/** \file
Random number generator and probability distribution
implementations, stolen from http://ftp.arl.mil/random/

The functionality has been converted into a Python extension.
*/

// Random.h: Definition and Implementation of Random Number Distribution Class
//           Ref: Richard Saucier, "Computer Generation of Statistical 
//                Distributions," ARL-TR-2168, US Army Research Laboratory,
//                Aberdeen Proving Ground, MD, 21005-5068, March 2000.
#include <stdio.h>
#include <list>
#include <algorithm>
#include <functional>
#include <cassert>
#include <climits>
#include <cmath>
#include <cfloat>     // for FLT_MIN and FLT_MAX

#include "random.h"

// Constants for Tausworthe random bit generator
// Ref: Tausworthe, Robert C., "Random Numbers Generated by Linear Recurrence
//      Modulo Two," Mathematics of Computation, vol. 19, pp. 201-209, 1965.

static const unsigned DEGREE_MAX = 32;   // max degree (bits per word)

static const unsigned BIT[1 + DEGREE_MAX] = {

	// Hexidecimal       Value      Degree   
	// -----------       -----      ------
	0x00000000,    // 0          0
	0x00000001,    // 2^0        1
	0x00000002,    // 2^1        2
	0x00000004,    // 2^2        3
	0x00000008,    // 2^3        4
	0x00000010,    // 2^4        5
	0x00000020,    // 2^5        6
	0x00000040,    // 2^6        7
	0x00000080,    // 2^7        8
	0x00000100,    // 2^8        9
	0x00000200,    // 2^9        10
	0x00000400,    // 2^10       11
	0x00000800,    // 2^11       12
	0x00001000,    // 2^12       13
	0x00002000,    // 2^13       14
	0x00004000,    // 2^14       15
	0x00008000,    // 2^15       16
	0x00010000,    // 2^16       17
	0x00020000,    // 2^17       18
	0x00040000,    // 2^18       19
	0x00080000,    // 2^19       20
	0x00100000,    // 2^20       21
	0x00200000,    // 2^21       22
	0x00400000,    // 2^22       23
	0x00800000,    // 2^23       24
	0x01000000,    // 2^24       25
	0x02000000,    // 2^25       26
	0x04000000,    // 2^26       27
	0x08000000,    // 2^27       28
	0x10000000,    // 2^28       29
	0x20000000,    // 2^29       30
	0x40000000,    // 2^30       31 
	0x80000000     // 2^31       32
};

// Coefficients that define a primitive polynomial (mod 2)
// Ref: Watson, E. J., "Primitive Polynomials (Mod 2),"
//      Mathematics of Computation, vol. 16, pp. 368-369, 1962.

static const unsigned MASK[1 + DEGREE_MAX] = {

	BIT[0],                                      // 0
	BIT[0],                                      // 1
	BIT[1],                                      // 2
	BIT[1],                                      // 3
	BIT[1],                                      // 4
	BIT[2],                                      // 5
	BIT[1],                                      // 6
	BIT[1],                                      // 7
	BIT[4] + BIT[3] + BIT[2],                    // 8
	BIT[4],                                      // 9
	BIT[3],                                      // 10
	BIT[2],                                      // 11
	BIT[6] + BIT[4] + BIT[1],                    // 12
	BIT[4] + BIT[3] + BIT[1],                    // 13
	BIT[5] + BIT[3] + BIT[1],                    // 14
	BIT[1],                                      // 15
	BIT[5] + BIT[3] + BIT[2],                    // 16
	BIT[3],                                      // 17
	BIT[5] + BIT[2] + BIT[1],                    // 18
	BIT[5] + BIT[2] + BIT[1],                    // 19
	BIT[3],                                      // 20
	BIT[2],                                      // 21
	BIT[1],                                      // 22
	BIT[5],                                      // 23
	BIT[4] + BIT[3] + BIT[1],                    // 24
	BIT[3],                                      // 25
	BIT[6] + BIT[2] + BIT[1],                    // 26
	BIT[5] + BIT[2] + BIT[1],                    // 27
	BIT[3],                                      // 28
	BIT[2],                                      // 29
	BIT[6] + BIT[4] + BIT[1],                    // 30
	BIT[3],                                      // 31
	BIT[7] + BIT[5] + BIT[3] + BIT[2] + BIT[1]   // 32
};

// private functions

void _seedTable(PyRandomObject *self)                // seeds the shuffle table
{
#ifndef _WIN32
	pthread_mutex_lock(&self->random_mutex);
#endif

	for (int i = _NTAB + 7; i >= 0; i--) {					// first perform 8 warm-ups

		long k = self->_seed / Q_;							// seed = ( A * seed ) % M
		self->_seed = A_ * (self->_seed - k * Q_) - k * R_; // without overflow by
		if (self->_seed < 0) self->_seed += _M;             // Schrage's method

		// store seeds into table
		if (i < _NTAB) PyList_SetItem((PyObject *)self->_table, i, PyLong_FromLong(self->_seed));
	}
	self->_next = PyLong_AsLong(PyList_GetItem((PyObject *)self->_table, 0));     // used as index next time

#ifndef _WIN32
	pthread_mutex_unlock(&self->random_mutex);
#endif
}

float _u(PyRandomObject *self)									// uniform rng
{
#ifndef _WIN32
	pthread_mutex_lock(&self->random_mutex);
#endif

	long k = self->_seed / Q_;								// seed = ( A*seed ) % M
	self->_seed = A_ * (self->_seed - k * Q_) - k * R_;     // without overflow by
	if (self->_seed < 0) self->_seed += _M;                 // Schrage's method

	int index = self->_next / _DIV;							// Bays-Durham shuffle
	// seed used for next time
	self->_next = PyLong_AsLong(PyList_GetItem((PyObject *)self->_table, index));
	// replace with new seed
	PyList_SetItem((PyObject *)self->_table, index, PyLong_FromLong(self->_seed));
	float retval = self->_next * _F;						// scale value within [0,1)

#ifndef _WIN32
	pthread_mutex_unlock(&self->random_mutex);
#endif

	return retval;
}

static float _parabola(float x, float xMin, float xMax)  // parabola
{
	if (x < xMin || x > xMax) return 0.0;

	float a = (float)(0.5 * (xMin + xMax));   // location parameter
	float b = (float)(0.5 * (xMax - xMin));   // scale parameter
	float yMax = (float)(0.75 / b);

	return float(yMax * (1. - (x - a) * (x - a) / (b * b)));
}


/*
* PyRandom_Type
*/

static PyMemberDef PyRandom_members[] = {
	{ "_table", T_OBJECT, offsetof(PyRandomObject, _table), READONLY,
	"shuffle table of seeds" },
	{ "_next", T_LONG, offsetof(PyRandomObject, _next), READONLY,
	"seed to be used as index into table" },
	{ "_seed", T_LONG, offsetof(PyRandomObject, _seed), READONLY,
	"current random number seed" },
	{ "_seed2", T_ULONG, offsetof(PyRandomObject, _seed2), READONLY,
	"seed for tausworthe random bit" },
	{ NULL }  /* Sentinel */
};


void PyRandom_dealloc(PyRandomObject* self)
{
	self->ob_type->tp_free((PyObject*)self);
}

PyObject * PyRandom_new(PyTypeObject *type, PyObject *args, PyObject *kwds)
{
	PyRandomObject *self;
	self = (PyRandomObject *)type->tp_alloc(type, 0);
	if (self != NULL) {
		self->_table = (PyListObject *)PyList_New(_NTAB);
		if (self->_table == NULL)
		{
			Py_DECREF(self);
			return NULL;
		}

		self->_seed = -1;
	}

	return (PyObject *)self;
}

int PyRandom_init(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	static char *kwlist[] = { "seed", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|l", kwlist, &self->_seed))
		return -1;

	reset(self);
	return 0;
}


PyObject * PyRandom_richcompare(PyObject *v, PyObject *w, int op)
{
	bool equal = false;
	PyRandomObject *p, *q;

	if (!PyRandom_Check(v) || !PyRandom_Check(w)) {
		Py_INCREF(Py_NotImplemented);
		return Py_NotImplemented;
	}

	p = (PyRandomObject *)v;
	q = (PyRandomObject *)w;

	switch (op) {
	case Py_LT:
	case Py_LE:
	case Py_GT:
	case Py_GE:
		Py_INCREF(Py_NotImplemented);
		return Py_NotImplemented;
	case Py_EQ:
	case Py_NE:
		equal = (p->_seed == q->_seed) && (p->_next == q->_next);
		for (int j = 0; j < _NTAB; j++)
			equal = equal && 
				(PyList_GetItem((PyObject *)p->_table, j) == PyList_GetItem((PyObject *)q->_table, j));
		PyObject *res;
		if (Py_NE) equal = !equal;
		res = (equal) ? Py_True : Py_False;
		Py_INCREF(res);
		return res;
	}

	Py_INCREF(Py_NotImplemented);
	return Py_NotImplemented;
}



PyObject * PyRandom_reset(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	static char *kwlist[] = { "seed", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|l", kwlist, &self->_seed))
		return NULL;

	reset(self);

	Py_RETURN_NONE;
}

// Continuous Distributions

// Arc Sine
PyObject * PyRandom_arcsine(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float xMin = 0., xMax = 1.;
	static char *kwlist[] = { "xMin", "xMax", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &xMin, &xMax))
		return NULL;

	return Py_BuildValue("f", arcsine(self, xMin, xMax));
}

// Beta (v > 0. and w > 0.)
PyObject * PyRandom_beta(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float v, w, xMin = 0., xMax = 1.;
	static char *kwlist[] = { "v", "w", "xMin", "xMax", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "ff|ff", kwlist, &v, &w, &xMin, xMax))
		return NULL;

	return Py_BuildValue("f", beta(self, v, w, xMin, xMax));
}

// Cauchy (or Lorentz)
PyObject * PyRandom_cauchy(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float a = 0., b = 1.;
	static char *kwlist[] = { "a", "b", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &a, &b))
		return NULL;

	return Py_BuildValue("f", cauchy(self, a, b));
}

// Chi-Square
PyObject * PyRandom_chiSquare(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	int df;
	static char *kwlist[] = { "df", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "i", kwlist, &df))
		return NULL;

	return Py_BuildValue("f", chiSquare(self, df));
}

// Cosine
PyObject * PyRandom_cosine(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float xMin = 0., xMax = 1.;
	static char *kwlist[] = { "xMin", "xMax", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &xMin, &xMax))
		return NULL;

	return Py_BuildValue("f", cosine(self, xMin, xMax));
}

// Double Log
PyObject * PyRandom_floatLog(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float xMin = -1., xMax = 1.;
	static char *kwlist[] = { "xMin", "xMax", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &xMin, &xMax))
		return NULL;

	return Py_BuildValue("f", floatLog(self, xMin, xMax));
}

// Erlang (b > 0. and c >= 1)
PyObject * PyRandom_erlang(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float b;
	int c;
	static char *kwlist[] = { "b", "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "fi", kwlist, &b, &c))
		return NULL;

	return Py_BuildValue("d", erlang(self, b, c));
}
// Exponential
PyObject * PyRandom_exponential(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float a = 0., c = 1.;		// location a, shape c
	static char *kwlist[] = { "a", "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &a, &c))
		return NULL;

	return Py_BuildValue("f", exponential(self, a, c));
}

// Extreme Value
PyObject * PyRandom_extremeValue(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float a = 0., c = 1.;		// location a, shape c
	static char *kwlist[] = { "a", "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &a, &c))
		return NULL;

	return Py_BuildValue("f", extremeValue(self, a, c));
}

// F Ratio (v and w >= 1)
PyObject * PyRandom_fRatio(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	int v, w;
	static char *kwlist[] = { "v", "w", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "ii", kwlist, &v, &w))
		return NULL;

	return Py_BuildValue("f", fRatio(self, v, w));
}

// Gamma
PyObject * PyRandom_gamma(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float a, b, c;		// location a, scale b, shape c
	static char *kwlist[] = { "a", "b", "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "fff", kwlist, &a, &b, &c))
		return NULL;

	return Py_BuildValue("f", gamma(self, a, b, c));
}

// Laplace
PyObject * PyRandom_laplace(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float a = 0., b = 1.;			// (or float exponential)
	static char *kwlist[] = { "a", "b", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &a, &b))
		return NULL;

	return Py_BuildValue("f", laplace(self, a, b));
}

// Logarithmic
PyObject * PyRandom_logarithmic(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float xMin = 0., xMax = 1.;
	static char *kwlist[] = { "xMin", "xMax", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &xMin, &xMax))
		return NULL;

	return Py_BuildValue("f", logarithmic(self, xMin, xMax));
}

// Logistic
PyObject * PyRandom_logistic(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float a = 0., c = 1.;
	static char *kwlist[] = { "a", "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &a, &c))
		return NULL;

	return Py_BuildValue("f", logistic(self, a, c));
}

// Lognormal
PyObject * PyRandom_lognormal(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float a, mu, sigma;
	static char *kwlist[] = { "a", "mu", "sigma", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "fff", kwlist, &a, &mu, &sigma))
		return NULL;

	return Py_BuildValue("f", gamma(self, a, mu, sigma));
}

// Normal
PyObject * PyRandom_normal(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float mu = 0., sigma = 1.;
	static char *kwlist[] = { "mu", "sigma", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &mu, &sigma))
		return NULL;

	return Py_BuildValue("f", normal(self, mu, sigma));
}

// Parabolic
PyObject * PyRandom_parabolic(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float xMin = 0., xMax = 1.;
	static char *kwlist[] = { "xMin", "xMax", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &xMin, &xMax))
		return NULL;

	return Py_BuildValue("f", parabolic(self, xMin, xMax));
}

// Pareto
PyObject * PyRandom_pareto(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float c;		// shape c
	static char *kwlist[] = { "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "f", kwlist, &c))
		return NULL;

	return Py_BuildValue("f", pareto(self, c));
}

// Pearson Type 5
PyObject * PyRandom_pearson5(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float b, c;		// scale b, shape c
	static char *kwlist[] = { "b", "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "ff", kwlist, &b, &c))
		return NULL;

	return Py_BuildValue("f", pearson5(self, b, c));
}

// Pearson Type 6
PyObject * PyRandom_pearson6(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float b, v, w;		// scale b, shape v & w
	static char *kwlist[] = { "b", "v", "w", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "fff", kwlist, &b, &v, &w))
		return NULL;

	return Py_BuildValue("f", pearson6(self, b, v, w));
}

// Power
PyObject * PyRandom_power(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float c;	// shape c
	static char *kwlist[] = { "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "f", kwlist, &c))
		return NULL;

	return Py_BuildValue("f", power(self, c));
}

// Rayleigh
PyObject * PyRandom_rayleigh(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float a, b;		// location a, scale b
	static char *kwlist[] = { "a", "b", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "ff", kwlist, &a, &b))
		return NULL;

	return Py_BuildValue("f", rayleigh(self, a, b));
}

// Student's T
PyObject * PyRandom_studentT(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	int df;			// degres of freedom df
	static char *kwlist[] = { "df", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "i", kwlist, &df))
		return NULL;

	return Py_BuildValue("f", studentT(self, df));
}

// Triangular with default interval [0,1) and default mode 0.5
PyObject * PyRandom_triangular(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float xMin = 0., xMax = 1., c = 0.5;
	static char *kwlist[] = { "xMin", "xMax", "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|fff", kwlist, &xMin, &xMax, &c))
		return NULL;

	return Py_BuildValue("f", triangular(self, xMin, xMax, c));
}

// Uniform on [xMin,xMax)
PyObject * PyRandom_uniform(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float xMin = 0., xMax = 1.;
	static char *kwlist[] = { "xMin", "xMax", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|ff", kwlist, &xMin, &xMax))
		return NULL;

	return Py_BuildValue("f", uniform(self, xMin, xMax));
}

//// User-Specified Distribution
//PyObject * PyRandom_userSpecified(PyRandomObject *self, PyObject *args, PyObject *kwds)
//{
//	//float(*usf)(float, float, float);	// pointer to user-specified function with parameters x, xMin, xMax
//	float xMin, xMax;     // function domain
//	float yMin, yMax;     // function range
//
//	if (!PyArg_ParseTuple(args, "Offff", &func, &xMin, &xMax, &yMin, &yMax))
//		   return NULL;
//
//	return Py_BuildValue("f", userSpecified(self, func, xMin, xMax, yMin, yMax));
//}

// Weibull
PyObject * PyRandom_weibull(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float a, b, c;		// location a, scale b,
	static char *kwlist[] = { "a", "b", "c", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "fff", kwlist, &a, &b, &c))
		return NULL;

	return Py_BuildValue("f", weibull(self, a, b, c));
}

// Discrete Distributions

// Bernoulli Trial
PyObject * PyRandom_bernoulli(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float p = 0.5;
	static char *kwlist[] = { "p", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|f", kwlist, &p))
		return NULL;

	return Py_BuildValue("i", (int)bernoulli(self, p));
}

// Binomial
PyObject * PyRandom_binomial(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	int n;
	float p;
	static char *kwlist[] = { "n", "p", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "if", kwlist, &n, &p))
		return NULL;

	return Py_BuildValue("i", binomial(self, n, p));
}

// Geometric
PyObject * PyRandom_geometric(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float p;
	static char *kwlist[] = { "p", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "f", kwlist, &p))
		return NULL;

	return Py_BuildValue("i", geometric(self, p));
}

// Hypergeometric
PyObject * PyRandom_hypergeometric(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	int n, N, K;		// trials n, size N, successes K
	static char *kwlist[] = { "n", "N", "K", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "iii", kwlist, &n, &N, &K))
		return NULL;

	return Py_BuildValue("i", hypergeometric(self, n, N, K));
}

// Multinomial
PyObject * PyRandom_multinomial(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	int n;			// trials n
	float *p;       // probability vector p,
	int *count;     // success vector count,
	int m;          // number of disjoint events m

	static char *kwlist[] = { "n", "p", "count", "m", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "iOOi", kwlist, &n, &p, &count, &m))
		return NULL;

	multinomial(self, n, p, count, m);
	return NULL;
}

// Negative Binomial
PyObject * PyRandom_negativeBinomial(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	int s;		// successes s
	float p;	// probability p
	static char *kwlist[] = { "s", "p", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "if", kwlist, &s, &p))
		return NULL;

	return Py_BuildValue("i", negativeBinomial(self, s, p));
}

// Pascal
PyObject * PyRandom_pascal(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	int s;		// successes s
	float p;	// probability p
	static char *kwlist[] = { "s", "p", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "if", kwlist, &s, &p))
		return NULL;

	return Py_BuildValue("i", pascal(self, s, p));
}

// Poisson
PyObject * PyRandom_poisson(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	float mu;
	static char *kwlist[] = { "mu", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "f", kwlist, &mu))
		return NULL;

	return Py_BuildValue("i", poisson(self, mu));
}

// Uniform Discrete
PyObject * PyRandom_uniformDiscrete(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	int i, j;		// inclusive i to j
	static char *kwlist[] = { "i", "j", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "ii", kwlist, &i, &j))
		return NULL;

	return Py_BuildValue("i", uniformDiscrete(self, i, j));
}

// Empirical and Data-Driven Distributions

// Empirical Continuous
PyObject * PyRandom_empirical(PyRandomObject *self)
{
	static vector<float> x, cdf;
	static int n;
	static bool init = false;

	if (!init) {
		ifstream in("empiricalDistribution");
		if (!in) {
			cerr << "Cannot open \"empiricalDistribution\" input file" << endl;
			exit(1);
		}
		float value, prob;
		while (in >> value >> prob) {   // read in empirical distribution
			x.push_back(value);
			cdf.push_back(prob);
		}
		n = x.size();
		init = true;

		// check that this is indeed a cumulative distribution

		assert(0. == cdf[0] && cdf[n - 1] == 1.);
		for (int i = 1; i < n; i++) assert(cdf[i - 1] < cdf[i]);
	}

	float p = _u(self);
	for (int i = 0; i < n - 1; i++)
		if (cdf[i] <= p && p < cdf[i + 1])
			return Py_BuildValue("f", x[i] + (x[i + 1] - x[i]) *
			(p - cdf[i]) / (cdf[i + 1] - cdf[i]));

	return Py_BuildValue("f", x[n - 1]);
}

// Empirical Discrete
PyObject * PyRandom_empiricalDiscrete(PyRandomObject *self)
{
	static vector< int >    k;
	static vector< float > f[2];   // f[ 0 ] is pdf and f[ 1 ] is cdf
	static float           max;
	static int              n;
	static bool             init = false;

	if (!init) {
		ifstream in("empiricalDiscrete");
		if (!in) {
			cerr << "Cannot open \"empiricalDiscrete\" input file" << endl;
			exit(1);
		}
		int value;
		float freq;
		while (in >> value >> freq) {   // read in empirical data
			k.push_back(value);
			f[0].push_back(freq);
		}
		n = k.size();
		init = true;

		// form the cumulative distribution

		f[1].push_back(f[0][0]);
		for (int i = 1; i < n; i++)
			f[1].push_back(f[1][i - 1] + f[0][i]);

		// check that the integer points are in ascending order

		for (int i = 1; i < n; i++) assert(k[i - 1] < k[i]);

		max = f[1][n - 1];
	}

	// select a uniform variate between 0 and the max value of the cdf

	float p = uniform(self, 0., max);

	// locate and return the corresponding index

	for (int i = 0; i < n; i++) if (p <= f[1][i]) return Py_BuildValue("i", k[i]);
	return Py_BuildValue("i", k[n - 1]);
}


// Sample w or w/o replacement from a distribution of 1-D data in a file
// vector for sampling with replacement flag that file has been read in
// number of data elements in the file subscript in the sequential order.
// Or sample from a given distribution of multi-dimensional data
PyObject * PyRandom_sample(PyRandomObject *self, PyObject *args, PyObject *kwds)
{
	bool replace = true;
	float *x = NULL;
	int ndim = 0;
	static char *kwlist[] = { "replace", "x", "ndim", NULL };

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|iOi", kwlist, &replace))
		return NULL;

	if (x) {
		sample(self, x, ndim);
		return NULL;
	}

	return Py_BuildValue("f", sample(self, replace));
}

static PyMethodDef PyRandom_methods[] = {
	{ "reset", (PyCFunction)PyRandom_reset, METH_VARARGS | METH_KEYWORDS,
	"Reset the seeds." },
	{ "arcsine", (PyCFunction)PyRandom_arcsine, METH_VARARGS | METH_KEYWORDS,
	"Arc Sine distribution." },
	{ "beta", (PyCFunction)PyRandom_beta, METH_VARARGS | METH_KEYWORDS,
	"Beta distribution." },
	{ "cauchy", (PyCFunction)PyRandom_cauchy, METH_VARARGS | METH_KEYWORDS,
	"Cauchy distribution." },
	{ "chiSquare", (PyCFunction)PyRandom_chiSquare, METH_VARARGS | METH_KEYWORDS,
	"Chi-Square distribution." },
	{ "cosine", (PyCFunction)PyRandom_cosine, METH_VARARGS | METH_KEYWORDS,
	"Cosine distribution." },
	{ "floatLog", (PyCFunction)PyRandom_floatLog, METH_VARARGS | METH_KEYWORDS,
	"Double Log distribution." },
	{ "erlang", (PyCFunction)PyRandom_erlang, METH_VARARGS | METH_KEYWORDS,
	"Erlang (b > 0. and c >= 1) distribution." },
	{ "exponential", (PyCFunction)PyRandom_exponential, METH_VARARGS | METH_KEYWORDS,
	"Exponential distribution." },
	{ "extremeValue", (PyCFunction)PyRandom_extremeValue, METH_VARARGS | METH_KEYWORDS,
	"Extreme Value distribution." },
	{ "fRatio", (PyCFunction)PyRandom_fRatio, METH_VARARGS | METH_KEYWORDS,
	"F Ratio (v and w >= 1) distribution." },
	{ "gamma", (PyCFunction)PyRandom_gamma, METH_VARARGS | METH_KEYWORDS,
	"Gamma distribution." },
	{ "laplace", (PyCFunction)PyRandom_laplace, METH_VARARGS | METH_KEYWORDS,
	"Laplace distribution." },
	{ "logarithmic", (PyCFunction)PyRandom_logarithmic, METH_VARARGS | METH_KEYWORDS,
	"Logarithmic distribution." },
	{ "logistic", (PyCFunction)PyRandom_logistic, METH_VARARGS | METH_KEYWORDS,
	"Logistic distribution." },
	{ "lognormal", (PyCFunction)PyRandom_lognormal, METH_VARARGS | METH_KEYWORDS,
	"Lognormal distribution." },
	{ "normal", (PyCFunction)PyRandom_normal, METH_VARARGS | METH_KEYWORDS,
	"Normal distribution." },
	{ "parabolic", (PyCFunction)PyRandom_parabolic, METH_VARARGS | METH_KEYWORDS,
	"Parabolic distribution." },
	{ "pareto", (PyCFunction)PyRandom_pareto, METH_VARARGS | METH_KEYWORDS,
	"Pareto distribution." },
	{ "pearson5", (PyCFunction)PyRandom_pearson5, METH_VARARGS | METH_KEYWORDS,
	"Pearson Type 5 distribution." },
	{ "pearson6", (PyCFunction)PyRandom_pearson6, METH_VARARGS | METH_KEYWORDS,
	"Pearson Type 6 distribution." },
	{ "power", (PyCFunction)PyRandom_power, METH_VARARGS | METH_KEYWORDS,
	"Power distribution." },
	{ "rayleigh", (PyCFunction)PyRandom_rayleigh, METH_VARARGS | METH_KEYWORDS,
	"Rayleigh distribution." },
	{ "studentT", (PyCFunction)PyRandom_studentT, METH_VARARGS | METH_KEYWORDS,
	"Student's T distribution." },
	{ "triangular", (PyCFunction)PyRandom_triangular, METH_VARARGS | METH_KEYWORDS,
	"Triangular distribution." },
	{ "uniform", (PyCFunction)PyRandom_uniform, METH_VARARGS | METH_KEYWORDS,
	"Uniform distribution." },
	//{ "userSpecified", (PyCFunction)PyRandom_userSpecified, METH_VARARGS | METH_KEYWORDS,
	//"Uer-Specified distribution." },
	{ "weibull", (PyCFunction)PyRandom_weibull, METH_VARARGS | METH_KEYWORDS,
	"Weibull distribution." },
	{ "bernoulli", (PyCFunction)PyRandom_bernoulli, METH_VARARGS | METH_KEYWORDS,
	"Bernoulli Trial." },
	{ "binomial", (PyCFunction)PyRandom_binomial, METH_VARARGS | METH_KEYWORDS,
	"Binomial distribution." },
	{ "hypergeometric", (PyCFunction)PyRandom_hypergeometric, METH_VARARGS | METH_KEYWORDS,
	"Hypergeometric distribution." },
	{ "multinomial", (PyCFunction)PyRandom_multinomial, METH_VARARGS | METH_KEYWORDS,
	"Multinomial distribution." },
	{ "negativeBinomial", (PyCFunction)PyRandom_negativeBinomial, METH_VARARGS | METH_KEYWORDS,
	"Negative Binomial distribution." },
	{ "pascal", (PyCFunction)PyRandom_pascal, METH_VARARGS | METH_KEYWORDS,
	"Pascal distribution." },
	{ "poisson", (PyCFunction)PyRandom_poisson, METH_VARARGS | METH_KEYWORDS,
	"Poisson distribution." },
	{ "uniformDiscrete", (PyCFunction)PyRandom_uniformDiscrete, METH_VARARGS | METH_KEYWORDS,
	"Uniform Discrete distribution." },
	{ "sample", (PyCFunction)PyRandom_sample, METH_VARARGS | METH_KEYWORDS,
	"Sample w or w/o replacement from a distribution of 1-D data in a file" \
	" vector for sampling with replacement flag that file has been read in" \
	" number of data elements in the file subscript in the sequential order or" \
	" sample from a given distribution of multi-dimensional data." },
	//{ "pareto", (PyCFunction)PyRandom_pareto, METH_VARARGS | METH_KEYWORDS,
	//"Pareto distribution." },
	//{ "pearson5", (PyCFunction)PyRandom_pearson5, METH_VARARGS | METH_KEYWORDS,
	//"Pearson Type 5 distribution." },
	//{ "pearson6", (PyCFunction)PyRandom_pearson6, METH_VARARGS | METH_KEYWORDS,
	//"Pearson Type 6 distribution." },
	{ NULL }  /* Sentinel */
};


PyTypeObject PyRandom_Type = {
	PyObject_HEAD_INIT(NULL)
	0,                               /*ob_size*/
	"Random",						 /*tp_name*/
	sizeof(PyRandomObject),          /*tp_basicsize*/
	0,                               /*tp_itemsize*/
	(destructor)PyRandom_dealloc,    /*tp_dealloc*/
	0,                               /*tp_print*/
	0,                               /*tp_getattr*/
	0,                               /*tp_setattr*/
	0,                               /*tp_compare*/
	0,                               /*tp_repr*/
	0,                               /*tp_as_number*/
	0,                               /*tp_as_sequence*/
	0,                               /*tp_as_mapping*/
	0,                               /*tp_hash */
	0,                               /*tp_call*/
	0,                               /*tp_str*/
	0,                               /*tp_getattro*/
	0,                               /*tp_setattro*/
	0,                               /*tp_as_buffer*/
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
	"Random objects",                /* tp_doc */
	0,		                         /* tp_traverse */
	0,		                         /* tp_clear */
	PyRandom_richcompare,		     /* tp_richcompare */
	0,		                         /* tp_weaklistoffset */
	0,		                         /* tp_iter */
	0,		                         /* tp_iternext */
	PyRandom_methods,                /* tp_methods */
	PyRandom_members,                /* tp_members */
	0,								 /* tp_getset */
	0,                               /* tp_base */
	0,                               /* tp_dict */
	0,                               /* tp_descr_get */
	0,                               /* tp_descr_set */
	0,                               /* tp_dictoffset */
	(initproc)PyRandom_init,         /* tp_init */
	0,                               /* tp_alloc */
	PyRandom_new,                    /* tp_new */
};




// Utility functions

void reset(PyRandomObject *self)
{
	if (self->_seed == -1) {
#ifndef _WIN32
		pthread_mutex_init(&self->random_mutex, NULL);
		self->_seed = long(getpid());
#else
		self->_seed = long(_getpid());
#endif
	}

	if (self->_seed == 0 || self->_seed == LONG_MAX) {
		char msg[50];
#ifndef _WIN32
		snprintf(msg, sizeof(msg), "The seed value must be not 0 or %l", LONG_MAX);
#else
		sprintf_s(msg, sizeof(msg), "The seed value must be not 0 or %l", LONG_MAX);
#endif
		PyErr_SetString(PyExc_TypeError, msg);
	}

	_seedTable(self);
	self->_seed2 = self->_seed | 1;   // so that all bits cannot be zero
}


// Continuous Distributions

// Arc Sine
float arcsine(PyRandomObject *self, float xMin, float xMax)
{
	float q = (float)sin(M_PI_2 * _u(self));
	return xMin + (xMax - xMin) * q * q;
}

// Beta
float beta(PyRandomObject *self, float v, float w,
	float xMin, float xMax)   // (v > 0. and w > 0.)
{
	if (v < w) return xMax - (xMax - xMin) * beta(self, w, v);
	float y1 = gamma(self, 0., 1., v);
	float y2 = gamma(self, 0., 1., w);
	return xMin + (xMax - xMin) * y1 / (y1 + y2);
}

// Cauchy (or Lorentz)
float cauchy(PyRandomObject *self, float a, float b)   
{
	// a is the location parameter and b is the scale parameter
	// b is the half width at half maximum (HWHM) and variance doesn't exist

	assert(b > 0.);

	return a + b * (float)tan(M_PI * uniform(self, -0.5, 0.5));
}

// Chi-Square
float chiSquare(PyRandomObject *self, int df)   
{
	assert(df >= 1);

	return gamma(self, 0.f, 2.f, 0.5f * float(df));
}

// Cosine
float cosine(PyRandomObject *self, float xMin, float xMax)   
{
	assert(xMin < xMax);

	float a = 0.5f * (xMin + xMax);    // location parameter
	float b = (xMax - xMin) / float(M_PI);   // scale parameter

	return a + b * asin(uniform(self, -1., 1.));
}

// Double Log
float floatLog(PyRandomObject *self, float xMin, float xMax)   
{
	assert(xMin < xMax);

	float a = 0.5f * (xMin + xMax);    // location parameter
	float b = 0.5f * (xMax - xMin);    // scale parameter

	if (bernoulli(self, 0.5f)) return a + b * _u(self) * _u(self);
	else                    return a - b * _u(self) * _u(self);
}

// Erlang (b > 0. and c >= 1)
float erlang(PyRandomObject *self, float b, int c)   
{
	assert(b > 0. && c >= 1);

	float prod = 1.;
	for (int i = 0; i < c; i++) prod *= _u(self);

	return -b * log(prod);
}

// Exponential
float exponential(PyRandomObject *self, float a, float c)   
{                                                    // location a, shape c
	assert(c > 0.0);

	return a - c * log(_u(self));
}

// Extreme Value
float extremeValue(PyRandomObject *self, float a, float c)   
{                                                     // location a, shape c
	assert(c > 0.);

	return a + c * log(-log(_u(self)));
}

// F Ratio (v and w >= 1)
float fRatio(PyRandomObject *self, int v, int w)   
{
	assert(v >= 1 && w >= 1);

	return (chiSquare(self, v) / v) / (chiSquare(self, w) / w);
}

// Gamma
float gamma(PyRandomObject *self, float a, float b, float c)  
{                                             // location a, scale b, shape c
	assert(b > 0. && c > 0.);

	const float A = 1.f / sqrt(2.f * c - 1.f);
	const float B = c - log(4.f);
	const float Q = c + 1.f / A;
	const float T = 4.5f;
	const float D = 1.f + log(T);
	const float C = 1.f + c / (float)M_E;

	if (c < 1.) {
		while (true) {
			float p = C * _u(self);
			if (p > 1.) {
				float y = -log((C - p) / c);
				if (_u(self) <= powf(y, c - 1.f)) return a + b * y;
			}
			else {
				float y = powf(p, 1.f / c);
				if (_u(self) <= exp(-y)) return a + b * y;
			}
		}
	}
	else if (c == 1.0) return exponential(self, a, b);
	else {
		while (true) {
			float p1 = _u(self);
			float p2 = _u(self);
			float v = A * log(p1 / (1.f - p1));
			float y = c * exp(v);
			float z = p1 * p1 * p2;
			float w = B + Q * v - y;
			if (w + D - T * z >= 0. || w >= log(z)) return a + b * y;
		}
	}
}

// Laplace (or float exponential)
float laplace(PyRandomObject *self, float a, float b)   
{                                             
	assert(b > 0.);

	// composition method

	if (bernoulli(self, 0.5)) return a + b * log(_u(self));
	else                    return a - b * log(_u(self));
}

// Logarithmic
float logarithmic(PyRandomObject *self, float xMin, float xMax)   
{
	assert(xMin < xMax);

	float a = xMin;          // location parameter
	float b = xMax - xMin;   // scale parameter

	// use convolution formula for product of two IID uniform variates

	return a + b * _u(self) * _u(self);
}

// Logistic
float logistic(PyRandomObject *self, float a, float c)   
{
	assert(c > 0.);

	return a - c * (float)log(1. / _u(self) - 1.);
}

// Lognormal
float lognormal(PyRandomObject *self, float a, float mu, float sigma)   
{
	return a + exp(normal(self, mu, sigma));
}

// Normal
float normal(PyRandomObject *self, float mu, float sigma)   
{
	assert(sigma > 0.);

	static bool f = true;
	static float p, p1, p2;
	float q;

	if (f) {
		do {
			p1 = uniform(self, -1., 1.);
			p2 = uniform(self, -1., 1.);
			p = p1 * p1 + p2 * p2;
		} while (p >= 1.);
		q = p1;
	}
	else
		q = p2;
	f = !f;

	return mu + sigma * q * (float)sqrt(-2. * log(p) / p);
}

// Parabolic
float parabolic(PyRandomObject *self, float xMin, float xMax)   
{
	assert(xMin < xMax);

	float a = 0.5f * (xMin + xMax);        // location parameter
	float yMax = _parabola(a, xMin, xMax);   // maximum function range

	return userSpecified(self, _parabola, xMin, xMax, 0., yMax);
}

// Pareto
float pareto(PyRandomObject *self, float c)   
{                           // shape c
	assert(c > 0.);

	return powf(_u(self), -1.f / c);
}

// Pearson Type 5
float pearson5(PyRandomObject *self, float b, float c)   
{                                       // scale b, shape c
	assert(b > 0. && c > 0.);

	return 1.f / gamma(self, 0., 1.f / b, c);
}

// Pearson Type 6
float pearson6(PyRandomObject *self, float b, float v, float w)   
{                                                 // scale b, shape v & w
	assert(b > 0. && v > 0. && w > 0.);

	return gamma(self, 0., b, v) / gamma(self, 0., b, w);
}

// Power
float power(PyRandomObject *self, float c)   
{                          // shape c
	assert(c > 0.);

	return powf(_u(self), 1.f / c);
}

// Rayleigh
float rayleigh(PyRandomObject *self, float a, float b)   
{                                       // location a, scale b
	assert(b > 0.);

	return a + b * sqrt(-log(_u(self)));
}

// Student's T
float studentT(PyRandomObject *self, int df)   
{                           // degres of freedom df
	assert(df >= 1);

	return normal(self) / sqrt(chiSquare(self, df) / df);
}

// Triangular
float triangular(PyRandomObject *self, float xMin,     
	float xMax,     // with default interval [0,1)
	float c)		// and default mode 0.5
{
	assert(xMin < xMax && xMin <= c && c <= xMax);

	float p = _u(self), q = 1.f - p;

	if (p <= (c - xMin) / (xMax - xMin))
		return xMin + sqrt((xMax - xMin) * (c - xMin) * p);
	else
		return xMax - sqrt((xMax - xMin) * (xMax - c) * q);
}

// Uniform on [xMin,xMax)
float uniform(PyRandomObject *self, float xMin, float xMax)
{
	assert(xMin < xMax);

	return xMin + (xMax - xMin) * _u(self);
}

// User-Specified Distribution
float userSpecified(
	PyRandomObject *self,
	float(*usf)(                // pointer to user-specified function
	float,						// x
	float,						// xMin
	float),						// xMax
	float xMin, float xMax,     // function domain
	float yMin, float yMax)     // function range
{
	assert(xMin < xMax && yMin < yMax);

	float x, y, areaMax = (xMax - xMin) * (yMax - yMin);

	// acceptance-rejection method

	do {
		x = uniform(self, 0.0, areaMax) / (yMax - yMin) + xMin;
		y = uniform(self, yMin, yMax);

	} while (y > usf(x, xMin, xMax));

	return x;
}

// Weibull
float weibull(PyRandomObject *self, float a, float b, float c)
{                                                // location a, scale b,
	assert(b > 0. && c > 0.);                    // shape c

	return a + b * powf(-log(_u(self)), 1.f / c);
}

// Discrete Distributions

// Bernoulli Trial
bool bernoulli(PyRandomObject *self, float p)
{
	assert(0. <= p && p <= 1.);

	return _u(self) < p;
}

// Binomial
int binomial(PyRandomObject *self, int n, float p)
{
	assert(n >= 1 && 0. <= p && p <= 1.);

	int sum = 0;
	for (int i = 0; i < n; i++) sum += bernoulli(self, p);
	return sum;
}

// Geometric
int geometric(PyRandomObject *self, float p)
{
	assert(0. < p && p < 1.);

	return int(log(_u(self)) / log(1. - p));
}

// Hypergeometric
int hypergeometric(PyRandomObject *self, int n, int N, int K)
{                                                // trials n, size N, successes K
	assert(0 <= n && n <= N && N >= 1 && K >= 0);

	int count = 0;
	for (int i = 0; i < n; i++, N--) {

		float p = float(K) / float(N);
		if (bernoulli(self, p)) { count++; K--; }
	}
	return count;
}

// Multinomial
void multinomial(PyRandomObject *self,
	int n,				// trials n
	float p[],          // probability vector p,
	int   count[],      // success vector count,
	int   m)            // number of disjoint events m
{
	assert(m >= 2);   // at least 2 events
	float sum = 0.;
	for (int bin = 0; bin < m; bin++) sum += p[bin];    // probabilities
	assert(sum == 1.);                                  // must sum to 1

	for (int bin = 0; bin < m; bin++) count[bin] = 0;   // initialize

	// generate n uniform variates in the interval [0,1) and bin the results

	for (int i = 0; i < n; i++) {

		float lower = 0., upper = 0., u = _u(self);

		for (int bin = 0; bin < m; bin++) {

			// locate subinterval, which is of length p[ bin ],
			// that contains the variate and increment the corresponding counter

			lower = upper;
			upper += p[bin];
			if (lower <= u && u < upper) { count[bin]++; break; }
		}
	}
}

// Negative Binomial
int negativeBinomial(PyRandomObject *self, int s, float p)
{                                         // successes s, probability p
	assert(s >= 1 && 0. < p && p < 1.);

	int sum = 0;
	for (int i = 0; i < s; i++) sum += geometric(self, p);
	return sum;
}

// Pascal
int pascal(PyRandomObject *self, int s, float p)
{                                          // successes s, probability p
	return negativeBinomial(self, s, p) + s;
}

// Poisson
int poisson(PyRandomObject *self, float mu)
{
	assert(mu > 0.);

	float a = exp(-mu);
	float b = 1.;

	int i;
	for (i = 0; b >= a; i++) b *= _u(self);
	return i - 1;
}

// Uniform Discrete
int uniformDiscrete(PyRandomObject *self, int i, int j)
{                                     // inclusive i to j
	assert(i < j);

	return i + int((j - i + 1) * _u(self));
}


// Sample w or w/o replacement from a distribution of 1-D data in a file
// vector for sampling with replacement flag that file has been read in
// number of data elements in the file subscript in the sequential order
float sample(PyRandomObject *self, bool replace)
{                                   
	static vector< float > v;       
	static bool init = false;       
	static int n;                   
	static int index = 0;           

	if (!init) {
		ifstream in("sampleData");
		if (!in) {
			cerr << "Cannot open \"sampleData\" file" << endl;
			exit(1);
		}
		float d;
		while (in >> d) v.push_back(d);
		in.close();
		n = v.size();
		init = true;
		if (replace == false) {   // sample without replacement

			// shuffle contents of v once and for all
			// Ref: Knuth, D. E., The Art of Computer Programming, Vol. 2:
			//      Seminumerical Algorithms. London: Addison-Wesley, 1969.

			for (int i = n - 1; i > 0; i--) {
				int j = int((i + 1) * _u(self));
				swap(v[i], v[j]);
			}
		}
	}

	// return a random sample

	if (replace)                                // sample w/ replacement
		return v[uniformDiscrete(self, 0, n - 1)];
	else {                                        // sample w/o replacement
		assert(index < n);                       // retrieve elements
		return v[index++];                       // in sequential order
	}
}

// Sample from a given distribution of multi-dimensional data
void sample(PyRandomObject *self, float x[], int ndim)
{
	static const int N_DIM = 6;
	assert(ndim <= N_DIM);

	static vector< float > v[N_DIM];
	static bool init = false;
	static int n;

	if (!init)  {
		ifstream in("sampleData");
		if (!in) {
			cerr << "Cannot open \"sampleData\" file" << endl;
			exit(1);
		}
		float d;
		while (!in.eof()) {
			for (int i = 0; i < ndim; i++) {
				in >> d;
				v[i].push_back(d);
			}
		}
		in.close();
		n = v[0].size();
		init = true;
	}
	int index = uniformDiscrete(self, 0, n - 1);
	for (int i = 0; i < ndim; i++) x[i] = v[i][index];
}


// comparison functor for determining the neighborhood of a data point

struct dSquared :
	public binary_function< cartesianCoord, cartesianCoord, bool > {
	bool operator()(cartesianCoord p, cartesianCoord q) {
		return p.x * p.x + p.y * p.y < q.x * q.x + q.y * q.y;
	}
};

// Stochastic Interpolation
cartesianCoord stochasticInterpolation(PyRandomObject *self)
// Refs: Taylor, M. S. and J. R. Thompson, Computational Statistics & Data 
//       Analysis, Vol. 4, pp. 93-101, 1986; Thompson, J. R., Empirical Model
//       Building, pp. 108-114, Wiley, 1989; Bodt, B. A. and M. S. Taylor,
//       A Data Based Random Number Generator for A Multivariate Distribution
//       - A User's Manual, ARBRL-TR-02439, BRL, APG, MD, Nov. 1982.
{
	static vector< cartesianCoord > data;
	static cartesianCoord           min, max;
	static int                      m;
	static float                   lower, upper;
	static bool                     init = false;

	if (!init) {
		ifstream in("stochasticData");
		if (!in) {
			cerr << "Cannot open \"stochasticData\" input file" << endl;
			exit(1);
		}

		// read in the data and set min and max values

		min.x = min.y = FLT_MAX;
		max.x = max.y = FLT_MIN;
		cartesianCoord p;
		while (in >> p.x >> p.y) {

			min.x = (p.x < min.x ? p.x : min.x);
			min.y = (p.y < min.y ? p.y : min.y);
			max.x = (p.x > max.x ? p.x : max.x);
			max.y = (p.y > max.y ? p.y : max.y);

			data.push_back(p);
		}
		in.close();
		init = true;

		// scale the data so that each dimension will have equal weight

		for (unsigned i = 0; i < data.size(); i++) {

			data[i].x = (data[i].x - min.x) / (max.x - min.x);
			data[i].y = (data[i].y - min.y) / (max.y - min.y);
		}

		// set m, the number of points in a neighborhood of a given point

		m = data.size() / 20;       // 5% of all the data points
		if (m < 5) m = 5;        // but no less than 5
		if (m > 20) m = 20;       // and no more than 20

		lower = (1.f - (float)sqrt(3. * (float(m) - 1.))) / float(m);
		upper = (1.f + (float)sqrt(3. * (float(m) - 1.))) / float(m);
	}

	// uniform random selection of a data point (with replacement)

	cartesianCoord origin = data[uniformDiscrete(self, 0, data.size() - 1)];

	// make this point the origin of the coordinate system

	for (unsigned n = 0; n < data.size(); n++) data[n] -= origin;

	// sort the data with respect to its distance (squared) from this origin

	sort(data.begin(), data.end(), dSquared());

	// find the mean value of the data in the neighborhood about this point

	cartesianCoord mean;
	mean.x = mean.y = 0.;
	for (int n = 0; n < m; n++) mean += data[n];
	mean /= float(m);

	// select a random linear combination of the points in this neighborhood

	cartesianCoord p;
	p.x = p.y = 0.;
	for (int n = 0; n < m; n++) {

		float rn;
		if (m == 1) rn = 1.;
		else          rn = uniform(self, lower, upper);
		p.x += rn * (data[n].x - mean.x);
		p.y += rn * (data[n].y - mean.y);
	}

	// restore the data to its original form

	for (unsigned n = 0; n < data.size(); n++) data[n] += origin;

	// use mean and original point to translate the randomly-chosen point

	p += mean;
	p += origin;

	// scale randomly-chosen point to the dimensions of the original data

	p.x = p.x * (max.x - min.x) + min.x;
	p.y = p.y * (max.y - min.y) + min.y;

	return p;
}


// Multivariate Distributions

// Bivariate Gaussian
cartesianCoord bivariateNormal(PyRandomObject *self, 
	float muX,   
	float sigmaX,
	float muY,
	float sigmaY)
{
	assert(sigmaX > 0. && sigmaY > 0.);

	cartesianCoord p;
	p.x = normal(self, muX, sigmaX);
	p.y = normal(self, muY, sigmaY);
	return p;
}

// Bivariate Uniform
cartesianCoord bivariateUniform(PyRandomObject *self, 
	float xMin,    
	float xMax,
	float yMin,
	float yMax)
{
	assert(xMin < xMax && yMin < yMax);

	float x0 = 0.5f * (xMin + xMax);
	float y0 = 0.5f * (yMin + yMax);
	float a = 0.5f * (xMax - xMin);
	float b = 0.5f * (yMax - yMin);
	float x, y;

	do {
		x = uniform(self, -1., 1.);
		y = uniform(self, -1., 1.);

	} while (x * x + y * y > 1.);

	cartesianCoord p;
	p.x = x0 + a * x;
	p.y = y0 + b * y;
	return p;
}

// Correlated Normal
cartesianCoord corrNormal(PyRandomObject *self, 
	float r,              
	float muX,
	float sigmaX,
	float muY,
	float sigmaY)
{
	assert(-1. <= r && r <= 1. &&          // bounds on correlation coeff
		sigmaX > 0. && sigmaY > 0.);   // positive std dev

	float x = normal(self);
	float y = normal(self);

	y = r * x + (float)sqrt(1. - r * r) * y;     // correlate the variables

	cartesianCoord p;
	p.x = muX + sigmaX * x;                 // translate and scale
	p.y = muY + sigmaY * y;
	return p;
}

// Correlated Uniform
cartesianCoord corrUniform(PyRandomObject *self, 
	float r,
	float xMin,
	float xMax,
	float yMin,
	float yMax)
{
	assert(-1. <= r && r <= 1. &&          // bounds on correlation coeff
		xMin < xMax && yMin < yMax);   // bounds on domain

	float x0 = 0.5f * (xMin + xMax);
	float y0 = 0.5f * (yMin + yMax);
	float a = 0.5f * (xMax - xMin);
	float b = 0.5f * (yMax - yMin);
	float x, y;

	do {
		x = uniform(self, -1., 1.);
		y = uniform(self, -1., 1.);

	} while (x * x + y * y > 1.);

	y = r * x + (float)sqrt(1. - r * r) * y;   // correlate the variables

	cartesianCoord p;
	p.x = x0 + a * x;                     // translate and scale
	p.y = y0 + b * y;
	return p;
}

// Uniform Spherical
sphericalCoord spherical(PyRandomObject *self, 
	float thMin,
	float thMax,
	float phMin,
	float phMax)
{
	assert(0. <= thMin && thMin < thMax && thMax <= M_PI &&
		0. <= phMin && phMin < phMax && phMax <= 2. * M_PI);

	sphericalCoord p;                       // azimuth
	p.theta = acos(uniform(self, cos(thMax), cos(thMin)));   // polar angle
	p.phi = uniform(self, phMin, phMax);                     // azimuth
	return p;
}
// Uniform over the surface of an n-dimensional unit sphere
void sphericalND(PyRandomObject *self, float x[], int n)   
{
	// generate a point inside the unit n-sphere by normal polar method

	float r2 = 0.;
	for (int i = 0; i < n; i++) {
		x[i] = normal(self);
		r2 += x[i] * x[i];
	}

	// project the point onto the surface of the unit n-sphere by scaling

	const float A = 1.f / sqrt(r2);
	for (int i = 0; i < n; i++) x[i] *= A;
}

// Number Theoretic Distributions

// Maximal Avoidance (1-D)
float avoidance(void)   
{                          // overloaded for convenience
	float x[1];
	avoidance(x, 1);
	return x[0];
}

// Maximal Avoidance (N-D)
void avoidance(float x[], unsigned ndim)   
{
	static const unsigned MAXBIT = 30;
	static const unsigned MAXDIM = 6;

	assert(ndim <= MAXDIM);

	static unsigned long ix[MAXDIM + 1] = { 0 };
	static unsigned long *u[MAXBIT + 1];
	static unsigned long mdeg[MAXDIM + 1] = { // degree of
		0,                                       // primitive polynomial
		1, 2, 3, 3, 4, 4
	};
	static unsigned long p[MAXDIM + 1] = {   // decimal encoded
		0,                                      // interior bits
		0, 1, 1, 2, 1, 4
	};
	static unsigned long v[MAXDIM * MAXBIT + 1] = {
		0,
		1, 1, 1, 1, 1, 1,
		3, 1, 3, 3, 1, 1,
		5, 7, 7, 3, 3, 5,
		15, 11, 5, 15, 13, 9
	};

	static float fac;
	static int in = -1;
	unsigned j, k;
	unsigned long i, m, pp;

	if (in == -1) {
		in = 0;
		fac = 1. / (1L << MAXBIT);
		for (j = 1, k = 0; j <= MAXBIT; j++, k += MAXDIM) u[j] = &v[k];
		for (k = 1; k <= MAXDIM; k++) {
			for (j = 1; j <= mdeg[k]; j++) u[j][k] <<= (MAXBIT - j);
			for (j = mdeg[k] + 1; j <= MAXBIT; j++) {
				pp = p[k];
				i = u[j - mdeg[k]][k];
				i ^= (i >> mdeg[k]);
				for (int n = mdeg[k] - 1; n >= 1; n--) {
					if (pp & 1) i ^= u[j - n][k];
					pp >>= 1;
				}
				u[j][k] = i;
			}
		}
	}
	m = in++;
	for (j = 0; j < MAXBIT; j++, m >>= 1) if (!(m & 1)) break;
	if (j >= MAXBIT) exit(1);
	m = j * MAXDIM;
	for (k = 0; k < ndim; k++) {
		ix[k + 1] ^= v[m + k + 1];
		x[k] = ix[k + 1] * fac;
	}
}
// Tausworthe random bit generator returns a single random bit
bool tausworthe(PyRandomObject *self, unsigned n)   
{
	assert(1 <= n && n <= 32);

	if (self->_seed2 & BIT[n]) {
		self->_seed2 = ((self->_seed2 ^ MASK[n]) << 1) | BIT[1];
		return true;
	}
	else {
		self->_seed2 <<= 1;
		return false;
	}
}

// Tausworthe random bit generator returns a bit vector of length n
void tausworthe(PyRandomObject *self, bool *bitvec, unsigned n)
	// It is guaranteed to cycle through all possible combinations of n bits
	// (except all zeros) before repeating, i.e., cycle has maximal length 2^n-1.
	// Ref: Press, W. H., B. P. Flannery, S. A. Teukolsky and W. T. Vetterling,
	//      Numerical Recipes in C, Cambridge Univ. Press, Cambridge, 1988.
{
	assert(1 <= n && n <= 32);   // length of bit vector

	if (self->_seed2 & BIT[n])
		self->_seed2 = ((self-> _seed2 ^ MASK[n]) << 1) | BIT[1];
	else
		self->_seed2 <<= 1;
	for (unsigned i = 0; i < n; i++) bitvec[i] = self->_seed2 & (BIT[n] >> i);
}
