# Copyright (c) 2020, Manfred Moitzi
# License: MIT License
import pytest
from ezdxf.lldxf.tags import Tags
from ezdxf.lldxf.tagwriter import TagCollector
from ezdxf.proxygraphic import (
    load_proxy_graphic,
    export_proxy_graphic,
    ProxyGraphic,
)
from ezdxf.protocols import SupportsVirtualEntities, query_virtual_entities


def test_load_proxy_graphic():
    binary_data = load_proxy_graphic(Tags.from_text(DATA))
    assert len(binary_data) == 968


def test_export_proxy_graphic():
    tagwriter = TagCollector()
    binary_data = load_proxy_graphic(Tags.from_text(DATA))
    export_proxy_graphic(binary_data, tagwriter)
    s = "".join(tag.dxfstr() for tag in tagwriter.tags)
    assert s == DATA


class TestProxyGraphic:
    @pytest.fixture(scope="class")
    def data(self) -> bytes:
        return load_proxy_graphic(Tags.from_text(DATA))

    @pytest.fixture
    def parser(self, data: bytes) -> ProxyGraphic:
        return ProxyGraphic(data)

    def test_info(self, parser: ProxyGraphic):
        indices = list(parser.info())
        assert len(indices) == 13
        index, size, type_ = indices[0]
        assert (index, size, type_) == (8, 84, "POLYLINE_WITH_NORMALS")

    def test_supports_virtual_entities_protocol(self, parser):
        assert isinstance(parser, SupportsVirtualEntities) is True
        assert len(query_virtual_entities(parser)) > 0

    def test_multi_leader_entities(self):
        # ATTRIBUTE_TRUE_COLOR; size: 12
        # UNKNOWN_TYPE_51; size: 12
        # ATTRIBUTE_MARKER; size: 12
        # UNICODE_TEXT2; size: 200
        # UNKNOWN_TYPE_51; size: 12
        # ATTRIBUTE_LAYER; size: 12
        # ATTRIBUTE_TRUE_COLOR; size: 12
        # ATTRIBUTE_LINETYPE; size: 12
        # ATTRIBUTE_MARKER; size: 12
        # ATTRIBUTE_FILL; size: 12
        # POLYGON; size: 84
        # ATTRIBUTE_MARKER; size: 12
        # POLYLINE; size: 60
        # ATTRIBUTE_MARKER; size: 12
        # POLYLINE; size: 60
        # ATTRIBUTE_TRUE_COLOR; size: 12
        # ATTRIBUTE_LINETYPE; size: 12
        # ATTRIBUTE_LINEWEIGHT; size: 12
        # ATTRIBUTE_MARKER; size: 12
        # ATTRIBUTE_TRUE_COLOR; size: 12
        # ATTRIBUTE_LINETYPE; size: 12
        # ATTRIBUTE_LINEWEIGHT; size: 12
        # UNKNOWN_TYPE_51; size: 12
        parser = ProxyGraphic(load_proxy_graphic(Tags.from_text(MULITILEADER)))
        indices = list(parser.info())
        assert len(indices) == 23
        entities = list(parser.virtual_entities())
        assert len(entities) == 4
        text = entities[0]
        assert text.dxftype() == "TEXT"
        assert text.dxf.text == "W410"
        assert text.dxf.layer == "0"  # no DXF document available
        assert text.dxf.color == 256  # by layer
        assert text.dxf.linetype == "BYLAYER"  # no DXF document available
        assert text.dxf.true_color is None

        hatch = entities[1]  # POLYGON
        assert hatch.dxftype() == "HATCH"
        assert len(hatch.paths[0].vertices) == 3
        assert hatch.dxf.layer == "0"  # no DXF document available
        assert hatch.dxf.color == 256  # by layer
        assert hatch.dxf.linetype == "BYLAYER"  # no DXF document available

        polyline = entities[2]
        assert polyline.is_closed is False
        assert polyline.dxftype() == "POLYLINE"
        assert len(polyline.vertices) == 2
        assert polyline.dxf.layer == "0"  # no DXF document available
        assert polyline.dxf.color == 256  # by layer
        assert polyline.dxf.linetype == "BYLAYER"  # no DXF document available

        polyline = entities[3]
        assert polyline.is_closed is False
        assert polyline.dxftype() == "POLYLINE"
        assert len(polyline.vertices) == 2
        assert polyline.dxf.layer == "0"  # no DXF document available
        assert polyline.dxf.color == 256  # by layer
        assert polyline.dxf.linetype == "BYLAYER"  # no DXF document available

    def test_image_entities(self):
        # UNICODE_TEXT2; size: 204
        # POLYLINE; size: 132
        parser = ProxyGraphic(load_proxy_graphic(Tags.from_text(IMAGE)))
        indices = list(parser.info())
        assert len(indices) == 2

        entities = list(parser.virtual_entities())
        assert len(indices) == 2

        text = entities[0]
        assert text.dxftype() == "TEXT"
        assert text.dxf.text == "AcDbRasterImage"
        assert text.dxf.layer == "0"  # no DXF document available
        assert text.dxf.color == 256  # by layer
        assert text.dxf.linetype == "BYLAYER"  # no DXF document available

        polyline = entities[1]
        assert polyline.is_closed is False
        assert polyline.dxftype() == "POLYLINE"
        assert len(polyline.vertices) == 5
        assert text.dxf.layer == "0"  # no DXF document available
        assert text.dxf.color == 256  # by layer
        assert text.dxf.linetype == "BYLAYER"  # no DXF document available

    def test_circular_arc_entities(self):
        # example "aec_doors_walls.dxf" of issue #497; ACAD_PROXY_ENTITY <59D>
        parser = ProxyGraphic(load_proxy_graphic(Tags.from_text(ARC_AEC_DOOR)))
        indices = list(parser.info())
        # [
        # ...
        # (96, 108, 'POLYGON'),
        # (204, 108, 'POLYGON'),
        # (312, 100, 'CIRCULAR_ARC'),
        # (412, 100, 'CIRCULAR_ARC'),
        # (512, 108, 'POLYGON'),
        # (620, 108, 'POLYGON')
        # ]
        assert len(indices) == 13
        entities = list(parser.virtual_entities())
        assert len(entities) == 6
        # 1. ARC - clockwise oriented
        arc1 = entities[2]
        assert arc1.dxf.extrusion.isclose(
            (0, 0, -1)
        ), "expected inverted extrusion vector"

        assert arc1.dxf.center.isclose(
            (-9333.398890018769, 10110.240054080468, 0.0)  # OCS!
        )
        assert arc1.dxf.start_angle == pytest.approx(90.0003859699524)
        assert arc1.dxf.end_angle == pytest.approx(-179.9996140300476)
        assert arc1.dxf.radius == pytest.approx(850)

        # 2. ARC - regular counter clockwise oriented arc
        arc2 = entities[3]
        assert arc2.dxf.extrusion.isclose(
            (0, 0, 1)
        ), "expected WCS aligned extrusion vector"

        assert arc2.dxf.center.isclose(
            (9333.41034197779, 11810.240054041897, 0.0)  # OCS == WCS
        )
        assert arc2.dxf.start_angle == pytest.approx(-90.0003859699524)
        assert arc2.dxf.end_angle == pytest.approx(-0.0003859699524042526)
        assert arc2.dxf.radius == pytest.approx(850)


DATA = """160
968
310
C80300000D000000540000002000000002000000033E695D8B227240B00D3CF1FB7B5540000000000000000082C85BAC2FDE7240FB1040429FB05740000000000000000000000000000000000000000000000000000000000000F03F5400000020000000020000004AF9442AE7FA60405A2D686189715A4000000000000000
310
00C0DC003571AE5F40043422DDA4515D40000000000000000000000000000000000000000000000000000000000000F03F64000000040000001EA72DF9806A69402CE3B4E7B59D34400000000000000000770FBC9D50855E4000000000000000000000000000000000000000000000F03FB634003D352CE93FB1DDE561C5C1
310
E33F00000000000000000418DC3967E1F83F000000000C0000001200000000000000D0000000260000001F8BC5F8B8B46A40197732241FF06140000000000000000000000000000000000000000000000000000000000000F03F0943D77B25BDEF3F417457E0C451C0BF00000000000000003100370032002C003400320000
310
00000006000000010000000000000000000440000000000000F03F0000000000000000000000000000F03F00000000000000000000000000000000000000000000000000000000000000000000000041007200690061006C00000061007200690061006C002E007400740066000000000000000C00000012000000FF7F0000
310
6400000004000000813C33FBB3606A400278BF21B8F4614000000000000000009AEFA7C64B37034000000000000000000000000000000000000000000000F03F0943D77B25BDEF3F437457E0C451C0BF0000000000000000182D4454FB210940000000000C00000010000000010000000C0000001700000000000000540000
310
0020000000020000001EA72DF9806A69402CE3B4E7B59D344000000000000000001EA72DF9806A69402CE3B4E7B59D3440000000000000000000000000000000000000000000000000000000000000F03F540000002000000002000000B296839B8D1A724001000000F06355400000000000000000B296839B8D1A72400100
310
0000F0635540000000000000000000000000000000000000000000000000000000000000F03F540000002000000002000000632D073753076140FFFFFFFF2F525A400000000000000000632D073753076140FFFFFFFF2F525A40000000000000000000000000000000000000000000000000000000000000F03F5400000020
310
000000020000000960E446A3456F405AF2DBF448AB604000000000000000000960E446A3456F405AF2DBF448AB6040000000000000000000000000000000000000000000000000000000000000F03F
"""

MULITILEADER = """160
640
310
80020000170000000C00000016000000000000C00C00000033000000000000000C00000013000000993A0000C800000026000000EC2335956D6D9440F0AEEB8E7B766840000000000000000000000000000000000000000000000000000000000000F03F000000000000F03F00000000000000000000000000000000570034
310
00310030000000000004000000010000000000000000000840626666666666F23F0000000000000000000000000000F03F0000000000000000000000000000000000000000000000000000000001000000000000000000000072006F006D0061006E0073002E0073006800780000005C00000000000C000000330000000000
310
00000C000000100000001D0000000C00000016000000000000C00C00000012000000FF7F00000C00000013000000010000000C00000014000000010000005400000007000000030000000EA778BBEFF9934072D00E24B4FE694000000000000000005E00F4D6D4EB9340264DF666FD0B6B40000000000000000028C9814D17
310
04944016652748DB316A4000000000000000000C00000013000000891300003C00000006000000020000001B387D8403FF9340C41A1BB647186A400000000000000000FB0DD6A357189440F0AEEB8E7BD6684000000000000000000C00000013000000112700003C0000000600000002000000FB0DD6A357189440F0AEEB8E
310
7BD66840000000000000000080F9275C765D9440F0AEEB8E7BD6684000000000000000000C00000016000000000000C00C00000012000000FF7F00000C00000017000000000000000C000000130000009A3A00000C00000016000000000000C00C00000012000000FF7F00000C00000017000000FFFFFFFF0C000000330000
310
0000000000
"""

IMAGE = """160
344
310
5801000002000000CC0000002600000092B5D7AAA19916C0BF88551BB606F83F000000000000000000000000000000000000000000000000000000000000F03F000000000000F03F00000000000000000000000000000000410063004400620052006100730074006500720049006D0061006700650000000F000000000000
310
000000000000000000000000000000F03F0000000000000000000000000000F03F0000000000000000000000000000000000000000000000000000000000000000000000000000000074007800740000000000000084000000060000000500000092B5D7AAA19916C0BF88551BB606F83F00000000000000008A21ADA701E6
310
01C0BF88551BB606F83F00000000000000008A21ADA701E601C06095C5228F990740000000000000000092B5D7AAA19916C06095C5228F990740000000000000000092B5D7AAA19916C0BF88551BB606F83F0000000000000000
"""

ARC_AEC_DOOR = """160
728
310
D80200000D0000000C00000010000000050000000C0000000E000000000100000C00000012000000FF7F0000100000002200000000000000000000000C00000010000000050000000C00000012000000000000000C0000000E000000010000006C00000007000000040000008705D40EB308C2404F852AD01EBFC340000000
310
0000000000188DCA03B308C240BF872AD01EA6C34000000000000000003988CA03B33AC240E19617BA1EA6C3400000000000000000A800D40EB33AC240719417BA1EBFC34000000000000000006C000000070000000400000043041686B408C2407A322AD01E11C7400000000000000000B27C1F91B408C2400A302AD01E2A
310
C7400000000000000000D3771F91B43AC2402C3F17BA1E2AC740000000000000000064FF1586B43AC2409C4117BA1E11C74000000000000000006400000004000000A800D40EB33AC240719417BA1EBFC34000000000000000000000000000908A4000000000000000000000000000000000010000000000F0BFD0A43C4F34
310
41DC3EADE1FCFFFFFFEF3F0000000000000000182D4454FB21F93F00000000640000000400000064FF1586B43AC2409C4117BA1E11C74000000000000000000000000000908A4000000000000000000000000000000000010000000000F03FD0A43C4F3441DCBEADE1FCFFFFFFEFBF0000000000000000182D4454FB21F93F
310
000000006C0000000700000004000000A800D40EB33AC240719417BA1EBFC3400000000000000000A36A8A18B33AC2404D9217BA1ED5C340000000000000000038418A18B3E3C340EE9276FE1DD5C34000000000000000003DD7D30EB3E3C340139576FE1DBFC34000000000000000006C000000070000000400000064FF15
310
86B43AC2409C4117BA1E11C740000000000000000069955F7CB43AC240C14317BA1EFBC6400000000000000000FE6B5F7CB4E3C340634476FE1DFBC6400000000000000000F9D51586B4E3C3403E4276FE1D11C7400000000000000000
"""

if __name__ == "__main__":
    pytest.main([__file__])
