# -*- coding: utf-8 -*-

import unittest
import datetime
import sys
sys.path.insert(0,'../..')
from shaman.src.helpers.mongo_accessor import dict_roll

class TestDictRoll(unittest.TestCase):
    def test_get_dict_roll_empty(self):
        d = dict()
        field = 'some'
        r = dict_roll(d,field)
        self.assertFalse(field in r)

    def test_get_dict_roll_one_false(self):
        d = {'f':0}
        field = 'some'
        r = dict_roll(d,field)
        self.assertFalse(field in r)
        self.assertTrue('f' in r)

    def test_get_dict_roll_one_false_1(self):
        d = {'some':0}
        field = 'some'
        r = dict_roll(d,field)
        self.assertTrue('some' in r)

    def test_get_dict_roll_one_true_1(self):
        d = {'some':{'some':0}}
        field = 'some'
        r = dict_roll(d,field)
        self.assertTrue('some.some' in r)

    def test_get_dict_roll_one_true_2(self):
        d = {'some':{'some':0,'some2':1}}
        field = 'some'
        r = dict_roll(d,field)
        self.assertTrue('some.some' in r)
        self.assertTrue('some.some2' in r)
        self.assertTrue('some' not in r)

    def test_get_dict_roll_one_true_3(self):
        d = {'some':{'some':0,'some2':{'some3':0}}}
        field = 'some'
        r = dict_roll(d,field)
        self.assertTrue('some.some' in r)
        self.assertTrue('some.some2.some3' in r)
        self.assertTrue('some' not in r)
        self.assertTrue('some.some2' not in r)

    def test_get_dict_roll_with_number(self):
        d = {'some':{'some1':0,'some2':{5:0,'some4':[1,2,3,4]}}}
        field = 'some'
        r = dict_roll(d,field)
        self.assertTrue('some.some1' in r)
        self.assertTrue('some.some2.5' in r)
        self.assertTrue('some.some2.some4' in r)
        self.assertTrue('some' not in r)
        self.assertTrue('some.some2' not in r)

    def test_get_dict_roll_with_ignore_set(self):
        d = {'some':{'some1':0,'some2':{'some3':0,'some4':[1,2,3,4]}}}
        field = 'some'

        r = dict_roll(d,field,ignore_set = set(['some2',]))

        self.assertTrue('some.some1' in r)
        self.assertTrue('some.some2.some3' in r)
        self.assertTrue('some.some2.some4' in r)
        self.assertTrue('some' not in r)
        self.assertTrue('some.some2' not in r)


    def test_get_dict_roll_one_true_real_case0(self):
        d = {'$set':{'extra': {'models': {
        'artm100': {64: 0.019954024, 5: 0.016493639, 70: 0.01192757, 71: 0.022308135, 72: 0.012004217, 10: 0.017928431,
                    78: 0.021609137, 80: 0.018059865, 19: 0.018677102, 84: 0.015546973, 21: 0.12822817, 22: 0.011010095,
                    23: 0.010688408, 88: 0.035388242, 25: 0.022487756, 95: 0.010216089, 32: 0.015358775,
                    97: 0.010981974, 99: 0.26748708, 39: 0.0162556, 40: 0.030478181, 42: 0.012391727, 44: 0.015909484,
                    51: 0.012709632, 56: 0.019558603, 57: 0.014174114, 59: 0.010830575, 85: 0.013902265}}},
        'modified_at': datetime.datetime(2017, 1, 9, 17, 51, 2, 131082), '_id': 'oplatye.ru',
        'lemmatized_text': ['платье', 'платье', 'главный', 'rss', 'email', 'вход', 'главный', 'платье', 'размер',
                            'шрифт', 'новость', 'предновогодний', 'наряд', 'леди', 'гага', 'платье', 'новогодний',
                            'елка', 'декабрь', 'известный', 'свой', 'неожиданный', 'наряд', 'выступление', 'леди',
                            'гага', 'решаться', 'новый', 'праздничный', 'имидж', 'именно', 'наряжаться',
                            'предновогодний', 'мини', 'костюмчик', 'платье', 'шорты', 'алый', 'наряд', 'певица', 'свет',
                            'камера', 'сверкать', 'менее', 'ярко', 'водружать', 'макушка', 'зеленый', 'волос',
                            'настоящий', 'новогодний', 'елка', 'оригинальный', 'головной', 'убор', 'вид', 'верхушка',
                            'лесной', 'разноцветный', 'игрушка', 'увенчивать', 'звезда', 'блестка', 'укреплять',
                            'высокий', 'шпиль', 'певица', 'удовольствие', 'демонстрировать', 'свой', 'новый', 'образ',
                            'позировать', 'фотокамера', 'широко', 'раскрывать', 'глаз', 'демонстрировать', 'сверкающий',
                            'белизна', 'зуб', 'свой', 'новое', 'приталивать', 'короткий', 'платье', 'выполнять',
                            'стиль', 'мешок', 'подарок', 'санта', 'клаус', 'певица', 'подбирать', 'излюбленный',
                            'черный', 'ботинок', 'высокий', 'платформа', 'подробно', 'категория', 'пост', 'новость',
                            'блеск', 'звездный', 'платье', 'золотой', 'граммофон', 'декабрь', 'большой', 'концертный',
                            'зал', 'кремлевский', 'дворец', 'вновь', 'состояться', 'ежегодно', 'проводить',
                            'насчитывать', 'восемнадцатый', 'сезон', 'церемония', 'вручение', 'популярный', 'премия',
                            'золотой', 'граммофон', 'исполнитель', 'наиболее', 'популярный', 'полюбиться', 'слушатель',
                            'русский', 'радио', 'песенный', 'хит', 'вручаться', 'памятный', 'золотой', 'статуэтка',
                            'год', 'наиболее', 'яркий', 'признавать', 'песня', 'григорий', 'лепс', 'аня', 'лорак',
                            'николай', 'басков', 'вера', 'брежнева', 'натали', 'группа', 'блестящий', 'тимати',
                            'российский', 'звезда', 'церемония', 'запоминаться', 'также', 'яркий', 'образ', 'который',
                            'представать', 'зритель', 'звезда', 'сцена', 'эффектный', 'смелый', 'самый', 'удачный',
                            'платье', 'костюм', 'прекрасный', 'звездный', 'половина', 'менее', 'звездный',
                            'мужественный', 'половина', 'приковывать', 'внимание', 'пресса', 'подробно', 'категория',
                            'пост', 'новость', 'модный', 'вечерний', 'платье', 'церемония', 'american', 'music',
                            'awards', 'ноябрь', 'торжественный', 'церемония', 'american', 'music', 'awards',
                            'состояться', 'ночь', 'лос', 'анджелес', 'представать', 'многий', 'первый', 'звезда',
                            'превращать', 'грандиозный', 'незабываемый', 'шоу', 'следовать', 'ожидать', 'звезда',
                            'пытаться', 'затмевать', 'друг', 'друг', 'свой', 'блистательный', 'наряд', 'прибывать',
                            'вечер', 'леди', 'гага', 'затмевать', 'весь', 'самый', 'яркий', 'образ', 'восседать',
                            'белый', 'конь', 'небесно', 'голубой', 'платье', 'макси', 'донателло', 'версачий',
                            'эпатажный', 'певица', 'выглядеть', 'очень', 'привлекательно', 'весьма', 'необычно',
                            'настолько', 'новый', 'образ', 'милый', 'принцесса', 'отличаться', 'ранее', 'известный',
                            'красный', 'дорожка', 'среди', 'звезда', 'блистать', 'агилер', 'тейлор', 'свифт', 'хайди',
                            'клум', 'лить', 'мама', 'келли', 'осборн', 'майль', 'сайрус', 'зоуи', 'салдан', 'кэти',
                            'перри', 'многий', 'многий', 'подробно', 'категория', 'пост', 'новость', 'школьный', 'мода',
                            'пря', 'коллекция', 'журнал', 'fashion', 'book', 'июнь', 'последний', 'номер', 'популярный',
                            'глянец', 'fashion', 'book', 'буквально', 'день', 'представлять', 'фотосессия', 'последний',
                            'пря', 'коллекция', 'pre', 'fall', 'автор', 'который', 'являться', 'известный', 'фотограф',
                            'мэтью', 'сезар', 'весь', 'съемка', 'выполнять', 'черно', 'белый', 'стиль', 'посвящаться',
                            'лишь', 'школьный', 'мода', 'задумывать', 'помогать', 'воплощать', 'жизнь', 'фотомодель',
                            'мария', 'локс', 'манон', 'лела', 'коллекция', 'производить', 'глубокий', 'впечатление',
                            'вызывать', 'бурный', 'ажиотаж', 'смелый', 'характер', 'представлять', 'модель', 'гардероб',
                            'вызывать', 'неоднозначный', 'оценка', 'среди', 'родитель', 'представитель',
                            'общественность', 'однако', 'согласно', 'мнение', 'журнал', 'современный', 'ученица',
                            'соответствовать', 'ритм', 'жизнь', 'должный', 'выглядеть', 'именно', 'контрастный', 'цвет',
                            'низ', 'верх', 'одежда', 'крупный', 'обувь', 'белый', 'однотонный', 'блузка', 'присутствие',
                            'мелкий', 'принт', 'горох', 'либо', 'геометрический', 'рисунок', 'кожаный', 'пояс',
                            'довольно', 'короткий', 'юбка', 'белый', 'лосина', 'подробно', 'категория', 'пост',
                            'новость', 'наряд', 'модный', 'бренд', 'каннский', 'фестиваль', 'май', 'завершаться',
                            'каннский', 'фестиваль', 'отзвучать', 'овация', 'проходить', 'хороший', 'премьера',
                            'знатный', 'публика', 'приветствовать', 'популярный', 'актер', 'режиссер', 'вручать',
                            'награда', 'красный', 'дорожка', 'представлять', 'платье', 'модный', 'бренд', 'эксперт',
                            'мода', 'оценивать', 'увидеть', 'наряд', 'составлять', 'свой', 'рейтинг', 'основной',
                            'черта', 'увидеть', 'наряд', 'становиться', 'минимализм', 'пышный', 'форма', 'скрывать',
                            'весьма', 'простой', 'лаконичный', 'фасон', 'однако', 'весь', 'увидеть', 'вариант',
                            'модный', 'лук', 'оказываться', 'удачно', 'выдержанный', 'сочетание', 'дороговизна',
                            'умеренность', 'чжань', 'цзы', 'сыграть', 'фильм', 'час', 'пик', 'великий', 'мастер', 'дом',
                            'летать', 'кинжал', 'представать', 'платье', 'модный', 'дом', 'giambattista', 'valli',
                            'цвет', 'морской', 'волна', 'полупрозрачный', 'ткань', 'пышный', 'юбка', 'усыпать',
                            'крупный', 'цветок', 'контур', 'подробно', 'категория', 'пост', 'новость', 'страница',
                            'десяток', 'выкройка', 'журнал', 'diana', 'moden', 'декабрь', 'выкройка', 'декабрь',
                            'журнал', 'diana', 'moden', 'превращать', 'декабрь', 'сказочный', 'удивительный', 'месяц',
                            'раскрывать', 'читатель', 'волшебный', 'образ', 'бальный', 'наряд', 'сезон', 'актуальный',
                            'легкое', 'веселый', 'цвет', 'аквамарин', 'бирюза', 'аметист', 'воздушность', 'кружевной',
                            'ткань', 'тюль', 'органза', 'газ', 'прекрасно', 'сочетаться', 'блеск', 'атлас', 'шелк',
                            'номер', 'опубликовывать', 'выкройка', 'карнавальный', 'костюм', 'ребенок', 'взрослый',
                            'интересный', 'модель', 'плюшевый', 'зайчик', 'утенок', 'божий', 'коровка', 'отлично',
                            'подходить', 'утренник', 'детский', 'сад', 'ассортимент', 'короткий', 'элегантный', 'наряд',
                            'создавать', 'эффектный', 'образ', 'уверенный', 'женщина', 'корпоративный', 'вечеринка',
                            'сиреневый', 'платье', 'нежный', 'лиф', 'пышный', 'юбка', 'схватывать', 'талия', 'широкий',
                            'атласный', 'пояс', 'позволять', 'почувствовать', 'атмосфера', 'праздник', 'полный', 'мера',
                            'подробно', 'категория', 'пост', 'выкройка', 'diana', 'moden', 'журнал', 'burda', 'декабрь',
                            'выкройка', 'ноябрь', 'главный', 'тематика', 'декабрьский', 'выпуск', 'журнал', 'burda',
                            'являться', 'грядущий', 'семейный', 'праздник', 'новый', 'год', 'лидировать', 'позиция',
                            'выходить', 'наряд', 'изумрудный', 'оттенок', 'тенденция', 'классический', 'вариант',
                            'кроить', 'тонкий', 'нотка', 'романтика', 'капелька', 'эксцентрик', 'ход', 'идти', 'нежный',
                            'мех', 'блестящий', 'пайетка', 'шелк', 'однако', 'важный', 'помнить', 'элегантность',
                            'умеренность', 'который', 'должный', 'наблюдаться', 'сочетать', 'ансамбль', 'нежный',
                            'розовый', 'пальто', 'тюльпан', 'пике', 'комбинация', 'дюшес', 'открывать', 'каталог',
                            'выкройка', 'номер', 'продолжение', 'идти', 'платье', 'шикарный', 'бант', 'шея', 'жакет',
                            'юбка', 'тюльпан', 'пике', 'линия', 'продолжаться', 'белоснежный', 'пальто', 'шерстяной',
                            'сукно', 'нежный', 'блуза', 'крепдешин', 'глубокий', 'вырез', 'грудь', 'подробно',
                            'категория', 'пост', 'выкройка', 'бурда', 'журнал', 'diana', 'moden', 'ноябрь', 'выкройка',
                            'ноябрь', 'ноябрьский', 'выпуск', 'журнал', 'diana', 'moden', 'стирать', 'граница',
                            'классический', 'стиль', 'предлагать', 'читатель', 'использовать', 'жанр', 'mix', 'тренд',
                            'самый', 'разнообразный', 'расцветка', 'сочетание', 'цветок', 'образ', 'множество',
                            'оборка', 'бесконечный', 'складочка', 'яркий', 'вышивка', 'цветной', 'пуговица',
                            'актуальный', 'сезон', 'широкий', 'безразмерный', 'блейзер', 'узкий', 'брючки',
                            'однотонный', 'блузка', 'юбка', 'клетка', 'актуальный', 'ноябрь', 'основной', 'новость',
                            'являться', 'неповторимый', 'модель', 'немецкий', 'дизайнер', 'длинный', 'жакет', 'широкий',
                            'рукав', 'схватывать', 'запястье', 'резинка', 'вышивать', 'пояс', 'веселый', 'цветочный',
                            'узор', 'который', 'вносить', 'тусклый', 'вечер', 'яркий', 'краска', 'подробно',
                            'категория', 'пост', 'выкройка', 'diana', 'moden', 'журнал', 'burda', 'ноябрь', 'выкройка',
                            'ноябрь', 'ноябрьский', 'сезон', 'журнал', 'burda', 'начинаться', 'волшебный', 'комбинация',
                            'несколько', 'стиль', 'ансамбль', 'костюм', 'присутствовать', 'восточный', 'колорит',
                            'броский', 'гламур', 'свобода', 'хиппи', 'лидер', 'выпуск', 'позиционироваться',
                            'удивительный', 'красный', 'цвет', 'всевозможный', 'оттенок', 'окраска', 'широкий', 'рукав',
                            'мелкий', 'волан', 'множество', 'оборка', 'очень', 'кстати', 'данный', 'сезон', 'выкройка',
                            'великолепный', 'серый', 'платье', 'тафта', 'шанжан', 'глубокий', 'вырез', 'сборка',
                            'центр', 'создавать', 'величественный', 'тонкий', 'образ', 'элегантный', 'облегать',
                            'платье', 'черный', 'красный', 'цвет', 'интересный', 'воротник', 'атлас', 'отлично',
                            'дополнять', 'гардероб', 'модный', 'женщина', 'модель', 'потрясать', 'блуза', 'запах',
                            'насыщенный', 'красный', 'цвет', 'внушительный', 'воротник', 'оригинальный', 'пояс',
                            'превращать', 'весь', 'недостаток', 'достоинство', 'подробно', 'категория', 'пост',
                            'выкройка', 'бурда', 'журнал', 'diana', 'moden', 'октябрь', 'выкройка', 'октябрь',
                            'октябрь', 'журнал', 'diana', 'moden', 'предлагать', 'читатель', 'комбинировать', 'весь',
                            'тенденция', 'умеренный', 'пропорция', 'ход', 'идти', 'оборка', 'карман', 'всевозможный',
                            'вставка', 'бант', 'ограниченный', 'количество', 'мягкий', 'теплый', 'тон', 'украшать',
                            'клеточка', 'полоска', 'основной', 'тема', 'сезон', 'сложный', 'воротник', 'многоярусный',
                            'карман', 'единственный', 'украшение', 'модель', 'серия', 'оригинальный', 'шапочка',
                            'выкройка', 'классический', 'брюки', 'являться', 'главный', 'украшение', 'номер',
                            'великолепный', 'ансамбль', 'коричневый', 'жакет', 'рукав', 'фонарик', 'короткий', 'платье',
                            'подол', 'трапеция', 'превращать', 'рабочий', 'будни', 'любой', 'женщина', 'праздник',
                            'бежевый', 'брючки', 'стиль', '80х', 'известный', 'банан', 'придавать', 'образ',
                            'ветреность', 'ущерб', 'деловой', 'репутация', 'подробно', 'категория', 'пост', 'выкройка',
                            'diana', 'moden', 'страница', 'статья', 'школьный', 'платье', 'возвращаться', 'август',
                            'женщина', 'средний', 'поколение', 'помнить', 'строгий', 'коричневый', 'синий', 'школьный',
                            'платье', 'белоснежный', 'воротничок', 'капроновый', 'фартук', 'мама', 'бабушка',
                            'пристально', 'следить', 'воротничок', 'выстирывать', 'обязательно', 'накрахмаливать',
                            'старшеклассница', 'отличница', 'заниматься', 'вещь', 'самый', 'модный', 'считаться',
                            'девушка', 'который', 'менять', 'воротничок', 'каждый', 'день', 'это', 'время', 'деньги',
                            'труд', 'старый', 'добрый', 'школьный', 'платье', 'школьный', 'форма', 'который', 'вводить',
                            'середина', 'впервые', 'развал', 'образовательный', 'система', 'ссср', 'настолько',
                            'идеальный', 'многий', 'девочка', 'надевать', 'новый', 'школьный', 'платье', 'неохота',
                            'несколько', 'год', 'свобода', 'школа', 'приходить', 'свитер', 'джинсы', 'яркий', 'цветной',
                            'кофточка', 'подействовать', 'расслаблять', 'одеваться', 'вновь', 'строгий', 'школьный',
                            'форма', 'никто', 'хотеться', 'подробно', 'категория', 'пост', 'платье', 'школьный',
                            'фартук', 'август', 'время', 'школа', 'девочка', 'носить', 'школьный', 'фартук', 'строгий',
                            'темный', 'платье', 'давно', 'проходить', 'однако', 'фартук', 'продолжать', 'интересовать',
                            'многий', 'родитель', 'часто', 'особый', 'интерес', 'фартук', 'возникать', 'родитель',
                            'школьник', 'выпускник', 'который', 'хотеть', 'надевать', 'подобный', 'школьный', 'форма',
                            'выпускной', 'праздник', 'доказательство', 'это', 'служить', 'множество', 'школьница',
                            'белый', 'передник', 'косичка', 'бант', 'встречаться', 'улица', 'приходить', 'время',
                            'окончание', 'учебный', 'год', 'год', 'ваш', 'семья', 'предстоять', 'важный', 'событие',
                            'дочь', 'хотеть', 'именно', 'наряд', 'позаботиться', 'следовать', 'заранее', 'хотеть',
                            'заказывать', 'индивидуальный', 'пошив', 'школьный', 'фартук', 'смело', 'браться', 'работа',
                            'самостоятельно', 'создавать', 'модель', 'требоваться', 'большой', 'усилие', 'опыт',
                            'наверно', 'многие', 'помнить', 'урок', 'труд', 'школа', 'учить', 'создавать', 'передник',
                            'фартук', 'отличаться', 'слишком', 'сильно', 'подробно', 'категория', 'пост', 'разный',
                            'платье', 'стиль', 'ампир', 'август', 'стиль', 'ампир', 'одежда', 'сегодня', 'становиться',
                            'снова', 'модный', 'наряд', 'придавать', 'женственность', 'блеск', 'пышность', 'история',
                            'стиль', 'ампир', 'одежда', 'стиль', 'ампир', 'искусство', 'одежда', 'зарождаться',
                            'начало', 'xix', 'век', 'считаться', 'законодательница', 'мода', 'стиль', 'ампир',
                            'жозефина', 'первый', 'супруга', 'наполеон', 'основной', 'черта', 'платье', 'стиль',
                            'ампир', 'глубокий', 'вырез', 'лиф', 'завышать', 'линия', 'талия', 'длинный', 'подол',
                            'складка', 'который', 'переходить', 'шлейф', 'рукав', 'фонарик', 'широкий', 'манжета',
                            'ткань', 'платье', 'выбирать', 'шелковый', 'прозрачный', 'тонкий', 'который', 'шиться',
                            'подкладка', 'плотный', 'шелк', 'постепенно', 'отделка', 'платье', 'появляться',
                            'разнообразный', 'орнамент', 'который', 'вышиваться', 'гладь', 'серебряный', 'золотой',
                            'нить', 'спустя', 'некоторый', 'время', 'шлейф', 'исчезать', 'подол', 'становиться',
                            'короче', 'открывать', 'сначала', 'туфля', 'щиколотка', 'декольте', 'становиться', 'мало',
                            'глубокий', 'основный', 'черта', 'женский', 'костюм', 'стиль', 'ампир', 'высокий', 'талия',
                            'узкий', 'лиф', 'длинный', 'прямая', 'юбка', 'подробно', 'категория', 'пост', 'платье',
                            'школьный', 'платье', 'август', 'россия', 'полно', 'оригинальный', 'традиция', 'например',
                            'советский', 'школьный', 'форма', 'церемония', 'последний', 'звонок', 'коричневый',
                            'платье', 'белый', 'воротничок', 'белый', 'фартук', 'огромный', 'бант', 'голова',
                            'выпускница', 'школьный', 'форма', 'шить', 'промышленный', 'объем', 'год', 'сильный',
                            'желание', 'родитель', 'учитель', 'школьник', 'соблюдать', 'традиция', 'добывать',
                            'форменный', 'наряд', 'передавать', 'наследство', 'десятая', 'знакомый', 'шить',
                            'заказывать', 'ателье', 'рассказывать', 'заведующая', 'архангельский', 'ателье',
                            'молодость', 'галина', 'лопасков', 'пошив', 'платьице', 'стоять', 'рубль', 'школьный',
                            'форма', 'действовать', 'скидка', 'говорить', 'платье', 'последний', 'звонок', 'купить',
                            'магазин', 'взрослый', 'интимный', 'товар', 'мужчина', 'костюм', 'школьница',
                            'пользоваться', 'спрос', 'магазин', 'игрушка', 'взрослый', 'подтверждать', 'костюм',
                            'школьница', 'часто', 'бывать', 'продажа', 'накануне', 'праздник', 'последний', 'звонок',
                            'стремительно', 'раскупать', 'подробно', 'категория', 'пост', 'платье', 'выбирать',
                            'свадебный', 'платье', 'август', 'многий', 'девушка', 'долго', 'замужество', 'представлять',
                            'мечта', 'свой', 'свадебный', 'платье', 'задолго', 'начало', 'подготовка', 'праздничный',
                            'торжество', 'необходимо', 'начинать', 'выбирать', 'свадебный', 'платье', 'это', 'делать',
                            'помощь', 'каталог', 'буклет', 'интернет', 'конец', 'конец', 'просто', 'прогуливаться',
                            'свадебный', 'салон', 'магазин', 'настоящий', 'время', 'очень', 'большой', 'выбор',
                            'свадебный', 'платье', 'платье', 'шлейф', 'скромный', 'узкий', 'пышный', 'множество',
                            'юбка', 'длинный', 'короткий', 'белый', 'цветной', 'каждый', 'вид', 'отвечать', 'место',
                            'проведение', 'свадьба', 'стиль', 'просто', 'вкус', 'жених', 'невеста', 'огромный',
                            'количество', 'предлагать', 'продукция', 'выбирать', 'именно', 'подходить', 'начало',
                            'останавливать', 'свой', 'выбор', 'несколько', 'салон', 'который', 'решать', 'посещать',
                            'примерять', 'предлагать', 'платье', 'естественно', 'жить', 'небольшой', 'город', 'пара',
                            'салон', 'обходить', 'весь', 'подробно', 'категория', 'пост', 'свадебный', 'платье',
                            'платье', 'греческий', 'стиль', 'май', 'современный', 'девушка', 'преображаться', 'самый',
                            'настоящий', 'богиня', 'сложный', 'стоить', 'выбирать', 'идеальный', 'платье', 'греческий',
                            'стиль', 'важный', 'выбирать', 'именно', 'фасон', 'который', 'подчеркивать', 'достоинство',
                            'фигура', 'скрывать', 'иметься', 'недостаток', 'складка', 'драпировка', 'греческий',
                            'платье', 'прекрасно', 'добавлять', 'объем', 'это', 'нужно', 'скрывать', 'лишний', 'выбор',
                            'платье', 'греческий', 'стиль', 'обращать', 'внимание', 'цвет', 'смотреться', 'наиболее',
                            'выигрышно', 'совершенно', 'необязательно', 'приобретать', 'именно', 'белый', 'платье',
                            'учет', 'ваш', 'цвет', 'кожа', 'выбирать', 'бежевый', 'розовый', 'голубой', 'прочий',
                            'цвет', 'платье', 'вешалка', 'выглядеть', 'абсолютно', 'иначе', 'тело', 'поэтому',
                            'полениться', 'мерить', 'весь', 'экземпляр', 'оно', 'настоящему', 'оживать', 'фигура',
                            'надевать', 'платье', 'греческий', 'стиль', 'мочь', 'молоденький', 'девушка', 'дама',
                            'солидный', 'возраст', 'оно', 'иметь', 'возрастной', 'граница', 'подробно', 'категория',
                            'пост', 'платье', 'январь', 'февраль', 'март', 'апрель', 'май', 'июнь', 'июль', 'август',
                            'сентябрь', 'октябрь', 'ноябрь', 'декабрь', 'декабрь', 'главный', 'весь', 'пост',
                            'свадебный', 'платье', 'вечерний', 'платье', 'новогодний', 'платье', 'платье', 'выпускной',
                            'разный', 'новость', 'выкройка', 'популярный', 'новость', 'новогодний', 'платье',
                            'маленький', 'модница', 'мода', 'осень', 'зима', 'элегантный', 'платье', 'тренд',
                            'соблазнительный', 'коктейльный', 'платье', 'новогодний', 'корпоратив', 'прозрачный',
                            'платье', 'полиэтилен', 'группа', 'фабрика', 'носить', 'короткий', 'кружевной', 'платье',
                            'модель', 'платье', 'полный', 'женщина', 'наталья', 'королева', 'вновь', 'изменять',
                            'имидж', 'новогодний', 'платье', 'маленький', 'модница', 'свадебный', 'платье', 'стиль',
                            'фасон', 'тенденция', 'мода', 'мода', 'зимний', 'сезон', 'монохромный', 'черно', 'белый',
                            'платье', 'мода', 'век', 'платье', 'вторсырье', 'мода', 'офисный', 'платье', 'pre', 'fall',
                            'narciso', 'rodrigue', 'наряд', 'модный', 'бренд', 'каннский', 'фестиваль', 'ошибка',
                            'выбор', 'платье', 'российский', 'звезда', 'розовый', 'главный', 'тренд', 'свадебный',
                            'платье', 'блеск', 'звездный', 'платье', 'золотой', 'граммофон', 'школьный', 'мода', 'пря',
                            'коллекция', 'журнал', 'fashion', 'book', 'золотой', 'платье', 'честь', 'мороженое', 'млн',
                            'доллар', 'предновогодний', 'наряд', 'леди', 'гага', 'платье', 'новогодний', 'елка',
                            'ажурный', 'платье', 'писк', 'мода', 'архив', 'хороший', 'коктейльный', 'платье',
                            'открывать', 'спина', 'новогодний', 'платье', 'беременный', 'кружевной', 'платье', 'хит',
                            'мода', 'показ', 'свадебный', 'платье', 'цвет', 'фасон', 'новогодний', 'платье', 'платье',
                            'шоколад', 'изысканный', 'гармония', 'звездный', 'битва', 'платье', 'неделя',
                            'мусульманский', 'мода', 'казань', 'мода', 'платье', 'новый', 'год', 'свадебный', 'платье',
                            'звездный', 'свадьба', 'год', 'искушение', 'вечерний', 'платье', 'весна', 'лето', 'zac',
                            'posen', 'новогодний', 'коллекция', 'платье', 'jovani', 'fashion', 'delicatesses',
                            'трикотажный', 'платье', 'коллекция', 'платье', 'птица', 'парк', 'джеймс', 'платье',
                            'пиппа', 'миддлтон', 'купить', 'интернет', 'корректировать', 'нижний', 'белье',
                            'коктейльный', 'платье', 'коллекция', 'вечерний', 'платье', 'кира', 'пластинин', 'украсть',
                            'коллекция', 'платье', 'марка', 'джейкобс', 'модный', 'свадебный', 'платье', 'осень',
                            'зима', 'модный', 'платье', 'осенне', 'зимний', 'сезон', 'помощь', 'раздел', 'карта',
                            'раздел', 'анонс', 'статья', 'поиск', 'раздел', 'статистика', 'раздел', 'подписка', 'rss',
                            'rss', 'раздел', 'комментарий', 'rss', 'раздел', 'коммент', 'платье', 'использование',
                            'материал', 'сайт', 'активный', 'гиперссылка', 'обязательный']}}
        field = 'extra'
        if '$set' in d:
            d['$set'] = dict_roll(d['$set'], 'extra')  # rolling out dict of dicts as extra.key1.key2:value, ... so on

        self.assertTrue('extra.models.artm100.44' in d['$set'])


    def test_get_dict_roll_one_true_real_case_with_ignore_set(self):
        d = {'$set': {'extra': {'models': {
            'artm100': {64: 0.019954024, 5: 0.016493639, 70: 0.01192757, 71: 0.022308135, 72: 0.012004217,
                        10: 0.017928431,
                        78: 0.021609137, 80: 0.018059865, 19: 0.018677102, 84: 0.015546973, 21: 0.12822817,
                        22: 0.011010095,
                        23: 0.010688408, 88: 0.035388242, 25: 0.022487756, 95: 0.010216089, 32: 0.015358775,
                        97: 0.010981974, 99: 0.26748708, 39: 0.0162556, 40: 0.030478181, 42: 0.012391727,
                        44: 0.015909484,
                        51: 0.012709632, 56: 0.019558603, 57: 0.014174114, 59: 0.010830575, 85: 0.013902265},
            'artm300': {64: 0.019954024, 5: 0.016493639, 70: 0.01192757, 71: 0.022308135, 72: 0.012004217,
                        10: 0.017928431,
                        78: 0.021609137, 80: 0.018059865, 19: 0.018677102, 84: 0.015546973, 21: 0.12822817,
                        22: 0.011010095,
                        23: 0.010688408, 88: 0.035388242, 25: 0.022487756, 95: 0.010216089, 32: 0.015358775,
                        97: 0.010981974, 99: 0.26748708, 39: 0.0162556, 40: 0.030478181, 42: 0.012391727,
                        44: 0.015909484,
                        51: 0.012709632, 56: 0.019558603, 57: 0.014174114, 59: 0.010830575, 85: 0.013902265}
        }},
            'modified_at': datetime.datetime(2017, 1, 9, 17, 51, 2, 131082), '_id': 'oplatye.ru',
            'lemmatized_text': ['платье', 'платье', 'главный', 'rss', 'email', 'вход', 'главный', 'платье',
                                'размер',
                                'шрифт', 'новость', 'предновогодний', 'наряд', 'леди', 'гага', 'платье',
                                'новогодний',
                                'елка', 'декабрь', 'известный', 'свой', 'неожиданный', 'наряд', 'выступление',
                                'леди',
                                'гага', 'решаться', 'новый', 'праздничный', 'имидж', 'именно', 'наряжаться',
                                'предновогодний', 'мини', 'костюмчик', 'платье', 'шорты', 'алый', 'наряд', 'певица',
                                'свет',
                                'камера', 'сверкать', 'менее', 'ярко', 'водружать', 'макушка', 'зеленый', 'волос',
                                'настоящий', 'новогодний', 'елка', 'оригинальный', 'головной', 'убор', 'вид',
                                'верхушка',
                                'лесной', 'разноцветный', 'игрушка', 'увенчивать', 'звезда', 'блестка', 'укреплять',
                                'высокий', 'шпиль', 'певица', 'удовольствие', 'демонстрировать', 'свой', 'новый',
                                'образ',
                                'позировать', 'фотокамера', 'широко', 'раскрывать', 'глаз', 'демонстрировать',
                                'сверкающий',
                                'белизна', 'зуб', 'свой', 'новое', 'приталивать', 'короткий', 'платье', 'выполнять',
                                'стиль', 'мешок', 'подарок', 'санта', 'клаус', 'певица', 'подбирать', 'излюбленный',
                                'черный', 'ботинок', 'высокий', 'платформа', 'подробно', 'категория', 'пост',
                                'новость',
                                'блеск', 'звездный', 'платье', 'золотой', 'граммофон', 'декабрь', 'большой',
                                'концертный',
                                'зал', 'кремлевский', 'дворец', 'вновь', 'состояться', 'ежегодно', 'проводить',
                                'насчитывать', 'восемнадцатый', 'сезон', 'церемония', 'вручение', 'популярный',
                                'премия',
                                'золотой', 'граммофон', 'исполнитель', 'наиболее', 'популярный', 'полюбиться',
                                'слушатель',
                                'русский', 'радио', 'песенный', 'хит', 'вручаться', 'памятный', 'золотой',
                                'статуэтка',
                                'год', 'наиболее', 'яркий', 'признавать', 'песня', 'григорий', 'лепс', 'аня',
                                'лорак',
                                'николай', 'басков', 'вера', 'брежнева', 'натали', 'группа', 'блестящий', 'тимати',
                                'российский', 'звезда', 'церемония', 'запоминаться', 'также', 'яркий', 'образ',
                                'который',
                                'представать', 'зритель', 'звезда', 'сцена', 'эффектный', 'смелый', 'самый',
                                'удачный',
                                'платье', 'костюм', 'прекрасный', 'звездный', 'половина', 'менее', 'звездный',
                                'мужественный', 'половина', 'приковывать', 'внимание', 'пресса', 'подробно',
                                'категория',
                                'пост', 'новость', 'модный', 'вечерний', 'платье', 'церемония', 'american', 'music',
                                'awards', 'ноябрь', 'торжественный', 'церемония', 'american', 'music', 'awards',
                                'состояться', 'ночь', 'лос', 'анджелес', 'представать', 'многий', 'первый',
                                'звезда',
                                'превращать', 'грандиозный', 'незабываемый', 'шоу', 'следовать', 'ожидать',
                                'звезда',
                                'пытаться', 'затмевать', 'друг', 'друг', 'свой', 'блистательный', 'наряд',
                                'прибывать',
                                'вечер', 'леди', 'гага', 'затмевать', 'весь', 'самый', 'яркий', 'образ',
                                'восседать',
                                'белый', 'конь', 'небесно', 'голубой', 'платье', 'макси', 'донателло', 'версачий',
                                'эпатажный', 'певица', 'выглядеть', 'очень', 'привлекательно', 'весьма', 'необычно',
                                'настолько', 'новый', 'образ', 'милый', 'принцесса', 'отличаться', 'ранее',
                                'известный',
                                'красный', 'дорожка', 'среди', 'звезда', 'блистать', 'агилер', 'тейлор', 'свифт',
                                'хайди',
                                'клум', 'лить', 'мама', 'келли', 'осборн', 'майль', 'сайрус', 'зоуи', 'салдан',
                                'кэти',
                                'перри', 'многий', 'многий', 'подробно', 'категория', 'пост', 'новость', 'школьный',
                                'мода',
                                'пря', 'коллекция', 'журнал', 'fashion', 'book', 'июнь', 'последний', 'номер',
                                'популярный',
                                'глянец', 'fashion', 'book', 'буквально', 'день', 'представлять', 'фотосессия',
                                'последний',
                                'пря', 'коллекция', 'pre', 'fall', 'автор', 'который', 'являться', 'известный',
                                'фотограф',
                                'мэтью', 'сезар', 'весь', 'съемка', 'выполнять', 'черно', 'белый', 'стиль',
                                'посвящаться',
                                'лишь', 'школьный', 'мода', 'задумывать', 'помогать', 'воплощать', 'жизнь',
                                'фотомодель',
                                'мария', 'локс', 'манон', 'лела', 'коллекция', 'производить', 'глубокий',
                                'впечатление',
                                'вызывать', 'бурный', 'ажиотаж', 'смелый', 'характер', 'представлять', 'модель',
                                'гардероб',
                                'вызывать', 'неоднозначный', 'оценка', 'среди', 'родитель', 'представитель',
                                'общественность', 'однако', 'согласно', 'мнение', 'журнал', 'современный',
                                'ученица',
                                'соответствовать', 'ритм', 'жизнь', 'должный', 'выглядеть', 'именно', 'контрастный',
                                'цвет',
                                'низ', 'верх', 'одежда', 'крупный', 'обувь', 'белый', 'однотонный', 'блузка',
                                'присутствие',
                                'мелкий', 'принт', 'горох', 'либо', 'геометрический', 'рисунок', 'кожаный', 'пояс',
                                'довольно', 'короткий', 'юбка', 'белый', 'лосина', 'подробно', 'категория', 'пост',
                                'новость', 'наряд', 'модный', 'бренд', 'каннский', 'фестиваль', 'май',
                                'завершаться',
                                'каннский', 'фестиваль', 'отзвучать', 'овация', 'проходить', 'хороший', 'премьера',
                                'знатный', 'публика', 'приветствовать', 'популярный', 'актер', 'режиссер',
                                'вручать',
                                'награда', 'красный', 'дорожка', 'представлять', 'платье', 'модный', 'бренд',
                                'эксперт',
                                'мода', 'оценивать', 'увидеть', 'наряд', 'составлять', 'свой', 'рейтинг',
                                'основной',
                                'черта', 'увидеть', 'наряд', 'становиться', 'минимализм', 'пышный', 'форма',
                                'скрывать',
                                'весьма', 'простой', 'лаконичный', 'фасон', 'однако', 'весь', 'увидеть', 'вариант',
                                'модный', 'лук', 'оказываться', 'удачно', 'выдержанный', 'сочетание', 'дороговизна',
                                'умеренность', 'чжань', 'цзы', 'сыграть', 'фильм', 'час', 'пик', 'великий',
                                'мастер', 'дом',
                                'летать', 'кинжал', 'представать', 'платье', 'модный', 'дом', 'giambattista',
                                'valli',
                                'цвет', 'морской', 'волна', 'полупрозрачный', 'ткань', 'пышный', 'юбка', 'усыпать',
                                'крупный', 'цветок', 'контур', 'подробно', 'категория', 'пост', 'новость',
                                'страница',
                                'десяток', 'выкройка', 'журнал', 'diana', 'moden', 'декабрь', 'выкройка', 'декабрь',
                                'журнал', 'diana', 'moden', 'превращать', 'декабрь', 'сказочный', 'удивительный',
                                'месяц',
                                'раскрывать', 'читатель', 'волшебный', 'образ', 'бальный', 'наряд', 'сезон',
                                'актуальный',
                                'легкое', 'веселый', 'цвет', 'аквамарин', 'бирюза', 'аметист', 'воздушность',
                                'кружевной',
                                'ткань', 'тюль', 'органза', 'газ', 'прекрасно', 'сочетаться', 'блеск', 'атлас',
                                'шелк',
                                'номер', 'опубликовывать', 'выкройка', 'карнавальный', 'костюм', 'ребенок',
                                'взрослый',
                                'интересный', 'модель', 'плюшевый', 'зайчик', 'утенок', 'божий', 'коровка',
                                'отлично',
                                'подходить', 'утренник', 'детский', 'сад', 'ассортимент', 'короткий', 'элегантный',
                                'наряд',
                                'создавать', 'эффектный', 'образ', 'уверенный', 'женщина', 'корпоративный',
                                'вечеринка',
                                'сиреневый', 'платье', 'нежный', 'лиф', 'пышный', 'юбка', 'схватывать', 'талия',
                                'широкий',
                                'атласный', 'пояс', 'позволять', 'почувствовать', 'атмосфера', 'праздник', 'полный',
                                'мера',
                                'подробно', 'категория', 'пост', 'выкройка', 'diana', 'moden', 'журнал', 'burda',
                                'декабрь',
                                'выкройка', 'ноябрь', 'главный', 'тематика', 'декабрьский', 'выпуск', 'журнал',
                                'burda',
                                'являться', 'грядущий', 'семейный', 'праздник', 'новый', 'год', 'лидировать',
                                'позиция',
                                'выходить', 'наряд', 'изумрудный', 'оттенок', 'тенденция', 'классический',
                                'вариант',
                                'кроить', 'тонкий', 'нотка', 'романтика', 'капелька', 'эксцентрик', 'ход', 'идти',
                                'нежный',
                                'мех', 'блестящий', 'пайетка', 'шелк', 'однако', 'важный', 'помнить',
                                'элегантность',
                                'умеренность', 'который', 'должный', 'наблюдаться', 'сочетать', 'ансамбль',
                                'нежный',
                                'розовый', 'пальто', 'тюльпан', 'пике', 'комбинация', 'дюшес', 'открывать',
                                'каталог',
                                'выкройка', 'номер', 'продолжение', 'идти', 'платье', 'шикарный', 'бант', 'шея',
                                'жакет',
                                'юбка', 'тюльпан', 'пике', 'линия', 'продолжаться', 'белоснежный', 'пальто',
                                'шерстяной',
                                'сукно', 'нежный', 'блуза', 'крепдешин', 'глубокий', 'вырез', 'грудь', 'подробно',
                                'категория', 'пост', 'выкройка', 'бурда', 'журнал', 'diana', 'moden', 'ноябрь',
                                'выкройка',
                                'ноябрь', 'ноябрьский', 'выпуск', 'журнал', 'diana', 'moden', 'стирать', 'граница',
                                'классический', 'стиль', 'предлагать', 'читатель', 'использовать', 'жанр', 'mix',
                                'тренд',
                                'самый', 'разнообразный', 'расцветка', 'сочетание', 'цветок', 'образ', 'множество',
                                'оборка', 'бесконечный', 'складочка', 'яркий', 'вышивка', 'цветной', 'пуговица',
                                'актуальный', 'сезон', 'широкий', 'безразмерный', 'блейзер', 'узкий', 'брючки',
                                'однотонный', 'блузка', 'юбка', 'клетка', 'актуальный', 'ноябрь', 'основной',
                                'новость',
                                'являться', 'неповторимый', 'модель', 'немецкий', 'дизайнер', 'длинный', 'жакет',
                                'широкий',
                                'рукав', 'схватывать', 'запястье', 'резинка', 'вышивать', 'пояс', 'веселый',
                                'цветочный',
                                'узор', 'который', 'вносить', 'тусклый', 'вечер', 'яркий', 'краска', 'подробно',
                                'категория', 'пост', 'выкройка', 'diana', 'moden', 'журнал', 'burda', 'ноябрь',
                                'выкройка',
                                'ноябрь', 'ноябрьский', 'сезон', 'журнал', 'burda', 'начинаться', 'волшебный',
                                'комбинация',
                                'несколько', 'стиль', 'ансамбль', 'костюм', 'присутствовать', 'восточный',
                                'колорит',
                                'броский', 'гламур', 'свобода', 'хиппи', 'лидер', 'выпуск', 'позиционироваться',
                                'удивительный', 'красный', 'цвет', 'всевозможный', 'оттенок', 'окраска', 'широкий',
                                'рукав',
                                'мелкий', 'волан', 'множество', 'оборка', 'очень', 'кстати', 'данный', 'сезон',
                                'выкройка',
                                'великолепный', 'серый', 'платье', 'тафта', 'шанжан', 'глубокий', 'вырез', 'сборка',
                                'центр', 'создавать', 'величественный', 'тонкий', 'образ', 'элегантный', 'облегать',
                                'платье', 'черный', 'красный', 'цвет', 'интересный', 'воротник', 'атлас', 'отлично',
                                'дополнять', 'гардероб', 'модный', 'женщина', 'модель', 'потрясать', 'блуза',
                                'запах',
                                'насыщенный', 'красный', 'цвет', 'внушительный', 'воротник', 'оригинальный', 'пояс',
                                'превращать', 'весь', 'недостаток', 'достоинство', 'подробно', 'категория', 'пост',
                                'выкройка', 'бурда', 'журнал', 'diana', 'moden', 'октябрь', 'выкройка', 'октябрь',
                                'октябрь', 'журнал', 'diana', 'moden', 'предлагать', 'читатель', 'комбинировать',
                                'весь',
                                'тенденция', 'умеренный', 'пропорция', 'ход', 'идти', 'оборка', 'карман',
                                'всевозможный',
                                'вставка', 'бант', 'ограниченный', 'количество', 'мягкий', 'теплый', 'тон',
                                'украшать',
                                'клеточка', 'полоска', 'основной', 'тема', 'сезон', 'сложный', 'воротник',
                                'многоярусный',
                                'карман', 'единственный', 'украшение', 'модель', 'серия', 'оригинальный', 'шапочка',
                                'выкройка', 'классический', 'брюки', 'являться', 'главный', 'украшение', 'номер',
                                'великолепный', 'ансамбль', 'коричневый', 'жакет', 'рукав', 'фонарик', 'короткий',
                                'платье',
                                'подол', 'трапеция', 'превращать', 'рабочий', 'будни', 'любой', 'женщина',
                                'праздник',
                                'бежевый', 'брючки', 'стиль', '80х', 'известный', 'банан', 'придавать', 'образ',
                                'ветреность', 'ущерб', 'деловой', 'репутация', 'подробно', 'категория', 'пост',
                                'выкройка',
                                'diana', 'moden', 'страница', 'статья', 'школьный', 'платье', 'возвращаться',
                                'август',
                                'женщина', 'средний', 'поколение', 'помнить', 'строгий', 'коричневый', 'синий',
                                'школьный',
                                'платье', 'белоснежный', 'воротничок', 'капроновый', 'фартук', 'мама', 'бабушка',
                                'пристально', 'следить', 'воротничок', 'выстирывать', 'обязательно',
                                'накрахмаливать',
                                'старшеклассница', 'отличница', 'заниматься', 'вещь', 'самый', 'модный',
                                'считаться',
                                'девушка', 'который', 'менять', 'воротничок', 'каждый', 'день', 'это', 'время',
                                'деньги',
                                'труд', 'старый', 'добрый', 'школьный', 'платье', 'школьный', 'форма', 'который',
                                'вводить',
                                'середина', 'впервые', 'развал', 'образовательный', 'система', 'ссср', 'настолько',
                                'идеальный', 'многий', 'девочка', 'надевать', 'новый', 'школьный', 'платье',
                                'неохота',
                                'несколько', 'год', 'свобода', 'школа', 'приходить', 'свитер', 'джинсы', 'яркий',
                                'цветной',
                                'кофточка', 'подействовать', 'расслаблять', 'одеваться', 'вновь', 'строгий',
                                'школьный',
                                'форма', 'никто', 'хотеться', 'подробно', 'категория', 'пост', 'платье', 'школьный',
                                'фартук', 'август', 'время', 'школа', 'девочка', 'носить', 'школьный', 'фартук',
                                'строгий',
                                'темный', 'платье', 'давно', 'проходить', 'однако', 'фартук', 'продолжать',
                                'интересовать',
                                'многий', 'родитель', 'часто', 'особый', 'интерес', 'фартук', 'возникать',
                                'родитель',
                                'школьник', 'выпускник', 'который', 'хотеть', 'надевать', 'подобный', 'школьный',
                                'форма',
                                'выпускной', 'праздник', 'доказательство', 'это', 'служить', 'множество',
                                'школьница',
                                'белый', 'передник', 'косичка', 'бант', 'встречаться', 'улица', 'приходить',
                                'время',
                                'окончание', 'учебный', 'год', 'год', 'ваш', 'семья', 'предстоять', 'важный',
                                'событие',
                                'дочь', 'хотеть', 'именно', 'наряд', 'позаботиться', 'следовать', 'заранее',
                                'хотеть',
                                'заказывать', 'индивидуальный', 'пошив', 'школьный', 'фартук', 'смело', 'браться',
                                'работа',
                                'самостоятельно', 'создавать', 'модель', 'требоваться', 'большой', 'усилие', 'опыт',
                                'наверно', 'многие', 'помнить', 'урок', 'труд', 'школа', 'учить', 'создавать',
                                'передник',
                                'фартук', 'отличаться', 'слишком', 'сильно', 'подробно', 'категория', 'пост',
                                'разный',
                                'платье', 'стиль', 'ампир', 'август', 'стиль', 'ампир', 'одежда', 'сегодня',
                                'становиться',
                                'снова', 'модный', 'наряд', 'придавать', 'женственность', 'блеск', 'пышность',
                                'история',
                                'стиль', 'ампир', 'одежда', 'стиль', 'ампир', 'искусство', 'одежда', 'зарождаться',
                                'начало', 'xix', 'век', 'считаться', 'законодательница', 'мода', 'стиль', 'ампир',
                                'жозефина', 'первый', 'супруга', 'наполеон', 'основной', 'черта', 'платье', 'стиль',
                                'ампир', 'глубокий', 'вырез', 'лиф', 'завышать', 'линия', 'талия', 'длинный',
                                'подол',
                                'складка', 'который', 'переходить', 'шлейф', 'рукав', 'фонарик', 'широкий',
                                'манжета',
                                'ткань', 'платье', 'выбирать', 'шелковый', 'прозрачный', 'тонкий', 'который',
                                'шиться',
                                'подкладка', 'плотный', 'шелк', 'постепенно', 'отделка', 'платье', 'появляться',
                                'разнообразный', 'орнамент', 'который', 'вышиваться', 'гладь', 'серебряный',
                                'золотой',
                                'нить', 'спустя', 'некоторый', 'время', 'шлейф', 'исчезать', 'подол', 'становиться',
                                'короче', 'открывать', 'сначала', 'туфля', 'щиколотка', 'декольте', 'становиться',
                                'мало',
                                'глубокий', 'основный', 'черта', 'женский', 'костюм', 'стиль', 'ампир', 'высокий',
                                'талия',
                                'узкий', 'лиф', 'длинный', 'прямая', 'юбка', 'подробно', 'категория', 'пост',
                                'платье',
                                'школьный', 'платье', 'август', 'россия', 'полно', 'оригинальный', 'традиция',
                                'например',
                                'советский', 'школьный', 'форма', 'церемония', 'последний', 'звонок', 'коричневый',
                                'платье', 'белый', 'воротничок', 'белый', 'фартук', 'огромный', 'бант', 'голова',
                                'выпускница', 'школьный', 'форма', 'шить', 'промышленный', 'объем', 'год',
                                'сильный',
                                'желание', 'родитель', 'учитель', 'школьник', 'соблюдать', 'традиция', 'добывать',
                                'форменный', 'наряд', 'передавать', 'наследство', 'десятая', 'знакомый', 'шить',
                                'заказывать', 'ателье', 'рассказывать', 'заведующая', 'архангельский', 'ателье',
                                'молодость', 'галина', 'лопасков', 'пошив', 'платьице', 'стоять', 'рубль',
                                'школьный',
                                'форма', 'действовать', 'скидка', 'говорить', 'платье', 'последний', 'звонок',
                                'купить',
                                'магазин', 'взрослый', 'интимный', 'товар', 'мужчина', 'костюм', 'школьница',
                                'пользоваться', 'спрос', 'магазин', 'игрушка', 'взрослый', 'подтверждать', 'костюм',
                                'школьница', 'часто', 'бывать', 'продажа', 'накануне', 'праздник', 'последний',
                                'звонок',
                                'стремительно', 'раскупать', 'подробно', 'категория', 'пост', 'платье', 'выбирать',
                                'свадебный', 'платье', 'август', 'многий', 'девушка', 'долго', 'замужество',
                                'представлять',
                                'мечта', 'свой', 'свадебный', 'платье', 'задолго', 'начало', 'подготовка',
                                'праздничный',
                                'торжество', 'необходимо', 'начинать', 'выбирать', 'свадебный', 'платье', 'это',
                                'делать',
                                'помощь', 'каталог', 'буклет', 'интернет', 'конец', 'конец', 'просто',
                                'прогуливаться',
                                'свадебный', 'салон', 'магазин', 'настоящий', 'время', 'очень', 'большой', 'выбор',
                                'свадебный', 'платье', 'платье', 'шлейф', 'скромный', 'узкий', 'пышный',
                                'множество',
                                'юбка', 'длинный', 'короткий', 'белый', 'цветной', 'каждый', 'вид', 'отвечать',
                                'место',
                                'проведение', 'свадьба', 'стиль', 'просто', 'вкус', 'жених', 'невеста', 'огромный',
                                'количество', 'предлагать', 'продукция', 'выбирать', 'именно', 'подходить',
                                'начало',
                                'останавливать', 'свой', 'выбор', 'несколько', 'салон', 'который', 'решать',
                                'посещать',
                                'примерять', 'предлагать', 'платье', 'естественно', 'жить', 'небольшой', 'город',
                                'пара',
                                'салон', 'обходить', 'весь', 'подробно', 'категория', 'пост', 'свадебный', 'платье',
                                'платье', 'греческий', 'стиль', 'май', 'современный', 'девушка', 'преображаться',
                                'самый',
                                'настоящий', 'богиня', 'сложный', 'стоить', 'выбирать', 'идеальный', 'платье',
                                'греческий',
                                'стиль', 'важный', 'выбирать', 'именно', 'фасон', 'который', 'подчеркивать',
                                'достоинство',
                                'фигура', 'скрывать', 'иметься', 'недостаток', 'складка', 'драпировка', 'греческий',
                                'платье', 'прекрасно', 'добавлять', 'объем', 'это', 'нужно', 'скрывать', 'лишний',
                                'выбор',
                                'платье', 'греческий', 'стиль', 'обращать', 'внимание', 'цвет', 'смотреться',
                                'наиболее',
                                'выигрышно', 'совершенно', 'необязательно', 'приобретать', 'именно', 'белый',
                                'платье',
                                'учет', 'ваш', 'цвет', 'кожа', 'выбирать', 'бежевый', 'розовый', 'голубой',
                                'прочий',
                                'цвет', 'платье', 'вешалка', 'выглядеть', 'абсолютно', 'иначе', 'тело', 'поэтому',
                                'полениться', 'мерить', 'весь', 'экземпляр', 'оно', 'настоящему', 'оживать',
                                'фигура',
                                'надевать', 'платье', 'греческий', 'стиль', 'мочь', 'молоденький', 'девушка',
                                'дама',
                                'солидный', 'возраст', 'оно', 'иметь', 'возрастной', 'граница', 'подробно',
                                'категория',
                                'пост', 'платье', 'январь', 'февраль', 'март', 'апрель', 'май', 'июнь', 'июль',
                                'август',
                                'сентябрь', 'октябрь', 'ноябрь', 'декабрь', 'декабрь', 'главный', 'весь', 'пост',
                                'свадебный', 'платье', 'вечерний', 'платье', 'новогодний', 'платье', 'платье',
                                'выпускной',
                                'разный', 'новость', 'выкройка', 'популярный', 'новость', 'новогодний', 'платье',
                                'маленький', 'модница', 'мода', 'осень', 'зима', 'элегантный', 'платье', 'тренд',
                                'соблазнительный', 'коктейльный', 'платье', 'новогодний', 'корпоратив',
                                'прозрачный',
                                'платье', 'полиэтилен', 'группа', 'фабрика', 'носить', 'короткий', 'кружевной',
                                'платье',
                                'модель', 'платье', 'полный', 'женщина', 'наталья', 'королева', 'вновь', 'изменять',
                                'имидж', 'новогодний', 'платье', 'маленький', 'модница', 'свадебный', 'платье',
                                'стиль',
                                'фасон', 'тенденция', 'мода', 'мода', 'зимний', 'сезон', 'монохромный', 'черно',
                                'белый',
                                'платье', 'мода', 'век', 'платье', 'вторсырье', 'мода', 'офисный', 'платье', 'pre',
                                'fall',
                                'narciso', 'rodrigue', 'наряд', 'модный', 'бренд', 'каннский', 'фестиваль',
                                'ошибка',
                                'выбор', 'платье', 'российский', 'звезда', 'розовый', 'главный', 'тренд',
                                'свадебный',
                                'платье', 'блеск', 'звездный', 'платье', 'золотой', 'граммофон', 'школьный', 'мода',
                                'пря',
                                'коллекция', 'журнал', 'fashion', 'book', 'золотой', 'платье', 'честь', 'мороженое',
                                'млн',
                                'доллар', 'предновогодний', 'наряд', 'леди', 'гага', 'платье', 'новогодний', 'елка',
                                'ажурный', 'платье', 'писк', 'мода', 'архив', 'хороший', 'коктейльный', 'платье',
                                'открывать', 'спина', 'новогодний', 'платье', 'беременный', 'кружевной', 'платье',
                                'хит',
                                'мода', 'показ', 'свадебный', 'платье', 'цвет', 'фасон', 'новогодний', 'платье',
                                'платье',
                                'шоколад', 'изысканный', 'гармония', 'звездный', 'битва', 'платье', 'неделя',
                                'мусульманский', 'мода', 'казань', 'мода', 'платье', 'новый', 'год', 'свадебный',
                                'платье',
                                'звездный', 'свадьба', 'год', 'искушение', 'вечерний', 'платье', 'весна', 'лето',
                                'zac',
                                'posen', 'новогодний', 'коллекция', 'платье', 'jovani', 'fashion', 'delicatesses',
                                'трикотажный', 'платье', 'коллекция', 'платье', 'птица', 'парк', 'джеймс', 'платье',
                                'пиппа', 'миддлтон', 'купить', 'интернет', 'корректировать', 'нижний', 'белье',
                                'коктейльный', 'платье', 'коллекция', 'вечерний', 'платье', 'кира', 'пластинин',
                                'украсть',
                                'коллекция', 'платье', 'марка', 'джейкобс', 'модный', 'свадебный', 'платье',
                                'осень',
                                'зима', 'модный', 'платье', 'осенне', 'зимний', 'сезон', 'помощь', 'раздел',
                                'карта',
                                'раздел', 'анонс', 'статья', 'поиск', 'раздел', 'статистика', 'раздел', 'подписка',
                                'rss',
                                'rss', 'раздел', 'комментарий', 'rss', 'раздел', 'коммент', 'платье',
                                'использование',
                                'материал', 'сайт', 'активный', 'гиперссылка', 'обязательный']}}
        field = 'extra'
        if '$set' in d:
            d['$set'] = dict_roll(d['$set'],
                                  'extra',
                                  ignore_set=set(['artm100',]))  # rolling out dict of dicts as extra.key1.key2:value, ... so on

        self.assertTrue('extra.models.artm300.44' in d['$set'])
        self.assertTrue('extra.models.artm100' in d['$set'])

    def test_get_dict_roll_one_true_real_case_with_ignore_set_2(self):
        d = {'$set': {'extra': {'models': {
            'artm100': {64: 0.019954024, 5: 0.016493639, 70: 0.01192757, 71: 0.022308135, 72: 0.012004217,
                        10: 0.017928431,
                        78: 0.021609137, 80: 0.018059865, 19: 0.018677102, 84: 0.015546973, 21: 0.12822817,
                        22: 0.011010095,
                        23: 0.010688408, 88: 0.035388242, 25: 0.022487756, 95: 0.010216089, 32: 0.015358775,
                        97: 0.010981974, 99: 0.26748708, 39: 0.0162556, 40: 0.030478181, 42: 0.012391727,
                        44: 0.015909484,
                        51: 0.012709632, 56: 0.019558603, 57: 0.014174114, 59: 0.010830575, 85: 0.013902265}}},
            'modified_at': datetime.datetime(2017, 1, 9, 17, 51, 2, 131082), '_id': 'oplatye.ru',
            'lemmatized_text': ['платье', 'платье', 'главный', 'rss', 'email', 'вход', 'главный', 'платье',
                                'размер',
                                'шрифт', 'новость', 'предновогодний', 'наряд', 'леди', 'гага', 'платье',
                                'новогодний',
                                'елка', 'декабрь', 'известный', 'свой', 'неожиданный', 'наряд', 'выступление',
                                'леди',
                                'гага', 'решаться', 'новый', 'праздничный', 'имидж', 'именно', 'наряжаться',
                                'предновогодний', 'мини', 'костюмчик', 'платье', 'шорты', 'алый', 'наряд', 'певица',
                                'свет',
                                'камера', 'сверкать', 'менее', 'ярко', 'водружать', 'макушка', 'зеленый', 'волос',
                                'настоящий', 'новогодний', 'елка', 'оригинальный', 'головной', 'убор', 'вид',
                                'верхушка',
                                'лесной', 'разноцветный', 'игрушка', 'увенчивать', 'звезда', 'блестка', 'укреплять',
                                'высокий', 'шпиль', 'певица', 'удовольствие', 'демонстрировать', 'свой', 'новый',
                                'образ',
                                'позировать', 'фотокамера', 'широко', 'раскрывать', 'глаз', 'демонстрировать',
                                'сверкающий',
                                'белизна', 'зуб', 'свой', 'новое', 'приталивать', 'короткий', 'платье', 'выполнять',
                                'стиль', 'мешок', 'подарок', 'санта', 'клаус', 'певица', 'подбирать', 'излюбленный',
                                'черный', 'ботинок', 'высокий', 'платформа', 'подробно', 'категория', 'пост',
                                'новость',
                                'блеск', 'звездный', 'платье', 'золотой', 'граммофон', 'декабрь', 'большой',
                                'концертный',
                                'зал', 'кремлевский', 'дворец', 'вновь', 'состояться', 'ежегодно', 'проводить',
                                'насчитывать', 'восемнадцатый', 'сезон', 'церемония', 'вручение', 'популярный',
                                'премия',
                                'золотой', 'граммофон', 'исполнитель', 'наиболее', 'популярный', 'полюбиться',
                                'слушатель',
                                'русский', 'радио', 'песенный', 'хит', 'вручаться', 'памятный', 'золотой',
                                'статуэтка',
                                'год', 'наиболее', 'яркий', 'признавать', 'песня', 'григорий', 'лепс', 'аня',
                                'лорак',
                                'николай', 'басков', 'вера', 'брежнева', 'натали', 'группа', 'блестящий', 'тимати',
                                'российский', 'звезда', 'церемония', 'запоминаться', 'также', 'яркий', 'образ',
                                'который',
                                'представать', 'зритель', 'звезда', 'сцена', 'эффектный', 'смелый', 'самый',
                                'удачный',
                                'платье', 'костюм', 'прекрасный', 'звездный', 'половина', 'менее', 'звездный',
                                'мужественный', 'половина', 'приковывать', 'внимание', 'пресса', 'подробно',
                                'категория',
                                'пост', 'новость', 'модный', 'вечерний', 'платье', 'церемония', 'american', 'music',
                                'awards', 'ноябрь', 'торжественный', 'церемония', 'american', 'music', 'awards',
                                'состояться', 'ночь', 'лос', 'анджелес', 'представать', 'многий', 'первый',
                                'звезда',
                                'превращать', 'грандиозный', 'незабываемый', 'шоу', 'следовать', 'ожидать',
                                'звезда',
                                'пытаться', 'затмевать', 'друг', 'друг', 'свой', 'блистательный', 'наряд',
                                'прибывать',
                                'вечер', 'леди', 'гага', 'затмевать', 'весь', 'самый', 'яркий', 'образ',
                                'восседать',
                                'белый', 'конь', 'небесно', 'голубой', 'платье', 'макси', 'донателло', 'версачий',
                                'эпатажный', 'певица', 'выглядеть', 'очень', 'привлекательно', 'весьма', 'необычно',
                                'настолько', 'новый', 'образ', 'милый', 'принцесса', 'отличаться', 'ранее',
                                'известный',
                                'красный', 'дорожка', 'среди', 'звезда', 'блистать', 'агилер', 'тейлор', 'свифт',
                                'хайди',
                                'клум', 'лить', 'мама', 'келли', 'осборн', 'майль', 'сайрус', 'зоуи', 'салдан',
                                'кэти',
                                'перри', 'многий', 'многий', 'подробно', 'категория', 'пост', 'новость', 'школьный',
                                'мода',
                                'пря', 'коллекция', 'журнал', 'fashion', 'book', 'июнь', 'последний', 'номер',
                                'популярный',
                                'глянец', 'fashion', 'book', 'буквально', 'день', 'представлять', 'фотосессия',
                                'последний',
                                'пря', 'коллекция', 'pre', 'fall', 'автор', 'который', 'являться', 'известный',
                                'фотограф',
                                'мэтью', 'сезар', 'весь', 'съемка', 'выполнять', 'черно', 'белый', 'стиль',
                                'посвящаться',
                                'лишь', 'школьный', 'мода', 'задумывать', 'помогать', 'воплощать', 'жизнь',
                                'фотомодель',
                                'мария', 'локс', 'манон', 'лела', 'коллекция', 'производить', 'глубокий',
                                'впечатление',
                                'вызывать', 'бурный', 'ажиотаж', 'смелый', 'характер', 'представлять', 'модель',
                                'гардероб',
                                'вызывать', 'неоднозначный', 'оценка', 'среди', 'родитель', 'представитель',
                                'общественность', 'однако', 'согласно', 'мнение', 'журнал', 'современный',
                                'ученица',
                                'соответствовать', 'ритм', 'жизнь', 'должный', 'выглядеть', 'именно', 'контрастный',
                                'цвет',
                                'низ', 'верх', 'одежда', 'крупный', 'обувь', 'белый', 'однотонный', 'блузка',
                                'присутствие',
                                'мелкий', 'принт', 'горох', 'либо', 'геометрический', 'рисунок', 'кожаный', 'пояс',
                                'довольно', 'короткий', 'юбка', 'белый', 'лосина', 'подробно', 'категория', 'пост',
                                'новость', 'наряд', 'модный', 'бренд', 'каннский', 'фестиваль', 'май',
                                'завершаться',
                                'каннский', 'фестиваль', 'отзвучать', 'овация', 'проходить', 'хороший', 'премьера',
                                'знатный', 'публика', 'приветствовать', 'популярный', 'актер', 'режиссер',
                                'вручать',
                                'награда', 'красный', 'дорожка', 'представлять', 'платье', 'модный', 'бренд',
                                'эксперт',
                                'мода', 'оценивать', 'увидеть', 'наряд', 'составлять', 'свой', 'рейтинг',
                                'основной',
                                'черта', 'увидеть', 'наряд', 'становиться', 'минимализм', 'пышный', 'форма',
                                'скрывать',
                                'весьма', 'простой', 'лаконичный', 'фасон', 'однако', 'весь', 'увидеть', 'вариант',
                                'модный', 'лук', 'оказываться', 'удачно', 'выдержанный', 'сочетание', 'дороговизна',
                                'умеренность', 'чжань', 'цзы', 'сыграть', 'фильм', 'час', 'пик', 'великий',
                                'мастер', 'дом',
                                'летать', 'кинжал', 'представать', 'платье', 'модный', 'дом', 'giambattista',
                                'valli',
                                'цвет', 'морской', 'волна', 'полупрозрачный', 'ткань', 'пышный', 'юбка', 'усыпать',
                                'крупный', 'цветок', 'контур', 'подробно', 'категория', 'пост', 'новость',
                                'страница',
                                'десяток', 'выкройка', 'журнал', 'diana', 'moden', 'декабрь', 'выкройка', 'декабрь',
                                'журнал', 'diana', 'moden', 'превращать', 'декабрь', 'сказочный', 'удивительный',
                                'месяц',
                                'раскрывать', 'читатель', 'волшебный', 'образ', 'бальный', 'наряд', 'сезон',
                                'актуальный',
                                'легкое', 'веселый', 'цвет', 'аквамарин', 'бирюза', 'аметист', 'воздушность',
                                'кружевной',
                                'ткань', 'тюль', 'органза', 'газ', 'прекрасно', 'сочетаться', 'блеск', 'атлас',
                                'шелк',
                                'номер', 'опубликовывать', 'выкройка', 'карнавальный', 'костюм', 'ребенок',
                                'взрослый',
                                'интересный', 'модель', 'плюшевый', 'зайчик', 'утенок', 'божий', 'коровка',
                                'отлично',
                                'подходить', 'утренник', 'детский', 'сад', 'ассортимент', 'короткий', 'элегантный',
                                'наряд',
                                'создавать', 'эффектный', 'образ', 'уверенный', 'женщина', 'корпоративный',
                                'вечеринка',
                                'сиреневый', 'платье', 'нежный', 'лиф', 'пышный', 'юбка', 'схватывать', 'талия',
                                'широкий',
                                'атласный', 'пояс', 'позволять', 'почувствовать', 'атмосфера', 'праздник', 'полный',
                                'мера',
                                'подробно', 'категория', 'пост', 'выкройка', 'diana', 'moden', 'журнал', 'burda',
                                'декабрь',
                                'выкройка', 'ноябрь', 'главный', 'тематика', 'декабрьский', 'выпуск', 'журнал',
                                'burda',
                                'являться', 'грядущий', 'семейный', 'праздник', 'новый', 'год', 'лидировать',
                                'позиция',
                                'выходить', 'наряд', 'изумрудный', 'оттенок', 'тенденция', 'классический',
                                'вариант',
                                'кроить', 'тонкий', 'нотка', 'романтика', 'капелька', 'эксцентрик', 'ход', 'идти',
                                'нежный',
                                'мех', 'блестящий', 'пайетка', 'шелк', 'однако', 'важный', 'помнить',
                                'элегантность',
                                'умеренность', 'который', 'должный', 'наблюдаться', 'сочетать', 'ансамбль',
                                'нежный',
                                'розовый', 'пальто', 'тюльпан', 'пике', 'комбинация', 'дюшес', 'открывать',
                                'каталог',
                                'выкройка', 'номер', 'продолжение', 'идти', 'платье', 'шикарный', 'бант', 'шея',
                                'жакет',
                                'юбка', 'тюльпан', 'пике', 'линия', 'продолжаться', 'белоснежный', 'пальто',
                                'шерстяной',
                                'сукно', 'нежный', 'блуза', 'крепдешин', 'глубокий', 'вырез', 'грудь', 'подробно',
                                'категория', 'пост', 'выкройка', 'бурда', 'журнал', 'diana', 'moden', 'ноябрь',
                                'выкройка',
                                'ноябрь', 'ноябрьский', 'выпуск', 'журнал', 'diana', 'moden', 'стирать', 'граница',
                                'классический', 'стиль', 'предлагать', 'читатель', 'использовать', 'жанр', 'mix',
                                'тренд',
                                'самый', 'разнообразный', 'расцветка', 'сочетание', 'цветок', 'образ', 'множество',
                                'оборка', 'бесконечный', 'складочка', 'яркий', 'вышивка', 'цветной', 'пуговица',
                                'актуальный', 'сезон', 'широкий', 'безразмерный', 'блейзер', 'узкий', 'брючки',
                                'однотонный', 'блузка', 'юбка', 'клетка', 'актуальный', 'ноябрь', 'основной',
                                'новость',
                                'являться', 'неповторимый', 'модель', 'немецкий', 'дизайнер', 'длинный', 'жакет',
                                'широкий',
                                'рукав', 'схватывать', 'запястье', 'резинка', 'вышивать', 'пояс', 'веселый',
                                'цветочный',
                                'узор', 'который', 'вносить', 'тусклый', 'вечер', 'яркий', 'краска', 'подробно',
                                'категория', 'пост', 'выкройка', 'diana', 'moden', 'журнал', 'burda', 'ноябрь',
                                'выкройка',
                                'ноябрь', 'ноябрьский', 'сезон', 'журнал', 'burda', 'начинаться', 'волшебный',
                                'комбинация',
                                'несколько', 'стиль', 'ансамбль', 'костюм', 'присутствовать', 'восточный',
                                'колорит',
                                'броский', 'гламур', 'свобода', 'хиппи', 'лидер', 'выпуск', 'позиционироваться',
                                'удивительный', 'красный', 'цвет', 'всевозможный', 'оттенок', 'окраска', 'широкий',
                                'рукав',
                                'мелкий', 'волан', 'множество', 'оборка', 'очень', 'кстати', 'данный', 'сезон',
                                'выкройка',
                                'великолепный', 'серый', 'платье', 'тафта', 'шанжан', 'глубокий', 'вырез', 'сборка',
                                'центр', 'создавать', 'величественный', 'тонкий', 'образ', 'элегантный', 'облегать',
                                'платье', 'черный', 'красный', 'цвет', 'интересный', 'воротник', 'атлас', 'отлично',
                                'дополнять', 'гардероб', 'модный', 'женщина', 'модель', 'потрясать', 'блуза',
                                'запах',
                                'насыщенный', 'красный', 'цвет', 'внушительный', 'воротник', 'оригинальный', 'пояс',
                                'превращать', 'весь', 'недостаток', 'достоинство', 'подробно', 'категория', 'пост',
                                'выкройка', 'бурда', 'журнал', 'diana', 'moden', 'октябрь', 'выкройка', 'октябрь',
                                'октябрь', 'журнал', 'diana', 'moden', 'предлагать', 'читатель', 'комбинировать',
                                'весь',
                                'тенденция', 'умеренный', 'пропорция', 'ход', 'идти', 'оборка', 'карман',
                                'всевозможный',
                                'вставка', 'бант', 'ограниченный', 'количество', 'мягкий', 'теплый', 'тон',
                                'украшать',
                                'клеточка', 'полоска', 'основной', 'тема', 'сезон', 'сложный', 'воротник',
                                'многоярусный',
                                'карман', 'единственный', 'украшение', 'модель', 'серия', 'оригинальный', 'шапочка',
                                'выкройка', 'классический', 'брюки', 'являться', 'главный', 'украшение', 'номер',
                                'великолепный', 'ансамбль', 'коричневый', 'жакет', 'рукав', 'фонарик', 'короткий',
                                'платье',
                                'подол', 'трапеция', 'превращать', 'рабочий', 'будни', 'любой', 'женщина',
                                'праздник',
                                'бежевый', 'брючки', 'стиль', '80х', 'известный', 'банан', 'придавать', 'образ',
                                'ветреность', 'ущерб', 'деловой', 'репутация', 'подробно', 'категория', 'пост',
                                'выкройка',
                                'diana', 'moden', 'страница', 'статья', 'школьный', 'платье', 'возвращаться',
                                'август',
                                'женщина', 'средний', 'поколение', 'помнить', 'строгий', 'коричневый', 'синий',
                                'школьный',
                                'платье', 'белоснежный', 'воротничок', 'капроновый', 'фартук', 'мама', 'бабушка',
                                'пристально', 'следить', 'воротничок', 'выстирывать', 'обязательно',
                                'накрахмаливать',
                                'старшеклассница', 'отличница', 'заниматься', 'вещь', 'самый', 'модный',
                                'считаться',
                                'девушка', 'который', 'менять', 'воротничок', 'каждый', 'день', 'это', 'время',
                                'деньги',
                                'труд', 'старый', 'добрый', 'школьный', 'платье', 'школьный', 'форма', 'который',
                                'вводить',
                                'середина', 'впервые', 'развал', 'образовательный', 'система', 'ссср', 'настолько',
                                'идеальный', 'многий', 'девочка', 'надевать', 'новый', 'школьный', 'платье',
                                'неохота',
                                'несколько', 'год', 'свобода', 'школа', 'приходить', 'свитер', 'джинсы', 'яркий',
                                'цветной',
                                'кофточка', 'подействовать', 'расслаблять', 'одеваться', 'вновь', 'строгий',
                                'школьный',
                                'форма', 'никто', 'хотеться', 'подробно', 'категория', 'пост', 'платье', 'школьный',
                                'фартук', 'август', 'время', 'школа', 'девочка', 'носить', 'школьный', 'фартук',
                                'строгий',
                                'темный', 'платье', 'давно', 'проходить', 'однако', 'фартук', 'продолжать',
                                'интересовать',
                                'многий', 'родитель', 'часто', 'особый', 'интерес', 'фартук', 'возникать',
                                'родитель',
                                'школьник', 'выпускник', 'который', 'хотеть', 'надевать', 'подобный', 'школьный',
                                'форма',
                                'выпускной', 'праздник', 'доказательство', 'это', 'служить', 'множество',
                                'школьница',
                                'белый', 'передник', 'косичка', 'бант', 'встречаться', 'улица', 'приходить',
                                'время',
                                'окончание', 'учебный', 'год', 'год', 'ваш', 'семья', 'предстоять', 'важный',
                                'событие',
                                'дочь', 'хотеть', 'именно', 'наряд', 'позаботиться', 'следовать', 'заранее',
                                'хотеть',
                                'заказывать', 'индивидуальный', 'пошив', 'школьный', 'фартук', 'смело', 'браться',
                                'работа',
                                'самостоятельно', 'создавать', 'модель', 'требоваться', 'большой', 'усилие', 'опыт',
                                'наверно', 'многие', 'помнить', 'урок', 'труд', 'школа', 'учить', 'создавать',
                                'передник',
                                'фартук', 'отличаться', 'слишком', 'сильно', 'подробно', 'категория', 'пост',
                                'разный',
                                'платье', 'стиль', 'ампир', 'август', 'стиль', 'ампир', 'одежда', 'сегодня',
                                'становиться',
                                'снова', 'модный', 'наряд', 'придавать', 'женственность', 'блеск', 'пышность',
                                'история',
                                'стиль', 'ампир', 'одежда', 'стиль', 'ампир', 'искусство', 'одежда', 'зарождаться',
                                'начало', 'xix', 'век', 'считаться', 'законодательница', 'мода', 'стиль', 'ампир',
                                'жозефина', 'первый', 'супруга', 'наполеон', 'основной', 'черта', 'платье', 'стиль',
                                'ампир', 'глубокий', 'вырез', 'лиф', 'завышать', 'линия', 'талия', 'длинный',
                                'подол',
                                'складка', 'который', 'переходить', 'шлейф', 'рукав', 'фонарик', 'широкий',
                                'манжета',
                                'ткань', 'платье', 'выбирать', 'шелковый', 'прозрачный', 'тонкий', 'который',
                                'шиться',
                                'подкладка', 'плотный', 'шелк', 'постепенно', 'отделка', 'платье', 'появляться',
                                'разнообразный', 'орнамент', 'который', 'вышиваться', 'гладь', 'серебряный',
                                'золотой',
                                'нить', 'спустя', 'некоторый', 'время', 'шлейф', 'исчезать', 'подол', 'становиться',
                                'короче', 'открывать', 'сначала', 'туфля', 'щиколотка', 'декольте', 'становиться',
                                'мало',
                                'глубокий', 'основный', 'черта', 'женский', 'костюм', 'стиль', 'ампир', 'высокий',
                                'талия',
                                'узкий', 'лиф', 'длинный', 'прямая', 'юбка', 'подробно', 'категория', 'пост',
                                'платье',
                                'школьный', 'платье', 'август', 'россия', 'полно', 'оригинальный', 'традиция',
                                'например',
                                'советский', 'школьный', 'форма', 'церемония', 'последний', 'звонок', 'коричневый',
                                'платье', 'белый', 'воротничок', 'белый', 'фартук', 'огромный', 'бант', 'голова',
                                'выпускница', 'школьный', 'форма', 'шить', 'промышленный', 'объем', 'год',
                                'сильный',
                                'желание', 'родитель', 'учитель', 'школьник', 'соблюдать', 'традиция', 'добывать',
                                'форменный', 'наряд', 'передавать', 'наследство', 'десятая', 'знакомый', 'шить',
                                'заказывать', 'ателье', 'рассказывать', 'заведующая', 'архангельский', 'ателье',
                                'молодость', 'галина', 'лопасков', 'пошив', 'платьице', 'стоять', 'рубль',
                                'школьный',
                                'форма', 'действовать', 'скидка', 'говорить', 'платье', 'последний', 'звонок',
                                'купить',
                                'магазин', 'взрослый', 'интимный', 'товар', 'мужчина', 'костюм', 'школьница',
                                'пользоваться', 'спрос', 'магазин', 'игрушка', 'взрослый', 'подтверждать', 'костюм',
                                'школьница', 'часто', 'бывать', 'продажа', 'накануне', 'праздник', 'последний',
                                'звонок',
                                'стремительно', 'раскупать', 'подробно', 'категория', 'пост', 'платье', 'выбирать',
                                'свадебный', 'платье', 'август', 'многий', 'девушка', 'долго', 'замужество',
                                'представлять',
                                'мечта', 'свой', 'свадебный', 'платье', 'задолго', 'начало', 'подготовка',
                                'праздничный',
                                'торжество', 'необходимо', 'начинать', 'выбирать', 'свадебный', 'платье', 'это',
                                'делать',
                                'помощь', 'каталог', 'буклет', 'интернет', 'конец', 'конец', 'просто',
                                'прогуливаться',
                                'свадебный', 'салон', 'магазин', 'настоящий', 'время', 'очень', 'большой', 'выбор',
                                'свадебный', 'платье', 'платье', 'шлейф', 'скромный', 'узкий', 'пышный',
                                'множество',
                                'юбка', 'длинный', 'короткий', 'белый', 'цветной', 'каждый', 'вид', 'отвечать',
                                'место',
                                'проведение', 'свадьба', 'стиль', 'просто', 'вкус', 'жених', 'невеста', 'огромный',
                                'количество', 'предлагать', 'продукция', 'выбирать', 'именно', 'подходить',
                                'начало',
                                'останавливать', 'свой', 'выбор', 'несколько', 'салон', 'который', 'решать',
                                'посещать',
                                'примерять', 'предлагать', 'платье', 'естественно', 'жить', 'небольшой', 'город',
                                'пара',
                                'салон', 'обходить', 'весь', 'подробно', 'категория', 'пост', 'свадебный', 'платье',
                                'платье', 'греческий', 'стиль', 'май', 'современный', 'девушка', 'преображаться',
                                'самый',
                                'настоящий', 'богиня', 'сложный', 'стоить', 'выбирать', 'идеальный', 'платье',
                                'греческий',
                                'стиль', 'важный', 'выбирать', 'именно', 'фасон', 'который', 'подчеркивать',
                                'достоинство',
                                'фигура', 'скрывать', 'иметься', 'недостаток', 'складка', 'драпировка', 'греческий',
                                'платье', 'прекрасно', 'добавлять', 'объем', 'это', 'нужно', 'скрывать', 'лишний',
                                'выбор',
                                'платье', 'греческий', 'стиль', 'обращать', 'внимание', 'цвет', 'смотреться',
                                'наиболее',
                                'выигрышно', 'совершенно', 'необязательно', 'приобретать', 'именно', 'белый',
                                'платье',
                                'учет', 'ваш', 'цвет', 'кожа', 'выбирать', 'бежевый', 'розовый', 'голубой',
                                'прочий',
                                'цвет', 'платье', 'вешалка', 'выглядеть', 'абсолютно', 'иначе', 'тело', 'поэтому',
                                'полениться', 'мерить', 'весь', 'экземпляр', 'оно', 'настоящему', 'оживать',
                                'фигура',
                                'надевать', 'платье', 'греческий', 'стиль', 'мочь', 'молоденький', 'девушка',
                                'дама',
                                'солидный', 'возраст', 'оно', 'иметь', 'возрастной', 'граница', 'подробно',
                                'категория',
                                'пост', 'платье', 'январь', 'февраль', 'март', 'апрель', 'май', 'июнь', 'июль',
                                'август',
                                'сентябрь', 'октябрь', 'ноябрь', 'декабрь', 'декабрь', 'главный', 'весь', 'пост',
                                'свадебный', 'платье', 'вечерний', 'платье', 'новогодний', 'платье', 'платье',
                                'выпускной',
                                'разный', 'новость', 'выкройка', 'популярный', 'новость', 'новогодний', 'платье',
                                'маленький', 'модница', 'мода', 'осень', 'зима', 'элегантный', 'платье', 'тренд',
                                'соблазнительный', 'коктейльный', 'платье', 'новогодний', 'корпоратив',
                                'прозрачный',
                                'платье', 'полиэтилен', 'группа', 'фабрика', 'носить', 'короткий', 'кружевной',
                                'платье',
                                'модель', 'платье', 'полный', 'женщина', 'наталья', 'королева', 'вновь', 'изменять',
                                'имидж', 'новогодний', 'платье', 'маленький', 'модница', 'свадебный', 'платье',
                                'стиль',
                                'фасон', 'тенденция', 'мода', 'мода', 'зимний', 'сезон', 'монохромный', 'черно',
                                'белый',
                                'платье', 'мода', 'век', 'платье', 'вторсырье', 'мода', 'офисный', 'платье', 'pre',
                                'fall',
                                'narciso', 'rodrigue', 'наряд', 'модный', 'бренд', 'каннский', 'фестиваль',
                                'ошибка',
                                'выбор', 'платье', 'российский', 'звезда', 'розовый', 'главный', 'тренд',
                                'свадебный',
                                'платье', 'блеск', 'звездный', 'платье', 'золотой', 'граммофон', 'школьный', 'мода',
                                'пря',
                                'коллекция', 'журнал', 'fashion', 'book', 'золотой', 'платье', 'честь', 'мороженое',
                                'млн',
                                'доллар', 'предновогодний', 'наряд', 'леди', 'гага', 'платье', 'новогодний', 'елка',
                                'ажурный', 'платье', 'писк', 'мода', 'архив', 'хороший', 'коктейльный', 'платье',
                                'открывать', 'спина', 'новогодний', 'платье', 'беременный', 'кружевной', 'платье',
                                'хит',
                                'мода', 'показ', 'свадебный', 'платье', 'цвет', 'фасон', 'новогодний', 'платье',
                                'платье',
                                'шоколад', 'изысканный', 'гармония', 'звездный', 'битва', 'платье', 'неделя',
                                'мусульманский', 'мода', 'казань', 'мода', 'платье', 'новый', 'год', 'свадебный',
                                'платье',
                                'звездный', 'свадьба', 'год', 'искушение', 'вечерний', 'платье', 'весна', 'лето',
                                'zac',
                                'posen', 'новогодний', 'коллекция', 'платье', 'jovani', 'fashion', 'delicatesses',
                                'трикотажный', 'платье', 'коллекция', 'платье', 'птица', 'парк', 'джеймс', 'платье',
                                'пиппа', 'миддлтон', 'купить', 'интернет', 'корректировать', 'нижний', 'белье',
                                'коктейльный', 'платье', 'коллекция', 'вечерний', 'платье', 'кира', 'пластинин',
                                'украсть',
                                'коллекция', 'платье', 'марка', 'джейкобс', 'модный', 'свадебный', 'платье',
                                'осень',
                                'зима', 'модный', 'платье', 'осенне', 'зимний', 'сезон', 'помощь', 'раздел',
                                'карта',
                                'раздел', 'анонс', 'статья', 'поиск', 'раздел', 'статистика', 'раздел', 'подписка',
                                'rss',
                                'rss', 'раздел', 'комментарий', 'rss', 'раздел', 'коммент', 'платье',
                                'использование',
                                'материал', 'сайт', 'активный', 'гиперссылка', 'обязательный']}}
        field = 'extra'
        if '$set' in d:
            d['$set'] = dict_roll(d['$set'],
                                  'extra',
                                  ignore_set=set(['artm100',]))  # rolling out dict of dicts as extra.key1.key2:value, ... so on

        self.assertTrue('extra.models.artm100.44' not in d['$set'])
        self.assertTrue('extra.models.artm100' in d['$set'])

    def test_get_dict_roll_one_true_4(self):
        d = {'some':{'some1':0,'some2':{'some3':0,'some4':[1,2,3,4]}}}
        field = 'some'
        r = dict_roll(d,field)
        self.assertTrue('some.some1' in r)
        self.assertTrue('some.some2.some3' in r)
        self.assertTrue('some.some2.some4' in r)
        self.assertTrue('some' not in r)
        self.assertTrue('some.some2' not in r)

    def test_battle_update_query(self):
        d = {'$set': {'modified_at': datetime.datetime(2016, 12, 26, 12, 17, 0, 829115), 'extra': {'models': {'topwords': ['строд', 'самара', 'монтажный', 'зао', 'strod', 'торговый', 'автодорожный', 'фирма', 'servis', 'стильсервис', 'утевская', 'сервис', 'ооо', 'строительство', 'ремонтный', 'стд', 'адма', 'специализированный', 'стк', 'дорожный', 'компания', 'организация', 'к1', 'подкатегория', 'добавлять'], 'artm500': 'strod-servis.samaracity.net 8^0.012994 59^0.0227405 66^0.0133365 104^0.0144554 109^0.278336 118^0.0106339 171^0.0124547 173^0.0460547 200^0.0361394 238^0.0139105 254^0.0193912 257^0.0121051 259^0.0101845 300^0.0179433 305^0.0389147 340^0.0243675 347^0.101209 358^0.0153185 411^0.0122034 453^0.0130601 487^0.0153976', 'artm100': 'strod-servis.samaracity.net 7^0.0402751 8^0.172989 12^0.0654858 14^0.0293553 26^0.0229211 30^0.0161813 31^0.117879 51^0.0138601 52^0.0200963 58^0.03034 59^0.0413681 66^0.0222964 68^0.0204344 72^0.0535075 76^0.0135972 83^0.0130732 89^0.016325 91^0.0505299 92^0.0650556 98^0.0205467'}}, 'lemmatized_text': 'строд сервис зао торговый монтажный фирма самара вход организация забывать пароль регистрация логин пароль организация самара добавлять объявление добавлять работа самара добавлять вакансия резюме рейтинг сайт участвовать организация самара строд сервис зао торговый монтажный фирма самара компания строд сервис зао торговый монтажный фирма автодорожный строительство специализированный дорожный средство предлагать продукция услуга направление строительство недвижимость подкатегория автодорожный строительство ремонт специализированный дорожный средство контакт город самара адрес самара утевская к1 факс самара гаражный этаж тело mail info strod servis сайт www strod servis сообщать ошибка строд сервис зао торговый монтажный фирма самара карта схема проезд статистика дата регистрация строд сервис зао торговый монтажный фирма каталог просмотр страница компания самара рубрика строительство недвижимость стк адма ооо строительный ремонтный компания стиль маркет ооо сеть магазин стильсервис проектный монтажный компания стд группа ооо торговый сервисный компания статус ооо строительный ремонтный компания copyright самара контакт реклама сайт правовой информация карта портал', '_id': 'strod-servis.samaracity.net'}}
        if '$set' in d:
            d['$set'] = dict_roll(d['$set'],'extra') # rolling out dict of dicts as extra.key1.key2:value, ... so on

        self.assertTrue('extra' not in d['$set'])

if __name__ == '__main__':
    unittest.main()
