#!/usr/bin/python
# -*- coding: latin1 -*- vim: ts=8 sts=4 sw=4 si et tw=79
#
# relpath.py

import os
from os.path import exists, isdir, abspath, normcase, \
        sep, curdir, pardir

__all__ = ['relpath',
           # exception classes:
           'InvalidArgument',
           'TargetMissing',
           'BaseMissing',
           'DriveMismatch',
           ]
__authors__ = ['R. Barran',
               'Tobias Herp <tobias.herp@gmx.net>',
               ]
VERSION = (1,	# initial version, compatible to R.Barran 30/08/2004
           0,	# modinfo, exception classes
           'rev-%s' % '$Rev: 839 $'[6:-2],
           )
__version__ = '.'.join(map(str, VERSION))

try:
    _
except NameError:
    def _(s): return s

class InvalidArgument(Exception):
    def __str__(self):
        return self.message
class TargetMissing(InvalidArgument, OSError):
    def __init__(self, target):
        self.message = _('Target "%(target)s" does not exist') % locals()
class BaseMissing(InvalidArgument, OSError):
    def __init__(self, base):
        self.message = _('Base "%(base)s" is not a directory'
                         ' or does not exist') % locals()
class DriveMismatch(InvalidArgument, OSError):
    def __init__(self, target_list, base_list):
        target = sep.join(target_list)
        target_drv = target_list[0].upper()
        base_drv = base_list[0].upper()
        self.message = _('Target "%(target)s" is on a different drive'
                         ' ("%(target_drv)s") to base ("%(base_drv)s")'
                         ) % locals()

if os.name in ['nt', 'dos', 'os2']:
    def _check_drives(pl1, pl2):
        if normcase(pl1[0]) != normcase(pl2[0]):
            raise DriveMismatch(pl1, pl2)
else:
    def _check_drives(pl1, pl2):
        pass


def relpath(target, base=curdir, as_url=0):
    """
    Return a relative path to the target from either the current dir or an optional base dir.
    Base can be a directory specified either as absolute or relative to current dir.
    """

    if not exists(target):
        raise TargetMissing(target)

    if not isdir(base):
        raise BaseMissing(base)

    base_list = abspath(base).split(sep)
    target_list = abspath(target).split(sep)

    # On the windows platform the target may be on a completely different drive from the base.
    _check_drives(target_list, base_list)

    # Starting from the filepath root, work out how much of the filepath is
    # shared by base and target.
    for i in range(min(len(base_list), len(target_list))):
        if base_list[i] <> target_list[i]:
            break
    else:
        # If we broke out of the loop, i is pointing to the first differing path elements.
        # If we didn't break out of the loop, i is pointing to identical path elements.
        # Increment i so that in all cases it points to the first differing path elements.
        i += 1

    if as_url:
        PARDIR, SEP = '..', '/'
    else:
        PARDIR, SEP = pardir, sep
    rel_list = [PARDIR] * (len(base_list)-i) + target_list[i:]
    return SEP.join(rel_list)

if __name__ == '__main__':
    import modinfo
    modinfo.main(version=__version__)

