# python module junclib - Junction support (needed on Windows only)
''"""\
junclib -- junction library (a Python module)

Might become replaced by the fst (file system tree) module;
see junclib.txt
"""
__author__ = 'Tobias Herp <tobias.herp@gmx.de>'
VERSION = (0,
           1, # get_junction_target works in some cases
           2, # junclib.txt
           )
__version__ = '.'.join(map(str, VERSION))


# for convenience:
from os.path import isdir, isfile, islink

isjunction, junction_target = (None, None)
try:
    set
except NameError:
    from sets import Set as set

# initially support English (<JUNCTION>) and German versions:
JUNCMARKERS = set(['<%s>' % word
                   for word in _('junction verbindung').upper().split()
                   ])
# <DIR> for German and English version, but one never knows...
DIRMARKERS = set(['<%s>' % word
                   for word in 'dir'.upper().split()
                   ])
class UnexpectedString(AssertionError):
    """
    an assumption about the DIR output format turns out be inaccurate
    """

class ParsingError(RuntimeError):
    """
    Unexpected (or not yet handled) output format
    """

class CreationError(ParsingError):
    """
    junction.exe didn't yield the expected output
    """

class NotAJunction(ValueError):
    """
    Something what needs to be a junction turned out to be something else
    """

_SIZECHARS = '0123456789.,\''

DIROPTS = ('/-C '    # keine Tausender-Trennzeichen
           '/-W /-B '# weder breit noch kurz
           '/-D '    # /D ist das bessere /W
           '/-P '    # keine Pausen
           '/-Q '    # keine Owner
           '/-S '    # keine Rekursion
           )

REPARSEPOINT_TYPES = ('JUNCTION',)

def parse_junction_output(s, single=0):
    """
    Parse the given junction.exe output.

    If single is True, return after the first junction.
    """
    lines = s.split('\r\n')
    cr_found = 0
    bl_found = 0 # blank line after copyright
    ju_found = 0
    idx = 0
    juncname = None
    import pdb
    # pdb.set_trace()
    for idx in range(len(lines)):
        line = lines[idx]
        if line:
            if bl_found:
                if line.startswith('No '):
                    break
                elif ':' in line:
                    if line[0] == ' ':
                        if not juncname:
                            raise UnexpectedString('Having this line ('
                                    '%(line)r), there should be a local '
                                    'name pending (%(juncname)s)'
                                    % locals())
                        dest = line.split(': ')[1]
                        yield juncname, dest[0].upper()+dest[1:]
                        if single:
                            return

                    elif juncname:
                        raise UnexpectedString('juncname should be empty in '
                                'this line (%(line)r); it is %(juncname)r'
                                % locals())
                    else:
                        second = line.rfind(':')
                        if not second > 1:
                            raise UnexpectedString('Expected a ":" somewhere '
                                    'at the end of the string (%(line)s)'
                                    % locals())
                        shouldbe = line[second+1:].strip()
                        if shouldbe not in REPARSEPOINT_TYPES:
                            types = ', '.join(REPARSEPOINT_TYPES)
                            raise UnexpectedString('Unexpected string '
                                    '%(shouldbe)r (currently handled:'
                                    ' %(types)s)'
                                    % locals())
                        juncname = line[:second].lstrip()
            elif cr_found:
                pass
            elif line.startswith('Junction v'):
                cr_found = 1
        elif juncname:
            juncname = None
        elif cr_found:
            bl_found = 1
        elif idx == 0:
            pass
        else:
            raise UnexpectedString(
                '\r'.join(['Unexpected blank line %(idx)d:','']
                         +lines[:idx]
                         +['<%(idx)d>']
                         +lines[idx+1:]) % locals())


def get_junction_targets(name):
    """
    Return the target of the (suspected) junction.
    If the argument is not the name of a junction,
    a NotAJunction exception is raised.

    Could be refactored to yield (junction, dest) tuples.
    """
    from subprocess import Popen, PIPE
    if 1:
        s = Popen(('junction.exe', name), stdout=PIPE, stderr=PIPE
                  ).communicate()[0]
    else:
        p = Popen(('junction.exe', name), stdout=PIPE, stderr=PIPE)
        s = p.communicate()[0]
    return parse_junction_output(s, 0)

def get_junction_target(name):
    """
    Return the target of the (suspected) junction.
    If the argument is not the name of a junction,
    a NotAJunction exception is raised.

    Could be refactored to yield (junction, dest) tuples.
    """
    from subprocess import Popen, PIPE
    if not isdir(name):
        if isfile(name):
            raise NotAJunction('"%(name)s" is not a junction'
                    ' (but a file)' % locals())
        if not islink(name):
            raise NotAJunction('"%(name)s" is not a junction'
                    ' (not even a directory)' % locals())

    if 1:
        s = Popen(('junction.exe', name), stdout=PIPE, stderr=PIPE
                  ).communicate()[0]
    else:
        p = Popen(('junction.exe', name), stdout=PIPE, stderr=PIPE)
        s = p.communicate()[0]
    # print dir(p)
    # p.close()
    for tup in parse_junction_output(s, 1):
        return tup[1]

dangerous_chars = set()
forbidden_chars = set()
def shell_quote(name):
    """
    return the quoted filename, suitable for commandline usage.

    Covers Windows only, so far (" being the only quote character,
    and no checks for $variable substitution)

    """
    global dangerous_chars, forbidden_chars
    if not dangerous_chars:
        forbidden_chars = set('"|*?') # und vielleicht weitere!
        # aus Hilfe von CMD (WinXP): 
        dangerous_chars = set("&()[]{}^=;!'+,`~")
    res = list()
    quote = 0

    name = name.strip('"')
    for ch in name:
        if ch in forbidden_chars:
            raise InvalidFilename('%(ch)r is not allowed in filenames'
                    ' (%(name)s)' % locals())
        if ch in dangerous_chars:
            quote = 1
            break
    if quote:
        return ''.join('"', name, '"')
    else:
        return name




def isjunction_shell(juncname, mustexist=1):
    """
    Tell whether the given name specifies a junction,
    by using the DIR shell command.
    NOTE that for a junction SUBSTituted with a drive letter,
    this will return False; but this shouldn't hurt, since this
    spec. can't be deleted with RD/RMDIR

    @param juncname name of the suspected junction
    @param mustexist   if true, raise an AssertionError if
                    *nothing* exists at the given path
                    (no junction nor anything else)
    """
    from os.path import normpath, basename, realpath, normcase
    from os import popen
    handover = normcase(realpath(normpath(juncname)))
    base = basename(handover)
    if not basename(handover): # e.g. C:\ (root directory)
        return False
    pi = popen('dir "%s?"' % handover)
    entriesfound = 0
    for dic in dir_dicts(handover+'?'):
        if dic['name'] in (base, handover):
            return dic['size'] in JUNCMARKERS
    if mustexist:
        raise AssertionError(_('suspected junction "%(juncname)s" not found'
                               ) % locals())
    else:
        return 0

def _get_interesting_lines(seq):
    """
    filters off the "uninteresting" lines from the DIR command output,
    leaving the lines containing filesystem entries only
    """
    waiting4entries = 1
    for line in seq:
        if not line or line.startswith(' '):
            if waiting4entries:
                continue
            else:
                return
        yield line
    return

REO_J = None
def junc_dicts(do_append=0, **kwargs):
    """
    takes named arguments (kwargs, a dictionary) and returns dictionaries.

    Important keys:

    name -- the directory being read (must be present in **kwargs)
    parent -- the parent directory (optional, but will be present when
              called recursively)

    Added in returned dictionaries:
    name -- the name of the entry
    dest -- the refered entry
    """
    global REO_J
    import re, os
    from os import popen, sep, curdir, altsep
    from os.path import join, split, isdir, dirname, normpath
    name = kwargs['name']
    junction_listing_re() 
    ldic = {'parent': name or curdir,
            'tail':   sep+'*',
            }
    mypa = kwargs.get('parent')
    if mypa:
        ldic['parent'] = os.join(mypa, ldic['parent'])
    dic = dict()
    if name <> curdir:
        if isdir(name):
            pass
        elif not do_append:
            pass
        elif name[-1] in (sep, altsep):
            name = normpath(dirname(name))
            tail = sep+'*'
        else:
            tail = ''

    fo = os.popen(r'junction "%(parent)s\*"' % ldic)
    for line in fo:
        mo = REO_J.search(line)
        if mo:
            gdic = mo.groupdict()
            for k in gdic:
                if gdic[k]:
                    dic[k] = gdic[k]
            # dic.update(mo.groupdict())
            if dic.get('path', None) and dic.get('dest', None):
                dic['parent'], dic['name'] = os.path.split(dic['path'])
                yield dic
                dic = dict()

def junction_listing_re():
    """
    unfertig...
    """

    r"""

Systems Internals - http://www.sysinternals.com

\\?\p:\alle\Verzeichniseintrag: JUNCTION
   Substitute Name: G:\Dokumente und Einstellun...
    """
    global REO_J
    if REO_J is None:
        import re
        bs = r'\\'
        bsbs = 2*bs
        kl_auf = '('
        oder   = '|'
        kl_zu  = ')'
        lit_qm = r'\?' # literal question mark
        _0or1 = '?'
        _1ormore = '+'
        # print  bs, kl_auf, oder, kl_zu, lit_qm, _0or1, _1ormore
        RE = (r'^'+kl_auf+
              bsbs+lit_qm+bs+
              r'(?P<path>'+'([a-zA-Z]:'+bs+')'+_0or1+'[^:]'+_1ormore+'): JUNCTION'+
              kl_zu+oder+kl_auf+
              r'^\s*Substitute Name: (?P<dest>.'+_1ormore+')'+
              r'\n?'+kl_zu+'$')
        # print RE
        REO_J = re.compile(RE)

def junc_names(**kwargs):
    """
    takes named arguments (kwargs, a dictionary) and returns the names
    of junctions (if any found).

    Important keys:

    name -- the directory being read (must be present in **kwargs)
    parent -- the parent directory (optional, but will be present when
              called recursively)
    """
    import re, os
    global REO_J
    junction_listing_re() 
    ldic = {'parent': kwargs['name'] or '.',
            }
    mypa = kwargs.get('parent')
    if mypa:
        ldic['parent'] = os.join(mypa, ldic['parent'])
    dic = dict()
    fo = os.popen(r'junction "%(parent)s\*"' % ldic)
    for line in fo:
        mo = REO_J.search(line)
        if mo:
            gdic = mo.groupdict()
            for k in gdic:
                if gdic[k]:
                    dic[k] = gdic[k]
            # dic.update(mo.groupdict())
            if dic.get('path', None) and dic.get('dest', None):
                yield os.path.basename(dic['path'])
                dic = dict()

REO = None
def dir_dicts(**kwargs):
    """
    takes named arguments (kwargs, a dictionary) and returns dictionaries.

    Important keys:

    name -- the directory being read (must be present in **kwargs)
    parent -- the parent directory (optional, but will be present when
              called recursively)

    Added in returned dictionaries:
    date -- the date
    time -- the time
    size -- the size of the listed file, or '<DIR>',
            *or* (something like) '<JUNCTION>'
    name -- the name of the entry
    """
    global REO
    import re, os
    if REO is None:
        RE = (r'^(?P<date>[-0-9.]+)\s+'
              r'(?P<time>[:0-9]+)\s+'
              r'(?P<size>\S+)\s+'
              r'(?P<name>.+)$')
        REO = re.compile(RE)
    os.environ['DIRCMD'] = '/-C '+kwargs.get('optstr','')
    ldic = {'parent': kwargs['name'] or '.',
            }
    mypa = kwargs.get('parent')
    if mypa:
        ldic['parent'] = os.join(mypa, ldic['parent'])
    import pdb
    if 0: pdb.set_trace()
    fo = os.popen(r'dir "%(parent)s\*"' % ldic)
    for line in fo:
        mo = REO.search(line)
        if mo:
            dic = mo.groupdict()
            dic.update(ldic)
            yield dic

def get_junctions_shell(dirname, mustexist=1, subsonly=1, fullnames=0):
    """
    return an iterator over all junctions in the given dir,
    using shell commands

    dirname -- name of the given directory

    mustexist -- if set to true (default), yield an AssertionError
                 if dirname is not a directory; if set to false,
                 just return

    subsonly -- if set to true (default), don't return the given
                dirname (or it's parent)
    """
    from os.path import normpath, basename, realpath, normcase, isdir
    if subsonly or 1:
        from os.path import curdir, pardir
    if fullnames:
        from os.path import join, normpath, realpath
        dirname = realpath(normpath(dirname))
    if not isdir(dirname):
        if mustexist:
            raise AssertionError(_('%(dirname)s is not a directory'
                                   ) % locals())
        else:
            return
    for dic in dir_dicts(dirname):
        if subsonly and dic['name'] in (curdir, pardir):
            continue
        if dic['size'] in JUNCMARKERS:
            yield (fullnames
                    and realpath(normpath(join(dirname, dic['name'])))
                    or dic['name'])
    return

isjunction = isjunction_shell
get_junctions = get_junctions_shell

class DirEntry:
    _isdir = 0
    _isfile = 0
    _islink = 0
    _isjunc = 0

    def __init__(self, name, *args):
        self.name = name
        if args:
            print 'WARNING (%r): argument[s] ignored (%s)' % (
                    self, ', '.join(args))

    def __str__(self):
        return '<%s %r>' % (self.__class__.__name__,
                            self.name)

    def isdir(self):
        return self._isdir

    def isfile(self):
        return self._isfile

    def islink(self):
        return self._islink

    def isjunc(self):
        return self._isjunc

class File(DirEntry):
    _isfile = 1

class Directory(DirEntry):
    _isdir = 1

class Symlink(DirEntry):
    _islink = 1
    def isfile(self):
        raise NotImplemented
    def isdir(self):
        raise NotImplemented

class Junction(DirEntry):
    """
    junctions can reference dirs only (as of WindowsXP)
    """
    _isjunc = 1
    def isfile(self):
        raise NotImplemented
    def isdir(self):
        raise NotImplemented


CLASSMAP = {'file': File,
            'dir':  Directory,
            'link': Symlink,
            'junc': Junction,
            }
def get_entryobjects_shell(dirname, mustexist=1, subsonly=1):
    """
    iterate the given directory and yield DirEntry objects
    """
    from os.path import normpath, basename, realpath, normcase, isdir
    if subsonly or 1:
        from os.path import curdir, pardir
    if not isdir(dirname):
        if mustexist:
            raise AssertionError(_('%(dirname)s is not a directory'
                                   ) % locals())
        else:
            return
    from os.path import join, normpath, realpath
    dirname = realpath(normpath(dirname))
    for dic in dir_dicts(dirname):
        if dic['size'] in JUNCMARKERS:
            yield CLASSMAP['junc'](dic['name'],
                                   isdir(normpath(join(dirname,
                                                       dic['name']))))
        elif dic['size'] in DIRMARKERS:
            yield CLASSMAP['dir'](dic['name'])
        else:
            yield CLASSMAP['file'](dic['name'])
    return

def buffered(seq):
    """
    iterate a sequence and yield 2-tuples of

    - the item
    - a boolean value indicating whether the item is the last
    """
    has_prev = 0
    for item in seq:
        if has_prev:
            yield prev, False
        else:
            has_prev = 1
        prev = item
    if has_prev:
        yield prev, True
    return

def create_junction_cmd(src, dest):
    return 'junction.exe "%(dest)s" "%(src)s"' % locals()

def create_junction_shell(src, dest):
    import os
    fo = os.popen(create_junction_cmd(src, dest))
    c, t = (0, 0)
    for line in fo:
        if line.startswith('Created:'):
            c = line.split(':', 1)[1].strip()
        elif line.startswith('Targetted at:'):
            t = line.split(':', 1)[1].strip()
    if not (c and t):
        raise CreationError(_('error when creating the junction'
                              ' %(dest)s pointing to %(src)s') % locals())

re_fullpath = r'[A-Z]:\\[^:]+' # ok, wenn Doppelpunkt oder Ende folgt
re_name = r'^(?P<prefix>\\\\\?\\)?(?P<path>'+re_fullpath+r'): JUNCTION\S*$'
re_target = r'^   Substitute Name: (?P<subst>'+re_fullpath+')$'

create_junction = create_junction_shell

if __name__ == '__main__':
    # print __doc__
    i, max = 0, 10
    import pdb
    if 0: pdb.set_trace()
    import sys, os.path
    if not sys.argv[1:]:
        sys.argv.append(os.path.sep.join((os.path.curdir, '*')))
    for tup in get_junction_targets(sys.argv[1]):
        i += 1
        print '%s --> %s' % tup
        if i > max:
            break

    for DIR in sys.argv[1:]:
        print DIR
        for name in junc_names(name=DIR, optstr='/AD'):
            print name
            i += 1
            if i >= max:
                break
        for dic in junc_dicts(name=DIR, optstr='/AD'):
            print dic
            i += 1
            if i >= max:
                break

# vim: ts=4 sts=4 et sw=4 si
