#!/usr/bin/env python
# -*- coding: latin1 -*- vim: ts=8 sts=4 sw=4 si et tw=79
"""\
echt aussehende User-Agent-Strings erzeugen

reale Beispiele (von echten Browsern):
Mozilla/5.0 (Windows NT 6.1; rv:5.0) Gecko/20100101 Firefox/5.0
Mozilla/5.0 (Windows NT 6.1; rv:5.0) Gecko/20110706 Firefox/5.0 SeaMonkey/2.2
Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.1; Trident/5.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; .NET4.0C)
Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0)
"""

__author__ = "Tobias Herp <tobias.herp@gmx.net>"
VERSION = (0,
           1,   # initial version
           'rev-%s' % '$Rev: 840 $'[6:-2],
           )
__version__ = '.'.join(map(str, VERSION))
# fr Module:
__all__ = [
    'FakeUserAgent',
    'FakeGecko',
    'FakeFirefox',
    'FakeSeamonkey',
    'FakeMSIE',
           ]

from time import strftime, localtime

ONEDAY = 24 * 60**2
import pdb

class FakeUserAgent:
    def __init__(self):
        pass

    def __str__(self):
        return 'Mozilla/5.0'

    def mozversion(self):
        return '5.0'

    def osinfo(self):
        return 'Windows NT 6.1'   # TODO: use real os info

    def inparentheses(self):
        return [self.osinfo()]

    def vars(self):
        return {'mv': self.mozversion(),
                'osinfo': self.osinfo(),
                }

    def primary(self):
        return 'Mozilla/%(mv)s'

    def strings(self):
        liz = [self.primary()]
        ipa = self.inparentheses()
        if ipa:
            liz.append('(%s)' % '; '.join(ipa))
        return liz

    def __str__(self):
        return ' '.join(self.strings()) % self.vars()

class FakeGecko(FakeUserAgent):
    def __init__(self, geckodate=None):
        self._geckodate = self.geckodate(geckodate)
        self._revision = '5.0'

    def geckodate(self, geckodate, geckoage=0):
        """
        geckodate - a time tuple, not None
        geckoage - an integer [days] (not yet used)
        """
        # pdb.set_trace()
        assert isinstance(geckoage, int)
        return strftime('%Y%m%d',
                        (geckodate or localtime())) # - (geckoage*ONEDAY)

    def vars(self):
        dic = FakeUserAgent.vars(self)
        dic['geckodate'] = self._geckodate
        dic['rv'] = self._revision
        return dic

    def inparentheses(self):
        return FakeUserAgent.inparentheses(self) + [
                'rv:%(rv)s',
                ]

    def strings(self):
        return FakeUserAgent.strings(self) + [
                'Gecko/%(geckodate)s',
                ]

class FakeFirefox(FakeGecko):
    def strings(self):
        return FakeGecko.strings(self)+['Firefox/%(rv)s']

class FakeSeamonkey(FakeFirefox):
    def strings(self):
        """
        Since v2.1, Seamonkey pretends to be Firefox
        """
        return FakeFirefox.strings(self)+['SeaMonkey/%(smv)s']

    def vars(self):
        dic = FakeFirefox.vars(self)
        dic['smv'] = '2.2'  # SeaMonkey-Version
        return dic


class FakeMSIE(FakeUserAgent):
    def __init__(self, msieversion=9.0):
        self._mozversion = (msieversion >= 8
                            ) and '5.0' or '4.0'
        self._msieversion = msieversion

    def mozversion(self):
        return self._mozversion

    def vars(self):
        dic = FakeUserAgent.vars(self)
        dic['msiev'] = '%.1f' % self._msieversion
        return dic

    def inparentheses(self):
        liz = ['compatible',
               'MSIE %(msiev)s',
               self.osinfo(),
               ]
        if self._msieversion >= 7:
            liz.append('Trident/5.0')
        return liz


if __name__ == '__main__':
    if 0:
        print FakeUserAgent()
        print FakeGecko()
        print FakeFirefox()
        print FakeSeamonkey()
        # pdb.set_trace()
        print FakeMSIE()
    from thebops import modinfo
    modinfo.main(version=__version__)

