# Rekall Memory Forensics
# Copyright (c) 2012, Michael Cohen <scudette@gmail.com>
# Copyright (c) 2010, 2011, 2012 Michael Ligh <michael.ligh@mnin.org>
# Copyright 2013 Google Inc. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

from rekall import utils
from rekall.plugins import yarascanner
from rekall.plugins.windows import common


class WinYaraScan(yarascanner.YaraScanMixin, common.WinProcessFilter):
    """Scan using yara signatures."""

    def collect_task_scan(self, task):
        """Scan a task's address space.

        In Windows pagetable entries outside the VAD might be uninitialized and
        this will lead to scanning massive regions of mostly unmapped memory.

        When asked to scan process memory we only scan memory inside the VAD.
        """
        # We have to change process context so the address resolver and task_as
        # line up.
        task_as = self.session.default_address_space

        count = 0
        scanner = yarascanner.BaseYaraASScanner(
            profile=self.profile, session=self.session,
            address_space=task_as, rules=self.rules)

        for vad in sorted(task.RealVadRoot.traverse(),
                          key=lambda x: x.Start):
            self.session.report_progress(
                "Scanning VAD %s from %#0x (%#0x)",
                task.name, vad.Start, vad.Length)

            # Only scan the VAD region.
            for hit in scanner.scan(vad.Start, vad.Length):
                count += 1
                rule = hit[0]
                address = hit[1]

                symbol = self.session.address_resolver.format_address(address)
                yield (task, rule, address, utils.HexDumpedString(
                    task_as.read(address, 0x40)), symbol)

                # If we exceed the total hit count we are done.
                if count > self.hits:
                    return
