#!/usr/bin/env python
# -*- coding: utf-8 -*-

""" search dialog for text/table-based widgets """

# pytkapp.tkw: search dialog for text/table-based widgets
#
# Copyright (c) 2015 Paul "Mid.Tier"
# Author e-mail: mid.tier@gmail.com

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

###################################
## import
###################################
import sys

import gettext
if __name__ == '__main__':
    if sys.hexversion >= 0x03000000:
        gettext.install(__name__)
    else:
        gettext.install(__name__, unicode=True)
elif '_' not in __builtins__:
    _ = gettext.gettext

if sys.hexversion >= 0x03000000:
    from tkinter import Label, Frame, Text
    from tkinter import Button, Entry, StringVar
    from tkinter.constants import N, S, W, E
    from tkinter.constants import TOP, LEFT, RIGHT, X, YES
    import tkinter.messagebox as messagebox
    import tkinter.scrolledtext as scrolledtext
else:
    from Tkinter import Label, Frame, Text
    from Tkinter import Button, Entry, StringVar
    from Tkconstants import N, S, W, E
    from Tkconstants import TOP, LEFT, RIGHT, X, YES
    import tkMessageBox as messagebox
    import ScrolledText as scrolledtext

from pytkapp.pta_routines import get_estr, xprint, tu
from pytkapp.tkw.tkw_routines import toplevel_footer, toplevel_header, is_basechild_subwidget
from pytkapp.tkw.tablelistwrapper import TableList

###################################
## classes
###################################


class SearchDialog:
    """ simple dialog for searching some text in simple widgets """

    def __init__(self, pw_caller, pw_datawidget, **kw):
        """ init
            pw_caller - who call
            pw_datawidget - widget that store data
            kw keys:
                searchresultbg - background for search result
                resultlinebg - background for line with search result
                lastsearch - stringvar holding last searched text
                postsearchcmd - if not None than this func will be fired after dialog
                research - None/single/all process search with lastsearch clause
        """
        self.__caller = pw_caller
        self.__datawidget = pw_datawidget
        self.__toplevel = None

        self.__prefs = {}
        self.__prefs['searchresultbg'] = kw.get('searchresultbg', 'orange')
        self.__prefs['resultlinebg'] = kw.get('resultlinebg', 'yellow')

        if kw.get('lastsearch', None) is None:
            self.__lastsearch = StringVar()
        else:
            self.__lastsearch = kw['lastsearch']

        self.__research = kw.get('research', None)

        if isinstance(self.__datawidget, scrolledtext.ScrolledText):
            self.__datawtype = 'text'
        elif isinstance(self.__datawidget, Text):
            self.__datawtype = 'text'
        elif isinstance(self.__datawidget, TableList):
            self.__datawtype = 'table'
        else:
            self.__datawtype = 'undef'

        if self.__datawtype in ('text',):
            self.__findallallowed = True
        elif self.__datawtype in ('table',):
            if self.__datawidget.cget("-selectmode") == "extended":
                self.__findallallowed = True
            else:
                self.__findallallowed = False
        else:
            self.__findallallowed = False

        self.__firstindex = None

        self.__postsearchcmd = kw.get('postsearchcmd', None)

        if self.__research is None:
            top_page, top_frame = toplevel_header(self.__datawidget.winfo_toplevel(),
                                                  title=_('Search'),
                                                  destroycmd=self.close_dialog,
                                                  noresize=1)
            top_frame.configure(padx=2, pady=2)

            self.__toplevel = top_page
            top_page.bind('<Escape>', self.close_dialog, '+')

            lw_label = Label(top_frame, text=_('Find what:'))
            lw_label.grid(row=0, column=0, sticky=N+E+W+S, padx=2, pady=2)

            eitem = Entry(top_frame, width=50, justify=LEFT, textvariable=self.__lastsearch)
            eitem.grid(row=0, column=1, sticky=N+E+W+S, padx=2, pady=2)
            eitem.bind('<Return>', self.search_smth)

            top_frame.pack(side=TOP)

            lw_ctrlframe = Frame(top_page)
            lw_btn = Button(lw_ctrlframe, text=_('Cancel'), width=12, command=top_page.destroy)
            lw_btn.pack(side=RIGHT, padx=2, pady=2)

            lw_btn = Button(lw_ctrlframe, text=_('Reset'), width=12, command=lambda x=1: self.reset_searchres(True))
            lw_btn.pack(side=RIGHT, padx=2, pady=2)

            if self.__findallallowed:
                lw_btn = Button(lw_ctrlframe, text=_('Find all'), width=12, command=self.search_all)
                lw_btn.pack(side=RIGHT, padx=2, pady=2)

            lw_btn = Button(lw_ctrlframe, text=_('Find next'), width=12, command=self.search_smth)
            lw_btn.pack(side=RIGHT, padx=2, pady=2)

            eitem.focus()

            lw_ctrlframe.pack(side=TOP, expand=YES, fill=X)

            if is_basechild_subwidget(self.__caller):
                lv_dograb = False
                lv_skipbackfocus = True
            else:
                lv_dograb = True
                lv_skipbackfocus = False

            toplevel_footer(top_page,
                            self.__datawidget.winfo_toplevel(),
                            skipbackfocus=lv_skipbackfocus,
                            dograb=lv_dograb)
        else:
            if self.__research == 'all' and self.__findallallowed:
                self.search_all()
            else:
                self.search_smth()

    def get_firstindex(self):
        """ return first result """

        return self.__firstindex

    def reset_searchres(self, pb_resetdata=False):
        """ clear all prev. marked results """

        try:
            if pb_resetdata:
                self.__lastsearch.set('')

            if self.__datawtype in ('text',):
                # remove highlight
                self.__datawidget.tag_delete("searchresult")
                self.__datawidget.tag_config("searchresult", background=self.__prefs['searchresultbg'])
                self.__datawidget.tag_delete("resultline")
                self.__datawidget.tag_config("resultline", background=self.__prefs['resultlinebg'])

                self.__datawidget.tag_raise("searchresult")
            elif self.__datawtype in ('table',):
                # remove current selection
                self.__datawidget.selection_clear(0, "end")
                self.__datawidget.event_generate('<<TablelistSelect>>')

        except:
            lv_message = 'failed to reset search result: %s' % get_estr()
            xprint(lv_message)

    def mark_searchres(self, pv_data, pv_indx1, pv_indx2=None):
        """ mark search results """

        if self.__datawtype in ('text',):
            if pv_indx2 is None:
                lv_indx2 = pv_indx1
            else:
                lv_indx2 = pv_indx2
            self.__datawidget.tag_add("resultline", '%s.0' % (pv_indx1.split('.')[0]), '%s.end' % pv_indx1.split('.')[0])
            self.__datawidget.tag_add("searchresult", pv_indx1, lv_indx2)
        elif self.__datawtype in ('table',):
            self.__datawidget.selection_set((pv_indx1,))
            self.__datawidget.event_generate('<<TablelistSelect>>')

    def search_smth(self, p_top=None):
        """ search one record """

        lv_data = self.__lastsearch.get().upper()
        lv_resindex = None
        if len(lv_data) > 0:

            lw_body = self.__datawidget

            if self.__datawtype in ('text',):
                self.reset_searchres()

                lv_curr = lw_body.index("current")
                lv_indx1 = lw_body.search(lv_data, index=lv_curr, stopindex="end", exact=False, nocase=1)
                if lv_indx1 != "":
                    lv_indx2 = lw_body.index(lv_indx1+'+'+str(len(lv_data))+'c')
                    self.mark_searchres(lv_data, lv_indx1, lv_indx2)
                    lv_resindex = lv_indx1
            elif self.__datawtype in ('table',):
                lv_tlsize = lw_body.size()
                if len(lw_body.curselection()) >= 1:
                    lv_index = lw_body.curselection()[0]
                    if lv_index < lv_tlsize-1:
                        lv_index += 1
                    else:
                        lv_index = 0
                else:
                    lv_index = 0

                ll_realcols = []
                for column_index in range(lw_body.columncount()):
                    if lw_body.columncget(column_index, "-hide") == 0:
                        ll_realcols.append(column_index)

                lv_result = False
                for row_index in range(lv_index, lv_tlsize):
                    ll_data = lw_body.rowcget(row_index, "-text")
                    lv_result = False
                    for indx, val in enumerate(ll_data):
                        if indx in ll_realcols:
                            if val is not None and tu(val).upper().find(lv_data) != -1:
                                lv_result = True
                                self.reset_searchres()
                                self.mark_searchres(lv_data, row_index)
                                lv_resindex = row_index
                                break
                    if lv_result:
                        break

            # finish
            if lv_resindex is not None:
                if self.__datawtype in ('text',):
                    self.__firstindex = lv_resindex
                elif self.__datawtype in ('table',):
                    self.__firstindex = lv_resindex

                if self.__research is None:
                    self.close_dialog()
            else:
                if self.__research is None:
                    messagebox.showinfo(_('Search failed'),
                                        _('Text "%s" not found') % (self.__lastsearch.get()),
                                        parent=self.__toplevel)

    def search_all(self):
        """ search all records """

        lv_data = self.__lastsearch.get().upper()
        lb_found = False

        if len(lv_data) > 0:
            self.reset_searchres()

            lw_body = self.__datawidget

            if self.__datawtype in ('text',):
                lv_curr = "1.0"

                lw_body = self.__datawidget

                while lv_curr != lw_body.index("end"):
                    lv_indx1 = lw_body.search(lv_data, index=lv_curr, stopindex="end", exact=False, nocase=1)
                    if lv_indx1 != "":
                        lv_indx2 = lw_body.index(lv_indx1+'+'+str(len(lv_data))+'c')
                        self.mark_searchres(lv_data, lv_indx1, lv_indx2)
                        lv_curr = lv_indx2
                        if not lb_found:
                            lb_found = True
                    else:
                        lv_curr = lw_body.index("end")
            elif self.__datawtype in ('table',):
                lv_tlsize = lw_body.size()
                lv_index = 0

                ll_realcols = []
                for column_index in range(lw_body.columncount()):
                    if lw_body.columncget(column_index, "-hide") == 0:
                        ll_realcols.append(column_index)

                for row_index in range(lv_index, lv_tlsize):
                    ll_data = lw_body.rowcget(row_index, "-text")
                    lv_result = False
                    for indx, val in enumerate(ll_data):
                        if indx in ll_realcols:
                            if val is not None and val.upper().find(lv_data) != -1:
                                if not lv_result:
                                    lv_result = True
                                if not lb_found:
                                    self.reset_searchres()
                                    lb_found = True
                                self.mark_searchres(lv_data, row_index)
                                break

            # finish
            if self.__research is None:
                if lb_found:
                    self.close_dialog()
                else:
                    messagebox.showinfo(_('Search failed'),
                                        _('Text "%s" not found') % (self.__lastsearch.get()),
                                        parent=self.__toplevel)

    def close_dialog(self, po_event=None):
        """ additional routines on closing """

        if self.__postsearchcmd is not None:
            self.__postsearchcmd()

        self.__toplevel.destroy()
