# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
Python-file importer as a picklable module.
Callables at the global scope of the module are also picklable.

>>> from bfc.importfile import import_file
>>> import cPickle as pickle, tempfile
>>> a = tempfile.NamedTemporaryFile(suffix = '.py')
>>> a.write('''print "Hello"
... x = 5
... def foo(x): print "%s" % (x,)
... ''')
>>> a.flush()
>>> module = PicklableModule(a.name)
Hello
>>> str(module) == '< PicklableModule from %s >' % (a.name, )
True
>>> repr(module) == 'PicklableModule(%r)' % (a.name, )
True
>>> str(module.foo) == '< PicklableCallable of %r named %s from %s>' % (module.foo._callable, module.foo._name, module)
True
>>> repr(module.foo) == 'PicklableCallable(%r, %r, %r)' % (module, module.foo._name, module.foo._callable)
True
>>> s = set()
>>> s.add(module)
>>> module.freeze() is module
True
>>> module.x
5
>>> type(module.x)
<type 'int'>
>>> module.foo(10)
10
>>> s.add(module.foo)
>>> module.foo.freeze() is module.foo
True
>>> module.foo.func_name
'foo'
>>> module.foo.__name__
'foo'
>>> del s
>>> b = tempfile.TemporaryFile(mode = 'w+')
>>> pickle.dump(module, b, -1)
>>> b.seek(0, 0)
>>> del module
>>> import_file.reset()
>>> module = pickle.load(b)
Hello
>>> module.foo(10)
10
>>> b.close()
>>> b = tempfile.TemporaryFile(mode = 'w+')
>>> pickle.dump(module.foo, b, -1)
>>> b.seek(0, 0)
>>> del module
>>> pickle.load(b)(10)
10


"""
from .importfile import import_file

import os
from collections import Callable

class PicklableCallable(object):
    __slots__ = ['_container',
                 '_name',
                 '_callable',
                 ]

    def __init__(self, container, name, callable):
        self._container = container
        self._name = name
        self._callable = callable

    def __getstate__(self):
        return {'c':self._container,'n':self._name}

    def __setstate__(self, state):
        self._container = state['c']
        self._name = state['n']
        self._callable = getattr(self._container._module, self._name)

    def __repr__(self):
        return '%s(%r, %r, %r)' % (self.__class__.__name__, self._container, self._name, self._callable)

    def __str__(self):
        return '< %s of %s named %s from %s>' %  (self.__class__.__name__, self._callable, self._name, self._container)

    def __call__(self, *args, **kws):
        return self._callable(*args, **kws)

    def __getattr__(self, attr):
        return getattr(self._callable, attr)

    def __hash__(self):
        return hash((self._container, self._name))

    def freeze(self):
        return self

    @property
    def func_name(self):
        return self._callable.func_name

    @property
    def __name__(self):
        return self._callable.__name__

class PicklableModule(object):
    __slots__ = ['_module',
                 '_filename',
                 '_cache',
                 ]

    def __init__(self, filename):
        self._init(os.path.abspath(filename))

    def _init(self, filename):
        self._module = import_file(filename)
        self._filename = filename
        self._cache = {}

    def __getstate__(self):
        return {'f' : self._filename}

    def __setstate__(self, state):
        self._init(state['f'])

    def __hash__(self):
        return hash(self._filename)

    def __getattr__(self, attr):
        v = getattr(self._module, attr)
        if isinstance(v, Callable):
            if attr in self._cache:
                return self._cache[attr]
            v = PicklableCallable(self, attr, v)
            self._cache[attr] = v
        return v

    def __repr__(self):
        return '%s(%r)' % (self.__class__.__name__, self._filename)

    def __str__(self):
        return '< %s from %s >' % (self.__class__.__name__, self._filename)
        
    def freeze(self):
        return self
