# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
A Pythonic implementation of double-linked lists.

The container class is LinkedList, and it contains BaseDataNode instances.
You should derive from BaseDataNode to contain your data in nodes,
or use the provided DataNode instead.

>>> l = LinkedList()
>>> ds = [DataNode(x) for x in xrange(5)]
>>> list(l)
[]
>>> ds[0].insert_after(l); list(l)
[DataNode(0)]
>>> ds[1].insert_after(l); list(l)
[DataNode(1), DataNode(0)]
>>> ds[2].insert_after(ds[0]); list(l)
[DataNode(1), DataNode(0), DataNode(2)]
>>> l = LinkedList()
>>> ds = [DataNode(x) for x in xrange(5)]
>>> list(l)
[]
>>> ds[0].is_attached
False
>>> ds[0].insert_after(l); list(l)
[DataNode(0)]
>>> ds[0].is_attached
True
>>> ds[1].insert_before(l); list(l)
[DataNode(0), DataNode(1)]
>>> ds[2].insert_after(ds[0]); list(l)
[DataNode(0), DataNode(2), DataNode(1)]
>>> ds[3].insert_before(ds[0]); list(l)
[DataNode(3), DataNode(0), DataNode(2), DataNode(1)]
>>> ds[4].insert_after(ds[1]); list(l)
[DataNode(3), DataNode(0), DataNode(2), DataNode(1), DataNode(4)]
>>> ds[2].detach(); list(l)
[DataNode(3), DataNode(0), DataNode(1), DataNode(4)]
>>> ds[2].insert_after(ds[4]); list(l)
[DataNode(3), DataNode(0), DataNode(1), DataNode(4), DataNode(2)]
>>> ds[3].detach();
>>> list(l)
[DataNode(0), DataNode(1), DataNode(4), DataNode(2)]
>>> ds[4].detach(); list(l)
[DataNode(0), DataNode(1), DataNode(2)]
>>> ds[1].detach(); list(l)
[DataNode(0), DataNode(2)]
>>> ds[0].detach(); list(l)
[DataNode(2)]
>>> ds[2].detach(); list(l)
[]


"""

__all__ = ['BaseDataNode', 'DataNode', 'LinkedList']

from weakref import ref as wr

class _Node(object):
    __slots__ = ['_prev_wr',
                 '_next_wr',
                 '__weakref__',
                 ]

    @property
    def prev(self):
        return self._prev_wr()

    @property
    def next(self):
        return self._next_wr()

class BaseDataNode(_Node):
    __slots__ = []

    def __init__(self):
        self._prev_wr = None
        self._next_wr = None

    def insert_after(self, node):
        assert self._prev_wr is None and self._next_wr is None
        self._prev_wr = wr(node)
        self._next_wr = node._next_wr
        my_wr = wr(self)
        self.next._prev_wr = my_wr
        node._next_wr = my_wr

    def insert_before(self, node):
        self.insert_after(node.prev)

    def detach(self):
        assert self._prev_wr is not None and self._next_wr is not None
        self.prev._next_wr = self._next_wr
        self.next._prev_wr = self._prev_wr
        self._prev_wr = None
        self._next_wr = None

    @property
    def is_attached(self):
        assert (self._prev_wr is None) == (self._next_wr is None)
        return self._prev_wr is not None

class DataNode(BaseDataNode):
    __slots__ = ['_data']

    def __init__(self, data=None):
        BaseDataNode.__init__(self)
        self._data = data

    @property
    def data(self):
        return self._data

    def __repr__(self):
        return '%s(%r)' % (self.__class__.__name__, self._data)

class LinkedList(_Node):
    __slots__ = []

    def __init__(self):
        my_wr = wr(self)
        self._prev_wr = my_wr
        self._next_wr = my_wr

    def __iter__(self):
        return self.datanodes()

    def datanodes(self):
        current = self.next
        while current is not self:
            yield current
            current = current.next

    def reveresed_datanodes(self):
        current = self.prev
        while current is not self:
            yield current
            current = current.prev
