# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
A dict object with inheritance.
It behaves mostly like the built-in dict:

>>> a = DerivedDict()
>>> a['a'] = 5
>>> a.update((('b',10),('c','x')))
>>> a.items()
[('a', 5), ('b', 10), ('c', 'x')]
>>> a.keys()
['a', 'b', 'c']
>>> a.values()
[5, 10, 'x']
>>> del a['b']
>>> a['c'] = 'X'
>>> a.items()
[('a', 5), ('c', 'X')]
>>> list(a)
['a', 'c']
>>> list(a.iteritems())
[('a', 5), ('c', 'X')]
>>> list(a.itervalues())
[5, 'X']

But it has support for being derived from other DerivedDict instances.
When creating a new instance, pass a sequence of fathers (will be converted into a tuple anyway).
The value for a given key is taken from the internal dict first.
If the internal dict does not contain the key, the fathers are asked for the value for the key.

>>> a = DerivedDict()
>>> b = DerivedDict(derivable_fathers = (a,))
>>> c = DerivedDict()
>>> d = DerivedDict(derivable_fathers = (b,c))
>>> a['a'] = 'A'
>>> d.items()
[('a', 'A')]
>>> c['a'] = 'C'
>>> d.items()
[('a', 'A')]
>>> b['a'] = 'B'
>>> d.items()
[('a', 'B')]
>>> del a['a']
>>> del b['a']
>>> d.items()
[('a', 'C')]
>>> d.get('a')
'C'
>>> d.get('b', 5)
5

Iteration works like DerivedSet's one.

>>> a = DerivedDict()
>>> b = DerivedDict(derivable_fathers = (a,))
>>> a.update({'a':1,'A':2})
>>> b.update({'b':3,'B':4})
>>> b.items()
[('B', 4), ('b', 3), ('a', 1), ('A', 2)]
>>> b['a'] = 100
>>> b.items()
[('B', 4), ('b', 3), ('a', 100), ('A', 2)]

Non-derivable fathers: like DerivedSet, an instance can have non-derivable fathers:

>>> a = DerivedDict()
>>> at = DerivedDict()
>>> b = DerivedDict(derivable_fathers = (a,), non_derivable_fathers = (at,))
>>> c = DerivedDict(derivable_fathers = (b,))
>>> a['a'] = 5
>>> at['at'] = 6
>>> b.items()
[('a', 5), ('at', 6)]
>>> c.items()
[('a', 5)]

A father can be a simple dict

>>> d = {1:2}
>>> a = DerivedDict(derivable_fathers = (d,))
>>> a[1]
2
>>> e = {3:4}
>>> b = DerivedDict(derivable_fathers = (a,), non_derivable_fathers = (e,))
>>> b[3]
4

DerivedSet and DerivedDict can live inside a DerivedDict and be affected by its fathers

>>> a = DerivedDict()
>>> b = DerivedDict(derivable_fathers = (a,))
>>> ib = b.derivedset('foo')
>>> a.derivedset('foo').add('a')
>>> ib.add('b')
>>> tuple(ib)
('b', 'a')
>>> a.deriveddict('bar')['a'] = 5
>>> ib = b['bar']
>>> ib['b'] = 10
>>> ib.items()
[('b', 10), ('a', 5)]
>>> c = DerivedDict(derivable_fathers = (b,))
>>> tuple(c.get('foo'))
('b', 'a')
>>> c.get('bar').items()
[('b', 10), ('a', 5)]
>>> a.derivedset('bar')
Traceback (most recent call last):
    ...
KeyError: 'bar'

DerivedDicts can be updated with the exec statment:

>>> a = DerivedDict()
>>> b = DerivedDict(derivable_fathers = (a,))
>>> a_code = '''
... foo = 5
... bar = 'a'
... DERIVEDSET('a').add(10)
... DERIVEDDICT('b')['c'] = 'd'
... '''
>>> a.update_by_exec(a_code)
>>> a.items()[:2]
[('foo', 5), ('bar', 'a')]
>>> list(a['a'])
[10]
>>> a['b'].items()
[('c', 'd')]
>>> b.items()[:2]
[('foo', 5), ('bar', 'a')]
>>> b_code = '''
... xip = foo * 30
... if 'a' == bar:
...     bar = 'A'
... DERIVEDSET('a').add(20)
... DERIVEDDICT('b')['C'] = DERIVEDDICT('b')['c'].upper()
... '''
>>> b.update_by_exec(b_code)
>>> b.items()[:2]
[('xip', 150), ('bar', 'A')]
>>> b.items()[4]
('foo', 5)
>>> list(b['a'])
[20, 10]
>>> b['b'].items()
[('C', 'D'), ('c', 'd')]

DerivedDicts are picklable

>>> import cPickle as pickle
>>> a = DerivedDict()
>>> b = DerivedDict(derivable_fathers = (a,))
>>> a['A'] = 'A'
>>> b['b'] = 'b'
>>> s = pickle.dumps(b, -1)
>>> del b
>>> del a
>>> b = pickle.loads(s)
>>> b.items()
[('b', 'b'), ('A', 'A')]

"""

from .base import DerivedBase
from .set import DerivedSet

from ..views.generator import GeneratorView
from ..ordereddict import OrderedDict
from ..executils import wrapped_exec

from abc import ABCMeta

__all__ = ['DerivedDict',
           'DerivedSetInsideDict',
           'DerivedDictInsideDict',
           'EMPTY',
           ]

class ContainedInDerivedDict(object):
    __slots__ = []
    __metaclass__ = ABCMeta

class DerivedDict(DerivedBase):
    __slots__ = []

    def __init__(self, sequence = (), derivable_fathers = (), non_derivable_fathers = (), freeze = False):
        d = OrderedDict(sequence)
        assert not any(isinstance(v, ContainedInDerivedDict) for v in d.itervalues()), 'When creating a derived dict, no initial value can be a contained-in one!'
        DerivedBase.__init__(self, d, derivable_fathers, non_derivable_fathers, freeze)

    def keys(self):
        return list(self.iterkeys())

    def values(self):
        return list(self.itervalues())

    def items(self):
        return list(self.iteritems())

    ### collections.Iterable requirements
    def __iter__(self):
        return self.iterkeys()

    def iterkeys(self):
        seen = set()
        for d in self._iter_internal_data():
            for key in d:
                if key not in seen:
                    yield key
            seen.update(d)

    def iteritems(self):
        seen = set()
        for d in self._iter_internal_data():
            for key, value in d.items():
                if key not in seen:
                    yield key, value
            seen.update(d)

    def itervalues(self):
        seen = set()
        for d in self._iter_internal_data():
            for key, value in d.items():
                if key not in seen:
                    yield value
            seen.update(d)

    def _get_at_father(self, key):
        for d in self._iter_father_internal_data():
            try:
                value = d[key]
            except KeyError:
                continue
            else:
                return value
        raise KeyError(key)

    def __getitem__(self, key):
        try:
            return self._internal_data[key]
        except KeyError:
            pass

        value = self._get_at_father(key)

        # If a father of me has a ContainedInDerivedDictMixin object,
        # I shall create a new one in my dict that derives from my fathers' ContainedInDerivedDictMixin objects.
        if isinstance(value, ContainedInDerivedDict):
            value = value.__class__(self, key)
            self._internal_data[key] = value
        return value

    def get(self, key, default = None):
        try:
            return self._internal_data[key]
        except KeyError:
            pass

        try:
            value = self._get_at_father(key)
        except KeyError:
            return default

        # If a father of me has a ContainedInDerivedDictMixin object,
        # I shall create a new one in my dict that derives from my fathers' ContainedInDerivedDictMixin objects.
        if isinstance(value, ContainedInDerivedDict):
            value = value.__class__(self, key)
            self._internal_data[key] = value
        return value
        
    def __setitem__(self, key, value):
        assert not self.is_frozen
        self._internal_data[key] = value

    def __delitem__(self, key):
        assert not self.is_frozen
        del self._internal_data[key]

    def derivedset(self, key):
        return self._get_derivedinside(key, DerivedSetInsideDict)

    def deriveddict(self, key):
        return self._get_derivedinside(key, DerivedDictInsideDict)

    def _get_derivedinside(self, key, inside_type):
        try:
            value = self[key]
        except KeyError:
            assert not self.is_frozen
            value = inside_type(self, key)
            self._internal_data[key] = value
        else:
            if not isinstance(value, inside_type):
                raise KeyError(key)
        return value

    @property
    def execution_reaodonly_parts(self):
        return ({'DERIVEDSET': self.derivedset, 'DERIVEDDICT' : self.deriveddict},)

    def update_by_exec(self, executed, readonly_parts = ()):
        # it is o.k. to exec even if frozen, just don't set anything
        wrapped_exec(executed, self,
                     self.execution_reaodonly_parts + readonly_parts)

def _contained_in_derived_dict_derivable_fathers_generator(container, key):
    for father in container._derivable_fathers:
        value = father.get(key)
        if value is not None:
            yield value

def _contained_in_derived_dict_derivable_fathers(container, key):
    return GeneratorView(_contained_in_derived_dict_derivable_fathers_generator, container, key)

def _contained_in_derived_dict_non_derivable_fathers_generator(container, key):
    for father in container._non_derivable_fathers:
        value = father.get(key)
        if value is not None:
            yield value

def _contained_in_derived_dict_non_derivable_fathers(container, key):
    return GeneratorView(_contained_in_derived_dict_non_derivable_fathers_generator, container, key)

class DerivedSetInsideDict(DerivedSet):
    __slots__ = ['_container']
    def __init__(self, container, key, sequence = ()):
        self._container = container
        container[key] = self
        DerivedSet.__init__(self,
                            sequence = sequence,
                            derivable_fathers = _contained_in_derived_dict_derivable_fathers(container, key),
                            non_derivable_fathers = _contained_in_derived_dict_non_derivable_fathers(container, key),
                            )

class DerivedDictInsideDict(DerivedDict):
    __slots__ = ['_container']
    def __init__(self, container, key, sequence = ()):
        self._container = container
        container[key] = self
        DerivedDict.__init__(self,
                             sequence = sequence,
                             derivable_fathers = _contained_in_derived_dict_derivable_fathers(container, key),
                             non_derivable_fathers = _contained_in_derived_dict_non_derivable_fathers(container, key),
                             )

ContainedInDerivedDict.register(DerivedSetInsideDict)
ContainedInDerivedDict.register(DerivedDictInsideDict)

EMPTY = DerivedDict({}, freeze = True)
