# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
Cache for functions with hashable arguments.

>>> def foo(x,y):
...     print x, y
...     return x + y
... 
>>> hashed_foo = HashableFuncCache(foo)
>>> hashed_foo(1,2)
1 2
3
>>> hashed_foo(1,2)
3
>>> hashed_foo(3,4)
3 4
7
>>> hashed_foo(5,6)
5 6
11
>>> hashed_foo(3,4)
7
>>> hashed_foo.reset()
>>> hashed_foo(3,4)
3 4
7
>>> hashed_foo.remember(8, 3, 4)
>>> hashed_foo(3, 4)
8
>>> hashed_foo.remember_new(10, 4, 5)
>>> hashed_foo.remember_new(11, 4, 5)
Traceback (most recent call last):
    ...
Exception: Cannot re-remember new result
>>> hashed_foo(4, 5)
10
>>> hashed_foo.forget(4, 5)
>>> hashed_foo.forget(4, 5)
>>> hashed_foo(4, 5)
4 5
9
>>> 'HashableFuncCache(<function %s at 0x%x>)' % (foo.func_name, id(foo)) == repr(hashed_foo)
True

As decorator:

>>> @hashable_cached
... def foo(x,y):
...     print x, y
...     return x + y
... 
>>> foo(1,2)
1 2
3
>>> foo(1,2)
3
>>> foo(3,4)
3 4
7
>>> foo(5,6)
5 6
11
>>> foo(3,4)
7
>>> foo.reset()
>>> foo(3,4)
3 4
7

"""

import functools

__all__ = [
    'HashableFuncCache', 'hashable_cached',
    'HashableNonNullFuncCache', 'hashable_non_null_cached',
    ]

from .base import BaseCache

class FuncCache(BaseCache):

    __slots__ = ['_func']

    def __init__(self, func):
        BaseCache.__init__(self)
        self._func = func

    def __repr__(self):
        return '%s(%r)' % (self.__class__.__name__, self._func)

class __NOVALUE__(object):
    __slots__ = []

__NOVALUE__ = __NOVALUE__()

class HashableFuncCache(FuncCache):
    __slots__ = ['_cache','_func']

    def __call__(self, *args, **kws):
        key = (args, tuple(kws.items()))
        result = self._cache.get(key, __NOVALUE__)
        if result is __NOVALUE__:
            result = self._func(*args, **kws)
            self._cache[key] = result
        return result

    def remember(self, result, *args, **kws):
        key = (args, tuple(kws.items()))
        self._cache[key] = result

    def remember_new(self, result, *args, **kws):
        key = (args, tuple(kws.items()))
        if key in self._cache:
            raise Exception('Cannot re-remember new result')
        self._cache[key] = result
        
    def forget(self, *args, **kws):
        key = (args, tuple(kws.items()))
        try:
            del self._cache[key]
        except KeyError:
            pass

def hashable_cached(func):
    _cache = HashableNonNullFuncCache(func)
    def wrapper(*args, **kws):
        return _cache(*args, **kws)
    functools.update_wrapper(wrapper, func)
    wrapper.reset = _cache.reset
    return wrapper

# backward compatability
class HashableNonNullFuncCache(HashableFuncCache): pass
hashable_non_null_cached = hashable_cached
