"""
This module contains the pdbParser database and methods related to the database information extraction.
"""
# standard libraries imports
import sys
from collections import OrderedDict

# external libraries imports
import numpy as np

# pdbParser library imports
from pdbParser.log import Logger

def is_element(element):
    """
    Check whether element is defined in database
    
    :Parameters:
        #. element (key): The database key dictionary key.
        
    :Returns:
        #. result (boolean): True if element exist, False for otherwise
    """
    assert isinstance(element, basestring), Logger.error("element must be a string") 
    element = str(element)
    return element.lower() in __atoms_database__.keys()

def is_element_property(prop):
    """
    Check whether property is defined in database
    
    :Parameters:
        #. prop (key): The database dictionary values key dictionary key.
        
    :Returns:
        #. result (boolean): True if property exist, False for otherwise
    """
    return prop in __atoms_database__['h'].keys()
    
def get_element_property(element, prop):
    """
    Check whether property is defined in database
    
    :Parameters:
        #. element (key): The database element dictionary key.
        #. prop (key): The database dictionary values property dictionary key.
        
    :Returns:
        #. value (object): The proerty value.
    """
    assert is_element(element), Logger.error("element '%s' don't exist in database"%element) 
    assert is_element_property(prop), Logger.error("property '%s' don't exist in database"%prop) 
    return __atoms_database__[element.lower()][prop]


# constants
__interMolecularMinimumDistance__ = 2.0 # A
__avogadroNumber__ = 6.0221415e+23 # mol
__mol__ = __avogadroNumber__
__eps0__ = 8.854187817e-12 # farads per meter (F/m) or in C^2 N^-1 M^-2
__coulombConstant__ = 8.9875517873681764e+9 # =1/(4Pi*eps0) in N m^2 C^-2
__elementaryCharge__ = 1.602176487e-19 # coulomb
__boltzmannConstant__ = 1.3806488e-23 #J*K^-1


# __atoms_database__
__atoms_database__ = OrderedDict()
__atoms_database__["h"]   = {'name':'Hydrogen',       'symbol':'H',  'group':1,  'atomicNumber':1,   'neutronCohb':-3.7390,        'neutronIncohb':None,         'neutronCohXs':1.7568,    'neutronIncohXs':80.26,   'atomicWeight':1.008,    'covalentRadius':0.32,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 0.489918, 'b4': 2.20159, 'a3': 0.196767, 'a2': 0.262003, 'a4': 0.049879, 'b2': 7.74039, 'b3': 49.5519, 'c': 0.001305, 'b1': 20.6593}, -1: {'a1': 0.897661, 'b4': 3.56709, 'a3': 0.415815, 'a2': 0.565616, 'a4': 0.116973, 'b2': 15.187, 'b3': 186.576, 'c': 0.002389, 'b1': 53.1368}}}
__atoms_database__["he"]  = {'name':'Helium',         'symbol':'He', 'group':18, 'atomicNumber':2,   'neutronCohb':3.26,           'neutronIncohb':None,         'neutronCohXs':1.34,      'neutronIncohXs':0,       'atomicWeight':4.003,    'covalentRadius':46,     'equal':1.0,  'atomicFormFactor':{0: {'a1': 0.8734, 'b4': 0.9821, 'a3': 0.3112, 'a2': 0.6309, 'a4': 0.178, 'b2': 3.3568, 'b3': 22.9276, 'c': 0.0064, 'b1': 9.1037}}}
__atoms_database__["li"]  = {'name':'Lithium',        'symbol':'Li', 'group':1,  'atomicNumber':3,   'neutronCohb':-1.90,          'neutronIncohb':None,         'neutronCohXs':0.454,     'neutronIncohXs':0.92,    'atomicWeight':6.941,    'covalentRadius':1.33,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 1.1282, 'b4': 168.261, 'a3': 0.6175, 'a2': 0.7508, 'a4': 0.4653, 'b2': 1.0524, 'b3': 85.3905, 'c': 0.0377, 'b1': 3.9546}, 1: {'a1': 0.6968, 'b4': 10.0953, 'a3': 0.3414, 'a2': 0.7888, 'a4': 0.1563, 'b2': 1.9557, 'b3': 0.6316, 'c': 0.0167, 'b1': 4.6237}}}
__atoms_database__["be"]  = {'name':'Beryllium',      'symbol':'Be', 'group':2,  'atomicNumber':4,   'neutronCohb':7.79,           'neutronIncohb':0.12,         'neutronCohXs':7.63,      'neutronIncohXs':0.0018,  'atomicWeight':9.012,    'covalentRadius':1.02,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 1.5919, 'b4': 0.542, 'a3': 0.5391, 'a2': 1.1278, 'a4': 0.7029, 'b2': 1.8623, 'b3': 103.483, 'c': 0.0385, 'b1': 43.6427}, 2: {'a1': 6.2603, 'b4': 5.1146, 'a3': 0.7993, 'a2': 0.8849, 'a4': 0.1647, 'b2': 0.8313, 'b3': 2.2758, 'c': -6.1092, 'b1': 0.0027}}}
__atoms_database__["b"]   = {'name':'Boron',          'symbol':'B',  'group':13, 'atomicNumber':5,   'neutronCohb':-0.1-1.066j,    'neutronIncohb':-4.7+1.231j,  'neutronCohXs':0.144,     'neutronIncohXs':3,       'atomicWeight':10.811,   'covalentRadius':0.85,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 2.0545, 'b4': 0.1403, 'a3': 1.0979, 'a2': 1.3326, 'a4': 0.7068, 'b2': 1.021, 'b3': 60.3498, 'c': -0.1932, 'b1': 23.2185}}}
__atoms_database__["c"]   = {'name':'Carbon',         'symbol':'C',  'group':14, 'atomicNumber':6,   'neutronCohb':6.6460,         'neutronIncohb':None,         'neutronCohXs':5.551,     'neutronIncohXs':0.001,   'atomicWeight':12.011,   'covalentRadius':0.75,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 2.31, 'b4': 51.6512, 'a3': 1.5886, 'a2': 1.02, 'a4': 0.865, 'b2': 10.2075, 'b3': 0.5687, 'c': 0.2156, 'b1': 20.8439}, 'val': {'a1': 2.26069, 'b4': 55.5949, 'a3': 1.05075, 'a2': 1.56165, 'a4': 0.839259, 'b2': 0.656665, 'b3': 9.75618, 'c': 0.286977, 'b1': 22.6907}}}
__atoms_database__["n"]   = {'name':'Nitrogen',       'symbol':'N',  'group':15, 'atomicNumber':7,   'neutronCohb':9.36,           'neutronIncohb':None,         'neutronCohXs':11.01,     'neutronIncohXs':0.5,     'atomicWeight':14.007,   'covalentRadius':0.71,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 12.2126, 'b4': 0.5826, 'a3': 2.0125, 'a2': 3.1322, 'a4': 1.1663, 'b2': 9.8933, 'b3': 28.9975, 'c': -11.529, 'b1': 0.0057}}}
__atoms_database__["o"]   = {'name':'Oxygen',         'symbol':'O',  'group':16, 'atomicNumber':8,   'neutronCohb':5.803,          'neutronIncohb':None,         'neutronCohXs':4.232,     'neutronIncohXs':0.0008,  'atomicWeight':15.999,   'covalentRadius':0.63,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 3.0485, 'b4': 32.9089, 'a3': 1.5463, 'a2': 2.2868, 'a4': 0.867, 'b2': 5.7011, 'b3': 0.3239, 'c': 0.2508, 'b1': 13.2771}, -1: {'a1': 4.1916, 'b4': -0.01404, 'a3': 1.52673, 'a2': 1.63969, 'a4': -20.307, 'b2': 4.17236, 'b3': 47.0179, 'c': 21.9412, 'b1': 12.8573}}}
__atoms_database__["f"]   = {'name':'Fluorine',       'symbol':'F',  'group':17, 'atomicNumber':9,   'neutronCohb':5.654,          'neutronIncohb':-0.082,       'neutronCohXs':4.017,     'neutronIncohXs':0.0008,  'atomicWeight':18.998,   'covalentRadius':0.64,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 3.5392, 'b4': 26.1476, 'a3': 1.517, 'a2': 2.6412, 'a4': 1.0243, 'b2': 4.2944, 'b3': 0.2615, 'c': 0.2776, 'b1': 10.2825}, -1: {'a1': 3.6322, 'b4': 47.3437, 'a3': 1.26064, 'a2': 3.51057, 'a4': 0.940706, 'b2': 14.7353, 'b3': 0.442258, 'c': 0.653396, 'b1': 5.27756}}}
__atoms_database__["ne"]  = {'name':'Neon',           'symbol':'Ne', 'group':18, 'atomicNumber':10,  'neutronCohb':4.566,          'neutronIncohb':None,         'neutronCohXs':2.62,      'neutronIncohXs':0.008,   'atomicWeight':20.18,    'covalentRadius':0.96,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 3.9553, 'b4': 21.7184, 'a3': 1.4546, 'a2': 3.1125, 'a4': 1.1251, 'b2': 3.4262, 'b3': 0.2306, 'c': 0.3515, 'b1': 8.4042}}}
__atoms_database__["na"]  = {'name':'Sodium',         'symbol':'Na', 'group':1,  'atomicNumber':11,  'neutronCohb':3.63,           'neutronIncohb':3.59,         'neutronCohXs':1.66,      'neutronIncohXs':1.62,    'atomicWeight':22.991,   'covalentRadius':1.6,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 4.7626, 'b4': 129.424, 'a3': 1.2674, 'a2': 3.1736, 'a4': 1.1128, 'b2': 8.8422, 'b3': 0.3136, 'c': 0.676, 'b1': 3.285}, 1: {'a1': 3.2565, 'b4': 14.039, 'a3': 1.3998, 'a2': 3.9362, 'a4': 1.0032, 'b2': 6.1153, 'b3': 0.2001, 'c': 0.404, 'b1': 2.6671}}}
__atoms_database__["mg"]  = {'name':'Magnesium',      'symbol':'Mg', 'group':2,  'atomicNumber':12,  'neutronCohb':5.375,          'neutronIncohb':None,         'neutronCohXs':3.631,     'neutronIncohXs':0.08,    'atomicWeight':24.305,   'covalentRadius':1.39,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 5.4204, 'b4': 7.1937, 'a3': 1.2269, 'a2': 2.1735, 'a4': 2.3073, 'b2': 79.2611, 'b3': 0.3808, 'c': 0.8584, 'b1': 2.8275}, 2: {'a1': 3.4988, 'b4': 10.1411, 'a3': 1.3284, 'a2': 3.8378, 'a4': 0.8497, 'b2': 4.7542, 'b3': 0.185, 'c': 0.4853, 'b1': 2.1676}}}
__atoms_database__["al"]  = {'name':'Aluminium',      'symbol':'Al', 'group':13, 'atomicNumber':13,  'neutronCohb':3.449,          'neutronIncohb':0.256,        'neutronCohXs':1.495,     'neutronIncohXs':0.0082,  'atomicWeight':26.982,   'covalentRadius':1.26,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 6.4202, 'b4': 85.0886, 'a3': 1.5936, 'a2': 1.9002, 'a4': 1.9646, 'b2': 0.7426, 'b3': 31.5472, 'c': 1.1151, 'b1': 3.0387}, 3: {'a1': 4.17448, 'b4': 8.28524, 'a3': 1.20296, 'a2': 3.3876, 'a4': 0.528137, 'b2': 4.14553, 'b3': 0.228753, 'c': 0.706786, 'b1': 1.93816}}}
__atoms_database__["si"]  = {'name':'Silicon',        'symbol':'Si', 'group':14, 'atomicNumber':14,  'neutronCohb':4.1491,         'neutronIncohb':None,         'neutronCohXs':2.163,     'neutronIncohXs':0.004,   'atomicWeight':28.086,   'covalentRadius':1.16,   'equal':1.0,  'atomicFormFactor':{4: {'a1': 4.43918, 'b4': 6.65365, 'a3': 1.19453, 'a2': 3.20345, 'a4': 0.41653, 'b2': 3.43757, 'b3': 0.2149, 'c': 0.746297, 'b1': 1.64167}, 'val': {'a1': 5.66269, 'b4': 93.5458, 'a3': 2.62446, 'a2': 3.07164, 'a4': 1.3932, 'b2': 38.6634, 'b3': 0.916946, 'c': 1.24707, 'b1': 2.6652}, 'v': {'a1': 6.2915, 'b4': 81.6937, 'a3': 1.9891, 'a2': 3.0353, 'a4': 1.541, 'b2': 32.3337, 'b3': 0.6785, 'c': 1.1407, 'b1': 2.4386}}}
__atoms_database__["p"]   = {'name':'Phosphorus',     'symbol':'P',  'group':15, 'atomicNumber':15,  'neutronCohb':5.13,           'neutronIncohb':0.2,          'neutronCohXs':3.307,     'neutronIncohXs':0.005,   'atomicWeight':30.974,   'covalentRadius':1.11,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 6.4345, 'b4': 68.1645, 'a3': 1.78, 'a2': 4.1791, 'a4': 1.4908, 'b2': 27.157, 'b3': 0.526, 'c': 1.1149, 'b1': 1.9067}}}
__atoms_database__["s"]   = {'name':'Sulphur',        'symbol':'S',  'group':16, 'atomicNumber':16,  'neutronCohb':2.847,          'neutronIncohb':None,         'neutronCohXs':1.0186,    'neutronIncohXs':0.007,   'atomicWeight':32.066,   'covalentRadius':1.03,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 6.9053, 'b4': 56.172, 'a3': 1.4379, 'a2': 5.2034, 'a4': 1.5863, 'b2': 22.2151, 'b3': 0.2536, 'c': 0.8669, 'b1': 1.4679}}}
__atoms_database__["cl"]  = {'name':'Chlorine',       'symbol':'Cl', 'group':17, 'atomicNumber':17,  'neutronCohb':9.5770,         'neutronIncohb':None,         'neutronCohXs':11.5257,   'neutronIncohXs':5.3,     'atomicWeight':35.453,   'covalentRadius':0.99,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 11.4604, 'b4': 47.7784, 'a3': 6.2556, 'a2': 7.1964, 'a4': 1.6455, 'b2': 1.1662, 'b3': 18.5194, 'c': -9.5574, 'b1': 0.0104}, -1: {'a1': 18.2915, 'b4': 60.4486, 'a3': 6.5337, 'a2': 7.2084, 'a4': 2.3386, 'b2': 1.1717, 'b3': 19.5424, 'c': -16.378, 'b1': 0.0066}}}
__atoms_database__["ar"]  = {'name':'Argon',          'symbol':'Ar', 'group':18, 'atomicNumber':18,  'neutronCohb':1.909,          'neutronIncohb':None,         'neutronCohXs':0.458,     'neutronIncohXs':0.225,   'atomicWeight':39.948,   'covalentRadius':1.07,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 7.4845, 'b4': 33.3929, 'a3': 0.6539, 'a2': 6.7723, 'a4': 1.6442, 'b2': 14.8407, 'b3': 43.8983, 'c': 1.4445, 'b1': 0.9072}}}
__atoms_database__["k"]   = {'name':'Potassium',      'symbol':'K',  'group':1,  'atomicNumber':19,  'neutronCohb':3.67,           'neutronIncohb':None,         'neutronCohXs':1.69,      'neutronIncohXs':0.27,    'atomicWeight':39.098,   'covalentRadius':1.96,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 8.2186, 'b4': 41.6841, 'a3': 1.0519, 'a2': 7.4398, 'a4': 0.8659, 'b2': 0.7748, 'b3': 213.187, 'c': 1.4228, 'b1': 12.7949}, 1: {'a1': 7.9578, 'b4': 31.9128, 'a3': 6.359, 'a2': 7.4917, 'a4': 1.1915, 'b2': 0.7674, 'b3': -0.002, 'c': -4.9978, 'b1': 12.6331}}}
__atoms_database__["ca"]  = {'name':'Calcium',        'symbol':'Ca', 'group':2,  'atomicNumber':20,  'neutronCohb':4.70,           'neutronIncohb':None,         'neutronCohXs':2.78,      'neutronIncohXs':0.05,    'atomicWeight':40.078,   'covalentRadius':1.71,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 8.6266, 'b4': 178.437, 'a3': 1.5899, 'a2': 7.3873, 'a4': 1.0211, 'b2': 0.6599, 'b3': 85.7484, 'c': 1.3751, 'b1': 10.4421}, 2: {'a1': 15.6348, 'b4': 25.9905, 'a3': 8.4372, 'a2': 7.9518, 'a4': 0.8537, 'b2': 0.6089, 'b3': 10.3116, 'c': -14.875, 'b1': -0.0074}}}
__atoms_database__["sc"]  = {'name':'Scandium',       'symbol':'Sc', 'group':3,  'atomicNumber':21,  'neutronCohb':12.29,          'neutronIncohb':-6.0,         'neutronCohXs':19,        'neutronIncohXs':4.5,     'atomicWeight':44.956,   'covalentRadius':1.48,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 9.189, 'b4': 51.3531, 'a3': 1.6409, 'a2': 7.3679, 'a4': 1.468, 'b2': 0.5729, 'b3': 136.108, 'c': 1.3329, 'b1': 9.0213}, 3: {'a1': 13.4008, 'b4': 16.0662, 'a3': 1.65943, 'a2': 8.0273, 'a4': 1.57936, 'b2': 7.9629, 'b3': -0.28604, 'c': -6.6667, 'b1': 0.29854}}}
__atoms_database__["ti"]  = {'name':'Titanium',       'symbol':'Ti', 'group':4,  'atomicNumber':22,  'neutronCohb':-3.438,         'neutronIncohb':None,         'neutronCohXs':1.485,     'neutronIncohXs':2.87,    'atomicWeight':47.867,   'covalentRadius':1.36,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 9.7595, 'b4': 116.105, 'a3': 1.6991, 'a2': 7.3558, 'a4': 1.9021, 'b2': 0.5, 'b3': 35.6338, 'c': 1.2807, 'b1': 7.8508}, 2: {'a1': 9.11423, 'b4': 61.6558, 'a3': 2.2793, 'a2': 7.62174, 'a4': 0.087899, 'b2': 0.457585, 'b3': 19.5361, 'c': 0.897155, 'b1': 7.5243}, 3: {'a1': 17.7344, 'b4': 15.9768, 'a3': 5.25691, 'a2': 8.73816, 'a4': 1.92134, 'b2': 7.04716, 'b3': -0.15762, 'c': -14.652, 'b1': 0.22061}, 4: {'a1': 19.5114, 'b4': 12.9464, 'a3': 2.01341, 'a2': 8.23473, 'a4': 1.5208, 'b2': 6.67018, 'b3': -0.29263, 'c': -13.28, 'b1': 0.178847}}}
__atoms_database__["v"]   = {'name':'Vanadium',       'symbol':'V',  'group':5,  'atomicNumber':23,  'neutronCohb':-0.3824,        'neutronIncohb':None,         'neutronCohXs':0.0184,    'neutronIncohXs':5.08,    'atomicWeight':50.942,   'covalentRadius':1.34,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 10.2971, 'b4': 102.478, 'a3': 2.0703, 'a2': 7.3511, 'a4': 2.0571, 'b2': 0.4385, 'b3': 26.8938, 'c': 1.2199, 'b1': 6.8657}, 2: {'a1': 10.106, 'b4': 115.122, 'a3': 2.2884, 'a2': 7.3541, 'a4': 0.0223, 'b2': 0.4409, 'b3': 20.3004, 'c': 1.2298, 'b1': 6.8818}, 3: {'a1': 9.43141, 'b4': 63.969, 'a3': 2.15343, 'a2': 7.7419, 'a4': 0.016865, 'b2': 0.383349, 'b3': 15.1908, 'c': 0.656565, 'b1': 6.39535}, 5: {'a1': 15.6887, 'b4': 0.940464, 'a3': 2.03081, 'a2': 8.14208, 'a4': -9.576, 'b2': 5.40135, 'b3': 9.97278, 'c': 1.7143, 'b1': 0.679003}}}
__atoms_database__["cr"]  = {'name':'Chromium',       'symbol':'Cr', 'group':6,  'atomicNumber':24,  'neutronCohb':3.635,          'neutronIncohb':None,         'neutronCohXs':1.66,      'neutronIncohXs':1.83,    'atomicWeight':51.996,   'covalentRadius':1.22,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 10.6406, 'b4': 98.7399, 'a3': 3.324, 'a2': 7.3537, 'a4': 1.4922, 'b2': 0.392, 'b3': 20.2626, 'c': 1.1832, 'b1': 6.1038}, 2: {'a1': 9.54034, 'b4': 32.4224, 'a3': 3.58274, 'a2': 7.7509, 'a4': 0.509107, 'b2': 0.344261, 'b3': 13.3075, 'c': 0.616898, 'b1': 5.66078}, 3: {'a1': 9.6809, 'b4': 32.8761, 'a3': 2.87603, 'a2': 7.81136, 'a4': 0.113575, 'b2': 0.334393, 'b3': 12.8288, 'c': 0.518275, 'b1': 5.59463}}}
__atoms_database__["mn"]  = {'name':'Manganese',      'symbol':'Mn', 'group':7,  'atomicNumber':25,  'neutronCohb':-3.73,          'neutronIncohb':1.79,         'neutronCohXs':1.75,      'neutronIncohXs':0.4,     'atomicWeight':54.938,   'covalentRadius':1.19,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 11.2819, 'b4': 83.7543, 'a3': 3.0193, 'a2': 7.3573, 'a4': 2.2441, 'b2': 0.3432, 'b3': 17.8674, 'c': 1.0896, 'b1': 5.3409}, 2: {'a1': 10.8061, 'b4': 41.3235, 'a3': 3.5268, 'a2': 7.362, 'a4': 0.2184, 'b2': 0.3435, 'b3': 14.343, 'c': 1.0874, 'b1': 5.2796}, 3: {'a1': 9.84521, 'b4': 24.1281, 'a3': 3.56531, 'a2': 7.87194, 'a4': 0.323613, 'b2': 0.294393, 'b3': 10.8171, 'c': 0.393974, 'b1': 4.91797}, 4: {'a1': 9.96253, 'b4': 27.573, 'a3': 2.76067, 'a2': 7.97057, 'a4': 0.054447, 'b2': 0.283303, 'b3': 10.4852, 'c': 0.251877, 'b1': 4.8485}}}
__atoms_database__["fe"]  = {'name':'Iron',           'symbol':'Fe', 'group':8,  'atomicNumber':26,  'neutronCohb':9.45,           'neutronIncohb':None,         'neutronCohXs':11.22,     'neutronIncohXs':0.4,     'atomicWeight':55.845,   'covalentRadius':1.16,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 11.7695, 'b4': 76.8805, 'a3': 3.5222, 'a2': 7.3573, 'a4': 2.3045, 'b2': 0.3072, 'b3': 15.3535, 'c': 1.0369, 'b1': 4.7611}, 2: {'a1': 11.0424, 'b4': 31.2809, 'a3': 4.1346, 'a2': 7.374, 'a4': 0.4399, 'b2': 0.3053, 'b3': 12.0546, 'c': 1.0097, 'b1': 4.6538}, 3: {'a1': 11.1764, 'b4': 38.5566, 'a3': 3.3948, 'a2': 7.3863, 'a4': 0.0724, 'b2': 0.3005, 'b3': 11.6729, 'c': 0.9707, 'b1': 4.6147}}}
__atoms_database__["co"]  = {'name':'Cobalt',         'symbol':'Co', 'group':9,  'atomicNumber':27,  'neutronCohb':2.49,           'neutronIncohb':-6.2,         'neutronCohXs':0.779,     'neutronIncohXs':4.8,     'atomicWeight':58.933,   'covalentRadius':1.11,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 12.2841, 'b4': 71.1692, 'a3': 4.0034, 'a2': 7.3409, 'a4': 2.3488, 'b2': 0.2784, 'b3': 13.5359, 'c': 1.0118, 'b1': 4.2791}, 2: {'a1': 11.2296, 'b4': 25.6466, 'a3': 4.7393, 'a2': 7.3883, 'a4': 0.7108, 'b2': 0.2726, 'b3': 10.2443, 'c': 0.9324, 'b1': 4.1231}, 3: {'a1': 10.338, 'b4': 18.3491, 'a3': 4.76795, 'a2': 7.88173, 'a4': 0.725591, 'b2': 0.238668, 'b3': 8.35583, 'c': 0.286667, 'b1': 3.90969}}}
__atoms_database__["ni"]  = {'name':'Nickel',         'symbol':'Ni', 'group':10, 'atomicNumber':28,  'neutronCohb':10.3,           'neutronIncohb':None,         'neutronCohXs':13.3,      'neutronIncohXs':5.2,     'atomicWeight':58.693,   'covalentRadius':1.1,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 12.8376, 'b4': 66.3421, 'a3': 4.4438, 'a2': 7.292, 'a4': 2.38, 'b2': 0.2565, 'b3': 12.1763, 'c': 1.0341, 'b1': 3.8785}, 2: {'a1': 11.4166, 'b4': 22.1626, 'a3': 5.3442, 'a2': 7.4005, 'a4': 0.9773, 'b2': 0.2449, 'b3': 8.873, 'c': 0.8614, 'b1': 3.6766}, 3: {'a1': 10.7806, 'b4': 16.9673, 'a3': 5.22746, 'a2': 7.75868, 'a4': 0.847114, 'b2': 0.22314, 'b3': 7.64468, 'c': 0.386044, 'b1': 3.5477}}}
__atoms_database__["cu"]  = {'name':'Copper',         'symbol':'Cu', 'group':11, 'atomicNumber':29,  'neutronCohb':7.718,          'neutronIncohb':None,         'neutronCohXs':7.485,     'neutronIncohXs':0.55,    'atomicWeight':63.546,   'covalentRadius':1.2,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 13.338, 'b4': 64.8126, 'a3': 5.6158, 'a2': 7.1676, 'a4': 1.6735, 'b2': 0.247, 'b3': 11.3966, 'c': 1.191, 'b1': 3.5828}, 1: {'a1': 11.9475, 'b4': 25.8487, 'a3': 6.2455, 'a2': 7.3573, 'a4': 1.5578, 'b2': 0.2274, 'b3': 8.6625, 'c': 0.89, 'b1': 3.3669}, 2: {'a1': 11.8168, 'b4': 19.897, 'a3': 5.78135, 'a2': 7.11181, 'a4': 1.14523, 'b2': 0.244078, 'b3': 7.9876, 'c': 1.14431, 'b1': 3.37484}}}
__atoms_database__["zn"]  = {'name':'Zinc',           'symbol':'Zn', 'group':12, 'atomicNumber':30,  'neutronCohb':5.680,          'neutronIncohb':None,         'neutronCohXs':4.054,     'neutronIncohXs':0.077,   'atomicWeight':65.39,    'covalentRadius':1.2,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 14.0743, 'b4': 58.7097, 'a3': 5.1652, 'a2': 7.0318, 'a4': 2.41, 'b2': 0.2333, 'b3': 10.3163, 'c': 1.3041, 'b1': 3.2655}, 2: {'a1': 11.9719, 'b4': 18.0995, 'a3': 6.4668, 'a2': 7.3862, 'a4': 1.394, 'b2': 0.2031, 'b3': 7.0826, 'c': 0.7807, 'b1': 2.9946}}}
__atoms_database__["ga"]  = {'name':'Gallium',        'symbol':'Ga', 'group':13, 'atomicNumber':31,  'neutronCohb':7.288,          'neutronIncohb':None,         'neutronCohXs':6.675,     'neutronIncohXs':0.16,    'atomicWeight':69.723,   'covalentRadius':1.24,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 15.2354, 'b4': 61.4135, 'a3': 4.3591, 'a2': 6.7006, 'a4': 2.9623, 'b2': 0.2412, 'b3': 10.7805, 'c': 1.7189, 'b1': 3.0669}, 3: {'a1': 12.692, 'b4': 14.4122, 'a3': 6.06692, 'a2': 6.69883, 'a4': 1.0066, 'b2': 0.22789, 'b3': 6.36441, 'c': 1.53545, 'b1': 2.81262}}}
__atoms_database__["ge"]  = {'name':'Germanium',      'symbol':'Ge', 'group':14, 'atomicNumber':32,  'neutronCohb':8.185,          'neutronIncohb':None,         'neutronCohXs':8.42,      'neutronIncohXs':0.18,    'atomicWeight':72.61,    'covalentRadius':1.21,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 16.0816, 'b4': 54.7625, 'a3': 3.7068, 'a2': 6.3747, 'a4': 3.683, 'b2': 0.2516, 'b3': 11.4468, 'c': 2.1313, 'b1': 2.8509}, 4: {'a1': 12.9172, 'b4': 11.603, 'a3': 6.06791, 'a2': 6.70003, 'a4': 0.859041, 'b2': 0.205855, 'b3': 5.47913, 'c': 1.45572, 'b1': 2.53718}}}
__atoms_database__["as"]  = {'name':'Arsenic',        'symbol':'As', 'group':15, 'atomicNumber':33,  'neutronCohb':6.58,           'neutronIncohb':-0.69,        'neutronCohXs':5.44,      'neutronIncohXs':0.06,    'atomicWeight':74.922,   'covalentRadius':1.21,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 16.6723, 'b4': 47.7972, 'a3': 3.4313, 'a2': 6.0701, 'a4': 4.2779, 'b2': 0.2647, 'b3': 12.9479, 'c': 2.531, 'b1': 2.6345}}}
__atoms_database__["se"]  = {'name':'Selenium',       'symbol':'Se', 'group':16, 'atomicNumber':34,  'neutronCohb':7.970,          'neutronIncohb':None,         'neutronCohXs':7.98,      'neutronIncohXs':0.32,    'atomicWeight':78.96,    'covalentRadius':1.16,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 17.0006, 'b4': 43.8163, 'a3': 3.9731, 'a2': 5.8196, 'a4': 4.3543, 'b2': 0.2726, 'b3': 15.2372, 'c': 2.8409, 'b1': 2.4098}}}
__atoms_database__["br"]  = {'name':'Bromine',        'symbol':'Br', 'group':17, 'atomicNumber':35,  'neutronCohb':6.795,          'neutronIncohb':None,         'neutronCohXs':5.8,       'neutronIncohXs':0.1,     'atomicWeight':79.904,   'covalentRadius':1.21,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 17.1789, 'b4': 41.4328, 'a3': 5.6377, 'a2': 5.2358, 'a4': 3.9851, 'b2': 16.5796, 'b3': 0.2609, 'c': 2.9557, 'b1': 2.1723}, -1: {'a1': 17.1718, 'b4': 58.1535, 'a3': 5.5754, 'a2': 6.3338, 'a4': 3.7272, 'b2': 19.3345, 'b3': 0.2871, 'c': 3.1776, 'b1': 2.2059}}}
__atoms_database__["kr"]  = {'name':'Krypton',        'symbol':'Kr', 'group':18, 'atomicNumber':36,  'neutronCohb':7.81,           'neutronIncohb':None,         'neutronCohXs':7.67,      'neutronIncohXs':0.01,    'atomicWeight':83.8,     'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 17.3555, 'b4': 39.3972, 'a3': 5.5493, 'a2': 6.7286, 'a4': 3.5375, 'b2': 16.5623, 'b3': 0.2261, 'c': 2.825, 'b1': 1.9384}}}
__atoms_database__["rb"]  = {'name':'Rubidium',       'symbol':'Rb', 'group':1,  'atomicNumber':37,  'neutronCohb':7.09,           'neutronIncohb':None,         'neutronCohXs':6.32,      'neutronIncohXs':0.5,     'atomicWeight':85.468,   'covalentRadius':2.2,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 17.1784, 'b4': 164.934, 'a3': 5.1399, 'a2': 9.6435, 'a4': 1.5292, 'b2': 17.3151, 'b3': 0.2748, 'c': 3.4873, 'b1': 1.7888}, 1: {'a1': 17.5816, 'b4': 31.2087, 'a3': 5.8981, 'a2': 7.6598, 'a4': 2.7817, 'b2': 14.7957, 'b3': 0.1603, 'c': 2.0782, 'b1': 1.7139}}}
__atoms_database__["sr"]  = {'name':'Strontium',      'symbol':'Sr', 'group':2,  'atomicNumber':38,  'neutronCohb':7.02,           'neutronIncohb':None,         'neutronCohXs':6.19,      'neutronIncohXs':0.06,    'atomicWeight':87.62,    'covalentRadius':1.95,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 17.5663, 'b4': 132.376, 'a3': 5.422, 'a2': 9.8184, 'a4': 2.6694, 'b2': 14.0988, 'b3': 0.1664, 'c': 2.5064, 'b1': 1.5564}, 2: {'a1': 18.0874, 'b4': -0.0138, 'a3': 2.5654, 'a2': 8.1373, 'a4': -34.193, 'b2': 12.6963, 'b3': 24.5651, 'c': 41.4025, 'b1': 1.4907}}}
__atoms_database__["y"]   = {'name':'Yttrium',        'symbol':'Y',  'group':3,  'atomicNumber':39,  'neutronCohb':7.75,           'neutronIncohb':1.1,          'neutronCohXs':7.55,      'neutronIncohXs':0.15,    'atomicWeight':88.906,   'covalentRadius':1.9,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 17.776, 'b4': 104.354, 'a3': 5.72629, 'a2': 10.2946, 'a4': 3.26588, 'b2': 12.8006, 'b3': 0.125599, 'c': 1.91213, 'b1': 1.4029}, 3: {'a1': 17.9268, 'b4': -0.01319, 'a3': 1.76795, 'a2': 9.1531, 'a4': -33.108, 'b2': 11.2145, 'b3': 22.6599, 'c': 40.2602, 'b1': 1.35417}}}
__atoms_database__["zr"]  = {'name':'Zirconium',      'symbol':'Zr', 'group':4,  'atomicNumber':40,  'neutronCohb':7.16,           'neutronIncohb':None,         'neutronCohXs':6.44,      'neutronIncohXs':0.02,    'atomicWeight':91.224,   'covalentRadius':1.75,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 17.8765, 'b4': 87.6627, 'a3': 5.41732, 'a2': 10.948, 'a4': 3.65721, 'b2': 11.916, 'b3': 0.117622, 'c': 2.06929, 'b1': 1.27618}, 4: {'a1': 18.1668, 'b4': -0.10276, 'a3': 1.01118, 'a2': 10.0562, 'a4': -2.6479, 'b2': 10.1483, 'b3': 21.6054, 'c': 9.41454, 'b1': 1.2148}}}
__atoms_database__["nb"]  = {'name':'Niobium',        'symbol':'Nb', 'group':5,  'atomicNumber':41,  'neutronCohb':7.054,          'neutronIncohb':-0.139,       'neutronCohXs':6.253,     'neutronIncohXs':0.0024,  'atomicWeight':92.906,   'covalentRadius':1.64,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 17.6142, 'b4': 69.7957, 'a3': 4.04183, 'a2': 12.0144, 'a4': 3.53346, 'b2': 11.766, 'b3': 0.204785, 'c': 3.75591, 'b1': 1.18865}, 3: {'a1': 19.8812, 'b4': 28.3389, 'a3': 11.0177, 'a2': 18.0653, 'a4': 1.94715, 'b2': 1.13305, 'b3': 10.1621, 'c': -12.912, 'b1': 0.019175}, 5: {'a1': 17.9163, 'b4': 25.7228, 'a3': 10.799, 'a2': 13.3417, 'a4': 0.337905, 'b2': 0.028781, 'b3': 9.28206, 'c': -6.3934, 'b1': 1.12446}}}
__atoms_database__["mo"]  = {'name':'Molybdenum',     'symbol':'Mo', 'group':6,  'atomicNumber':42,  'neutronCohb':6.715,          'neutronIncohb':None,         'neutronCohXs':5.67,      'neutronIncohXs':0.04,    'atomicWeight':95.94,    'covalentRadius':1.54,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 3.7025, 'b4': 61.6584, 'a3': 12.8876, 'a2': 17.2356, 'a4': 3.7429, 'b2': 1.0958, 'b3': 11.004, 'c': 4.3875, 'b1': 0.2772}, 3: {'a1': 21.1664, 'b4': 26.6307, 'a3': 11.7423, 'a2': 18.2017, 'a4': 2.30951, 'b2': 1.03031, 'b3': 9.53659, 'c': -14.421, 'b1': 0.014734}, 5: {'a1': 21.0149, 'b4': 23.3452, 'a3': 11.4632, 'a2': 18.0992, 'a4': 0.740625, 'b2': 1.02238, 'b3': 8.78809, 'c': -14.316, 'b1': 0.014345}, 6: {'a1': 17.8871, 'b4': 0.0, 'a3': 6.57891, 'a2': 11.175, 'a4': 0.0, 'b2': 8.48061, 'b3': 0.058881, 'c': 0.344941, 'b1': 1.03649}}}
__atoms_database__["tc"]  = {'name':'Technetium',     'symbol':'Tc', 'group':7,  'atomicNumber':43,  'neutronCohb':6.8,            'neutronIncohb':None,         'neutronCohXs':5.8,       'neutronIncohXs':0.5,     'atomicWeight':98,       'covalentRadius':1.47,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.1301, 'b4': 86.8472, 'a3': 4.64901, 'a2': 11.0948, 'a4': 2.71263, 'b2': 8.14487, 'b3': 21.5707, 'c': 5.40428, 'b1': 0.864132}}}
__atoms_database__["ru"]  = {'name':'Ruthenium',      'symbol':'Ru', 'group':8,  'atomicNumber':44,  'neutronCohb':7.03,           'neutronIncohb':None,         'neutronCohXs':6.21,      'neutronIncohXs':0.4,     'atomicWeight':101.07,   'covalentRadius':1.46,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.2674, 'b4': 94.2928, 'a3': 4.86337, 'a2': 12.9182, 'a4': 1.56756, 'b2': 8.43467, 'b3': 24.7997, 'c': 5.37874, 'b1': 0.80852}, 3: {'a1': 18.5638, 'b4': 22.887, 'a3': 9.32602, 'a2': 13.2885, 'a4': 3.00964, 'b2': 8.37164, 'b3': 0.017662, 'c': -3.1892, 'b1': 0.847329}, 4: {'a1': 18.5003, 'b4': 20.8504, 'a3': 4.71304, 'a2': 13.1787, 'a4': 2.18535, 'b2': 8.12534, 'b3': 0.36495, 'c': 1.42357, 'b1': 0.844582}}}
__atoms_database__["rh"]  = {'name':'Rhodium',        'symbol':'Rh', 'group':9,  'atomicNumber':45,  'neutronCohb':5.88,           'neutronIncohb':None,         'neutronCohXs':4.34,      'neutronIncohXs':0.3,     'atomicWeight':102.906,  'covalentRadius':1.45,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.2957, 'b4': 98.6062, 'a3': 4.73425, 'a2': 14.3501, 'a4': 1.28918, 'b2': 8.21758, 'b3': 25.8749, 'c': 5.328, 'b1': 0.751536}, 3: {'a1': 18.8785, 'b4': -0.01036, 'a3': 3.32515, 'a2': 14.1259, 'a4': -6.1989, 'b2': 7.84438, 'b3': 21.2487, 'c': 11.8678, 'b1': 0.764252}, 4: {'a1': 18.8545, 'b4': -0.0102, 'a3': 2.53464, 'a2': 13.9806, 'a4': -5.6526, 'b2': 7.62436, 'b3': 19.3317, 'c': 11.2835, 'b1': 0.760825}}}
__atoms_database__["pd"]  = {'name':'Palladium',      'symbol':'Pd', 'group':10, 'atomicNumber':46,  'neutronCohb':5.91,           'neutronIncohb':None,         'neutronCohXs':4.39,      'neutronIncohXs':0.093,   'atomicWeight':106.42,   'covalentRadius':1.39,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.3319, 'b4': 76.8986, 'a3': 5.29537, 'a2': 15.5017, 'a4': 0.605844, 'b2': 7.98929, 'b3': 25.2052, 'c': 5.26593, 'b1': 0.698655}, 2: {'a1': 19.1701, 'b4': 0.0, 'a3': 4.32234, 'a2': 15.2096, 'a4': 0.0, 'b2': 7.55573, 'b3': 22.5057, 'c': 5.2916, 'b1': 0.696219}, 4: {'a1': 19.2493, 'b4': 0.005127, 'a3': 2.89289, 'a2': 14.79, 'a4': -7.9492, 'b2': 7.14833, 'b3': 17.9144, 'c': 13.0174, 'b1': 0.683839}}}
__atoms_database__["ag"]  = {'name':'Silver',         'symbol':'Ag', 'group':11, 'atomicNumber':47,  'neutronCohb':5.922,          'neutronIncohb':None,         'neutronCohXs':4.407,     'neutronIncohXs':0.58,    'atomicWeight':107.868,  'covalentRadius':1.45,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.2808, 'b4': 99.8156, 'a3': 4.8045, 'a2': 16.6885, 'a4': 1.0463, 'b2': 7.4726, 'b3': 24.6605, 'c': 5.179, 'b1': 0.6446}, 1: {'a1': 19.1812, 'b4': 66.1147, 'a3': 5.27475, 'a2': 15.9719, 'a4': 0.357534, 'b2': 7.19123, 'b3': 21.7326, 'c': 5.21572, 'b1': 0.646179}, 2: {'a1': 19.1643, 'b4': 0.0, 'a3': 4.3709, 'a2': 16.2456, 'a4': 0.0, 'b2': 7.18544, 'b3': 21.4072, 'c': 5.21404, 'b1': 0.645643}}}
__atoms_database__["cd"]  = {'name':'Cadmium',        'symbol':'Cd', 'group':12, 'atomicNumber':48,  'neutronCohb':4.87-0.70j,     'neutronIncohb':None,         'neutronCohXs':3.04,      'neutronIncohXs':3.46,    'atomicWeight':112.411,  'covalentRadius':1.44,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.2214, 'b4': 87.4825, 'a3': 4.461, 'a2': 17.6444, 'a4': 1.6029, 'b2': 6.9089, 'b3': 24.7008, 'c': 5.0694, 'b1': 0.5946}, 2: {'a1': 19.1514, 'b4': 0.0, 'a3': 4.47128, 'a2': 17.2535, 'a4': 0.0, 'b2': 6.80639, 'b3': 20.2521, 'c': 5.11937, 'b1': 0.597922}}}
__atoms_database__["in"]  = {'name':'Indium',         'symbol':'In', 'group':13, 'atomicNumber':49,  'neutronCohb':4.065-0.0539j,  'neutronIncohb':None,         'neutronCohXs':2.08,      'neutronIncohXs':0.54,    'atomicWeight':114.818,  'covalentRadius':1.42,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.1624, 'b4': 92.8029, 'a3': 4.2948, 'a2': 18.5596, 'a4': 2.0396, 'b2': 6.3776, 'b3': 25.8499, 'c': 4.9391, 'b1': 0.5476}, 3: {'a1': 19.1045, 'b4': 0.0, 'a3': 3.78897, 'a2': 18.1108, 'a4': 0.0, 'b2': 6.3247, 'b3': 17.3595, 'c': 4.99635, 'b1': 0.551522}}}
__atoms_database__["sn"]  = {'name':'Tin',            'symbol':'Sn', 'group':14, 'atomicNumber':50,  'neutronCohb':6.225,          'neutronIncohb':None,         'neutronCohXs':4.871,     'neutronIncohXs':0.022,   'atomicWeight':118.71,   'covalentRadius':1.39,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.1889, 'b4': 83.9571, 'a3': 4.4585, 'a2': 19.1005, 'a4': 2.4663, 'b2': 0.5031, 'b3': 26.8909, 'c': 4.7821, 'b1': 5.8303}, 2: {'a1': 19.1094, 'b4': 62.2061, 'a3': 4.5648, 'a2': 19.0548, 'a4': 0.487, 'b2': 5.8378, 'b3': 23.3752, 'c': 4.7861, 'b1': 0.5036}, 4: {'a1': 18.9333, 'b4': -0.7583, 'a3': 3.4182, 'a2': 19.7131, 'a4': 0.0193, 'b2': 0.4655, 'b3': 14.0049, 'c': 3.9182, 'b1': 5.764}}}
__atoms_database__["sb"]  = {'name':'Antimony',       'symbol':'Sb', 'group':15, 'atomicNumber':51,  'neutronCohb':5.57,           'neutronIncohb':None,         'neutronCohXs':3.9,       'neutronIncohXs':0.007,   'atomicWeight':121.76,   'covalentRadius':1.39,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.6418, 'b4': 75.2825, 'a3': 5.0371, 'a2': 19.0455, 'a4': 2.6827, 'b2': 0.4607, 'b3': 27.9074, 'c': 4.5909, 'b1': 5.3034}, 3: {'a1': 18.9755, 'b4': 55.5113, 'a3': 5.10789, 'a2': 18.933, 'a4': 0.288753, 'b2': 5.22126, 'b3': 19.5902, 'c': 4.69626, 'b1': 0.467196}, 5: {'a1': 19.8685, 'b4': 0.0, 'a3': 2.41253, 'a2': 19.0302, 'a4': 0.0, 'b2': 0.467973, 'b3': 14.1259, 'c': 4.69263, 'b1': 5.44853}}}
__atoms_database__["te"]  = {'name':'Tellurium',      'symbol':'Te', 'group':16, 'atomicNumber':52,  'neutronCohb':5.80,           'neutronIncohb':None,         'neutronCohXs':4.23,      'neutronIncohXs':0.09,    'atomicWeight':127.6,    'covalentRadius':1.47,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 19.9644, 'b4': 70.8403, 'a3': 6.14487, 'a2': 19.0138, 'a4': 2.5239, 'b2': 0.420885, 'b3': 28.5284, 'c': 4.352, 'b1': 4.81742}}}
__atoms_database__["i"]   = {'name':'Iodine',         'symbol':'I',  'group':17, 'atomicNumber':53,  'neutronCohb':5.28,           'neutronIncohb':1.58,         'neutronCohXs':3.5,       'neutronIncohXs':0.31,    'atomicWeight':126.904,  'covalentRadius':1.4,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 20.1472, 'b4': 66.8776, 'a3': 7.5138, 'a2': 18.9949, 'a4': 2.2735, 'b2': 0.3814, 'b3': 27.766, 'c': 4.0712, 'b1': 4.347}, -1: {'a1': 20.2332, 'b4': 84.9304, 'a3': 7.8069, 'a2': 18.997, 'a4': 2.8868, 'b2': 0.3815, 'b3': 29.5259, 'c': 4.0714, 'b1': 4.3579}}}
__atoms_database__["xe"]  = {'name':'Xenon',          'symbol':'Xe', 'group':18, 'atomicNumber':54,  'neutronCohb':4.92,           'neutronIncohb':3.04,         'neutronCohXs':2.96,      'neutronIncohXs':0,       'atomicWeight':131.29,   'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 20.2933, 'b4': 64.2658, 'a3': 8.9767, 'a2': 19.0298, 'a4': 1.99, 'b2': 0.344, 'b3': 26.4659, 'c': 3.7118, 'b1': 3.9282}}}
__atoms_database__["cs"]  = {'name':'Caesium',        'symbol':'Cs', 'group':1,  'atomicNumber':55,  'neutronCohb':5.42,           'neutronIncohb':1.29,         'neutronCohXs':3.69,      'neutronIncohXs':0.21,    'atomicWeight':132.905,  'covalentRadius':2.44,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 20.3892, 'b4': 213.904, 'a3': 10.662, 'a2': 19.1062, 'a4': 1.4953, 'b2': 0.3107, 'b3': 24.3879, 'c': 3.3352, 'b1': 3.569}, 1: {'a1': 20.3524, 'b4': 59.4565, 'a3': 10.2821, 'a2': 19.1278, 'a4': 0.9615, 'b2': 0.3086, 'b3': 23.7128, 'c': 3.2791, 'b1': 3.552}}}
__atoms_database__["ba"]  = {'name':'Barium',         'symbol':'Ba', 'group':2,  'atomicNumber':56,  'neutronCohb':5.07,           'neutronIncohb':None,         'neutronCohXs':3.23,      'neutronIncohXs':0.15,    'atomicWeight':137.327,  'covalentRadius':2.15,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 20.3361, 'b4': 167.202, 'a3': 10.888, 'a2': 19.297, 'a4': 2.6959, 'b2': 0.2756, 'b3': 20.2073, 'c': 2.7731, 'b1': 3.216}, 2: {'a1': 20.1807, 'b4': 51.746, 'a3': 10.9054, 'a2': 19.1136, 'a4': 0.77634, 'b2': 0.28331, 'b3': 20.0558, 'c': 3.02902, 'b1': 3.21367}}}
__atoms_database__["lu"]  = {'name':'Lutetium',       'symbol':'Lu', 'group':3,  'atomicNumber':71,  'neutronCohb':7.21,           'neutronIncohb':None,         'neutronCohXs':6.53,      'neutronIncohXs':0.7,     'atomicWeight':174.967,  'covalentRadius':1.87,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 28.9476, 'b4': 84.3298, 'a3': 15.1, 'a2': 15.2208, 'a4': 3.71601, 'b2': 9.98519, 'b3': 0.261033, 'c': 7.97628, 'b1': 1.90182}, 3: {'a1': 28.4628, 'b4': 16.3535, 'a3': 12.8429, 'a2': 18.121, 'a4': 5.59415, 'b2': 0.142292, 'b3': 7.33727, 'c': 2.97573, 'b1': 1.68216}}}
__atoms_database__["hf"]  = {'name':'Hafnium',        'symbol':'Hf', 'group':4,  'atomicNumber':72,  'neutronCohb':7.7,            'neutronIncohb':None,         'neutronCohXs':7.6,       'neutronIncohXs':2.6,     'atomicWeight':178.49,   'covalentRadius':1.75,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 29.144, 'b4': 72.029, 'a3': 14.7586, 'a2': 15.1726, 'a4': 4.30013, 'b2': 9.5999, 'b3': 0.275116, 'c': 8.58154, 'b1': 1.83262}, 4: {'a1': 28.8131, 'b4': 14.0366, 'a3': 12.7285, 'a2': 18.4601, 'a4': 5.59927, 'b2': 0.128903, 'b3': 6.76232, 'c': 2.39699, 'b1': 1.59136}}}
__atoms_database__["ta"]  = {'name':'Tantalum',       'symbol':'Ta', 'group':5,  'atomicNumber':73,  'neutronCohb':6.91,           'neutronIncohb':None,         'neutronCohXs':6,         'neutronIncohXs':0.01,    'atomicWeight':180.948,  'covalentRadius':1.7,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 29.2024, 'b4': 63.3644, 'a3': 14.5135, 'a2': 15.2293, 'a4': 4.76492, 'b2': 9.37046, 'b3': 0.295977, 'c': 9.24354, 'b1': 1.77333}, 5: {'a1': 29.1587, 'b4': 12.4244, 'a3': 12.8268, 'a2': 18.8407, 'a4': 5.38695, 'b2': 0.116741, 'b3': 6.31524, 'c': 1.78555, 'b1': 1.50711}}}
__atoms_database__["w"]   = {'name':'Tungsten',       'symbol':'W',  'group':6,  'atomicNumber':74,  'neutronCohb':4.86,           'neutronIncohb':None,         'neutronCohXs':2.97,      'neutronIncohXs':1.63,    'atomicWeight':183.84,   'covalentRadius':1.62,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 29.0818, 'b4': 57.056, 'a3': 14.4327, 'a2': 15.43, 'a4': 5.11982, 'b2': 9.2259, 'b3': 0.321703, 'c': 9.8875, 'b1': 1.72029}, 6: {'a1': 29.4936, 'b4': 11.1972, 'a3': 13.0544, 'a2': 19.3763, 'a4': 5.06412, 'b2': 0.104621, 'b3': 5.93667, 'c': 1.01074, 'b1': 1.42755}}}
__atoms_database__["re"]  = {'name':'Rhenium',        'symbol':'Re', 'group':7,  'atomicNumber':75,  'neutronCohb':9.2,            'neutronIncohb':None,         'neutronCohXs':10.6,      'neutronIncohXs':0.9,     'atomicWeight':186.207,  'covalentRadius':1.51,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 28.7621, 'b4': 52.0861, 'a3': 14.5564, 'a2': 15.7189, 'a4': 5.44174, 'b2': 9.09227, 'b3': 0.3505, 'c': 10.472, 'b1': 1.67191}}}
__atoms_database__["os"]  = {'name':'Osmium',         'symbol':'Os', 'group':8,  'atomicNumber':76,  'neutronCohb':10.7,           'neutronIncohb':None,         'neutronCohXs':14.4,      'neutronIncohXs':0.3,     'atomicWeight':190.23,   'covalentRadius':1.44,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 28.1894, 'b4': 48.1647, 'a3': 14.9305, 'a2': 16.155, 'a4': 5.67589, 'b2': 8.97948, 'b3': 0.382661, 'c': 11.0005, 'b1': 1.62903}, 4: {'a1': 30.419, 'b4': 18.003, 'a3': 14.7458, 'a2': 15.2637, 'a4': 5.06795, 'b2': 6.84706, 'b3': 0.165191, 'c': 6.49804, 'b1': 1.37113}}}
__atoms_database__["ir"]  = {'name':'Iridium',        'symbol':'Ir', 'group':9,  'atomicNumber':77,  'neutronCohb':10.6,           'neutronIncohb':None,         'neutronCohXs':14.1,      'neutronIncohXs':0.,      'atomicWeight':192.217,  'covalentRadius':1.41,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 27.3049, 'b4': 45.0011, 'a3': 15.6115, 'a2': 16.7296, 'a4': 5.83377, 'b2': 8.86553, 'b3': 0.417916, 'c': 11.4722, 'b1': 1.59279}, 3: {'a1': 30.4156, 'b4': 20.3254, 'a3': 13.6145, 'a2': 15.862, 'a4': 5.82008, 'b2': 7.10909, 'b3': 0.204633, 'c': 8.27903, 'b1': 1.34323}, 4: {'a1': 30.7058, 'b4': 17.4911, 'a3': 14.2326, 'a2': 15.5512, 'a4': 5.53672, 'b2': 6.71983, 'b3': 0.167252, 'c': 6.96824, 'b1': 1.30923}}}
__atoms_database__["pt"]  = {'name':'Platinum',       'symbol':'Pt', 'group':10, 'atomicNumber':78,  'neutronCohb':9.60,           'neutronIncohb':None,         'neutronCohXs':11.58,     'neutronIncohXs':0.13,    'atomicWeight':195.078,  'covalentRadius':1.36,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 27.0059, 'b4': 38.6103, 'a3': 15.7131, 'a2': 17.7639, 'a4': 5.7837, 'b2': 8.81174, 'b3': 0.424593, 'c': 11.6883, 'b1': 1.51293}, 2: {'a1': 29.8429, 'b4': 22.9426, 'a3': 13.2153, 'a2': 16.7224, 'a4': 6.35234, 'b2': 7.38979, 'b3': 0.263297, 'c': 9.85329, 'b1': 1.32927}, 4: {'a1': 30.9612, 'b4': 16.9392, 'a3': 13.7348, 'a2': 15.9829, 'a4': 5.92034, 'b2': 6.60834, 'b3': 0.16864, 'c': 7.39534, 'b1': 1.24813}}}
__atoms_database__["au"]  = {'name':'Gold',           'symbol':'Au', 'group':11, 'atomicNumber':79,  'neutronCohb':7.63,           'neutronIncohb':-1.84,        'neutronCohXs':7.32,      'neutronIncohXs':0.43,    'atomicWeight':196.967,  'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 16.8819, 'b4': 36.3956, 'a3': 25.5582, 'a2': 18.5913, 'a4': 5.86, 'b2': 8.6216, 'b3': 1.4826, 'c': 12.0658, 'b1': 0.4611}, 1: {'a1': 28.0109, 'b4': 26.4043, 'a3': 14.3359, 'a2': 17.8204, 'a4': 6.58077, 'b2': 7.7395, 'b3': 0.356752, 'c': 11.2299, 'b1': 1.35321}, 3: {'a1': 30.6886, 'b4': 18.659, 'a3': 12.7801, 'a2': 16.9029, 'a4': 6.52354, 'b2': 6.82872, 'b3': 0.212867, 'c': 9.0968, 'b1': 1.2199}}}
__atoms_database__["hg"]  = {'name':'Mercury',        'symbol':'Hg', 'group':12, 'atomicNumber':80,  'neutronCohb':12.692,         'neutronIncohb':None,         'neutronCohXs':20.24,     'neutronIncohXs':6.6,     'atomicWeight':200.59,   'covalentRadius':1.32,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 20.6809, 'b4': 38.3246, 'a3': 21.6575, 'a2': 19.0417, 'a4': 5.9676, 'b2': 8.4484, 'b3': 1.5729, 'c': 12.6089, 'b1': 0.545}, 1: {'a1': 25.0853, 'b4': 28.2262, 'a3': 16.8883, 'a2': 18.4973, 'a4': 6.48216, 'b2': 7.65105, 'b3': 0.443378, 'c': 12.0205, 'b1': 1.39507}, 2: {'a1': 29.5641, 'b4': 20.7482, 'a3': 12.8374, 'a2': 18.06, 'a4': 6.89912, 'b2': 7.05639, 'b3': 0.284738, 'c': 10.6268, 'b1': 1.21152}}}
__atoms_database__["tl"]  = {'name':'Thallium',       'symbol':'Tl', 'group':13, 'atomicNumber':81,  'neutronCohb':8.776,          'neutronIncohb':None,         'neutronCohXs':9.678,     'neutronIncohXs':0.21,    'atomicWeight':204.383,  'covalentRadius':1.45,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 27.5446, 'b4': 45.8149, 'a3': 15.538, 'a2': 19.1584, 'a4': 5.52593, 'b2': 8.70751, 'b3': 1.96347, 'c': 13.1746, 'b1': 0.65515}, 1: {'a1': 21.3985, 'b4': 28.8482, 'a3': 18.7478, 'a2': 20.4723, 'a4': 6.82847, 'b2': 0.517394, 'b3': 7.43463, 'c': 12.5258, 'b1': 1.4711}, 3: {'a1': 30.8695, 'b4': 17.2114, 'a3': 11.9328, 'a2': 18.3481, 'a4': 7.00574, 'b2': 6.53852, 'b3': 0.219074, 'c': 9.8027, 'b1': 1.1008}}}
__atoms_database__["pb"]  = {'name':'Lead',           'symbol':'Pb', 'group':14, 'atomicNumber':82,  'neutronCohb':9.405,          'neutronIncohb':None,         'neutronCohXs':11.115,    'neutronIncohXs':0.003,   'atomicWeight':207.2,    'covalentRadius':1.46,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 31.0617, 'b4': 47.2579, 'a3': 18.442, 'a2': 13.0637, 'a4': 5.9696, 'b2': 2.3576, 'b3': 8.618, 'c': 13.4118, 'b1': 0.6902}, 2: {'a1': 21.7886, 'b4': 23.8132, 'a3': 19.1406, 'a2': 19.5682, 'a4': 7.01107, 'b2': 0.488383, 'b3': 6.7727, 'c': 12.4734, 'b1': 1.3366}, 4: {'a1': 32.1244, 'b4': 14.714, 'a3': 12.0175, 'a2': 18.8003, 'a4': 6.96886, 'b2': 6.10926, 'b3': 0.147041, 'c': 8.08428, 'b1': 1.00566}}}
__atoms_database__["bi"]  = {'name':'Bismuth',        'symbol':'Bi', 'group':15, 'atomicNumber':83,  'neutronCohb':8.532,          'neutronIncohb':None,         'neutronCohXs':9.148,     'neutronIncohXs':0.0084,  'atomicWeight':208.98,   'covalentRadius':1.48,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 33.3689, 'b4': 48.0093, 'a3': 16.5877, 'a2': 12.951, 'a4': 6.4692, 'b2': 2.9238, 'b3': 8.7937, 'c': 13.5782, 'b1': 0.704}, 3: {'a1': 21.8053, 'b4': 20.3185, 'a3': 19.1053, 'a2': 19.5026, 'a4': 7.10295, 'b2': 6.24149, 'b3': 0.469999, 'c': 12.4711, 'b1': 1.2356}, 5: {'a1': 33.5364, 'b4': 12.8285, 'a3': 19.2497, 'a2': 25.0946, 'a4': 6.91555, 'b2': 0.39042, 'b3': 5.71414, 'c': -6.7994, 'b1': 0.91654}}}
__atoms_database__["po"]  = {'name':'Polonium',       'symbol':'Po', 'group':16, 'atomicNumber':84,  'neutronCohb':None,           'neutronIncohb':0.259,        'neutronCohXs':0,         'neutronIncohXs':None,    'atomicWeight':210,      'covalentRadius':1.4,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 34.6726, 'b4': 47.0045, 'a3': 13.1138, 'a2': 15.4733, 'a4': 7.02588, 'b2': 3.55078, 'b3': 9.55642, 'c': 13.677, 'b1': 0.700999}}}
__atoms_database__["at"]  = {'name':'Astatine',       'symbol':'At', 'group':17, 'atomicNumber':85,  'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':0,         'neutronIncohXs':None,    'atomicWeight':210,      'covalentRadius':1.21,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 35.3163, 'b4': 45.4715, 'a3': 9.49887, 'a2': 19.0211, 'a4': 7.42518, 'b2': 3.97458, 'b3': 11.3824, 'c': 13.7108, 'b1': 0.68587}}}
__atoms_database__["rn"]  = {'name':'Radon',          'symbol':'Rn', 'group':18, 'atomicNumber':86,  'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':0,         'neutronIncohXs':None,    'atomicWeight':222,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 35.5631, 'b4': 44.2473, 'a3': 8.0037, 'a2': 21.2816, 'a4': 7.4433, 'b2': 4.0691, 'b3': 14.0422, 'c': 13.6905, 'b1': 0.6631}}}
__atoms_database__["ce"]  = {'name':'Cerium',         'symbol':'Ce', 'group':19, 'atomicNumber':58,  'neutronCohb':4.84,           'neutronIncohb':None,         'neutronCohXs':2.94,      'neutronIncohXs':0.001,   'atomicWeight':140.116,  'covalentRadius':2.04,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 21.1671, 'b4': 127.113, 'a3': 11.8513, 'a2': 19.7695, 'a4': 3.33049, 'b2': 0.226836, 'b3': 17.6083, 'c': 1.86264, 'b1': 2.81219}, 3: {'a1': 20.8036, 'b4': 43.1692, 'a3': 11.9369, 'a2': 19.559, 'a4': 0.612376, 'b2': 0.23154, 'b3': 16.5408, 'c': 2.09013, 'b1': 2.77691}, 4: {'a1': 20.3235, 'b4': 62.2355, 'a3': 12.1233, 'a2': 19.8186, 'a4': 0.144583, 'b2': 0.21885, 'b3': 15.7992, 'c': 1.5918, 'b1': 2.65941}}}
__atoms_database__["dy"]  = {'name':'Dysprosium',     'symbol':'Dy', 'group':19, 'atomicNumber':66,  'neutronCohb':16.9-0.276j,    'neutronIncohb':None,         'neutronCohXs':35.9,      'neutronIncohXs':54.4,    'atomicWeight':162.5,    'covalentRadius':1.92,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 26.507, 'b4': 111.874, 'a3': 14.5596, 'a2': 17.6383, 'a4': 2.96577, 'b2': 0.202172, 'b3': 12.1899, 'c': 4.29728, 'b1': 2.1802}, 3: {'a1': 25.5395, 'b4': 19.581, 'a3': 11.9812, 'a2': 20.2861, 'a4': 4.50073, 'b2': 0.143384, 'b3': 9.34972, 'c': 0.68969, 'b1': 1.9804}}}
__atoms_database__["er"]  = {'name':'Erbium',         'symbol':'Er', 'group':19, 'atomicNumber':68,  'neutronCohb':7.79,           'neutronIncohb':None,         'neutronCohXs':7.63,      'neutronIncohXs':1.1,     'atomicWeight':167.26,   'covalentRadius':1.89,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 27.6563, 'b4': 105.703, 'a3': 14.9779, 'a2': 16.4285, 'a4': 2.98233, 'b2': 0.223545, 'b3': 11.3604, 'c': 5.92046, 'b1': 2.07356}, 3: {'a1': 26.722, 'b4': 17.8974, 'a3': 12.1506, 'a2': 19.7748, 'a4': 5.17379, 'b2': 0.13729, 'b3': 8.36225, 'c': 1.17613, 'b1': 1.84659}}}
__atoms_database__["eu"]  = {'name':'Europium',       'symbol':'Eu', 'group':19, 'atomicNumber':63,  'neutronCohb':7.22-1.26j,     'neutronIncohb':None,         'neutronCohXs':6.57,      'neutronIncohXs':2.5,     'atomicWeight':151.964,  'covalentRadius':1.98,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 24.6274, 'b4': 123.174, 'a3': 13.7603, 'a2': 19.0886, 'a4': 2.9227, 'b2': 0.1942, 'b3': 13.7546, 'c': 2.5745, 'b1': 2.3879}, 2: {'a1': 24.0063, 'b4': 26.5156, 'a3': 11.8034, 'a2': 19.9504, 'a4': 3.87243, 'b2': 0.17353, 'b3': 11.6096, 'c': 1.36389, 'b1': 2.27783}, 3: {'a1': 23.7497, 'b4': 22.9966, 'a3': 11.8509, 'a2': 20.3745, 'a4': 3.26503, 'b2': 0.16394, 'b3': 11.311, 'c': 0.759344, 'b1': 2.22258}}}
__atoms_database__["gd"]  = {'name':'Gadolinium',     'symbol':'Gd', 'group':19, 'atomicNumber':64,  'neutronCohb':6.5-13.82j,     'neutronIncohb':None,         'neutronCohXs':29.3,      'neutronIncohXs':151.,    'atomicWeight':157.25,   'covalentRadius':1.96,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 25.0709, 'b4': 101.398, 'a3': 13.8518, 'a2': 19.0798, 'a4': 3.54545, 'b2': 0.181951, 'b3': 12.9331, 'c': 2.4196, 'b1': 2.25341}, 3: {'a1': 24.3466, 'b4': 21.7029, 'a3': 11.8708, 'a2': 20.4208, 'a4': 3.7149, 'b2': 0.155525, 'b3': 10.5782, 'c': 0.645089, 'b1': 2.13553}}}
__atoms_database__["ho"]  = {'name':'Holmium',        'symbol':'Ho', 'group':19, 'atomicNumber':67,  'neutronCohb':8.01,           'neutronIncohb':-1.70,        'neutronCohXs':8.06,      'neutronIncohXs':0.36,    'atomicWeight':164.93,   'covalentRadius':1.92,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 26.9049, 'b4': 92.6566, 'a3': 14.5583, 'a2': 17.294, 'a4': 3.63837, 'b2': 0.19794, 'b3': 11.4407, 'c': 4.56796, 'b1': 2.07051}, 3: {'a1': 26.1296, 'b4': 18.5908, 'a3': 11.9788, 'a2': 20.0994, 'a4': 4.93676, 'b2': 0.139358, 'b3': 8.80018, 'c': 0.852795, 'b1': 1.91072}}}
__atoms_database__["la"]  = {'name':'Lanthanum',      'symbol':'La', 'group':19, 'atomicNumber':57,  'neutronCohb':8.24,           'neutronIncohb':None,         'neutronCohXs':8.53,      'neutronIncohXs':1.13,    'atomicWeight':138.906,  'covalentRadius':2.07,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 20.578, 'b4': 133.124, 'a3': 11.3727, 'a2': 19.599, 'a4': 3.28719, 'b2': 0.244475, 'b3': 18.7726, 'c': 2.14678, 'b1': 2.94817}, 3: {'a1': 20.2489, 'b4': 54.9453, 'a3': 11.6323, 'a2': 19.3763, 'a4': 0.336048, 'b2': 0.250698, 'b3': 17.8211, 'c': 2.4086, 'b1': 2.9207}}}
__atoms_database__["nd"]  = {'name':'Neodymium',      'symbol':'Nd', 'group':19, 'atomicNumber':60,  'neutronCohb':7.69,           'neutronIncohb':None,         'neutronCohXs':7.43,      'neutronIncohXs':9.2,     'atomicWeight':144.24,   'covalentRadius':2.01,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 22.6845, 'b4': 137.903, 'a3': 12.774, 'a2': 19.6847, 'a4': 2.85137, 'b2': 0.210628, 'b3': 15.885, 'c': 1.98486, 'b1': 2.66248}, 3: {'a1': 21.961, 'b4': 30.8717, 'a3': 12.12, 'a2': 19.9339, 'a4': 1.51031, 'b2': 0.199237, 'b3': 14.1783, 'c': 1.47588, 'b1': 2.52722}}}
__atoms_database__["pm"]  = {'name':'Promethium',     'symbol':'Pm', 'group':19, 'atomicNumber':61,  'neutronCohb':12.6,           'neutronIncohb':None,         'neutronCohXs':20.0,      'neutronIncohXs':1.3,     'atomicWeight':145,      'covalentRadius':1.99,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 23.3405, 'b4': 132.721, 'a3': 13.1235, 'a2': 19.6095, 'a4': 2.87516, 'b2': 0.202088, 'b3': 15.1009, 'c': 2.02876, 'b1': 2.5627}, 3: {'a1': 22.5527, 'b4': 27.4491, 'a3': 12.0671, 'a2': 20.1108, 'a4': 2.07492, 'b2': 0.185769, 'b3': 13.1275, 'c': 1.19499, 'b1': 2.4174}}}
__atoms_database__["pr"]  = {'name':'Praseodymium',   'symbol':'Pr', 'group':19, 'atomicNumber':59,  'neutronCohb':4.58,           'neutronIncohb':-0.35,        'neutronCohXs':2.64,      'neutronIncohXs':0.015,   'atomicWeight':140.908,  'covalentRadius':2.03,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 22.044, 'b4': 143.644, 'a3': 12.3856, 'a2': 19.6697, 'a4': 2.82428, 'b2': 0.222087, 'b3': 16.7669, 'c': 2.0583, 'b1': 2.77393}, 3: {'a1': 21.3727, 'b4': 36.4065, 'a3': 12.1329, 'a2': 19.7491, 'a4': 0.97518, 'b2': 0.214299, 'b3': 15.323, 'c': 1.77132, 'b1': 2.6452}, 4: {'a1': 20.9413, 'b4': 45.4643, 'a3': 12.4668, 'a2': 20.0539, 'a4': 0.296689, 'b2': 0.202481, 'b3': 14.8137, 'c': 1.24285, 'b1': 2.54467}}}
__atoms_database__["sm"]  = {'name':'Samarium',       'symbol':'Sm', 'group':19, 'atomicNumber':62,  'neutronCohb':0.80-1.65j,     'neutronIncohb':None,         'neutronCohXs':0.422,     'neutronIncohXs':39.,     'atomicWeight':150.36,   'covalentRadius':1.98,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 24.0042, 'b4': 128.007, 'a3': 13.4396, 'a2': 19.4258, 'a4': 2.89604, 'b2': 0.196451, 'b3': 14.3996, 'c': 2.20963, 'b1': 2.47274}, 3: {'a1': 23.1504, 'b4': 24.8242, 'a3': 11.9202, 'a2': 20.2599, 'a4': 2.71488, 'b2': 0.174081, 'b3': 12.1571, 'c': 0.954586, 'b1': 2.31641}}}
__atoms_database__["tb"]  = {'name':'Terbium',        'symbol':'Tb', 'group':19, 'atomicNumber':65,  'neutronCohb':7.38,           'neutronIncohb':-0.17,        'neutronCohXs':6.84,      'neutronIncohXs':0.004,   'atomicWeight':158.925,  'covalentRadius':1.94,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 25.8976, 'b4': 115.362, 'a3': 14.3167, 'a2': 18.2185, 'a4': 2.95354, 'b2': 0.196143, 'b3': 12.6648, 'c': 3.58324, 'b1': 2.24256}, 3: {'a1': 24.9559, 'b4': 21.2773, 'a3': 12.2471, 'a2': 20.3271, 'a4': 3.773, 'b2': 0.149525, 'b3': 10.0499, 'c': 0.691967, 'b1': 2.05601}}}
__atoms_database__["tm"]  = {'name':'Thulium',        'symbol':'Tm', 'group':19, 'atomicNumber':69,  'neutronCohb':7.07,           'neutronIncohb':0.9,          'neutronCohXs':6.28,      'neutronIncohXs':0.1,     'atomicWeight':168.934,  'covalentRadius':1.9,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 28.1819, 'b4': 102.961, 'a3': 15.1542, 'a2': 15.8851, 'a4': 2.98706, 'b2': 0.238849, 'b3': 10.9975, 'c': 6.75621, 'b1': 2.02859}, 3: {'a1': 27.3083, 'b4': 17.2922, 'a3': 12.3339, 'a2': 19.332, 'a4': 5.38348, 'b2': 0.136974, 'b3': 7.96778, 'c': 1.63929, 'b1': 1.78711}}}
__atoms_database__["yb"]  = {'name':'Ytterbium',      'symbol':'Yb', 'group':19, 'atomicNumber':70,  'neutronCohb':12.43,          'neutronIncohb':None,         'neutronCohXs':19.42,     'neutronIncohXs':4,       'atomicWeight':173.04,   'covalentRadius':1.87,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 28.6641, 'b4': 100.417, 'a3': 15.3087, 'a2': 15.4345, 'a4': 2.98963, 'b2': 0.257119, 'b3': 10.6647, 'c': 7.56672, 'b1': 1.9889}, 2: {'a1': 28.1209, 'b4': 20.39, 'a3': 13.3335, 'a2': 17.6817, 'a4': 5.14657, 'b2': 0.15997, 'b3': 8.18304, 'c': 3.70983, 'b1': 1.78503}, 3: {'a1': 27.8917, 'b4': 16.8153, 'a3': 12.6072, 'a2': 18.7614, 'a4': 5.47647, 'b2': 0.13879, 'b3': 7.64412, 'c': 2.26001, 'b1': 1.73272}}}
__atoms_database__["fr"]  = {'name':'Francium',       'symbol':'Fr', 'group':1,  'atomicNumber':87,  'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':0,         'neutronIncohXs':None,    'atomicWeight':223,      'covalentRadius':2.6,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 35.9299, 'b4': 150.645, 'a3': 12.1439, 'a2': 23.0547, 'a4': 2.11253, 'b2': 4.17619, 'b3': 23.1052, 'c': 13.7247, 'b1': 0.646453}}}
__atoms_database__["ra"]  = {'name':'Radium',         'symbol':'Ra', 'group':2,  'atomicNumber':88,  'neutronCohb':10.0,           'neutronIncohb':0,            'neutronCohXs':13.,       'neutronIncohXs':0,       'atomicWeight':226,      'covalentRadius':2.21,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 35.763, 'b4': 142.325, 'a3': 12.4739, 'a2': 22.9064, 'a4': 3.21097, 'b2': 3.87135, 'b3': 19.9887, 'c': 13.6211, 'b1': 0.616341}, 2: {'a1': 35.215, 'b4': 29.8436, 'a3': 7.91342, 'a2': 21.67, 'a4': 7.65078, 'b2': 3.5767, 'b3': 12.601, 'c': 13.5431, 'b1': 0.604909}}}
__atoms_database__["lr"]  = {'name':'Lawrencium',     'symbol':'Lr', 'group':3,  'atomicNumber':103, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':262,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["lw"]  = {'name':'Lawrencium',     'symbol':'Lw', 'group':3,  'atomicNumber':103, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':262,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["rf"]  = {'name':'Rutherfordium',  'symbol':'Rf', 'group':4,  'atomicNumber':104, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':261,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["db"]  = {'name':'Dubnium',        'symbol':'Db', 'group':5,  'atomicNumber':105, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':262,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["sg"]  = {'name':'Seaborgium',     'symbol':'Sg', 'group':6,  'atomicNumber':106, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':266,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["bh"]  = {'name':'Bohrium',        'symbol':'Bh', 'group':7,  'atomicNumber':107, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':264,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["hs"]  = {'name':'Hassium',        'symbol':'Hs', 'group':8,  'atomicNumber':108, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':269,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["mt"]  = {'name':'Meitnerium',     'symbol':'Mt', 'group':9,  'atomicNumber':109, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':268,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["ds"]  = {'name':'Darmstadtium',   'symbol':'Ds', 'group':10, 'atomicNumber':110, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':271,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["ac"]  = {'name':'Actinium',       'symbol':'Ac', 'group':20, 'atomicNumber':89,  'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':0,         'neutronIncohXs':None,    'atomicWeight':227,      'covalentRadius':2.15,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 35.6597, 'b4': 117.02, 'a3': 12.5977, 'a2': 23.1032, 'a4': 4.08655, 'b2': 3.65155, 'b3': 18.599, 'c': 13.5266, 'b1': 0.589092}, 3: {'a1': 35.1736, 'b4': 25.9443, 'a3': 8.19216, 'a2': 22.1112, 'a4': 7.05545, 'b2': 3.41437, 'b3': 12.9187, 'c': 13.4637, 'b1': 0.579689}}}
__atoms_database__["am"]  = {'name':'Americium',      'symbol':'Am', 'group':20, 'atomicNumber':95,  'neutronCohb':8.3,            'neutronIncohb':None,         'neutronCohXs':8.7,       'neutronIncohXs':0.3,     'atomicWeight':243,      'covalentRadius':1.8,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 36.6706, 'b4': 102.273, 'a3': 17.3415, 'a2': 24.0992, 'a4': 3.49331, 'b2': 3.20647, 'b3': 14.3136, 'c': 13.3592, 'b1': 0.483629}}}
__atoms_database__["bk"]  = {'name':'Berkelium',      'symbol':'Bk', 'group':20, 'atomicNumber':97,  'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':247,      'covalentRadius':1.54,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 36.7881, 'b4': 86.003, 'a3': 17.8919, 'a2': 24.7736, 'a4': 4.23284, 'b2': 3.04619, 'b3': 12.8946, 'c': 13.2754, 'b1': 0.451018}}}
__atoms_database__["cf"]  = {'name':'Californium',    'symbol':'Cf', 'group':20, 'atomicNumber':98,  'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':251,      'covalentRadius':1.83,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 36.9185, 'b4': 83.7881, 'a3': 18.3317, 'a2': 25.1995, 'a4': 4.24391, 'b2': 3.00775, 'b3': 12.4044, 'c': 13.2674, 'b1': 0.437533}}}
__atoms_database__["cm"]  = {'name':'Curium',         'symbol':'Cm', 'group':20, 'atomicNumber':96,  'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':0,         'neutronIncohXs':None,    'atomicWeight':247,      'covalentRadius':1.69,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 36.6488, 'b4': 88.4834, 'a3': 17.399, 'a2': 24.4096, 'a4': 4.21665, 'b2': 3.08997, 'b3': 13.4346, 'c': 13.2887, 'b1': 0.465154}}}
__atoms_database__["es"]  = {'name':'Einsteinium',    'symbol':'Es', 'group':20, 'atomicNumber':99,  'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':252,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["fm"]  = {'name':'Fermium',        'symbol':'Fm', 'group':20, 'atomicNumber':100, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':257,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["md"]  = {'name':'Mendelevium',    'symbol':'Md', 'group':20, 'atomicNumber':101, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':258,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["no"]  = {'name':'Nobelium',       'symbol':'No', 'group':20, 'atomicNumber':102, 'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':259,      'covalentRadius':1.5,    'equal':1.0,  'atomicFormFactor':{}}
__atoms_database__["np"]  = {'name':'Neptunium',      'symbol':'Np', 'group':20, 'atomicNumber':93,  'neutronCohb':10.55,          'neutronIncohb':None,         'neutronCohXs':14,        'neutronIncohXs':0.5,     'atomicWeight':237,      'covalentRadius':1.9,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 36.1874, 'b4': 97.4908, 'a3': 15.6402, 'a2': 23.5964, 'a4': 4.1855, 'b2': 3.25396, 'b3': 15.3622, 'c': 13.3573, 'b1': 0.511929}, 3: {'a1': 35.7074, 'b4': 25.4928, 'a3': 12.9898, 'a2': 22.613, 'a4': 5.43227, 'b2': 3.03807, 'b3': 12.1449, 'c': 13.2544, 'b1': 0.502322}, 4: {'a1': 35.5103, 'b4': 22.7502, 'a3': 12.7766, 'a2': 22.5787, 'a4': 4.92159, 'b2': 2.96627, 'b3': 11.9484, 'c': 13.2116, 'b1': 0.498626}, 6: {'a1': 35.0136, 'b4': 22.6581, 'a3': 14.3884, 'a2': 22.7286, 'a4': 1.75669, 'b2': 2.81099, 'b3': 12.33, 'c': 13.113, 'b1': 0.48981}}}
__atoms_database__["pa"]  = {'name':'Protactinium',   'symbol':'Pa', 'group':20, 'atomicNumber':91,  'neutronCohb':9.1,            'neutronIncohb':None,         'neutronCohXs':10.4,      'neutronIncohXs':0.1,     'atomicWeight':231.036,  'covalentRadius':2.0,    'equal':1.0,  'atomicFormFactor':{0: {'a1': 35.8847, 'b4': 105.251, 'a3': 14.1891, 'a2': 23.2948, 'a4': 4.17287, 'b2': 3.41519, 'b3': 16.9235, 'c': 13.4287, 'b1': 0.547751}}}
__atoms_database__["pu"]  = {'name':'Plutonium',      'symbol':'Pu', 'group':20, 'atomicNumber':94,  'neutronCohb':None,           'neutronIncohb':None,         'neutronCohXs':None,      'neutronIncohXs':None,    'atomicWeight':244,      'covalentRadius':1.87,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 36.5254, 'b4': 105.98, 'a3': 16.7707, 'a2': 23.8083, 'a4': 3.47947, 'b2': 3.26371, 'b3': 14.9455, 'c': 13.3812, 'b1': 0.499384}, 3: {'a1': 35.84, 'b4': 24.3992, 'a3': 13.5807, 'a2': 22.7169, 'a4': 5.66016, 'b2': 2.96118, 'b3': 11.5331, 'c': 13.1991, 'b1': 0.484938}, 4: {'a1': 35.6493, 'b4': 21.8301, 'a3': 13.3595, 'a2': 22.646, 'a4': 5.18831, 'b2': 2.8902, 'b3': 11.316, 'c': 13.1555, 'b1': 0.481422}, 6: {'a1': 35.1736, 'b4': 20.9303, 'a3': 14.7635, 'a2': 22.7181, 'a4': 2.28678, 'b2': 2.73848, 'b3': 11.553, 'c': 13.0582, 'b1': 0.473204}}}
__atoms_database__["th"]  = {'name':'Thorium',        'symbol':'Th', 'group':20, 'atomicNumber':90,  'neutronCohb':10.31,          'neutronIncohb':0,            'neutronCohXs':13.36,     'neutronIncohXs':0,       'atomicWeight':232.038,  'covalentRadius':2.06,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 35.5645, 'b4': 99.1722, 'a3': 12.7473, 'a2': 23.4219, 'a4': 4.80703, 'b2': 3.46204, 'b3': 17.8309, 'c': 13.4314, 'b1': 0.563359}, 4: {'a1': 35.1007, 'b4': 23.9533, 'a3': 9.78554, 'a2': 22.4418, 'a4': 5.29444, 'b2': 3.24498, 'b3': 13.4661, 'c': 13.376, 'b1': 0.555054}}}
__atoms_database__["u"]   = {'name':'Uranium',        'symbol':'U',  'group':20, 'atomicNumber':92,  'neutronCohb':8.417,          'neutronIncohb':None,         'neutronCohXs':8.903,     'neutronIncohXs':0.005,   'atomicWeight':238.029,  'covalentRadius':1.96,   'equal':1.0,  'atomicFormFactor':{0: {'a1': 36.0228, 'b4': 100.613, 'a3': 14.9491, 'a2': 23.4128, 'a4': 4.188, 'b2': 3.3253, 'b3': 16.0927, 'c': 13.3966, 'b1': 0.5293}, 3: {'a1': 35.5747, 'b4': 26.3394, 'a3': 12.2165, 'a2': 22.5259, 'a4': 5.37073, 'b2': 3.12293, 'b3': 12.7148, 'c': 13.3092, 'b1': 0.52048}, 4: {'a1': 35.3715, 'b4': 23.4582, 'a3': 12.0291, 'a2': 22.5326, 'a4': 4.7984, 'b2': 3.05053, 'b3': 12.5723, 'c': 13.2671, 'b1': 0.516598}, 6: {'a1': 34.8509, 'b4': 25.2017, 'a3': 14.0099, 'a2': 22.7584, 'a4': 1.21457, 'b2': 2.8903, 'b3': 13.1767, 'c': 13.1665, 'b1': 0.507079}}}


def export_database(fd=sys.stdout):
    """
    Export all database as a python dictionary instance.
    
    :Parameters:
        #. fd (file handler): The file handler to which the database would be exported.\n
           By default it's the standard output of the system.
    """
    fd.write('__atoms_database__ = {}\n')
    for el in __atoms_database__.keys():
        val = __atoms_database__[el]
        fd.write(str.ljust('__atoms_database__["%s"]'%el, 26))
        fd.write("= {")
        # name
        fd.write(str.ljust("'name':'%s',"%val['name'], 25))
        # symbol
        fd.write(str.ljust("'symbol':'%s',"%val['symbol'], 15))
        # group
        fd.write(str.ljust("'group':%s,"%val['group'], 12))
        # atomic number
        fd.write(str.ljust("'atomicNumber':%s,"%val['atomicNumber'], 20))
        # neutron coherent scattering length
        fd.write(str.ljust("'neutronCohb':%s,"%val['neutronCohb'], 30))
        # neutron incoherent scattering length
        fd.write(str.ljust("'neutronIncohb':%s,"%val['neutronIncohb'], 30))
        # neutron coherent cross section
        fd.write(str.ljust("'neutronCohXs':%s,"%val['neutronCohXs'], 26))
        # neutron incoherent cross section
        fd.write(str.ljust("'neutronIncohXs':%s,"%val['neutronIncohXs'], 26))
        # atomic weight
        fd.write(str.ljust("'atomicWeight':%s,"%val['atomicWeight'], 25))
        # covalent radius
        fd.write(str.ljust("'covalentRadius':%s,"%val['covalentRadius'], 25))
        # equal weighting
        fd.write(str.ljust("'equal':%s,"%val['equal'], 14))
        # equal atomicFormFactor
        fd.write(str.ljust("'atomicFormFactor':%s}"%val['atomicFormFactor'], 12))
        fd.write('\n')

# atoms and molecules definitions
__ATOM__ = {'segment_identifier': '    ',\
 'record_name': 'ATOM  ',\
 'coordinates_x': 0.0, \
 'coordinates_y': 0.0, \
 'coordinates_z': 0.0, \
 'charge': '  ', \
 'code_of_insertion': ' ', \
 'location_indicator': ' ',\
 'temperature_factor': 0.0, \
 'residue_name': 'TIP', \
 'occupancy': 1.0, \
 'serial_number': 1, \
 'atom_name': '', \
 'chain_identifier': ' ', \
 'sequence_number': 1, \
 'element_symbol': ''}

WAT_O = {'segment_identifier': '    ',\
 'record_name': 'ATOM  ',\
 'coordinates_x': 0.0, \
 'coordinates_y': -0.293, \
 'coordinates_z': 0.0, \
 'charge': '  ', \
 'code_of_insertion': ' ', \
 'location_indicator': ' ',\
 'temperature_factor': 0.0, \
 'residue_name': 'TIP', \
 'occupancy': 1.0, \
 'serial_number': 1, \
 'atom_name': 'OH2', \
 'chain_identifier': ' ', \
 'sequence_number': 1, \
 'element_symbol': ' O'}

WAT_H1 = {'segment_identifier': '    ',\
 'record_name': 'ATOM  ',\
 'coordinates_x': -0.757, \
 'coordinates_y': 0.293, \
 'coordinates_z': 0.0, \
 'charge': '  ', \
 'code_of_insertion': ' ', \
 'location_indicator': ' ',\
 'temperature_factor': 0.0, \
 'residue_name': 'TIP', \
 'occupancy': 1.0, \
 'serial_number': 2, \
 'atom_name': 'H1', \
 'chain_identifier': ' ', \
 'sequence_number': 1, \
 'element_symbol': ' H'}

WAT_H2 = {'segment_identifier': '    ',\
 'record_name': 'ATOM  ',\
 'coordinates_x': 0.757, \
 'coordinates_y': 0.293, \
 'coordinates_z': 0.0, \
 'charge': '  ', \
 'code_of_insertion': ' ', \
 'location_indicator': ' ',\
 'temperature_factor': 0.0, \
 'residue_name': 'TIP', \
 'occupancy': 1.0, \
 'serial_number': 3, \
 'atom_name': 'H2', \
 'chain_identifier': ' ', \
 'sequence_number': 1, \
 'element_symbol': ' H'}

__WATER__ = []
__WATER__.extend([WAT_O , WAT_H1 , WAT_H2])
