/*
 * ISDA CDS Standard Model
 *
 * Copyright (C) 2009 International Swaps and Derivatives Association, Inc.
 * Developed and supported in collaboration with Markit
 * 
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the ISDA CDS Standard Model Public License.
 */

#include <time.h>
#include "cgeneral.h"
#include "convert.h"
#include "dateconv.h"
#include "macros.h"
#include "cerror.h"

#define DAYS_IN_1_YEAR      365L
#define DAYS_IN_4_YEARS     1461L
#define DAYS_IN_100_YEARS  (DAYS_IN_4_YEARS * 25 - 1)
#define DAYS_IN_400_YEARS  (DAYS_IN_100_YEARS * 4 + 1)

#define MULTIPLY_BY_1461(N) (((N) << 10) + \
                 ((N) << 8) + \
                 ((N) << 7) + \
                 ((N) << 5) + \
                 ((N) << 4) + \
                 ((N) << 2) + \
                 (N))

/* Make these long because they're compared against TDates, which
 * are also long. Since this is a time-critical part of the routine,
 * we want to make it as fast as possible.
 */
static long  leapCumDays[] = {
    -1, 30, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365};

static long  cumDays[] = {
    -1, 30, 58, 89, 119, 150, 180, 211, 242, 272, 303, 333, 364};

static int  leapDays[] = {
    0, 31,  29,  31,  30,  31,  30,  31,  31,  30,  31,  30,  31};
    /* JAN  FEB  MAR  APR  MAY  JUN  JUL  AUG  SEP  OCT  NOV  DEC */

static int  days[] = {
    0, 31,  28,  31,  30,  31,  30,  31,  31,  30,  31,  30,  31};
    /* JAN  FEB  MAR  APR  MAY  JUN  JUL  AUG  SEP  OCT  NOV  DEC */


/*
 * Typedefs for date conversion cache.
 *
 * Basic idea:
 *    The time consuming part of date conversion always arises from the variable
 *    number of days in a month.  Having leap years makes it worse.  It's
 *    faster if we store the starting dates for months.  We can
 *    make finding cached items very fast.
 *    Keep an array of <TDate, m/y> pairs, one for each month for many years.
 *    To convert from TDate d to m/d/y:
 *         find element e in cache such that e.date <= d
 *         return e.mdy, after replacing  e.day with d - e.date + 1
 *    To convert from m/d/y m to TDate d:
 *         find element e in cache such that e.month=m.month && e.year==m.year
 *         return e.date + m.day - 1
 *
 *  gDateCacheArray  is initialized to contain the TDate,Month,Year for the
 *   first day of each month from 1/1/1995 until 12/1/2119
 */

typedef struct                         /* One entry for each month in cache */
{
    TDate           date;              /* Date of first day in cached month  */
    short int       month;             /* month= 1..12 */
    short int       year;              /* E.g. 1996  (NOT 96) */
} TDateCacheEntry;

static TDateCacheEntry gDateCacheArray[] = {
{143905,1,1995},{143936,2,1995},{143964,3,1995},{143995,4,1995},
{144025,5,1995},{144056,6,1995},{144086,7,1995},{144117,8,1995},
{144148,9,1995},{144178,10,1995},{144209,11,1995},{144239,12,1995},
{144270,1,1996},{144301,2,1996},{144330,3,1996},{144361,4,1996},
{144391,5,1996},{144422,6,1996},{144452,7,1996},{144483,8,1996},
{144514,9,1996},{144544,10,1996},{144575,11,1996},{144605,12,1996},
{144636,1,1997},{144667,2,1997},{144695,3,1997},{144726,4,1997},
{144756,5,1997},{144787,6,1997},{144817,7,1997},{144848,8,1997},
{144879,9,1997},{144909,10,1997},{144940,11,1997},{144970,12,1997},
{145001,1,1998},{145032,2,1998},{145060,3,1998},{145091,4,1998},
{145121,5,1998},{145152,6,1998},{145182,7,1998},{145213,8,1998},
{145244,9,1998},{145274,10,1998},{145305,11,1998},{145335,12,1998},
{145366,1,1999},{145397,2,1999},{145425,3,1999},{145456,4,1999},
{145486,5,1999},{145517,6,1999},{145547,7,1999},{145578,8,1999},
{145609,9,1999},{145639,10,1999},{145670,11,1999},{145700,12,1999},
{145731,1,2000},{145762,2,2000},{145791,3,2000},{145822,4,2000},
{145852,5,2000},{145883,6,2000},{145913,7,2000},{145944,8,2000},
{145975,9,2000},{146005,10,2000},{146036,11,2000},{146066,12,2000},
{146097,1,2001},{146128,2,2001},{146156,3,2001},{146187,4,2001},
{146217,5,2001},{146248,6,2001},{146278,7,2001},{146309,8,2001},
{146340,9,2001},{146370,10,2001},{146401,11,2001},{146431,12,2001},
{146462,1,2002},{146493,2,2002},{146521,3,2002},{146552,4,2002},
{146582,5,2002},{146613,6,2002},{146643,7,2002},{146674,8,2002},
{146705,9,2002},{146735,10,2002},{146766,11,2002},{146796,12,2002},
{146827,1,2003},{146858,2,2003},{146886,3,2003},{146917,4,2003},
{146947,5,2003},{146978,6,2003},{147008,7,2003},{147039,8,2003},
{147070,9,2003},{147100,10,2003},{147131,11,2003},{147161,12,2003},
{147192,1,2004},{147223,2,2004},{147252,3,2004},{147283,4,2004},
{147313,5,2004},{147344,6,2004},{147374,7,2004},{147405,8,2004},
{147436,9,2004},{147466,10,2004},{147497,11,2004},{147527,12,2004},
{147558,1,2005},{147589,2,2005},{147617,3,2005},{147648,4,2005},
{147678,5,2005},{147709,6,2005},{147739,7,2005},{147770,8,2005},
{147801,9,2005},{147831,10,2005},{147862,11,2005},{147892,12,2005},
{147923,1,2006},{147954,2,2006},{147982,3,2006},{148013,4,2006},
{148043,5,2006},{148074,6,2006},{148104,7,2006},{148135,8,2006},
{148166,9,2006},{148196,10,2006},{148227,11,2006},{148257,12,2006},
{148288,1,2007},{148319,2,2007},{148347,3,2007},{148378,4,2007},
{148408,5,2007},{148439,6,2007},{148469,7,2007},{148500,8,2007},
{148531,9,2007},{148561,10,2007},{148592,11,2007},{148622,12,2007},
{148653,1,2008},{148684,2,2008},{148713,3,2008},{148744,4,2008},
{148774,5,2008},{148805,6,2008},{148835,7,2008},{148866,8,2008},
{148897,9,2008},{148927,10,2008},{148958,11,2008},{148988,12,2008},
{149019,1,2009},{149050,2,2009},{149078,3,2009},{149109,4,2009},
{149139,5,2009},{149170,6,2009},{149200,7,2009},{149231,8,2009},
{149262,9,2009},{149292,10,2009},{149323,11,2009},{149353,12,2009},
{149384,1,2010},{149415,2,2010},{149443,3,2010},{149474,4,2010},
{149504,5,2010},{149535,6,2010},{149565,7,2010},{149596,8,2010},
{149627,9,2010},{149657,10,2010},{149688,11,2010},{149718,12,2010},
{149749,1,2011},{149780,2,2011},{149808,3,2011},{149839,4,2011},
{149869,5,2011},{149900,6,2011},{149930,7,2011},{149961,8,2011},
{149992,9,2011},{150022,10,2011},{150053,11,2011},{150083,12,2011},
{150114,1,2012},{150145,2,2012},{150174,3,2012},{150205,4,2012},
{150235,5,2012},{150266,6,2012},{150296,7,2012},{150327,8,2012},
{150358,9,2012},{150388,10,2012},{150419,11,2012},{150449,12,2012},
{150480,1,2013},{150511,2,2013},{150539,3,2013},{150570,4,2013},
{150600,5,2013},{150631,6,2013},{150661,7,2013},{150692,8,2013},
{150723,9,2013},{150753,10,2013},{150784,11,2013},{150814,12,2013},
{150845,1,2014},{150876,2,2014},{150904,3,2014},{150935,4,2014},
{150965,5,2014},{150996,6,2014},{151026,7,2014},{151057,8,2014},
{151088,9,2014},{151118,10,2014},{151149,11,2014},{151179,12,2014},
{151210,1,2015},{151241,2,2015},{151269,3,2015},{151300,4,2015},
{151330,5,2015},{151361,6,2015},{151391,7,2015},{151422,8,2015},
{151453,9,2015},{151483,10,2015},{151514,11,2015},{151544,12,2015},
{151575,1,2016},{151606,2,2016},{151635,3,2016},{151666,4,2016},
{151696,5,2016},{151727,6,2016},{151757,7,2016},{151788,8,2016},
{151819,9,2016},{151849,10,2016},{151880,11,2016},{151910,12,2016},
{151941,1,2017},{151972,2,2017},{152000,3,2017},{152031,4,2017},
{152061,5,2017},{152092,6,2017},{152122,7,2017},{152153,8,2017},
{152184,9,2017},{152214,10,2017},{152245,11,2017},{152275,12,2017},
{152306,1,2018},{152337,2,2018},{152365,3,2018},{152396,4,2018},
{152426,5,2018},{152457,6,2018},{152487,7,2018},{152518,8,2018},
{152549,9,2018},{152579,10,2018},{152610,11,2018},{152640,12,2018},
{152671,1,2019},{152702,2,2019},{152730,3,2019},{152761,4,2019},
{152791,5,2019},{152822,6,2019},{152852,7,2019},{152883,8,2019},
{152914,9,2019},{152944,10,2019},{152975,11,2019},{153005,12,2019},
{153036,1,2020},{153067,2,2020},{153096,3,2020},{153127,4,2020},
{153157,5,2020},{153188,6,2020},{153218,7,2020},{153249,8,2020},
{153280,9,2020},{153310,10,2020},{153341,11,2020},{153371,12,2020},
{153402,1,2021},{153433,2,2021},{153461,3,2021},{153492,4,2021},
{153522,5,2021},{153553,6,2021},{153583,7,2021},{153614,8,2021},
{153645,9,2021},{153675,10,2021},{153706,11,2021},{153736,12,2021},
{153767,1,2022},{153798,2,2022},{153826,3,2022},{153857,4,2022},
{153887,5,2022},{153918,6,2022},{153948,7,2022},{153979,8,2022},
{154010,9,2022},{154040,10,2022},{154071,11,2022},{154101,12,2022},
{154132,1,2023},{154163,2,2023},{154191,3,2023},{154222,4,2023},
{154252,5,2023},{154283,6,2023},{154313,7,2023},{154344,8,2023},
{154375,9,2023},{154405,10,2023},{154436,11,2023},{154466,12,2023},
{154497,1,2024},{154528,2,2024},{154557,3,2024},{154588,4,2024},
{154618,5,2024},{154649,6,2024},{154679,7,2024},{154710,8,2024},
{154741,9,2024},{154771,10,2024},{154802,11,2024},{154832,12,2024},
{154863,1,2025},{154894,2,2025},{154922,3,2025},{154953,4,2025},
{154983,5,2025},{155014,6,2025},{155044,7,2025},{155075,8,2025},
{155106,9,2025},{155136,10,2025},{155167,11,2025},{155197,12,2025},
{155228,1,2026},{155259,2,2026},{155287,3,2026},{155318,4,2026},
{155348,5,2026},{155379,6,2026},{155409,7,2026},{155440,8,2026},
{155471,9,2026},{155501,10,2026},{155532,11,2026},{155562,12,2026},
{155593,1,2027},{155624,2,2027},{155652,3,2027},{155683,4,2027},
{155713,5,2027},{155744,6,2027},{155774,7,2027},{155805,8,2027},
{155836,9,2027},{155866,10,2027},{155897,11,2027},{155927,12,2027},
{155958,1,2028},{155989,2,2028},{156018,3,2028},{156049,4,2028},
{156079,5,2028},{156110,6,2028},{156140,7,2028},{156171,8,2028},
{156202,9,2028},{156232,10,2028},{156263,11,2028},{156293,12,2028},
{156324,1,2029},{156355,2,2029},{156383,3,2029},{156414,4,2029},
{156444,5,2029},{156475,6,2029},{156505,7,2029},{156536,8,2029},
{156567,9,2029},{156597,10,2029},{156628,11,2029},{156658,12,2029},
{156689,1,2030},{156720,2,2030},{156748,3,2030},{156779,4,2030},
{156809,5,2030},{156840,6,2030},{156870,7,2030},{156901,8,2030},
{156932,9,2030},{156962,10,2030},{156993,11,2030},{157023,12,2030},
{157054,1,2031},{157085,2,2031},{157113,3,2031},{157144,4,2031},
{157174,5,2031},{157205,6,2031},{157235,7,2031},{157266,8,2031},
{157297,9,2031},{157327,10,2031},{157358,11,2031},{157388,12,2031},
{157419,1,2032},{157450,2,2032},{157479,3,2032},{157510,4,2032},
{157540,5,2032},{157571,6,2032},{157601,7,2032},{157632,8,2032},
{157663,9,2032},{157693,10,2032},{157724,11,2032},{157754,12,2032},
{157785,1,2033},{157816,2,2033},{157844,3,2033},{157875,4,2033},
{157905,5,2033},{157936,6,2033},{157966,7,2033},{157997,8,2033},
{158028,9,2033},{158058,10,2033},{158089,11,2033},{158119,12,2033},
{158150,1,2034},{158181,2,2034},{158209,3,2034},{158240,4,2034},
{158270,5,2034},{158301,6,2034},{158331,7,2034},{158362,8,2034},
{158393,9,2034},{158423,10,2034},{158454,11,2034},{158484,12,2034},
{158515,1,2035},{158546,2,2035},{158574,3,2035},{158605,4,2035},
{158635,5,2035},{158666,6,2035},{158696,7,2035},{158727,8,2035},
{158758,9,2035},{158788,10,2035},{158819,11,2035},{158849,12,2035},
{158880,1,2036},{158911,2,2036},{158940,3,2036},{158971,4,2036},
{159001,5,2036},{159032,6,2036},{159062,7,2036},{159093,8,2036},
{159124,9,2036},{159154,10,2036},{159185,11,2036},{159215,12,2036},
{159246,1,2037},{159277,2,2037},{159305,3,2037},{159336,4,2037},
{159366,5,2037},{159397,6,2037},{159427,7,2037},{159458,8,2037},
{159489,9,2037},{159519,10,2037},{159550,11,2037},{159580,12,2037},
{159611,1,2038},{159642,2,2038},{159670,3,2038},{159701,4,2038},
{159731,5,2038},{159762,6,2038},{159792,7,2038},{159823,8,2038},
{159854,9,2038},{159884,10,2038},{159915,11,2038},{159945,12,2038},
{159976,1,2039},{160007,2,2039},{160035,3,2039},{160066,4,2039},
{160096,5,2039},{160127,6,2039},{160157,7,2039},{160188,8,2039},
{160219,9,2039},{160249,10,2039},{160280,11,2039},{160310,12,2039},
{160341,1,2040},{160372,2,2040},{160401,3,2040},{160432,4,2040},
{160462,5,2040},{160493,6,2040},{160523,7,2040},{160554,8,2040},
{160585,9,2040},{160615,10,2040},{160646,11,2040},{160676,12,2040},
{160707,1,2041},{160738,2,2041},{160766,3,2041},{160797,4,2041},
{160827,5,2041},{160858,6,2041},{160888,7,2041},{160919,8,2041},
{160950,9,2041},{160980,10,2041},{161011,11,2041},{161041,12,2041},
{161072,1,2042},{161103,2,2042},{161131,3,2042},{161162,4,2042},
{161192,5,2042},{161223,6,2042},{161253,7,2042},{161284,8,2042},
{161315,9,2042},{161345,10,2042},{161376,11,2042},{161406,12,2042},
{161437,1,2043},{161468,2,2043},{161496,3,2043},{161527,4,2043},
{161557,5,2043},{161588,6,2043},{161618,7,2043},{161649,8,2043},
{161680,9,2043},{161710,10,2043},{161741,11,2043},{161771,12,2043},
{161802,1,2044},{161833,2,2044},{161862,3,2044},{161893,4,2044},
{161923,5,2044},{161954,6,2044},{161984,7,2044},{162015,8,2044},
{162046,9,2044},{162076,10,2044},{162107,11,2044},{162137,12,2044},
{162168,1,2045},{162199,2,2045},{162227,3,2045},{162258,4,2045},
{162288,5,2045},{162319,6,2045},{162349,7,2045},{162380,8,2045},
{162411,9,2045},{162441,10,2045},{162472,11,2045},{162502,12,2045},
{162533,1,2046},{162564,2,2046},{162592,3,2046},{162623,4,2046},
{162653,5,2046},{162684,6,2046},{162714,7,2046},{162745,8,2046},
{162776,9,2046},{162806,10,2046},{162837,11,2046},{162867,12,2046},
{162898,1,2047},{162929,2,2047},{162957,3,2047},{162988,4,2047},
{163018,5,2047},{163049,6,2047},{163079,7,2047},{163110,8,2047},
{163141,9,2047},{163171,10,2047},{163202,11,2047},{163232,12,2047},
{163263,1,2048},{163294,2,2048},{163323,3,2048},{163354,4,2048},
{163384,5,2048},{163415,6,2048},{163445,7,2048},{163476,8,2048},
{163507,9,2048},{163537,10,2048},{163568,11,2048},{163598,12,2048},
{163629,1,2049},{163660,2,2049},{163688,3,2049},{163719,4,2049},
{163749,5,2049},{163780,6,2049},{163810,7,2049},{163841,8,2049},
{163872,9,2049},{163902,10,2049},{163933,11,2049},{163963,12,2049},
{163994,1,2050},{164025,2,2050},{164053,3,2050},{164084,4,2050},
{164114,5,2050},{164145,6,2050},{164175,7,2050},{164206,8,2050},
{164237,9,2050},{164267,10,2050},{164298,11,2050},{164328,12,2050},
{164359,1,2051},{164390,2,2051},{164418,3,2051},{164449,4,2051},
{164479,5,2051},{164510,6,2051},{164540,7,2051},{164571,8,2051},
{164602,9,2051},{164632,10,2051},{164663,11,2051},{164693,12,2051},
{164724,1,2052},{164755,2,2052},{164784,3,2052},{164815,4,2052},
{164845,5,2052},{164876,6,2052},{164906,7,2052},{164937,8,2052},
{164968,9,2052},{164998,10,2052},{165029,11,2052},{165059,12,2052},
{165090,1,2053},{165121,2,2053},{165149,3,2053},{165180,4,2053},
{165210,5,2053},{165241,6,2053},{165271,7,2053},{165302,8,2053},
{165333,9,2053},{165363,10,2053},{165394,11,2053},{165424,12,2053},
{165455,1,2054},{165486,2,2054},{165514,3,2054},{165545,4,2054},
{165575,5,2054},{165606,6,2054},{165636,7,2054},{165667,8,2054},
{165698,9,2054},{165728,10,2054},{165759,11,2054},{165789,12,2054},
{165820,1,2055},{165851,2,2055},{165879,3,2055},{165910,4,2055},
{165940,5,2055},{165971,6,2055},{166001,7,2055},{166032,8,2055},
{166063,9,2055},{166093,10,2055},{166124,11,2055},{166154,12,2055},
{166185,1,2056},{166216,2,2056},{166245,3,2056},{166276,4,2056},
{166306,5,2056},{166337,6,2056},{166367,7,2056},{166398,8,2056},
{166429,9,2056},{166459,10,2056},{166490,11,2056},{166520,12,2056},
{166551,1,2057},{166582,2,2057},{166610,3,2057},{166641,4,2057},
{166671,5,2057},{166702,6,2057},{166732,7,2057},{166763,8,2057},
{166794,9,2057},{166824,10,2057},{166855,11,2057},{166885,12,2057},
{166916,1,2058},{166947,2,2058},{166975,3,2058},{167006,4,2058},
{167036,5,2058},{167067,6,2058},{167097,7,2058},{167128,8,2058},
{167159,9,2058},{167189,10,2058},{167220,11,2058},{167250,12,2058},
{167281,1,2059},{167312,2,2059},{167340,3,2059},{167371,4,2059},
{167401,5,2059},{167432,6,2059},{167462,7,2059},{167493,8,2059},
{167524,9,2059},{167554,10,2059},{167585,11,2059},{167615,12,2059},
{167646,1,2060},{167677,2,2060},{167706,3,2060},{167737,4,2060},
{167767,5,2060},{167798,6,2060},{167828,7,2060},{167859,8,2060},
{167890,9,2060},{167920,10,2060},{167951,11,2060},{167981,12,2060},
{168012,1,2061},{168043,2,2061},{168071,3,2061},{168102,4,2061},
{168132,5,2061},{168163,6,2061},{168193,7,2061},{168224,8,2061},
{168255,9,2061},{168285,10,2061},{168316,11,2061},{168346,12,2061},
{168377,1,2062},{168408,2,2062},{168436,3,2062},{168467,4,2062},
{168497,5,2062},{168528,6,2062},{168558,7,2062},{168589,8,2062},
{168620,9,2062},{168650,10,2062},{168681,11,2062},{168711,12,2062},
{168742,1,2063},{168773,2,2063},{168801,3,2063},{168832,4,2063},
{168862,5,2063},{168893,6,2063},{168923,7,2063},{168954,8,2063},
{168985,9,2063},{169015,10,2063},{169046,11,2063},{169076,12,2063},
{169107,1,2064},{169138,2,2064},{169167,3,2064},{169198,4,2064},
{169228,5,2064},{169259,6,2064},{169289,7,2064},{169320,8,2064},
{169351,9,2064},{169381,10,2064},{169412,11,2064},{169442,12,2064},
{169473,1,2065},{169504,2,2065},{169532,3,2065},{169563,4,2065},
{169593,5,2065},{169624,6,2065},{169654,7,2065},{169685,8,2065},
{169716,9,2065},{169746,10,2065},{169777,11,2065},{169807,12,2065},
{169838,1,2066},{169869,2,2066},{169897,3,2066},{169928,4,2066},
{169958,5,2066},{169989,6,2066},{170019,7,2066},{170050,8,2066},
{170081,9,2066},{170111,10,2066},{170142,11,2066},{170172,12,2066},
{170203,1,2067},{170234,2,2067},{170262,3,2067},{170293,4,2067},
{170323,5,2067},{170354,6,2067},{170384,7,2067},{170415,8,2067},
{170446,9,2067},{170476,10,2067},{170507,11,2067},{170537,12,2067},
{170568,1,2068},{170599,2,2068},{170628,3,2068},{170659,4,2068},
{170689,5,2068},{170720,6,2068},{170750,7,2068},{170781,8,2068},
{170812,9,2068},{170842,10,2068},{170873,11,2068},{170903,12,2068},
{170934,1,2069},{170965,2,2069},{170993,3,2069},{171024,4,2069},
{171054,5,2069},{171085,6,2069},{171115,7,2069},{171146,8,2069},
{171177,9,2069},{171207,10,2069},{171238,11,2069},{171268,12,2069},
{171299,1,2070},{171330,2,2070},{171358,3,2070},{171389,4,2070},
{171419,5,2070},{171450,6,2070},{171480,7,2070},{171511,8,2070},
{171542,9,2070},{171572,10,2070},{171603,11,2070},{171633,12,2070},
{171664,1,2071},{171695,2,2071},{171723,3,2071},{171754,4,2071},
{171784,5,2071},{171815,6,2071},{171845,7,2071},{171876,8,2071},
{171907,9,2071},{171937,10,2071},{171968,11,2071},{171998,12,2071},
{172029,1,2072},{172060,2,2072},{172089,3,2072},{172120,4,2072},
{172150,5,2072},{172181,6,2072},{172211,7,2072},{172242,8,2072},
{172273,9,2072},{172303,10,2072},{172334,11,2072},{172364,12,2072},
{172395,1,2073},{172426,2,2073},{172454,3,2073},{172485,4,2073},
{172515,5,2073},{172546,6,2073},{172576,7,2073},{172607,8,2073},
{172638,9,2073},{172668,10,2073},{172699,11,2073},{172729,12,2073},
{172760,1,2074},{172791,2,2074},{172819,3,2074},{172850,4,2074},
{172880,5,2074},{172911,6,2074},{172941,7,2074},{172972,8,2074},
{173003,9,2074},{173033,10,2074},{173064,11,2074},{173094,12,2074},
{173125,1,2075},{173156,2,2075},{173184,3,2075},{173215,4,2075},
{173245,5,2075},{173276,6,2075},{173306,7,2075},{173337,8,2075},
{173368,9,2075},{173398,10,2075},{173429,11,2075},{173459,12,2075},
{173490,1,2076},{173521,2,2076},{173550,3,2076},{173581,4,2076},
{173611,5,2076},{173642,6,2076},{173672,7,2076},{173703,8,2076},
{173734,9,2076},{173764,10,2076},{173795,11,2076},{173825,12,2076},
{173856,1,2077},{173887,2,2077},{173915,3,2077},{173946,4,2077},
{173976,5,2077},{174007,6,2077},{174037,7,2077},{174068,8,2077},
{174099,9,2077},{174129,10,2077},{174160,11,2077},{174190,12,2077},
{174221,1,2078},{174252,2,2078},{174280,3,2078},{174311,4,2078},
{174341,5,2078},{174372,6,2078},{174402,7,2078},{174433,8,2078},
{174464,9,2078},{174494,10,2078},{174525,11,2078},{174555,12,2078},
{174586,1,2079},{174617,2,2079},{174645,3,2079},{174676,4,2079},
{174706,5,2079},{174737,6,2079},{174767,7,2079},{174798,8,2079},
{174829,9,2079},{174859,10,2079},{174890,11,2079},{174920,12,2079},
{174951,1,2080},{174982,2,2080},{175011,3,2080},{175042,4,2080},
{175072,5,2080},{175103,6,2080},{175133,7,2080},{175164,8,2080},
{175195,9,2080},{175225,10,2080},{175256,11,2080},{175286,12,2080},
{175317,1,2081},{175348,2,2081},{175376,3,2081},{175407,4,2081},
{175437,5,2081},{175468,6,2081},{175498,7,2081},{175529,8,2081},
{175560,9,2081},{175590,10,2081},{175621,11,2081},{175651,12,2081},
{175682,1,2082},{175713,2,2082},{175741,3,2082},{175772,4,2082},
{175802,5,2082},{175833,6,2082},{175863,7,2082},{175894,8,2082},
{175925,9,2082},{175955,10,2082},{175986,11,2082},{176016,12,2082},
{176047,1,2083},{176078,2,2083},{176106,3,2083},{176137,4,2083},
{176167,5,2083},{176198,6,2083},{176228,7,2083},{176259,8,2083},
{176290,9,2083},{176320,10,2083},{176351,11,2083},{176381,12,2083},
{176412,1,2084},{176443,2,2084},{176472,3,2084},{176503,4,2084},
{176533,5,2084},{176564,6,2084},{176594,7,2084},{176625,8,2084},
{176656,9,2084},{176686,10,2084},{176717,11,2084},{176747,12,2084},
{176778,1,2085},{176809,2,2085},{176837,3,2085},{176868,4,2085},
{176898,5,2085},{176929,6,2085},{176959,7,2085},{176990,8,2085},
{177021,9,2085},{177051,10,2085},{177082,11,2085},{177112,12,2085},
{177143,1,2086},{177174,2,2086},{177202,3,2086},{177233,4,2086},
{177263,5,2086},{177294,6,2086},{177324,7,2086},{177355,8,2086},
{177386,9,2086},{177416,10,2086},{177447,11,2086},{177477,12,2086},
{177508,1,2087},{177539,2,2087},{177567,3,2087},{177598,4,2087},
{177628,5,2087},{177659,6,2087},{177689,7,2087},{177720,8,2087},
{177751,9,2087},{177781,10,2087},{177812,11,2087},{177842,12,2087},
{177873,1,2088},{177904,2,2088},{177933,3,2088},{177964,4,2088},
{177994,5,2088},{178025,6,2088},{178055,7,2088},{178086,8,2088},
{178117,9,2088},{178147,10,2088},{178178,11,2088},{178208,12,2088},
{178239,1,2089},{178270,2,2089},{178298,3,2089},{178329,4,2089},
{178359,5,2089},{178390,6,2089},{178420,7,2089},{178451,8,2089},
{178482,9,2089},{178512,10,2089},{178543,11,2089},{178573,12,2089},
{178604,1,2090},{178635,2,2090},{178663,3,2090},{178694,4,2090},
{178724,5,2090},{178755,6,2090},{178785,7,2090},{178816,8,2090},
{178847,9,2090},{178877,10,2090},{178908,11,2090},{178938,12,2090},
{178969,1,2091},{179000,2,2091},{179028,3,2091},{179059,4,2091},
{179089,5,2091},{179120,6,2091},{179150,7,2091},{179181,8,2091},
{179212,9,2091},{179242,10,2091},{179273,11,2091},{179303,12,2091},
{179334,1,2092},{179365,2,2092},{179394,3,2092},{179425,4,2092},
{179455,5,2092},{179486,6,2092},{179516,7,2092},{179547,8,2092},
{179578,9,2092},{179608,10,2092},{179639,11,2092},{179669,12,2092},
{179700,1,2093},{179731,2,2093},{179759,3,2093},{179790,4,2093},
{179820,5,2093},{179851,6,2093},{179881,7,2093},{179912,8,2093},
{179943,9,2093},{179973,10,2093},{180004,11,2093},{180034,12,2093},
{180065,1,2094},{180096,2,2094},{180124,3,2094},{180155,4,2094},
{180185,5,2094},{180216,6,2094},{180246,7,2094},{180277,8,2094},
{180308,9,2094},{180338,10,2094},{180369,11,2094},{180399,12,2094},
{180430,1,2095},{180461,2,2095},{180489,3,2095},{180520,4,2095},
{180550,5,2095},{180581,6,2095},{180611,7,2095},{180642,8,2095},
{180673,9,2095},{180703,10,2095},{180734,11,2095},{180764,12,2095},
{180795,1,2096},{180826,2,2096},{180855,3,2096},{180886,4,2096},
{180916,5,2096},{180947,6,2096},{180977,7,2096},{181008,8,2096},
{181039,9,2096},{181069,10,2096},{181100,11,2096},{181130,12,2096},
{181161,1,2097},{181192,2,2097},{181220,3,2097},{181251,4,2097},
{181281,5,2097},{181312,6,2097},{181342,7,2097},{181373,8,2097},
{181404,9,2097},{181434,10,2097},{181465,11,2097},{181495,12,2097},
{181526,1,2098},{181557,2,2098},{181585,3,2098},{181616,4,2098},
{181646,5,2098},{181677,6,2098},{181707,7,2098},{181738,8,2098},
{181769,9,2098},{181799,10,2098},{181830,11,2098},{181860,12,2098},
{181891,1,2099},{181922,2,2099},{181950,3,2099},{181981,4,2099},
{182011,5,2099},{182042,6,2099},{182072,7,2099},{182103,8,2099},
{182134,9,2099},{182164,10,2099},{182195,11,2099},{182225,12,2099},
{182256,1,2100},{182287,2,2100},{182315,3,2100},{182346,4,2100},
{182376,5,2100},{182407,6,2100},{182437,7,2100},{182468,8,2100},
{182499,9,2100},{182529,10,2100},{182560,11,2100},{182590,12,2100},
{182621,1,2101},{182652,2,2101},{182680,3,2101},{182711,4,2101},
{182741,5,2101},{182772,6,2101},{182802,7,2101},{182833,8,2101},
{182864,9,2101},{182894,10,2101},{182925,11,2101},{182955,12,2101},
{182986,1,2102},{183017,2,2102},{183045,3,2102},{183076,4,2102},
{183106,5,2102},{183137,6,2102},{183167,7,2102},{183198,8,2102},
{183229,9,2102},{183259,10,2102},{183290,11,2102},{183320,12,2102},
{183351,1,2103},{183382,2,2103},{183410,3,2103},{183441,4,2103},
{183471,5,2103},{183502,6,2103},{183532,7,2103},{183563,8,2103},
{183594,9,2103},{183624,10,2103},{183655,11,2103},{183685,12,2103},
{183716,1,2104},{183747,2,2104},{183776,3,2104},{183807,4,2104},
{183837,5,2104},{183868,6,2104},{183898,7,2104},{183929,8,2104},
{183960,9,2104},{183990,10,2104},{184021,11,2104},{184051,12,2104},
{184082,1,2105},{184113,2,2105},{184141,3,2105},{184172,4,2105},
{184202,5,2105},{184233,6,2105},{184263,7,2105},{184294,8,2105},
{184325,9,2105},{184355,10,2105},{184386,11,2105},{184416,12,2105},
{184447,1,2106},{184478,2,2106},{184506,3,2106},{184537,4,2106},
{184567,5,2106},{184598,6,2106},{184628,7,2106},{184659,8,2106},
{184690,9,2106},{184720,10,2106},{184751,11,2106},{184781,12,2106},
{184812,1,2107},{184843,2,2107},{184871,3,2107},{184902,4,2107},
{184932,5,2107},{184963,6,2107},{184993,7,2107},{185024,8,2107},
{185055,9,2107},{185085,10,2107},{185116,11,2107},{185146,12,2107},
{185177,1,2108},{185208,2,2108},{185237,3,2108},{185268,4,2108},
{185298,5,2108},{185329,6,2108},{185359,7,2108},{185390,8,2108},
{185421,9,2108},{185451,10,2108},{185482,11,2108},{185512,12,2108},
{185543,1,2109},{185574,2,2109},{185602,3,2109},{185633,4,2109},
{185663,5,2109},{185694,6,2109},{185724,7,2109},{185755,8,2109},
{185786,9,2109},{185816,10,2109},{185847,11,2109},{185877,12,2109},
{185908,1,2110},{185939,2,2110},{185967,3,2110},{185998,4,2110},
{186028,5,2110},{186059,6,2110},{186089,7,2110},{186120,8,2110},
{186151,9,2110},{186181,10,2110},{186212,11,2110},{186242,12,2110},
{186273,1,2111},{186304,2,2111},{186332,3,2111},{186363,4,2111},
{186393,5,2111},{186424,6,2111},{186454,7,2111},{186485,8,2111},
{186516,9,2111},{186546,10,2111},{186577,11,2111},{186607,12,2111},
{186638,1,2112},{186669,2,2112},{186698,3,2112},{186729,4,2112},
{186759,5,2112},{186790,6,2112},{186820,7,2112},{186851,8,2112},
{186882,9,2112},{186912,10,2112},{186943,11,2112},{186973,12,2112},
{187004,1,2113},{187035,2,2113},{187063,3,2113},{187094,4,2113},
{187124,5,2113},{187155,6,2113},{187185,7,2113},{187216,8,2113},
{187247,9,2113},{187277,10,2113},{187308,11,2113},{187338,12,2113},
{187369,1,2114},{187400,2,2114},{187428,3,2114},{187459,4,2114},
{187489,5,2114},{187520,6,2114},{187550,7,2114},{187581,8,2114},
{187612,9,2114},{187642,10,2114},{187673,11,2114},{187703,12,2114},
{187734,1,2115},{187765,2,2115},{187793,3,2115},{187824,4,2115},
{187854,5,2115},{187885,6,2115},{187915,7,2115},{187946,8,2115},
{187977,9,2115},{188007,10,2115},{188038,11,2115},{188068,12,2115},
{188099,1,2116},{188130,2,2116},{188159,3,2116},{188190,4,2116},
{188220,5,2116},{188251,6,2116},{188281,7,2116},{188312,8,2116},
{188343,9,2116},{188373,10,2116},{188404,11,2116},{188434,12,2116},
{188465,1,2117},{188496,2,2117},{188524,3,2117},{188555,4,2117},
{188585,5,2117},{188616,6,2117},{188646,7,2117},{188677,8,2117},
{188708,9,2117},{188738,10,2117},{188769,11,2117},{188799,12,2117},
{188830,1,2118},{188861,2,2118},{188889,3,2118},{188920,4,2118},
{188950,5,2118},{188981,6,2118},{189011,7,2118},{189042,8,2118},
{189073,9,2118},{189103,10,2118},{189134,11,2118},{189164,12,2118},
{189195,1,2119},{189226,2,2119},{189254,3,2119},{189285,4,2119},
{189315,5,2119},{189346,6,2119},{189376,7,2119},{189407,8,2119},
{189438,9,2119},{189468,10,2119},{189499,11,2119},{189529,12,2119}
};


#define JPMCDS_DATE_CACHE_NUM_ITEMS  (sizeof(gDateCacheArray)/sizeof(TDateCacheEntry))

/*
***************************************************************************
** Converts year, month, day to a date and returns the date. This is useful
** for initializing dates within test routines. Returns FAILURE for invalid
** dates.
***************************************************************************
*/
TDate JpmcdsDate
(long year,  /* (I) Year */
 long month, /* (I) Month */
 long day    /* (I) Day */
)
{
    static char routine[]="JpmcdsDate";

    TMonthDayYear mdy;
    TDate         date; /* returned */

    mdy.month = month;
    mdy.day   = day;
    mdy.year  = year;

    if (JpmcdsMDYToDate (&mdy, &date) != SUCCESS)
    {
        JpmcdsErrMsg ("%s: Failed.\n", routine);
        date = FAILURE;
    }

    return date;
}


/*
***************************************************************************
** Converts TDate to Month, Day, Year.
**
** NOTE: routine was written to avoid divides especially as well as 
** multiplies. Whenever possible, the type int is used-which is presumably 
** the fastest type on a given architecture.
***************************************************************************
*/
int JpmcdsDateToMDY
    (TDate         date,                /* (I) Days since 1/1/BASE_YEAR. */
     TMonthDayYear *mdy)                /* (O) Month/Day/Year format */
{
    static char routine[]="JpmcdsDateToMDY";
    long        day;
    int         month;
    int         year = JPMCDS_TDATE_BASE_YEAR;
    long        fourYearBlocks;
    int         count;
    long       *cumDaysp;

    /* Check if date is covered by cache
     */
    if (gDateCacheArray[0].date <= date &&
        date <= gDateCacheArray[JPMCDS_DATE_CACHE_NUM_ITEMS-1].date)
    {
        int i = (int) (date - gDateCacheArray[0].date);
        /* i = 12*(i/365) + (i%365)/29; */ /* make guess at proper index */
        i /= 29;                /* make guess at proper index */
        if (i>JPMCDS_DATE_CACHE_NUM_ITEMS-1)
        {
            i = JPMCDS_DATE_CACHE_NUM_ITEMS-1;
        }
        while (gDateCacheArray[i].date > date) /* can guess too high */
        {
            i--;
        }

        mdy->year =  gDateCacheArray[i].year;
        mdy->month = gDateCacheArray[i].month;
        mdy->day =   date - gDateCacheArray[i].date + 1;
        return SUCCESS;
    }



    if (date < 0)
    {
        JpmcdsErrMsg("%s: negative  TDate %ld received.\n", routine, date);
        return FAILURE;
    }

    /* Get year */
    while (date >= DAYS_IN_400_YEARS)
    {
        date -= DAYS_IN_400_YEARS;
        year += 400;
    }

    /* Go through this loop at most 3 times so that Dec 31 in the
     * year 2000, 2400, etc doesn't get moved to the year 2001, 2401.
     */
    for (count = 3; date >= DAYS_IN_100_YEARS && count > 0; count--)
    {
        date -= DAYS_IN_100_YEARS;
        year += 100;
    }

    /* Dont have to make sure we go through at most 24 times since
     * the last 4 years (of 100) has *less* (or the same number of)
     * days than the other groups of 4 years.
     */
    if (date >= DAYS_IN_4_YEARS)
    {
        fourYearBlocks = date/DAYS_IN_4_YEARS;
        date -= MULTIPLY_BY_1461(fourYearBlocks);
        year += (int)fourYearBlocks << 2;   /* Multiply by 4 */
    }

    /* Go through this loop at most 3 times so that Dec 31 in a leap
     * year does not get moved to the next year.
     */
    for (count = 3; date >= DAYS_IN_1_YEAR && count > 0; count--)
    {
        date -= DAYS_IN_1_YEAR;
        year += 1;
    }

    /* Get month and date
     */

    /* date/32 is a good lower bound for month. */
    month = (date >> 5) + 1;

    if (JPMCDS_IS_LEAP(year))
        cumDaysp = leapCumDays + month;
    else
        cumDaysp = cumDays + month;

    /* There is an extra increment and decrement of cumDaysp here, but
       it's necessary in order to set month correctly. */
    for ( ; date > *cumDaysp; month++)
       cumDaysp++;
    day = date - *(--cumDaysp);


    mdy->year = (long)year;
    mdy->month = (long)month;
    mdy->day = day;

    return SUCCESS;
}


/*f
***************************************************************************
** Converts Month, Day, Year to TDate.
**
** NOTE: routine was written to avoid divides (especially) and multiplies.
***************************************************************************
*/
int JpmcdsMDYToDate
    (TMonthDayYear *mdy,                /* (I) Month/Day/Year format */
     TDate         *odate)              /* (O) Days since 1/1/BASE_YEAR. */
{
    static char routine[]="JpmcdsMDYToDate";

    TDate    date = 0;                  /* Assign to odate at end */
    long     fourYearBlocks;
    int      year =  (int) mdy->year;
    int      month = (int) mdy->month;
    int      day =   (int) mdy->day;
    TBoolean isLeap;

    {
       /* Check if date is covered by cache */
       if (gDateCacheArray[0].year <= mdy->year &&
           mdy->year <= gDateCacheArray[JPMCDS_DATE_CACHE_NUM_ITEMS-1].year)
       {
           int i = 12*(year - gDateCacheArray[0].year) + month-1;  /* index */

           if (day < 1  ||  31 < day || month < 1  ||  12 < month)
           {
               JpmcdsErrMsg("%s: Invalid date: %d/%d/%d\n", routine,month,day,year);
               return FAILURE;
           }

           *odate = gDateCacheArray[i].date + day - 1;
           if (i < JPMCDS_DATE_CACHE_NUM_ITEMS-1 &&
               *odate >= gDateCacheArray[i+1].date)
           {   /* note: don't have to check last, as it's december */
               JpmcdsErrMsg("%s: Invalid date: %d/%d/%d\n", routine,month,day,year);
               return FAILURE;
           }

           return SUCCESS;
       }
     }

    year = year - JPMCDS_TDATE_BASE_YEAR;                  /* Avoid ptr derefs*/
    isLeap = JPMCDS_IS_LEAP(mdy->year);

    /* Make sure day is in range.
     */
    if (day >= 1 && day <= 28)
       /*EMPTY*/;                      /* Guaranteed to be OK */
                    /* Avoid doing check below */
    else if (day < 1 ||
        (isLeap ? day > leapDays[month] : day > (days[month])))
    {
        JpmcdsErrMsg("%s: date %ld/%ld/%ld  not valid.\n", routine, mdy->month, mdy->day, mdy->year);
        return FAILURE;
    }

    /* Make sure month and year are in range.
     */
    if (month < 1 || month > JPMCDS_MONTHS_PER_YEAR ||
        mdy->year < JPMCDS_TDATE_BASE_YEAR)
    {
        JpmcdsErrMsg("%s: date %ld/%ld/%ld is not valid.\n", routine, mdy->month, mdy->day, mdy->year);
        return FAILURE;
    }

    /* Take years into account */
    while (year >= 400)
    {
        year -= 400;
        date += DAYS_IN_400_YEARS;
    }

    while (year >= 100)
    {
        year -= 100;
        date += DAYS_IN_100_YEARS;
    }

    if (year >= 4)
    {
        fourYearBlocks = (long)(year>>2);       /* Divide by 4 */
        year -= (int)(fourYearBlocks<<2);       /* Multiply by 4 */
        date += MULTIPLY_BY_1461(fourYearBlocks);
    }

    while (year >= 1)
    {
        year -= 1;
        date += DAYS_IN_1_YEAR;
    }

    if (isLeap)
        date += leapCumDays[month-1] + day;
    else
        date += cumDays[month-1] + day;

    *odate = date;
    return SUCCESS;
}


/*
***************************************************************************
** Converts TDate to day of week (0-6)
** 0 is Sunday, 1 is Monday.... 6 is Saturday
***************************************************************************
*/
int JpmcdsDayOfWeek
    (TDate date,                        /* (I) Days since 1/1/BASE_YEAR. */
     long *dayOfWeek)                   /* (O) Day of week (0-6) */
{
    *dayOfWeek = (date+1) % 7;
    return SUCCESS;
}


/*f
***************************************************************************
** Normalizes a month/day/year. If month is out of range, it is brought 
** into range, and the years are incremented or decremented as appropriate.
** If day belongs to a month/year combination which does not exist, (such 
** as April 31) the day is reduced so that it becomes valid.(to April 30).
***************************************************************************
*/
int JpmcdsNormalizeMDY(TMonthDayYear *mdy)  /* (I/O) */
{
    static char routine[]="JpmcdsNormalizeMDY";

    int    month = (int) mdy->month;            /* store in local vars for */
    int    year =  (int) mdy->year;             /* ... speed */
    int    day =   (int) mdy->day;

    while (month < 1)
    {
        month += JPMCDS_MONTHS_PER_YEAR;
        year--;
    }

    while (month > JPMCDS_MONTHS_PER_YEAR)
    {
        month -= JPMCDS_MONTHS_PER_YEAR;
        year++;
    }

    if (day < 1 || day > MAX_DAYS_PER_MONTH)
    {
        JpmcdsErrMsg("%s: day %d invalid.\n", routine, day);
        return FAILURE;
    }

    if (JPMCDS_IS_LEAP(year))
    {
        if (day > leapDays[month])
        {
            day = leapDays[month];
        }
    }
    else
    {
        if (day > days[month])
        {
            day = days[month];
        }
    }

    mdy->month = month;
    mdy->year =  year;
    mdy->day  =  day;
    return SUCCESS;
}


/*
***************************************************************************
** Returns # days in month.
***************************************************************************
*/
int JpmcdsDaysInMonth
    (long year,                         /* (I) */
     long month                         /* (I) Month (1-12) */
     )
{
    if (JPMCDS_IS_LEAP(year))
        return leapDays[month];
    else
        return days[month];
}


/*
***************************************************************************
** Moves a date to the end of the current month.
***************************************************************************
*/
int JpmcdsDateToEOM
(TDate  inDate,  /* (I) Input date */
 TDate *outDate  /* (O) Output date - at end of month, in the same month */
)
{
    static char routine[] = "JpmcdsDateToEOM";
    int         status    = FAILURE;

    TMonthDayYear mdy;

    if (JpmcdsDateToMDY (inDate, &mdy) != SUCCESS)
        goto done;

    mdy.day = JpmcdsDaysInMonth (mdy.year, mdy.month);

    if (JpmcdsMDYToDate (&mdy, outDate) != SUCCESS)
        goto done;

    status = SUCCESS;

done:

    if (status != SUCCESS)
        JpmcdsErrMsgFailure (routine);

    return status;
}
