#!/usr/bin/python
# -*- coding: utf-8 -*-

# Hive Netius System
# Copyright (C) 2008-2015 Hive Solutions Lda.
#
# This file is part of Hive Netius System.
#
# Hive Netius System is free software: you can redistribute it and/or modify
# it under the terms of the Apache License as published by the Apache
# Foundation, either version 2.0 of the License, or (at your option) any
# later version.
#
# Hive Netius System is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# Apache License for more details.
#
# You should have received a copy of the Apache License along with
# Hive Netius System. If not, see <http://www.apache.org/licenses/>.

__author__ = "João Magalhães <joamag@hive.pt>"
""" The author(s) of the module """

__version__ = "1.0.0"
""" The version of the module """

__revision__ = "$LastChangedRevision$"
""" The revision number of the module """

__date__ = "$LastChangedDate$"
""" The last change date of the module """

__copyright__ = "Copyright (c) 2008-2015 Hive Solutions Lda."
""" The copyright for the module """

__license__ = "Apache License, Version 2.0"
""" The license for the module """

import threading

import netius

CALLABLE_WORK = 1

class Thread(threading.Thread):

    def __init__(self, identifier, owner = None, *args, **kwargs):
        threading.Thread.__init__(self, *args, **kwargs)
        self.identifier = identifier
        self.owner = owner
        self._run = False

    def stop(self):
        self._run = False

    def run(self):
        threading.Thread.run(self)
        self._run = True
        while self._run: self.tick()

    def tick(self):
        self.owner.condition.acquire()
        while not self.owner.peek() and self._run:
            self.owner.condition.wait()
        try:
            if not self._run: return
            work = self.owner.pop()
        finally:
            self.owner.condition.release()
        self.execute(work)

    def execute(self, work):
        type = work[0]
        if type == CALLABLE_WORK: work[1]()
        else: raise netius.NotImplemented("Cannot execute type '%d'" % type)

class ThreadPool(object):

    def __init__(self, base = Thread, count = 30):
        self.base = base
        self.count = count
        self.instances = []
        self.queue = []
        self.condition = threading.Condition()
        self._built = False

    def start(self):
        self.build()
        for instance in self.instances:
            instance.start()

    def stop(self, join = True):
        for instance in self.instances: instance.stop()
        self.condition.acquire()
        try: self.condition.notify_all()
        finally: self.condition.release()
        if not join: return
        for instance in self.instances: instance.join()

    def build(self):
        if self._built: return
        for index in range(self.count):
            instance = self.base(index, owner = self)
            self.instances.append(instance)
        self._built = True

    def peek(self):
        if not self.queue: return None
        return self.queue[0]

    def pop(self, lock = True):
        lock and self.condition.acquire()
        try: value = self.queue.pop(0)
        finally: lock and self.condition.release()
        return value

    def push(self, work, lock = True):
        lock and self.condition.acquire()
        try:
            value = self.queue.append(work)
            self.condition.notify()
        finally:
            lock and self.condition.release()
        return value

    def push_callable(self, callable):
        work = (CALLABLE_WORK, callable)
        self.push(work)
