#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os            as _os
import itertools     as _itertools

from Bio             import SeqIO as _SeqIO

from pygenome._pretty import pretty_str as _ps

from pygenome.systematic_name import _systematic_name

from pygenome._data_files import _chromosome_files

data_dir = _os.path.join( _os.getenv("pygenome_data_dir"), "Saccharomyces_cerevisiae")


def intergenic_sequence(upgene, dngene):
    '''
    Returns the intergenic sequence between two genes on the same chromosome.

    Parameters
    ----------
    upgene : str
        standard name (eg. CYC1) or a systematic name (eg. YJR048W)
    dngene : str
        standard name (eg. CYC1) or a systematic name (eg. YJR048W)

    Returns
    -------
    out : Bio.SeqRecord
        Bio.SeqRecord object

    See Also
    --------
    intergenic_sequence_genbank_accession

    Examples
    --------
    >>> from pygenome import sg
    >>> sg.gene["TDH3"]
    Gene TDH3/YGR192C
    >>> sg.gene["TDH3"].upstream_gene
    Gene PDX1/YGR193C
    >>> sg.gene["YGR193C"].upstream_gene
    Gene XKS1/YGR194C
    >>> from pygenome.intergenic import intergenic_sequence
    >>> len(intergenic_sequence("YGR192C", "YGR193C"))
    698
    >>> len(intergenic_sequence("YGR192C", "YGR194C"))
    2262
    >>>

    '''

    upgene = _systematic_name(upgene)
    dngene = _systematic_name(dngene)

    if not upgene and dngene:
        raise Exception("Both upgene and dngene are needed.")
    #print upgene[1], dngene[1]
    if upgene[1] != dngene[1]:
        raise Exception("Both genes has to be on the same chromosome.")

    _krom = _SeqIO.read(_os.path.join(data_dir, _chromosome_files[upgene[1]]),"gb")
    cds  = {f.qualifiers['locus_tag'][0] :  f for f in [f for f in _krom.features if f.type=="CDS"]}
    upfeature = cds[upgene]
    startup, stopup  = upfeature.location.start,upfeature.location.end
    dnfeature = cds[dngene]
    startdn, stopdn = dnfeature.location.start, dnfeature.location.end

    assert sorted( (startup, stopup, startdn, stopdn) ) == list(_itertools.chain.from_iterable(sorted( [sorted((startup,stopup)),sorted((startdn, stopdn))] )))

    length,a,b = min([ (abs(a-b), a, b) for a, b in _itertools.product((startup,stopup),(startdn,stopdn))])

    if a<b:
        igs = _krom[a:b]
        igs.description = "{} REGION: {}..{}".format(_krom.id, a+1,b)
    else:
        igs = _krom[b:a].reverse_complement()
        igs.description = "{} REGION: complement({}..{})".format(_krom.id, b+1,a)

    igs.name = _krom.name

    igs.id = _krom.id

    igs.pydna_code = _ps("gb = pydna.Genbank('my@email.com')\n"
                         "seq = gb.nucleotide('{}')".format(igs.description))
    return igs


if __name__ is "__main__":
    pass


