"""Wille App Module

Using Apps:

>>> import app

Load Fileserver app from folder

>>> fileserver_app = app.App('fileserver', '../../../apps')

App variables can be accessed:

>>> fileserver_app.name
'fileserver'

URL to app can be generated by invoking:

>>> fileserver_app.make_url('http://localhost')
'http://localhost/apps/fileserver'

App can be also reloaded when code changes:

>>> fileserver_app.reload()
True

"""

import mimetypes
import os
import sys
import imp
import traceback

from pyjavaproperties import Properties

import utils

DEFAULT_PROPERTIES_FILENAME = 'willeapp.properties'

class App:
	""" Base Class For Wille Apps """
	
	def __init__(self, name, path):
		# Basic vars
		self.code_loaded = False
		self.error_mode = False
		self.error_view = None 
		self.name = name
		self.path = path
		self.full_path = os.path.realpath( os.path.join(self.path, self.name) )
		self.urls = None
		self.properties_filename = \
			os.path.realpath( \
				os.path.join(self.full_path, DEFAULT_PROPERTIES_FILENAME) )
				
		# Load code & resolve urls
		self.__load_code()
		
		# load properties
		self.__load_properties()
		
		# Parse profiles from properties
		self.profiles = None
		if len(self.properties['profile']):
			self.profiles = self.properties['profile'].split(',')

	def __str__(self):
		return ('App(%s)' % (self.name) )
	
	def make_url(self, base_url):
		"""Make URL for the app using given base URL"""
		
		return ("%s/apps/%s" % (base_url, self.name))

	def reload(self):
		"""Reload App"""
		self.__load_code()
		self.__load_properties()
		return True
		
	def __load_properties(self):
		try:
			self.properties = Properties()
			self.properties.load(open(self.properties_filename))
		except Exception, e:
			pass
			# We no longer require properties file for apps
			# self.__set_error(PropertiesErrorView) 
		
	def __set_error(self, error_view):
		if not self.error_mode:
			self.error_mode = True
			self.error_view = error_view
			self.urls = ( ("/(.*)", error_view), )
		
	def __load_code(self):
		"""Load Actual App code
		
		   Raises an exception on error
		"""
		
		# Register app folder as a python module		
		if self.code_loaded:
			try:
				reload(sys.modules[self.name])
			except:
				pass
			return
		
		# Register python module
		utils.register_python_module(self.path, self.name)
		
		# Find module
		try:
			f, filename, description = imp.find_module(self.name)
		except:
			self.__set_error(SourceCodeErrorView)
		
		# Load module
		try:
			app = imp.load_module(self.name, f, filename, description)
			
			# Update URLs
			self.urls = app.urls
		
			# Mark code as loaded
			self.code_loaded = True
		except:
			self.__set_error(SourceCodeErrorView)
		
class SourceCodeErrorView:
	""" Displayed in case there is an error in source code """
	def POST(self, context, path):
		return self.GET(context, path)
	
	def GET(self, context, path):
		# Parse modname
		modname = os.path.split(context.workdir)[-1]
		error = 'Unknown error'
		stacktrace = 'Try restarting Wille Server'
		
		# Try to find the module
		found = False
		try:
			f, filename, description = imp.find_module(modname)
			found = True
		except:
			error = 'App module missing (%s.py)' % modname
			stacktrace = traceback.format_exc()			
		
		# Try to load the module
		if found:
			try:
				app = imp.load_module(modname, f, filename, description)
				
				# urls var found?
				try:
					app.urls
				except:
					error = 'URL mappings missing'
					stacktrace = 'Add <code>urls</code> var to %s.py with valid URL mappings' % (modname) 
			except:
				error = 'Error in %s.py' % modname 
				stacktrace = traceback.format_exc()						
				
		return '<h1>%s</h1><pre>%s</pre>' % (error, stacktrace)

class PropertiesErrorView:
	def GET(self, *args): 
		return '<h1>Broken properties file</h1><p>Create new <code>%s</code> with valid properties data</p>' % DEFAULT_PROPERTIES_FILENAME 

# Run tests 
if __name__ == "__main__":
	import doctest
	doctest.testmod()
	