# -*- coding: utf-8 -*-
"""
    glashammer.utils.forms
    ~~~~~~~~~~~~~~~~~~~~~~

    This module implements a sophisticated form validation and rendering
    system that is based on diva with concepts from django newforms and
    wtforms incorporated.

    It can validate nested structures and works in both ways.  It can also
    handle intelligent backredirects (via :mod:`glashammer.utils.http`) and supports
    basic CSRF protection.

    For usage informations see :class:`Form`

    :copyright: Copyright 2007-2008 by Armin Ronacher, Christopher Lenz.
    :license: GNU GPL.
"""
import re
from datetime import datetime
from unicodedata import normalize
from time import strptime
from itertools import chain
try:
    from hashlib import sha1
except ImportError:
    from sha import new as sha1

from werkzeug import html, escape, MultiDict

from glashammer.utils.local import get_request, url_for
from glashammer.bundles.sqladb import db
from glashammer.bundles.i18n import _, ngettext, parse_datetime, \
     format_system_datetime
lazy_gettext = lambda s: s
from glashammer.utils.http import get_redirect_target, redirect
from glashammer.utils.crypto import gen_random_identifier


class ValidationError(ValueError):
    """Exception raised when invalid data is encountered."""

    def __init__(self, message):
        if not isinstance(message, (list, tuple)):
            messages = [message]
        # make all items in the list unicode (this also evaluates
        # lazy translations in there)
        messages = map(unicode, messages)
        Exception.__init__(self, messages[0])

        self.messages = ErrorList(messages)

    def unpack(self, key=None):
        return {key: self.messages}


class DataIntegrityError(ValueError):
    """Raised if data was fed to the form handling system that was not
    generated by it.
    """


def _decode(data):
    """Decodes the flat dictionary d into a nested structure.

    >>> _decode({'foo': 'bar'})
    {'foo': 'bar'}
    >>> _decode({'foo.0': 'bar', 'foo.1': 'baz'})
    {'foo': ['bar', 'baz']}
    >>> data = _decode({'foo.bar': '1', 'foo.baz': '2'})
    >>> data == {'foo': {'bar': '1', 'baz': '2'}}
    True

    More complex mappings work too:

    >>> _decode({'foo.bar.0': 'baz', 'foo.bar.1': 'buzz'})
    {'foo': {'bar': ['baz', 'buzz']}}
    >>> _decode({'foo.0.bar': '23', 'foo.1.baz': '42'})
    {'foo': [{'bar': '23'}, {'baz': '42'}]}
    >>> _decode({'foo.0.0': '23', 'foo.0.1': '42'})
    {'foo': [['23', '42']]}
    >>> _decode({'foo': ['23', '42']})
    {'foo': ['23', '42']}

    Missing items in lists are ignored for convenience reasons:

    >>> _decode({'foo.42': 'a', 'foo.82': 'b'})
    {'foo': ['a', 'b']}

    This can be used for help client side DOM processing (inserting and
    deleting rows in dynamic forms).

    It also supports werkzeug's multi dicts:

    >>> _decode(MultiDict({"foo": ['1', '2']}))
    {'foo': ['1', '2']}
    >>> _decode(MultiDict({"foo.0": '1', "foo.1": '2'}))
    {'foo': ['1', '2']}

    Those two submission ways can also be used combined:

    >>> _decode(MultiDict({"foo": ['1'], "foo.0": '2', "foo.1": '3'}))
    {'foo': ['1', '2', '3']}

    This function will never raise exceptions except for argument errors
    but the recovery behavior for invalid form data is undefined.
    """
    list_marker = object()
    value_marker = object()

    if isinstance(data, MultiDict):
        listiter = data.iterlists()
    else:
        listiter = ((k, [v]) for k, v in data.iteritems())

    def _split_key(name):
        result = name.split('.')
        for idx, part in enumerate(result):
            if part.isdigit():
                result[idx] = int(part)
        return result

    def _enter_container(container, key):
        if key not in container:
            return container.setdefault(key, {list_marker: False})
        return container[key]

    def _convert(container):
        if value_marker in container:
            force_list = False
            values = container.pop(value_marker)
            if container.pop(list_marker):
                force_list = True
                values.extend(_convert(x[1]) for x in
                              sorted(container.items()))
            if not force_list and len(values) == 1:
                values = values[0]
            return values
        elif container.pop(list_marker):
            return [_convert(x[1]) for x in sorted(container.items())]
        return dict((k, _convert(v)) for k, v in container.iteritems())

    result = {list_marker: False}
    for key, values in listiter:
        parts = _split_key(key)
        if not parts:
            continue
        container = result
        for part in parts:
            last_container = container
            container = _enter_container(container, part)
            last_container[list_marker] = isinstance(part, (int, long))
        container[value_marker] = values[:]

    return _convert(result)


def _bind(obj, form, memo):
    """Helper for the field binding.  This is inspired by the way `deepcopy`
    is implemented.
    """
    if memo is None:
        memo = {}
    obj_id = id(obj)
    if obj_id in memo:
        return memo[obj_id]
    rv = obj._bind(form, memo)
    memo[obj_id] = rv
    return rv


def _force_dict(value, silent=False):
    """If the value is not a dict, raise an exception."""
    if value is None:
        return {}
    elif not isinstance(value, dict):
        if silent:
            return {}
        raise DataIntegrityError('expected a dict, got %r' %
                                 value.__class__.__name__)
    return value


def _force_list(value, silent=False):
    """If the value is not a list, raise an exception"""
    if value is None:
        return []
    try:
        if isinstance(value, basestring):
            raise TypeError()
        return list(value)
    except TypeError:
        if silent:
            return []
        raise DataIntegrityError('expected a list, got %r' %
                                 value.__class__.__name__)


def _make_widget(field, name, value, errors):
    """Shortcut for widget creation."""
    return field.widget(field, name, value, errors)


def _make_name(parent, child):
    """Joins a name."""
    if parent is None:
        result = child
    else:
        result = '%s.%s' % (parent, child)

    # try to return a ascii only bytestring if possible
    try:
        return str(result)
    except UnicodeError:
        return unicode(result)


def _to_string(value):
    """Convert a value to unicode, None means empty string."""
    if value is None:
        return u''
    return unicode(value)


def _to_list(value):
    """Similar to `_force_list` but always succeeds and never drops data."""
    if isinstance(value, basestring):
        return [value]
    try:
        return list(value)
    except TypeError:
        return [value]


def _iter_choices(choices):
    """Iterate over choices."""
    if choices is not None:
        for choice in choices:
            if not isinstance(choice, tuple):
                choice = (choice, choice)
            yield choice


class _ListSupport(object):
    """A mixin for iterable objects that yield html."""

    def as_ul(self, **attrs):
        if attrs.pop('hide_empty', False) and not self:
            return u''
        return html.ul(*(html.li(unicode(item)) for item in self), **attrs)

    def as_ol(self, **attrs):
        if attrs.pop('hide_empty', False) and not self:
            return u''
        return html.ol(*(html.li(unicode(item)) for item in self), **attrs)


class _Renderable(object):
    """Mixin for renderable HTML objects."""

    def render(self):
        return u''

    def __call__(self, *args, **kwargs):
        return self.render(*args, **kwargs)

    def __unicode__(self):
        return self()

    def __str__(self):
        return unicode(self).encode('utf-8')


class Widget(_Renderable):
    """Baseclass for all widgets.  All widgets share a common interface
    that can be used from within templates.

    Take this form as an example:

    >>> class LoginForm(Form):
    ...     username = TextField(required=True)
    ...     password = TextField(widget=PasswordInput)
    ...     flags = MultiChoiceField(choices=[1, 2, 3])
    ...
    >>> form = LoginForm()
    >>> form.validate({'username': '', 'password': '',
    ...                'flags': [1, 3]})
    False
    >>> widget = form.as_widget()

    You can get the subwidgets by using the normal indexing operators:

    >>> username = widget['username']
    >>> password = widget['password']

    The conversion to unicode calls the widget which renders it and displays
    an error list as unordered list next to it.

    >>> unicode(username) == username()
    True

    To render a widget you can usually invoke the `render()` method.  All
    keyword parameters are used as HTML attribute in the resulting tag.
    You can also call the widget itself (``username()`` instead of
    ``username.render()``) which does the same if there are no errors for
    the field but adds the default error list after the widget if there
    are errors.

    Widgets have some public attributes:

    `errors`

        gives the list of errors:

        >>> username.errors
        [u'This field is required.']

        This error list is printable:

        >>> print username.errors
        <ul class="errors"><li>This field is required.</li></ul>

        Like any other sequence that yields list items it provides
        `as_ul` and `as_ol` methods:

        >>> print username.errors.as_ul()
        <ul><li>This field is required.</li></ul>

        Keep in mind that ``widget.errors()`` is equivalent to
        ``widget.errors.as_ul(class_='errors', hide_empty=True)``.

    `value`

        returns the value of the widget as primitive.  For basic
        widgets this is always a string, for widgets with subwidgets or
        widgets with multiple values a dict or a list:

        >>> username.value
        u''
        >>> widget['flags'].value
        [u'1', u'3']

    `name` gives you the name of the field for form submissions:

        >>> username.name
        'username'

        Please keep in mind that the name is not always that obvious.  Glashammer
        supports nested form fields so it's a good idea to always use the
        name attribute.

    `id`

        gives you the default domain for the widget.  This is either none
        if there is no idea for the field or `f_` + the field name with
        underscores instead of dots:

        >>> username.id
        'f_username'

    `all_errors`

        like `errors` but also contains the errors of child
        widgets.
    """

    def __init__(self, field, name, value, all_errors):
        self._field = field
        self._value = value
        self._all_errors = all_errors
        self.name = name

    @property
    def id(self):
        """The proposed id for this widget."""
        if self.name is not None:
            return 'f_' + self.name.replace('.', '_')

    @property
    def value(self):
        """The primitive value for this widget."""
        return self._field.to_primitive(self._value)

    @property
    def errors(self):
        """The direct errors of this widget."""
        if self.name in self._all_errors:
            return self._all_errors[self.name]
        return ErrorList()

    @property
    def all_errors(self):
        """The current errors and the errors of all child widgets."""
        items = sorted(self._all_errors.items())
        if self.name is None:
            return ErrorList(chain(*(item[1] for item in items)))
        result = ErrorList()
        for key, value in items:
            if key == self.name or key.startswith(self.name + '.'):
                result.extend(value)
        return result

    def _attr_setdefault(self, attrs):
        """Add an ID to the attrs if there is none."""
        if 'id' not in attrs and self.id is not None:
            attrs['id'] = self.id

    def __call__(self, **attrs):
        """The default display is the form + error list as ul if needed."""
        return self.render(**attrs) + self.errors()


class Label(_Renderable):
    """Holds a label."""

    def __init__(self, text, linked_to=None):
        self.text = text
        self.linked_to = linked_to

    def render(self, **attrs):
        attrs.setdefault('for', self.linked_to)
        return html.label(escape(self.text), **attrs)


class InternalWidget(Widget):
    """Special widgets are widgets that can't be used on arbitrary
    form fields but belong to others.
    """

    def __init__(self, parent):
        self._parent = parent

    value = name = None
    errors = all_errors = property(lambda: ErrorList())


class Input(Widget):
    """A widget that is a HTML input field."""
    hide_value = False
    type = None

    def render(self, **attrs):
        self._attr_setdefault(attrs)
        value = self.value
        if self.hide_value:
            value = u''
        return html.input(name=self.name, value=value, type=self.type,
                          **attrs)


class TextInput(Input):
    """A widget that holds text."""
    type = 'text'



class PasswordInput(TextInput):
    """A widget that holds a password."""
    type = 'password'
    hide_value = True


class HiddenInput(Input):
    """A hidden input field for text."""
    type = 'hidden'


class Textarea(Widget):
    """Displays a textarea."""

    def _attr_setdefault(self, attrs):
        Widget._attr_setdefault(self, attrs)
        attrs.setdefault('rows', 8)
        attrs.setdefault('cols', 40)

    def render(self, **attrs):
        self._attr_setdefault(attrs)
        return html.textarea(self.value, name=self.name, **attrs)


class Checkbox(Widget):
    """A simple checkbox."""

    def render(self, **attrs):
        self._attr_setdefault(attrs)
        return html.input(name=self.name, type='checkbox',
                          checked=self.value, **attrs)


class SelectBox(Widget):
    """A select box."""

    def _attr_setdefault(self, attrs):
        Widget._attr_setdefault(self, attrs)
        attrs.setdefault('multiple', self._field.multiple_choices)

    def render(self, **attrs):
        self._attr_setdefault(attrs)
        items = []
        for choice in self._field.choices:
            if isinstance(choice, tuple):
                key, value = choice
            else:
                key = value = choice
            items.append(html.option(unicode(value), value=unicode(key),
                                     selected=key == self.value))
        return html.select(name=self.name, *items, **attrs)


class _InputGroupMember(InternalWidget):
    """A widget that is a single radio button."""

    def __init__(self, parent, value, label):
        InternalWidget.__init__(self, parent)
        self.value = unicode(value)
        self.label = Label(label, self.id)

    @property
    def name(self):
        return self._parent.name

    @property
    def id(self):
        return 'f_%s_%s' % (self._parent.name, self.value)

    @property
    def checked(self):
        if self._parent._field.multiple_choices:
            return self.value in self._parent.value
        return self._parent.value == self.value

    def render(self, **attrs):
        self._attr_setdefault(attrs)
        return html.input(type=self.type, name=self.name, value=self.value,
                          checked=self.checked, **attrs)


class RadioButton(_InputGroupMember):
    """A radio button in an input group."""
    type = 'radio'


class GroupCheckbox(_InputGroupMember):
    """A checkbox in an input group."""
    type = 'checkbox'


class _InputGroup(Widget):

    def __init__(self, field, name, value, all_errors):
        Widget.__init__(self, field, name, value, all_errors)
        self.choices = []
        self._subwidgets = {}
        for value, label in _iter_choices(self._field.choices):
            widget = self.subwidget(self, value, label)
            self.choices.append(widget)
            self._subwidgets[value] = widget

    def __getitem__(self, value):
        """Return a subwidget."""
        return self._subwidgets[value]

    def _as_list(self, list_type, attrs):
        if attrs.pop('hide_empty', False) and not self.choices:
            return u''
        self._attr_setdefault(attrs)
        return list_type(*[u'<li>%s %s</li>' % (
            choice,
            choice.label
        ) for choice in self.choices], **attrs)

    def as_ul(self, **attrs):
        """Render the radio buttons widget as <ul>"""
        return self._as_list(html.ul, attrs)

    def as_ol(self, **attrs):
        """Render the radio buttons widget as <ol>"""
        return self._as_list(html.ol, attrs)

    def as_table(self, **attrs):
        """Render the radio buttons widget as <table>"""
        self._attr_setdefault(attrs)
        return list_type(*[u'<tr><td>%s</td><td>%s</td></tr>' % (
            choice,
            choice.label
        ) for choice in self.choices], **attrs)

    def render(self, **attrs):
        return self.as_ul(**attrs)


class RadioButtonGroup(_InputGroup):
    """A group of radio buttons."""
    subwidget = RadioButton


class CheckboxGroup(_InputGroup):
    """A group of checkboxes."""
    subwidget = GroupCheckbox


class MappingWidget(Widget):
    """Special widget for dict-like fields."""

    def __init__(self, field, name, value, all_errors):
        Widget.__init__(self, field, name, _force_dict(value), all_errors)
        self._subwidgets = {}

    def __getitem__(self, name):
        subwidget = self._subwidgets.get(name)
        if subwidget is None:
            # this could raise a KeyError we pass through
            field = self._field.fields[name]
            subwidget = _make_widget(field,
                                     _make_name(self.name, name),
                                     self._value.get(name) or field.default,
                                     self._all_errors)
            self._subwidgets[name] = subwidget
        return subwidget

    def as_dl(self, **attrs):
        return html.dl(*(html.dt(key.title()) + html.dd(self[key]())
                         for key in self.get_input_fields()), **attrs)

    def actions_container(self, **attrs):
        return html.div(*(self[key]() for key in self.get_action_fields()), **attrs)

    def get_input_fields(self):
        return [key for key in self if self._field.fields[key].is_input]

    def get_action_fields(self):
        return [key for key in self if not self._field.fields[key].is_input]

    def __call__(self, *args, **kwargs):
        return self.as_dl(*args, **kwargs)

    def __iter__(self):
        return iter(self._field.fields)


class FormWidget(MappingWidget):
    """A widget for forms."""

    def get_hidden_fields(self):
        """This method is called by the `hidden_fields` property to return
        a list of (key, value) pairs for the special hidden fields.
        """
        fields = []
        if self._field.form.request is not None:
            if self._field.form.csrf_protected:
                fields.append(('_csrf_token', self.csrf_token))
            if self._field.form.redirect_tracking:
                target = self.redirect_target
                if target is not None:
                    fields.append(('_redirect_target', target))
        return fields

    @property
    def hidden_fields(self):
        """The hidden fields as string."""
        return u''.join(html.input(type='hidden', name=name, value=value)
                        for name, value in self.get_hidden_fields())

    @property
    def csrf_token(self):
        """Forward the CSRF check token for templates."""
        return self._field.form.csrf_token

    @property
    def redirect_target(self):
        """The redirect target for this form."""
        return self._field.form.redirect_target

    def render(self, action='', method='post', **attrs):
        self._attr_setdefault(attrs)
        with_errors = attrs.pop('with_errors', False)

        # support jinja's caller
        caller = attrs.pop('caller', None)
        if caller is not None:
            body = caller()
        else:
            body = self.as_dl()

        body = body + self.actions_container()

        hidden = self.hidden_fields
        if hidden:
            # if there are hidden fields we put an invisible div around
            # it.  the HTML standard doesn't allow input fields as direct
            # childs of a <form> tag...
            body = '<div style="display: none">%s</div>%s' % (hidden, body)

        if with_errors:
            body = self.errors() + body
        return html.form(body, action=action, method=method, **attrs)

    def __call__(self, *args, **attrs):
        attrs.setdefault('with_errors', True)
        return self.render(*args, **attrs)


class ListWidget(Widget, _ListSupport):
    """Special widget for list-like fields."""

    def __init__(self, field, name, value, all_errors):
        Widget.__init__(self, field, name, _force_list(value), all_errors)
        self._subwidgets = {}

    def __getitem__(self, index):
        if not isinstance(index, (int, long)):
            raise TypeError('list widget indices must be integers')
        subwidget = self._subwidgets.get(index)
        if subwidget is None:
            try:
                value = self._value[index]
            except IndexError:
                # return an widget without value if we try
                # to access a field not in the list
                value = self._field.field.default
            subwidget = _make_widget(self._field.field,
                                     _make_name(self.name, index), value,
                                     self._all_errors)
            self._subwidgets[index] = subwidget
        return subwidget


    def __iter__(self):
        for index in xrange(len(self)):
            yield self[index]

    def __len__(self):
        return len(self._value)

    def __call__(self, *args, **kwargs):
        return self.as_ul(*args, **kwargs)


class ErrorList(_Renderable, _ListSupport, list):
    """The class that is used to display the errors."""

    def render(self, **attrs):
        return self.as_ul(**attrs)

    def __call__(self, **attrs):
        attrs.setdefault('class', attrs.pop('class_', 'errors'))
        attrs.setdefault('hide_empty', True)
        return self.render(**attrs)


class MultipleValidationErrors(ValidationError):
    """A validation error subclass for multiple errors raised by
    subfields.  This is used by the mapping and list fields.
    """

    def __init__(self, errors):
        ValidationError.__init__(self, '%d error%s' % (
            len(errors), len(errors) != 1 and 's' or ''
        ))
        self.errors = errors

    def __unicode__(self):
        return ', '.join(map(unicode, self.errors.itervalues()))

    def unpack(self, key=None):
        rv = {}
        for name, error in self.errors.iteritems():
            rv.update(error.unpack(_make_name(key, name)))
        return rv


class FieldMeta(type):

    def __new__(cls, name, bases, d):
        messages = {}
        for base in reversed(bases):
            if hasattr(base, 'messages'):
                messages.update(base.messages)
        if 'messages' in d:
            messages.update(d['messages'])
        d['messages'] = messages
        return type.__new__(cls, name, bases, d)


class Field(object):
    """Abstract field base class."""

    __metaclass__ = FieldMeta
    widget = TextInput
    messages = dict(required=lazy_gettext('This field is required.'))
    form = None
    is_input = True

    def __init__(self, validators=None, widget=None, messages=None, default=None):
        if validators is None:
            validators = []
        self.validators = validators
        self.custom_converter = None
        if widget is not None:
            self.widget = widget
        if messages:
            self.messages = self.messages.copy()
            self.messages.update(messages)
        self.default = default
        assert not issubclass(self.widget, InternalWidget), \
            'can\'t use internal widgets as widgets for fields'

    def __call__(self, value):
        value = self.convert(value)
        self.apply_validators(value)
        return value

    def apply_validators(self, value):
        """Applies all validators on the value."""
        if self.should_validate(value):
            for validate in self.validators:
                validate(self.form, value)

    def should_validate(self, value):
        """Per default validate if the value is not None.  This method is
        called before the custom validators are applied to not perform
        validation if the field is empty and not required.

        For example a validator like `is_valid_ip` is never alled if the
        value is an empty string and the field hasn't raised a validation
        error when checking if the field is required.
        """
        return value is not None

    def convert(self, value):
        """This can be overridden by subclasses and performs the value
        conversion.
        """
        return unicode(value)

    def to_primitive(self, value):
        """Convert a value into a primitve (string or a list/dict of lists,
        dicts or strings).

        This method must never fail!
        """
        return _to_string(value)

    def _bind(self, form, memo):
        """Method that binds a field to a form."""
        if self.bound:
            raise TypeError('%r already bound' % type(obj).__name__)
        rv = object.__new__(self.__class__)
        rv.__dict__.update(self.__dict__)
        rv.validators = self.validators[:]
        rv.messages = self.messages.copy()
        rv.form = form
        return rv

    @property
    def bound(self):
        """True if the form is bound."""
        return 'form' in self.__dict__

    def __repr__(self):
        rv = object.__repr__(self)
        if self.bound:
            rv = rv[:-1] + ' [bound]>'
        return rv


class Mapping(Field):
    """Apply a set of fields to a dictionary of values.

    >>> field = Mapping(name=TextField(), age=IntegerField())
    >>> field({'name': u'John Doe', 'age': u'42'})
    {'age': 42, 'name': u'John Doe'}
    """

    widget = MappingWidget

    def __init__(self, **fields):
        Field.__init__(self)
        self.fields = fields

    def convert(self, value):
        value = _force_dict(value)
        errors = {}
        result = {}
        for name, field in self.fields.iteritems():
            try:
                result[name] = field(value.get(name))
            except ValidationError, e:
                errors[name] = e
        if errors:
            raise MultipleValidationErrors(errors)
        return result

    def to_primitive(self, value):
        value = _force_dict(value, silent=True)
        result = {}
        for key, field in self.fields.iteritems():
            result[key] = field.to_primitive(value.get(key))
        return result

    def _bind(self, form, memo):
        rv = Field._bind(self, form, memo)
        rv.fields = {}
        for key, field in self.fields.iteritems():
            rv.fields[key] = _bind(field, form, memo)
        return rv


class FormMapping(Mapping):
    """Like a mapping but does csrf protection and stuff."""

    widget = FormWidget

    def convert(self, value):
        if self.form is None:
            raise TypeError('form mapping without form passed is unable '
                            'to convert data')
        if self.form.csrf_protected and self.form.request is not None:
            token = self.form.request.values.get('_csrf_token')
            if token != self.form.csrf_token:
                raise ValidationError(_('Invalid security token submitted.'))
        return Mapping.convert(self, value)


class FormAsField(Mapping):
    """If a form is converted into a field the returned field object is an
    instance of this class.  The behavior is mostly equivalent to a normal
    :class:`Mapping` field with the difference that it as an attribute called
    :attr:`form_class` that points to the form class it was created from.
    """

    def __init__(self):
        raise TypeError('can\'t create %r instances' %
                        self.__class__.__name__)


class Multiple(Field):
    """Apply a single field to a sequence of values.

    >>> field = Multiple(IntegerField())
    >>> field([u'1', u'2', u'3'])
    [1, 2, 3]
    """

    widget = ListWidget
    messages = dict(too_small=None, too_big=None)

    def __init__(self, field, min_size=None, max_size=None,
                 messages=None):
        Field.__init__(self, messages=messages)
        self.field = field
        self.min_size = min_size
        self.max_size = max_size

    def convert(self, value):
        value = _force_list(value)
        if self.min_size is not None and len(value) < self.min_size:
            message = self.messages['too_small']
            if message is None:
                message = ngettext('Please provide at least %d item.',
                                   'Please provide at least %d items.',
                                    self.min_size) % self.min_size
            raise ValidationError(message)
        if self.max_size is not None and len(value) > self.max_size:
            message = self.messages['too_big']
            if message is None:
                message = ngettext('Please provide no more than %d item.',
                                   'Please provide no more than %d items.',
                                    self.min_size) % self.min_size
            raise ValidationError(message)
        result = []
        errors = {}
        for idx, item in enumerate(value):
            try:
                result.append(self.field(item))
            except ValidationError, e:
                errors[idx] = e
        if errors:
            raise MultipleValidationErrors(errors)
        return result

    def to_primitive(self, value):
        return map(self.field.to_primitive, _force_list(value, silent=True))

    def _bind(self, form, memo):
        rv = Field._bind(self, form, memo)
        rv.field = _bind(self.field, form, memo)
        return rv


class TextField(Field):
    """Field for strings.

    >>> field = TextField(required=True, min_length=6)
    >>> field('foo bar')
    u'foo bar'
    >>> field('')
    Traceback (most recent call last):
      ...
    ValidationError: This field is required.
    """

    messages = dict(too_short=None, too_long=None)

    def __init__(self, required=False, min_length=None, max_length=None,
                 validators=None, widget=None, messages=None):
        Field.__init__(self, validators, widget, messages)
        self.required = required
        self.min_length = min_length
        self.max_length = max_length

    def convert(self, value):
        value = _to_string(value)
        if self.required and not value:
            raise ValidationError(self.messages['required'])
        if self.min_length is not None and len(value) < self.min_length:
            message = self.messages['too_short']
            if message is None:
                message = ngettext('Please enter at least %d character.',
                                   'Please enter at least %d characters.',
                                   self.min_length) % self.min_length
            raise ValidationError(message)
        if self.max_length is not None and len(value) > self.max_length:
            message = self.messages['too_long']
            if message is None:
                message = ngettext('Please enter no more than %d character.',
                                   'Please enter no more than %d characters.',
                                   self.max_length) % self.max_length
            raise ValidationError(message)
        return value

    def should_validate(self, value):
        """Validate if the string is not empty."""
        return bool(value)


class DateTimeField(Field):
    """Field for datetime objects.

    >>> field = DateTimeField()
    >>> field('1970-01-12 00:00')
    datetime.datetime(1970, 1, 12, 0, 0)

    >>> field('foo')
    Traceback (most recent call last):
      ...
    ValidationError: Please enter a valid date.
    """

    messages = dict(invalid_date=lazy_gettext('Please enter a valid date.'))

    def __init__(self, required=False, format='%m/%d/%Y',
                 validators=None, widget=None, messages=None):
        Field.__init__(self, validators, widget, messages)
        self.required = required
        self.format = format

    def convert(self, value):
        if isinstance(value, datetime):
            return value
        value = _to_string(value)
        if not value:
            if self.required:
                raise ValidationError(self.messages['required'])
            return None
        try:
            return datetime(*strptime(value, self.format)[:7])
        except ValueError:
            raise ValidationError(self.messages['invalid_date'])

    def to_primitive(self, value):
        if isinstance(value, datetime):
            value = value.strftime(self.format)
        return value


class ModelField(Field):
    """A field that queries for a model.

    The first argument is the name of the model, the second the named
    argument for `filter_by` (eg: `User` and ``'username'``).
    """
    messages = dict(not_found=lazy_gettext('"%(value)s" does not exist'))

    def __init__(self, model, key, required=False, message=None,
                 validators=None, widget=None, messages=None):
        Field.__init__(self, validators, widget, messages)
        self.model = model
        self.key = key
        self.required = required
        self.message = message

    def convert(self, value):
        if isinstance(value, self.model):
            return value
        if not value:
            if self.required:
                raise ValidationError(self.messages['required'])
            return None
        value = self._coerce_value(value)

        rv = self.model.objects.filter_by(**{self.key: value}).first()
        if rv is None:
            raise ValidationError(self.messages['not_found'] %
                                  {'value': value})
        return rv

    def _coerce_value(self, value):
        return value

    def to_primitive(self, value):
        if value is None:
            return u''
        elif isinstance(value, self.model):
            value = getattr(value, self.key)
        return unicode(value)


class HiddenModelField(ModelField):
    """A hidden field that points to a model identified by primary key.
    Can be used to pass models through a form.
    """
    widget = HiddenInput

    # these messages should never show up unless ...
    #   ... the user tempered with the form data
    #   ... or the object went away in the meantime.
    messages = dict(
        invalid=lazy_gettext('Invalid value.'),
        not_found=lazy_gettext('Key does not exist.')
    )

    def __init__(self, model, key=None, required=False, message=None,
                 validators=None, widget=None, messages=None):
        if key is None:
            keys = db.class_mapper(model).primary_key
            assert len(keys) == 1, 'Model has multiple primary keys'
            key = keys[0].name
        ModelField.__init__(self, model, key, required=False, message=None,
                            validators=None, widget=None, messages=None)

    def _coerce_value(self, value):
        try:
            return int(value)
        except (TypeError, ValueError):
            raise ValidationError(self.messages['invalid'])


class ChoiceField(Field):
    """A field that lets a user select one out of many choices.

    A choice field accepts some choices that are valid values for it.
    Values are compared after converting to unicode which means that
    ``1 == "1"``:

    >>> field = ChoiceField(choices=[1, 2, 3])
    >>> field('1')
    1
    >>> field('42')
    Traceback (most recent call last):
      ...
    ValidationError: Please enter a valid choice.

    A choice field also accepts lists of tuples as argument where the
    first item is used for comparing and the second for displaying
    (which is used by the `SelectBoxWidget`):

    >>> field = ChoiceField(choices=[(0, 'inactive', 1, 'active')])
    >>> field('0')
    0

    Because all fields are bound to the form before validation it's
    possible to assign the choices later:

    >>> class MyForm(Form):
    ...     status = ChoiceField()
    ...
    >>> form = MyForm()
    >>> form.fields['status'].choices = [(0, 'inactive', 1, 'active')]
    >>> form.validate({'status': '0'})
    True
    >>> form.data
    {'status': 0}
    """

    widget = SelectBox
    multiple_choices = False
    messages = dict(
        invalid_choice=lazy_gettext('Please enter a valid choice.')
    )

    def __init__(self, required=True, choices=None, validators=None,
                 widget=None, messages=None):
        Field.__init__(self, validators, widget, messages)
        self.required = required
        self.choices = choices

    def convert(self, value):
        if not value and not self.required:
            return
        for choice in self.choices:
            if isinstance(choice, tuple):
                choice = choice[0]
            # perform type interpolation.  If the key from the choices
            # is a integer (42) and the value is a string ("42") we
            # return the integer.
            if value == choice or unicode(value) == unicode(choice):
                return choice
        raise ValidationError(self.messages['invalid_choice'])

    def _bind(self, form, memo):
        rv = Field._bind(self, form, memo)
        if self.choices is not None:
            rv.choices = list(self.choices)
        return rv


class MultiChoiceField(ChoiceField):
    """A field that lets a user select multiple choices."""

    multiple_choices = True
    messages = dict(too_small=None, too_big=None)

    def __init__(self, choices=None, min_size=None, max_size=None,
                 validators=None, widget=None, messages=None):
        ChoiceField.__init__(self, min_size > 0, choices, validators,
                             widget, messages)
        self.min_size = min_size
        self.max_size = max_size

    def convert(self, value):
        values = _to_list(value)
        container = set(values) | set(map(unicode, values))
        result = []

        for choice in self.choices:
            if isinstance(choice, tuple):
                choice = choice[0]
            if choice not in container:
                raise ValidationError(_('Select a valid choice.'))
            result.append(choice)

        if self.min_size is not None and len(result) < self.min_size:
            message = self.messages['too_small']
            if message is None:
                message = ngettext('Please provide at least %d item.',
                                   'Please provide at least %d items.',
                                    self.min_size) % self.min_size
            raise ValidationError(message)
        if self.max_size is not None and len(result) > self.max_size:
            message = self.messages['too_big']
            if message is None:
                message = ngettext('Please provide no more than %d item.',
                                   'Please provide no more than %d items.',
                                    self.min_size) % self.min_size
            raise ValidationError(message)

        return result

    def to_primitive(self, value):
        return map(unicode, _force_list(value, silent=True))


class IntegerField(Field):
    """Field for integers.

    >>> field = IntegerField(min_value=0, max_value=99)
    >>> field('13')
    13

    >>> field('thirteen')
    Traceback (most recent call last):
      ...
    ValidationError: Please enter a whole number.

    >>> field('193')
    Traceback (most recent call last):
      ...
    ValidationError: Ensure this value is less than or equal to 99.
    """

    messages = dict(
        too_small=None,
        too_big=None,
        no_integer=lazy_gettext('Please enter a whole number.')
    )

    def __init__(self, required=False, min_value=None, max_value=None,
                 validators=None, widget=None, messages=None):
        Field.__init__(self, validators, widget, messages=None)
        self.required = required
        self.min_value = min_value
        self.max_value = max_value

    def convert(self, value):
        value = _to_string(value)
        if not value:
            if self.required:
                raise ValidationError(self.messages['required'])
            return None
        try:
            value = int(value)
        except ValueError:
            raise ValidationError(self.messages['no_integer'])

        if self.min_value is not None and value < self.min_value:
            message = self.messages['too_small']
            if message is None:
                message = _('Ensure this value is greater than or '
                            'equal to %s.') % self.min_value
            raise ValidationError(message)
        if self.max_value is not None and value > self.max_value:
            message = self.messages['too_big']
            if message is None:
                message = _('Ensure this value is less than or '
                            'equal to %s.') % self.max_value
            raise ValidationError(message)

        return int(value)


class BooleanField(Field):
    """Field for boolean values.

    >>> field = BooleanField()
    >>> field('1')
    True

    >>> field = BooleanField()
    >>> field('')
    False
    """

    widget = Checkbox

    def convert(self, value):
        return bool(value)

    def to_primitive(self, value):
        if value:
            return '1'
        return ''


class FormMeta(type):
    """Meta class for forms.  Handles form inheritance and registers
    validator functions.
    """

    def __new__(cls, name, bases, d):
        fields = {}
        validator_functions = {}
        root_validator_functions = []

        for base in reversed(bases):
            if hasattr(base, '_root_field'):
                # base._root_field is always a FormMapping field
                fields.update(base._root_field.fields)
                root_validator_functions.extend(base._root_field.validators)

        for key, value in d.iteritems():
            if key.startswith('validate_') and callable(value):
                validator_functions[key[9:]] = value
            elif isinstance(value, Field):
                fields[key] = value
                d[key] = FieldDescriptor(key)

        for field_name, func in validator_functions.iteritems():
            if field_name in fields:
                fields[field_name].validators.append(func)

        d['_root_field'] = root = FormMapping(**fields)
        context_validate = d.get('context_validate')
        root.validators.extend(root_validator_functions)
        if context_validate is not None:
            root.validators.append(context_validate)

        return type.__new__(cls, name, bases, d)

    def as_field(cls):
        """Returns a field object for this form.  The field object returned
        is independent of the form and can be modified in the same manner as
        a bound field.
        """
        field = object.__new__(FormAsField)
        field.__dict__.update(cls._root_field.__dict__)
        field.form_class = cls
        field.validators = cls._root_field.validators[:]
        field.fields = cls._root_field.fields.copy()
        return field

    @property
    def validators(cls):
        return cls._root_field.validators

    @property
    def fields(cls):
        return cls._root_field.fields


class FieldDescriptor(object):

    def __init__(self, name):
        self.name = name

    def __get__(self, obj, type=None):
        return (obj or type).fields[self.name]

    def __set__(self, obj, value):
        obj.fields[self.name] = value


class Form(object):
    """Form base class.

    >>> class PersonForm(Form):
    ...     name = TextField(required=True)
    ...     age = IntegerField()

    >>> form = PersonForm()
    >>> form.validate({'name': 'johnny', 'age': '42'})
    True
    >>> form.data['name']
    u'johnny'
    >>> form.data['age']
    42

    Let's cause a simple validation error:

    >>> form = PersonForm()
    >>> form.validate({'name': '', 'age': 'fourty-two'})
    False
    >>> print form.errors['age'][0]
    Please enter a whole number.
    >>> print form.errors['name'][0]
    This field is required.

    You can also add custom validation routines for fields by adding methods
    that start with the prefix ``validate_`` and the field name that take the
    value as argument. For example:

    >>> class PersonForm(Form):
    ...     name = TextField(required=True)
    ...     age = IntegerField()
    ...
    ...     def validate_name(self, value):
    ...         if not value.isalpha():
    ...             raise ValidationError(u'The value must only contain letters')

    >>> form = PersonForm()
    >>> form.validate({'name': 'mr.t', 'age': '42'})
    False
    >>> form.errors
    {'name': [u'The value must only contain letters']}

    You can also validate multiple fields in the context of other fields.
    That validation is performed after all other validations.  Just add a
    method called ``context_validate`` that is passed the dict of all fields::

    >>> class RegisterForm(Form):
    ...     username = TextField(required=True)
    ...     password = TextField(required=True)
    ...     password_again = TextField(required=True)
    ...
    ...     def context_validate(self, data):
    ...         if data['password'] != data['password_again']:
    ...             raise ValidationError(u'The two passwords must be the same')

    >>> form = RegisterForm()
    >>> form.validate({'username': 'admin', 'password': 'blah',
    ...                'password_again': 'blag'})
    ...
    False
    >>> form.errors
    {None: [u'The two passwords must be the same']}

    Forms can be used as fields for other forms.  To create a form field of
    a form you can call the `as_field` class method::

    >>> field = RegisterForm.as_field()

    This field can be used like any other field class.  What's important about
    forms as fields is that validators don't get an instance of `RegisterForm`
    passed as `form` / `self` but the form where it's used in if the field is
    used from a form.

    Form fields are bound to the form on form instanciation.  This makes it
    possible to modify a particular instance of the form.  For example you
    can create an instance of it and drop some fiels by using
    ``del form.fields['name']`` or reassign choices of choice fields.  It's
    however not easily possible to add new fields to an instance because newly
    added fields wouldn't be bound.  The fields that are stored directly on
    the form can also be accessed with their name like a regular attribute.

    Example usage::

    >>> class StatusForm(Form):
    ...     status = ChoiceField()
    ...
    >>> StatusForm.status.bound
    False
    >>> form = StatusForm()
    >>> form.status.bound
    True
    >>> form.status.choices = [u'happy', u'unhappy']
    >>> form.validate({'status': u'happy'})
    True
    >>> form['status']
    u'happy'
    """
    __metaclass__ = FormMeta

    csrf_protected = True
    redirect_tracking = True

    def __init__(self, initial=None):
        self.request = get_request()
        if initial is None:
            initial = {}
        self.initial = initial
        self.invalid_redirect_targets = set()

        self._root_field = _bind(self.__class__._root_field, self, {})
        self.reset()

    def __getitem__(self, key):
        return self.data[key]

    def __contains__(self, key):
        return key in self.data

    def as_widget(self):
        """Return the form as widget."""
        # if there is submitted data, use that for the widget
        if self.raw_data is not None:
            data = self.raw_data
        # otherwise go with the data from the source (eg: database)
        else:
            data = self.data
        return _make_widget(self._root_field, None, data, self.errors)

    def add_invalid_redirect_target(self, *args, **kwargs):
        """Add an invalid target. Invalid targets are URLs we don't want to
        visit again. For example if a post is deleted from the post edit page
        it's a bad idea to redirect back to the edit page because in that
        situation the edit page would return a page not found.

        This function accepts the same parameters as `url_for`.
        """
        self.invalid_redirect_targets.add(url_for(*args, **kwargs))

    @property
    def redirect_target(self):
        """The back-redirect target for this form."""
        return get_redirect_target(self.invalid_redirect_targets,
                                   self.request)

    def redirect(self, *args, **kwargs):
        """Redirects to the url rule given or back to the URL where we are
        comming from if `redirect_tracking` is enabled.
        """
        target = None
        if self.redirect_tracking:
            target = self.redirect_target
        if target is None:
            target = url_for(*args, **kwargs)
        return redirect(target)

    @property
    def csrf_token(self):
        """The unique CSRF security token for this form."""
        if self.request is None:
            raise AttributeError('no csrf token because form not bound '
                                 'to request')
        path = self.request.path
        user_id = -1
        #XXX
        #if self.request.user.is_somebody:
        #    user_id = self.request.user.user_id
        #key = self.request.app.cfg['secret_key']
        key = 'I am secret'
        return sha1(('%s|%s|%s' % (path, user_id, key))
                     .encode('utf-8')).hexdigest()

    @property
    def is_valid(self):
        """True if the form is valid."""
        return not self.errors

    @property
    def has_changed(self):
        """True if the form has changed."""
        return self._root_field.to_primitive(self.initial) != \
               self._root_field.to_primitive(self.data)

    @property
    def fields(self):
        return self._root_field.fields

    @property
    def validators(self):
        return self._root_field.validators

    def reset(self):
        """Resets the form."""
        self.data = self.initial.copy()
        self.errors = {}
        self.raw_data = None

    def validate(self, data):
        """Validate the form against the data passed."""
        self.raw_data = _decode(data)
        d = self.data.copy()
        d.update(self.raw_data)
        errors = {}
        try:
            data = self._root_field(d)
        except ValidationError, e:
            errors = e.unpack()
        self.errors = errors
        if errors:
            return False
        self.data.update(data)
        return True

    def populate(self, obj):
        for k in self.data:
            setattr(obj, k, self.data[k])


# glashammer extras



class SubmitButton(Input):
    """A submit button field"""
    type = 'submit'

class ResetButton(Input):
    type = 'submit'

class SubmitField(Field):
    widget = SubmitButton
    is_input = False

class ResetField(Field):
    widget = ResetButton
    is_input = False
