# -*- coding: utf-8 -*-
"""
Utilities
"""
from __future__ import print_function

import os
import io
from collections import namedtuple
import xml.etree.ElementTree as ET
import datetime
import pprint
try:
    # Python 3
    from urllib.parse import urlencode
except ImportError:
    # Python 2
    from urllib import urlencode

import six
from owslib.util import openURL

from .sos import BASE_URL
from . import SOS
from . import metadata


def get_feature_of_interest(feature_id):

    request = {'service': 'SOS',
               'version': SOS.version,
               'request': 'GetFeatureOfInterest'}
    request['FeatureOfInterestId'] = feature_id

    data = urlencode(request)
    response = openURL(BASE_URL, data, method='Get').read()

    if isinstance(response, bytes):
        response = io.BytesIO(response)
    elif isinstance(response, str):
        response = io.StringIO(response)

    #tree = ET.parse('temp.xml')
    tree = ET.parse(response)

    root = tree.getroot()

    if not root.tag == '{http://www.opengis.net/sampling/1.0}SamplingPoint':
        raise ValueError(
            "feature_id {feature_id} is not a SamplingPoint".format(
                feature_id=feature_id))

    st_info = {}
    st_info['id'] = root.attrib['{http://www.opengis.net/gml}id']
    st_info['description'] = root.find('{http://www.opengis.net/gml}description').text
    st_info['long_name'] = root.find('{http://www.opengis.net/gml}name').text
    name, _ = st_info['long_name'].split(' - ')
    st_info['name'] = name
    position = root.find('{http://www.opengis.net/sampling/1.0}position')
    point = position.find('{http://www.opengis.net/gml}Point')
    pos = point.find('{http://www.opengis.net/gml}pos').text
    x, y = pos.split(' ')
    st_info['x'] = x
    st_info['y'] = y

    return st_info


def _list_stations():

    stations_lists = []

    for off in SOS.offerings:
        stations_lists.append(off.features_of_interest)

    stations = [station for stations in stations_lists for station in stations]

    return set(stations)


_STATIONS_FILE_TEMPLATE = """# -*- coding: utf-8 -*-
\"\"\"
Overview of all stations.

ATTENTION: this is a auto-generated file!
Generated by 'util.py:_generate_stations_list' at {timestamp}.
\"\"\"

STATIONS = \\
{station_dict}
"""


def _generate_stations_list():

    stations = _list_stations()

    st_info = {}

    for station in stations:
        try:
            st_info[station] = get_feature_of_interest(station)
            print("Got feature_id {0}".format(station))
        except ValueError as exc:
            print(exc)
            continue

    filename = os.path.realpath(__file__)
    base_dir = os.path.abspath(os.path.dirname(filename))

    with open(os.path.join(base_dir, '_stations.py'), 'w') as stations_file:

        content = _STATIONS_FILE_TEMPLATE.format(
            station_dict=pprint.pformat(st_info),
            timestamp=datetime.datetime.today().replace(microsecond=0))

        stations_file.write(content)


###############################################################################
# Pretty table printing
###############################################################################

def pprinttable(rows):
    """
    Source: http://stackoverflow.com/questions/5909873/python-pretty-printing-ascii-tables/5910078#5910078
    """
    if len(rows) > 1:
        headers = rows[0]._fields
        lens = []
        for i in range(len(rows[0])):
            lens.append(len(max([x[i] for x in rows] + [headers[i]],key=lambda x:len(str(x.encode('UTF-8'))))))
        formats = []
        hformats = []
        for i in range(len(rows[0])):
            if isinstance(rows[0][i], int):
                formats.append("%%%dd" % lens[i])
            else:
                formats.append("%%-%ds" % lens[i])
            hformats.append("%%-%ds" % lens[i])
        pattern = " | ".join(formats)
        hpattern = " | ".join(hformats)
        separator = "-+-".join(['-' * n for n in lens])
        print(hpattern % tuple(headers))
        print(separator)
        _u = lambda t: t.decode('UTF-8', 'replace') if isinstance(t, six.binary_type) else t
        for line in rows:
            print(pattern % tuple(_u(t) for t in line))
    elif len(rows) == 1:
        row = rows[0]
        hwidth = len(max(row._fields,key=lambda x: len(x)))
        for i in range(len(row)):
            print("%*s = %s" % (hwidth,row._fields[i],row[i]))


def print_stations(stations=None):

    if stations is None:
        stations = metadata.STATIONS
    else:
        stations = {st: metadata.STATIONS[st] for st in stations if st in metadata.STATIONS}

    Row = namedtuple('Row', ['name', 'EU_code', 'location', 'region', 'type'])
    rows = [Row(st['name'], st['id'], st['long_name'].split(' - ')[1],
                *st['description'].split(' - '))
            for (code, st) in stations.items()]
    pprinttable(sorted(rows))


def print_pollutants(pollutants=None):

    if pollutants is None:
        pollutants = metadata.POLLUTANTS
    else:
        pollutants = {pol: metadata.POLLUTANTS[pol] for pol in pollutants}

    short_names = {val: key for (key, val) in metadata.SAROAD_CODE.items()}
    Row = namedtuple('Row', ['id', 'short', 'name', 'stations'])
    rows = [Row(pol['id'], short_names.get(pol['id'], ''), pol['name'],
                str(len(pol['features_of_interest'])))
            for (code, pol) in pollutants.items()]
    pprinttable(sorted(rows))
