# Copyright (c) 2014, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

import stix
import stix.bindings.exploit_target as exploit_target_binding
from stix.common import StructuredText


class Configuration(stix.Entity):
    """Implementation of STIX ``Configuration``.

    Args:
        cce_id(optional): Common Configuration Enumeration value as a string
        description (optional): A string description.
        short_description (optional): A string short description.

    """
    _binding = exploit_target_binding
    _binding_class = _binding.ConfigurationType
    _namespace = "http://stix.mitre.org/ExploitTarget-1"

    def __init__(self, description=None, short_description=None, cce_id=None):
        self.description = description
        self.short_description = short_description
        self.cce_id = cce_id

    @property
    def description(self):
        """The ``description`` property for this :class:`Configuration`.

        Default Value: ``None``

        Note:
            If set to a value that is not an instance of
            :class:`stix.common.structured_text.StructuredText`, an attempt to
            will be made to convert the value into an instance of
            :class:`stix.common.structured_text.StructuredText`.

        Returns:
            An instance of
            :class:`stix.common.structured_text.StructuredText`

        """
        return self._description

    @description.setter
    def description(self, value):
        if value:
            if isinstance(value, StructuredText):
                self._description = value
            else:
                self._description = StructuredText(value=value)
        else:
            self._description = None

    @property
    def short_description(self):
        return self._short_description

    @short_description.setter
    def short_description(self, value):
        """The ``short_description`` property for this :class:`Configuration`.

        Default Value: ``None``

        Note:
            If set to a value that is not an instance of
            :class:`stix.common.structured_text.StructuredText`, an attempt to
            will be made to convert the value into an instance of
            :class:`stix.common.structured_text.StructuredText`.

        Returns:
            An instance of
            :class:`stix.common.structured_text.StructuredText`

        """
        if value:
            if isinstance(value, StructuredText):
                self._short_description = value
            else:
                self._short_description = StructuredText(value=value)
        else:
            self._short_description = None

    @property
    def cce_id(self):
        """Common Configuration Enumeration value for this :class:`Configuration`.

        Default Value: ``None``

        Returns:
            A string representing the CCE ID
        """
        return self._cce_id

    @cce_id.setter
    def cce_id(self, value):
        self._cce_id = value

    def to_obj(self, return_obj=None, ns_info=None):
        super(Configuration, self).to_obj(return_obj=return_obj, ns_info=ns_info)

        if not return_obj:
            return_obj = self._binding_class()

        if self.description:
            return_obj.Description = self.description.to_obj(ns_info=ns_info)
        if self.short_description:
            return_obj.Short_Description = self.short_description.to_obj(ns_info=ns_info)
        return_obj.CCE_ID = self.cce_id

        return return_obj

    @classmethod
    def from_obj(cls, obj, return_obj=None):
        if not obj:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.description = StructuredText.from_obj(obj.Description)
        return_obj.short_description = StructuredText.from_obj(obj.Short_Description)
        return_obj.cce_id = obj.CCE_ID

        return return_obj

    def to_dict(self):
        d = {}

        if self.description:
            d['description'] = self.description.to_dict()
        if self.short_description:
            d['short_description'] = self.short_description.to_dict()
        if self.cce_id:
            d['cce_id'] = self.cce_id

        return d

    @classmethod
    def from_dict(cls, dict_repr, return_obj=None):
        if not dict_repr:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.description = StructuredText.from_dict(dict_repr.get('description'))
        return_obj.short_description = StructuredText.from_dict(dict_repr.get('short_description'))
        return_obj.cce_id = dict_repr.get('cce_id')

        return return_obj
