/* C bindings to the sentry relay library */

#ifndef RELAY_H_INCLUDED
#define RELAY_H_INCLUDED

/* Generated with cbindgen:0.14.3 */

/* Warning, this file is autogenerated. Do not modify this manually. */

#include <stdarg.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>

/**
 * Classifies the type of data that is being ingested.
 */
enum RelayDataCategory {
  /**
   * Reserved and unused.
   */
  RELAY_DATA_CATEGORY_DEFAULT,
  /**
   * Error events and Events with an `event_type` not explicitly listed below.
   */
  RELAY_DATA_CATEGORY_ERROR,
  /**
   * Transaction events.
   */
  RELAY_DATA_CATEGORY_TRANSACTION,
  /**
   * Events with an event type of `csp`, `hpkp`, `expectct` and `expectstaple`.
   */
  RELAY_DATA_CATEGORY_SECURITY,
  /**
   * An attachment. Quantity is the size of the attachment in bytes.
   */
  RELAY_DATA_CATEGORY_ATTACHMENT,
  /**
   * Session updates. Quantity is the number of updates in the batch.
   */
  RELAY_DATA_CATEGORY_SESSION,
  /**
   * Any other data category not known by this Relay.
   */
  RELAY_DATA_CATEGORY_UNKNOWN = -1,
};
typedef int8_t RelayDataCategory;

/**
 * Controls the globbing behaviors.
 */
enum GlobFlags {
  GLOB_FLAGS_DOUBLE_STAR = 1,
  GLOB_FLAGS_CASE_INSENSITIVE = 2,
  GLOB_FLAGS_PATH_NORMALIZE = 4,
  GLOB_FLAGS_ALLOW_NEWLINE = 8,
};
typedef uint32_t GlobFlags;

/**
 * Represents all possible error codes
 */
enum RelayErrorCode {
  RELAY_ERROR_CODE_NO_ERROR = 0,
  RELAY_ERROR_CODE_PANIC = 1,
  RELAY_ERROR_CODE_UNKNOWN = 2,
  RELAY_ERROR_CODE_INVALID_JSON_ERROR = 101,
  RELAY_ERROR_CODE_KEY_PARSE_ERROR_BAD_ENCODING = 1000,
  RELAY_ERROR_CODE_KEY_PARSE_ERROR_BAD_KEY = 1001,
  RELAY_ERROR_CODE_UNPACK_ERROR_BAD_SIGNATURE = 1003,
  RELAY_ERROR_CODE_UNPACK_ERROR_BAD_PAYLOAD = 1004,
  RELAY_ERROR_CODE_UNPACK_ERROR_SIGNATURE_EXPIRED = 1005,
  RELAY_ERROR_CODE_PROCESSING_ERROR_INVALID_TRANSACTION = 2001,
  RELAY_ERROR_CODE_PROCESSING_ERROR_INVALID_GEO_IP = 2002,
  RELAY_ERROR_CODE_INVALID_RELEASE_ERROR_TOO_LONG = 3001,
  RELAY_ERROR_CODE_INVALID_RELEASE_ERROR_RESTRICTED_NAME = 3002,
  RELAY_ERROR_CODE_INVALID_RELEASE_ERROR_BAD_CHARACTERS = 3003,
};
typedef uint32_t RelayErrorCode;

/**
 * Trace status.
 *
 * Values from https://github.com/open-telemetry/opentelemetry-specification/blob/8fb6c14e4709e75a9aaa64b0dbbdf02a6067682a/specification/api-tracing.md#status
 * Mapping to HTTP from https://github.com/open-telemetry/opentelemetry-specification/blob/8fb6c14e4709e75a9aaa64b0dbbdf02a6067682a/specification/data-http.md#status
 */
enum RelaySpanStatus {
  /**
   * The operation completed successfully.
   *
   * HTTP status 100..299 + successful redirects from the 3xx range.
   */
  RELAY_SPAN_STATUS_OK = 0,
  /**
   * The operation was cancelled (typically by the user).
   */
  RELAY_SPAN_STATUS_CANCELLED = 1,
  /**
   * Unknown. Any non-standard HTTP status code.
   *
   * "We do not know whether the transaction failed or succeeded"
   */
  RELAY_SPAN_STATUS_UNKNOWN = 2,
  /**
   * Client specified an invalid argument. 4xx.
   *
   * Note that this differs from FailedPrecondition. InvalidArgument indicates arguments that
   * are problematic regardless of the state of the system.
   */
  RELAY_SPAN_STATUS_INVALID_ARGUMENT = 3,
  /**
   * Deadline expired before operation could complete.
   *
   * For operations that change the state of the system, this error may be returned even if the
   * operation has been completed successfully.
   *
   * HTTP redirect loops and 504 Gateway Timeout
   */
  RELAY_SPAN_STATUS_DEADLINE_EXCEEDED = 4,
  /**
   * 404 Not Found. Some requested entity (file or directory) was not found.
   */
  RELAY_SPAN_STATUS_NOT_FOUND = 5,
  /**
   * Already exists (409)
   *
   * Some entity that we attempted to create already exists.
   */
  RELAY_SPAN_STATUS_ALREADY_EXISTS = 6,
  /**
   * 403 Forbidden
   *
   * The caller does not have permission to execute the specified operation.
   */
  RELAY_SPAN_STATUS_PERMISSION_DENIED = 7,
  /**
   * 429 Too Many Requests
   *
   * Some resource has been exhausted, perhaps a per-user quota or perhaps the entire file
   * system is out of space.
   */
  RELAY_SPAN_STATUS_RESOURCE_EXHAUSTED = 8,
  /**
   * Operation was rejected because the system is not in a state required for the operation's
   * execution
   */
  RELAY_SPAN_STATUS_FAILED_PRECONDITION = 9,
  /**
   * The operation was aborted, typically due to a concurrency issue.
   */
  RELAY_SPAN_STATUS_ABORTED = 10,
  /**
   * Operation was attempted past the valid range.
   */
  RELAY_SPAN_STATUS_OUT_OF_RANGE = 11,
  /**
   * 501 Not Implemented
   *
   * Operation is not implemented or not enabled.
   */
  RELAY_SPAN_STATUS_UNIMPLEMENTED = 12,
  /**
   * Other/generic 5xx.
   */
  RELAY_SPAN_STATUS_INTERNAL_ERROR = 13,
  /**
   * 503 Service Unavailable
   */
  RELAY_SPAN_STATUS_UNAVAILABLE = 14,
  /**
   * Unrecoverable data loss or corruption
   */
  RELAY_SPAN_STATUS_DATA_LOSS = 15,
  /**
   * 401 Unauthorized (actually does mean unauthenticated according to RFC 7235)
   *
   * Prefer PermissionDenied if a user is logged in.
   */
  RELAY_SPAN_STATUS_UNAUTHENTICATED = 16,
};
typedef uint8_t RelaySpanStatus;

typedef struct RelayGeoIpLookup RelayGeoIpLookup;

/**
 * Represents a public key in relay.
 */
typedef struct RelayPublicKey RelayPublicKey;

/**
 * Represents a secret key in relay.
 */
typedef struct RelaySecretKey RelaySecretKey;

typedef struct RelayStoreNormalizer RelayStoreNormalizer;

/**
 * A binary buffer of known length.
 *
 * If the buffer is owned, indicated by the `owned` flag, the owner must call the `free` function
 * on this buffer. The convention is:
 *
 *  - When obtained as instance through return values, always free the buffer.
 *  - When obtained as pointer through field access, never free the buffer.
 */
typedef struct {
  /**
   * Pointer to the raw data.
   */
  uint8_t *data;
  /**
   * The length of the buffer pointed to by `data`.
   */
  uintptr_t len;
  /**
   * Indicates that the buffer is owned and must be freed.
   */
  bool owned;
} RelayBuf;

/**
 * A length-prefixed UTF-8 string.
 *
 * As opposed to C strings, this string is not null-terminated. If the string is owned, indicated
 * by the `owned` flag, the owner must call the `free` function on this string. The convention is:
 *
 *  - When obtained as instance through return values, always free the string.
 *  - When obtained as pointer through field access, never free the string.
 */
typedef struct {
  /**
   * Pointer to the UTF-8 encoded string data.
   */
  char *data;
  /**
   * The length of the string pointed to by `data`.
   */
  uintptr_t len;
  /**
   * Indicates that the string is owned and must be freed.
   */
  bool owned;
} RelayStr;

/**
 * Represents a key pair from key generation.
 */
typedef struct {
  RelayPublicKey *public_key;
  RelaySecretKey *secret_key;
} RelayKeyPair;

/**
 * A 16-byte UUID.
 */
typedef struct {
  /**
   * UUID bytes in network byte order (big endian).
   */
  uint8_t data[16];
} RelayUuid;

/**
 * Frees a Relay buf.
 *
 * If the buffer is marked as not owned then this function does not
 * do anything.
 */
void relay_buf_free(RelayBuf *b);

/**
 * Convert an old datascrubbing config to the new PII config format.
 */
RelayStr relay_convert_datascrubbing_config(const RelayStr *config);

/**
 * Creates a challenge from a register request and returns JSON.
 */
RelayStr relay_create_register_challenge(const RelayBuf *data,
                                         const RelayStr *signature,
                                         uint32_t max_age);

/**
 * Parses a `DataCategory` from an event type.
 */
RelayDataCategory relay_data_category_from_event_type(const RelayStr *event_type);

/**
 * Returns the API name of the given `DataCategory`.
 */
RelayStr relay_data_category_name(RelayDataCategory category);

/**
 * Parses a `DataCategory` from its API name.
 */
RelayDataCategory relay_data_category_parse(const RelayStr *name);

/**
 * Clears the last error.
 */
void relay_err_clear(void);

/**
 * Returns the panic information as string.
 */
RelayStr relay_err_get_backtrace(void);

/**
 * Returns the last error code.
 *
 * If there is no error, 0 is returned.
 */
RelayErrorCode relay_err_get_last_code(void);

/**
 * Returns the last error message.
 *
 * If there is no error an empty string is returned.  This allocates new memory
 * that needs to be freed with `relay_str_free`.
 */
RelayStr relay_err_get_last_message(void);

/**
 * Generates a secret, public key pair.
 */
RelayKeyPair relay_generate_key_pair(void);

/**
 * Randomly generates an relay id
 */
RelayUuid relay_generate_relay_id(void);

void relay_geoip_lookup_free(RelayGeoIpLookup *lookup);

RelayGeoIpLookup *relay_geoip_lookup_new(const char *path);

/**
 * Given just the data from a register response returns the
 * conained relay id without validating the signature.
 */
RelayUuid relay_get_register_response_relay_id(const RelayBuf *data);

/**
 * Initializes the library
 */
void relay_init(void);

bool relay_is_glob_match(const RelayBuf *value,
                         const RelayStr *pat,
                         GlobFlags flags);

RelayStr relay_parse_release(const RelayStr *value);

/**
 * Walk through the event and collect selectors that can be applied to it in a PII config. This
 * function is used in the UI to provide auto-completion of selectors.
 */
RelayStr relay_pii_selector_suggestions_from_event(const RelayStr *event);

/**
 * DEPRECATED: Use relay_pii_selector_suggestions_from_event
 */
RelayStr relay_pii_selectors_from_event(const RelayStr *event);

/**
 * Scrub an event using new PII stripping config.
 */
RelayStr relay_pii_strip_event(const RelayStr *config, const RelayStr *event);

/**
 * Frees a public key.
 */
void relay_publickey_free(RelayPublicKey *spk);

/**
 * Parses a public key from a string.
 */
RelayPublicKey *relay_publickey_parse(const RelayStr *s);

/**
 * Converts a public key into a string.
 */
RelayStr relay_publickey_to_string(const RelayPublicKey *spk);

/**
 * Verifies a signature
 */
bool relay_publickey_verify(const RelayPublicKey *spk,
                            const RelayBuf *data,
                            const RelayStr *sig);

/**
 * Verifies a signature
 */
bool relay_publickey_verify_timestamp(const RelayPublicKey *spk,
                                      const RelayBuf *data,
                                      const RelayStr *sig,
                                      uint32_t max_age);

/**
 * Scrub an event using old data scrubbing settings.
 */
RelayStr relay_scrub_event(const RelayStr *config, const RelayStr *event);

/**
 * Frees a secret key.
 */
void relay_secretkey_free(RelaySecretKey *spk);

/**
 * Parses a secret key from a string.
 */
RelaySecretKey *relay_secretkey_parse(const RelayStr *s);

/**
 * Verifies a signature
 */
RelayStr relay_secretkey_sign(const RelaySecretKey *spk, const RelayBuf *data);

/**
 * Converts a secret key into a string.
 */
RelayStr relay_secretkey_to_string(const RelaySecretKey *spk);

RelayStr relay_split_chunks(const RelayStr *string, const RelayStr *remarks);

void relay_store_normalizer_free(RelayStoreNormalizer *normalizer);

RelayStoreNormalizer *relay_store_normalizer_new(const RelayStr *config,
                                                 const RelayGeoIpLookup *geoip_lookup);

RelayStr relay_store_normalizer_normalize_event(RelayStoreNormalizer *normalizer,
                                                const RelayStr *event);

/**
 * Frees a Relay str.
 *
 * If the string is marked as not owned then this function does not
 * do anything.
 */
void relay_str_free(RelayStr *s);

/**
 * Creates a Relay string from a c string.
 */
RelayStr relay_str_from_cstr(const char *s);

void relay_test_panic(void);

bool relay_translate_legacy_python_json(RelayStr *event);

/**
 * Returns true if the uuid is nil.
 */
bool relay_uuid_is_nil(const RelayUuid *uuid);

/**
 * Formats the UUID into a string.
 *
 * The string is newly allocated and needs to be released with
 * `relay_cstr_free`.
 */
RelayStr relay_uuid_to_str(const RelayUuid *uuid);

/**
 * Returns a list of all valid platform identifiers.
 */
const RelayStr *relay_valid_platforms(uintptr_t *size_out);

/**
 * Validate a PII config against the schema. Used in project options UI.
 */
RelayStr relay_validate_pii_config(const RelayStr *value);

/**
 * Validates a register response.
 */
RelayStr relay_validate_register_response(const RelayPublicKey *pk,
                                          const RelayBuf *data,
                                          const RelayStr *signature,
                                          uint32_t max_age);

/**
 * Returns true if the given version is supported by this library.
 */
bool relay_version_supported(const RelayStr *version);

#endif /* RELAY_H_INCLUDED */
