#!/usr/bin/env python
# vim: set fileencoding=utf-8 :
# Elie Khoury <elie.khoury@idiap.ch>
# Mon  6 Jul 15:12:22 CEST 2015
#
# Copyright (C) 2012-2015 Idiap Research Institute, Martigny, Switzerland
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the ipyplotied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function

"""This script evaluates the given score files and computes EER and Spoofing FAR with regards to 10 types of voice attacks""" 

import bob.measure

import argparse
import numpy, math
import os


import bob.core
logger = bob.core.log.setup("bob.bio.base")




def load_development_file(filename):
  # split in positives and negatives
  female_neg = []
  female_pos = []
  male_neg    = []
  male_pos    = []
    
  # read four column list line by line
  for (client_id, probe_id, filename, score) in bob.measure.load.four_column(filename):
    if client_id == probe_id:
      if 'female' in filename:
        female_pos.append(score)
      else:
        male_pos.append(score)
    else:
      if 'female' in filename:
        female_neg.append(score)
      else:
        male_neg.append(score)
  results = {}
  results['female'] = (numpy.array(female_neg, numpy.float64), numpy.array(female_pos, numpy.float64))
  results['male']    = (numpy.array(male_neg, numpy.float64), numpy.array(male_pos, numpy.float64))        
  return results



attacks = ['replay_phone1', 'replay_phone2', 'replay_laptop', 'replay_laptop_HQ', 'speech_synthesis_logical_access', 'speech_synthesis_physical_access', 'speech_synthesis_physical_access_HQ', 'voice_conversion_logical_access', 'voice_conversion_physical_access', 'voice_conversion_physical_access_HQ']

def load_evaluation_file(filename):
  # split in positives and negatives
  female_neg = []
  female_pos = []
  male_neg    = []
  male_pos    = []
   
  results = {}
  results['female'] = {}
  results['male'] = {}
  
  for i in attacks:
    for gender in ['female', 'male']:
      results[gender][i] = []
  
  # read four column list line by line
  for (client_id, attack, filename, score) in bob.measure.load.four_column(filename):
    if 'female' in filename:
      results['female'][attack].append(score)
    else:
      results['male'][attack].append(score)
  
  for i in attacks:
    for gender in ['female', 'male']:
      results[gender][i] =  numpy.array(results[gender][i], numpy.float64)
  return results
  

def command_line_arguments(command_line_parameters):
  """Parse the program options"""

  # set up command line parser
  parser = argparse.ArgumentParser(description=__doc__,
      formatter_class=argparse.ArgumentDefaultsHelpFormatter)

  parser.add_argument('-d', '--dev-file', required=True, help = "The score file of the development set.")
  parser.add_argument('-e', '--eval-file', required=True, help = "The score files of the evaluation set.")

  
  # add verbose option
  bob.core.log.add_command_line_option(parser)

  # parse arguments
  args = parser.parse_args(command_line_parameters)

  # set verbosity level
  bob.core.log.set_verbosity_level(logger, args.verbose)

  return args



def main(command_line_parameters=None):
  """Reads score files, computes error measures and plots curves."""

  args = command_line_arguments(command_line_parameters)


  # First, read the score files
  logger.info("Loading %d score file of the development set", len(args.dev_file))
  scores_dev  = load_development_file(args.dev_file)
  
  logger.info("Loading %d score file of the evaluation set", len(args.eval_file))
  scores_eval = load_evaluation_file(args.eval_file)
  
  thresholds = {}
  for gender in ['male', 'female']:
    print('---------------  %s  -----------------' % gender)
    print('----------------------------------------------')
    thresholds[gender] = bob.measure.eer_threshold(scores_dev[gender][0], scores_dev[gender][1])
    far, frr = bob.measure.farfrr(scores_dev[gender][0], scores_dev[gender][1], thresholds[gender])
    print("EER = %2.1f%% \t Threshold = %2.3f" % ((far + frr) * 50., thresholds[gender]))
    for a in attacks:
      far, frr = bob.measure.farfrr(scores_eval[gender][a], scores_dev[gender][1], thresholds[gender])
      print('%s : %2.1f%%' %(a, far * 100.))
    print("")

