# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
bfc.setup is a helper module for python packages that use Mercurial as
their source-control.

This module provides the following services:
- computing the version information from the repository or an .hg_archival.txt;
- storing the version information in an official release;
- all basic needs of a setup.py script;
- a release file generator - invoke as python -m bfc.setup
  (python -m bfc.setup -h for help!)

An example usage of this module can be found in:
- this package's setup.py
- pybfc-based packages, such as priest (http://priest.sf.net/)

"""

from . import importfile

import os, subprocess, sys, optparse

class ReadVersionError(Exception): pass

def read_version_info_from_file(project_root, package_name):
    filename = os.path.join(project_root, package_name, '__version__.py')
    if not os.path.exists(filename):
        raise ReadVersionError('No such file: %s' % (filename,))
    try:
        return importfile.import_file(filename).version_info
    except Exception, e:
        raise ReadVersionError('Failed to read package __version__.py', e)

def tag_to_version_info(tag, distance):
    t = tag.split('.')
    return tuple([int(v) for v in t] + ([0] * (3 - len(t))) + [int(distance), ''])

def read_version_info(project_root, package_name):
    # check for hg_archive
    filename = os.path.join(project_root, '.hg_archival.txt')
    if os.path.exists(filename):
        kw = dict([[t.strip() for t in l.split(':', 1)]
                   for l in open(filename)])
        if 'tag' in kw:
            return tag_to_version_info(kw['tag'], 0)
        if 'latesttag' in kw:
            return tag_to_version_info(kw['latesttag'], kw['latesttagdistance'])
        return (0, 0, 0, 0, kw.get('node', '')[:12])
    # check for .hg
    if os.path.exists(os.path.join(project_root, '.hg')):
        p = subprocess.Popen(['hg', 'parent', '--template', '{node|short}\n{latesttag}\n{latesttagdistance}'],
                             stdout = subprocess.PIPE, cwd = project_root)
        out, err = p.communicate(None)
        r = p.wait()
        if r or err:
            raise ReadVersionError('Mercurial returned an error', r, err)
        node, latesttag, tagdistance = out.split('\n')
        if latesttag != 'null':
            return tag_to_version_info(latesttag, tagdistance)
        return (0, 0, 0, 0, node[:12])
    raise ReadVersionError('Unable to find version information')

def version_info_to_version(version_info):
    vi = list(version_info[:3])
    while vi and vi[-1] == 0:
        vi.pop()
    version = '.'.join(str(v) for v in vi)
    if version_info[3]:
        version = version + '+%s' % (version_info[3],)
    if version_info[4]:
        version = version + '-%s' % (version_info[4],)
    return version

def write_version_file(project_root, package_name, version_info):
    version = version_info_to_version(version_info)
    with open(os.path.join(project_root, package_name, '__version__.py'), 'wb') as f:
        f.write('# this file is autogenerated by bfc.setup\n')
        f.write('version_info = %r\n' % (version_info, ))
        f.write('version = "%s"\n' % (version,))
    return version

def setup_packages(project_root, package_path):
    return [root[len(project_root)+len(os.path.sep):].replace(os.path.sep, '.')
            for root, dirs, files in os.walk(os.path.join(project_root, package_path))]

def _get_current_revision():
    p = subprocess.Popen(['hg', 'parent', '--template', '{rev}'],
                         stdout = subprocess.PIPE)
    rev, err = p.communicate()
    r = p.wait()
    if r:
        raise Exception('Getting current revision failed', r)
    return rev

def _create_tag(tag):
    print 'Creating tag:', tag
    p = subprocess.Popen(['hg', 'tag', tag])
    r = p.wait()
    if r:
        raise Exception('Creating a tag failed', r)

def _update_to(rev):
    print 'Updating to revision:', rev
    p = subprocess.Popen(['hg', 'up', '-c', rev])
    r = p.wait()
    if r:
        raise Exception('Creating a tag failed', r)
    
def _generate_package(formats):
    if os.path.exists('MANIFEST'):
        os.unlink('MANIFEST')
    print 'Generating a package in formats:', formats
    p = subprocess.Popen([sys.executable, 'setup.py', 'sdist', '--formats', formats])
    r = p.wait()
    if r:
        raise Exception('Generating a package failed', r)

def _check_uncommited():
    p = subprocess.Popen(['hg', 'st'],
                         stdout = subprocess.PIPE)
    out, err = p.communicate()
    r = p.wait()
    if r:
        raise Exception('Checking commits failed', r)
    if out:
        raise Exception('Uncommitted local changes')
    

def build_release_file(formats = 'bztar', tag = None):

    _check_uncommited()

    if tag is not None:
        _create_tag(tag)
        revision = _get_current_revision()
        _update_to(tag)

    _generate_package(formats)

    if tag is not None:
        _update_to(revision)
        
if __name__ == '__main__':
    parser = optparse.OptionParser(usage="usage: %prog [options]")
    parser.add_option('-f', '--formats', dest='formats', default='bztar', metavar='FORMATS',
                      help='a comma-separated of source distributions of your packages (.tar.bz2 by default)')
    parser.add_option('-t', '--tag', dest='tag', default=None, metavar='TAG',
                      help='a tag to generate for the release. If not specified a bleeding-edge one will be created')

    options, args = parser.parse_args()
    if args:
        parser.print_help()
    build_release_file(options.formats, options.tag)
    
                      
