# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
A frozen dictionary: one that cannot be changed after creation.
Therefore it might a hash, and it is sortable.

>>> FrozenDict(one=1,two=2)
FrozenDict({'two': 2, 'one': 1})
>>> f = FrozenDict(((1,2),(3,4)))
>>> f
FrozenDict({1: 2, 3: 4})
>>> len(f)
2
>>> f[1]
2
>>> 1 in f
True
>>> 2 in f
False
>>> list(f)
[1, 3]
>>> list(f.iterkeys())
[1, 3]
>>> f.keys()
[1, 3]
>>> list(f.itervalues())
[2, 4]
>>> f.values()
[2, 4]
>>> list(f.iteritems())
[(1, 2), (3, 4)]
>>> f.items()
[(1, 2), (3, 4)]
>>> str(f)
'{1: 2, 3: 4}'
>>> f.get(1)
2
>>> f.get(2)
>>> f.get(2, 10)
10
>>> h = hash(f)
>>> f < FrozenDict(((1,2),(3,5)))
True
>>> f < FrozenDict(((1,2),(3,3)))
False
>>> import cPickle as pickle
>>> s = pickle.dumps(f)
>>> f2 = pickle.loads(s)
>>> f2
FrozenDict({1: 2, 3: 4})
>>> f2 == f
True
>>> f.freeze() is f
True

"""

from collections import Mapping

__all__ = ['FrozenDict']

class FrozenDict(object):
    __slots__ = ['_dict',
                 '_essence',
                 '_hash',
        ]
    def __init__(self, *args, **kws):
        self._dict = dict(*args, **kws)
        self._essence = None
        self._hash = None

    def __len__(self):
        return len(self._dict)

    def __iter__(self):
        return iter(self._dict)

    def __contains__(self, key):
        return key in self._dict

    def iterkeys(self):
        return self._dict.iterkeys()

    def keys(self):
        return self._dict.keys()

    def itervalues(self):
        return self._dict.itervalues()

    def values(self):
        return self._dict.values()

    def iteritems(self):
        return self._dict.iteritems()

    def items(self):
        return self._dict.items()

    def __getitem__(self, key):
        return self._dict[key]

    def get(self, key, default = None):
        return self._dict.get(key, default)

    def _essence_(self):
        if self._essence is None:
            items = self.items()
            items.sort()
            self._essence = tuple(items)
        return self._essence
    
    def __hash__(self):
        if self._hash is None:
            self._hash = hash(self._essence_())
        return self._hash

    def __cmp__(self, other):
        return cmp(self._essence_(), other._essence_())

    def freeze(self):
        return self

    def __getstate__(self):
        return self._dict

    def __setstate__(self, s):
        self._dict = s
        self._essence = None
        self._hash = None

    def __repr__(self):
        return '%s(%r)' % (self.__class__.__name__, self._dict)

    def __str__(self):
        return str(self._dict)

Mapping.register(FrozenDict)
