# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
A simple derived set type: its fathers are constant.
It behaves mostly like the OrderedSet type.

Note that the iteration starts with the internal set of the instance, and then the fathers' ones,
while skipping elements that were already yielded.

The internal set is actually an OrderedSet.

>>> a = DerivedSet()
>>> a.update('aA')
>>> tuple(a)
('a', 'A')
>>> a.add('x')
>>> tuple(a)
('a', 'A', 'x')
>>> a.discard('x')
>>> tuple(a)
('a', 'A')
>>> for x in a: print(x)
a
A
>>> 'b' in a
False
>>> 'a' in a
True
>>> a
DerivedSet(OrderedSet(['a', 'A']), (), ())

But it has support for being derived from other DerivedSet instances.
When creating a new instance, pass a sequence of fathers
(it must be able to be iterated upon unlimited number of times!).

>>> a = DerivedSet()
>>> a.update('aA')
>>> b = DerivedSet(derivable_fathers = (a,))
>>> tuple(b)
('a', 'A')
>>> b.update('bB')
>>> tuple(b)
('b', 'B', 'a', 'A')
>>> a.discard('a')
>>> tuple(b)
('b', 'B', 'A')
>>> a.update('axyz')
>>> tuple(b)
('b', 'B', 'A', 'a', 'x', 'y', 'z')
>>> 'x' in b
True
>>> 'B' in b
True
>>> 'B' in a
False
>>> b += 'xyz'
>>> tuple(b)
('b', 'B', 'x', 'y', 'z', 'A', 'a')
>>> for x in 'xyz': a.discard(x)
... 
>>> tuple(b)
('b', 'B', 'x', 'y', 'z', 'A', 'a')
>>> tuple(a)
('A', 'a')
>>> b.clear()
>>> tuple(b)
('A', 'a')

A father can be any sequence (must be iterable multiple times):

>>> a = DerivedSet(derivable_fathers = ((1,2,3),))
>>> a.add(4)
>>> tuple(a)
(4, 1, 2, 3)

Non-derivable fathers: if a DerivedSet b has non-derivable fathers, these fathers will not affect b's children:

>>> a = DerivedSet()
>>> at = DerivedSet()
>>> b = DerivedSet(derivable_fathers = (a,), non_derivable_fathers = (at,))
>>> c = DerivedSet(derivable_fathers = (b,))
>>> a.add(5)
>>> at.add(6)
>>> tuple(b)
(5, 6)
>>> tuple(c)
(5,)
>>> b.derivable_fathers[0] is a
True
>>> b.non_derivable_fathers[0] is at
True
>>> f = list(b.fathers)
>>> f[0] is a
True
>>> f[1] is at
True

A father can be any type of set, even not a (Frozen)DerivedSet one. An (Frozen)OrderdedSet is recommended.

>>> a = DerivedSet('ab')
>>> a.add('c')
>>> tuple(a)
('a', 'b', 'c')

Additionally, it can be freezed. A freezed copy can be a father for new sets, and it is comparable and hashable

>>> a = DerivedSet('a')
>>> b = DerivedSet('b', derivable_fathers = (a,))
>>> b.freeze()
>>> a.is_frozen
True
>>> a.freeze()
>>> tuple(b)
('b', 'a')
>>> s = set([b])
>>> b in s
True
>>> b == DerivedSet('b', derivable_fathers = (DerivedSet('a'),), freeze = True)
True

If you want to forget about your fathers, you can flatten:

>>> a = DerivedSet('a')
>>> A = DerivedSet('A')
>>> b = DerivedSet('b', derivable_fathers = (a,), non_derivable_fathers = (A,))
>>> b.flatten()
>>> ''.join(b)
'baA'
>>> list(b.fathers)
[]

DerivedSets are picklable:

>>> import cPickle as pickle
>>> a = DerivedSet()
>>> b = DerivedSet(derivable_fathers = (a,))
>>> a.add('A')
>>> b.add('b')
>>> s = pickle.dumps(b, -1)
>>> del b
>>> del a
>>> b = pickle.loads(s)
>>> list(b)
['b', 'A']
>>> import cPickle as pickle
>>> a = DerivedSet()
>>> a.add('A')
>>> a.freeze()
>>> b = DerivedSet(derivable_fathers = (a,))
>>> b.add('b')
>>> b.freeze()
>>> s = pickle.dumps(b, -1)
>>> del b
>>> del a
>>> b = pickle.loads(s)
>>> list(b)
['b', 'A']

"""

from .base import DerivedBase

__all__ = ['DerivedSet', 'EMPTY']

from ..orderedset import OrderedSet

class DerivedSet(DerivedBase):
    __slots__ = []

    def __init__(self, sequence = (), derivable_fathers = (), non_derivable_fathers = (), freeze = False):
        DerivedBase.__init__(self, OrderedSet(sequence), derivable_fathers, non_derivable_fathers, freeze)
    
    ### collections.Iterable requirements
    def __iter__(self):
        seen = set()

        for s in self._iter_internal_data():
            for x in s:
                if x not in seen:
                    yield x
            seen.update(s)

    def add(self, obj):
        assert not self.is_frozen
        self._internal_data.add(obj)

    def discard(self, obj):
        assert not self.is_frozen
        self._internal_data.discard(obj)

    def __iadd__(self, other):
        self.update(other)
        return self

EMPTY = DerivedSet((), freeze = False)
