!***************************************************************************************************
! Copyright 1987, D. Bilitza, 2007 S. Bourdarie
!
! This file is part of IRBEM-LIB.
!
!    IRBEM-LIB is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Lesser General Public License as published by
!    the Free Software Foundation, either version 3 of the License, or
!    (at your option) any later version.
!
!    IRBEM-LIB is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU Lesser General Public License for more details.
!
!    You should have received a copy of the GNU Lesser General Public License
!    along with IRBEM-LIB.  If not, see <http://www.gnu.org/licenses/>.
!
!
! CREATION: S. Bourdarie - ONERA-DESP
!
! FILE CONTENT:
!               SUBROUTINE fly_in_nasa_aeap1: Allows to fly a spacecraft in AE8-AP8 min and max models.
!               SUBROUTINE get_AE8_AP8_flux: Compute flux from AE8 or AP8 for an energy, B/B0 and L
!               SUBROUTINE TRARA1: Finds particle fluxes for given energies, magnetic field strength and L-value
!               FUNCTION TRARA2: Interpolates linearly in L-B/B0-MAP to obtain the logarithm of integral flux at given L and B/B0
!               SUBROUTINE Init_AE8max: Set block data for NASA AE8max model
!               SUBROUTINE Init_AE8min: Set block data for NASA AE8min model
!               SUBROUTINE Init_AP8max: Set block data for NASA AP8max model
!               SUBROUTINE Init_AP8min: Set block data for NASA AP8min model
!
!***************************************************************************************************
!--------------------------------------------------------------------------------------------
!+
! NAME:
!	fly_in_nasa_aeap1
!
! PURPOSE:
!	This subroutine Allows to fly a spacecraft in AE8-AP8 min and max models.
!
! CATEGORY:
!	Radiation belt model
!
! CALLING SEQUENCE:
!	CALL fly_in_nasa_aeap1(ntime,sysaxes,whichm,whatf,energy,xIN1,xIN2,xIN3,flux)
!
! INPUTS:
!	ntime -> maximum number of time to compute (long integer)
!	sysaxes -> define which coordinate system is provided in
!	whichm -> which model to use, 1=AE8min 2=AE8max 3=AP8min 4=AP8max (long integer)
!	whatf -> which kind of flux, 1=differential 2=E range 3=integral (long integer)
!	Nene -> Number of energy channels to compute
!	energy -> energy (MeV) at which fluxes must be computed (double array [2,25])
!	iyear,idoy,UT -> times when flux are to be computed (not usefull if imput position is not in GSE, GSM, SM,GEI) (respectively long array(ntime_max), long array(ntime_max), double array(ntime_max))
!	xIN1 -> first coordinate in the chosen system (double array [ntime_max])
!	xIN2 -> second coordinate in the chosen system (double array [ntime_max])
!	xIN3 -> third coordinate in the chosen system (double array [ntime_max])
!
! OUTPUTS:
!	flux -> Computed fluxes (MeV-1 cm-2 s-1 or cm-2 s-1) (double array [ntime_max,25])
!
! COMMON BLOCKS:
!	COMMON/GENER/ERA,AQUAD,BQUAD
!	COMMON /magmod/k_ext,k_l,kint
!	COMMON /flag_L/Ilflag
!	COMMON /dip_ang/tilt
!
! MODIFICATION HISTORY:
!	Written by: S. Bourdarie (introduced in version 3.0), September 2005
!				S. Bourdarie - March 2007 (add multi channel calculcations;
!					add time input for coordinates transformations) - V4.1
!-
!--------------------------------------------------------------------------------------------
      SUBROUTINE fly_in_nasa_aeap1(ntime,sysaxes,whichm,whatf,nene,
     &                        energy,iyear,idoy,UT,xIN1,xIN2,xIN3,flux)
c
      IMPLICIT NONE
      INCLUDE 'variables.inc'
      INCLUDE 'ntime_max.inc'
C
c declare inputs
      INTEGER*4 nene_max
      PARAMETER (nene_max=25)
      INTEGER*4  ntime,sysaxes,whichm,whatf,Nene
      INTEGER*4  iyear(ntime_max),idoy(ntime_max)
      REAL*8     energy(2,nene_max)
      REAL*8     UT(ntime_max)
      real*8     xIN1(ntime_max),xIN2(ntime_max),xIN3(ntime_max)
c Declare internal variables
      INTEGER*4  k_ext,k_l,isat,kint
      INTEGER*4  t_resol,r_resol,Ilflag
      INTEGER*4  iyear_dip,idoy_dip
      REAL*8     xGEO(3),xMAG(3),xSUN(3),rM,MLAT,Mlon1
      REAL*8     xGSM(3),xSM(3),xGEI(3),xGSE(3)
      real*8     alti,lati,longi,UT_dip,psi,tilt
      REAL*8     ERA,AQUAD,BQUAD
      REAL*8     BLOCAL(ntime_max),BMIN(ntime_max),XJ(ntime_max)
      REAL*8     Lm(ntime_max),Lstar(ntime_max),BBo(ntime_max)
c
c Declare output variables
      REAL*8     flux(ntime_max,nene_max)
C
      COMMON/GENER/ERA,AQUAD,BQUAD
      COMMON /magmod/k_ext,k_l,kint
      COMMON /flag_L/Ilflag
      COMMON /dip_ang/tilt
        REAL*8     pi,rad
        common /rconst/rad,pi
      DATA  xSUN /1.d0,0.d0,0.d0/
C
      Ilflag=0
      k_ext=0
      t_resol=3
      r_resol=0
      k_l=0
      if (ABS(whichm) .lt. 1 .or. ABS(whichm) .gt. 4) then
         whichm=1
         WRITE(6,*)
         WRITE(6,*)'!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!'
         WRITE(6,*)'Invalid NASA AE8 or AP8 specification'
         WRITE(6,*)'Selecting AE8 min'
         WRITE(6,*)'!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!'
         WRITE(6,*)
      endif
      if (ABS(whichm) .eq. 4) then
         kint=3
      else
         kint=2
      endif
      if (whatf .lt. 1 .or. whatf .gt. 3) then
         whatf=1
         WRITE(6,*)
         WRITE(6,*)'!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!'
         WRITE(6,*)'Invalid flux output specification'
         WRITE(6,*)'Selecting differential flux'
         WRITE(6,*)'!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!'
         WRITE(6,*)
      endif
c
c
      CALL INITIZE
      if (kint .eq. 2) then
         CALL JensenANDCain1960
         iyear_dip=1960
      endif
      if (kint .eq. 3) then
         CALL GSFC1266
        iyear_dip=1970
      endif
c
      idoy_dip=0
      UT_dip=0.d0
      DO isat = 1,ntime
         ! need to reinitialize sun for GEI to GEO coordinate transforms
         CALL INIT_GSM(iyear(isat),idoy(isat),UT(isat),psi) ! compute sun among other things
         tilt = psi/rad ! in common block dip_ang

	    call get_coordinates ( sysaxes, 
     6        xIN1(isat), xIN2(isat), xIN3(isat), 
     6        alti, lati, longi, xGEO )

      CALL calcul_Lstar_opt(t_resol,r_resol,xGEO
     &     ,Lm(isat),Lstar(isat),XJ(isat),BLOCAL(isat),BMIN(isat))
C           BBo(isat)=BLOCAL(isat)/BMIN(isat)  ! removed to use McIlwain Gmagmo for AP/AE
      if (Lm(isat) .le. 0.D0 .and. Lm(isat) .ne. -1.D31)
     &       Lm(isat)=-Lm(isat)

         BBo(isat)=BLOCAL(isat)/(31165.3/Lm(isat)**3)  ! use McIlwain Gmagmo for B0 calc.
!        write(6,'(I10,3f12.5)')isat,Lm(isat),BLOCAL(isat),BMIN(isat)
      enddo
      call get_AE8_AP8_flux(ntime,whichm,whatf,nene,
     &                             energy,BBo,Lm,flux)
        end

!
!--------------------------------------------------------------------------------------------
!+
! NAME:
!	get_AE8_AP8_flux
!
! PURPOSE:
!	This subroutine Allows to fly a spacecraft in AE8-AP8 min and max models.
!
! CATEGORY:
!	Radiation belt model
!
! CALLING SEQUENCE:
!	CALL get_AE8_AP8_flux(ntmax,whichm,whatf,nene,energy,BBo,L,flux)
!
! INPUTS:
!	ntmax -> maximum number of time to compute (long integer)
!	whichm -> which model to use, 1=AE8min 2=AE8max 3=AP8min 4=AP8max (long integer)
!	whatf -> which kind of flux, 1=differential 2=E range 3=integral (long integer)
!	Nene -> Number of energy channels to compute
!	energy -> energy (MeV) at which fluxes must be computed (double array [2,25])
!	BBo -> Blocal/Bequator (double array [ntime_max])
!	L -> McIlwain L (double array [ntime_max])
!
! OUTPUTS:
!	flux -> Computed fluxes (MeV-1 cm-2 s-1 or cm-2 s-1) (double array [ntime_max,25])
!
! COMMON BLOCKS:
!	COMMON /PROMIN/ IHEADPMIN, MAPPMIN
!	COMMON /PROMAX/ IHEADPMAX, MAPPMAX
!	COMMON /ELEMIN/ IHEADEMIN, MAPEMIN
!	COMMON /ELEMAX/ IHEADEMAX, MAPEMAX
!
! MODIFICATION HISTORY:
!	Written by: S. Bourdarie (introduced in version 3.0), September 2005
!				S. Bourdarie - March 2007 (add multi channel calculcations) - V4.1
!-
!--------------------------------------------------------------------------------------------
      SUBROUTINE get_AE8_AP8_flux(ntmax,whichm,whatf,nene,
     &                             energy,BBo,L,flux)
c
      IMPLICIT NONE
      INCLUDE 'variables.inc'
      INCLUDE 'ntime_max.inc'
c
      REAL*8       xMinP_L, xMaxP_L, XMinE_L, xMaxE_L
      PARAMETER   (xMINP_L = 1.0)   ! minimum Lshell to calculate proton fluxes
      PARAMETER   (xMAXP_L = 11.0)  ! maximum Lshell to calculate proton fluxes

      PARAMETER   (xMINE_L = 1.2)   ! minimum Lshell to calculate electron fluxes
      PARAMETER   (xMAXE_L = 12.0)  ! maximum Lshell to calculate electron fluxes

      INTEGER*4   ntmax,i,nene,ieny
      INTEGER*4   whichm  !1=AE8min 2=AE8max 3=AP8min 4=AP8max
      INTEGER*4   whatf  !1=diff 2=E range 3=int
      INTEGER*4 MAPPMIN(16582), MAPPMAX(16291),
     &           MAPEMIN(13168), MAPEMAX(13548)
      REAL*8      energy(2,25)
      REAL*8      flux(ntime_max,25),BBo(ntime_max),L(ntime_max)
      REAL*8    FL,FL1
      INTEGER*4 IHEADPMIN(8),IHEADPMAX(8),IHEADEMIN(8),IHEADEMAX(8)
c
      COMMON /PROMIN/ IHEADPMIN, MAPPMIN
      COMMON /PROMAX/ IHEADPMAX, MAPPMAX
      COMMON /ELEMIN/ IHEADEMIN, MAPEMIN
      COMMON /ELEMAX/ IHEADEMAX, MAPEMAX
c
c  init
      DO i=1,ntmax
         do ieny=1,25
            Flux(i,ieny) = baddata
         enddo
      enddo
c
c  AE8min
c
      if (ABS(whichm) .EQ. 1) then
         CALL Init_AE8min
         if (whatf.EQ. 1) then
            DO i=1,ntmax
               if (L(i).GE.xMINE_l .AND. L(i).LE. xMAXE_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADEMIN,MAPEMIN,L(i),BBo(i),
     &                    energy(1,ieny)-0.001D0,FL, whichm)
                     FL1 = 10.D0** FL
                     CALL TRARA(IHEADEMIN,MAPEMIN,L(i),BBo(i),
     &                    energy(1,ieny)+0.001D0,FL, whichm)
                     if (FL1.GT.1.D0 .AND. FL.GT.0.D0)
     &                   Flux(i,ieny) = (FL1-10.D0** FL)/0.002D0
                     if (energy(1,ieny) .lt. 0.05D0 .or.
     &               energy(1,ieny) .gt. 7.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
         if (whatf.EQ. 2) then
            DO i=1,ntmax
               if (L(i).GE. xMINE_L .AND. L(i).LE. xMAXE_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADEMIN,MAPEMIN,L(i),BBo(i),
     &                     energy(1,ieny),FL, whichm)
                     FL1 = 10.D0** FL
                     CALL TRARA(IHEADEMIN,MAPEMIN,L(i),BBo(i),
     &                        energy(2,ieny),FL, whichm)
                     if (FL1.GT.1.D0 .AND. FL.GT.0.D0)
     &                  Flux(i,ieny) = (FL1-10.D0** FL)/
     &                          (energy(2,ieny)-energy(1,ieny))
                     if (energy(1,ieny) .lt. 0.05D0 .or.
     &               energy(1,ieny) .gt. 7.D0) Flux(i,ieny)=baddata
                     if (energy(2,ieny) .lt. 0.05D0 .or.
     &               energy(2,ieny) .gt. 7.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
         if (whatf.EQ. 3) then
            DO i=1,ntmax
               if (L(i).GE. xMINE_L .AND. L(i).LE. xMAXE_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADEMIN,MAPEMIN,L(i),BBo(i),
     &                        energy(1,ieny),FL, whichm)
                     if (FL.GT.0.D0) Flux(i,ieny) = 10.D0** FL
                     if (energy(1,ieny) .lt. 0.05D0 .or.
     &               energy(1,ieny) .gt. 7.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
      endif
c
c  AE8max
C
      if (ABS(whichm) .EQ. 2) then
         CALL Init_AE8max
         if (whatf.EQ. 1) then
            DO i=1,ntmax
               if (L(i).GE. xMINE_L .AND. L(i).LE. xMAXE_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADEMAX,MAPEMAX,L(i),BBo(i),
     &                     energy(1,ieny)-0.001D0,FL, whichm)
                     FL1 = 10.D0** FL
                     CALL TRARA(IHEADEMAX,MAPEMAX,L(i),BBo(i),
     &                        energy(1,ieny)+0.001D0,FL, whichm)
                     if (FL1.GT.1.D0 .AND. FL.GT.0.D0)
     &                     Flux(i,ieny) = (FL1-10.D0** FL)/0.002D0
                     if (energy(1,ieny) .lt. 0.05D0 .or.
     &               energy(1,ieny) .gt. 7.D0) Flux(i,ieny)=baddata
c             if (Flux(i,ieny).LE.0.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
         if (whatf.EQ. 2) then
            DO i=1,ntmax
               if (L(i).GE. xMINE_L .AND. L(i).LE. xMAXE_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADEMAX,MAPEMAX,L(i),BBo(i),
     &                     energy(1,ieny),FL, whichm)
                     FL1 = 10.D0** FL
                     CALL TRARA(IHEADEMAX,MAPEMAX,L(i),BBo(i),
     &                        energy(2,ieny),FL, whichm)
                     if (FL1.GT.1.D0 .AND. FL.GT.0.D0)
     &                     Flux(i,ieny) = (FL1-10.D0** FL)/
     &                    (energy(2,ieny)-energy(1,ieny))
                     if (energy(1,ieny) .lt. 0.05D0 .or.
     &               energy(1,ieny) .gt. 7.D0) Flux(i,ieny)=baddata
                     if (energy(2,ieny) .lt. 0.05D0 .or.
     &               energy(2,ieny) .gt. 7.D0) Flux(i,ieny)=baddata
c             if (Flux(i,ieny).LE.0.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
         if (whatf.EQ. 3) then
            DO i=1,ntmax
               if (L(i).GE. xMINE_L .AND. L(i).LE. xMAXE_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADEMAX,MAPEMAX,L(i),BBo(i),
     &                        energy(1,ieny),FL, whichm)
                     if (FL.GT.0.D0) Flux(i,ieny) = 10.D0** FL
                     if (energy(1,ieny) .lt. 0.05D0 .or.
     &               energy(1,ieny) .gt. 7.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
      endif
c
c  AP8min
C
      if (ABS(whichm) .EQ. 3) then
         CALL Init_AP8min
         if (whatf.EQ. 1) then
            DO i=1,ntmax
               if (L(i).GE. xMINP_L .AND. L(i).LE.xMAXP_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADPMIN,MAPPMIN,L(i),BBo(i),
     &                     energy(1,ieny)-0.001D0,FL, whichm)
                     FL1 = 10.D0** FL
                     CALL TRARA(IHEADPMIN,MAPPMIN,L(i),BBo(i),
     &                        energy(1,ieny)+0.001D0,FL, whichm)
                     if (FL1.GT.1.D0 .AND. FL.GT.0.D0)
     &                     Flux(i,ieny) = (FL1-10.D0** FL)/0.002D0
                     if (energy(1,ieny) .lt. 0.1D0 .or.
     &               energy(1,ieny) .gt. 300.D0) Flux(i,ieny)=baddata
c             if (Flux(i,ieny).LE.0.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
         if (whatf.EQ. 2) then
            DO i=1,ntmax
               if (L(i).GE. xMINP_L .AND. L(i).LE.xMAXP_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADPMIN,MAPPMIN,L(i),BBo(i),
     &                     energy(1,ieny),FL, whichm)
                     FL1 = 10.D0** FL
                     CALL TRARA(IHEADPMIN,MAPPMIN,L(i),BBo(i),
     &                        energy(2,ieny),FL, whichm)
                     if (FL1.GT.1.D0 .AND. FL.GT.0.D0)
     &                     Flux(i,ieny) = (FL1-10.D0** FL)/
     &                     (energy(2,ieny)-energy(1,ieny))
                     if (energy(1,ieny) .lt. 0.1D0 .or.
     &               energy(1,ieny) .gt. 300.D0) Flux(i,ieny)=baddata
                     if (energy(2,ieny) .lt. 0.1D0 .or.
     &               energy(2,ieny) .gt. 300.D0) Flux(i,ieny)=baddata
c             if (Flux(i,ieny).LE.0.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
         if (whatf.EQ. 3) then
            DO i=1,ntmax
               if (L(i).GE. xMINP_L .AND. L(i).LE.xMAXP_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADPMIN,MAPPMIN,L(i),BBo(i),
     &                        energy(1,ieny),FL, whichm)
                     if (FL.GT.0.D0) Flux(i,ieny) = 10.D0** FL
                     if (energy(1,ieny) .lt. 0.1D0 .or.
     &               energy(1,ieny) .gt. 300.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
      endif
c
c  AP8max
C
      if (ABS(whichm) .EQ. 4) then
         CALL Init_AP8max
         if (whatf.EQ. 1) then
            DO i=1,ntmax
               if (L(i).GE. xMINP_L .AND. L(i).LE.xMAXP_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADPMAX,MAPPMAX,L(i),BBo(i),
     &                     energy(1,ieny)-0.001D0,FL, whichm)
                     FL1 = 10.D0** FL
                     CALL TRARA(IHEADPMAX,MAPPMAX,L(i),BBo(i),
     &                        energy(1,ieny)+0.001D0,FL, whichm)
                     if (FL1.GT.1.D0 .AND. FL.GT.0.D0)
     &                     Flux(i,ieny) = (FL1-10.D0** FL)/0.002D0
                     if (energy(1,ieny) .lt. 0.1D0 .or.
     &               energy(1,ieny) .gt. 300.D0) Flux(i,ieny)=baddata
c             if (Flux(i,ieny).LE.0.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
         if (whatf.EQ. 2) then
            DO i=1,ntmax
               if (L(i).GE. xMINP_L .AND. L(i).LE.xMAXP_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADPMAX,MAPPMAX,L(i),BBo(i),
     &                     energy(1,ieny),FL, whichm)
                     FL1 = 10.D0** FL
                     CALL TRARA(IHEADPMAX,MAPPMAX,L(i),BBo(i),
     &                        energy(2,ieny),FL, whichm)
                     if (FL1.GT.1.D0 .AND. FL.GT.0.D0)
     &                     Flux(i,ieny) = (FL1-10.D0** FL)/
     &                     (energy(2,ieny)-energy(1,ieny))
                     if (energy(1,ieny) .lt. 0.1D0 .or.
     &               energy(1,ieny) .gt. 300.D0) Flux(i,ieny)=baddata
                     if (energy(2,ieny) .lt. 0.1D0 .or.
     &               energy(2,ieny) .gt. 300.D0) Flux(i,ieny)=baddata
c             if (Flux(i,ieny).LE.0.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
         if (whatf.EQ. 3) then
            DO i=1,ntmax
               if (L(i).GE. xMINP_L .AND. L(i).LE.xMAXP_L) then
c loop on energies
                  do ieny=1,nene
                     CALL TRARA(IHEADPMAX,MAPPMAX,L(i),BBo(i),
     &                        energy(1,ieny),FL, whichm)
                     if (FL.GT.0.D0) Flux(i,ieny) = 10.D0** FL
                     if (energy(1,ieny) .lt. 0.1D0 .or.
     &               energy(1,ieny) .gt. 300.D0) Flux(i,ieny)=baddata
                  enddo
               endif
            enddo
         endif
      endif
      end
!
!--------------------------------------------------------------------------------------------
!+
! NAME:
!	TRARA

! PURPOSE:
!	This subroutine wraps the original TRARA1 and the ESA_TRARA1 functions.
!       if WHATM is negative, then the ESA models are used, if positive, then the
!       original TRARA1 is used.
!
! CATEGORY:
!	Radiation belt model
!
! CALLING SEQUENCE:
!	CALL TRARA(DESCR,MAP,FL,BB0,E,F, whichm)
!
! INPUTS:
!	DESCR(8) -> Header of specified trapped dadiation model
!	MAP(...) -> MAP of trapped radiation model (see block data)
!	FL -> L-value
!	BB0 -> B/B0 magnetic field strength normalized to field strength at magnetic equator
!	E -> energy in MeV
!       whichm -> Which model to use.
!
! OUTPUTS:
!	F -> Decadic logarithm of integral fluxes in #/(cm2 s)
!
! COMMON BLOCKS:
!
! MODIFICATION HISTORY:
!	Written by: H. Evans - April 2009
!-
!--------------------------------------------------------------------------------------------
      SUBROUTINE TRARA(DESCR, MAP, FL, BB0, E, F, whichm)
      IMPLICIT NONE

      INTEGER*4 MAP(*)
      INTEGER*4 DESCR(8)
      REAL*8    FL, BB0, E, F
      INTEGER*4 whichm

      REAL*4    xDESCR(8)
      INTEGER*4 i

      IF (whichm .GT. 0) THEN
         CALL TRARA1    ( DESCR, MAP, FL, BB0, E, F)
      ELSE
C        ESA versions of A*-8 models used REAL*4 numbers in
C        the map headers.
         DO i=1,8
            xDESCR(i) = DESCR(i)
         ENDDO
         CALL ESA_TRARA1( xDESCR, MAP, FL, BB0, E, F)
      ENDIF
      RETURN
      END
      

!--------------------------------------------------------------------------------------------
!+
! NAME:
!	TRARA1

! PURPOSE:
!	This subroutine finds particle fluxes for given energies, magnetic field strength and L-value.
!
! CATEGORY:
!	Radiation belt model
!
! CALLING SEQUENCE:
!	CALL call TRARA1(DESCR,MAP,FL,BB0,E,F)
!
! INPUTS:
!	DESCR(8) -> Header of specified trapped dadiation model
!	MAP(...) -> MAP of trapped radiation model (see block data)
!	FL -> L-value
!	BB0 -> B/B0 magnetic field strength normalized to field strength at magnetic equator
!	E -> energy in MeV
!
! OUTPUTS:
!	F -> Decadic logarithm of integral fluxes in #/(cm2 s)
!
! COMMON BLOCKS:
!	COMMON/TRA2/FISTEP
!
! MODIFICATION HISTORY:
!	Written by: D. Bilitza, 1987
!				S. Bourdarie (introduced in version 3.0) - September 2005
!-
!--------------------------------------------------------------------------------------------
      SUBROUTINE TRARA1(DESCR,MAP,FL,BB0,E,F)
c
      IMPLICIT NONE
      LOGICAL S0,S1,S2
      INTEGER*4 MAP(1)
      INTEGER*4 DESCR(8)
      INTEGER*4 NL,NB
      INTEGER*4 I0,I1,I2,I3,L3
      REAL*8    E,F,XNL,BB0,FL,TRARA2
      REAL*8    FISTEP,ESCALE,FSCALE
      REAL*8    E0,E1,E2,F0,F1,F2
      COMMON/TRA2/FISTEP
      DATA F1,F2/1.001D0,1.002D0/
C
      FISTEP=dble(DESCR(7)/DESCR(2))
      ESCALE=dble(DESCR(4))
      FSCALE=dble(DESCR(7))
      XNL=1.D0*MIN(15.6D0,ABS(FL))
      NL=INT(XNL*DESCR(5))
      IF(BB0.LT.1.D0) BB0=1.D0
      NB=INT((BB0-1.D0)*DESCR(6))
C
C I2 IS THE NUMBER OF ELEMENTS IN THE FLUX MAP FOR THE FIRST ENERGY.
C I3 IS THE INDEX OF THE LAST ELEMENT OF THE SECOND ENERGY MAP.
C L3 IS THE LENGTH OF THE MAP FOR THE THIRD ENERGY.
C E1 IS THE ENERGY OF THE FIRST ENERGY MAP (UNSCALED)
C E2 IS THE ENERGY OF THE SECOND ENERGY MAP (UNSCALED)
C
      I1=0
      I2=MAP(1)
      I3=I2+MAP(I2+1)
      L3=MAP(I3+1)
      E1=MAP(I1+2)/ESCALE
      E2=MAP(I2+2)/ESCALE
C
C S0, S1, S2 ARE LOGICAL VARIABLES WHICH INDICATE WHETHER THE FLUX FOR
C A PARTICULAR E, B, L POINT HAS ALREADY BEEN FOUND IN A PREVIOUS CALL
C TO FUNCTION TRARA2. IF NOT, S.. =.TRUE.
C
      S1=.TRUE.
      S2=.TRUE.
C
C FOR THE ENERGY E FIND THE SUCCESSIVE ENERGIES E0,E1,E2 IN
C MODEL MAP, WHICH OBEY  E0 < E1 < E < E2 .
C
  1   IF((E.LE.E2).OR.(L3.EQ.0)) GOTO 2
          I0=I1
          I1=I2
          I2=I3
          I3=I3+L3
          L3=MAP(I3+1)
          E0=E1
          E1=E2
          E2=MAP(I2+2)/ESCALE
          S0=S1
          S1=S2
          S2=.TRUE.
          F0=F1
          F1=F2
          GOTO 1
C
C CALL TRARA2 TO INTERPOLATE THE FLUX-MAPS FOR E1,E2 IN L-B/B0-
C SPACE TO FIND FLUXES F1,F2 [IF THEY HAVE NOT ALREADY BEEN
C CALCULATED FOR A PREVIOUS E(I)].
C
  2   IF(S1) F1=TRARA2(MAP(I1+3),NL,NB)/FSCALE
      IF(S2) F2=TRARA2(MAP(I2+3),NL,NB)/FSCALE
      S1=.FALSE.
      S2=.FALSE.
C
C FINALLY, INTERPOLATE IN ENERGY.
C
      F=F1+(F2-F1)*(E-E1)/(E2-E1)
      IF(F2.GT.0.0D0) GOTO 3
      IF(I1.EQ.0)   GOTO 3
C
C --------- SPECIAL INTERPOLATION ---------------------------------
C IF THE FLUX FOR THE SECOND ENERGY CANNOT BE FOUND (I.E. F2=0.0),
C AND THE ZEROTH ENERGY MAP HAS BEEN DEFINED (I.E. I1 NOT EQUAL 0),
C THEN INTERPOLATE USING THE FLUX MAPS FOR THE ZEROTH AND FIRST
C ENERGY AND CHOOSE THE MINIMUM OF THIS INTERPOLATIONS AND THE
C INTERPOLATION THAT WAS DONE WITH F2=0.
C
      IF(S0) F0=TRARA2(MAP(I0+3),NL,NB)/FSCALE
      S0=.FALSE.
      F=1.D0*MIN(F,F0+(F1-F0)*(E-E0)/(E1-E0))
C
C THE LOGARITHMIC FLUX IS ALWAYS KEPT GREATER OR EQUAL ZERO.
C
  3   F=1.D0*MAX(F,0.D0)
      RETURN
      END
C
!--------------------------------------------------------------------------------------------
!+
! NAME:
!	TRARA2

! PURPOSE:
!	This function interpolates linearly in L-B/B0-MAP to obtain the logarithm of integral flux at given L and B/B0
!
! CATEGORY:
!	Radiation belt model
!
! CALLING SEQUENCE:
!	result=TRARA2(MAP,IL,IB)
!
! INPUTS:
!	MAP(...) -> is the sub-MAP (for specific energy) of trapped radiation model (see block data)
!	IL -> Scaled L-value
!	IB -> scaled B/B0 magnetic field strength normalized to field strength at magnetic equator
!	E -> energy in MeV
!
! OUTPUTS:
!	TRARA2 -> Scaled logarithm of integral fluxes
!
! COMMON BLOCKS:
!	COMMON/TRA2/FISTEP
!
! MODIFICATION HISTORY:
!	Written by: D. Bilitza, 1987
!				S. Bourdarie (introduced in version 3.0) - September 2005
!-
!--------------------------------------------------------------------------------------------
      FUNCTION TRARA2(MAP,IL,IB)
C
      IMPLICIT NONE
      INTEGER*4 MAP(1)
      INTEGER*4 IL,IB
      INTEGER*4 ITIME,I1,I2,L1,L2,KT
      INTEGER*4 FLL1,FLL2
      INTEGER*4 J1,J2,FINCR1,FINCR2
      REAL*8 FISTEP,FLOG1,FLOG2
      REAL*8 TRARA2,FNL,FNB
      REAL*8 DFL,FKB1,FKB2,SL1,SL2,FKBJ1
      REAL*8 FLOGM,FKBM,FKBJ2,FLOG,FKB
      COMMON/TRA2/FISTEP
      FNL=IL
      FNB=IB
      ITIME=0
      I2=0
c
C FIND CONSECUTIVE SUB-SUB-MAPS FOR SCALED L-VALUES LS1,LS2,
C WITH IL LESS OR EQUAL LS2.  L1,L2 ARE LENGTHS OF SUB-SUB-MAPS.
C I1,I2 ARE INDECES OF FIRST ELEMENTS MINUS 1.
C
  1   L2=MAP(I2+1)
      IF(MAP(I2+2).GT.IL) GOTO 2
      I1=I2
      L1=L2
      I2=I2+L2
      GOTO 1
  2   CONTINUE
C
C IF SUB-SUB-MAPS ARE EMPTY, I. E. LENGTH LESS 4, THAN TRARA2=0
C
      IF((L1.LT.4).AND.(L2.LT.4)) GOTO 50
C
C IF FLOG2 LESS FLOG1, THAN LS2 FIRST MAP AND LS1 SECOND MAP
C
      IF(MAP(I2+3).GT.MAP(I1+3))  GOTO 10
    5     KT=I1
          I1=I2
          I2=KT
          KT=L1
          L1=L2
          L2=KT
C
C DETERMINE INTERPOLATE IN SCALED L-VALUE
C
  10  FLL1=MAP(I1+2)
      FLL2=MAP(I2+2)
      DFL=(FNL-FLL1)/(FLL2-FLL1)
      FLOG1=MAP(I1+3)
      FLOG2=MAP(I2+3)
      FKB1=0.D0
      FKB2=0.D0
      IF(L1.LT.4) GOTO 32
C
C B/B0 LOOP
C
      DO 17 J2=4,L2
         FINCR2=MAP(I2+J2)
         IF(FKB2+FINCR2.GT.FNB) GOTO 23
         FKB2=FKB2+FINCR2
   17 FLOG2=FLOG2-FISTEP
      ITIME=ITIME+1
      IF(ITIME.EQ.1)GO TO 5
      GO TO 50
   23 IF(ITIME.EQ.1)GO TO 30
      IF(J2.EQ.4)GO TO 28
      SL2=FLOG2/FKB2
      DO 27 J1=4,L1
          FINCR1=MAP(I1+J1)
          FKB1=FKB1+FINCR1
          FLOG1=FLOG1-FISTEP
          FKBJ1=((FLOG1/FISTEP)*FINCR1+FKB1)/((FINCR1/FISTEP)*SL2+1.)
          IF(FKBJ1.LE.FKB1) GOTO 31
   27 CONTINUE
      IF(FKBJ1.LE.FKB2) GOTO 50
   31 IF(FKBJ1.LE.FKB2) GOTO 29
      FKB1=0.D0
   30 FKB2=0.D0
   32 J2=4
      FINCR2=MAP(I2+J2)
      FLOG2=MAP(I2+3)
      FLOG1=MAP(I1+3)
   28 FLOGM=FLOG1+(FLOG2-FLOG1)*DFL
      FKBM=0.D0
      FKB2=FKB2+FINCR2
      FLOG2=FLOG2-FISTEP
      SL2=FLOG2/FKB2
      IF(L1.LT.4) GOTO 35
      J1=4
      FINCR1=MAP(I1+J1)
      FKB1=FKB1+FINCR1
      FLOG1=FLOG1-FISTEP
      SL1=FLOG1/FKB1
      GOTO 15
   29 FKBM=FKBJ1+(FKB2-FKBJ1)*DFL
      FLOGM=FKBM*SL2
      FLOG2=FLOG2-FISTEP
      FKB2=FKB2+FINCR2
      SL1=FLOG1/FKB1
      SL2=FLOG2/FKB2
   15 IF(SL1.LT.SL2) GOTO 20
          FKBJ2=((FLOG2/FISTEP)*FINCR2+FKB2)/((FINCR2/FISTEP)*SL1+1.D0)
          FKB=FKB1+(FKBJ2-FKB1)*DFL
          FLOG=FKB*SL1
          IF(FKB.GE.FNB) GOTO 60
          FKBM=FKB
          FLOGM=FLOG
          IF(J1.GE.L1) GOTO 50
          J1=J1+1
          FINCR1=MAP(I1+J1)
          FLOG1=FLOG1-FISTEP
          FKB1=FKB1+FINCR1
          SL1=FLOG1/FKB1
          GOTO 15
   20 FKBJ1=((FLOG1/FISTEP)*FINCR1+FKB1)/((FINCR1/FISTEP)*SL2+1.D0)
      FKB=FKBJ1+(FKB2-FKBJ1)*DFL
      FLOG=FKB*SL2
      IF(FKB.GE.FNB) GOTO 60
          FKBM=FKB
          FLOGM=FLOG
          IF(J2.GE.L2) GOTO 50
          J2=J2+1
          FINCR2=MAP(I2+J2)
          FLOG2=FLOG2-FISTEP
          FKB2=FKB2+FINCR2
          SL2=FLOG2/FKB2
          GOTO 15
   35     FINCR1=0.D0
          SL1=-900000.D0
          GOTO 20
   60 IF(FKB.LT.FKBM+1.D-10) GOTO 50
      TRARA2=FLOGM+(FLOG-FLOGM)*((FNB-FKBM)/(FKB-FKBM))
      TRARA2=1.D0*MAX(TRARA2,0.D0)
      RETURN
   50 TRARA2=0.D0
      RETURN
      END

!--------------------------------------------------------------------------------------------
!+
! NAME:
!	Init_AE8max

! PURPOSE:
!	This subroutine sets block data for AE8max
!
! CATEGORY:
!	Radiation belt model
!
! CALLING SEQUENCE:
!	call Init_AE8max
!
! INPUTS:
!	None
!
! OUTPUTS:
!	None
!
! COMMON BLOCKS:
!	COMMON /ELEMAX/ IHEADEMAX, MAPEMAX
!
! MODIFICATION HISTORY:
!	Written by: D. Bilitza, 1987
!				S. Bourdarie (introduced in version 3.0) - September 2005
!-
!--------------------------------------------------------------------------------------------
      SUBROUTINE Init_AE8max
!
!      BLOCK DATA AE8MAX

      COMMON /ELEMAX/ IHEADEMAX, MAPEMAX

      INTEGER*4 IHEADEMAX(8)
      INTEGER*4 MAPEMAX(13548)

      DATA IHEADEMAX /7,4,1970,6400,2100,1024,1024,13548/

      DATA (MAPEMAX (I), I=    1,  450) /
     &   1442,   256,     3,     0,     0,     3,  2310,     0,    27,
     &   2520,  5907,    37,    19,    19,    19,    18,    17,    16,
     &     15,    13,    12,    11,    10,     9,     8,     7,     6,
     &      6,     5,     4,     4,     3,     3,     3,     2,    32,
     &   2625,  7393,    72,    33,    33,    32,    30,    28,    26,
     &     24,    22,    19,    17,    16,    14,    12,    11,     9,
     &      8,     7,     6,     6,     5,     4,     4,     3,     2,
     &      3,     2,     2,     2,    34,  2730,  7926,   171,    55,
     &     49,    46,    42,    38,    34,    30,    27,    23,    20,
     &     18,    15,    13,    11,    10,     8,     7,     6,     5,
     &      4,     4,     2,     3,     2,     1,     2,     2,     1,
     &      1,     0,    35,  2835,  8153,   279,    85,    72,    65,
     &     58,    51,    44,    37,    32,    26,    22,    18,    15,
     &     12,    10,     8,     6,     6,     4,     4,     2,     3,
     &      3,     2,     1,     0,     0,     0,     0,     0,     0,
     &      0,    36,  2940,  8339,   389,   123,   105,    93,    80,
     &     67,    55,    45,    36,    29,    23,    18,    14,    11,
     &      9,     7,     5,     4,     4,     2,     1,     1,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,    37,  3045,  8464,   402,   184,   168,   145,   119,
     &     95,    73,    55,    41,    30,    22,    16,    11,    10,
     &      7,     5,     3,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,    37,  3150,  8568,   354,   261,   245,   212,
     &    172,   131,    97,    69,    48,    33,    23,    16,    13,
     &      8,     4,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,    38,  3360,  8720,   381,   340,   342,
     &    313,   265,   211,   159,   115,    81,    56,    38,    26,
     &     21,    14,     8,     5,     3,     1,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,    38,  3570,  8812,   444,
     &    421,   445,   426,   374,   305,   235,   172,   122,    85,
     &     58,    38,    28,    20,    13,     8,     4,     2,     1,
     &      1,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,    38,  3780,
     &   8842,   442,   490,   547,   552,   506,   427,   336,   251,
     &    179,   124,    85,    57,    38,    31,    20,    12,     7,
     &      4,     2,     1,     1,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &     38,  3990,  8814,   413,   559,   657,   695,   662,   574,
     &    461,   348,   251,   175,   119,    79,    52,    43,    27,
     &     17,     9,     5,     3,     2,     1,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,    38,  4200,  8766,   468,   664,   807,   874,
     &    842,   732,   584,   435,   308,   212,   140,    93,    61,
     &     47,    30,    17,    10,     5,     3,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,    37,  4620,  8669,   530,   795,
     &   1037,  1203,  1239,  1136,   944,   723,   522,   361,   241/
      DATA (MAPEMAX (I), I=  451,  900) /
     &    159,   101,    78,    51,    31,    17,    10,     6,     3,
     &      2,     1,     1,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,    37,  5039,  8562,  1331,
     &    307,   410,   512,   614,   614,   819,   819,   819,   922,
     &    819,   819,   819,   717,   614,   614,   512,   410,   307,
     &    350,   255,   210,   174,   141,   117,    94,    77,    62,
     &     51,    41,    32,    27,    21,     8,    37,  5250,  8477,
     &   1331,   410,   512,   614,   717,   819,   922,  1024,  1024,
     &   1126,  1126,  1024,  1024,   819,   819,   614,   614,   512,
     &    307,   373,   269,   219,   177,   142,   115,    91,    74,
     &     58,    46,    38,    29,    24,    18,     2,    36,  5460,
     &   8386,  1331,   512,   614,   717,   922,  1024,  1229,  1331,
     &   1434,  1434,  1331,  1331,  1229,  1024,   922,   717,   614,
     &    512,   410,   374,   270,   215,   170,   133,   105,    82,
     &     64,    51,    39,    31,    24,    19,    11,    36,  5669,
     &   8284,  1434,   512,   717,   922,  1229,  1434,  1536,  1843,
     &   1843,  1843,  1741,  1638,  1434,  1229,  1024,   819,   717,
     &    512,   406,   329,   256,   198,   151,   118,    90,    69,
     &     52,    41,    31,    23,    18,    13,     4,    35,  5880,
     &   8114,  1536,   614,   922,  1126,  1536,  1946,  2253,  2355,
     &   2560,  2458,  2253,  2048,  1741,  1434,  1229,   819,   819,
     &    512,   417,   322,   244,   184,   138,   103,    78,    58,
     &     44,    32,    24,    18,    13,     7,    34,  6089,  7856,
     &   1536,   819,  1229,  1638,  2150,  2662,  3072,  3379,  3379,
     &   3072,  2765,  2355,  1843,  1536,  1126,   819,   717,   443,
     &    351,   257,   189,   136,   100,    73,    52,    39,    27,
     &     19,    14,    10,     6,    34,  6300,  7806,  1638,  1024,
     &   1638,  2253,  3072,  3789,  4301,  4506,  4198,  3789,  3072,
     &   2458,  1843,  1434,   922,   819,   489,   375,   265,   188,
     &    132,    92,    65,    47,    31,    22,    16,    11,     7,
     &      6,     2,    34,  6719,  7856,  2253,  2355,  4198,  6246,
     &   7680,  7680,  6451,  4915,  3482,  2355,  1434,  1024,   600,
     &    388,   243,   151,    93,    59,    37,    23,    14,     8,
     &      6,     3,     2,     1,     1,     0,     0,     1,     0,
     &     34,  7139,  7909,  2355,  2867,  5222,  8090,  9830,  9523,
     &   7885,  5734,  3994,  2560,  1536,  1024,   664,   386,   236,
     &    144,    88,    54,    33,    20,    12,     7,     4,     3,
     &      2,     1,     1,     0,     0,     0,     0,    35,  7350,
     &   7942,  2458,  2970,  5734,  8909, 10957, 10547,  8602,  6349,
     &   4198,  2765,  1741,  1126,   626,   403,   245,   148,    90,
     &     55,    33,    20,    12,     7,     4,     3,     2,     1,
     &      1,     0,     0,     0,     0,     0,    35,  7560,  7964,
     &   2458,  3174,  6246,  9830, 12186, 11674,  9421,  6861,  4608,
     &   2970,  1741,  1126,   736,   417,   251,   152,    91,    55,
     &     33,    20,    12,     7,     4,     3,     1,     1,     1,
     &      0,     0,     0,     0,     0,    35,  7980,  8000,  2560,
     &   3379,  7066, 11366, 14438, 14234, 11469,  8294,  5530,  3482,
     &   2253,  1331,   782,   486,   292,   174,   105,    62,    37,
     &     22,    14,     8,     4,     3,     2,     1,     1,     0,
     &      0,     0,     0,     0,    35,  8400,  8045,  2560,  3789/
      DATA (MAPEMAX (I), I=  901, 1350) /
     &   7782, 13210, 17101, 16896, 13824,  9933,  6554,  4198,  2560,
     &   1536,   922,   568,   332,   197,   117,    70,    41,    24,
     &     15,     8,     5,     3,     2,     1,     1,     0,     0,
     &      0,     0,     0,    35,  8820,  8069,  2560,  3891,  8090,
     &  14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,  1946,
     &   1229,   688,   418,   249,   147,    88,    52,    30,    19,
     &     10,     7,     4,     2,     1,     1,     1,     0,     0,
     &      0,     0,    35,  9239,  8092,  2662,  3789,  8499, 15462,
     &  21709, 23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,
     &    819,   528,   310,   184,   109,    65,    38,    23,    13,
     &      8,     5,     3,     1,     1,     1,     0,     0,     0,
     &      0,    35,  9450,  8098,  2662,  3891,  8602, 15974, 22938,
     &  24986, 21811, 16282, 11059,  7168,  4403,  2662,  1638,  1024,
     &    535,   345,   205,   121,    72,    43,    25,    15,     9,
     &      5,     3,     2,     1,     1,     0,     0,     0,     0,
     &     35,  9660,  8109,  2662,  3891,  8704, 16486, 24166, 26829,
     &  23654, 17818, 12186,  7885,  4915,  2970,  1741,  1126,   622,
     &    382,   227,   135,    80,    47,    28,    17,    10,     5,
     &      4,     2,     1,     1,     0,     0,     0,     0,    35,
     &  10079,  8109,  2662,  3891,  9011, 17408, 26522, 30515, 27750,
     &  21197, 14746,  9523,  5939,  3584,  2253,  1229,   834,   467,
     &    277,   165,    97,    58,    34,    21,    12,     7,     4,
     &      3,     1,     1,     0,     0,     0,     1,    35, 10500,
     &   8098,  2662,  3994,  9216, 18227, 28877, 34509, 32051, 25088,
     &  17510, 11469,  7168,  4403,  2662,  1536,  1024,   537,   336,
     &    199,   118,    70,    42,    25,    14,     9,     5,     3,
     &      2,     1,     0,     0,     1,     0,    35, 11550,  8039,
     &   2662,  4096,  9626, 20275, 34611, 44954, 44954, 36762, 26419,
     &  17510, 11162,  6861,  4096,  2560,  1434,   922,   510,   313,
     &    186,   110,    66,    39,    22,    14,     8,     5,     3,
     &      1,     1,     1,     0,     0,    35, 12600,  7964,  2662,
     &   4096, 10035, 21811, 39936, 56115, 59904, 51507, 38093, 25805,
     &  16486, 10240,  6246,  3687,  2253,  1331,   822,   472,   280,
     &    166,    99,    58,    35,    20,    13,     7,     4,     3,
     &      1,     0,     1,     0,    34, 13860,  7846,  2662,  4198,
     &  10240, 23347, 45670, 69837, 80998, 73728, 56730, 39322, 25498,
     &  15974,  9728,  5939,  3482,  2151,  1229,   728,   442,   262,
     &    155,    92,    55,    32,    19,    11,     7,     4,     2,
     &      1,     1,    34, 14700,  7761,  2662,  4198, 10445, 24166,
     &  48845, 78746, 95846, 90829, 71783, 50483, 33075, 20787, 12698,
     &   7783,  4608,  2765,  1639,  1024,   550,   344,   204,   121,
     &     72,    42,    25,    15,     9,     5,     3,     2,     0,
     &     33, 15750,  7671,  2662,  4198, 10547, 25088, 52429, 89395,
     & 115610,115098, 94003, 67482, 44749, 28365, 17408, 10650,  6349,
     &   3789,  2253,  1434,   748,   474,   281,   167,    98,    59,
     &     35,    20,    12,     7,     4,     3,    33, 16800,  7557,
     &   2662,  4198, 10650, 25702, 55706, 99430,136192,142438,120320,
     &  88269, 59290, 37786, 23450, 14234,  8602,  5120,  3072,  1741,
     &   1126,   636,   379,   225,   133,    79,    47,    27,    16,
     &     10,     6,     2,    32, 18900,  7301,  2765,  4915, 12902/
      DATA (MAPEMAX (I), I= 1351, 1800) /
     &  32870, 75674,142131,200295,210227,175719,126567, 83558, 52429,
     &  31847, 19047, 11264,  6656,  3891,  2253,  1331,   794,   455,
     &    266,   156,    91,    53,    30,    18,    11,     3,    31,
     &  21000,  6996,  3789, 10035, 34918,106701,239616,342323,323686,
     & 235725,148992, 87655, 49562, 27546, 15053,  8294,  4404,  2458,
     &   1331,   725,   389,   211,   114,    62,    33,    18,    10,
     &      5,     3,     1,    29, 23100,  6585, 11162, 95334,448410,
     & 664474,473600,256819,126362, 59801, 27853, 12902,  5940,  2764,
     &   1229,   572,   265,   122,    56,    25,    12,     5,     3,
     &      1,     0,     0,     0,     0,     3, 25200,     0,     3,
     &  32767,     0,  1391,   640,     3,     0,     0,     3,  2310,
     &      0,    26,  2520,  5801,    37,    19,    19,    18,    18,
     &     17,    16,    15,    13,    12,    11,    10,     9,     8,
     &      7,     6,     6,     5,     4,     4,     4,     3,     3,
     &     32,  2625,  7285,    72,    33,    33,    32,    30,    28,
     &     26,    24,    22,    20,    17,    16,    14,    12,    11,
     &      9,     8,     7,     6,     5,     5,     4,     4,     3,
     &      2,     3,     2,     2,     2,    34,  2730,  7818,   171,
     &     54,    48,    45,    42,    38,    34,    30,    27,    23,
     &     20,    18,    15,    13,    11,    10,     8,     7,     6,
     &      5,     4,     4,     2,     3,     2,     1,     2,     2,
     &      1,     1,     0,    35,  2835,  8028,   278,    84,    71,
     &     65,    58,    51,    44,    37,    32,    27,    22,    19,
     &     15,    12,    11,     9,     7,     6,     4,     4,     2,
     &      3,     3,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,    36,  2940,  8206,   387,   121,   104,    92,
     &     80,    67,    56,    45,    37,    29,    23,    19,    14,
     &     11,     9,     7,     6,     4,     4,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,    36,  3045,  8332,   398,   179,   165,   144,
     &    120,    96,    75,    57,    43,    32,    23,    17,    12,
     &     11,     7,     5,     3,     2,     1,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,    36,  3150,  8428,   344,   250,   239,   211,
     &    174,   136,   102,    74,    52,    36,    24,    16,    15,
     &      9,     5,     3,     2,     1,     1,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,    37,  3360,  8572,   365,   321,   327,   307,
     &    268,   218,   169,   124,    88,    63,    43,    29,    21,
     &     16,    10,     6,     3,     2,     1,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,    37,  3570,  8667,   421,   393,   420,
     &    413,   374,   315,   249,   188,   136,    95,    66,    44,
     &     30,    24,    16,     9,     5,     3,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,    37,  3780,  8695,   413,   450,
     &    507,   525,   499,   438,   357,   275,   202,   143,    99,
     &     67,    44,    35,    23,    15,     8,     5,     3,     1,
     &      1,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,    37,  3990,  8656,   378/
      DATA (MAPEMAX (I), I= 1801, 2250) /
     &    505,   598,   649,   643,   584,   491,   385,   285,   205,
     &    141,    96,    64,    46,    33,    22,    12,     7,     4,
     &      2,     1,     1,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,    37,  4200,  8608,
     &    424,   591,   723,   806,   814,   746,   627,   489,   358,
     &    253,   171,   114,    74,    55,    37,    23,    13,     7,
     &      4,     2,     1,     1,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,    37,  4620,
     &   8507,   468,   684,   886,  1055,  1145,  1125,  1005,   824,
     &    626,   452,   312,   209,   137,    90,    70,    44,    26,
     &     15,     8,     5,     3,     1,     1,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,    36,
     &   5039,  8395,  1331,   307,   410,   512,   614,   614,   819,
     &    819,   819,   922,   819,   819,   819,   717,   614,   614,
     &    512,   410,   307,   350,   255,   210,   174,   141,   117,
     &     94,    77,    62,    51,    40,    33,    27,    18,    36,
     &   5250,  8301,  1331,   410,   512,   614,   717,   819,   922,
     &   1024,  1024,  1126,  1126,  1024,  1024,   819,   819,   614,
     &    614,   512,   307,   373,   269,   219,   177,   142,   115,
     &     91,    74,    58,    46,    37,    30,    24,     8,    36,
     &   5460,  8196,  1331,   512,   614,   717,   922,  1024,  1229,
     &   1331,  1434,  1434,  1331,  1331,  1229,  1024,   922,   717,
     &    614,   512,   410,   374,   270,   215,   170,   133,   105,
     &     82,    64,    51,    39,    31,    24,    19,     0,    35,
     &   5669,  8064,  1434,   512,   717,   922,  1229,  1434,  1536,
     &   1843,  1843,  1843,  1741,  1638,  1434,  1229,  1024,   819,
     &    717,   512,   406,   329,   256,   198,   151,   118,    90,
     &     69,    52,    41,    30,    24,    18,     7,    34,  5880,
     &   7816,  1536,   614,   922,  1126,  1536,  1946,  2253,  2355,
     &   2560,  2458,  2253,  2048,  1741,  1434,  1229,   819,   819,
     &    512,   417,   322,   244,   184,   138,   103,    78,    58,
     &     44,    32,    24,    18,     8,    33,  6089,  7497,  1536,
     &    819,  1229,  1638,  2150,  2662,  3072,  3379,  3379,  3072,
     &   2765,  2355,  1843,  1536,  1126,   819,   717,   443,   351,
     &    257,   189,   136,   100,    73,    52,    39,    26,    20,
     &     14,     4,    33,  6300,  7518,  1638,  1024,  1638,  2253,
     &   3072,  3789,  4301,  4506,  4198,  3789,  3072,  2458,  1843,
     &   1434,   922,   819,   489,   375,   265,   188,   132,    92,
     &     65,    47,    31,    22,    16,    11,     7,     2,    33,
     &   6719,  7601,  2253,  2355,  4198,  6246,  7680,  7680,  6451,
     &   4915,  3482,  2355,  1434,  1024,   600,   388,   243,   151,
     &     93,    59,    37,    23,    14,     8,     6,     3,     2,
     &      1,     1,     0,     0,     0,    33,  7139,  7678,  2355,
     &   2867,  5222,  8090,  9830,  9523,  7885,  5734,  3994,  2560,
     &   1536,  1024,   664,   386,   236,   144,    88,    54,    33,
     &     20,    12,     7,     4,     3,     2,     1,     0,     0,
     &      1,     0,    34,  7350,  7712,  2458,  2970,  5734,  8909,
     &  10957, 10547,  8602,  6349,  4198,  2765,  1741,  1126,   626,
     &    403,   245,   148,    90,    55,    33,    20,    12,     7,
     &      4,     3,     2,     1,     1,     0,     0,     0,     0/
      DATA (MAPEMAX (I), I= 2251, 2700) /
     &     34,  7560,  7749,  2458,  3174,  6246,  9830, 12186, 11674,
     &   9421,  6861,  4608,  2970,  1741,  1126,   736,   417,   251,
     &    152,    91,    55,    33,    20,    12,     7,     4,     3,
     &      1,     1,     1,     0,     0,     0,     0,    34,  7980,
     &   7784,  2560,  3379,  7066, 11366, 14438, 14234, 11469,  8294,
     &   5530,  3482,  2253,  1331,   782,   486,   292,   174,   105,
     &     62,    37,    22,    14,     8,     4,     3,     2,     1,
     &      1,     0,     0,     0,     0,    34,  8400,  7826,  2560,
     &   3789,  7782, 13210, 17101, 16896, 13824,  9933,  6554,  4198,
     &   2560,  1536,   922,   568,   332,   197,   117,    70,    41,
     &     24,    15,     8,     5,     3,     2,     1,     1,     0,
     &      0,     0,     0,    34,  8820,  7865,  2560,  3891,  8090,
     &  14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,  1946,
     &   1229,   688,   418,   249,   147,    88,    52,    30,    19,
     &     10,     7,     4,     2,     1,     1,     0,     0,     0,
     &      1,    34,  9239,  7892,  2662,  3789,  8499, 15462, 21709,
     &  23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,   819,
     &    528,   310,   184,   109,    65,    38,    23,    13,     8,
     &      5,     3,     1,     1,     1,     0,     0,     0,    34,
     &   9450,  7901,  2662,  3891,  8602, 15974, 22938, 24986, 21811,
     &  16282, 11059,  7168,  4403,  2662,  1638,  1024,   535,   345,
     &    205,   121,    72,    43,    25,    15,     9,     5,     3,
     &      2,     1,     0,     1,     0,     0,    34,  9660,  7909,
     &   2662,  3891,  8704, 16486, 24166, 26829, 23654, 17818, 12186,
     &   7885,  4915,  2970,  1741,  1126,   622,   382,   227,   135,
     &     80,    47,    28,    17,    10,     5,     4,     2,     1,
     &      0,     1,     0,     0,    34, 10079,  7917,  2662,  3891,
     &   9011, 17408, 26522, 30515, 27750, 21197, 14746,  9523,  5939,
     &   3584,  2253,  1229,   834,   467,   277,   165,    97,    58,
     &     34,    21,    12,     7,     4,     3,     1,     0,     1,
     &      0,     0,    34, 10500,  7917,  2662,  3994,  9216, 18227,
     &  28877, 34509, 32051, 25088, 17510, 11469,  7168,  4403,  2662,
     &   1536,  1024,   537,   336,   199,   118,    70,    42,    25,
     &     14,     9,     5,     3,     2,     0,     1,     0,     1,
     &     34, 11550,  7865,  2662,  4096,  9626, 20275, 34611, 44954,
     &  44954, 36762, 26419, 17510, 11162,  6861,  4096,  2560,  1434,
     &    922,   510,   313,   186,   110,    66,    39,    22,    14,
     &      8,     5,     3,     1,     1,     1,     0,    34, 12600,
     &   7816,  2662,  4096, 10035, 21811, 39936, 56115, 59904, 51507,
     &  38093, 25805, 16486, 10240,  6246,  3687,  2253,  1331,   822,
     &    472,   280,   166,    99,    58,    35,    20,    13,     7,
     &      4,     2,     2,     0,     1,    33, 13860,  7678,  2662,
     &   4198, 10240, 23347, 45670, 69837, 80998, 73728, 56730, 39322,
     &  25498, 15974,  9728,  5939,  3482,  2151,  1229,   728,   442,
     &    262,   155,    92,    55,    32,    19,    11,     7,     4,
     &      2,     1,    33, 14700,  7557,  2662,  4198, 10445, 24166,
     &  48845, 78746, 95846, 90829, 71783, 50483, 33075, 20787, 12698,
     &   7783,  4608,  2765,  1639,  1024,   550,   344,   204,   121,
     &     72,    42,    25,    15,     8,     6,     3,     1,    32,
     &  15750,  7371,  2662,  4198, 10547, 25088, 52429, 89395,115610/
      DATA (MAPEMAX (I), I= 2701, 3150) /
     & 115098, 94003, 67482, 44749, 28365, 17408, 10650,  6349,  3789,
     &   2253,  1434,   748,   474,   281,   167,    98,    59,    35,
     &     20,    12,     7,     4,    31, 16800,  7145,  2662,  4198,
     &  10650, 25702, 55706, 99430,136192,142438,120320, 88269, 59290,
     &  37786, 23450, 14234,  8602,  5120,  3072,  1741,  1126,   636,
     &    379,   225,   133,    79,    46,    28,    16,     9,    29,
     &  18900,  6523,  2765,  4915, 12902, 32870, 75674,142131,200295,
     & 210227,175719,126567, 83558, 52429, 31847, 19047, 11264,  6656,
     &   3891,  2253,  1331,   794,   455,   266,   155,    91,    53,
     &     17,    25, 21000,  5577,  3789, 10035, 34918,106701,239616,
     & 342323,323686,235725,148992, 87655, 49562, 27546, 15053,  8294,
     &   4404,  2458,  1331,   725,   389,   210,   114,    52,    20,
     &  23100,  4177, 11162, 95334,448410,664474,473600,256819,126362,
     &  59801, 27853, 12902,  5940,  2764,  1229,   572,   265,   121,
     &     23,     3, 25200,     0,     3, 32767,     0,  1307,  1600,
     &      3,     0,     0,     3,  2310,     0,    25,  2520,  5498,
     &     37,    18,    19,    18,    18,    17,    16,    14,    13,
     &     12,    11,    10,     9,     8,     7,     7,     6,     5,
     &      5,     4,     4,     3,    31,  2625,  6948,    72,    32,
     &     32,    31,    30,    28,    26,    24,    22,    20,    18,
     &     16,    14,    12,    11,    10,     9,     7,     7,     6,
     &      5,     4,     4,     3,     3,     3,     2,     2,    33,
     &   2730,  7377,   170,    53,    47,    44,    41,    37,    34,
     &     30,    27,    24,    21,    18,    16,    14,    12,    10,
     &      9,     7,     6,     5,     5,     4,     3,     3,     2,
     &      1,     1,     3,     1,     1,    34,  2835,  7584,   276,
     &     82,    68,    63,    57,    50,    44,    38,    33,    28,
     &     23,    19,    16,    13,    11,     9,     8,     6,     4,
     &      5,     3,     2,     3,     2,     1,     1,     0,     0,
     &      0,     0,     0,    34,  2940,  7737,   382,   117,   100,
     &     90,    79,    68,    57,    47,    39,    31,    25,    20,
     &     16,    12,    10,     7,     7,     4,     5,     3,     2,
     &      1,     1,     0,     0,     0,     0,     0,     0,     0,
     &      0,    35,  3045,  7856,   387,   165,   156,   140,   121,
     &    100,    80,    63,    48,    36,    27,    20,    14,    11,
     &      8,     6,     4,     2,     1,     1,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &     35,  3150,  7959,   317,   220,   218,   203,   178,   147,
     &    116,    88,    64,    45,    31,    22,    16,    12,     8,
     &      4,     2,     1,     1,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,    35,
     &   3360,  8092,   324,   270,   285,   283,   264,   232,   192,
     &    151,   113,    83,    58,    41,    27,    21,    15,     9,
     &      5,     3,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,    35,  3570,
     &   8189,   368,   321,   351,   364,   356,   326,   282,   230,
     &    177,   132,    94,    66,    45,    30,    25,    16,    10,
     &      5,     3,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,    36,  3780,  8218,
     &    347,   355,   405,   438,   449,   434,   394,   335,   270/
      DATA (MAPEMAX (I), I= 3151, 3600) /
     &    205,   150,   106,    73,    49,    36,    26,    17,    10,
     &      5,     3,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,    35,  3990,  8156,
     &    300,   384,   456,   514,   549,   554,   523,   463,   384,
     &    301,   224,   160,   111,    75,    49,    41,    26,    16,
     &      9,     5,     3,     2,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,    35,  4200,  8066,   328,
     &    437,   531,   616,   673,   694,   667,   598,   498,   389,
     &    288,   204,   138,    93,    61,    49,    31,    18,    10,
     &      6,     3,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,    34,  4620,  7864,   348,   470,
     &    591,   717,   829,   913,   951,   927,   844,   713,   564,
     &    421,   298,   205,   135,    89,    70,    45,    26,    15,
     &      8,     5,     3,     1,     1,     0,     0,     0,     0,
     &      0,     0,    34,  5039,  7685,  1331,   307,   410,   512,
     &    614,   614,   819,   819,   819,   922,   819,   819,   819,
     &    717,   614,   614,   512,   410,   307,   350,   255,   210,
     &    174,   141,   117,    94,    77,    62,    50,    41,     1,
     &     33,  5250,  7547,  1331,   410,   512,   614,   717,   819,
     &    922,  1024,  1024,  1126,  1126,  1024,  1024,   819,   819,
     &    614,   614,   512,   307,   373,   269,   219,   177,   142,
     &    115,    91,    73,    58,    47,    18,    32,  5460,  7390,
     &   1331,   512,   614,   717,   922,  1024,  1229,  1331,  1434,
     &   1434,  1331,  1331,  1229,  1024,   922,   717,   614,   512,
     &    410,   374,   270,   215,   170,   133,   105,    82,    64,
     &     50,    35,    32,  5669,  7210,  1434,   512,   717,   922,
     &   1229,  1434,  1536,  1843,  1843,  1843,  1741,  1638,  1434,
     &   1229,  1024,   819,   717,   512,   406,   329,   256,   198,
     &    151,   118,    90,    68,    53,    40,     6,    31,  5880,
     &   7040,  1536,   614,   922,  1126,  1536,  1946,  2253,  2355,
     &   2560,  2458,  2253,  2048,  1741,  1434,  1229,   819,   819,
     &    512,   417,   322,   244,   184,   138,   103,    78,    58,
     &     43,    17,    31,  6089,  6996,  1536,   819,  1229,  1638,
     &   2150,  2662,  3072,  3379,  3379,  3072,  2765,  2355,  1843,
     &   1536,  1126,   819,   717,   443,   351,   257,   189,   136,
     &    100,    73,    52,    38,    27,     7,    31,  6300,  7034,
     &   1638,  1024,  1638,  2253,  3072,  3789,  4301,  4506,  4198,
     &   3789,  3072,  2458,  1843,  1434,   922,   819,   489,   375,
     &    265,   188,   132,    92,    65,    47,    31,    22,    16,
     &      5,    31,  6719,  7140,  2253,  2355,  4198,  6246,  7680,
     &   7680,  6451,  4915,  3482,  2355,  1434,  1024,   600,   388,
     &    243,   151,    93,    59,    37,    23,    14,     8,     6,
     &      3,     1,     2,     1,     0,    32,  7139,  7237,  2355,
     &   2867,  5222,  8090,  9830,  9523,  7885,  5734,  3994,  2560,
     &   1536,  1024,   664,   386,   236,   144,    88,    54,    33,
     &     20,    12,     7,     4,     3,     2,     0,     1,     0,
     &      0,    32,  7350,  7267,  2458,  2970,  5734,  8909, 10957,
     &  10547,  8602,  6349,  4198,  2765,  1741,  1126,   626,   403,
     &    245,   148,    90,    55,    33,    20,    12,     7,     4,
     &      3,     2,     0,     1,     0,     0,    32,  7560,  7333/
      DATA (MAPEMAX (I), I= 3601, 4050) /
     &   2458,  3174,  6246,  9830, 12186, 11674,  9421,  6861,  4608,
     &   2970,  1741,  1126,   736,   417,   251,   152,    91,    55,
     &     33,    20,    12,     7,     4,     3,     1,     1,     0,
     &      1,     0,    32,  7980,  7403,  2560,  3379,  7066, 11366,
     &  14438, 14234, 11469,  8294,  5530,  3482,  2253,  1331,   782,
     &    486,   292,   174,   105,    62,    37,    22,    14,     8,
     &      4,     3,     2,     0,     1,     0,     1,    33,  8400,
     &   7487,  2560,  3789,  7782, 13210, 17101, 16896, 13824,  9933,
     &   6554,  4198,  2560,  1536,   922,   568,   332,   197,   117,
     &     70,    41,    24,    15,     8,     5,     3,     2,     1,
     &      0,     1,     0,     0,    33,  8820,  7538,  2560,  3891,
     &   8090, 14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,
     &   1946,  1229,   688,   418,   249,   147,    88,    52,    30,
     &     19,    10,     7,     4,     2,     1,     1,     0,     0,
     &      0,    33,  9239,  7566,  2662,  3789,  8499, 15462, 21709,
     &  23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,   819,
     &    528,   310,   184,   109,    65,    38,    23,    13,     8,
     &      5,     3,     1,     1,     1,     0,     0,    33,  9450,
     &   7575,  2662,  3891,  8602, 15974, 22938, 24986, 21811, 16282,
     &  11059,  7168,  4403,  2662,  1638,  1024,   535,   345,   205,
     &    121,    72,    43,    25,    15,     9,     5,     3,     2,
     &      1,     0,     1,     0,    33,  9660,  7601,  2662,  3891,
     &   8704, 16486, 24166, 26829, 23654, 17818, 12186,  7885,  4915,
     &   2970,  1741,  1126,   622,   382,   227,   135,    80,    47,
     &     28,    17,    10,     5,     4,     2,     1,     0,     1,
     &      0,    33, 10079,  7601,  2662,  3891,  9011, 17408, 26522,
     &  30515, 27750, 21197, 14746,  9523,  5939,  3584,  2253,  1229,
     &    834,   467,   277,   165,    97,    58,    34,    21,    12,
     &      7,     4,     3,     1,     0,     1,     0,    33, 10500,
     &   7575,  2662,  3994,  9216, 18227, 28877, 34509, 32051, 25088,
     &  17510, 11469,  7168,  4403,  2662,  1536,  1024,   537,   336,
     &    199,   118,    70,    42,    25,    14,     9,     5,     3,
     &      1,     1,     1,     0,    33, 11550,  7508,  2662,  4096,
     &   9626, 20275, 34611, 44954, 44954, 36762, 26419, 17510, 11162,
     &   6861,  4096,  2560,  1434,   922,   510,   313,   186,   110,
     &     66,    39,    22,    14,     8,     5,     2,     2,     1,
     &      0,    32, 12600,  7398,  2662,  4096, 10035, 21811, 39936,
     &  56115, 59904, 51507, 38093, 25805, 16486, 10240,  6246,  3687,
     &   2253,  1331,   822,   472,   280,   166,    99,    58,    35,
     &     20,    13,     6,     5,     2,     1,    32, 13860,  7210,
     &   2662,  4198, 10240, 23347, 45670, 69837, 80998, 73728, 56730,
     &  39322, 25498, 15974,  9728,  5939,  3482,  2151,  1229,   728,
     &    442,   262,   155,    92,    55,    32,    19,    11,     7,
     &      4,     0,    31, 14700,  7051,  2662,  4198, 10445, 24166,
     &  48845, 78746, 95846, 90829, 71783, 50483, 33075, 20787, 12698,
     &   7783,  4608,  2765,  1639,  1024,   550,   344,   204,   121,
     &     72,    42,    25,    15,     8,     4,    30, 15750,  6812,
     &   2662,  4198, 10547, 25088, 52429, 89395,115610,115098, 94003,
     &  67482, 44749, 28365, 17408, 10650,  6349,  3789,  2253,  1434,
     &    748,   474,   281,   167,    98,    58,    35,    21,     8/
      DATA (MAPEMAX (I), I= 4051, 4500) /
     &     29, 16800,  6484,  2662,  4198, 10650, 25702, 55706, 99430,
     & 136192,142438,120320, 88269, 59290, 37786, 23450, 14234,  8602,
     &   5120,  3072,  1741,  1126,   636,   379,   225,   133,    78,
     &     47,    11,    25, 18900,  5608,  2765,  4915, 12902, 32870,
     &  75674,142131,200295,210227,175719,126567, 83558, 52429, 31847,
     &  19047, 11264,  6656,  3891,  2253,  1331,   793,   456,   247,
     &     20, 21000,  4229,  3789, 10035, 34918,106701,239616,342323,
     & 323686,235725,148992, 87655, 49562, 27546, 15053,  8192,  4506,
     &   2458,   717,    10, 23100,  1617, 11162, 95334,448410,664474,
     & 473498,256820, 50790,     3, 25200,     0,     3, 32767,     0,
     &   1200,  3200,     3,     0,     0,     3,  2310,     0,    24,
     &   2520,  4305,    37,    18,    19,    18,    17,    17,    16,
     &     14,    13,    12,    11,    10,     9,     8,     7,     7,
     &      6,     5,     5,     4,     4,    30,  2625,  5677,    71,
     &     32,    32,    31,    30,    28,    26,    24,    22,    20,
     &     18,    16,    14,    13,    11,    10,     9,     7,     7,
     &      6,     5,     4,     4,     3,     3,     3,     2,    32,
     &   2730,  6186,   169,    53,    47,    44,    41,    37,    34,
     &     31,    27,    24,    21,    18,    16,    14,    12,    10,
     &      9,     7,     6,     5,     5,     4,     3,     3,     3,
     &      1,     1,     3,     2,    32,  2835,  6494,   275,    82,
     &     68,    63,    57,    50,    44,    38,    33,    28,    23,
     &     20,    16,    14,    11,     9,     8,     6,     5,     5,
     &      3,     2,     4,     2,     1,     1,     0,     0,     0,
     &     32,  2940,  6686,   381,   116,    98,    89,    79,    68,
     &     57,    48,    39,    31,    25,    20,    16,    13,    10,
     &      7,     7,     4,     5,     3,     2,     1,     1,     0,
     &      0,     0,     0,     0,     0,    32,  3045,  6802,   385,
     &    161,   153,   139,   121,   101,    82,    64,    49,    37,
     &     28,    21,    15,    11,     9,     6,     4,     2,     1,
     &      1,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,    32,  3150,  6902,   311,   212,   212,   200,   178,
     &    149,   119,    91,    67,    48,    34,    23,    16,    14,
     &      8,     5,     3,     2,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,    32,  3360,  7045,
     &    315,   258,   273,   274,   260,   233,   197,   158,   121,
     &     89,    64,    44,    30,    22,    17,    11,     6,     3,
     &      2,     1,     1,     0,     0,     0,     0,     0,     0,
     &      0,     0,    31,  3570,  7130,   356,   304,   334,   349,
     &    346,   325,   287,   239,   189,   143,   103,    74,    50,
     &     34,    27,    18,    12,     7,     4,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,    32,  3780,  7195,
     &    332,   334,   380,   415,   431,   425,   395,   346,   286,
     &    223,   167,   120,    84,    57,    38,    32,    20,    12,
     &      7,     4,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,    32,  3990,  7173,   283,   359,   424,   480,
     &    519,   533,   518,   472,   405,   327,   250,   183,   128,
     &     89,    59,    44,    31,    20,    11,     6,     4,     2,
     &      1,     1,     0,     0,     0,     0,     0,    31,  4200,
     &   7052,   308,   404,   491,   569,   630,   663,   656,   608/
      DATA (MAPEMAX (I), I= 4501, 4950) /
     &    526,   427,   325,   236,   164,   111,    73,    55,    37,
     &     23,    13,     7,     4,     2,     1,     1,     0,     0,
     &      0,     0,    30,  4620,  6798,   324,   429,   538,   646,
     &    751,   840,   896,   905,   858,   762,   631,   489,   362,
     &    253,   172,   113,    82,    57,    37,    21,    12,     7,
     &      4,     2,     1,     1,     0,    30,  5039,  6749,  1331,
     &    307,   410,   512,   614,   614,   819,   819,   819,   922,
     &    819,   819,   819,   717,   614,   614,   512,   410,   307,
     &    350,   255,   210,   174,   141,   116,    95,    30,    30,
     &   5250,  6681,  1331,   410,   512,   614,   717,   819,   922,
     &   1024,  1024,  1126,  1126,  1024,  1024,   819,   819,   614,
     &    614,   512,   307,   373,   269,   219,   177,   141,   115,
     &     92,     8,    29,  5459,  6625,  1331,   512,   614,   717,
     &    922,  1024,  1229,  1331,  1434,  1434,  1331,  1331,  1229,
     &   1024,   922,   717,   614,   512,   410,   374,   270,   215,
     &    169,   134,   104,    74,    29,  5669,  6574,  1434,   512,
     &    717,   922,  1229,  1434,  1536,  1843,  1843,  1843,  1741,
     &   1638,  1434,  1229,  1024,   819,   717,   512,   406,   329,
     &    256,   198,   151,   117,    90,    49,    29,  5879,  6551,
     &   1536,   614,   922,  1126,  1536,  1946,  2253,  2355,  2560,
     &   2458,  2253,  2048,  1741,  1434,  1229,   819,   819,   512,
     &    417,   322,   244,   184,   138,   103,    78,    36,    29,
     &   6089,  6577,  1536,   819,  1229,  1638,  2150,  2662,  3072,
     &   3379,  3379,  3072,  2765,  2355,  1843,  1536,  1126,   819,
     &    717,   443,   351,   257,   189,   136,   100,    72,    53,
     &     27,    29,  6300,  6632,  1638,  1024,  1638,  2253,  3072,
     &   3789,  4301,  4506,  4198,  3789,  3072,  2458,  1843,  1434,
     &    922,   819,   489,   375,   265,   188,   132,    92,    65,
     &     46,    33,    20,    30,  6719,  6771,  2253,  2355,  4198,
     &   6246,  7680,  7680,  6451,  4915,  3482,  2355,  1434,  1024,
     &    600,   388,   243,   151,    93,    59,    37,    23,    14,
     &      8,     6,     2,     2,     2,     0,    30,  7139,  6877,
     &   2355,  2867,  5222,  8090,  9830,  9523,  7885,  5734,  3994,
     &   2560,  1536,  1024,   664,   386,   236,   144,    88,    54,
     &     33,    20,    12,     7,     4,     3,     1,     1,     1,
     &     31,  7350,  6933,  2458,  2970,  5734,  8909, 10957, 10547,
     &   8602,  6349,  4198,  2765,  1741,  1126,   626,   403,   245,
     &    148,    90,    55,    33,    20,    12,     7,     4,     3,
     &      1,     1,     1,     0,    31,  7559,  6976,  2458,  3174,
     &   6246,  9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,
     &   1126,   736,   417,   251,   152,    91,    55,    33,    20,
     &     12,     7,     4,     3,     1,     1,     0,     1,    31,
     &   7979,  7051,  2560,  3379,  7066, 11366, 14438, 14234, 11469,
     &   8294,  5530,  3482,  2253,  1331,   782,   486,   292,   174,
     &    105,    62,    37,    22,    14,     8,     4,     3,     1,
     &      1,     1,     0,    31,  8400,  7126,  2560,  3789,  7782,
     &  13210, 17101, 16896, 13824,  9933,  6554,  4198,  2560,  1536,
     &    922,   568,   332,   197,   117,    70,    41,    24,    15,
     &      8,     5,     3,     2,     1,     0,     1,    32,  8819,
     &   7172,  2560,  3891,  8090, 14438, 19354, 19968, 16691, 12288/
      DATA (MAPEMAX (I), I= 4951, 5400) /
     &   8090,  5222,  3174,  1946,  1229,   688,   418,   249,   147,
     &     88,    52,    30,    19,    10,     7,     3,     3,     1,
     &      0,     1,     0,    32,  9239,  7210,  2662,  3789,  8499,
     &  15462, 21709, 23347, 19968, 14848, 10035,  6451,  3994,  2355,
     &   1536,   819,   528,   310,   184,   109,    65,    38,    23,
     &     13,     8,     5,     3,     1,     1,     1,     0,    32,
     &   9450,  7214,  2662,  3891,  8602, 15974, 22938, 24986, 21811,
     &  16282, 11059,  7168,  4403,  2662,  1638,  1024,   535,   345,
     &    205,   121,    72,    43,    25,    15,     9,     5,     3,
     &      1,     2,     0,     0,    32,  9659,  7218,  2662,  3891,
     &   8704, 16486, 24166, 26829, 23654, 17818, 12186,  7885,  4915,
     &   2970,  1741,  1126,   622,   382,   227,   135,    80,    47,
     &     28,    17,     9,     6,     4,     1,     1,     1,     0,
     &     32, 10079,  7206,  2662,  3891,  9011, 17408, 26522, 30515,
     &  27750, 21197, 14746,  9523,  5939,  3584,  2253,  1229,   834,
     &    467,   277,   165,    97,    58,    34,    21,    12,     7,
     &      4,     2,     2,     0,     1,    32, 10500,  7189,  2662,
     &   3994,  9216, 18227, 28877, 34509, 32051, 25088, 17510, 11469,
     &   7168,  4403,  2662,  1536,  1024,   537,   336,   199,   118,
     &     70,    42,    25,    14,     9,     5,     3,     1,     1,
     &      1,    31, 11550,  7057,  2662,  4096,  9626, 20275, 34611,
     &  44954, 44954, 36762, 26419, 17510, 11162,  6861,  4096,  2560,
     &   1434,   922,   510,   313,   186,   110,    66,    39,    22,
     &     14,     8,     4,     3,     1,    30, 12600,  6885,  2662,
     &   4096, 10035, 21811, 39936, 56115, 59904, 51507, 38093, 25805,
     &  16486, 10240,  6246,  3687,  2253,  1331,   822,   472,   280,
     &    166,    99,    58,    35,    20,    12,     7,     4,    29,
     &  13859,  6617,  2662,  4198, 10240, 23347, 45670, 69837, 80998,
     &  73728, 56730, 39322, 25498, 15974,  9728,  5939,  3482,  2151,
     &   1229,   728,   441,   262,   155,    92,    54,    33,    19,
     &     10,    29, 14700,  6417,  2662,  4198, 10445, 24166, 48845,
     &  78746, 95846, 90829, 71783, 50483, 33075, 20787, 12698,  7783,
     &   4608,  2765,  1639,  1024,   550,   344,   204,   121,    71,
     &     42,    26,     1,    27, 15750,  6071,  2662,  4198, 10547,
     &  25088, 52429, 89395,115610,115098, 94003, 67482, 44749, 28365,
     &  17408, 10650,  6349,  3789,  2253,  1434,   748,   474,   281,
     &    166,    99,    45,    25, 16800,  5630,  2662,  4198, 10650,
     &  25702, 55706, 99430,136192,142438,120320, 88269, 59290, 37786,
     &  23450, 14234,  8602,  5120,  3072,  1741,  1024,   738,   379,
     &    223,    20, 18900,  4290,  2765,  4915, 12902, 32870, 75674,
     & 142131,200295,210227,175719,126567, 83558, 52429, 31847, 18944,
     &  11264,  6656,  3175,    12, 21000,  2136,  3789, 10035, 34918,
     & 106701,239616,342221,323789,235725, 60109,     3, 23100,     0,
     &      3, 32767,     0,  1133,  4800,     3,     0,     0,     3,
     &   2310,     0,    23,  2520,  3623,    37,    18,    19,    18,
     &     17,    17,    16,    14,    13,    12,    11,    10,     9,
     &      8,     7,     7,     6,     5,     5,     4,    29,  2625,
     &   4922,    71,    32,    32,    31,    30,    28,    26,    24,
     &     22,    20,    18,    16,    14,    13,    11,    10,     9,
     &      7,     7,     6,     5,     4,     4,     3,     3,     3/
      DATA (MAPEMAX (I), I= 5401, 5850) /
     &     31,  2730,  5527,   169,    53,    47,    44,    41,    37,
     &     34,    31,    27,    24,    21,    18,    16,    14,    12,
     &     10,     9,     7,     6,     5,     5,     4,     3,     3,
     &      3,     1,     1,     3,    31,  2835,  5946,   275,    82,
     &     68,    63,    57,    50,    44,    38,    33,    28,    23,
     &     20,    16,    14,    11,     9,     8,     6,     5,     5,
     &      3,     2,     4,     2,     1,     1,     0,     0,    31,
     &   2940,  6243,   381,   116,    98,    89,    79,    68,    57,
     &     48,    39,    31,    25,    20,    16,    13,    10,     7,
     &      7,     4,     5,     3,     2,     1,     1,     0,     0,
     &      0,     0,     0,    31,  3045,  6375,   385,   161,   153,
     &    139,   121,   101,    82,    64,    49,    37,    28,    21,
     &     15,    11,     9,     6,     4,     2,     1,     1,     0,
     &      0,     0,     0,     0,     0,     0,     0,    31,  3150,
     &   6418,   311,   212,   212,   200,   178,   149,   119,    91,
     &     67,    48,    34,    23,    16,    14,     8,     5,     3,
     &      2,     1,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,    30,  3360,  6380,   315,   258,   273,   274,
     &    260,   233,   197,   158,   121,    89,    64,    44,    30,
     &     22,    17,    11,     6,     3,     2,     1,     1,     0,
     &      0,     0,     0,     0,     0,    30,  3570,  6294,   356,
     &    304,   334,   349,   346,   325,   287,   239,   189,   143,
     &    103,    74,    50,    34,    27,    18,    12,     7,     4,
     &      2,     1,     1,     0,     0,     0,     0,     0,    29,
     &   3780,  6186,   332,   334,   380,   415,   431,   425,   395,
     &    346,   286,   223,   167,   120,    84,    57,    38,    32,
     &     20,    12,     7,     4,     2,     1,     1,     0,     0,
     &      0,    28,  3990,  6045,   283,   359,   424,   480,   519,
     &    533,   518,   472,   405,   327,   250,   183,   128,    89,
     &     59,    44,    31,    20,    11,     6,     4,     2,     1,
     &      1,     0,    28,  4200,  5870,   308,   404,   491,   569,
     &    630,   663,   656,   608,   526,   427,   325,   236,   164,
     &    111,    73,    55,    37,    23,    13,     7,     4,     2,
     &      1,     1,     0,    27,  4620,  5553,   324,   429,   538,
     &    646,   751,   840,   896,   905,   858,   762,   631,   489,
     &    362,   253,   172,   113,    82,    57,    37,    21,    12,
     &      7,     4,     2,    26,  5039,  5713,  1331,   307,   410,
     &    512,   614,   614,   819,   819,   819,   922,   819,   819,
     &    819,   717,   614,   614,   512,   410,   307,   350,   254,
     &    211,    58,    26,  5250,  5835,  1331,   410,   512,   614,
     &    717,   819,   922,  1024,  1024,  1126,  1126,  1024,  1024,
     &    819,   819,   614,   614,   512,   307,   372,   269,   219,
     &    144,    27,  5459,  5952,  1331,   512,   614,   717,   922,
     &   1024,  1229,  1331,  1434,  1434,  1331,  1331,  1229,  1024,
     &    922,   717,   614,   512,   410,   374,   270,   214,   170,
     &     37,    27,  5669,  6071,  1434,   512,   717,   922,  1229,
     &   1434,  1536,  1843,  1843,  1843,  1741,  1638,  1434,  1229,
     &   1024,   819,   717,   512,   406,   329,   255,   198,   152,
     &     87,    28,  5879,  6186,  1536,   614,   922,  1126,  1536,
     &   1946,  2253,  2355,  2560,  2458,  2253,  2048,  1741,  1434/
      DATA (MAPEMAX (I), I= 5851, 6300) /
     &   1229,   819,   819,   512,   417,   322,   244,   184,   138,
     &    103,    14,    28,  6089,  6277,  1536,   819,  1229,  1638,
     &   2150,  2662,  3072,  3379,  3379,  3072,  2765,  2355,  1843,
     &   1536,  1126,   819,   717,   443,   351,   257,   189,   136,
     &    100,    72,    29,    28,  6300,  6366,  1638,  1024,  1638,
     &   2253,  3072,  3789,  4301,  4506,  4198,  3789,  3072,  2458,
     &   1843,  1434,   922,   819,   489,   375,   265,   188,   132,
     &     92,    65,    46,    28,    29,  6719,  6533,  2253,  2355,
     &   4198,  6246,  7680,  7680,  6451,  4915,  3482,  2355,  1434,
     &   1024,   600,   388,   243,   151,    93,    59,    37,    23,
     &     14,     8,     5,     3,     2,     1,    29,  7139,  6632,
     &   2355,  2867,  5222,  8090,  9830,  9523,  7885,  5734,  3994,
     &   2560,  1536,  1024,   664,   386,   236,   144,    88,    54,
     &     33,    20,    12,     7,     4,     3,     1,     1,    30,
     &   7350,  6688,  2458,  2970,  5734,  8909, 10957, 10547,  8602,
     &   6349,  4198,  2765,  1741,  1126,   626,   403,   245,   148,
     &     90,    55,    33,    20,    12,     7,     4,     3,     1,
     &      1,     0,    30,  7559,  6725,  2458,  3174,  6246,  9830,
     &  12186, 11674,  9421,  6861,  4608,  2970,  1741,  1126,   736,
     &    417,   251,   152,    91,    55,    33,    20,    12,     7,
     &      4,     2,     2,     1,     0,    30,  7979,  6792,  2560,
     &   3379,  7066, 11366, 14438, 14234, 11469,  8294,  5530,  3482,
     &   2253,  1331,   782,   486,   292,   174,   105,    62,    37,
     &     22,    14,     8,     4,     3,     1,     1,     1,    30,
     &   8400,  6859,  2560,  3789,  7782, 13210, 17101, 16896, 13824,
     &   9933,  6554,  4198,  2560,  1536,   922,   568,   332,   197,
     &    117,    70,    41,    24,    15,     8,     5,     3,     2,
     &      1,     0,    31,  8819,  6918,  2560,  3891,  8090, 14438,
     &  19354, 19968, 16691, 12288,  8090,  5222,  3174,  1946,  1229,
     &    688,   418,   249,   147,    88,    52,    30,    19,    10,
     &      7,     3,     2,     2,     0,     0,    31,  9239,  6940,
     &   2662,  3789,  8499, 15462, 21709, 23347, 19968, 14848, 10035,
     &   6451,  3994,  2355,  1536,   819,   528,   310,   184,   109,
     &     65,    38,    23,    13,     8,     5,     2,     2,     1,
     &      0,    31,  9450,  6955,  2662,  3891,  8602, 15974, 22938,
     &  24986, 21811, 16282, 11059,  7168,  4403,  2662,  1638,  1024,
     &    535,   345,   205,   121,    72,    43,    25,    15,     9,
     &      5,     3,     1,     2,     0,    31,  9659,  6940,  2662,
     &   3891,  8704, 16486, 24166, 26829, 23654, 17818, 12186,  7885,
     &   4915,  2970,  1741,  1126,   622,   382,   227,   135,    80,
     &     47,    28,    17,     9,     6,     3,     2,     1,     1,
     &     30, 10079,  6910,  2662,  3891,  9011, 17408, 26522, 30515,
     &  27750, 21197, 14746,  9523,  5939,  3584,  2253,  1229,   834,
     &    467,   277,   165,    97,    58,    34,    21,    12,     6,
     &      5,     2,     2,    30, 10500,  6841,  2662,  3994,  9216,
     &  18227, 28877, 34509, 32051, 25088, 17510, 11469,  7168,  4403,
     &   2662,  1536,  1024,   537,   336,   199,   118,    70,    42,
     &     25,    14,     8,     6,     3,     1,    30, 11550,  6668,
     &   2662,  4096,  9626, 20275, 34611, 44954, 44954, 36762, 26419,
     &  17510, 11162,  6861,  4096,  2560,  1434,   922,   510,   313/
      DATA (MAPEMAX (I), I= 6301, 6750) /
     &    186,   110,    66,    39,    22,    14,     8,     4,     1,
     &     29, 12600,  6452,  2662,  4096, 10035, 21811, 39936, 56115,
     &  59904, 51507, 38093, 25805, 16486, 10240,  6246,  3687,  2253,
     &   1331,   822,   472,   280,   166,    99,    58,    34,    21,
     &     12,     2,    28, 13859,  6206,  2662,  4198, 10240, 23347,
     &  45670, 69837, 80998, 73728, 56730, 39322, 25498, 15974,  9728,
     &   5939,  3482,  2151,  1229,   728,   441,   262,   155,    92,
     &     54,    33,     5,    27, 14700,  5894,  2662,  4198, 10445,
     &  24166, 48845, 78746, 95846, 90829, 71783, 50483, 33075, 20787,
     &  12698,  7783,  4608,  2765,  1639,  1024,   550,   344,   203,
     &    121,    72,     1,    25, 15750,  5509,  2662,  4198, 10547,
     &  25088, 52429, 89395,115610,115098, 94003, 67482, 44749, 28365,
     &  17408, 10650,  6349,  3789,  2253,  1434,   748,   474,   281,
     &     97,    23, 16800,  5020,  2662,  4198, 10650, 25702, 55706,
     &  99430,136192,142438,120320, 88269, 59290, 37786, 23450, 14234,
     &   8602,  5120,  2970,  1843,  1024,   529,    17, 18900,  3560,
     &   2765,  4915, 12902, 32870, 75674,142131,200295,210227,175719,
     & 126567, 83558, 52327, 31847, 17715,     9, 21000,  1332,  3789,
     &  10035, 34816,106701,239718, 65126,     3, 23100,     0,     3,
     &  32767,     0,  1084,  6400,     3,     0,     0,     3,  2310,
     &      0,    23,  2520,  3149,    37,    18,    19,    18,    17,
     &     17,    16,    14,    13,    12,    11,    10,     9,     8,
     &      7,     7,     6,     5,     5,     4,    28,  2625,  4466,
     &     71,    32,    32,    31,    30,    28,    26,    24,    22,
     &     20,    18,    16,    14,    13,    11,    10,     9,     7,
     &      7,     6,     5,     4,     4,     3,     3,    30,  2730,
     &   5120,   169,    53,    47,    44,    41,    37,    34,    31,
     &     27,    24,    21,    18,    16,    14,    12,    10,     9,
     &      7,     6,     5,     5,     4,     3,     3,     3,     1,
     &      1,    31,  2835,  5586,   275,    82,    68,    63,    57,
     &     50,    44,    38,    33,    28,    23,    20,    16,    14,
     &     11,     9,     8,     6,     5,     5,     3,     2,     4,
     &      2,     1,     1,     0,     0,    31,  2940,  5917,   381,
     &    116,    98,    89,    79,    68,    57,    48,    39,    31,
     &     25,    20,    16,    13,    10,     7,     7,     4,     5,
     &      3,     2,     1,     1,     0,     0,     0,     0,     0,
     &     31,  3045,  6066,   385,   161,   153,   139,   121,   101,
     &     82,    64,    49,    37,    28,    21,    15,    11,     9,
     &      6,     4,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,    31,  3150,  6077,   311,   212,
     &    212,   200,   178,   149,   119,    91,    67,    48,    34,
     &     23,    16,    14,     8,     5,     3,     2,     1,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,    30,
     &   3360,  5985,   315,   258,   273,   274,   260,   233,   197,
     &    158,   121,    89,    64,    44,    30,    22,    17,    11,
     &      6,     3,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,    29,  3570,  5803,   356,   304,   334,   349,
     &    346,   325,   287,   239,   189,   143,   103,    74,    50,
     &     34,    27,    18,    12,     7,     4,     2,     1,     1,
     &      0,     0,     0,     0,    28,  3780,  5558,   332,   334/
      DATA (MAPEMAX (I), I= 6751, 7200) /
     &    380,   415,   431,   425,   395,   346,   286,   223,   167,
     &    120,    84,    57,    38,    32,    20,    12,     7,     4,
     &      2,     1,     1,     0,     0,    26,  3990,  5253,   283,
     &    359,   424,   480,   519,   533,   518,   472,   405,   327,
     &    250,   183,   128,    89,    59,    44,    31,    20,    11,
     &      6,     4,     2,     1,    26,  4200,  5010,   308,   404,
     &    491,   569,   630,   663,   656,   608,   526,   427,   325,
     &    236,   164,   111,    73,    55,    37,    23,    13,     7,
     &      4,     2,     1,    24,  4620,  4713,   324,   429,   538,
     &    646,   751,   840,   896,   905,   858,   762,   631,   489,
     &    362,   253,   172,   113,    82,    57,    37,    21,    12,
     &     22,  5039,  4854,  1331,   307,   410,   512,   614,   614,
     &    819,   819,   819,   922,   819,   819,   819,   717,   614,
     &    512,   512,   410,   410,    23,  5250,  5119,  1331,   410,
     &    512,   614,   717,   819,   922,  1024,  1024,  1126,  1126,
     &   1024,  1024,   819,   819,   614,   512,   512,   410,   372,
     &     25,  5459,  5405,  1331,   512,   614,   717,   922,  1024,
     &   1229,  1331,  1434,  1434,  1331,  1331,  1229,  1024,   922,
     &    717,   614,   512,   410,   374,   270,    27,    26,  5669,
     &   5644,  1434,   512,   717,   922,  1229,  1434,  1536,  1843,
     &   1843,  1843,  1741,  1638,  1434,  1229,  1024,   819,   717,
     &    512,   406,   329,   255,   198,     8,    27,  5879,  5894,
     &   1536,   614,   922,  1126,  1536,  1946,  2253,  2355,  2560,
     &   2458,  2253,  2048,  1741,  1434,  1229,   819,   819,   512,
     &    417,   322,   244,   184,   138,     2,    27,  6089,  6071,
     &   1536,   819,  1229,  1638,  2150,  2662,  3072,  3379,  3379,
     &   3072,  2765,  2355,  1843,  1536,  1126,   819,   717,   443,
     &    351,   257,   188,   137,   100,    54,    28,  6300,  6178,
     &   1638,  1024,  1638,  2253,  3072,  3789,  4301,  4506,  4198,
     &   3789,  3072,  2458,  1843,  1434,   922,   819,   489,   375,
     &    265,   188,   132,    92,    65,    46,     5,    28,  6719,
     &   6324,  2253,  2355,  4198,  6246,  7680,  7680,  6451,  4915,
     &   3482,  2355,  1434,  1024,   600,   388,   243,   151,    93,
     &     59,    37,    23,    14,     8,     5,     3,     2,    29,
     &   7139,  6440,  2355,  2867,  5222,  8090,  9830,  9523,  7885,
     &   5734,  3994,  2560,  1536,  1024,   664,   386,   236,   144,
     &     88,    54,    33,    20,    12,     7,     4,     3,     1,
     &      1,    29,  7350,  6484,  2458,  2970,  5734,  8909, 10957,
     &  10547,  8602,  6349,  4198,  2765,  1741,  1126,   626,   403,
     &    245,   148,    90,    55,    33,    20,    12,     7,     4,
     &      3,     1,     1,    29,  7559,  6533,  2458,  3174,  6246,
     &   9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,  1126,
     &    736,   417,   251,   152,    91,    55,    33,    20,    12,
     &      7,     4,     2,     2,     1,    29,  7979,  6609,  2560,
     &   3379,  7066, 11366, 14438, 14234, 11469,  8294,  5530,  3482,
     &   2253,  1331,   782,   486,   292,   174,   105,    62,    37,
     &     22,    14,     8,     4,     3,     1,     1,    30,  8400,
     &   6661,  2560,  3789,  7782, 13210, 17101, 16896, 13824,  9933,
     &   6554,  4198,  2560,  1536,   922,   568,   332,   197,   117,
     &     70,    41,    24,    15,     8,     5,     3,     2,     1/
      DATA (MAPEMAX (I), I= 7201, 7650) /
     &      0,    30,  8819,  6725,  2560,  3891,  8090, 14438, 19354,
     &  19968, 16691, 12288,  8090,  5222,  3174,  1946,  1229,   688,
     &    418,   249,   147,    88,    52,    30,    19,    10,     7,
     &      3,     2,     2,     0,    30,  9239,  6743,  2662,  3789,
     &   8499, 15462, 21709, 23347, 19968, 14848, 10035,  6451,  3994,
     &   2355,  1536,   819,   528,   310,   184,   109,    65,    38,
     &     23,    13,     8,     5,     2,     2,     0,    30,  9450,
     &   6737,  2662,  3891,  8602, 15974, 22938, 24986, 21811, 16282,
     &  11059,  7168,  4403,  2662,  1638,  1024,   535,   345,   205,
     &    121,    72,    43,    25,    15,     9,     4,     4,     1,
     &      1,    30,  9659,  6731,  2662,  3891,  8704, 16486, 24166,
     &  26829, 23654, 17818, 12186,  7885,  4915,  2970,  1741,  1126,
     &    622,   382,   227,   135,    80,    47,    28,    17,     9,
     &      6,     3,     2,     1,    30, 10079,  6681,  2662,  3891,
     &   9011, 17408, 26522, 30515, 27750, 21197, 14746,  9523,  5939,
     &   3584,  2253,  1229,   834,   467,   277,   165,    97,    58,
     &     34,    21,    12,     6,     5,     2,     0,    29, 10500,
     &   6601,  2662,  3994,  9216, 18227, 28877, 34509, 32051, 25088,
     &  17510, 11469,  7168,  4403,  2662,  1536,  1024,   537,   336,
     &    199,   118,    70,    42,    25,    14,     8,     6,     2,
     &     28, 11550,  6392,  2662,  4096,  9626, 20275, 34611, 44954,
     &  44954, 36762, 26419, 17510, 11162,  6861,  4096,  2560,  1434,
     &    922,   510,   313,   186,   110,    66,    38,    23,    14,
     &      7,    27, 12600,  6143,  2662,  4096, 10035, 21811, 39936,
     &  56115, 59904, 51507, 38093, 25805, 16486, 10240,  6246,  3687,
     &   2253,  1331,   822,   472,   280,   166,    98,    59,    34,
     &     21,    26, 13859,  5817,  2662,  4198, 10240, 23347, 45670,
     &  69837, 80998, 73728, 56730, 39322, 25498, 15974,  9728,  5939,
     &   3482,  2151,  1229,   728,   441,   262,   155,    92,    42,
     &     25, 14700,  5544,  2662,  4198, 10445, 24166, 48845, 78746,
     &  95846, 90829, 71783, 50483, 33075, 20787, 12698,  7783,  4608,
     &   2765,  1639,  1024,   549,   344,   204,    87,    23, 15750,
     &   5119,  2662,  4198, 10547, 25088, 52429, 89395,115610,115098,
     &  94003, 67482, 44749, 28365, 17408, 10650,  6349,  3789,  2253,
     &   1331,   850,   474,    21, 16800,  4569,  2662,  4198, 10650,
     &  25702, 55706, 99430,136192,142438,120320, 88269, 59290, 37786,
     &  23450, 14234,  8499,  5120,  3072,  1639,    15, 18900,  2944,
     &   2765,  4915, 12902, 32870, 75674,142131,200295,210227,175616,
     & 126669, 83558, 29287,     6, 21000,   715,  3686, 10035, 24064,
     &      3, 23100,     0,     3, 32767,     0,  1008,  9600,     3,
     &      0,     0,     3,  2310,     0,    22,  2520,  2572,    37,
     &     18,    19,    18,    17,    17,    16,    14,    13,    12,
     &     11,    10,     9,     8,     7,     7,     6,     5,     5,
     &     27,  2625,  3869,    71,    32,    32,    31,    30,    28,
     &     26,    24,    22,    20,    18,    16,    14,    13,    11,
     &     10,     9,     7,     7,     6,     5,     4,     4,     3,
     &     30,  2730,  4606,   169,    53,    47,    44,    41,    37,
     &     34,    31,    27,    24,    21,    18,    16,    14,    12,
     &     10,     9,     7,     6,     5,     5,     4,     3,     3,
     &      3,     1,     1,    30,  2835,  5099,   275,    82,    68/
      DATA (MAPEMAX (I), I= 7651, 8100) /
     &     63,    57,    50,    44,    38,    33,    28,    23,    20,
     &     16,    14,    11,     9,     8,     6,     5,     5,     3,
     &      2,     4,     2,     1,     1,     0,    30,  2940,  5470,
     &    381,   116,    98,    89,    79,    68,    57,    48,    39,
     &     31,    25,    20,    16,    13,    10,     7,     7,     4,
     &      5,     3,     2,     1,     1,     0,     0,     0,     0,
     &     30,  3045,  5664,   385,   161,   153,   139,   121,   101,
     &     82,    64,    49,    37,    28,    21,    15,    11,     9,
     &      6,     4,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,    30,  3150,  5690,   311,   212,   212,
     &    200,   178,   149,   119,    91,    67,    48,    34,    23,
     &     16,    14,     8,     5,     3,     2,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,    28,  3360,  5545,
     &    315,   258,   273,   274,   260,   233,   197,   158,   121,
     &     89,    64,    44,    30,    22,    17,    11,     6,     3,
     &      2,     1,     1,     0,     0,     0,     0,    27,  3570,
     &   5335,   356,   304,   334,   349,   346,   325,   287,   239,
     &    189,   143,   103,    74,    50,    34,    27,    18,    12,
     &      7,     4,     2,     1,     1,     0,     0,    25,  3780,
     &   5042,   332,   334,   380,   415,   431,   425,   395,   346,
     &    286,   223,   167,   120,    84,    57,    38,    32,    20,
     &     12,     7,     4,     2,     1,    24,  3990,  4620,   283,
     &    359,   424,   480,   519,   533,   518,   472,   405,   327,
     &    250,   183,   128,    89,    59,    44,    31,    20,    11,
     &      6,     4,    23,  4200,  4229,   308,   404,   491,   569,
     &    630,   663,   656,   608,   526,   427,   325,   236,   164,
     &    111,    73,    55,    37,    23,    13,     7,    21,  4620,
     &   3846,   324,   429,   538,   646,   751,   840,   896,   905,
     &    858,   762,   631,   489,   362,   253,   172,   113,    82,
     &     57,    20,  5039,  4096,  1331,   307,   410,   512,   614,
     &    614,   819,   819,   819,   922,   819,   819,   819,   614,
     &    717,   512,     0,    22,  5250,  4712,  1331,   410,   512,
     &    614,   717,   819,   922,  1024,  1024,  1126,  1126,  1024,
     &   1024,   819,   819,   614,   512,   512,   205,    23,  5459,
     &   5073,  1331,   512,   614,   717,   922,  1024,  1229,  1331,
     &   1434,  1434,  1331,  1331,  1229,  1024,   922,   717,   614,
     &    512,   410,   317,    24,  5669,  5306,  1434,   512,   717,
     &    922,  1229,  1434,  1536,  1843,  1843,  1843,  1741,  1638,
     &   1434,  1229,  1024,   819,   717,   410,   509,   329,   191,
     &     25,  5879,  5499,  1536,   614,   922,  1126,  1536,  1946,
     &   2253,  2355,  2560,  2458,  2253,  2048,  1741,  1434,  1229,
     &    819,   819,   512,   417,   322,   244,    95,    26,  6089,
     &   5707,  1536,   819,  1229,  1638,  2150,  2662,  3072,  3379,
     &   3379,  3072,  2765,  2355,  1843,  1536,  1126,   819,   717,
     &    443,   351,   257,   188,   137,    33,    26,  6300,  5835,
     &   1638,  1024,  1638,  2253,  3072,  3789,  4301,  4506,  4198,
     &   3789,  3072,  2458,  1843,  1434,   922,   819,   489,   375,
     &    265,   187,   132,    93,    54,    27,  6719,  6010,  2253,
     &   2355,  4198,  6246,  7680,  7680,  6451,  4915,  3482,  2355,
     &   1434,  1024,   600,   388,   243,   151,    93,    59,    37/
      DATA (MAPEMAX (I), I= 8101, 8550) /
     &     23,    13,     9,     5,     2,    27,  7139,  6121,  2355,
     &   2867,  5222,  8090,  9830,  9523,  7885,  5734,  3994,  2560,
     &   1536,  1024,   664,   386,   236,   144,    88,    54,    33,
     &     20,    11,     8,     4,     3,    28,  7350,  6165,  2458,
     &   2970,  5734,  8909, 10957, 10547,  8602,  6349,  4198,  2765,
     &   1741,  1126,   626,   403,   245,   148,    90,    55,    33,
     &     20,    12,     7,     4,     3,     0,    28,  7559,  6260,
     &   2458,  3174,  6246,  9830, 12186, 11674,  9421,  6861,  4608,
     &   2970,  1741,  1126,   736,   417,   251,   152,    91,    55,
     &     33,    20,    12,     7,     4,     2,     1,    28,  7979,
     &   6277,  2560,  3379,  7066, 11366, 14438, 14234, 11469,  8294,
     &   5530,  3482,  2253,  1331,   782,   486,   292,   174,   105,
     &     62,    37,    22,    14,     7,     5,     3,     1,    28,
     &   8400,  6353,  2560,  3789,  7782, 13210, 17101, 16896, 13824,
     &   9933,  6554,  4198,  2560,  1536,   922,   568,   332,   197,
     &    117,    70,    41,    24,    15,     8,     5,     3,     1,
     &     28,  8819,  6353,  2560,  3891,  8090, 14438, 19354, 19968,
     &  16691, 12288,  8090,  5222,  3174,  1946,  1229,   688,   418,
     &    249,   147,    88,    52,    30,    19,    10,     6,     4,
     &      2,    28,  9239,  6318,  2662,  3789,  8499, 15462, 21709,
     &  23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,   819,
     &    528,   310,   184,   109,    65,    38,    23,    13,     8,
     &      5,     2,    28,  9450,  6293,  2662,  3891,  8602, 15974,
     &  22938, 24986, 21811, 16282, 11059,  7168,  4403,  2662,  1638,
     &   1024,   535,   345,   205,   121,    72,    43,    25,    14,
     &      9,     5,     2,    28,  9659,  6267,  2662,  3891,  8704,
     &  16486, 24166, 26829, 23654, 17818, 12186,  7885,  4915,  2970,
     &   1741,  1126,   622,   382,   227,   135,    80,    47,    28,
     &     16,    10,     6,     2,    28, 10079,  6186,  2662,  3891,
     &   9011, 17408, 26522, 30515, 27750, 21197, 14746,  9523,  5939,
     &   3584,  2253,  1229,   834,   467,   277,   165,    97,    58,
     &     34,    20,    12,     7,     1,    27, 10500,  6097,  2662,
     &   3994,  9216, 18227, 28877, 34509, 32051, 25088, 17510, 11469,
     &   7168,  4403,  2662,  1536,  1024,   537,   336,   199,   118,
     &     70,    42,    24,    15,     7,    26, 11550,  5869,  2662,
     &   4096,  9626, 20275, 34611, 44954, 44954, 36762, 26419, 17510,
     &  11162,  6861,  4096,  2560,  1434,   922,   510,   313,   186,
     &    110,    65,    39,    22,    25, 12600,  5585,  2662,  4096,
     &  10035, 21811, 39936, 56115, 59904, 51507, 38093, 25805, 16486,
     &  10240,  6246,  3687,  2253,  1331,   822,   472,   280,   166,
     &     98,    50,    24, 13859,  5201,  2662,  4198, 10240, 23347,
     &  45670, 69837, 80998, 73728, 56730, 39322, 25498, 15974,  9728,
     &   5939,  3482,  2151,  1229,   727,   442,   262,    58,    23,
     &  14700,  4885,  2662,  4198, 10445, 24166, 48845, 78746, 95846,
     &  90829, 71783, 50483, 33075, 20787, 12698,  7783,  4608,  2765,
     &   1639,   922,   652,    36,    21, 15750,  4392,  2662,  4198,
     &  10547, 25088, 52429, 89395,115610,115098, 94003, 67482, 44749,
     &  28365, 17408, 10650,  6349,  3789,  2253,   205,    18, 16800,
     &   3730,  2662,  4198, 10650, 25702, 55706, 99430,136192,142438,
     & 120320, 88269, 59290, 37786, 23450, 14234,  5427,    11, 18900/
      DATA (MAPEMAX (I), I= 8551, 9000) /
     &   2025,  2765,  4915, 12902, 32870, 75674,142029,200294,192512,
     &      3, 21000,     0,     3, 32767,     0,   940, 12800,     3,
     &      0,     0,     3,  2310,     0,    20,  2520,  2228,    37,
     &     18,    19,    18,    17,    17,    16,    14,    13,    12,
     &     11,    10,     9,     8,     7,     7,     6,    26,  2625,
     &   3497,    71,    32,    32,    31,    30,    28,    26,    24,
     &     22,    20,    18,    16,    14,    13,    11,    10,     9,
     &      7,     7,     6,     5,     4,     4,    28,  2730,  4229,
     &    169,    53,    47,    44,    41,    37,    34,    31,    27,
     &     24,    21,    18,    16,    14,    12,    10,     9,     7,
     &      6,     5,     5,     4,     3,     3,     3,    29,  2835,
     &   4739,   275,    82,    68,    63,    57,    50,    44,    38,
     &     33,    28,    23,    20,    16,    14,    11,     9,     8,
     &      6,     5,     5,     3,     2,     4,     2,     1,     1,
     &     29,  2940,  5097,   381,   116,    98,    89,    79,    68,
     &     57,    48,    39,    31,    25,    20,    16,    13,    10,
     &      7,     7,     4,     5,     3,     2,     1,     1,     0,
     &      0,     0,    29,  3045,  5309,   385,   161,   153,   139,
     &    121,   101,    82,    64,    49,    37,    28,    21,    15,
     &     11,     9,     6,     4,     2,     1,     1,     0,     0,
     &      0,     0,     0,     0,    28,  3150,  5356,   311,   212,
     &    212,   200,   178,   149,   119,    91,    67,    48,    34,
     &     23,    16,    14,     8,     5,     3,     2,     1,     0,
     &      0,     0,     0,     0,     0,    27,  3360,  5170,   315,
     &    258,   273,   274,   260,   233,   197,   158,   121,    89,
     &     64,    44,    30,    22,    17,    11,     6,     3,     2,
     &      1,     1,     0,     0,     0,    25,  3570,  4882,   356,
     &    304,   334,   349,   346,   325,   287,   239,   189,   143,
     &    103,    74,    50,    34,    27,    18,    12,     7,     4,
     &      2,     1,     1,    23,  3780,  4426,   332,   334,   380,
     &    415,   431,   425,   395,   346,   286,   223,   167,   120,
     &     84,    57,    38,    32,    20,    12,     7,     4,    22,
     &   3990,  3931,   283,   359,   424,   480,   519,   533,   518,
     &    472,   405,   327,   250,   183,   128,    89,    59,    44,
     &     31,    20,    11,    20,  4200,  3589,   308,   404,   491,
     &    569,   630,   663,   656,   608,   526,   427,   325,   236,
     &    164,   111,    73,    55,    37,    18,  4620,  3171,   324,
     &    429,   538,   646,   751,   840,   896,   905,   858,   762,
     &    631,   489,   362,   253,   172,    16,  5039,  3171,  1331,
     &    307,   410,   512,   614,   614,   819,   819,   819,   819,
     &    819,   922,   307,    20,  5250,  4126,  1331,   410,   512,
     &    614,   717,   819,   922,  1024,  1024,  1126,  1126,  1024,
     &   1024,   819,   717,   717,     0,    22,  5459,  4646,  1331,
     &    512,   614,   717,   922,  1024,  1229,  1331,  1434,  1434,
     &   1331,  1331,  1229,  1024,   922,   614,   717,   512,     0,
     &     23,  5669,  4928,  1434,   512,   717,   922,  1229,  1434,
     &   1536,  1843,  1843,  1843,  1741,  1638,  1434,  1229,  1024,
     &    819,   614,   512,   509,    90,    23,  5879,  5119,  1536,
     &    614,   922,  1126,  1536,  1946,  2253,  2355,  2560,  2458,
     &   2253,  2048,  1741,  1434,  1229,   819,   717,   512,   519/
      DATA (MAPEMAX (I), I= 9001, 9450) /
     &    322,    24,  6089,  5355,  1536,   819,  1229,  1638,  2150,
     &   2662,  3072,  3379,  3379,  3072,  2765,  2355,  1843,  1536,
     &   1126,   819,   717,   443,   351,   257,   176,    25,  6300,
     &   5470,  1638,  1024,  1638,  2253,  3072,  3789,  4301,  4506,
     &   4198,  3789,  3072,  2458,  1843,  1434,   922,   819,   489,
     &    375,   265,   187,   132,    39,    26,  6719,  5677,  2253,
     &   2355,  4198,  6246,  7680,  7680,  6451,  4915,  3482,  2355,
     &   1434,  1024,   600,   388,   243,   151,    93,    59,    37,
     &     22,    14,     9,     1,    26,  7139,  5798,  2355,  2867,
     &   5222,  8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,
     &   1024,   664,   386,   236,   144,    88,    54,    33,    19,
     &     12,     8,     3,    26,  7350,  5861,  2458,  2970,  5734,
     &   8909, 10957, 10547,  8602,  6349,  4198,  2765,  1741,  1126,
     &    626,   403,   245,   148,    90,    55,    33,    19,    12,
     &      8,     4,    27,  7559,  5909,  2458,  3174,  6246,  9830,
     &  12186, 11674,  9421,  6861,  4608,  2970,  1741,  1126,   736,
     &    417,   251,   152,    91,    55,    33,    20,    11,     8,
     &      4,     0,    27,  7979,  5952,  2560,  3379,  7066, 11366,
     &  14438, 14234, 11469,  8294,  5530,  3482,  2253,  1331,   782,
     &    486,   292,   174,   105,    62,    37,    22,    13,     8,
     &      5,     1,    27,  8400,  5978,  2560,  3789,  7782, 13210,
     &  17101, 16896, 13824,  9933,  6554,  4198,  2560,  1536,   922,
     &    568,   332,   197,   117,    70,    41,    24,    14,     9,
     &      5,     1,    27,  8819,  5931,  2560,  3891,  8090, 14438,
     &  19354, 19968, 16691, 12288,  8090,  5222,  3174,  1946,  1229,
     &    688,   418,   249,   147,    88,    52,    30,    18,    11,
     &      6,     1,    27,  9239,  5894,  2662,  3789,  8499, 15462,
     &  21709, 23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,
     &    819,   528,   310,   184,   109,    65,    38,    22,    14,
     &      8,     0,    26,  9450,  5853,  2662,  3891,  8602, 15974,
     &  22938, 24986, 21811, 16282, 11059,  7168,  4403,  2662,  1638,
     &   1024,   535,   345,   205,   121,    72,    42,    25,    15,
     &      8,    26,  9659,  5817,  2662,  3891,  8704, 16486, 24166,
     &  26829, 23654, 17818, 12186,  7885,  4915,  2970,  1741,  1126,
     &    622,   382,   227,   135,    80,    46,    28,    17,     8,
     &     26, 10079,  5736,  2662,  3891,  9011, 17408, 26522, 30515,
     &  27750, 21197, 14746,  9523,  5939,  3584,  2253,  1229,   834,
     &    467,   277,   165,    97,    58,    34,    20,     6,    26,
     &  10500,  5650,  2662,  3994,  9216, 18227, 28877, 34509, 32051,
     &  25088, 17510, 11469,  7168,  4403,  2662,  1536,  1024,   537,
     &    336,   199,   118,    70,    42,    24,     2,    25, 11550,
     &   5393,  2662,  4096,  9626, 20275, 34611, 44954, 44954, 36762,
     &  26419, 17510, 11162,  6861,  4096,  2560,  1434,   922,   510,
     &    313,   186,   110,    65,     4,    23, 12600,  5115,  2662,
     &   4096, 10035, 21811, 39936, 56115, 59904, 51507, 38093, 25805,
     &  16486, 10240,  6246,  3687,  2253,  1331,   821,   473,   280,
     &    164,    22, 13859,  4712,  2662,  4198, 10240, 23347, 45670,
     &  69837, 80998, 73728, 56730, 39322, 25498, 15974,  9728,  5939,
     &   3482,  2048,  1229,   830,   208,    20, 14700,  4331,  2662,
     &   4198, 10445, 24166, 48845, 78746, 95846, 90829, 71783, 50483/
      DATA (MAPEMAX (I), I= 9451, 9900) /
     &  33075, 20787, 12698,  7680,  4711,  2765,  1536,    18, 15750,
     &   3750,  2662,  4198, 10547, 25088, 52429, 89395,115610,115098,
     &  94003, 67482, 44749, 28262, 17511, 10547,  4506,    15, 16800,
     &   3049,  2662,  4198, 10650, 25702, 55706, 99430,136192,142438,
     & 120320, 88167, 59290, 35123,     9, 18900,  1431,  2765,  4915,
     &  12800, 32973, 75674, 75366,     3, 21000,     0,     3, 32767,
     &      0,   864, 16000,     3,     0,     0,     3,  2310,     0,
     &     18,  2520,  1766,    37,    18,    19,    18,    17,    17,
     &     16,    14,    13,    12,    11,    10,     9,     8,     7,
     &     24,  2625,  3044,    71,    32,    32,    31,    30,    28,
     &     26,    24,    22,    20,    18,    16,    14,    13,    11,
     &     10,     9,     7,     7,     6,     5,    26,  2730,  3869,
     &    169,    53,    47,    44,    41,    37,    34,    31,    27,
     &     24,    21,    18,    16,    14,    12,    10,     9,     7,
     &      6,     5,     5,     4,     3,    26,  2835,  4404,   275,
     &     82,    68,    63,    57,    50,    44,    38,    33,    28,
     &     23,    20,    16,    14,    11,     9,     8,     6,     5,
     &      5,     3,     2,     4,    27,  2940,  4757,   381,   116,
     &     98,    89,    79,    68,    57,    48,    39,    31,    25,
     &     20,    16,    13,    10,     7,     7,     4,     5,     3,
     &      2,     1,     1,     0,    27,  3045,  4986,   385,   161,
     &    153,   139,   121,   101,    82,    64,    49,    37,    28,
     &     21,    15,    11,     9,     6,     4,     2,     1,     1,
     &      0,     0,     0,     0,    26,  3150,  5020,   311,   212,
     &    212,   200,   178,   149,   119,    91,    67,    48,    34,
     &     23,    16,    14,     8,     5,     3,     2,     1,     0,
     &      0,     0,     0,    25,  3360,  4770,   315,   258,   273,
     &    274,   260,   233,   197,   158,   121,    89,    64,    44,
     &     30,    22,    17,    11,     6,     3,     2,     1,     1,
     &      0,    24,  3570,  4426,   356,   304,   334,   349,   346,
     &    325,   287,   239,   189,   143,   103,    74,    50,    34,
     &     27,    18,    12,     7,     4,     2,     1,    21,  3780,
     &   3911,   332,   334,   380,   415,   431,   425,   395,   346,
     &    286,   223,   167,   120,    84,    57,    38,    32,    20,
     &     12,    19,  3990,  3345,   280,   358,   423,   479,   518,
     &    533,   518,   473,   406,   328,   251,   184,   129,    89,
     &     60,    45,    17,  4200,  2956,   308,   404,   491,   569,
     &    630,   663,   656,   608,   526,   427,   325,   236,   164,
     &    111,    15,  4620,  2493,   324,   429,   538,   646,   751,
     &    840,   896,   905,   858,   762,   631,   489,    12,  5039,
     &   2283,  1331,   307,   410,   512,   614,   614,   717,   819,
     &    819,    17,  5250,  3412,  1331,   410,   512,   614,   717,
     &    819,   922,  1024,  1024,  1126,  1024,  1126,   922,   307,
     &     20,  5459,  4117,  1331,   512,   614,   717,   922,  1024,
     &   1229,  1331,  1434,  1434,  1331,  1331,  1229,   922,   922,
     &    717,   102,    21,  5669,  4494,  1434,   512,   717,   922,
     &   1229,  1434,  1536,  1843,  1843,  1843,  1741,  1638,  1434,
     &   1229,   922,   922,   614,   307,    22,  5879,  4712,  1536,
     &    614,   922,  1126,  1536,  1946,  2253,  2355,  2560,  2458,
     &   2253,  2048,  1741,  1434,  1229,   819,   717,   512,   281/
      DATA (MAPEMAX (I), I= 9901,10350) /
     &     23,  6089,  4992,  1536,   819,  1229,  1638,  2150,  2662,
     &   3072,  3379,  3379,  3072,  2765,  2355,  1843,  1536,  1126,
     &    819,   614,   545,   350,   139,    23,  6300,  5111,  1638,
     &   1024,  1638,  2253,  3072,  3789,  4301,  4506,  4198,  3789,
     &   3072,  2458,  1843,  1434,   922,   819,   489,   374,   266,
     &    182,    24,  6719,  5314,  2253,  2355,  4198,  6246,  7680,
     &   7680,  6451,  4915,  3482,  2355,  1434,  1024,   600,   388,
     &    243,   151,    93,    59,    37,    22,    11,    25,  7139,
     &   5499,  2355,  2867,  5222,  8090,  9830,  9523,  7885,  5734,
     &   3994,  2560,  1536,  1024,   664,   386,   236,   144,    88,
     &     54,    33,    19,    12,     4,    25,  7350,  5536,  2458,
     &   2970,  5734,  8909, 10957, 10547,  8602,  6349,  4198,  2765,
     &   1741,  1126,   626,   403,   245,   148,    90,    55,    32,
     &     20,    12,     5,    25,  7559,  5601,  2458,  3174,  6246,
     &   9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,  1126,
     &    736,   417,   251,   152,    91,    55,    33,    19,    12,
     &      7,    26,  7979,  5650,  2560,  3379,  7066, 11366, 14438,
     &  14234, 11469,  8294,  5530,  3482,  2253,  1331,   782,   486,
     &    292,   174,   105,    62,    37,    22,    13,     8,     1,
     &     25,  8400,  5630,  2560,  3789,  7782, 13210, 17101, 16896,
     &  13824,  9933,  6554,  4198,  2560,  1536,   922,   568,   332,
     &    197,   117,    70,    40,    25,    14,     9,    25,  8819,
     &   5544,  2560,  3891,  8090, 14438, 19354, 19968, 16691, 12288,
     &   8090,  5222,  3174,  1946,  1229,   688,   418,   249,   147,
     &     88,    51,    31,    18,     8,    25,  9239,  5470,  2662,
     &   3789,  8499, 15462, 21709, 23347, 19968, 14848, 10035,  6451,
     &   3994,  2355,  1536,   819,   528,   310,   184,   109,    64,
     &     39,    22,     6,    25,  9450,  5428,  2662,  3891,  8602,
     &  15974, 22938, 24986, 21811, 16282, 11059,  7168,  4403,  2662,
     &   1638,  1024,   535,   345,   205,   121,    71,    43,    25,
     &      4,    25,  9659,  5381,  2662,  3891,  8704, 16486, 24166,
     &  26829, 23654, 17818, 12186,  7885,  4915,  2970,  1741,  1126,
     &    622,   382,   227,   135,    79,    47,    28,     1,    24,
     &  10079,  5253,  2662,  3891,  9011, 17408, 26522, 30515, 27750,
     &  21197, 14746,  9523,  5939,  3584,  2253,  1229,   834,   467,
     &    277,   164,    98,    58,    20,    24, 10500,  5201,  2662,
     &   3994,  9216, 18227, 28877, 34509, 32051, 25088, 17510, 11469,
     &   7168,  4403,  2662,  1536,  1024,   537,   336,   199,   118,
     &     70,    16,    23, 11550,  4921,  2662,  4096,  9626, 20275,
     &  34611, 44954, 44954, 36762, 26419, 17510, 11162,  6861,  4096,
     &   2560,  1434,   922,   509,   314,   186,    30,    21, 12600,
     &   4584,  2662,  4096, 10035, 21811, 39936, 56115, 59904, 51507,
     &  38093, 25805, 16486, 10240,  6246,  3687,  2253,  1331,   821,
     &    439,    20, 13859,  4117,  2662,  4198, 10240, 23347, 45670,
     &  69837, 80998, 73728, 56730, 39322, 25498, 15974,  9728,  5837,
     &   3584,  2048,   103,    18, 14700,  3688,  2662,  4198, 10445,
     &  24166, 48845, 78746, 95846, 90829, 71783, 50483, 33075, 20685,
     &  12800,  7680,  2253,    16, 15750,  3126,  2662,  4198, 10547,
     &  25088, 52429, 89395,115610,115098, 94003, 67379, 44749, 28365,
     &   4506,    13, 16800,  2529,  2662,  4198, 10650, 25702, 55706/
      DATA (MAPEMAX (I), I=10351,10800) /
     &  99430,136192,142438,120320, 79258,     8, 18900,  1086,  2765,
     &   4813, 12902, 32973, 13312,     3, 21000,     0,     3, 32767,
     &      0,   784, 19200,     3,     0,     0,     3,  2310,     0,
     &     16,  2520,  1230,    37,    18,    19,    18,    17,    17,
     &     16,    14,    13,    12,    11,    10,     9,    21,  2625,
     &   2537,    71,    32,    32,    31,    30,    28,    26,    24,
     &     22,    20,    18,    16,    14,    13,    11,    10,     9,
     &      7,    24,  2730,  3308,   169,    53,    47,    44,    41,
     &     37,    34,    31,    27,    24,    21,    18,    16,    14,
     &     12,    10,     9,     7,     6,     5,     5,    24,  2835,
     &   3715,   275,    82,    68,    63,    57,    50,    44,    38,
     &     33,    28,    23,    20,    16,    14,    11,     9,     8,
     &      6,     5,     5,     3,    24,  2940,  3968,   381,   116,
     &     98,    89,    79,    68,    57,    48,    39,    31,    25,
     &     20,    16,    13,    10,     7,     7,     4,     5,     3,
     &      2,    24,  3045,  4138,   385,   161,   153,   139,   121,
     &    101,    82,    64,    49,    37,    28,    21,    15,    11,
     &      9,     6,     4,     2,     1,     1,     0,    24,  3150,
     &   4138,   311,   212,   212,   200,   178,   149,   119,    91,
     &     67,    48,    34,    23,    16,    14,     8,     5,     3,
     &      2,     1,     0,     0,    23,  3360,  3869,   315,   258,
     &    273,   274,   260,   233,   197,   158,   121,    89,    64,
     &     44,    30,    22,    17,    11,     6,     3,     2,     1,
     &     22,  3570,  3514,   356,   304,   334,   349,   346,   325,
     &    287,   239,   189,   143,   103,    74,    50,    34,    27,
     &     18,    12,     7,     4,    19,  3780,  3081,   332,   334,
     &    380,   415,   431,   425,   395,   346,   286,   223,   167,
     &    120,    84,    57,    38,    32,    16,  3990,  2551,   283,
     &    359,   424,   480,   519,   533,   518,   472,   405,   327,
     &    250,   183,   128,    14,  4200,  2086,   308,   404,   491,
     &    569,   630,   663,   656,   608,   526,   427,   325,    11,
     &   4620,  1404,   324,   429,   538,   646,   751,   840,   896,
     &    905,     8,  5039,  1024,  1331,   307,   410,   512,     0,
     &     13,  5250,  2558,  1331,   410,   512,   614,   717,   819,
     &    819,  1024,  1126,  1126,    17,  5459,  3479,  1331,   512,
     &    614,   717,   922,  1024,  1229,  1331,  1434,  1434,  1331,
     &   1331,  1126,   614,    19,  5669,  3996,  1434,   512,   717,
     &    922,  1229,  1434,  1536,  1843,  1843,  1843,  1741,  1638,
     &   1434,  1229,   922,   614,    20,  5879,  4290,  1536,   614,
     &    922,  1126,  1536,  1946,  2253,  2355,  2560,  2458,  2253,
     &   2048,  1741,  1434,  1126,   922,   512,    22,  6089,  4626,
     &   1536,   819,  1229,  1638,  2150,  2662,  3072,  3379,  3379,
     &   3072,  2765,  2355,  1843,  1536,  1126,   819,   614,   545,
     &     29,    22,  6300,  4744,  1638,  1024,  1638,  2253,  3072,
     &   3789,  4301,  4506,  4198,  3789,  3072,  2458,  1843,  1434,
     &    922,   717,   590,   375,   153,    23,  6719,  4980,  2253,
     &   2355,  4198,  6246,  7680,  7680,  6451,  4915,  3482,  2355,
     &   1434,  1024,   600,   388,   243,   151,    93,    59,    37,
     &     11,    24,  7139,  5128,  2355,  2867,  5222,  8090,  9830,
     &   9523,  7885,  5734,  3994,  2560,  1536,  1024,   664,   386/
      DATA (MAPEMAX (I), I=10801,11250) /
     &    236,   144,    88,    54,    33,    19,     1,    24,  7350,
     &   5201,  2458,  2970,  5734,  8909, 10957, 10547,  8602,  6349,
     &   4198,  2765,  1741,  1126,   626,   403,   245,   148,    90,
     &     54,    33,    20,     5,    24,  7559,  5219,  2458,  3174,
     &   6246,  9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,
     &   1126,   736,   417,   251,   152,    91,    55,    33,    19,
     &      6,    24,  7979,  5219,  2560,  3379,  7066, 11366, 14438,
     &  14234, 11469,  8294,  5530,  3482,  2253,  1331,   782,   486,
     &    292,   174,   105,    61,    38,    22,     6,    24,  8400,
     &   5150,  2560,  3789,  7782, 13210, 17101, 16896, 13824,  9933,
     &   6554,  4198,  2560,  1536,   922,   568,   332,   197,   117,
     &     69,    41,    25,     2,    23,  8819,  5101,  2560,  3891,
     &   8090, 14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,
     &   1946,  1229,   688,   418,   249,   147,    87,    52,    29,
     &     23,  9239,  5020,  2662,  3789,  8499, 15462, 21709, 23347,
     &  19968, 14848, 10035,  6451,  3994,  2355,  1536,   819,   528,
     &    310,   183,   110,    64,    26,    23,  9450,  4992,  2662,
     &   3891,  8602, 15974, 22938, 24986, 21811, 16282, 11059,  7168,
     &   4403,  2662,  1638,  1024,   535,   345,   204,   121,    72,
     &     25,    23,  9659,  4941,  2662,  3891,  8704, 16486, 24166,
     &  26829, 23654, 17818, 12186,  7885,  4915,  2970,  1741,  1126,
     &    622,   382,   227,   134,    80,    17,    22, 10079,  4837,
     &   2662,  3891,  9011, 17408, 26522, 30515, 27750, 21197, 14746,
     &   9523,  5939,  3584,  2253,  1229,   834,   466,   278,   164,
     &     90,    22, 10500,  4734,  2662,  3994,  9216, 18227, 28877,
     &  34509, 32051, 25088, 17510, 11469,  7168,  4403,  2662,  1536,
     &   1024,   536,   336,   200,    66,    21, 11550,  4466,  2662,
     &   4096,  9626, 20275, 34611, 44954, 44954, 36762, 26419, 17510,
     &  11162,  6861,  4096,  2560,  1434,   819,   612,   160,    20,
     &  12600,  4142,  2662,  4096, 10035, 21811, 39936, 56115, 59904,
     &  51507, 38093, 25805, 16486, 10240,  6246,  3687,  2253,  1331,
     &    202,    18, 13859,  3653,  2662,  4198, 10240, 23347, 45670,
     &  69837, 80998, 73728, 56730, 39322, 25498, 15872,  9830,  5837,
     &   1127,    16, 14700,  3266,  2662,  4198, 10445, 24166, 48845,
     &  78746, 95846, 90829, 71783, 50381, 33075, 20787, 10343,    14,
     &  15750,  2754,  2662,  4198, 10547, 25088, 52429, 89395,115610,
     & 115098, 94003, 67379, 35840,    12, 16800,  2197,  2662,  4198,
     &  10650, 25702, 55706, 99328,136294,142438, 74035,     7, 18900,
     &    852,  2765,  4813, 12902,  7782,     3, 21000,     0,     3,
     &  32767,     0,   705, 22400,     3,     0,     0,     3,  2310,
     &      0,    14,  2520,   615,    37,    18,    19,    18,    17,
     &     17,    16,    14,    13,    12,    11,    19,  2625,  1603,
     &     71,    32,    32,    31,    30,    28,    26,    24,    22,
     &     20,    18,    16,    14,    13,    11,    10,    22,  2730,
     &   2270,   169,    53,    47,    44,    41,    37,    34,    31,
     &     27,    24,    21,    18,    16,    14,    12,    10,     9,
     &      7,     6,    20,  2835,  2642,   275,    82,    68,    63,
     &     57,    50,    44,    38,    33,    28,    23,    20,    16,
     &     14,    11,     9,     8,    20,  2940,  2887,   381,   116,
     &     98,    89,    79,    68,    57,    48,    39,    31,    25/
      DATA (MAPEMAX (I), I=11251,11700) /
     &     20,    16,    13,    10,     7,     7,    20,  3045,  3023,
     &    385,   161,   153,   139,   121,   101,    82,    64,    49,
     &     37,    28,    21,    15,    11,     9,     6,     4,    22,
     &   3150,  3026,   311,   212,   212,   200,   178,   149,   119,
     &     91,    67,    48,    34,    23,    16,    14,     8,     5,
     &      3,     2,     1,    21,  3360,  2856,   315,   258,   273,
     &    274,   260,   233,   197,   158,   121,    89,    64,    44,
     &     30,    22,    17,    11,     6,     3,    20,  3570,  2604,
     &    356,   304,   334,   349,   346,   325,   287,   239,   189,
     &    143,   103,    74,    50,    34,    27,    18,    12,    18,
     &   3780,  2256,   332,   334,   380,   415,   431,   425,   395,
     &    346,   286,   223,   167,   120,    84,    57,    38,    16,
     &   3990,  1809,   283,   359,   424,   480,   519,   533,   518,
     &    472,   405,   327,   250,   183,   128,    14,  4200,  1351,
     &    308,   404,   491,   569,   630,   663,   656,   608,   526,
     &    427,   325,     5,  4620,   317,  1229,     0,     3,  5040,
     &      0,    10,  5250,  1548,  1331,   410,   512,   512,   717,
     &    819,   102,    14,  5459,  2675,  1331,   512,   614,   717,
     &    922,  1024,  1229,  1331,  1331,  1434,   614,    17,  5669,
     &   3422,  1434,   512,   717,   922,  1229,  1434,  1536,  1843,
     &   1843,  1843,  1741,  1638,  1434,   410,    19,  5879,  3868,
     &   1536,   614,   922,  1126,  1536,  1946,  2253,  2355,  2560,
     &   2458,  2253,  2048,  1741,  1434,  1126,   102,    20,  6089,
     &   4229,  1536,   819,  1229,  1638,  2150,  2662,  3072,  3379,
     &   3379,  3072,  2765,  2355,  1843,  1434,  1126,   922,   307,
     &     21,  6300,  4357,  1638,  1024,  1638,  2253,  3072,  3789,
     &   4301,  4506,  4198,  3789,  3072,  2458,  1843,  1434,   922,
     &    717,   590,    10,    22,  6719,  4626,  2253,  2355,  4198,
     &   6246,  7680,  7680,  6451,  4915,  3482,  2355,  1434,  1024,
     &    600,   388,   243,   150,    94,    59,     3,    22,  7139,
     &   4764,  2355,  2867,  5222,  8090,  9830,  9523,  7885,  5734,
     &   3994,  2560,  1536,  1024,   664,   386,   236,   144,    88,
     &     54,    21,    22,  7350,  4811,  2458,  2970,  5734,  8909,
     &  10957, 10547,  8602,  6349,  4198,  2765,  1741,  1126,   626,
     &    403,   245,   148,    90,    54,    28,    22,  7559,  4811,
     &   2458,  3174,  6246,  9830, 12186, 11674,  9421,  6861,  4608,
     &   2970,  1741,  1126,   736,   417,   251,   151,    92,    55,
     &     27,    22,  7979,  4774,  2560,  3379,  7066, 11366, 14438,
     &  14234, 11469,  8294,  5530,  3482,  2253,  1331,   782,   486,
     &    292,   174,   104,    62,    27,    22,  8400,  4712,  2560,
     &   3789,  7782, 13210, 17101, 16896, 13824,  9933,  6554,  4198,
     &   2560,  1536,   922,   568,   332,   197,   117,    69,    20,
     &     22,  8819,  4633,  2560,  3891,  8090, 14438, 19354, 19968,
     &  16691, 12288,  8090,  5222,  3174,  1946,  1229,   688,   418,
     &    248,   148,    87,     7,    21,  9239,  4545,  2662,  3789,
     &   8499, 15462, 21709, 23347, 19968, 14848, 10035,  6451,  3994,
     &   2355,  1536,   819,   528,   310,   183,    88,    21,  9450,
     &   4503,  2662,  3891,  8602, 15974, 22938, 24986, 21811, 16282,
     &  11059,  7168,  4403,  2662,  1638,  1024,   535,   344,   205,
     &     79,    21,  9659,  4466,  2662,  3891,  8704, 16486, 24166/
      DATA (MAPEMAX (I), I=11701,12150) /
     &  26829, 23654, 17818, 12186,  7885,  4915,  2970,  1741,  1126,
     &    622,   382,   227,    69,    21, 10079,  4369,  2662,  3891,
     &   9011, 17408, 26522, 30515, 27750, 21197, 14746,  9523,  5939,
     &   3584,  2253,  1229,   834,   466,   278,    14,    20, 10500,
     &   4261,  2662,  3994,  9216, 18227, 28877, 34509, 32051, 25088,
     &  17510, 11469,  7168,  4403,  2662,  1536,   922,   639,   236,
     &     19, 11550,  3968,  2662,  4096,  9626, 20275, 34611, 44954,
     &  44954, 36762, 26419, 17510, 11162,  6861,  4096,  2458,  1536,
     &    512,    18, 12600,  3641,  2662,  4096, 10035, 21811, 39936,
     &  56115, 59904, 51507, 38093, 25805, 16486, 10138,  6246,  3789,
     &    614,    16, 13859,  3266,  2662,  4198, 10240, 23347, 45670,
     &  69837, 80998, 73728, 56730, 39219, 25498, 15974,  7885,    15,
     &  14700,  2873,  2662,  4198, 10445, 24166, 48845, 78746, 95846,
     &  90829, 71783, 50381, 33075,  5632,    13, 15750,  2398,  2662,
     &   4198, 10547, 25088, 52429, 89395,115610,115098, 94003, 27853,
     &     11, 16800,  1889,  2662,  4198, 10650, 25702, 55603, 99430,
     & 136294, 55296,     6, 18900,   658,  2765,  4813,  5837,     3,
     &  21000,     0,     3, 32767,     0,   565, 25600,     3,     0,
     &      0,     3,  2520,     0,    11,  2625,   669,    71,    32,
     &     32,    31,    30,    28,    26,    24,    13,  2730,  1233,
     &    169,    53,    47,    44,    41,    37,    34,    31,    27,
     &     24,    14,  2835,  1568,   275,    82,    68,    63,    57,
     &     50,    44,    38,    33,    28,    23,    15,  2940,  1806,
     &    381,   116,    98,    89,    79,    68,    57,    48,    39,
     &     31,    25,    20,    16,  3045,  1908,   385,   161,   153,
     &    139,   121,   101,    82,    64,    49,    37,    28,    21,
     &     15,    16,  3150,  1914,   311,   212,   212,   200,   178,
     &    149,   119,    91,    67,    48,    34,    23,    16,    16,
     &   3360,  1842,   315,   258,   273,   274,   260,   233,   197,
     &    158,   121,    89,    64,    44,    30,    15,  3570,  1693,
     &    356,   304,   334,   349,   346,   325,   287,   239,   189,
     &    143,   103,    74,    12,  3780,  1431,   332,   334,   380,
     &    415,   431,   425,   395,   346,   286,     9,  3990,  1066,
     &    283,   359,   424,   480,   519,   533,     6,  4200,   616,
     &    308,   404,   491,     3,  4620,     0,     3,  5040,     0,
     &      5,  5250,   457,  1331,   307,    10,  5459,  1548,  1331,
     &    512,   614,   614,   922,  1126,     0,    14,  5669,  2806,
     &   1434,   512,   717,   922,  1229,  1434,  1536,  1741,  1946,
     &   1843,  1638,    17,  5879,  3368,  1536,   614,   922,  1126,
     &   1536,  1946,  2253,  2355,  2560,  2458,  2253,  2048,  1741,
     &    205,    18,  6089,  3830,  1536,   819,  1229,  1638,  2150,
     &   2662,  3072,  3379,  3379,  3072,  2765,  2355,  1843,  1434,
     &   1126,    19,  6300,  3979,  1638,  1024,  1638,  2253,  3072,
     &   3789,  4301,  4506,  4198,  3789,  3072,  2458,  1741,  1434,
     &   1024,   410,    20,  6719,  4261,  2253,  2355,  4198,  6246,
     &   7680,  7680,  6451,  4915,  3482,  2355,  1434,  1024,   600,
     &    387,   243,   151,    66,    20,  7139,  4344,  2355,  2867,
     &   5222,  8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,
     &   1024,   664,   385,   237,   144,    86,    20,  7350,  4344,
     &   2458,  2970,  5734,  8909, 10957, 10547,  8602,  6349,  4198/
      DATA (MAPEMAX (I), I=12151,12600) /
     &   2765,  1741,  1126,   626,   402,   245,   149,    88,    20,
     &   7559,  4331,  2458,  3174,  6246,  9830, 12186, 11674,  9421,
     &   6861,  4608,  2970,  1741,  1126,   736,   416,   252,   151,
     &     86,    20,  7979,  4245,  2560,  3379,  7066, 11366, 14438,
     &  14234, 11469,  8294,  5530,  3482,  2253,  1331,   782,   485,
     &    292,   175,    67,    20,  8400,  4158,  2560,  3789,  7782,
     &  13210, 17101, 16896, 13824,  9933,  6554,  4198,  2560,  1536,
     &    922,   567,   332,   198,    34,    19,  8819,  4077,  2560,
     &   3891,  8090, 14438, 19354, 19968, 16691, 12288,  8090,  5222,
     &   3174,  1946,  1126,   790,   418,   235,    19,  9239,  3996,
     &   2662,  3789,  8499, 15462, 21709, 23347, 19968, 14848, 10035,
     &   6451,  3994,  2355,  1434,   922,   528,   208,    19,  9450,
     &   3943,  2662,  3891,  8602, 15974, 22938, 24986, 21811, 16282,
     &  11059,  7168,  4403,  2662,  1638,   922,   637,   162,    19,
     &   9659,  3904,  2662,  3891,  8704, 16486, 24166, 26829, 23654,
     &  17818, 12186,  7885,  4915,  2970,  1741,  1024,   724,   116,
     &     18, 10079,  3796,  2662,  3891,  9011, 17408, 26522, 30515,
     &  27750, 21197, 14746,  9523,  5939,  3584,  2150,  1331,   727,
     &     18, 10500,  3699,  2662,  3994,  9216, 18227, 28877, 34509,
     &  32051, 25088, 17510, 11469,  7168,  4301,  2662,  1638,   512,
     &     17, 11550,  3412,  2662,  4096,  9626, 20275, 34611, 44954,
     &  44954, 36762, 26419, 17510, 11059,  6861,  4198,   922,    16,
     &  12600,  3093,  2662,  4096, 10035, 21811, 39936, 56115, 59904,
     &  51507, 38093, 25702, 16486, 10240,   717,    14, 13859,  2706,
     &   2662,  4198, 10240, 23347, 45670, 69837, 80998, 73728, 56730,
     &  39219, 16077,    13, 14700,  2427,  2662,  4198, 10445, 24166,
     &  48845, 78746, 95744, 90931, 71783, 26829,    12, 15750,  2048,
     &   2662,  4198, 10547, 25088, 52429, 89293,115712,115098,     0,
     &     10, 16800,  1629,  2662,  4198, 10650, 25702, 55603, 99430,
     &  46694,     5, 18900,   433,  2765,  2765,     3, 21000,     0,
     &      3, 32767,     0,   428, 28800,     3,     0,     0,     3,
     &   2730,     0,     6,  2835,   389,   275,    82,    68,     8,
     &   2940,   659,   381,   116,    98,    89,    79,     9,  3045,
     &    797,   385,   161,   153,   139,   121,   101,    10,  3150,
     &    818,   311,   212,   212,   200,   178,   149,   119,    11,
     &   3360,   754,   315,   258,   273,   274,   260,   233,   197,
     &    158,    10,  3570,   559,   356,   304,   334,   349,   346,
     &    325,   287,     7,  3780,   149,   332,   334,   380,   415,
     &      3,  3990,     0,     3,  4200,     0,     3,  4600,     0,
     &      3,  5040,     0,     3,  5250,     0,     5,  5459,   285,
     &   1331,     0,    12,  5669,  2110,  1434,   512,   717,   922,
     &   1229,  1331,  1638,  1741,   512,    14,  5879,  2806,  1536,
     &    614,   922,  1126,  1536,  1946,  2253,  2355,  2458,  2560,
     &   2150,    17,  6089,  3380,  1536,   819,  1229,  1638,  2150,
     &   2662,  3072,  3379,  3379,  3072,  2765,  2355,  1843,   307,
     &     18,  6300,  3609,  1638,  1024,  1638,  2253,  3072,  3789,
     &   4301,  4506,  4198,  3789,  3072,  2355,  1843,  1434,   102,
     &     18,  6719,  3821,  2253,  2355,  4198,  6246,  7680,  7680,
     &   6451,  4915,  3482,  2355,  1434,   922,   703,   387,   230,
     &     19,  7139,  3853,  2355,  2867,  5222,  8090,  9830,  9523/
      DATA (MAPEMAX (I), I=12601,13050) /
     &   7885,  5734,  3994,  2560,  1536,  1024,   663,   386,   237,
     &      9,    18,  7350,  3796,  2458,  2970,  5734,  8909, 10957,
     &  10547,  8602,  6349,  4198,  2765,  1741,  1024,   728,   403,
     &    211,    18,  7559,  3769,  2458,  3174,  6246,  9830, 12186,
     &  11674,  9421,  6861,  4608,  2970,  1741,  1126,   735,   417,
     &    195,    18,  7979,  3671,  2560,  3379,  7066, 11366, 14438,
     &  14234, 11469,  8294,  5530,  3482,  2253,  1229,   819,   551,
     &    117,    17,  8400,  3560,  2560,  3789,  7782, 13210, 17101,
     &  16896, 13824,  9933,  6554,  4198,  2458,  1536,  1024,   528,
     &     17,  8819,  3461,  2560,  3891,  8090, 14438, 19354, 19968,
     &  16691, 12288,  8090,  5222,  3174,  1946,  1126,   498,    17,
     &   9239,  3357,  2662,  3789,  8499, 15462, 21709, 23347, 19968,
     &  14848, 10035,  6451,  3891,  2458,  1434,   102,    16,  9450,
     &   3300,  2662,  3891,  8602, 15974, 22938, 24986, 21811, 16282,
     &  11059,  7066,  4403,  2765,  1434,    16,  9659,  3249,  2662,
     &   3891,  8704, 16486, 24166, 26829, 23654, 17818, 12186,  7782,
     &   4915,  2970,  1331,    16, 10079,  3145,  2662,  3891,  9011,
     &  17408, 26522, 30515, 27750, 21197, 14746,  9421,  5939,  3686,
     &    717,    15, 10500,  3030,  2662,  3994,  9216, 18227, 28877,
     &  34509, 32051, 25088, 17510, 11366,  7270,  3789,    14, 11550,
     &   2763,  2662,  4096,  9626, 20275, 34611, 44954, 44954, 36659,
     &  26419, 17613,  9216,    13, 12600,  2464,  2662,  4096, 10035,
     &  21811, 39936, 56115, 59904, 51405, 38093, 17408,    12, 13859,
     &   2110,  2662,  4198, 10240, 23347, 45670, 69837, 80896, 73830,
     &  15463,    11, 14700,  1849,  2662,  4198, 10445, 24166, 48845,
     &  78643, 95846, 21606,     9, 15750,  1489,  2662,  4198, 10547,
     &  24986, 52531, 70554,     8, 16800,  1086,  2662,  4198, 10650,
     &  25702, 10035,     3, 18900,     0,     3, 32767,     0,   266,
     &  31999,     3,     0,     0,     3,  5647,     0,     9,  5669,
     &   1343,  1434,   512,   717,   922,  1126,   410,    12,  5879,
     &   2181,  1536,   614,   922,  1126,  1536,  1843,  2253,  2458,
     &   1331,    15,  6089,  2844,  1536,   819,  1229,  1638,  2150,
     &   2662,  3072,  3379,  3277,  3174,  2765,   307,    16,  6300,
     &   3252,  1638,  1024,  1638,  2253,  3072,  3789,  4301,  4506,
     &   4198,  3686,  3072,  2458,  1434,    17,  6719,  3451,  2253,
     &   2355,  4198,  6246,  7680,  7680,  6451,  4915,  3482,  2355,
     &   1434,   922,   703,   210,    16,  7139,  3307,  2355,  2867,
     &   5222,  8090,  9830,  9523,  7885,  5734,  3994,  2458,  1638,
     &   1024,   625,    16,  7350,  3221,  2458,  2970,  5734,  8909,
     &  10957, 10547,  8602,  6349,  4198,  2765,  1741,  1024,   493,
     &     16,  7559,  3153,  2458,  3174,  6246,  9830, 12186, 11674,
     &   9421,  6861,  4608,  2867,  1843,  1126,   304,    15,  7979,
     &   2994,  2560,  3379,  7066, 11366, 14438, 14234, 11469,  8294,
     &   5427,  3584,  2150,  1024,    15,  8400,  2852,  2560,  3789,
     &   7782, 13210, 17101, 16896, 13824,  9933,  6554,  4096,  2560,
     &    307,    14,  8819,  2706,  2560,  3891,  8090, 14438, 19354,
     &  19968, 16691, 12186,  8192,  5120,  2048,    13,  9239,  2558,
     &   2662,  3789,  8499, 15462, 21709, 23347, 19968, 14746, 10035,
     &   6451,    13,  9450,  2481,  2662,  3891,  8602, 15974, 22938,
     &  24986, 21709, 16384, 11059,  5222,    13,  9659,  2408,  2662/
      DATA (MAPEMAX (I), I=13051,13500) /
     &   3891,  8704, 16486, 24166, 26829, 23552, 17818, 12288,  3584,
     &     12, 10079,  2242,  2662,  3891,  9011, 17408, 26522, 30515,
     &  27648, 21299, 11674,    12, 10500,  2069,  2662,  3994,  9216,
     &  18227, 28877, 34406, 32154, 25088,  1741,    10, 11550,  1662,
     &   2662,  4096,  9626, 20173, 34611, 45056, 22733,     8, 12600,
     &   1233,  2662,  4096,  9933, 21914, 31130,     6, 13859,   648,
     &   2560,  4198,  4403,     4, 14700,   165,  1843,     3, 15750,
     &      0,     3, 32767,     0,   188, 35199,     3,     0,     0,
     &      3,  5647,     0,     6,  5669,   544,  1331,   614,   102,
     &      9,  5879,  1497,  1536,   614,   819,  1229,  1536,  1536,
     &     12,  6089,  2242,  1536,   819,  1229,  1638,  2150,  2560,
     &   3174,  3379,  2560,    15,  6300,  2866,  1638,  1024,  1638,
     &   2253,  3072,  3789,  4301,  4506,  4198,  3686,  3072,   614,
     &     15,  6719,  2866,  2253,  2355,  4198,  6246,  7680,  7680,
     &   6451,  4915,  3379,  2355,  1536,   205,    14,  7139,  2653,
     &   2355,  2867,  5222,  8090,  9830,  9523,  7885,  5734,  3891,
     &   2560,   717,    13,  7350,  2551,  2458,  2970,  5734,  8909,
     &  10957, 10547,  8602,  6246,  4301,  2662,    13,  7559,  2455,
     &   2458,  3174,  6246,  9830, 12186, 11674,  9421,  6861,  4506,
     &   1946,    12,  7979,  2242,  2560,  3379,  7066, 11366, 14438,
     &  14131, 11571,  8192,  4506,    11,  8400,  2025,  2560,  3789,
     &   7782, 13210, 16998, 16998, 13824,  9114,    11,  8819,  1805,
     &   2560,  3891,  8090, 14438, 19251, 20070, 16691,   819,    10,
     &   9239,  1581,  2662,  3789,  8499, 15360, 21811, 23245,  3891,
     &      9,  9450,  1457,  2662,  3891,  8499, 15974, 22938, 17510,
     &      9,  9659,  1343,  2662,  3891,  8704, 16486, 24064,  6656,
     &      8, 10079,  1105,  2662,  3891,  8909, 17510,  7475,     7,
     &  10500,   865,  2560,  3994,  9216,  5734,     3, 11500,     0,
     &      3, 32767,     0,   123, 38399,     3,     0,     0,     3,
     &   5857,     0,     7,  5879,   774,  1434,   614,   922,     0,
     &     10,  6089,  1581,  1536,   819,  1229,  1536,  2150,  2662,
     &    614,    13,  6300,  2356,  1638,  1024,  1638,  2253,  3072,
     &   3789,  4301,  4506,  4198,   819,    12,  6719,  2228,  2253,
     &   2355,  4198,  6246,  7680,  7578,  6451,  4915,  2662,    11,
     &   7139,  1948,  2355,  2867,  5222,  8090,  9830,  9523,  7885,
     &   3686,    10,  7350,  1790,  2458,  2970,  5734,  8909, 10854,
     &  10650,  8499,    10,  7559,  1640,  2458,  3174,  6246,  9830,
     &  12083, 11674,  4198,     9,  7979,  1413,  2560,  3379,  6963,
     &  11469, 14438,  7578,     8,  8400,  1045,  2560,  3686,  7782,
     &  13312,  1331,     6,  8819,   715,  2560,  3789,  6042,     5,
     &   9239,   360,  2560,  1229,     4,  9450,   165,  1843,     4,
     &   9659,     0,  1843,     3, 10080,     0,     3, 32767,     0,
     &     76, 41599,     3,     0,     0,     3,  5857,     0,     4,
     &   5879,     0,  1843,     7,  6089,   845,  1536,   819,  1126,
     &    512,    10,  6300,  1640,  1638,  1024,  1638,  2150,  3072,
     &   3789,  1741,     9,  6719,  1505,  2253,  2355,  4096,  6349,
     &   7680,  6758,     8,  7139,  1204,  2355,  2765,  5325,  8090,
     &   6861,     8,  7350,  1045,  2458,  2867,  5734,  9011,   819,
     &      7,  7559,   884,  2458,  3174,  6246,  3994,     5,  7979,
     &    488,  2458,  3072,     4,  8400,     0,  5530,     3,  8820/
      DATA (MAPEMAX (I), I=13501,13548) /
     &      0,     3, 32767,     0,    43, 44799,     3,     0,     0,
     &      3,  6067,     0,     4,  6089,    81,  1126,     6,  6300,
     &    517,  1638,  1024,     0,     6,  6719,   715,  2150,  2355,
     &   3174,     5,  7139,   407,  2355,  1331,     4,  7350,   248,
     &   2253,     4,  7559,    62,  1229,     3,  7980,     0,     3,
     &  32767,     0,     0/

      END
!--------------------------------------------------------------------------------------------
!+
! NAME:
!	Init_AE8min

! PURPOSE:
!	This subroutine sets block data for AE8min
!
! CATEGORY:
!	Radiation belt model
!
! CALLING SEQUENCE:
!	call Init_AE8min
!
! INPUTS:
!	None
!
! OUTPUTS:
!	None
!
! COMMON BLOCKS:
!	COMMON /ELEMIN/ IHEADEMIN, MAPEMIN
!
! MODIFICATION HISTORY:
!	Written by: D. Bilitza, 1987
!				S. Bourdarie (introduced in version 3.0) - September 2005
!-
!--------------------------------------------------------------------------------------------
!---------------------------------------------------------------------------------------------------
!                              Introduced in version 3.0
!
! CREATION: S. Bourdarie - September 2005
! MODIFICATION: S. Bourdarie - April 2006
!
! DESCRIPTION: Set block data for AE8min
!
! INPUT: None
! OUTPUT: None
!
! CALLING SEQUENCE: call Init_AE8min
!---------------------------------------------------------------------------------------------------
      SUBROUTINE Init_AE8min
!      BLOCK DATA AE8MIN

      COMMON /ELEMIN/ IHEADEMIN, MAPEMIN

      INTEGER*4 IHEADEMIN(8)
      INTEGER*4 MAPEMIN(13168)

      DATA IHEADEMIN /8,4,1964,6400,2100,1024,1024,13168/

      DATA (MAPEMIN (I), I=    1,  450) /
     &   1408,   256,     3,     0,     0,     3,  2310,     0,    27,
     &   2520,  5907,    37,    19,    19,    19,    18,    17,    16,
     &     15,    13,    12,    11,    10,     9,     8,     7,     6,
     &      6,     5,     4,     4,     3,     3,     3,     2,    32,
     &   2625,  7073,    72,    33,    33,    32,    30,    28,    26,
     &     24,    22,    19,    17,    16,    14,    12,    11,     9,
     &      8,     7,     6,     6,     5,     4,     4,     3,     2,
     &      3,     2,     2,     2,    34,  2730,  7700,   171,    55,
     &     49,    46,    42,    38,    34,    30,    27,    23,    20,
     &     18,    15,    13,    11,    10,     8,     7,     6,     5,
     &      4,     4,     2,     3,     2,     1,     2,     2,     1,
     &      1,     0,    35,  2835,  7982,   279,    85,    72,    65,
     &     58,    51,    44,    37,    32,    26,    22,    18,    15,
     &     12,    10,     8,     6,     6,     4,     4,     2,     3,
     &      3,     2,     1,     0,     0,     0,     0,     0,     0,
     &      0,    36,  2940,  8120,   389,   123,   105,    93,    80,
     &     67,    55,    45,    36,    29,    23,    18,    14,    11,
     &      9,     7,     5,     4,     4,     2,     1,     1,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,    37,  3045,  8178,   402,   184,   168,   145,   119,
     &     95,    73,    55,    41,    30,    22,    16,    11,    10,
     &      7,     5,     3,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,    37,  3150,  8219,   354,   261,   245,   212,
     &    172,   131,    97,    69,    48,    33,    23,    16,    13,
     &      8,     4,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,    38,  3360,  8277,   381,   340,   342,
     &    313,   265,   211,   159,   115,    81,    56,    38,    26,
     &     21,    14,     8,     5,     3,     1,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,    38,  3570,  8322,   444,
     &    421,   445,   426,   374,   305,   235,   172,   122,    85,
     &     58,    38,    28,    20,    13,     8,     4,     2,     1,
     &      1,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,    38,  3780,
     &   8312,   442,   490,   547,   552,   506,   427,   336,   251,
     &    179,   124,    85,    57,    38,    31,    20,    12,     7,
     &      4,     2,     1,     1,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &     38,  3990,  8254,   413,   559,   657,   695,   662,   574,
     &    461,   348,   251,   175,   119,    79,    52,    43,    27,
     &     17,     9,     5,     3,     2,     1,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,    38,  4200,  8201,   468,   664,   807,   874,
     &    842,   732,   584,   435,   308,   212,   140,    93,    61,
     &     47,    30,    17,    10,     5,     3,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,    37,  4620,  8110,   530,   795,
     &   1037,  1203,  1239,  1136,   944,   723,   522,   361,   241/
      DATA (MAPEMIN (I), I=  451,  900) /
     &    159,   101,    78,    51,    31,    17,    10,     6,     3,
     &      2,     1,     1,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,    35,  5039,  7986,  1331,
     &    307,   410,   512,   614,   614,   819,   819,   819,   922,
     &    819,   819,   819,   717,   614,   614,   512,   410,   307,
     &    350,   255,   210,   174,   141,   117,    94,    77,    62,
     &     50,    41,    33,     6,    34,  5250,  7926,  1331,   410,
     &    512,   614,   717,   819,   922,  1024,  1024,  1126,  1126,
     &   1024,  1024,   819,   819,   614,   614,   512,   307,   373,
     &    269,   219,   177,   142,   115,    91,    74,    57,    47,
     &     37,    29,    34,  5459,  7856,  1331,   512,   614,   717,
     &    922,  1024,  1229,  1331,  1434,  1434,  1331,  1331,  1229,
     &   1024,   922,   717,   614,   512,   410,   374,   270,   215,
     &    170,   133,   105,    82,    64,    50,    40,    31,    17,
     &     34,  5669,  7761,  1434,   512,   717,   922,  1229,  1434,
     &   1536,  1843,  1843,  1843,  1741,  1638,  1434,  1229,  1024,
     &    819,   717,   512,   406,   329,   256,   198,   151,   118,
     &     90,    69,    52,    40,    31,    24,     6,    34,  5879,
     &   7685,  1536,   614,   922,  1126,  1536,  1946,  2253,  2355,
     &   2560,  2458,  2253,  2048,  1741,  1434,  1229,   819,   819,
     &    512,   417,   322,   244,   184,   138,   103,    78,    58,
     &     43,    33,    24,    18,     0,    33,  6089,  7601,  1536,
     &    819,  1229,  1638,  2150,  2662,  3072,  3379,  3379,  3072,
     &   2765,  2355,  1843,  1536,  1126,   819,   717,   443,   351,
     &    257,   189,   136,   100,    73,    52,    39,    26,    20,
     &     14,     8,    33,  6300,  7528,  1638,  1024,  1638,  2253,
     &   3072,  3789,  4301,  4506,  4198,  3789,  3072,  2458,  1843,
     &   1434,   922,   819,   489,   375,   265,   188,   132,    92,
     &     65,    47,    31,    22,    16,    11,     7,     3,    32,
     &   6719,  7362,  2253,  2355,  4198,  6246,  7680,  7680,  6451,
     &   4915,  3482,  2355,  1434,  1024,   600,   388,   243,   151,
     &     93,    59,    37,    23,    14,     8,     6,     3,     2,
     &      1,     1,     0,     0,    32,  7139,  7323,  2355,  2867,
     &   5222,  8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,
     &   1024,   664,   386,   236,   144,    88,    54,    33,    20,
     &     12,     7,     4,     3,     2,     0,     1,     0,     0,
     &     32,  7350,  7333,  2458,  2970,  5734,  8909, 10957, 10547,
     &   8602,  6349,  4198,  2765,  1741,  1126,   626,   403,   245,
     &    148,    90,    55,    33,    20,    12,     7,     4,     3,
     &      2,     0,     1,     0,     0,    32,  7559,  7362,  2458,
     &   3174,  6246,  9830, 12186, 11674,  9421,  6861,  4608,  2970,
     &   1741,  1126,   736,   417,   251,   152,    91,    55,    33,
     &     20,    12,     7,     4,     3,     1,     1,     0,     1,
     &      0,    33,  7979,  7441,  2560,  3379,  7066, 11366, 14438,
     &  14234, 11469,  8294,  5530,  3482,  2253,  1331,   782,   486,
     &    292,   174,   105,    62,    37,    22,    14,     8,     4,
     &      3,     2,     1,     0,     0,     0,     1,    33,  8400,
     &   7510,  2560,  3789,  7782, 13210, 17101, 16896, 13824,  9933,
     &   6554,  4198,  2560,  1536,   922,   568,   332,   197,   117,
     &     70,    41,    24,    15,     8,     5,     3,     2,     1/
      DATA (MAPEMIN (I), I=  901, 1350) /
     &      0,     1,     0,     0,    33,  8819,  7589,  2560,  3891,
     &   8090, 14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,
     &   1946,  1229,   688,   418,   249,   147,    88,    52,    30,
     &     19,    10,     7,     3,     3,     1,     0,     1,     0,
     &      0,    34,  9239,  7706,  2662,  3789,  8499, 15462, 21709,
     &  23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,   819,
     &    528,   310,   184,   109,    65,    38,    23,    13,     8,
     &      5,     3,     1,     1,     1,     0,     0,     0,    34,
     &   9450,  7753,  2662,  3891,  8602, 15974, 22938, 24986, 21811,
     &  16282, 11059,  7168,  4403,  2662,  1638,  1024,   535,   345,
     &    205,   121,    72,    43,    25,    15,     9,     5,     3,
     &      2,     1,     0,     1,     0,     0,    34,  9659,  7800,
     &   2662,  3891,  8704, 16486, 24166, 26829, 23654, 17818, 12186,
     &   7885,  4915,  2970,  1741,  1126,   622,   382,   227,   135,
     &     80,    47,    28,    17,     9,     6,     4,     2,     1,
     &      0,     1,     0,     0,    34, 10079,  7859,  2662,  3891,
     &   9011, 17408, 26522, 30515, 27750, 21197, 14746,  9523,  5939,
     &   3584,  2253,  1229,   834,   467,   277,   165,    97,    58,
     &     34,    21,    12,     7,     4,     3,     1,     0,     1,
     &      0,     0,    34, 10500,  7907,  2662,  3994,  9216, 18227,
     &  28877, 34509, 32051, 25088, 17510, 11469,  7168,  4403,  2662,
     &   1536,  1024,   537,   336,   199,   118,    70,    42,    25,
     &     14,     9,     5,     3,     2,     0,     1,     0,     1,
     &     35, 11550,  7942,  2662,  4096,  9626, 20275, 34611, 44954,
     &  44954, 36762, 26419, 17510, 11162,  6861,  4096,  2560,  1434,
     &    922,   510,   313,   186,   110,    66,    39,    22,    14,
     &      8,     5,     3,     1,     1,     1,     0,     0,    34,
     &  12600,  7926,  2662,  4096, 10035, 21811, 39936, 56115, 59904,
     &  51507, 38093, 25805, 16486, 10240,  6246,  3687,  2253,  1331,
     &    822,   472,   280,   166,    99,    58,    35,    20,    13,
     &      7,     4,     2,     2,     0,     1,    34, 13859,  7846,
     &   2662,  4198, 10240, 23347, 45670, 69837, 80998, 73728, 56730,
     &  39322, 25498, 15974,  9728,  5939,  3482,  2151,  1229,   728,
     &    441,   262,   155,    92,    55,    32,    19,    12,     6,
     &      4,     2,     2,     0,    34, 14700,  7761,  2662,  4198,
     &  10445, 24166, 48845, 78746, 95846, 90829, 71783, 50483, 33075,
     &  20787, 12698,  7783,  4608,  2765,  1639,  1024,   550,   344,
     &    204,   121,    72,    42,    25,    15,     9,     5,     3,
     &      2,     0,    33, 15750,  7671,  2662,  4198, 10547, 25088,
     &  52429, 89395,115610,115098, 94003, 67482, 44749, 28365, 17408,
     &  10650,  6349,  3789,  2253,  1434,   748,   474,   281,   167,
     &     98,    59,    35,    20,    12,     7,     4,     3,    33,
     &  16800,  7557,  2662,  4198, 10650, 25702, 55706, 99430,136192,
     & 142438,120320, 88269, 59290, 37786, 23450, 14234,  8602,  5120,
     &   3072,  1741,  1126,   636,   379,   225,   133,    79,    47,
     &     27,    16,    10,     6,     2,    32, 18900,  7301,  2765,
     &   4915, 12902, 32870, 75674,142131,200295,210227,175719,126567,
     &  83558, 52429, 31847, 19047, 11264,  6656,  3891,  2253,  1331,
     &    794,   455,   266,   156,    91,    53,    30,    18,    11,
     &      3,    31, 21000,  6996,  3789, 10035, 34918,106701,239616/
      DATA (MAPEMIN (I), I= 1351, 1800) /
     & 342323,323686,235725,148992, 87655, 49562, 27546, 15053,  8294,
     &   4404,  2458,  1331,   725,   389,   211,   114,    62,    33,
     &     18,    10,     5,     3,     1,    29, 23100,  6585, 11162,
     &  95334,448410,664474,473600,256819,126362, 59801, 27853, 12902,
     &   5940,  2764,  1229,   572,   265,   122,    56,    25,    12,
     &      5,     3,     1,     0,     0,     0,     0,     3, 25200,
     &      0,     3, 32767,     0,  1357,   640,     3,     0,     0,
     &      3,  2310,     0,    26,  2520,  5736,    37,    19,    19,
     &     18,    18,    17,    16,    15,    13,    12,    11,    10,
     &      9,     8,     7,     6,     6,     5,     4,     4,     4,
     &      3,     3,    32,  2625,  6885,    72,    33,    33,    32,
     &     30,    28,    26,    24,    22,    20,    17,    16,    14,
     &     12,    11,     9,     8,     7,     6,     5,     5,     4,
     &      4,     3,     2,     3,     2,     2,     2,    34,  2730,
     &   7529,   171,    54,    48,    45,    42,    38,    34,    30,
     &     27,    23,    20,    18,    15,    13,    11,    10,     8,
     &      7,     6,     5,     4,     4,     2,     3,     2,     1,
     &      2,     2,     1,     1,     0,    35,  2835,  7817,   278,
     &     84,    71,    65,    58,    51,    44,    37,    32,    27,
     &     22,    19,    15,    12,    11,     9,     7,     6,     4,
     &      4,     2,     3,     3,     2,     1,     1,     0,     0,
     &      0,     0,     0,     0,    36,  2940,  7965,   387,   121,
     &    104,    92,    80,    67,    56,    45,    37,    29,    23,
     &     19,    14,    11,     9,     7,     6,     4,     4,     2,
     &      1,     1,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,    36,  3045,  8024,   398,   179,
     &    165,   144,   120,    96,    75,    57,    43,    32,    23,
     &     17,    12,    11,     7,     5,     3,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,    36,  3150,  8061,   344,   250,
     &    239,   211,   174,   136,   102,    74,    52,    36,    24,
     &     16,    15,     9,     5,     3,     2,     1,     1,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,    37,  3360,  8118,   365,   321,
     &    327,   307,   268,   218,   169,   124,    88,    63,    43,
     &     29,    21,    16,    10,     6,     3,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,    37,  3570,  8160,   421,
     &    393,   420,   413,   374,   315,   249,   188,   136,    95,
     &     66,    44,    30,    24,    16,     9,     5,     3,     2,
     &      1,     1,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,    37,  3780,  8155,
     &    413,   450,   507,   525,   499,   438,   357,   275,   202,
     &    143,    99,    67,    44,    35,    23,    15,     8,     5,
     &      3,     1,     1,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,    37,  3990,
     &   8081,   378,   505,   598,   649,   643,   584,   491,   385,
     &    285,   205,   141,    96,    64,    46,    33,    22,    12,
     &      7,     4,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,    37/
      DATA (MAPEMIN (I), I= 1801, 2250) /
     &   4200,  8017,   424,   591,   723,   806,   814,   746,   627,
     &    489,   358,   253,   171,   114,    74,    55,    37,    23,
     &     13,     7,     4,     2,     1,     1,     0,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &     37,  4620,  7906,   468,   684,   886,  1055,  1145,  1125,
     &   1005,   824,   626,   452,   312,   209,   137,    90,    70,
     &     44,    26,    15,     8,     5,     3,     1,     1,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,    34,  5039,  7761,  1331,   307,   410,   512,   614,
     &    614,   819,   819,   819,   922,   819,   819,   819,   717,
     &    614,   614,   512,   410,   307,   350,   255,   210,   174,
     &    141,   117,    94,    77,    62,    50,    41,    11,    34,
     &   5250,  7692,  1331,   410,   512,   614,   717,   819,   922,
     &   1024,  1024,  1126,  1126,  1024,  1024,   819,   819,   614,
     &    614,   512,   307,   373,   269,   219,   177,   142,   115,
     &     91,    74,    57,    47,    37,     2,    33,  5459,  7617,
     &   1331,   512,   614,   717,   922,  1024,  1229,  1331,  1434,
     &   1434,  1331,  1331,  1229,  1024,   922,   717,   614,   512,
     &    410,   374,   270,   215,   170,   133,   105,    82,    64,
     &     50,    40,    24,    33,  5669,  7508,  1434,   512,   717,
     &    922,  1229,  1434,  1536,  1843,  1843,  1843,  1741,  1638,
     &   1434,  1229,  1024,   819,   717,   512,   406,   329,   256,
     &    198,   151,   118,    90,    69,    52,    40,    31,     8,
     &     32,  5879,  7390,  1536,   614,   922,  1126,  1536,  1946,
     &   2253,  2355,  2560,  2458,  2253,  2048,  1741,  1434,  1229,
     &    819,   819,   512,   417,   322,   244,   184,   138,   103,
     &     78,    58,    43,    33,    21,    32,  6089,  7249,  1536,
     &    819,  1229,  1638,  2150,  2662,  3072,  3379,  3379,  3072,
     &   2765,  2355,  1843,  1536,  1126,   819,   717,   443,   351,
     &    257,   189,   136,   100,    73,    52,    38,    27,    20,
     &      5,    31,  6300,  7100,  1638,  1024,  1638,  2253,  3072,
     &   3789,  4301,  4506,  4198,  3789,  3072,  2458,  1843,  1434,
     &    922,   819,   489,   375,   265,   188,   132,    92,    65,
     &     47,    31,    22,    16,     8,    31,  6719,  6976,  2253,
     &   2355,  4198,  6246,  7680,  7680,  6451,  4915,  3482,  2355,
     &   1434,  1024,   600,   388,   243,   151,    93,    59,    37,
     &     23,    14,     8,     6,     3,     1,     2,     1,     0,
     &     31,  7139,  6983,  2355,  2867,  5222,  8090,  9830,  9523,
     &   7885,  5734,  3994,  2560,  1536,  1024,   664,   386,   236,
     &    144,    88,    54,    33,    20,    12,     7,     4,     3,
     &      1,     1,     1,     0,    31,  7350,  7009,  2458,  2970,
     &   5734,  8909, 10957, 10547,  8602,  6349,  4198,  2765,  1741,
     &   1126,   626,   403,   245,   148,    90,    55,    33,    20,
     &     12,     7,     4,     3,     1,     1,     1,     0,    31,
     &   7559,  7051,  2458,  3174,  6246,  9830, 12186, 11674,  9421,
     &   6861,  4608,  2970,  1741,  1126,   736,   417,   251,   152,
     &     91,    55,    33,    20,    12,     7,     4,     3,     1,
     &      1,     0,     1,    31,  7979,  7159,  2560,  3379,  7066,
     &  11366, 14438, 14234, 11469,  8294,  5530,  3482,  2253,  1331,
     &    782,   486,   292,   174,   105,    62,    37,    22,    14/
      DATA (MAPEMIN (I), I= 2251, 2700) /
     &      8,     4,     3,     1,     1,     1,     0,    32,  8400,
     &   7291,  2560,  3789,  7782, 13210, 17101, 16896, 13824,  9933,
     &   6554,  4198,  2560,  1536,   922,   568,   332,   197,   117,
     &     70,    41,    24,    15,     8,     5,     3,     2,     1,
     &      0,     1,     0,    32,  8819,  7416,  2560,  3891,  8090,
     &  14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,  1946,
     &   1229,   688,   418,   249,   147,    88,    52,    30,    19,
     &     10,     7,     3,     3,     1,     0,     1,     0,    33,
     &   9239,  7557,  2662,  3789,  8499, 15462, 21709, 23347, 19968,
     &  14848, 10035,  6451,  3994,  2355,  1536,   819,   528,   310,
     &    184,   109,    65,    38,    23,    13,     8,     5,     3,
     &      1,     1,     1,     0,     0,    33,  9450,  7625,  2662,
     &   3891,  8602, 15974, 22938, 24986, 21811, 16282, 11059,  7168,
     &   4403,  2662,  1638,  1024,   535,   345,   205,   121,    72,
     &     43,    25,    15,     9,     5,     3,     2,     1,     0,
     &      1,     0,    33,  9659,  7659,  2662,  3891,  8704, 16486,
     &  24166, 26829, 23654, 17818, 12186,  7885,  4915,  2970,  1741,
     &   1126,   622,   382,   227,   135,    80,    47,    28,    17,
     &      9,     6,     4,     2,     0,     1,     1,     0,    34,
     &  10079,  7730,  2662,  3891,  9011, 17408, 26522, 30515, 27750,
     &  21197, 14746,  9523,  5939,  3584,  2253,  1229,   834,   467,
     &    277,   165,    97,    58,    34,    21,    12,     7,     4,
     &      3,     1,     0,     1,     0,     0,    34, 10500,  7777,
     &   2662,  3994,  9216, 18227, 28877, 34509, 32051, 25088, 17510,
     &  11469,  7168,  4403,  2662,  1536,  1024,   537,   336,   199,
     &    118,    70,    42,    25,    14,     9,     5,     3,     2,
     &      0,     1,     0,     1,    34, 11550,  7787,  2662,  4096,
     &   9626, 20275, 34611, 44954, 44954, 36762, 26419, 17510, 11162,
     &   6861,  4096,  2560,  1434,   922,   510,   313,   186,   110,
     &     66,    39,    22,    14,     8,     5,     3,     1,     1,
     &      1,     0,    34, 12600,  7755,  2662,  4096, 10035, 21811,
     &  39936, 56115, 59904, 51507, 38093, 25805, 16486, 10240,  6246,
     &   3687,  2253,  1331,   822,   472,   280,   166,    99,    58,
     &     35,    20,    13,     7,     4,     2,     2,     0,     1,
     &     33, 13859,  7678,  2662,  4198, 10240, 23347, 45670, 69837,
     &  80998, 73728, 56730, 39322, 25498, 15974,  9728,  5939,  3482,
     &   2151,  1229,   728,   441,   262,   155,    92,    55,    32,
     &     19,    12,     6,     4,     2,     2,    33, 14700,  7557,
     &   2662,  4198, 10445, 24166, 48845, 78746, 95846, 90829, 71783,
     &  50483, 33075, 20787, 12698,  7783,  4608,  2765,  1639,  1024,
     &    550,   344,   204,   121,    72,    42,    25,    15,     8,
     &      6,     3,     1,    32, 15750,  7371,  2662,  4198, 10547,
     &  25088, 52429, 89395,115610,115098, 94003, 67482, 44749, 28365,
     &  17408, 10650,  6349,  3789,  2253,  1434,   748,   474,   281,
     &    167,    98,    59,    35,    20,    12,     7,     4,    31,
     &  16800,  7145,  2662,  4198, 10650, 25702, 55706, 99430,136192,
     & 142438,120320, 88269, 59290, 37786, 23450, 14234,  8602,  5120,
     &   3072,  1741,  1126,   636,   379,   225,   133,    79,    46,
     &     28,    16,     9,    29, 18900,  6523,  2765,  4915, 12902,
     &  32870, 75674,142131,200295,210227,175719,126567, 83558, 52429/
      DATA (MAPEMIN (I), I= 2701, 3150) /
     &  31847, 19047, 11264,  6656,  3891,  2253,  1331,   794,   455,
     &    266,   155,    91,    53,    17,    25, 21000,  5577,  3789,
     &  10035, 34918,106701,239616,342323,323686,235725,148992, 87655,
     &  49562, 27546, 15053,  8294,  4404,  2458,  1331,   725,   389,
     &    210,   114,    52,    20, 23100,  4177, 11162, 95334,448410,
     & 664474,473600,256819,126362, 59801, 27853, 12902,  5940,  2764,
     &   1229,   572,   265,   121,    23,     3, 25200,     0,     3,
     &  32767,     0,  1271,  1600,     3,     0,     0,     3,  2310,
     &      0,    25,  2520,  5285,    37,    18,    19,    18,    18,
     &     17,    16,    14,    13,    12,    11,    10,     9,     8,
     &      7,     7,     6,     5,     5,     4,     4,     3,    31,
     &   2625,  6405,    72,    32,    32,    31,    30,    28,    26,
     &     24,    22,    20,    18,    16,    14,    12,    11,    10,
     &      9,     7,     7,     6,     5,     4,     4,     3,     3,
     &      3,     2,     2,    33,  2730,  6922,   170,    53,    47,
     &     44,    41,    37,    34,    30,    27,    24,    21,    18,
     &     16,    14,    12,    10,     9,     7,     6,     5,     5,
     &      4,     3,     3,     2,     1,     1,     3,     1,     1,
     &     34,  2835,  7159,   276,    82,    68,    63,    57,    50,
     &     44,    38,    33,    28,    23,    19,    16,    13,    11,
     &      9,     8,     6,     4,     5,     3,     2,     3,     2,
     &      1,     1,     0,     0,     0,     0,     0,    34,  2940,
     &   7298,   382,   117,   100,    90,    79,    68,    57,    47,
     &     39,    31,    25,    20,    16,    12,    10,     7,     7,
     &      4,     5,     3,     2,     1,     1,     0,     0,     0,
     &      0,     0,     0,     0,     0,    35,  3045,  7380,   387,
     &    165,   156,   140,   121,   100,    80,    63,    48,    36,
     &     27,    20,    14,    11,     8,     6,     4,     2,     1,
     &      1,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,     0,    35,  3150,  7422,   317,   220,
     &    218,   203,   178,   147,   116,    88,    64,    45,    31,
     &     22,    16,    12,     8,     4,     2,     1,     1,     0,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,     0,    35,  3360,  7476,   324,   270,   285,
     &    283,   264,   232,   192,   151,   113,    83,    58,    41,
     &     27,    21,    15,     9,     5,     3,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,     0,    35,  3570,  7500,   368,   321,   351,   364,
     &    356,   326,   282,   230,   177,   132,    94,    66,    45,
     &     30,    25,    16,    10,     5,     3,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,    36,  3780,  7451,   347,   355,   405,   438,   449,
     &    434,   394,   335,   270,   205,   150,   106,    73,    49,
     &     36,    26,    17,    10,     5,     3,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,    35,  3990,  7348,   300,   384,   456,   514,   549,
     &    554,   523,   463,   384,   301,   224,   160,   111,    75,
     &     49,    41,    26,    16,     9,     5,     3,     2,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &     35,  4200,  7222,   328,   437,   531,   616,   673,   694/
      DATA (MAPEMIN (I), I= 3151, 3600) /
     &    667,   598,   498,   389,   288,   204,   138,    93,    61,
     &     49,    31,    18,    10,     6,     3,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,    34,
     &   4620,  7022,   348,   470,   591,   717,   829,   913,   951,
     &    927,   844,   713,   564,   421,   298,   205,   135,    89,
     &     70,    45,    26,    15,     8,     5,     3,     1,     1,
     &      0,     0,     0,     0,     0,     0,    30,  5039,  6868,
     &   1331,   307,   410,   512,   614,   614,   819,   819,   819,
     &    922,   819,   819,   819,   717,   614,   614,   512,   410,
     &    307,   350,   255,   210,   174,   141,   116,    95,    64,
     &     30,  5250,  6776,  1331,   410,   512,   614,   717,   819,
     &    922,  1024,  1024,  1126,  1126,  1024,  1024,   819,   819,
     &    614,   614,   512,   307,   373,   269,   219,   177,   141,
     &    115,    92,    37,    30,  5459,  6713,  1331,   512,   614,
     &    717,   922,  1024,  1229,  1331,  1434,  1434,  1331,  1331,
     &   1229,  1024,   922,   717,   614,   512,   410,   374,   270,
     &    215,   170,   133,   104,    83,    15,    29,  5669,  6647,
     &   1434,   512,   717,   922,  1229,  1434,  1536,  1843,  1843,
     &   1843,  1741,  1638,  1434,  1229,  1024,   819,   717,   512,
     &    406,   329,   256,   198,   151,   117,    90,    67,    29,
     &   5879,  6601,  1536,   614,   922,  1126,  1536,  1946,  2253,
     &   2355,  2560,  2458,  2253,  2048,  1741,  1434,  1229,   819,
     &    819,   512,   417,   322,   244,   184,   138,   103,    78,
     &     47,    29,  6089,  6568,  1536,   819,  1229,  1638,  2150,
     &   2662,  3072,  3379,  3379,  3072,  2765,  2355,  1843,  1536,
     &   1126,   819,   717,   443,   351,   257,   189,   136,   100,
     &     72,    53,    26,    29,  6300,  6551,  1638,  1024,  1638,
     &   2253,  3072,  3789,  4301,  4506,  4198,  3789,  3072,  2458,
     &   1843,  1434,   922,   819,   489,   375,   265,   188,   132,
     &     92,    65,    46,    33,    14,    29,  6719,  6560,  2253,
     &   2355,  4198,  6246,  7680,  7680,  6451,  4915,  3482,  2355,
     &   1434,  1024,   600,   388,   243,   151,    93,    59,    37,
     &     23,    14,     8,     5,     3,     2,     1,    29,  7139,
     &   6638,  2355,  2867,  5222,  8090,  9830,  9523,  7885,  5734,
     &   3994,  2560,  1536,  1024,   664,   386,   236,   144,    88,
     &     54,    33,    20,    12,     7,     4,     3,     1,     1,
     &     30,  7350,  6685,  2458,  2970,  5734,  8909, 10957, 10547,
     &   8602,  6349,  4198,  2765,  1741,  1126,   626,   403,   245,
     &    148,    90,    55,    33,    20,    12,     7,     4,     3,
     &      1,     1,     0,    30,  7559,  6737,  2458,  3174,  6246,
     &   9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,  1126,
     &    736,   417,   251,   152,    91,    55,    33,    20,    12,
     &      7,     4,     2,     2,     1,     0,    30,  7979,  6853,
     &   2560,  3379,  7066, 11366, 14438, 14234, 11469,  8294,  5530,
     &   3482,  2253,  1331,   782,   486,   292,   174,   105,    62,
     &     37,    22,    14,     8,     4,     3,     1,     1,     1,
     &     31,  8400,  6993,  2560,  3789,  7782, 13210, 17101, 16896,
     &  13824,  9933,  6554,  4198,  2560,  1536,   922,   568,   332,
     &    197,   117,    70,    41,    24,    15,     8,     5,     3,
     &      2,     1,     0,     1,    31,  8819,  7106,  2560,  3891/
      DATA (MAPEMIN (I), I= 3601, 4050) /
     &   8090, 14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,
     &   1946,  1229,   688,   418,   249,   147,    88,    52,    30,
     &     19,    10,     7,     3,     2,     2,     0,     1,    32,
     &   9239,  7241,  2662,  3789,  8499, 15462, 21709, 23347, 19968,
     &  14848, 10035,  6451,  3994,  2355,  1536,   819,   528,   310,
     &    184,   109,    65,    38,    23,    13,     8,     5,     3,
     &      1,     1,     1,     0,    32,  9450,  7333,  2662,  3891,
     &   8602, 15974, 22938, 24986, 21811, 16282, 11059,  7168,  4403,
     &   2662,  1638,  1024,   535,   345,   205,   121,    72,    43,
     &     25,    15,     9,     5,     3,     1,     2,     0,     0,
     &     32,  9659,  7396,  2662,  3891,  8704, 16486, 24166, 26829,
     &  23654, 17818, 12186,  7885,  4915,  2970,  1741,  1126,   622,
     &    382,   227,   135,    80,    47,    28,    17,     9,     6,
     &      4,     1,     1,     1,     1,    33, 10079,  7482,  2662,
     &   3891,  9011, 17408, 26522, 30515, 27750, 21197, 14746,  9523,
     &   5939,  3584,  2253,  1229,   834,   467,   277,   165,    97,
     &     58,    34,    21,    12,     7,     4,     3,     1,     0,
     &      1,     0,    33, 10500,  7528,  2662,  3994,  9216, 18227,
     &  28877, 34509, 32051, 25088, 17510, 11469,  7168,  4403,  2662,
     &   1536,  1024,   537,   336,   199,   118,    70,    42,    25,
     &     14,     9,     5,     3,     1,     1,     1,     0,    33,
     &  11550,  7508,  2662,  4096,  9626, 20275, 34611, 44954, 44954,
     &  36762, 26419, 17510, 11162,  6861,  4096,  2560,  1434,   922,
     &    510,   313,   186,   110,    66,    39,    22,    14,     8,
     &      5,     2,     2,     1,     0,    32, 12600,  7398,  2662,
     &   4096, 10035, 21811, 39936, 56115, 59904, 51507, 38093, 25805,
     &  16486, 10240,  6246,  3687,  2253,  1331,   822,   472,   280,
     &    166,    99,    58,    35,    20,    13,     6,     5,     2,
     &      1,    32, 13859,  7210,  2662,  4198, 10240, 23347, 45670,
     &  69837, 80998, 73728, 56730, 39322, 25498, 15974,  9728,  5939,
     &   3482,  2151,  1229,   728,   441,   262,   155,    92,    55,
     &     32,    19,    11,     7,     4,     0,    31, 14700,  7051,
     &   2662,  4198, 10445, 24166, 48845, 78746, 95846, 90829, 71783,
     &  50483, 33075, 20787, 12698,  7783,  4608,  2765,  1639,  1024,
     &    550,   344,   204,   121,    72,    42,    25,    15,     8,
     &      4,    30, 15750,  6812,  2662,  4198, 10547, 25088, 52429,
     &  89395,115610,115098, 94003, 67482, 44749, 28365, 17408, 10650,
     &   6349,  3789,  2253,  1434,   748,   474,   281,   167,    98,
     &     58,    35,    21,     8,    29, 16800,  6484,  2662,  4198,
     &  10650, 25702, 55706, 99430,136192,142438,120320, 88269, 59290,
     &  37786, 23450, 14234,  8602,  5120,  3072,  1741,  1126,   636,
     &    379,   225,   133,    78,    47,    11,    25, 18900,  5608,
     &   2765,  4915, 12902, 32870, 75674,142131,200295,210227,175719,
     & 126567, 83558, 52429, 31847, 19047, 11264,  6656,  3891,  2253,
     &   1331,   793,   456,   247,    20, 21000,  4229,  3789, 10035,
     &  34918,106701,239616,342323,323686,235725,148992, 87655, 49562,
     &  27546, 15053,  8192,  4506,  2458,   717,    10, 23100,  1617,
     &  11162, 95334,448410,664474,473498,256820, 50790,     3, 25200,
     &      0,     3, 32767,     0,  1163,  3200,     3,     0,     0,
     &      3,  2310,     0,    24,  2520,  4291,    37,    18,    19/
      DATA (MAPEMIN (I), I= 4051, 4500) /
     &     18,    17,    17,    16,    14,    13,    12,    11,    10,
     &      9,     8,     7,     7,     6,     5,     5,     4,     4,
     &     30,  2625,  5471,    71,    32,    32,    31,    30,    28,
     &     26,    24,    22,    20,    18,    16,    14,    13,    11,
     &     10,     9,     7,     7,     6,     5,     4,     4,     3,
     &      3,     3,     2,    32,  2730,  6097,   169,    53,    47,
     &     44,    41,    37,    34,    31,    27,    24,    21,    18,
     &     16,    14,    12,    10,     9,     7,     6,     5,     5,
     &      4,     3,     3,     3,     1,     1,     3,     2,    32,
     &   2835,  6408,   275,    82,    68,    63,    57,    50,    44,
     &     38,    33,    28,    23,    20,    16,    14,    11,     9,
     &      8,     6,     5,     5,     3,     2,     4,     2,     1,
     &      1,     0,     0,     0,    32,  2940,  6617,   381,   116,
     &     98,    89,    79,    68,    57,    48,    39,    31,    25,
     &     20,    16,    13,    10,     7,     7,     4,     5,     3,
     &      2,     1,     1,     0,     0,     0,     0,     0,     0,
     &     32,  3045,  6726,   385,   161,   153,   139,   121,   101,
     &     82,    64,    49,    37,    28,    21,    15,    11,     9,
     &      6,     4,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,    32,  3150,  6771,   311,
     &    212,   212,   200,   178,   149,   119,    91,    67,    48,
     &     34,    23,    16,    14,     8,     5,     3,     2,     1,
     &      0,     0,     0,     0,     0,     0,     0,     0,     0,
     &      0,    32,  3360,  6771,   315,   258,   273,   274,   260,
     &    233,   197,   158,   121,    89,    64,    44,    30,    22,
     &     17,    11,     6,     3,     2,     1,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,    31,  3570,  6695,
     &    356,   304,   334,   349,   346,   325,   287,   239,   189,
     &    143,   103,    74,    50,    34,    27,    18,    12,     7,
     &      4,     2,     1,     1,     0,     0,     0,     0,     0,
     &      0,    32,  3780,  6526,   332,   334,   380,   415,   431,
     &    425,   395,   346,   286,   223,   167,   120,    84,    57,
     &     38,    32,    20,    12,     7,     4,     2,     1,     1,
     &      0,     0,     0,     0,     0,     0,    32,  3990,  6321,
     &    283,   359,   424,   480,   519,   533,   518,   472,   405,
     &    327,   250,   183,   128,    89,    59,    44,    31,    20,
     &     11,     6,     4,     2,     1,     1,     0,     0,     0,
     &      0,     0,    31,  4200,  6148,   308,   404,   491,   569,
     &    630,   663,   656,   608,   526,   427,   325,   236,   164,
     &    111,    73,    55,    37,    23,    13,     7,     4,     2,
     &      1,     1,     0,     0,     0,     0,    30,  4620,  6058,
     &    324,   429,   538,   646,   751,   840,   896,   905,   858,
     &    762,   631,   489,   362,   253,   172,   113,    82,    57,
     &     37,    21,    12,     7,     4,     2,     1,     1,     0,
     &     27,  5039,  6010,  1331,   307,   410,   512,   614,   614,
     &    819,   819,   819,   922,   819,   819,   819,   717,   614,
     &    614,   512,   410,   307,   350,   254,   211,   173,    72,
     &     27,  5250,  5985,  1331,   410,   512,   614,   717,   819,
     &    922,  1024,  1024,  1126,  1126,  1024,  1024,   819,   819,
     &    614,   614,   512,   307,   373,   268,   219,   177,    58/
      DATA (MAPEMIN (I), I= 4501, 4950) /
     &     27,  5459,  5965,  1331,   512,   614,   717,   922,  1024,
     &   1229,  1331,  1434,  1434,  1331,  1331,  1229,  1024,   922,
     &    717,   614,   512,   410,   374,   270,   214,   170,    44,
     &     27,  5669,  5952,  1434,   512,   717,   922,  1229,  1434,
     &   1536,  1843,  1843,  1843,  1741,  1638,  1434,  1229,  1024,
     &    819,   717,   512,   406,   329,   255,   198,   152,    33,
     &     27,  5879,  5952,  1536,   614,   922,  1126,  1536,  1946,
     &   2253,  2355,  2560,  2458,  2253,  2048,  1741,  1434,  1229,
     &    819,   819,   512,   417,   322,   244,   184,   138,    29,
     &     27,  6089,  5978,  1536,   819,  1229,  1638,  2150,  2662,
     &   3072,  3379,  3379,  3072,  2765,  2355,  1843,  1536,  1126,
     &    819,   717,   443,   351,   257,   188,   137,   100,    28,
     &     27,  6300,  6033,  1638,  1024,  1638,  2253,  3072,  3789,
     &   4301,  4506,  4198,  3789,  3072,  2458,  1843,  1434,   922,
     &    819,   489,   375,   265,   188,   131,    93,    65,    28,
     &     27,  6719,  6143,  2253,  2355,  4198,  6246,  7680,  7680,
     &   6451,  4915,  3482,  2355,  1434,  1024,   600,   388,   243,
     &    151,    93,    59,    37,    23,    13,     9,     5,     3,
     &     28,  7139,  6277,  2355,  2867,  5222,  8090,  9830,  9523,
     &   7885,  5734,  3994,  2560,  1536,  1024,   664,   386,   236,
     &    144,    88,    54,    33,    20,    12,     7,     4,     3,
     &      1,    28,  7350,  6338,  2458,  2970,  5734,  8909, 10957,
     &  10547,  8602,  6349,  4198,  2765,  1741,  1126,   626,   403,
     &    245,   148,    90,    55,    33,    20,    12,     7,     4,
     &      3,     1,    28,  7559,  6392,  2458,  3174,  6246,  9830,
     &  12186, 11674,  9421,  6861,  4608,  2970,  1741,  1126,   736,
     &    417,   251,   152,    91,    55,    33,    20,    12,     7,
     &      4,     2,     2,    29,  7979,  6523,  2560,  3379,  7066,
     &  11366, 14438, 14234, 11469,  8294,  5530,  3482,  2253,  1331,
     &    782,   486,   292,   174,   105,    62,    37,    22,    14,
     &      8,     4,     3,     1,     1,    30,  8400,  6674,  2560,
     &   3789,  7782, 13210, 17101, 16896, 13824,  9933,  6554,  4198,
     &   2560,  1536,   922,   568,   332,   197,   117,    70,    41,
     &     24,    15,     8,     5,     3,     2,     1,     0,    30,
     &   8819,  6792,  2560,  3891,  8090, 14438, 19354, 19968, 16691,
     &  12288,  8090,  5222,  3174,  1946,  1229,   688,   418,   249,
     &    147,    88,    52,    30,    19,    10,     7,     3,     2,
     &      2,     0,    31,  9239,  6928,  2662,  3789,  8499, 15462,
     &  21709, 23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,
     &    819,   528,   310,   184,   109,    65,    38,    23,    13,
     &      8,     5,     2,     2,     1,     0,    31,  9450,  6996,
     &   2662,  3891,  8602, 15974, 22938, 24986, 21811, 16282, 11059,
     &   7168,  4403,  2662,  1638,  1024,   535,   345,   205,   121,
     &     72,    43,    25,    15,     9,     5,     3,     1,     2,
     &      0,    31,  9659,  7053,  2662,  3891,  8704, 16486, 24166,
     &  26829, 23654, 17818, 12186,  7885,  4915,  2970,  1741,  1126,
     &    622,   382,   227,   135,    80,    47,    28,    17,     9,
     &      6,     3,     2,     1,     1,    31, 10079,  7112,  2662,
     &   3891,  9011, 17408, 26522, 30515, 27750, 21197, 14746,  9523,
     &   5939,  3584,  2253,  1229,   834,   467,   277,   165,    97/
      DATA (MAPEMIN (I), I= 4951, 5400) /
     &     58,    34,    21,    12,     7,     4,     2,     2,     0,
     &     31, 10500,  7140,  2662,  3994,  9216, 18227, 28877, 34509,
     &  32051, 25088, 17510, 11469,  7168,  4403,  2662,  1536,  1024,
     &    537,   336,   199,   118,    70,    42,    25,    14,     9,
     &      5,     3,     1,     1,    31, 11550,  7057,  2662,  4096,
     &   9626, 20275, 34611, 44954, 44954, 36762, 26419, 17510, 11162,
     &   6861,  4096,  2560,  1434,   922,   510,   313,   186,   110,
     &     66,    39,    22,    14,     8,     4,     3,     1,    30,
     &  12600,  6885,  2662,  4096, 10035, 21811, 39936, 56115, 59904,
     &  51507, 38093, 25805, 16486, 10240,  6246,  3687,  2253,  1331,
     &    822,   472,   280,   166,    99,    58,    35,    20,    12,
     &      7,     4,    29, 13859,  6617,  2662,  4198, 10240, 23347,
     &  45670, 69837, 80998, 73728, 56730, 39322, 25498, 15974,  9728,
     &   5939,  3482,  2151,  1229,   728,   441,   262,   155,    92,
     &     54,    33,    19,    10,    29, 14700,  6417,  2662,  4198,
     &  10445, 24166, 48845, 78746, 95846, 90829, 71783, 50483, 33075,
     &  20787, 12698,  7783,  4608,  2765,  1639,  1024,   550,   344,
     &    204,   121,    71,    42,    26,     1,    27, 15750,  6071,
     &   2662,  4198, 10547, 25088, 52429, 89395,115610,115098, 94003,
     &  67482, 44749, 28365, 17408, 10650,  6349,  3789,  2253,  1434,
     &    748,   474,   281,   166,    99,    45,    25, 16800,  5630,
     &   2662,  4198, 10650, 25702, 55706, 99430,136192,142438,120320,
     &  88269, 59290, 37786, 23450, 14234,  8602,  5120,  3072,  1741,
     &   1024,   738,   379,   223,    20, 18900,  4290,  2765,  4915,
     &  12902, 32870, 75674,142131,200295,210227,175719,126567, 83558,
     &  52429, 31847, 18944, 11264,  6656,  3175,    12, 21000,  2136,
     &   3789, 10035, 34918,106701,239616,342221,323789,235725, 60109,
     &      3, 23100,     0,     3, 32767,     0,  1102,  4800,     3,
     &      0,     0,     3,  2310,     0,    23,  2520,  3596,    37,
     &     18,    19,    18,    17,    17,    16,    14,    13,    12,
     &     11,    10,     9,     8,     7,     7,     6,     5,     5,
     &      4,    29,  2625,  4794,    71,    32,    32,    31,    30,
     &     28,    26,    24,    22,    20,    18,    16,    14,    13,
     &     11,    10,     9,     7,     7,     6,     5,     4,     4,
     &      3,     3,     3,    31,  2730,  5470,   169,    53,    47,
     &     44,    41,    37,    34,    31,    27,    24,    21,    18,
     &     16,    14,    12,    10,     9,     7,     6,     5,     5,
     &      4,     3,     3,     3,     1,     1,     3,    31,  2835,
     &   5894,   275,    82,    68,    63,    57,    50,    44,    38,
     &     33,    28,    23,    20,    16,    14,    11,     9,     8,
     &      6,     5,     5,     3,     2,     4,     2,     1,     1,
     &      0,     0,    31,  2940,  6206,   381,   116,    98,    89,
     &     79,    68,    57,    48,    39,    31,    25,    20,    16,
     &     13,    10,     7,     7,     4,     5,     3,     2,     1,
     &      1,     0,     0,     0,     0,     0,    31,  3045,  6333,
     &    385,   161,   153,   139,   121,   101,    82,    64,    49,
     &     37,    28,    21,    15,    11,     9,     6,     4,     2,
     &      1,     1,     0,     0,     0,     0,     0,     0,     0,
     &      0,    31,  3150,  6367,   311,   212,   212,   200,   178,
     &    149,   119,    91,    67,    48,    34,    23,    16,    14/
      DATA (MAPEMIN (I), I= 5401, 5850) /
     &      8,     5,     3,     2,     1,     0,     0,     0,     0,
     &      0,     0,     0,     0,     0,    30,  3360,  6315,   315,
     &    258,   273,   274,   260,   233,   197,   158,   121,    89,
     &     64,    44,    30,    22,    17,    11,     6,     3,     2,
     &      1,     1,     0,     0,     0,     0,     0,     0,    30,
     &   3570,  6161,   356,   304,   334,   349,   346,   325,   287,
     &    239,   189,   143,   103,    74,    50,    34,    27,    18,
     &     12,     7,     4,     2,     1,     1,     0,     0,     0,
     &      0,     0,    29,  3780,  5931,   332,   334,   380,   415,
     &    431,   425,   395,   346,   286,   223,   167,   120,    84,
     &     57,    38,    32,    20,    12,     7,     4,     2,     1,
     &      1,     0,     0,     0,    28,  3990,  5692,   283,   359,
     &    424,   480,   519,   533,   518,   472,   405,   327,   250,
     &    183,   128,    89,    59,    44,    31,    20,    11,     6,
     &      4,     2,     1,     1,     0,    28,  4200,  5513,   308,
     &    404,   491,   569,   630,   663,   656,   608,   526,   427,
     &    325,   236,   164,   111,    73,    55,    37,    23,    13,
     &      7,     4,     2,     1,     1,     0,    27,  4620,  5359,
     &    324,   429,   538,   646,   751,   840,   896,   905,   858,
     &    762,   631,   489,   362,   253,   172,   113,    82,    57,
     &     37,    21,    12,     7,     4,     2,    24,  5039,  5342,
     &   1331,   307,   410,   512,   614,   614,   819,   819,   819,
     &    922,   819,   819,   819,   717,   614,   614,   512,   410,
     &    307,   350,   223,    25,  5250,  5439,  1331,   410,   512,
     &    614,   717,   819,   922,  1024,  1024,  1126,  1126,  1024,
     &   1024,   819,   819,   614,   614,   512,   307,   372,   269,
     &     58,    25,  5459,  5577,  1331,   512,   614,   717,   922,
     &   1024,  1229,  1331,  1434,  1434,  1331,  1331,  1229,  1024,
     &    922,   717,   614,   512,   410,   374,   270,   173,    25,
     &   5669,  5608,  1434,   512,   717,   922,  1229,  1434,  1536,
     &   1843,  1843,  1843,  1741,  1638,  1434,  1229,  1024,   819,
     &    717,   512,   406,   329,   255,   182,    26,  5879,  5637,
     &   1536,   614,   922,  1126,  1536,  1946,  2253,  2355,  2560,
     &   2458,  2253,  2048,  1741,  1434,  1229,   819,   819,   512,
     &    417,   322,   244,   184,     3,    26,  6089,  5678,  1536,
     &    819,  1229,  1638,  2150,  2662,  3072,  3379,  3379,  3072,
     &   2765,  2355,  1843,  1536,  1126,   819,   717,   443,   351,
     &    257,   188,   137,    20,    26,  6300,  5767,  1638,  1024,
     &   1638,  2253,  3072,  3789,  4301,  4506,  4198,  3789,  3072,
     &   2458,  1843,  1434,   922,   819,   489,   375,   265,   187,
     &    132,    93,    38,    27,  6719,  5934,  2253,  2355,  4198,
     &   6246,  7680,  7680,  6451,  4915,  3482,  2355,  1434,  1024,
     &    600,   388,   243,   151,    93,    59,    37,    23,    13,
     &      9,     5,     1,    27,  7139,  6033,  2355,  2867,  5222,
     &   8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,  1024,
     &    664,   386,   236,   144,    88,    54,    33,    20,    11,
     &      8,     4,     2,    27,  7350,  6105,  2458,  2970,  5734,
     &   8909, 10957, 10547,  8602,  6349,  4198,  2765,  1741,  1126,
     &    626,   403,   245,   148,    90,    55,    33,    20,    11,
     &      8,     4,     2,    28,  7559,  6225,  2458,  3174,  6246/
      DATA (MAPEMIN (I), I= 5851, 6300) /
     &   9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,  1126,
     &    736,   417,   251,   152,    91,    55,    33,    20,    12,
     &      7,     4,     2,     1,    28,  7979,  6243,  2560,  3379,
     &   7066, 11366, 14438, 14234, 11469,  8294,  5530,  3482,  2253,
     &   1331,   782,   486,   292,   174,   105,    62,    37,    22,
     &     14,     7,     5,     3,     0,    28,  8400,  6551,  2560,
     &   3789,  7782, 13210, 17101, 16896, 13824,  9933,  6554,  4198,
     &   2560,  1536,   922,   568,   332,   197,   117,    70,    41,
     &     24,    15,     8,     5,     3,     2,    29,  8819,  6623,
     &   2560,  3891,  8090, 14438, 19354, 19968, 16691, 12288,  8090,
     &   5222,  3174,  1946,  1229,   688,   418,   249,   147,    88,
     &     52,    30,    19,    10,     6,     4,     2,     2,    30,
     &   9239,  6686,  2662,  3789,  8499, 15462, 21709, 23347, 19968,
     &  14848, 10035,  6451,  3994,  2355,  1536,   819,   528,   310,
     &    184,   109,    65,    38,    23,    13,     8,     5,     2,
     &      2,     0,    30,  9450,  6767,  2662,  3891,  8602, 15974,
     &  22938, 24986, 21811, 16282, 11059,  7168,  4403,  2662,  1638,
     &   1024,   535,   345,   205,   121,    72,    43,    25,    15,
     &      9,     4,     4,     1,     1,    30,  9659,  6803,  2662,
     &   3891,  8704, 16486, 24166, 26829, 23654, 17818, 12186,  7885,
     &   4915,  2970,  1741,  1126,   622,   382,   227,   135,    80,
     &     47,    28,    17,     9,     6,     3,     2,     1,    30,
     &  10079,  6835,  2662,  3891,  9011, 17408, 26522, 30515, 27750,
     &  21197, 14746,  9523,  5939,  3584,  2253,  1229,   834,   467,
     &    277,   165,    97,    58,    34,    21,    12,     6,     5,
     &      2,     1,    30, 10500,  6841,  2662,  3994,  9216, 18227,
     &  28877, 34509, 32051, 25088, 17510, 11469,  7168,  4403,  2662,
     &   1536,  1024,   537,   336,   199,   118,    70,    42,    25,
     &     14,     8,     6,     3,     1,    30, 11550,  6668,  2662,
     &   4096,  9626, 20275, 34611, 44954, 44954, 36762, 26419, 17510,
     &  11162,  6861,  4096,  2560,  1434,   922,   510,   313,   186,
     &    110,    66,    39,    22,    14,     8,     4,     1,    29,
     &  12600,  6452,  2662,  4096, 10035, 21811, 39936, 56115, 59904,
     &  51507, 38093, 25805, 16486, 10240,  6246,  3687,  2253,  1331,
     &    822,   472,   280,   166,    99,    58,    34,    21,    12,
     &      2,    28, 13859,  6206,  2662,  4198, 10240, 23347, 45670,
     &  69837, 80998, 73728, 56730, 39322, 25498, 15974,  9728,  5939,
     &   3482,  2151,  1229,   728,   441,   262,   155,    92,    54,
     &     33,     5,    27, 14700,  5894,  2662,  4198, 10445, 24166,
     &  48845, 78746, 95846, 90829, 71783, 50483, 33075, 20787, 12698,
     &   7783,  4608,  2765,  1639,  1024,   550,   344,   203,   121,
     &     72,     1,    25, 15750,  5509,  2662,  4198, 10547, 25088,
     &  52429, 89395,115610,115098, 94003, 67482, 44749, 28365, 17408,
     &  10650,  6349,  3789,  2253,  1434,   748,   474,   281,    97,
     &     23, 16800,  5020,  2662,  4198, 10650, 25702, 55706, 99430,
     & 136192,142438,120320, 88269, 59290, 37786, 23450, 14234,  8602,
     &   5120,  2970,  1843,  1024,   529,    17, 18900,  3560,  2765,
     &   4915, 12902, 32870, 75674,142131,200295,210227,175719,126567,
     &  83558, 52327, 31847, 17715,     9, 21000,  1332,  3789, 10035,
     &  34816,106701,239718, 65126,     3, 23100,     0,     3, 32767/
      DATA (MAPEMIN (I), I= 6301, 6750) /
     &      0,  1058,  6400,     3,     0,     0,     3,  2310,     0,
     &     23,  2520,  3149,    37,    18,    19,    18,    17,    17,
     &     16,    14,    13,    12,    11,    10,     9,     8,     7,
     &      7,     6,     5,     5,     4,    28,  2625,  4466,    71,
     &     32,    32,    31,    30,    28,    26,    24,    22,    20,
     &     18,    16,    14,    13,    11,    10,     9,     7,     7,
     &      6,     5,     4,     4,     3,     3,    30,  2730,  5120,
     &    169,    53,    47,    44,    41,    37,    34,    31,    27,
     &     24,    21,    18,    16,    14,    12,    10,     9,     7,
     &      6,     5,     5,     4,     3,     3,     3,     1,     1,
     &     31,  2835,  5586,   275,    82,    68,    63,    57,    50,
     &     44,    38,    33,    28,    23,    20,    16,    14,    11,
     &      9,     8,     6,     5,     5,     3,     2,     4,     2,
     &      1,     1,     0,     0,    31,  2940,  5917,   381,   116,
     &     98,    89,    79,    68,    57,    48,    39,    31,    25,
     &     20,    16,    13,    10,     7,     7,     4,     5,     3,
     &      2,     1,     1,     0,     0,     0,     0,     0,    31,
     &   3045,  6066,   385,   161,   153,   139,   121,   101,    82,
     &     64,    49,    37,    28,    21,    15,    11,     9,     6,
     &      4,     2,     1,     1,     0,     0,     0,     0,     0,
     &      0,     0,     0,    31,  3150,  6077,   311,   212,   212,
     &    200,   178,   149,   119,    91,    67,    48,    34,    23,
     &     16,    14,     8,     5,     3,     2,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,     0,    30,  3360,
     &   5985,   315,   258,   273,   274,   260,   233,   197,   158,
     &    121,    89,    64,    44,    30,    22,    17,    11,     6,
     &      3,     2,     1,     1,     0,     0,     0,     0,     0,
     &      0,    29,  3570,  5803,   356,   304,   334,   349,   346,
     &    325,   287,   239,   189,   143,   103,    74,    50,    34,
     &     27,    18,    12,     7,     4,     2,     1,     1,     0,
     &      0,     0,     0,    28,  3780,  5558,   332,   334,   380,
     &    415,   431,   425,   395,   346,   286,   223,   167,   120,
     &     84,    57,    38,    32,    20,    12,     7,     4,     2,
     &      1,     1,     0,     0,    26,  3990,  5253,   283,   359,
     &    424,   480,   519,   533,   518,   472,   405,   327,   250,
     &    183,   128,    89,    59,    44,    31,    20,    11,     6,
     &      4,     2,     1,    26,  4200,  5010,   308,   404,   491,
     &    569,   630,   663,   656,   608,   526,   427,   325,   236,
     &    164,   111,    73,    55,    37,    23,    13,     7,     4,
     &      2,     1,    24,  4620,  4713,   324,   429,   538,   646,
     &    751,   840,   896,   905,   858,   762,   631,   489,   362,
     &    253,   172,   113,    82,    57,    37,    21,    12,    22,
     &   5039,  4811,  1331,   307,   410,   512,   614,   614,   819,
     &    819,   819,   922,   819,   819,   819,   717,   614,   512,
     &    512,   410,   307,    23,  5250,  4992,  1331,   410,   512,
     &    614,   717,   819,   922,  1024,  1024,  1126,  1126,  1024,
     &   1024,   819,   819,   614,   512,   512,   410,   215,    24,
     &   5459,  5182,  1331,   512,   614,   717,   922,  1024,  1229,
     &   1331,  1434,  1434,  1331,  1331,  1229,  1024,   922,   717,
     &    614,   512,   410,   374,    72,    24,  5669,  5300,  1434/
      DATA (MAPEMIN (I), I= 6751, 7200) /
     &    512,   717,   922,  1229,  1434,  1536,  1843,  1843,  1843,
     &   1741,  1638,  1434,  1229,  1024,   819,   717,   410,   509,
     &    329,   186,    25,  5879,  5405,  1536,   614,   922,  1126,
     &   1536,  1946,  2253,  2355,  2560,  2458,  2253,  2048,  1741,
     &   1434,  1229,   819,   819,   512,   417,   322,   244,    24,
     &     25,  6089,  5499,  1536,   819,  1229,  1638,  2150,  2662,
     &   3072,  3379,  3379,  3072,  2765,  2355,  1843,  1536,  1126,
     &    819,   717,   443,   351,   257,   188,    72,    25,  6300,
     &   5569,  1638,  1024,  1638,  2253,  3072,  3789,  4301,  4506,
     &   4198,  3789,  3072,  2458,  1843,  1434,   922,   819,   489,
     &    375,   265,   187,   132,    73,    26,  6719,  5736,  2253,
     &   2355,  4198,  6246,  7680,  7680,  6451,  4915,  3482,  2355,
     &   1434,  1024,   600,   388,   243,   151,    93,    59,    37,
     &     22,    14,     9,     2,    27,  7139,  5894,  2355,  2867,
     &   5222,  8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,
     &   1024,   664,   386,   236,   144,    88,    54,    33,    20,
     &     11,     8,     4,     0,    27,  7350,  5965,  2458,  2970,
     &   5734,  8909, 10957, 10547,  8602,  6349,  4198,  2765,  1741,
     &   1126,   626,   403,   245,   148,    90,    55,    33,    20,
     &     11,     8,     4,     1,    27,  7559,  6033,  2458,  3174,
     &   6246,  9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,
     &   1126,   736,   417,   251,   152,    91,    55,    33,    20,
     &     11,     8,     4,     1,    28,  7979,  6165,  2560,  3379,
     &   7066, 11366, 14438, 14234, 11469,  8294,  5530,  3482,  2253,
     &   1331,   782,   486,   292,   174,   105,    62,    37,    22,
     &     14,     7,     5,     3,     0,    28,  8400,  6293,  2560,
     &   3789,  7782, 13210, 17101, 16896, 13824,  9933,  6554,  4198,
     &   2560,  1536,   922,   568,   332,   197,   117,    70,    41,
     &     24,    15,     8,     5,     3,     1,    29,  8819,  6417,
     &   2560,  3891,  8090, 14438, 19354, 19968, 16691, 12288,  8090,
     &   5222,  3174,  1946,  1229,   688,   418,   249,   147,    88,
     &     52,    30,    19,    10,     6,     4,     2,     0,    29,
     &   9239,  6533,  2662,  3789,  8499, 15462, 21709, 23347, 19968,
     &  14848, 10035,  6451,  3994,  2355,  1536,   819,   528,   310,
     &    184,   109,    65,    38,    23,    13,     8,     5,     2,
     &      1,    29,  9450,  6577,  2662,  3891,  8602, 15974, 22938,
     &  24986, 21811, 16282, 11059,  7168,  4403,  2662,  1638,  1024,
     &    535,   345,   205,   121,    72,    43,    25,    15,     8,
     &      5,     4,     1,    29,  9659,  6617,  2662,  3891,  8704,
     &  16486, 24166, 26829, 23654, 17818, 12186,  7885,  4915,  2970,
     &   1741,  1126,   622,   382,   227,   135,    80,    47,    28,
     &     17,     9,     6,     3,     2,    29, 10079,  6625,  2662,
     &   3891,  9011, 17408, 26522, 30515, 27750, 21197, 14746,  9523,
     &   5939,  3584,  2253,  1229,   834,   467,   277,   165,    97,
     &     58,    34,    21,    11,     7,     5,     2,    29, 10500,
     &   6593,  2662,  3994,  9216, 18227, 28877, 34509, 32051, 25088,
     &  17510, 11469,  7168,  4403,  2662,  1536,  1024,   537,   336,
     &    199,   118,    70,    42,    25,    14,     8,     6,     2,
     &     28, 11550,  6392,  2662,  4096,  9626, 20275, 34611, 44954,
     &  44954, 36762, 26419, 17510, 11162,  6861,  4096,  2560,  1434/
      DATA (MAPEMIN (I), I= 7201, 7650) /
     &    922,   510,   313,   186,   110,    66,    38,    23,    14,
     &      7,    27, 12600,  6143,  2662,  4096, 10035, 21811, 39936,
     &  56115, 59904, 51507, 38093, 25805, 16486, 10240,  6246,  3687,
     &   2253,  1331,   822,   472,   280,   166,    98,    59,    34,
     &     21,    26, 13859,  5817,  2662,  4198, 10240, 23347, 45670,
     &  69837, 80998, 73728, 56730, 39322, 25498, 15974,  9728,  5939,
     &   3482,  2151,  1229,   728,   441,   262,   155,    92,    42,
     &     25, 14700,  5544,  2662,  4198, 10445, 24166, 48845, 78746,
     &  95846, 90829, 71783, 50483, 33075, 20787, 12698,  7783,  4608,
     &   2765,  1639,  1024,   549,   344,   204,    87,    23, 15750,
     &   5119,  2662,  4198, 10547, 25088, 52429, 89395,115610,115098,
     &  94003, 67482, 44749, 28365, 17408, 10650,  6349,  3789,  2253,
     &   1331,   850,   474,    21, 16800,  4569,  2662,  4198, 10650,
     &  25702, 55706, 99430,136192,142438,120320, 88269, 59290, 37786,
     &  23450, 14234,  8499,  5120,  3072,  1639,    15, 18900,  2944,
     &   2765,  4915, 12902, 32870, 75674,142131,200295,210227,175616,
     & 126669, 83558, 29287,     6, 21000,   715,  3686, 10035, 24064,
     &      3, 23100,     0,     3, 32767,     0,   982,  9600,     3,
     &      0,     0,     3,  2310,     0,    22,  2520,  2572,    37,
     &     18,    19,    18,    17,    17,    16,    14,    13,    12,
     &     11,    10,     9,     8,     7,     7,     6,     5,     5,
     &     27,  2625,  3869,    71,    32,    32,    31,    30,    28,
     &     26,    24,    22,    20,    18,    16,    14,    13,    11,
     &     10,     9,     7,     7,     6,     5,     4,     4,     3,
     &     30,  2730,  4606,   169,    53,    47,    44,    41,    37,
     &     34,    31,    27,    24,    21,    18,    16,    14,    12,
     &     10,     9,     7,     6,     5,     5,     4,     3,     3,
     &      3,     1,     1,    30,  2835,  5099,   275,    82,    68,
     &     63,    57,    50,    44,    38,    33,    28,    23,    20,
     &     16,    14,    11,     9,     8,     6,     5,     5,     3,
     &      2,     4,     2,     1,     1,     0,    30,  2940,  5470,
     &    381,   116,    98,    89,    79,    68,    57,    48,    39,
     &     31,    25,    20,    16,    13,    10,     7,     7,     4,
     &      5,     3,     2,     1,     1,     0,     0,     0,     0,
     &     30,  3045,  5664,   385,   161,   153,   139,   121,   101,
     &     82,    64,    49,    37,    28,    21,    15,    11,     9,
     &      6,     4,     2,     1,     1,     0,     0,     0,     0,
     &      0,     0,     0,    30,  3150,  5690,   311,   212,   212,
     &    200,   178,   149,   119,    91,    67,    48,    34,    23,
     &     16,    14,     8,     5,     3,     2,     1,     0,     0,
     &      0,     0,     0,     0,     0,     0,    28,  3360,  5545,
     &    315,   258,   273,   274,   260,   233,   197,   158,   121,
     &     89,    64,    44,    30,    22,    17,    11,     6,     3,
     &      2,     1,     1,     0,     0,     0,     0,    27,  3570,
     &   5335,   356,   304,   334,   349,   346,   325,   287,   239,
     &    189,   143,   103,    74,    50,    34,    27,    18,    12,
     &      7,     4,     2,     1,     1,     0,     0,    25,  3780,
     &   5042,   332,   334,   380,   415,   431,   425,   395,   346,
     &    286,   223,   167,   120,    84,    57,    38,    32,    20,
     &     12,     7,     4,     2,     1,    24,  3990,  4620,   283/
      DATA (MAPEMIN (I), I= 7651, 8100) /
     &    359,   424,   480,   519,   533,   518,   472,   405,   327,
     &    250,   183,   128,    89,    59,    44,    31,    20,    11,
     &      6,     4,    23,  4200,  4229,   308,   404,   491,   569,
     &    630,   663,   656,   608,   526,   427,   325,   236,   164,
     &    111,    73,    55,    37,    23,    13,     7,    21,  4620,
     &   3846,   324,   429,   538,   646,   751,   840,   896,   905,
     &    858,   762,   631,   489,   362,   253,   172,   113,    82,
     &     57,    19,  5039,  3917,  1331,   307,   410,   512,   614,
     &    614,   819,   819,   819,   922,   819,   819,   717,   717,
     &    717,   205,    20,  5250,  4245,  1331,   410,   512,   614,
     &    717,   819,   922,  1024,  1024,  1126,  1126,  1024,  1024,
     &    819,   717,   717,   307,    21,  5459,  4584,  1331,   512,
     &    614,   717,   922,  1024,  1229,  1331,  1434,  1434,  1331,
     &   1331,  1229,  1024,   819,   717,   717,   410,    22,  5669,
     &   4811,  1434,   512,   717,   922,  1229,  1434,  1536,  1843,
     &   1843,  1843,  1741,  1638,  1434,  1229,  1024,   819,   614,
     &    512,   431,    23,  5879,  5020,  1536,   614,   922,  1126,
     &   1536,  1946,  2253,  2355,  2560,  2458,  2253,  2048,  1741,
     &   1434,  1229,   819,   717,   512,   519,   208,    23,  6089,
     &   5092,  1536,   819,  1229,  1638,  2150,  2662,  3072,  3379,
     &   3379,  3072,  2765,  2355,  1843,  1536,  1126,   819,   614,
     &    545,   350,   234,    24,  6300,  5182,  1638,  1024,  1638,
     &   2253,  3072,  3789,  4301,  4506,  4198,  3789,  3072,  2458,
     &   1843,  1434,   922,   819,   489,   374,   266,   187,    37,
     &     25,  6719,  5410,  2253,  2355,  4198,  6246,  7680,  7680,
     &   6451,  4915,  3482,  2355,  1434,  1024,   600,   388,   243,
     &    151,    93,    59,    37,    22,    14,     1,    25,  7139,
     &   5561,  2355,  2867,  5222,  8090,  9830,  9523,  7885,  5734,
     &   3994,  2560,  1536,  1024,   664,   386,   236,   144,    88,
     &     54,    33,    19,    12,     6,    26,  7350,  5637,  2458,
     &   2970,  5734,  8909, 10957, 10547,  8602,  6349,  4198,  2765,
     &   1741,  1126,   626,   403,   245,   148,    90,    55,    33,
     &     19,    12,     8,     0,    26,  7559,  5701,  2458,  3174,
     &   6246,  9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,
     &   1126,   736,   417,   251,   152,    91,    55,    33,    19,
     &     12,     8,     1,    26,  7979,  5853,  2560,  3379,  7066,
     &  11366, 14438, 14234, 11469,  8294,  5530,  3482,  2253,  1331,
     &    782,   486,   292,   174,   105,    62,    37,    22,    14,
     &      8,     4,    27,  8400,  5952,  2560,  3789,  7782, 13210,
     &  17101, 16896, 13824,  9933,  6554,  4198,  2560,  1536,   922,
     &    568,   332,   197,   117,    70,    41,    24,    14,     9,
     &      5,     1,    27,  8819,  6055,  2560,  3891,  8090, 14438,
     &  19354, 19968, 16691, 12288,  8090,  5222,  3174,  1946,  1229,
     &    688,   418,   249,   147,    88,    52,    30,    18,    11,
     &      6,     3,    28,  9239,  6152,  2662,  3789,  8499, 15462,
     &  21709, 23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,
     &    819,   528,   310,   184,   109,    65,    38,    23,    13,
     &      8,     5,     0,    28,  9450,  6178,  2662,  3891,  8602,
     &  15974, 22938, 24986, 21811, 16282, 11059,  7168,  4403,  2662,
     &   1638,  1024,   535,   345,   205,   121,    72,    43,    25/
      DATA (MAPEMIN (I), I= 8101, 8550) /
     &     14,     9,     5,     1,    28,  9659,  6194,  2662,  3891,
     &   8704, 16486, 24166, 26829, 23654, 17818, 12186,  7885,  4915,
     &   2970,  1741,  1126,   622,   382,   227,   135,    80,    47,
     &     28,    16,    10,     6,     1,    28, 10079,  6157,  2662,
     &   3891,  9011, 17408, 26522, 30515, 27750, 21197, 14746,  9523,
     &   5939,  3584,  2253,  1229,   834,   467,   277,   165,    97,
     &     58,    34,    20,    12,     7,     1,    27, 10500,  6097,
     &   2662,  3994,  9216, 18227, 28877, 34509, 32051, 25088, 17510,
     &  11469,  7168,  4403,  2662,  1536,  1024,   537,   336,   199,
     &    118,    70,    42,    24,    15,     7,    26, 11550,  5869,
     &   2662,  4096,  9626, 20275, 34611, 44954, 44954, 36762, 26419,
     &  17510, 11162,  6861,  4096,  2560,  1434,   922,   510,   313,
     &    186,   110,    65,    39,    22,    25, 12600,  5585,  2662,
     &   4096, 10035, 21811, 39936, 56115, 59904, 51507, 38093, 25805,
     &  16486, 10240,  6246,  3687,  2253,  1331,   822,   472,   280,
     &    166,    98,    50,    24, 13859,  5201,  2662,  4198, 10240,
     &  23347, 45670, 69837, 80998, 73728, 56730, 39322, 25498, 15974,
     &   9728,  5939,  3482,  2151,  1229,   727,   442,   262,    58,
     &     23, 14700,  4885,  2662,  4198, 10445, 24166, 48845, 78746,
     &  95846, 90829, 71783, 50483, 33075, 20787, 12698,  7783,  4608,
     &   2765,  1639,   922,   652,    36,    21, 15750,  4392,  2662,
     &   4198, 10547, 25088, 52429, 89395,115610,115098, 94003, 67482,
     &  44749, 28365, 17408, 10650,  6349,  3789,  2253,   205,    18,
     &  16800,  3730,  2662,  4198, 10650, 25702, 55706, 99430,136192,
     & 142438,120320, 88269, 59290, 37786, 23450, 14234,  5427,    11,
     &  18900,  2025,  2765,  4915, 12902, 32870, 75674,142029,200294,
     & 192512,     3, 21000,     0,     3, 32767,     0,   910, 12800,
     &      3,     0,     0,     3,  2310,     0,    20,  2520,  2228,
     &     37,    18,    19,    18,    17,    17,    16,    14,    13,
     &     12,    11,    10,     9,     8,     7,     7,     6,    26,
     &   2625,  3497,    71,    32,    32,    31,    30,    28,    26,
     &     24,    22,    20,    18,    16,    14,    13,    11,    10,
     &      9,     7,     7,     6,     5,     4,     4,    28,  2730,
     &   4229,   169,    53,    47,    44,    41,    37,    34,    31,
     &     27,    24,    21,    18,    16,    14,    12,    10,     9,
     &      7,     6,     5,     5,     4,     3,     3,     3,    29,
     &   2835,  4739,   275,    82,    68,    63,    57,    50,    44,
     &     38,    33,    28,    23,    20,    16,    14,    11,     9,
     &      8,     6,     5,     5,     3,     2,     4,     2,     1,
     &      1,    29,  2940,  5097,   381,   116,    98,    89,    79,
     &     68,    57,    48,    39,    31,    25,    20,    16,    13,
     &     10,     7,     7,     4,     5,     3,     2,     1,     1,
     &      0,     0,     0,    29,  3045,  5309,   385,   161,   153,
     &    139,   121,   101,    82,    64,    49,    37,    28,    21,
     &     15,    11,     9,     6,     4,     2,     1,     1,     0,
     &      0,     0,     0,     0,     0,    28,  3150,  5356,   311,
     &    212,   212,   200,   178,   149,   119,    91,    67,    48,
     &     34,    23,    16,    14,     8,     5,     3,     2,     1,
     &      0,     0,     0,     0,     0,     0,    27,  3360,  5170,
     &    315,   258,   273,   274,   260,   233,   197,   158,   121/
      DATA (MAPEMIN (I), I= 8551, 9000) /
     &     89,    64,    44,    30,    22,    17,    11,     6,     3,
     &      2,     1,     1,     0,     0,     0,    25,  3570,  4882,
     &    356,   304,   334,   349,   346,   325,   287,   239,   189,
     &    143,   103,    74,    50,    34,    27,    18,    12,     7,
     &      4,     2,     1,     1,    23,  3780,  4426,   332,   334,
     &    380,   415,   431,   425,   395,   346,   286,   223,   167,
     &    120,    84,    57,    38,    32,    20,    12,     7,     4,
     &     22,  3990,  3931,   283,   359,   424,   480,   519,   533,
     &    518,   472,   405,   327,   250,   183,   128,    89,    59,
     &     44,    31,    20,    11,    20,  4200,  3589,   308,   404,
     &    491,   569,   630,   663,   656,   608,   526,   427,   325,
     &    236,   164,   111,    73,    55,    37,    18,  4620,  3171,
     &    324,   429,   538,   646,   751,   840,   896,   905,   858,
     &    762,   631,   489,   362,   253,   172,    15,  5039,  3071,
     &   1331,   307,   410,   512,   614,   614,   819,   819,   717,
     &    922,   819,   922,    17,  5250,  3496,  1331,   410,   512,
     &    614,   717,   819,   922,  1024,  1024,  1126,  1024,  1126,
     &    922,   614,    19,  5459,  3924,  1331,   512,   614,   717,
     &    922,  1024,  1229,  1331,  1434,  1434,  1331,  1331,  1126,
     &   1024,   922,   307,    20,  5669,  4212,  1434,   512,   717,
     &    922,  1229,  1434,  1536,  1843,  1843,  1843,  1741,  1638,
     &   1434,  1229,   922,   922,   307,    21,  5879,  4503,  1536,
     &    614,   922,  1126,  1536,  1946,  2253,  2355,  2560,  2458,
     &   2253,  2048,  1741,  1434,  1126,   922,   717,   307,    22,
     &   6089,  4728,  1536,   819,  1229,  1638,  2150,  2662,  3072,
     &   3379,  3379,  3072,  2765,  2355,  1843,  1536,  1126,   819,
     &    614,   545,   179,    23,  6300,  4871,  1638,  1024,  1638,
     &   2253,  3072,  3789,  4301,  4506,  4198,  3789,  3072,  2458,
     &   1843,  1434,   922,   819,   489,   374,   266,     6,    23,
     &   6719,  5078,  2253,  2355,  4198,  6246,  7680,  7680,  6451,
     &   4915,  3482,  2355,  1434,  1024,   600,   388,   243,   151,
     &     93,    59,    37,    19,    24,  7139,  5269,  2355,  2867,
     &   5222,  8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,
     &   1024,   664,   386,   236,   144,    88,    54,    33,    19,
     &      8,    24,  7350,  5361,  2458,  2970,  5734,  8909, 10957,
     &  10547,  8602,  6349,  4198,  2765,  1741,  1126,   626,   403,
     &    245,   148,    90,    54,    33,    20,    12,    25,  7559,
     &   5432,  2458,  3174,  6246,  9830, 12186, 11674,  9421,  6861,
     &   4608,  2970,  1741,  1126,   736,   417,   251,   152,    91,
     &     55,    33,    19,    12,     2,    25,  7979,  5536,  2560,
     &   3379,  7066, 11366, 14438, 14234, 11469,  8294,  5530,  3482,
     &   2253,  1331,   782,   486,   292,   174,   105,    62,    37,
     &     22,    13,     6,    26,  8400,  5642,  2560,  3789,  7782,
     &  13210, 17101, 16896, 13824,  9933,  6554,  4198,  2560,  1536,
     &    922,   568,   332,   197,   117,    70,    41,    24,    14,
     &      9,     0,    26,  8819,  5704,  2560,  3891,  8090, 14438,
     &  19354, 19968, 16691, 12288,  8090,  5222,  3174,  1946,  1229,
     &    688,   418,   249,   147,    88,    52,    30,    18,    11,
     &      2,    26,  9239,  5766,  2662,  3789,  8499, 15462, 21709,
     &  23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,   819/
      DATA (MAPEMIN (I), I= 9001, 9450) /
     &    528,   310,   184,   109,    65,    38,    22,    14,     5,
     &     26,  9450,  5770,  2662,  3891,  8602, 15974, 22938, 24986,
     &  21811, 16282, 11059,  7168,  4403,  2662,  1638,  1024,   535,
     &    345,   205,   121,    72,    42,    25,    15,     6,    26,
     &   9659,  5758,  2662,  3891,  8704, 16486, 24166, 26829, 23654,
     &  17818, 12186,  7885,  4915,  2970,  1741,  1126,   622,   382,
     &    227,   135,    80,    46,    28,    17,     6,    26, 10079,
     &   5713,  2662,  3891,  9011, 17408, 26522, 30515, 27750, 21197,
     &  14746,  9523,  5939,  3584,  2253,  1229,   834,   467,   277,
     &    165,    97,    58,    34,    20,     5,    26, 10500,  5650,
     &   2662,  3994,  9216, 18227, 28877, 34509, 32051, 25088, 17510,
     &  11469,  7168,  4403,  2662,  1536,  1024,   537,   336,   199,
     &    118,    70,    42,    24,     2,    25, 11550,  5430,  2662,
     &   4096,  9626, 20275, 34611, 44954, 44954, 36762, 26419, 17510,
     &  11162,  6861,  4096,  2560,  1434,   922,   510,   313,   186,
     &    110,    65,    10,    23, 12600,  5115,  2662,  4096, 10035,
     &  21811, 39936, 56115, 59904, 51507, 38093, 25805, 16486, 10240,
     &   6246,  3687,  2253,  1331,   821,   473,   280,   164,    22,
     &  13859,  4712,  2662,  4198, 10240, 23347, 45670, 69837, 80998,
     &  73728, 56730, 39322, 25498, 15974,  9728,  5939,  3482,  2048,
     &   1229,   830,   208,    20, 14700,  4331,  2662,  4198, 10445,
     &  24166, 48845, 78746, 95846, 90829, 71783, 50483, 33075, 20787,
     &  12698,  7680,  4711,  2765,  1536,    18, 15750,  3750,  2662,
     &   4198, 10547, 25088, 52429, 89395,115610,115098, 94003, 67482,
     &  44749, 28262, 17511, 10547,  4506,    15, 16800,  3049,  2662,
     &   4198, 10650, 25702, 55706, 99430,136192,142438,120320, 88269,
     &  59290, 35123,     9, 18900,  1431,  2765,  4915, 12800, 32973,
     &  75674, 75366,     3, 21000,     0,     3, 32767,     0,   834,
     &  16000,     3,     0,     0,     3,  2310,     0,    18,  2520,
     &   1766,    37,    18,    19,    18,    17,    17,    16,    14,
     &     13,    12,    11,    10,     9,     8,     7,    24,  2625,
     &   3044,    71,    32,    32,    31,    30,    28,    26,    24,
     &     22,    20,    18,    16,    14,    13,    11,    10,     9,
     &      7,     7,     6,     5,    26,  2730,  3869,   169,    53,
     &     47,    44,    41,    37,    34,    31,    27,    24,    21,
     &     18,    16,    14,    12,    10,     9,     7,     6,     5,
     &      5,     4,     3,    26,  2835,  4404,   275,    82,    68,
     &     63,    57,    50,    44,    38,    33,    28,    23,    20,
     &     16,    14,    11,     9,     8,     6,     5,     5,     3,
     &      2,     4,    27,  2940,  4757,   381,   116,    98,    89,
     &     79,    68,    57,    48,    39,    31,    25,    20,    16,
     &     13,    10,     7,     7,     4,     5,     3,     2,     1,
     &      1,     0,    27,  3045,  4986,   385,   161,   153,   139,
     &    121,   101,    82,    64,    49,    37,    28,    21,    15,
     &     11,     9,     6,     4,     2,     1,     1,     0,     0,
     &      0,     0,    26,  3150,  5020,   311,   212,   212,   200,
     &    178,   149,   119,    91,    67,    48,    34,    23,    16,
     &     14,     8,     5,     3,     2,     1,     0,     0,     0,
     &      0,    25,  3360,  4770,   315,   258,   273,   274,   260,
     &    233,   197,   158,   121,    89,    64,    44,    30,    22/
      DATA (MAPEMIN (I), I= 9451, 9900) /
     &     17,    11,     6,     3,     2,     1,     1,     0,    24,
     &   3570,  4426,   356,   304,   334,   349,   346,   325,   287,
     &    239,   189,   143,   103,    74,    50,    34,    27,    18,
     &     12,     7,     4,     2,     1,    21,  3780,  3911,   332,
     &    334,   380,   415,   431,   425,   395,   346,   286,   223,
     &    167,   120,    84,    57,    38,    32,    20,    12,    19,
     &   3990,  3345,   280,   358,   423,   479,   518,   533,   518,
     &    473,   406,   328,   251,   184,   129,    89,    60,    45,
     &     17,  4200,  2956,   308,   404,   491,   569,   630,   663,
     &    656,   608,   526,   427,   325,   236,   164,   111,    15,
     &   4620,  2493,   324,   429,   538,   646,   751,   840,   896,
     &    905,   858,   762,   631,   489,    12,  5039,  2283,  1331,
     &    307,   410,   512,   614,   614,   717,   819,   819,    14,
     &   5250,  2745,  1331,   410,   512,   614,   717,   819,   922,
     &    922,  1126,  1126,   819,    16,  5459,  3252,  1331,   512,
     &    614,   717,   922,  1024,  1229,  1331,  1434,  1331,  1434,
     &   1331,   819,    18,  5669,  3641,  1434,   512,   717,   922,
     &   1229,  1434,  1536,  1843,  1843,  1843,  1741,  1638,  1434,
     &   1229,   205,    19,  5879,  3985,  1536,   614,   922,  1126,
     &   1536,  1946,  2253,  2355,  2560,  2458,  2253,  2048,  1741,
     &   1434,  1126,   512,    20,  6089,  4290,  1536,   819,  1229,
     &   1638,  2150,  2662,  3072,  3379,  3379,  3072,  2765,  2355,
     &   1843,  1434,  1126,   922,   512,    21,  6300,  4494,  1638,
     &   1024,  1638,  2253,  3072,  3789,  4301,  4506,  4198,  3789,
     &   3072,  2458,  1843,  1434,   922,   717,   590,   224,    22,
     &   6719,  4764,  2253,  2355,  4198,  6246,  7680,  7680,  6451,
     &   4915,  3482,  2355,  1434,  1024,   600,   388,   243,   150,
     &     94,    59,    24,    23,  7139,  4967,  2355,  2867,  5222,
     &   8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,  1024,
     &    664,   386,   236,   144,    88,    54,    33,     9,    23,
     &   7350,  5022,  2458,  2970,  5734,  8909, 10957, 10547,  8602,
     &   6349,  4198,  2765,  1741,  1126,   626,   403,   245,   148,
     &     90,    54,    33,    14,    23,  7559,  5114,  2458,  3174,
     &   6246,  9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,
     &   1126,   736,   417,   251,   152,    91,    55,    33,    19,
     &     24,  7979,  5243,  2560,  3379,  7066, 11366, 14438, 14234,
     &  11469,  8294,  5530,  3482,  2253,  1331,   782,   486,   292,
     &    174,   105,    61,    38,    22,     7,    24,  8400,  5342,
     &   2560,  3789,  7782, 13210, 17101, 16896, 13824,  9933,  6554,
     &   4198,  2560,  1536,   922,   568,   332,   197,   117,    69,
     &     41,    25,    13,    25,  8819,  5398,  2560,  3891,  8090,
     &  14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,  1946,
     &   1229,   688,   418,   249,   147,    88,    51,    31,    18,
     &      1,    25,  9239,  5400,  2662,  3789,  8499, 15462, 21709,
     &  23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,   819,
     &    528,   310,   184,   109,    64,    39,    22,     2,    25,
     &   9450,  5386,  2662,  3891,  8602, 15974, 22938, 24986, 21811,
     &  16282, 11059,  7168,  4403,  2662,  1638,  1024,   535,   345,
     &    205,   121,    71,    43,    25,     1,    24,  9659,  5368,
     &   2662,  3891,  8704, 16486, 24166, 26829, 23654, 17818, 12186/
      DATA (MAPEMIN (I), I= 9901,10350) /
     &   7885,  4915,  2970,  1741,  1126,   622,   382,   227,   134,
     &     80,    47,    28,    24, 10079,  5285,  2662,  3891,  9011,
     &  17408, 26522, 30515, 27750, 21197, 14746,  9523,  5939,  3584,
     &   2253,  1229,   834,   467,   277,   164,    98,    58,    24,
     &     24, 10500,  5201,  2662,  3994,  9216, 18227, 28877, 34509,
     &  32051, 25088, 17510, 11469,  7168,  4403,  2662,  1536,  1024,
     &    537,   336,   199,   118,    70,    16,    23, 11550,  4921,
     &   2662,  4096,  9626, 20275, 34611, 44954, 44954, 36762, 26419,
     &  17510, 11162,  6861,  4096,  2560,  1434,   922,   509,   314,
     &    186,    30,    21, 12600,  4584,  2662,  4096, 10035, 21811,
     &  39936, 56115, 59904, 51507, 38093, 25805, 16486, 10240,  6246,
     &   3687,  2253,  1331,   821,   439,    20, 13859,  4117,  2662,
     &   4198, 10240, 23347, 45670, 69837, 80998, 73728, 56730, 39322,
     &  25498, 15974,  9728,  5837,  3584,  2048,   103,    18, 14700,
     &   3688,  2662,  4198, 10445, 24166, 48845, 78746, 95846, 90829,
     &  71783, 50483, 33075, 20685, 12800,  7680,  2253,    16, 15750,
     &   3126,  2662,  4198, 10547, 25088, 52429, 89395,115610,115098,
     &  94003, 67379, 44749, 28365,  4506,    13, 16800,  2529,  2662,
     &   4198, 10650, 25702, 55706, 99430,136192,142438,120320, 79258,
     &      8, 18900,  1086,  2765,  4813, 12902, 32973, 13312,     3,
     &  21000,     0,     3, 32767,     0,   756, 19200,     3,     0,
     &      0,     3,  2310,     0,    16,  2520,  1230,    37,    18,
     &     19,    18,    17,    17,    16,    14,    13,    12,    11,
     &     10,     9,    21,  2625,  2537,    71,    32,    32,    31,
     &     30,    28,    26,    24,    22,    20,    18,    16,    14,
     &     13,    11,    10,     9,     7,    24,  2730,  3308,   169,
     &     53,    47,    44,    41,    37,    34,    31,    27,    24,
     &     21,    18,    16,    14,    12,    10,     9,     7,     6,
     &      5,     5,    24,  2835,  3715,   275,    82,    68,    63,
     &     57,    50,    44,    38,    33,    28,    23,    20,    16,
     &     14,    11,     9,     8,     6,     5,     5,     3,    24,
     &   2940,  3968,   381,   116,    98,    89,    79,    68,    57,
     &     48,    39,    31,    25,    20,    16,    13,    10,     7,
     &      7,     4,     5,     3,     2,    24,  3045,  4138,   385,
     &    161,   153,   139,   121,   101,    82,    64,    49,    37,
     &     28,    21,    15,    11,     9,     6,     4,     2,     1,
     &      1,     0,    24,  3150,  4138,   311,   212,   212,   200,
     &    178,   149,   119,    91,    67,    48,    34,    23,    16,
     &     14,     8,     5,     3,     2,     1,     0,     0,    23,
     &   3360,  3869,   315,   258,   273,   274,   260,   233,   197,
     &    158,   121,    89,    64,    44,    30,    22,    17,    11,
     &      6,     3,     2,     1,    22,  3570,  3514,   356,   304,
     &    334,   349,   346,   325,   287,   239,   189,   143,   103,
     &     74,    50,    34,    27,    18,    12,     7,     4,    19,
     &   3780,  3081,   332,   334,   380,   415,   431,   425,   395,
     &    346,   286,   223,   167,   120,    84,    57,    38,    32,
     &     16,  3990,  2551,   283,   359,   424,   480,   519,   533,
     &    518,   472,   405,   327,   250,   183,   128,    14,  4200,
     &   2086,   308,   404,   491,   569,   630,   663,   656,   608,
     &    526,   427,   325,    11,  4620,  1404,   324,   429,   538/
      DATA (MAPEMIN (I), I=10351,10800) /
     &    646,   751,   840,   896,   905,     8,  5039,  1024,  1331,
     &    307,   410,   512,     0,    11,  5250,  1920,  1331,   410,
     &    512,   614,   614,   819,   922,   512,    13,  5459,  2455,
     &   1331,   512,   614,   717,   922,  1024,  1229,  1331,  1331,
     &    922,    15,  5669,  2961,  1434,   512,   717,   922,  1229,
     &   1434,  1536,  1843,  1843,  1843,  1741,   922,    17,  5879,
     &   3380,  1536,   614,   922,  1126,  1536,  1946,  2253,  2355,
     &   2560,  2458,  2253,  2048,  1741,   307,    19,  6089,  3996,
     &   1536,   819,  1229,  1638,  2150,  2662,  3072,  3379,  3379,
     &   3072,  2765,  2355,  1843,  1434,  1126,   614,    20,  6300,
     &   4245,  1638,  1024,  1638,  2253,  3072,  3789,  4301,  4506,
     &   4198,  3789,  3072,  2458,  1843,  1331,  1024,   717,   393,
     &     21,  6719,  4520,  2253,  2355,  4198,  6246,  7680,  7680,
     &   6451,  4915,  3482,  2355,  1434,  1024,   600,   388,   242,
     &    151,    94,    42,    22,  7139,  4723,  2355,  2867,  5222,
     &   8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,  1024,
     &    664,   386,   236,   144,    88,    54,    16,    22,  7350,
     &   4793,  2458,  2970,  5734,  8909, 10957, 10547,  8602,  6349,
     &   4198,  2765,  1741,  1126,   626,   403,   245,   148,    90,
     &     54,    26,    22,  7559,  4837,  2458,  3174,  6246,  9830,
     &  12186, 11674,  9421,  6861,  4608,  2970,  1741,  1126,   736,
     &    417,   251,   151,    92,    55,    30,    23,  7979,  4928,
     &   2560,  3379,  7066, 11366, 14438, 14234, 11469,  8294,  5530,
     &   3482,  2253,  1331,   782,   486,   292,   174,   104,    62,
     &     38,     6,    23,  8400,  4972,  2560,  3789,  7782, 13210,
     &  17101, 16896, 13824,  9933,  6554,  4198,  2560,  1536,   922,
     &    568,   332,   197,   117,    69,    41,    12,    23,  8819,
     &   4992,  2560,  3891,  8090, 14438, 19354, 19968, 16691, 12288,
     &   8090,  5222,  3174,  1946,  1229,   688,   418,   249,   147,
     &     87,    52,    17,    23,  9239,  4973,  2662,  3789,  8499,
     &  15462, 21709, 23347, 19968, 14848, 10035,  6451,  3994,  2355,
     &   1536,   819,   528,   310,   183,   110,    64,    19,    23,
     &   9450,  4961,  2662,  3891,  8602, 15974, 22938, 24986, 21811,
     &  16282, 11059,  7168,  4403,  2662,  1638,  1024,   535,   345,
     &    204,   121,    72,    19,    23,  9659,  4941,  2662,  3891,
     &   8704, 16486, 24166, 26829, 23654, 17818, 12186,  7885,  4915,
     &   2970,  1741,  1126,   622,   382,   227,   134,    80,    17,
     &     22, 10079,  4837,  2662,  3891,  9011, 17408, 26522, 30515,
     &  27750, 21197, 14746,  9523,  5939,  3584,  2253,  1229,   834,
     &    466,   278,   164,    90,    22, 10500,  4734,  2662,  3994,
     &   9216, 18227, 28877, 34509, 32051, 25088, 17510, 11469,  7168,
     &   4403,  2662,  1536,  1024,   536,   336,   200,    66,    21,
     &  11550,  4466,  2662,  4096,  9626, 20275, 34611, 44954, 44954,
     &  36762, 26419, 17510, 11162,  6861,  4096,  2560,  1434,   819,
     &    612,   160,    20, 12600,  4142,  2662,  4096, 10035, 21811,
     &  39936, 56115, 59904, 51507, 38093, 25805, 16486, 10240,  6246,
     &   3687,  2253,  1331,   202,    18, 13859,  3653,  2662,  4198,
     &  10240, 23347, 45670, 69837, 80998, 73728, 56730, 39322, 25498,
     &  15872,  9830,  5837,  1127,    16, 14700,  3266,  2662,  4198,
     &  10445, 24166, 48845, 78746, 95846, 90829, 71783, 50381, 33075/
      DATA (MAPEMIN (I), I=10801,11250) /
     &  20787, 10343,    14, 15750,  2754,  2662,  4198, 10547, 25088,
     &  52429, 89395,115610,115098, 94003, 67379, 35840,    12, 16800,
     &   2197,  2662,  4198, 10650, 25702, 55706, 99328,136294,142438,
     &  74035,     7, 18900,   852,  2765,  4813, 12902,  7782,     3,
     &  21000,     0,     3, 32767,     0,   681, 22400,     3,     0,
     &      0,     3,  2310,     0,    14,  2520,   615,    37,    18,
     &     19,    18,    17,    17,    16,    14,    13,    12,    11,
     &     18,  2625,  1602,    71,    32,    32,    31,    30,    28,
     &     26,    24,    22,    20,    18,    16,    14,    13,    11,
     &     20,  2730,  2270,   169,    53,    47,    44,    41,    37,
     &     34,    31,    27,    24,    21,    18,    16,    14,    12,
     &     10,     9,    22,  2835,  2641,   275,    82,    68,    63,
     &     57,    50,    44,    38,    33,    28,    23,    20,    16,
     &     14,    11,     9,     8,     6,     5,    22,  2940,  2887,
     &    381,   116,    98,    89,    79,    68,    57,    48,    39,
     &     31,    25,    20,    16,    13,    10,     7,     7,     4,
     &      5,    22,  3045,  3024,   385,   161,   153,   139,   121,
     &    101,    82,    64,    49,    37,    28,    21,    15,    11,
     &      9,     6,     4,     2,     1,    22,  3150,  3026,   311,
     &    212,   212,   200,   178,   149,   119,    91,    67,    48,
     &     34,    23,    16,    14,     8,     5,     3,     2,     1,
     &     22,  3360,  2856,   315,   258,   273,   274,   260,   233,
     &    197,   158,   121,    89,    64,    44,    30,    22,    17,
     &     11,     6,     3,     2,    20,  3570,  2603,   356,   304,
     &    334,   349,   346,   325,   287,   239,   189,   143,   103,
     &     74,    50,    34,    27,    18,    12,    17,  3780,  2256,
     &    332,   334,   380,   415,   431,   425,   395,   346,   286,
     &    223,   167,   120,    84,    57,    15,  3990,  1809,   283,
     &    359,   424,   480,   519,   533,   518,   472,   405,   327,
     &    250,   183,    13,  4200,  1351,   308,   404,   491,   569,
     &    630,   663,   656,   608,   526,   427,     8,  4620,   315,
     &    324,   429,   538,   646,   751,     3,  5040,     0,     8,
     &   5250,  1024,  1331,   307,   512,   614,     0,     9,  5459,
     &   1527,  1331,   512,   512,   717,   922,  1024,    12,  5669,
     &   2228,  1434,   512,   717,   922,  1229,  1331,  1638,  1741,
     &   1331,    15,  5879,  2944,  1536,   614,   922,  1126,  1536,
     &   1946,  2253,  2355,  2458,  2560,  2253,  1024,    18,  6089,
     &   3736,  1536,   819,  1229,  1638,  2150,  2662,  3072,  3379,
     &   3379,  3072,  2765,  2355,  1843,  1434,   717,    19,  6300,
     &   3982,  1638,  1024,  1638,  2253,  3072,  3789,  4301,  4506,
     &   4198,  3789,  3072,  2458,  1741,  1434,  1024,   410,    20,
     &   6719,  4290,  2253,  2355,  4198,  6246,  7680,  7680,  6451,
     &   4915,  3482,  2355,  1434,  1024,   600,   387,   243,   151,
     &     76,    21,  7139,  4474,  2355,  2867,  5222,  8090,  9830,
     &   9523,  7885,  5734,  3994,  2560,  1536,  1024,   664,   386,
     &    236,   144,    88,    28,    21,  7350,  4520,  2458,  2970,
     &   5734,  8909, 10957, 10547,  8602,  6349,  4198,  2765,  1741,
     &   1126,   626,   403,   244,   149,    90,    39,    21,  7559,
     &   4553,  2458,  3174,  6246,  9830, 12186, 11674,  9421,  6861,
     &   4608,  2970,  1741,  1126,   736,   417,   251,   151,    92/
      DATA (MAPEMIN (I), I=11251,11700) /
     &     45,    21,  7979,  4584,  2560,  3379,  7066, 11366, 14438,
     &  14234, 11469,  8294,  5530,  3482,  2253,  1331,   782,   486,
     &    291,   175,   104,    58,    21,  8400,  4599,  2560,  3789,
     &   7782, 13210, 17101, 16896, 13824,  9933,  6554,  4198,  2560,
     &   1536,   922,   568,   331,   198,   117,    67,    21,  8819,
     &   4569,  2560,  3891,  8090, 14438, 19354, 19968, 16691, 12288,
     &   8090,  5222,  3174,  1946,  1229,   688,   418,   248,   148,
     &     77,    21,  9239,  4537,  2662,  3789,  8499, 15462, 21709,
     &  23347, 19968, 14848, 10035,  6451,  3994,  2355,  1536,   819,
     &    528,   310,   183,    85,    21,  9450,  4494,  2662,  3891,
     &   8602, 15974, 22938, 24986, 21811, 16282, 11059,  7168,  4403,
     &   2662,  1638,  1024,   535,   344,   205,    75,    21,  9659,
     &   4466,  2662,  3891,  8704, 16486, 24166, 26829, 23654, 17818,
     &  12186,  7885,  4915,  2970,  1741,  1126,   622,   382,   227,
     &     69,    21, 10079,  4369,  2662,  3891,  9011, 17408, 26522,
     &  30515, 27750, 21197, 14746,  9523,  5939,  3584,  2253,  1229,
     &    834,   466,   278,    14,    20, 10500,  4261,  2662,  3994,
     &   9216, 18227, 28877, 34509, 32051, 25088, 17510, 11469,  7168,
     &   4403,  2662,  1536,   922,   639,   236,    19, 11550,  3968,
     &   2662,  4096,  9626, 20275, 34611, 44954, 44954, 36762, 26419,
     &  17510, 11162,  6861,  4096,  2458,  1536,   512,    18, 12600,
     &   3641,  2662,  4096, 10035, 21811, 39936, 56115, 59904, 51507,
     &  38093, 25805, 16486, 10138,  6246,  3789,   614,    16, 13859,
     &   3266,  2662,  4198, 10240, 23347, 45670, 69837, 80998, 73728,
     &  56730, 39219, 25498, 15974,  7885,    15, 14700,  2873,  2662,
     &   4198, 10445, 24166, 48845, 78746, 95846, 90829, 71783, 50381,
     &  33075,  5632,    13, 15750,  2398,  2662,  4198, 10547, 25088,
     &  52429, 89395,115610,115098, 94003, 27853,    11, 16800,  1889,
     &   2662,  4198, 10650, 25702, 55603, 99430,136294, 55296,     6,
     &  18900,   658,  2765,  4813,  5837,     3, 21000,     0,     3,
     &  32767,     0,   549, 25600,     3,     0,     0,     3,  2520,
     &      0,    11,  2625,   669,    71,    32,    32,    31,    30,
     &     28,    26,    24,    13,  2730,  1233,   169,    53,    47,
     &     44,    41,    37,    34,    31,    27,    24,    14,  2835,
     &   1568,   275,    82,    68,    63,    57,    50,    44,    38,
     &     33,    28,    23,    15,  2940,  1806,   381,   116,    98,
     &     89,    79,    68,    57,    48,    39,    31,    25,    20,
     &     16,  3045,  1908,   385,   161,   153,   139,   121,   101,
     &     82,    64,    49,    37,    28,    21,    15,    16,  3150,
     &   1914,   311,   212,   212,   200,   178,   149,   119,    91,
     &     67,    48,    34,    23,    16,    16,  3360,  1842,   315,
     &    258,   273,   274,   260,   233,   197,   158,   121,    89,
     &     64,    44,    30,    15,  3570,  1693,   356,   304,   334,
     &    349,   346,   325,   287,   239,   189,   143,   103,    74,
     &     12,  3780,  1431,   332,   334,   380,   415,   431,   425,
     &    395,   346,   286,     9,  3990,  1066,   283,   359,   424,
     &    480,   519,   533,     6,  4200,   616,   308,   404,   491,
     &      3,  4620,     0,     3,  5040,     0,     4,  5250,     0,
     &  13414,     6,  5459,   616,  1331,   410,   307,    10,  5669,
     &   1541,  1434,   512,   717,   922,  1126,  1434,     0,    13/
      DATA (MAPEMIN (I), I=11701,12150) /
     &   5879,  2455,  1536,   614,   922,  1126,  1536,  1946,  2150,
     &   2458,  2458,  1536,    17,  6089,  3444,  1536,   819,  1229,
     &   1638,  2150,  2662,  3072,  3379,  3379,  3072,  2765,  2355,
     &   1843,   717,    18,  6300,  3721,  1638,  1024,  1638,  2253,
     &   3072,  3789,  4301,  4506,  4198,  3789,  3072,  2355,  1843,
     &   1434,   614,    19,  6719,  4062,  2253,  2355,  4198,  6246,
     &   7680,  7680,  6451,  4915,  3482,  2355,  1434,  1024,   600,
     &    387,   243,   135,    20,  7139,  4173,  2355,  2867,  5222,
     &   8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,  1024,
     &    664,   385,   237,   144,    31,    20,  7350,  4184,  2458,
     &   2970,  5734,  8909, 10957, 10547,  8602,  6349,  4198,  2765,
     &   1741,  1126,   626,   402,   245,   149,    36,    20,  7559,
     &   4184,  2458,  3174,  6246,  9830, 12186, 11674,  9421,  6861,
     &   4608,  2970,  1741,  1126,   736,   416,   252,   151,    37,
     &     20,  7979,  4173,  2560,  3379,  7066, 11366, 14438, 14234,
     &  11469,  8294,  5530,  3482,  2253,  1331,   782,   485,   292,
     &    175,    37,    20,  8400,  4138,  2560,  3789,  7782, 13210,
     &  17101, 16896, 13824,  9933,  6554,  4198,  2560,  1536,   922,
     &    567,   332,   198,    24,    19,  8819,  4068,  2560,  3891,
     &   8090, 14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,
     &   1946,  1126,   790,   418,   228,    19,  9239,  3996,  2662,
     &   3789,  8499, 15462, 21709, 23347, 19968, 14848, 10035,  6451,
     &   3994,  2355,  1434,   922,   528,   208,    19,  9450,  3943,
     &   2662,  3891,  8602, 15974, 22938, 24986, 21811, 16282, 11059,
     &   7168,  4403,  2662,  1638,   922,   637,   162,    19,  9659,
     &   3904,  2662,  3891,  8704, 16486, 24166, 26829, 23654, 17818,
     &  12186,  7885,  4915,  2970,  1741,  1024,   724,   116,    18,
     &  10079,  3796,  2662,  3891,  9011, 17408, 26522, 30515, 27750,
     &  21197, 14746,  9523,  5939,  3584,  2150,  1331,   727,    18,
     &  10500,  3699,  2662,  3994,  9216, 18227, 28877, 34509, 32051,
     &  25088, 17510, 11469,  7168,  4301,  2662,  1638,   512,    17,
     &  11550,  3412,  2662,  4096,  9626, 20275, 34611, 44954, 44954,
     &  36762, 26419, 17510, 11059,  6861,  4198,   922,    16, 12600,
     &   3093,  2662,  4096, 10035, 21811, 39936, 56115, 59904, 51507,
     &  38093, 25702, 16486, 10240,   717,    14, 13859,  2706,  2662,
     &   4198, 10240, 23347, 45670, 69837, 80998, 73728, 56730, 39219,
     &  16077,    13, 14700,  2427,  2662,  4198, 10445, 24166, 48845,
     &  78746, 95744, 90931, 71783, 26829,    12, 15750,  2048,  2662,
     &   4198, 10547, 25088, 52429, 89293,115712,115098,     0,    10,
     &  16800,  1629,  2662,  4198, 10650, 25702, 55603, 99430, 46694,
     &      5, 18900,   433,  2765,  2765,     3, 21000,     0,     3,
     &  32767,     0,   414, 28800,     3,     0,     0,     3,  2730,
     &      0,     6,  2835,   389,   275,    82,    68,     8,  2940,
     &    659,   381,   116,    98,    89,    79,     9,  3045,   797,
     &    385,   161,   153,   139,   121,   101,    10,  3150,   818,
     &    311,   212,   212,   200,   178,   149,   119,    11,  3360,
     &    754,   315,   258,   273,   274,   260,   233,   197,   158,
     &     10,  3570,   559,   356,   304,   334,   349,   346,   325,
     &    287,     7,  3780,   149,   332,   334,   380,   415,     3,
     &   3990,     0,     3,  4200,     0,     3,  4620,     0,     3/
      DATA (MAPEMIN (I), I=12151,12600) /
     &   5040,     0,     3,  5250,     0,     3,  5460,     0,     6,
     &   5669,   715,  1331,   614,   512,    11,  5879,  1856,  1536,
     &    614,   922,  1126,  1536,  1843,  2253,   614,    16,  6089,
     &   3130,  1536,   819,  1229,  1638,  2150,  2662,  3072,  3379,
     &   3379,  3072,  2765,  2355,   410,    17,  6300,  3457,  1638,
     &   1024,  1638,  2253,  3072,  3789,  4301,  4506,  4198,  3789,
     &   2970,  2458,  1843,   819,    18,  6719,  3727,  2253,  2355,
     &   4198,  6246,  7680,  7680,  6451,  4915,  3482,  2355,  1434,
     &    922,   703,   387,   150,    18,  7139,  3776,  2355,  2867,
     &   5222,  8090,  9830,  9523,  7885,  5734,  3994,  2560,  1536,
     &   1024,   663,   386,   188,    18,  7350,  3746,  2458,  2970,
     &   5734,  8909, 10957, 10547,  8602,  6349,  4198,  2765,  1741,
     &   1024,   728,   403,   169,    18,  7559,  3727,  2458,  3174,
     &   6246,  9830, 12186, 11674,  9421,  6861,  4608,  2970,  1741,
     &   1126,   735,   417,   157,    18,  7979,  3650,  2560,  3379,
     &   7066, 11366, 14438, 14234, 11469,  8294,  5530,  3482,  2253,
     &   1229,   819,   551,    90,    17,  8400,  3560,  2560,  3789,
     &   7782, 13210, 17101, 16896, 13824,  9933,  6554,  4198,  2458,
     &   1536,  1024,   528,    17,  8819,  3461,  2560,  3891,  8090,
     &  14438, 19354, 19968, 16691, 12288,  8090,  5222,  3174,  1946,
     &   1126,   498,    17,  9239,  3357,  2662,  3789,  8499, 15462,
     &  21709, 23347, 19968, 14848, 10035,  6451,  3891,  2458,  1434,
     &    102,    16,  9450,  3300,  2662,  3891,  8602, 15974, 22938,
     &  24986, 21811, 16282, 11059,  7066,  4403,  2765,  1434,    16,
     &   9659,  3249,  2662,  3891,  8704, 16486, 24166, 26829, 23654,
     &  17818, 12186,  7782,  4915,  2970,  1331,    16, 10079,  3145,
     &   2662,  3891,  9011, 17408, 26522, 30515, 27750, 21197, 14746,
     &   9421,  5939,  3686,   717,    15, 10500,  3030,  2662,  3994,
     &   9216, 18227, 28877, 34509, 32051, 25088, 17510, 11366,  7270,
     &   3789,    14, 11550,  2763,  2662,  4096,  9626, 20275, 34611,
     &  44954, 44954, 36659, 26419, 17613,  9216,    13, 12600,  2464,
     &   2662,  4096, 10035, 21811, 39936, 56115, 59904, 51405, 38093,
     &  17408,    12, 13859,  2110,  2662,  4198, 10240, 23347, 45670,
     &  69837, 80896, 73830, 15463,    11, 14700,  1849,  2662,  4198,
     &  10445, 24166, 48845, 78643, 95846, 21606,     9, 15750,  1489,
     &   2662,  4198, 10547, 24986, 52531, 70554,     8, 16800,  1086,
     &   2662,  4198, 10650, 25702, 10035,     3, 18900,     0,     3,
     &  32767,     0,   265, 31999,     3,     0,     0,     3,  5647,
     &      0,     3,  5040,     0,     3,  5250,     0,     3,  5460,
     &      0,     4,  5669,     0,  1536,     8,  5879,  1233,  1536,
     &    512,   922,  1229,  1229,    14,  6089,  2704,  1536,   819,
     &   1229,  1638,  2150,  2662,  3072,  3379,  3277,  3174,  1638,
     &     16,  6300,  3192,  1638,  1024,  1638,  2253,  3072,  3789,
     &   4301,  4506,  4198,  3686,  3072,  2458,   922,    17,  6719,
     &   3414,  2253,  2355,  4198,  6246,  7680,  7680,  6451,  4915,
     &   3482,  2355,  1434,   922,   703,   151,    16,  7139,  3307,
     &   2355,  2867,  5222,  8090,  9830,  9523,  7885,  5734,  3994,
     &   2458,  1638,  1024,   625,    16,  7350,  3221,  2458,  2970,
     &   5734,  8909, 10957, 10547,  8602,  6349,  4198,  2765,  1741,
     &   1024,   493,    16,  7559,  3153,  2458,  3174,  6246,  9830/
      DATA (MAPEMIN (I), I=12601,13050) /
     &  12186, 11674,  9421,  6861,  4608,  2867,  1843,  1126,   304,
     &     15,  7979,  2994,  2560,  3379,  7066, 11366, 14438, 14234,
     &  11469,  8294,  5427,  3584,  2150,  1024,    15,  8400,  2852,
     &   2560,  3789,  7782, 13210, 17101, 16896, 13824,  9933,  6554,
     &   4096,  2560,   307,    14,  8819,  2706,  2560,  3891,  8090,
     &  14438, 19354, 19968, 16691, 12186,  8192,  5120,  2048,    13,
     &   9239,  2558,  2662,  3789,  8499, 15462, 21709, 23347, 19968,
     &  14746, 10035,  6451,    13,  9450,  2481,  2662,  3891,  8602,
     &  15974, 22938, 24986, 21709, 16384, 11059,  5222,    13,  9659,
     &   2408,  2662,  3891,  8704, 16486, 24166, 26829, 23552, 17818,
     &  12288,  3584,    12, 10079,  2242,  2662,  3891,  9011, 17408,
     &  26522, 30515, 27648, 21299, 11674,    12, 10500,  2069,  2662,
     &   3994,  9216, 18227, 28877, 34406, 32154, 25088,  1741,    10,
     &  11550,  1662,  2662,  4096,  9626, 20173, 34611, 45056, 22733,
     &      8, 12600,  1233,  2662,  4096,  9933, 21914, 31130,     6,
     &  13859,   648,  2560,  4198,  4403,     4, 14700,   165,  1843,
     &      3, 15750,     0,     3, 32767,     0,   179, 35199,     3,
     &      0,     0,     3,  5857,     0,     6,  5879,   581,  1434,
     &    614,   205,    12,  6089,  2207,  1536,   819,  1229,  1638,
     &   2150,  2560,  3174,  3379,  2048,    15,  6300,  2866,  1638,
     &   1024,  1638,  2253,  3072,  3789,  4301,  4506,  4198,  3686,
     &   3072,   614,    15,  6719,  2866,  2253,  2355,  4198,  6246,
     &   7680,  7680,  6451,  4915,  3379,  2355,  1536,   205,    14,
     &   7139,  2653,  2355,  2867,  5222,  8090,  9830,  9523,  7885,
     &   5734,  3891,  2560,   717,    13,  7350,  2551,  2458,  2970,
     &   5734,  8909, 10957, 10547,  8602,  6246,  4301,  2662,    13,
     &   7559,  2455,  2458,  3174,  6246,  9830, 12186, 11674,  9421,
     &   6861,  4506,  1946,    12,  7979,  2242,  2560,  3379,  7066,
     &  11366, 14438, 14131, 11571,  8192,  4506,    11,  8400,  2025,
     &   2560,  3789,  7782, 13210, 16998, 16998, 13824,  9114,    11,
     &   8819,  1805,  2560,  3891,  8090, 14438, 19251, 20070, 16691,
     &    819,    10,  9239,  1581,  2662,  3789,  8499, 15360, 21811,
     &  23245,  3891,     9,  9450,  1457,  2662,  3891,  8499, 15974,
     &  22938, 17510,     9,  9659,  1343,  2662,  3891,  8704, 16486,
     &  24064,  6656,     8, 10079,  1105,  2662,  3891,  8909, 17510,
     &   7475,     7, 10500,   865,  2560,  3994,  9216,  5734,     3,
     &  11550,     0,     3, 32767,     0,   119, 38399,     3,     0,
     &      0,     3,  5857,     0,     3,  5879,     0,    10,  6089,
     &   1581,  1536,   819,  1229,  1536,  2150,  2662,   614,    13,
     &   6300,  2356,  1638,  1024,  1638,  2253,  3072,  3789,  4301,
     &   4506,  4198,   819,    12,  6719,  2228,  2253,  2355,  4198,
     &   6246,  7680,  7578,  6451,  4915,  2662,    11,  7139,  1948,
     &   2355,  2867,  5222,  8090,  9830,  9523,  7885,  3686,    10,
     &   7350,  1790,  2458,  2970,  5734,  8909, 10854, 10650,  8499,
     &     10,  7559,  1640,  2458,  3174,  6246,  9830, 12083, 11674,
     &   4198,     9,  7979,  1413,  2560,  3379,  6963, 11469, 14438,
     &   7578,     8,  8400,  1045,  2560,  3686,  7782, 13312,  1331,
     &      6,  8819,   715,  2560,  3789,  6042,     5,  9239,   360,
     &   2560,  1229,     4,  9450,   165,  1843,     4,  9659,     0,
     &   1843,     3, 10080,     0,     3, 32767,     0,    76, 41599/
      DATA (MAPEMIN (I), I=13051,13168) /
     &      3,     0,     0,     3,  5857,     0,     4,  5879,     0,
     &   1843,     7,  6089,   845,  1536,   819,  1126,   512,    10,
     &   6300,  1640,  1638,  1024,  1638,  2150,  3072,  3789,  1741,
     &      9,  6719,  1505,  2253,  2355,  4096,  6349,  7680,  6758,
     &      8,  7139,  1204,  2355,  2765,  5325,  8090,  6861,     8,
     &   7350,  1045,  2458,  2867,  5734,  9011,   819,     7,  7559,
     &    884,  2458,  3174,  6246,  3994,     5,  7979,   488,  2458,
     &   3072,     4,  8400,     0,  5530,     3,  8820,     0,     3,
     &  32767,     0,    43, 44799,     3,     0,     0,     3,  6067,
     &      0,     4,  6089,    81,  1126,     6,  6300,   517,  1638,
     &   1024,     0,     6,  6719,   715,  2150,  2355,  3174,     5,
     &   7139,   407,  2355,  1331,     4,  7350,   248,  2253,     4,
     &   7559,    62,  1229,     3,  7980,     0,     3, 32767,     0,
     &      0/

      END
!--------------------------------------------------------------------------------------------
!+
! NAME:
!	Init_AP8max

! PURPOSE:
!	This subroutine sets block data for AP8max
!
! CATEGORY:
!	Radiation belt model
!
! CALLING SEQUENCE:
!	call Init_AP8max
!
! INPUTS:
!	None
!
! OUTPUTS:
!	None
!
! COMMON BLOCKS:
!	COMMON /PROMAX/ IHEADPMAX, MAPPMAX
!
! MODIFICATION HISTORY:
!	Written by: D. Bilitza, 1987
!				S. Bourdarie (introduced in version 3.0) - September 2005
!-
!--------------------------------------------------------------------------------------------
      SUBROUTINE Init_AP8max
!      BLOCK DATA AP8MAX

      COMMON /PROMAX/ IHEADPMAX, MAPPMAX

      INTEGER*4 IHEADPMAX(8)
      INTEGER*4 MAPPMAX(16291)

      DATA IHEADPMAX /2,4,1970,100,2048,2048,1024,16291/

      DATA (MAPPMAX(I), I=    1,  490) /
     & 1877,     10,      3,      0,      0,      3,   2333,
     &    0,     10,   2355,   1740,     26,     24,     19,
     &   16,     14,     11,      7,     14,   2396,   2814,
     &   47,     35,     30,     19,     11,     10,      6,
     &    7,      6,      4,      4,     17,   2457,   3519,
     &   88,     72,     59,     50,     44,     30,     23,
     &   17,     12,      9,      7,      5,      5,      9,
     &   22,   2662,   4651,    197,    199,    174,    143,
     &  108,     75,     62,     51,     42,     35,     25,
     &   21,     17,     15,     14,     12,     12,     15,
     &   21,     24,   2867,   5303,    199,    226,    254,
     &  258,    242,    220,    188,    163,     96,     73,
     &   61,     32,     25,     18,     15,     14,     14,
     &   15,     16,     19,     25,     26,   3072,   5864,
     &  230,    255,    288,    310,    313,    320,    307,
     &  273,    231,    199,    139,    110,     85,     63,
     &   50,     37,     28,     21,     13,     11,      9,
     &    9,      9,     28,   3276,   6182,    286,    324,
     &  358,    383,    401,    395,    383,    365,    336,
     &  297,    212,    183,    155,    131,    114,     85,
     &   73,     59,     50,     39,     33,     21,     20,
     &   22,     32,     29,   3481,   6577,    294,    340,
     &  389,    450,    477,    460,    481,    478,    436,
     &  377,    297,    254,    233,    212,    189,    170,
     &  155,    136,    112,     83,     60,     48,     29,
     &   20,     24,     45,     30,   3686,   6855,    312,
     &  364,    416,    478,    519,    540,    587,    600,
     &  550,    508,    482,    417,    340,    286,    253,
     &  218,    186,    175,    158,    136,    117,     97,
     &   78,     65,     37,     38,     42,     31,   3891,
     & 7129,    314,    375,    448,    542,    624,    699,
     &  795,    874,    874,    745,    605,    468,    341,
     &  284,    250,    223,    179,    142,    136,    119,
     &  107,    104,    103,    104,    119,    136,    154,
     &  176,     32,   4096,   7407,    334,    394,    475,
     &  577,    703,    827,    972,   1093,   1188,   1074,
     &  908,    636,    435,    348,    274,    219,    165,
     &  164,    158,    139,    125,    119,    115,    115,
     &  114,    119,    125,    125,    127,     33,   4300,
     & 7621,    366,    450,    557,    675,    769,    967,
     & 1265,   1501,   1460,   1187,    946,    678,    526,
     &  425,    350,    258,    249,    241,    211,    203,
     &  181,    155,    141,    142,    142,    146,    149,
     &  149,    160,    209,     34,   4505,   7867,    382,
     &  471,    591,    752,    959,   1200,   1505,   1721,
     & 1812,   1669,   1402,    941,    613,    451,    303,
     &  216,    190,    188,    190,    187,    172,    155,
     &  142,    145,    149,    159,    168,    172,    180,
     &  205,    237,     35,   4710,   8054,    415,    508,
     &  642,    842,   1147,   1476,   1802,   2034,   2054,
     & 1878,   1601,   1346,    780,    611,    400,    311,
     &  287,    277,    258,    218,    188,    181,    176,
     &  168,    162,    159,    162,    166,    170,    179,
     &  197,    281,     35,   4915,   8191,    429,    552,
     &  758,   1046,   1448,   1880,   2204,   2295,   2373,
     & 2118,   1778,   1391,    880,    660,    384,    284,
     &  315,    322,    322,    314,    309,    290,    254,
     &  228,    219,    216,    215,    216,    224,    228,
     &  228,    233,     36,   5120,   8348,    436,    579,
     &  831,   1177,   1604,   2078,   2467,   2696,   2584,
     & 2459,   2157,   1616,   1211,    974,    791,    678,
     &  614,    603,    520,    356,    287,    168,    169,
     &  167,    172,    177,    176,    178,    182,    185,
     &  184,    194,    240,     36,   5324,   8441,    478,
     &  648,    958,   1333,   1836,   2343,   2773,   3311,
     & 3419,   2803,   2140,   1621,   1319,   1072,    844,
     &  694,    522,    447,    366,    324,    325,    326,
     &  328,    330,    331,    334,    335,    330,    324,
     &  324,    330,    346,    380,     37,   5529,   8530,
     &  535,    724,   1032,   1500,   2095,   2747,   3316/

      DATA (MAPPMAX (I), I=    491,  980) /
     & 4200,   3472,   2907,   2306,   2010,   1668,   1471,
     & 1297,   1124,    775,    608,    544,    450,    359,
     &  315,    313,    315,    319,    320,    321,    322,
     &  323,    324,    329,    335,    368,    305,     37,
     & 5734,   8610,    564,    781,   1155,   1669,   2312,
     & 3090,   4074,   5030,   4602,   3857,   3266,   1855,
     & 1449,   1031,    825,    803,    769,    715,    630,
     &  517,    423,    405,    399,    395,    392,    410,
     &  422,    421,    414,    406,    396,    398,    404,
     &  416,     37,   5939,   8675,    582,    833,   1259,
     & 1906,   2829,   3866,   5092,   5899,   5327,   4479,
     & 3301,   2164,   1743,   1398,   1286,   1159,    806,
     &  659,    542,    452,    444,    449,    449,    447,
     &  451,    455,    451,    447,    444,    443,    442,
     &  439,    440,    436,     38,   6144,   8722,    604,
     &  898,   1472,   2354,   3537,   5035,   6435,   7318,
     & 5921,   4842,   3838,   3039,   2426,   1935,   1450,
     & 1152,    921,    737,    566,    486,    412,    325,
     &  316,    310,    311,    309,    319,    328,    328,
     &  325,    317,    302,    301,    295,    280,     38,
     & 6348,   8768,    614,    994,   1649,   2753,   4419,
     & 6504,   8256,   7781,   6309,   4899,   3226,   2528,
     & 1992,   1677,   1409,   1282,   1227,   1049,    858,
     &  720,    721,    714,    631,    538,    442,    368,
     &  364,    370,    375,    375,    375,    370,    361,
     &  349,    325,     38,   6553,   8793,    632,   1027,
     & 1837,   3274,   5576,   7960,   9469,   7581,   6321,
     & 4676,   3714,   3020,   2513,   2112,   1887,   1731,
     & 1408,   1393,   1359,   1236,   1043,    919,    822,
     &  716,    414,    330,    249,    251,    249,    250,
     &  253,    253,    249,    240,    224,     38,   6758,
     & 8790,    663,   1131,   2137,   3947,   6752,   9468,
     &10966,   8668,   6878,   5892,   3838,   2367,   2154,
     & 2026,   1932,   1789,   1474,   1308,   1252,   1189,
     & 1103,    982,    853,    587,    601,    682,    509,
     &  405,    254,    237,    246,    245,    247,    239,
     &  232,     38,   6963,   8763,    745,   1340,   2634,
     & 4863,   8313,  11557,  12095,   9931,   8798,   4460,
     & 3611,   2250,   1768,   1661,   1348,   1094,    886,
     &  860,    794,    728,    706,    870,    860,    847,
     &  854,    868,    878,    878,    875,    870,    869,
     &  850,    832,    804,    778,     38,   7168,   8711,
     &  812,   1573,   2889,   5612,   9755,  13321,  13968,
     &11093,   8522,   5144,   4171,   3210,   2652,   2098,
     & 1824,   1559,   1306,   1098,    936,   1073,   1128,
     &  738,    712,    677,    668,    659,    668,    673,
     &  670,    674,    679,    679,    665,    639,    583,
     &   37,   7372,   8632,    966,   1835,   3701,   6786,
     &10735,  15042,  13841,  12011,   7985,   5449,   4253,
     & 2902,   2557,   2415,   2032,   1781,   1650,   1528,
     & 1385,   1275,   1156,   1045,    933,    851,    829,
     &  820,    827,    845,    852,    845,    832,    818,
     &  801,    772,     37,   7577,   8559,   1040,   1986,
     & 4117,   7854,  12742,  18249,  15231,  13868,   7061,
     & 5599,   3795,   2676,   2102,   1904,   1832,   1706,
     & 1638,   1613,   1519,   1397,   1340,   1304,   1283,
     & 1256,   1245,   1227,   1254,   1281,   1269,   1254,
     & 1223,   1167,   1125,   1010,     37,   7782,   8451,
     & 1037,   2053,   4320,   8234,  13247,  19364,  16095,
     &14423,   8192,   6289,   5014,   3402,   2768,   2848,
     & 2891,   2928,   2726,   2438,   2119,   1818,   1556,
     & 1405,   1251,   1121,   1098,   1078,   1085,   1087,
     & 1092,   1090,   1060,   1020,    948,    798,     36,
     & 7987,   8398,   1040,   2238,   4737,   9429,  15286,
     &20257,  17131,  15236,   9150,   7359,   6088,   5045,
     & 3915,   3181,   2606,   2270,   2114,   1920,   1647,
     & 1372,   1191,   1089,   1223,   1438,   1491,   1439,
     & 1440,   1463,   1484,   1484,   1473,   1442,   1371,
     &   36,   8192,   8322,    992,   1981,   5377,  10700,
     &16271,  21436,  18570,  16084,  11366,   8239,   6445/

      DATA (MAPPMAX (I), I=    981,1470)/
     & 4987,   3301,   3042,   3084,   3095,   3126,   3010,
     & 2801,   2527,   2169,   1640,   1256,   1294,   1271,
     & 1281,   1287,   1312,   1303,   1303,   1277,   1267,
     & 1200,     36,   8396,   8254,   1041,   2217,   5596,
     &11187,  17150,  22697,  20371,  18072,  11189,   8195,
     & 6516,   4650,   3902,   3445,   3211,   3220,   3271,
     & 3298,   3244,   3038,   2642,   2216,   1927,   1921,
     & 1923,   1934,   1945,   1944,   1931,   1913,   1818,
     & 1611,   1213,     36,   8601,   8201,   1062,   2374,
     & 5797,  11755,  18805,  25569,  23593,  20955,  12019,
     & 8725,   6250,   4196,   3353,   3278,   3143,   2868,
     & 2756,   2757,   2671,   2585,   2560,   2549,   2551,
     & 2574,   2602,   2610,   2605,   2580,   2526,   2444,
     & 2308,   2126,   1593,     35,   8806,   8142,   1054,
     & 2438,   6321,  12529,  20567,  29223,  23710,  20002,
     &12878,   9315,   5969,   3639,   3307,   2748,   2263,
     & 2187,   2404,   2992,   3311,   3372,   3364,   3216,
     & 3090,   3025,   2996,   2966,   2948,   2972,   2986,
     & 2965,   2912,   2796,     35,   9011,   8101,   1096,
     & 2839,   6736,  13317,  20827,  26672,  27018,  22557,
     &17850,  11686,   8629,   8386,   7514,   5331,   3986,
     & 3658,   3642,   3478,   3239,   2928,   2551,   2174,
     & 2162,   2157,   2160,   2164,   2211,   2227,   2200,
     & 2127,   2063,   1973,     35,   9216,   8060,   1199,
     & 2964,   6971,  14122,  22414,  28972,  29148,  25542,
     &13803,  10575,   6310,   4825,   4250,   3870,   3880,
     & 3939,   4005,   4006,   3905,   3743,   3714,   3696,
     & 3682,   3697,   3721,   3755,   3751,   3725,   3709,
     & 3663,   3569,   3395,     35,   9420,   8014,   1038,
     & 2625,   7002,  17078,  28580,  38218,  28555,  23784,
     &14905,  11120,   8605,   6453,   4668,   4644,   4592,
     & 4364,   4336,   4353,   4256,   3923,   3552,   3377,
     & 3348,   3337,   3353,   3360,   3388,   3398,   3367,
     & 3347,   3357,   3459,     35,   9625,   7974,   1058,
     & 2778,   8142,  18112,  29158,  38589,  31886,  27614,
     &15379,  11756,   7404,   5758,   4377,   3611,   3336,
     & 3442,   3565,   3785,   4003,   4132,   4345,   4772,
     & 4936,   4992,   5054,   5105,   5101,   5017,   4927,
     & 4705,   4488,   3563,     34,   9830,   7929,   1050,
     & 2852,   7987,  21384,  34239,  43063,  31320,  26541,
     &14358,  11096,   9236,   5255,   5474,   5489,   5449,
     & 5006,   4596,   4461,   4138,   3589,   3142,   3382,
     & 4136,   4720,   4771,   4773,   4900,   4989,   4978,
     & 4896,   4809,     34,  10035,   7880,   1018,   3042,
     & 8752,  26094,  39580,  47541,  33710,  26599,  13861,
     &10928,   7420,   4878,   5170,   4977,   4500,   4616,
     & 5115,   6316,   6277,   5592,   5176,   5024,   4955,
     & 4931,   4938,   5003,   5037,   5092,   5157,   5148,
     & 5274,     34,  10240,   7822,   1097,   3528,  11320,
     &30959,  45133,  53296,  39292,  32724,  20426,  14875,
     &10911,   8529,   5325,   4152,   3082,   3242,   3447,
     & 3667,   3830,   4060,   4536,   4778,   4570,   4412,
     & 4498,   4488,   4240,   4153,   4479,   5171,    839,
     &   34,  10444,   7781,    968,   3330,  11732,  28699,
     &48053,  61860,  38808,  34666,  15336,  12283,   8061,
     & 6482,   6524,   6397,   6063,   5917,   5631,   4974,
     & 4500,   4625,   5421,   5630,   5729,   5650,   5422,
     & 5448,   5637,   5754,   5860,   6570,   1102,     34,
     &10649,   7726,    949,   3800,  15914,  44698,  55677,
     &51712,  41313,  22846,  15318,  12262,   9180,   9015,
     & 8739,   8014,   5932,   4976,   4967,   5779,   6215,
     & 6448,   6513,   6615,   6554,   6502,   6480,   6498,
     & 6628,   6815,   7346,   3383,    628,     34,  10854,
     & 7702,    967,   3603,  14439,  37488,  56986,  69984,
     &48191,  29525,  18211,  11889,   7308,   8061,   8707,
     & 8031,   6669,   6262,   6167,   6127,   6021,   6048,
     & 6238,   6304,   6275,   6251,   6136,   6125,   6194,
     & 6358,   6602,   7457,   2562,     33,  11059,   7646,
     & 1065,   4282,  17494,  43181,  59981,  64441,  49349,
     &28609,  18246,  13018,   6982,   6107,   7135,   7927/

      DATA (MAPPMAX (I), I= 1471,1960 ) /
     & 7979,   7986,   7816,   7772,   7717,   7981,   8334,
     & 8423,   8332,   7992,   7709,   7722,   7903,   8205,
     & 8819,   6261,     33,  11264,   7605,    973,   3803,
     &17998,  46677,  62123,  61152,  47671,  35256,  27396,
     &19708,  13998,  11838,   7860,   7709,   7271,   7088,
     & 7038,   6993,   6975,   7063,   7193,   7321,   7394,
     & 7338,   6942,   6839,   6984,   7192,   7622,   8761,
     &   33,  11468,   7548,    942,   3855,  19292,  44027,
     &64792,  78143,  55940,  35263,  21597,  16626,  13239,
     & 9941,   7579,   7709,   8131,   8605,   8456,   7487,
     & 7011,   7228,   7701,   8007,   8637,   8777,   8512,
     & 8557,   8534,   8790,   9390,  10623,     33,  11673,
     & 7506,    947,   3878,  19826,  46235,  65269,  75468,
     &53427,  46865,  23871,  19784,  16734,  10687,   8942,
     & 7088,   6477,   6806,   7200,   7817,   9634,   9854,
     &10034,  10061,  10011,   9877,   9425,   9649,  10030,
     &10162,  10325,  11420,     33,  11878,   7460,    907,
     & 3666,  19982,  45897,  66887,  80849,  56741,  50996,
     &23345,  18925,  12802,  10291,   9864,   8434,   7716,
     & 8141,   9632,  10005,  10160,  10223,  10100,  10126,
     & 9977,   9974,   9772,   9970,  10363,  10692,  10949,
     &13084,     32,  12083,   7412,    811,   3434,  17320,
     &42273,  67189,  84315,  55937,  48618,  27688,  23055,
     &19461,  16287,  13763,  12860,  12357,  11523,  10926,
     &10808,  10812,  11237,  11275,  11269,  11159,  10811,
     &10811,  11011,  11278,  11900,  13908,     32,  12288,
     & 7377,    828,   3421,  18339,  42434,  68271,  86460,
     &57767,  47630,  30906,  22593,  17972,  14424,  12692,
     &13367,  13302,  12054,  11412,  11977,  12466,  12773,
     &13205,  13097,  12909,  12830,  12615,  12871,  13077,
     &13745,  15513,     32,  12492,   7298,    783,   3006,
     &12800,  41684,  68536,  86332,  60430,  51133,  33738,
     &23827,  18950,  14154,  12822,  14521,  14969,  14684,
     &14673,  14740,  14907,  15092,  15430,  15373,  15240,
     &14907,  14810,  15295,  15691,  16197,  19143,     32,
     &12697,   7264,    884,   3067,  13701,  36703,  65354,
     &85331,  66156,  54399,  38935,  25315,  20801,  17818,
     &16832,  15661,  14372,  14753,  14899,  15089,  15249,
     &15555,  15778,  15866,  15722,  15568,  15074,  15197,
     &15275,  15974,  17333,     32,  12902,   7218,    872,
     & 2653,  12620,  32563,  61213,  80744,  74805,  59579,
     &41742,  25717,  21047,  16106,  17239,  18633,  19574,
     &19841,  20158,  20443,  20652,  20770,  20608,  20472,
     &20016,  20063,  19981,  20821,  22631,  16755,   2324,
     &   31,  13107,   7166,    847,   2232,  12119,  31211,
     &59073,  77927,  76545,  61683,  50501,  41778,  34428,
     &30660,  25422,  21215,  19711,  16970,  16424,  15505,
     &16446,  17930,  18467,  18070,  17821,  17406,  17638,
     &18178,  18809,  20450,     31,  13312,   7134,    896,
     & 2442,  11240,  29292,  56591,  76880,  77929,  63341,
     &55209,  31200,  24727,  22873,  24281,  25283,  25051,
     &24383,  24423,  24909,  25432,  25990,  26029,  25486,
     &25144,  25155,  25658,  26977,  30621,   4292,     31,
     &13516,   7063,    913,   2335,   9874,  30730,  56329,
     &75941,  78564,  67183,  58077,  49459,  43291,  37201,
     &31822,  27507,  25464,  25615,  24834,  23578,  23705,
     &23609,  23880,  24027,  24368,  24596,  25015,  27160,
     &24452,   2795,      3,  20480,      0,      3,  32767,
     &    0,   1823,     20,      3,      0,      0,      3,
     & 2333,      0,     10,   2355,   1731,     26,     24,
     &   19,     16,     14,     11,      7,     14,   2396,
     & 2810,     47,     35,     30,     19,     11,     10,
     &    6,      7,      6,      4,      4,     17,   2457,
     & 3517,     87,     73,     59,     49,     44,     30,
     &   23,     17,     12,     10,      5,      6,      5,
     &    5,     22,   2662,   4649,    195,    199,    176,
     &  143,    107,     75,     62,     51,     42,     35,
     &   25,     21,     18,     14,     15,     12,     11,
     &   15,     14,     24,   2867,   5300,    200,    225,
     &  255,    255,    246,    218,    189,    161,     97/

      DATA (MAPPMAX (I), I= 1961, 2450) /
     &   73,     60,     32,     26,     17,     15,     14,
     &   14,     15,     16,     19,     18,     26,   3072,
     & 5861,    229,    254,    290,    310,    312,    319,
     &  311,    272,    229,    198,    141,    110,     84,
     &   63,     50,     38,     28,     20,     15,      9,
     &    9,      9,     10,     28,   3276,   6178,    286,
     &  328,    355,    381,    402,    398,    381,    362,
     &  338,    299,    211,    181,    157,    131,    113,
     &   85,     73,     59,     49,     40,     32,     21,
     &   20,     22,     23,     29,   3481,   6571,    295,
     &  338,    390,    451,    479,    460,    483,    478,
     &  436,    374,    295,    256,    232,    212,    190,
     &  171,    153,    137,    111,     84,     59,     48,
     &   28,     19,     26,     44,     30,   3686,   6844,
     &  315,    363,    415,    478,    524,    543,    589,
     &  602,    553,    508,    477,    414,    341,    285,
     &  252,    216,    187,    174,    158,    135,    116,
     &   97,     77,     63,     38,     38,     38,     31,
     & 3891,   7090,    323,    390,    457,    557,    629,
     &  720,    796,    890,    857,    741,    579,    451,
     &  340,    285,    246,    223,    172,    145,    132,
     &  118,    108,    104,    101,    107,    121,    140,
     &  153,    154,     32,   4096,   7354,    344,    409,
     &  487,    590,    718,    854,   1001,   1128,   1196,
     & 1044,    890,    585,    424,    337,    262,    210,
     &  163,    165,    157,    137,    123,    116,    114,
     &  115,    116,    121,    124,    125,    131,     33,
     & 4300,   7570,    372,    452,    549,    667,    803,
     & 1017,   1308,   1529,   1404,   1162,    917,    641,
     &  521,    427,    354,    262,    258,    244,    211,
     &  204,    179,    154,    142,    142,    142,    146,
     &  148,    151,    164,    216,     34,   4505,   7793,
     &  379,    455,    570,    731,    977,   1227,   1535,
     & 1743,   1780,   1630,   1365,    939,    620,    455,
     &  352,    222,    194,    193,    198,    192,    175,
     &  154,    141,    144,    147,    161,    171,    173,
     &  180,    211,    218,     35,   4710,   7966,    404,
     &  488,    613,    831,   1185,   1531,   1878,   2067,
     & 2052,   1835,   1553,   1300,    773,    605,    400,
     &  315,    290,    285,    260,    221,    193,    183,
     &  177,    167,    162,    162,    163,    166,    171,
     &  181,    200,    286,     35,   4915,   8093,    426,
     &  545,    746,   1062,   1465,   1848,   2164,   2343,
     & 2409,   2131,   1785,   1365,    876,    642,    372,
     &  276,    326,    334,    327,    311,    304,    287,
     &  253,    229,    219,    216,    217,    219,    220,
     &  225,    231,    238,     36,   5120,   8218,    448,
     &  595,    877,   1251,   1678,   2096,   2446,   2692,
     & 2680,   2424,   2103,   1476,   1135,    949,    791,
     &  677,    601,    582,    524,    362,    306,    172,
     &  170,    166,    172,    175,    173,    173,    177,
     &  188,    187,    193,    237,     36,   5324,   8322,
     &  478,    645,    949,   1348,   1882,   2446,   2841,
     & 3274,   3281,   2732,   2126,   1643,   1330,   1072,
     &  838,    695,    537,    459,    375,    328,    320,
     &  317,    320,    322,    328,    331,    334,    333,
     &  327,    328,    332,    344,    382,     36,   5529,
     & 8423,    502,    675,   1008,   1498,   2133,   2841,
     & 3447,   4058,   3374,   2842,   2267,   2051,   1689,
     & 1478,   1308,   1117,    828,    601,    553,    459,
     &  374,    313,    303,    303,    314,    321,    323,
     &  322,    322,    326,    330,    339,    368,     37,
     & 5734,   8507,    511,    703,   1047,   1615,   2311,
     & 3191,   4062,   4984,   4264,   3746,   3357,   1963,
     & 1500,   1113,    860,    804,    783,    725,    672,
     &  562,    462,    398,    400,    401,    408,    408,
     &  407,    408,    414,    413,    412,    411,    403,
     &  400,     37,   5939,   8572,    533,    728,   1124,
     & 1800,   2678,   3732,   4900,   5751,   5401,   4487,
     & 3657,   2377,   1846,   1480,   1313,   1196,    849/

      DATA (MAPPMAX (I), I= 2451, 2940) /
     &  674,    544,    460,    433,    434,    448,    470,
     &  468,    453,    446,    445,    446,    446,    447,
     &  450,    448,    435,     37,   6144,   8612,    558,
     &  774,   1201,   2043,   3199,   4581,   6000,   6729,
     & 6150,   5230,   4177,   3435,   2699,   2119,   1659,
     & 1148,   1024,    808,    647,    476,    444,    353,
     &  311,    308,    306,    306,    309,    321,    330,
     &  322,    317,    317,    315,    307,     37,   6348,
     & 8651,    567,    797,   1288,   2279,   3628,   5301,
     & 7024,   8013,   6774,   5598,   4258,   2803,   2257,
     & 1862,   1452,   1320,   1326,   1278,   1009,    808,
     &  651,    712,    689,    593,    508,    427,    364,
     &  366,    371,    377,    378,    379,    367,    350,
     &   37,   6553,   8664,    568,    851,   1389,   2557,
     & 4452,   6355,   7759,   8490,   6640,   5263,   4350,
     & 3194,   2883,   2465,   2113,   1914,   1751,   1610,
     & 1507,   1383,   1114,    956,    888,    855,    758,
     &  479,    360,    241,    254,    257,    255,    256,
     &  261,    255,     37,   6758,   8629,    627,    970,
     & 1672,   2901,   5100,   7644,   9516,   9425,   8151,
     & 6673,   4935,   3594,   2152,   2092,   2000,   1879,
     & 1764,   1461,   1356,   1258,   1094,    972,    876,
     &  759,    543,    600,    662,    500,    393,    265,
     &  245,    248,    247,    243,     37,   6963,   8587,
     &  692,   1088,   1992,   3453,   5800,   8838,  11312,
     &10008,   9045,   8377,   4344,   3557,   2187,   1751,
     & 1627,   1330,   1087,    885,    832,    748,    674,
     &  665,    784,    829,    847,    853,    863,    871,
     &  877,    881,    872,    866,    849,    833,     37,
     & 7168,   8524,    715,   1232,   2275,   3899,   6702,
     &10390,  13224,  11731,   9835,   7850,   5081,   4068,
     & 3125,   2531,   2014,   1830,   1581,   1366,   1165,
     & 1009,   1110,   1086,    813,    723,    676,    663,
     &  658,    665,    675,    675,    667,    665,    658,
     &  624,     37,   7372,   8451,    773,   1367,   2618,
     & 4576,   7871,  11806,  13858,  12629,  10811,   7403,
     & 5101,   4004,   2979,   2674,   2530,   2053,   1841,
     & 1661,   1433,   1244,   1213,   1144,   1057,    899,
     &  841,    819,    814,    804,    811,    857,    857,
     &  857,    840,    792,     36,   7577,   8366,    861,
     & 1485,   2861,   5426,   8971,  13447,  18196,  14032,
     &12413,   6198,   4924,   3669,   2440,   2059,   1959,
     & 1818,   1724,   1688,   1612,   1467,   1341,   1318,
     & 1274,   1235,   1225,   1278,   1305,   1267,   1238,
     & 1268,   1264,   1204,   1139,     36,   7782,   8258,
     &  862,   1621,   3341,   5892,   9631,  14765,  19120,
     &14600,  12656,   6845,   5547,   4497,   3150,   2854,
     & 2905,   2883,   2817,   2688,   2447,   2071,   1730,
     & 1483,   1247,   1149,   1112,   1059,   1040,   1063,
     & 1097,   1103,   1101,   1096,   1039,     35,   7987,
     & 8189,    892,   1900,   3740,   6765,  11433,  16464,
     &19962,  15739,  13022,   8269,   6788,   5572,   4483,
     & 3716,   2840,   2311,   2091,   1984,   1747,   1434,
     & 1275,   1107,   1102,   1230,   1434,   1454,   1433,
     & 1429,   1478,   1481,   1482,   1464,     35,   8192,
     & 8121,    893,   1662,   3598,   7085,  11657,  16786,
     &20939,  17674,  15079,  10598,   7683,   6093,   4489,
     & 3115,   3248,   3309,   3329,   3165,   2798,   2588,
     & 2236,   1870,   1297,   1277,   1240,   1259,   1314,
     & 1331,   1373,   1314,   1305,   1263,     35,   8396,
     & 8046,    949,   1822,   4004,   8000,  12815,  17519,
     &21269,  19367,  17952,   9727,   7601,   5913,   4644,
     & 3782,   3391,   3375,   3446,   3450,   3383,   3109,
     & 2671,   2240,   1930,   1876,   1848,   1879,   1912,
     & 1966,   1972,   1891,   1780,   1612,     35,   8601,
     & 7974,    909,   1760,   4040,   8630,  14664,  20515,
     &24478,  21433,  18823,  11208,   8233,   5687,   3540,
     & 3378,   3348,   3101,   3036,   2968,   2768,   2624,
     & 2540,   2486,   2502,   2548,   2613,   2644,   2580,
     & 2511,   2496,   2451,   2307,   2033,     34,   8806/

      DATA (MAPPMAX (I), I= 2941, 3430) /
     & 7902,    946,   1850,   4436,   9078,  15439,  22575,
     &27647,  22347,  16858,  10728,  10253,   3840,   3736,
     & 3182,   2651,   2470,   2718,   3189,   3482,   3741,
     & 3661,   3257,   3049,   2929,   2880,   2977,   3024,
     & 3006,   3015,   3039,   2989,     34,   9011,   7827,
     &  893,   2088,   4736,   9558,  15551,  21121,  24982,
     &26651,  22471,  16142,  10776,   9585,   8246,   6922,
     & 4064,   3745,   3716,   3543,   3316,   3001,   2625,
     & 2245,   2217,   2160,   2165,   2143,   2147,   2180,
     & 2202,   2213,   2103,     34,   9216,   7787,    969,
     & 2107,   4394,   9918,  16365,  22139,  28447,  27878,
     &26460,  11084,   9183,   5611,   4977,   3908,   3907,
     & 4064,   4355,   4686,   4484,   4063,   3759,   3547,
     & 3563,   3692,   3741,   3758,   3838,   3905,   3749,
     & 3542,   3466,     34,   9420,   7720,    887,   1868,
     & 5173,  10913,  19119,  28304,  35276,  26805,  21006,
     &14469,  10736,   8279,   6230,   4862,   5056,   5146,
     & 5163,   5185,   4901,   4182,   3603,   3372,   3367,
     & 3365,   3396,   3434,   3484,   3467,   3370,   3259,
     & 3202,     33,   9625,   7664,    833,   1782,   5252,
     &10431,  18537,  28047,  35189,  30570,  25615,  16505,
     &11291,   7874,   6220,   4617,   3941,   3961,   4097,
     & 4264,   4447,   4609,   4886,   5173,   5166,   4911,
     & 4944,   5253,   5267,   5010,   4959,   4783,     33,
     & 9830,   7586,    809,   1782,   5289,  12249,  23281,
     &33366,  39547,  29082,  24363,  14495,  10840,   8907,
     & 5723,   6086,   6165,   6071,   5592,   5431,   4704,
     & 3818,   3428,   3810,   4571,   4732,   4810,   4957,
     & 4999,   4963,   4884,   4844,     33,  10035,   7519,
     &  842,   1923,   5289,  12332,  24004,  35165,  42545,
     &32331,  27205,  16801,  11950,   9101,   6117,   6557,
     & 5528,   5535,   5851,   6465,   6238,   5554,   5119,
     & 4816,   4764,   4870,   5013,   5145,   5175,   5135,
     & 5206,   5030,     33,  10240,   7449,    847,   2149,
     & 5927,  13285,  27861,  38919,  44550,  36595,  30908,
     &25054,  17307,  13158,   9905,   7109,   4811,   3696,
     & 2974,   3018,   3271,   3714,   3828,   3741,   3948,
     & 4071,   3945,   4183,   4416,   4491,   4365,   4285,
     &   32,  10444,   7363,    913,   2481,   6500,  15562,
     &27859,  42378,  54425,  37530,  33806,  15636,  12266,
     & 8819,   5946,   5793,   5784,   5861,   6251,   6438,
     & 6028,   5301,   4757,   4736,   4882,   5287,   5526,
     & 5406,   6009,   5916,   5180,     32,  10649,   7301,
     &  869,   2290,   7873,  19873,  40272,  52849,  42336,
     &35009,  22993,  16587,  13010,  10166,   8113,   8567,
     & 8434,   8051,   6759,   5430,   5728,   6792,   6807,
     & 7206,   6976,   6556,   6616,   6484,   6640,   6855,
     & 6693,     32,  10854,   7249,    840,   2024,   7399,
     &18463,  37618,  53738,  62446,  45182,  28634,  16794,
     &12581,   7275,   6331,   7164,   8492,   7984,   7797,
     & 6957,   6167,   6207,   6023,   5999,   6058,   6361,
     & 6082,   6139,   5709,   5864,   6608,     32,  11059,
     & 7177,    983,   2665,   8552,  23797,  43003,  57900,
     &57193,  46483,  26874,  17786,  12822,   6715,   5511,
     & 5561,   6707,   7522,   8023,   7809,   7947,   7609,
     & 7882,   8281,   8195,   8021,   8227,   7350,   7354,
     & 7736,   7697,     31,  11264,   7123,    798,   2023,
     & 9118,  28050,  46908,  59170,  51113,  42329,  32758,
     &26764,  18817,  13107,  11586,   8984,   7765,   7692,
     & 7070,   6926,   6862,   7121,   6862,   7313,   7223,
     & 7169,   7064,   7133,   6529,   7041,     31,  11468,
     & 7069,    816,   2218,   9145,  23576,  45698,  62901,
     &70282,  51752,  33441,  21171,  17100,  13787,   9689,
     & 6890,   6219,   6603,   7678,   9234,   8943,   7392,
     & 7545,   8746,   9303,   8955,   8919,   8665,   8596,
     & 8845,     31,  11673,   6985,    802,   2165,  10532,
     &30534,  50991,  64779,  61210,  47415,  40398,  22348,
     &18785,  15802,  12064,  10115,   7791,   6735,   6658,
     & 7032,   7604,   8852,   9547,   9180,   9627,   9700,
     & 9202,  10118,   9615,  10918,     31,  11878,   6935/

      DATA (MAPPMAX (I), I= 3431, 3920) /
     &  786,   2158,  10637,  30692,  55830,  70648,  61262,
     &49479,  43215,  23275,  18317,  13253,  11368,  10653,
     & 8577,   7821,   8762,  10707,  10591,  10145,  10013,
     & 9352,   9163,  10316,  10097,  10816,  11137,  11150,
     &   30,  12083,   6861,    781,   2085,   8396,  32957,
     &52876,  66387,  67951,  51881,  39716,  25957,  20368,
     &16678,  14465,  12938,  11539,  11738,  11258,  11286,
     &10851,  11017,  10691,  11580,  11117,  11673,  11007,
     &11578,  11159,     30,  12288,   6799,    777,   2110,
     &10075,  29071,  56377,  72072,  69929,  52809,  40026,
     &28183,  21131,  16579,  12927,  10434,  12876,  14824,
     &12506,  11152,  12073,  12446,  12927,  13002,  13291,
     &12850,  13021,  12279,  13234,     30,  12492,   6697,
     &  770,   2530,   8512,  27223,  57189,  73132,  70744,
     &53750,  42295,  30148,  21479,  16879,  11707,  12612,
     &16502,  16769,  14420,  14744,  14444,  15125,  14757,
     &15252,  14182,  15472,  16062,  15976,  15639,     30,
     &12697,   6661,    798,   2124,   8389,  28781,  61426,
     &78363,  68808,  53700,  41431,  27940,  22047,  18396,
     &16722,  16322,  16592,  16526,  15177,  13807,  15680,
     &16577,  15893,  14985,  15902,  14450,  15633,  15975,
     &17475,     29,  12902,   6577,    869,   2417,   9169,
     &25418,  51350,  70191,  75936,  57538,  46416,  32032,
     &24399,  19990,  17923,  19133,  20442,  20406,  19557,
     &19364,  18710,  17371,  19273,  21410,  21267,  21752,
     &20135,  19495,     29,  13107,   6514,    755,   2246,
     & 9174,  25906,  54271,  71536,  76131,  56801,  45977,
     &35401,  33015,  31522,  30385,  24834,  20630,  19125,
     &17975,  17774,  18423,  18585,  18098,  17453,  16263,
     &17724,  18445,  18827,     29,  13312,   6448,    863,
     & 2427,  10327,  27125,  54926,  74155,  75530,  59260,
     &49164,  31744,  25283,  22775,  23179,  24323,  24960,
     &24483,  24390,  25180,  25726,  26282,  26240,  25519,
     &25507,  26115,  26368,  27263,     29,  13516,   6400,
     &  745,   1890,   7095,  25419,  45079,  65303,  79899,
     &68012,  60001,  52424,  45500,  38955,  34167,  30242,
     &25962,  23415,  24372,  24157,  23102,  22849,  22146,
     &24514,  26260,  24343,  22772,  26066,      3,  20480,
     &    0,      3,  32767,      0,   1729,     40,      3,
     &    0,      0,      3,   2333,      0,     10,   2355,
     & 1722,     26,     24,     19,     16,     14,     11,
     &   12,     14,   2396,   2806,     47,     35,     30,
     &   19,     11,     10,      6,      7,      6,      4,
     &    3,     17,   2457,   3515,     86,     74,     58,
     &   50,     43,     30,     23,     17,     12,     10,
     &    5,      6,      5,      5,     22,   2662,   4647,
     &  195,    195,    180,    143,    107,     74,     62,
     &   51,     42,     35,     25,     21,     18,     14,
     &   15,     12,     11,     15,     15,     24,   2867,
     & 5297,    198,    227,    256,    253,    249,    218,
     &  187,    160,     98,     72,     61,     31,     26,
     &   17,     16,     13,     15,     14,     17,     18,
     &   18,     26,   3072,   5857,    230,    255,    287,
     &  312,    313,    319,    310,    274,    229,    197,
     &  140,    109,     84,     64,     49,     37,     28,
     &   20,     15,     10,      8,     10,      9,     28,
     & 3276,   6170,    288,    326,    360,    379,    402,
     &  395,    386,    360,    341,    295,    212,    180,
     &  159,    129,    113,     83,     73,     60,     48,
     &   40,     32,     21,     20,     22,     23,     29,
     & 3481,   6559,    295,    339,    391,    450,    480,
     &  465,    485,    476,    435,    372,    296,    253,
     &  233,    213,    192,    170,    153,    136,    110,
     &   82,     59,     47,     28,     20,     25,     45,
     &   30,   3686,   6824,    312,    367,    415,    476,
     &  521,    554,    599,    605,    551,    511,    476,
     &  409,    336,    287,    249,    216,    184,    177,
     &  159,    133,    114,     96,     76,     61,     38,
     &   38,     39,     31,   3891,   7016,    348,    410,
     &  478,    581,    651,    735,    822,    885,    829/

      DATA (MAPPMAX (I), I= 3921, 4410) /
     &  719,    554,    430,    338,    284,    243,    223,
     &  159,    145,    129,    116,    108,    104,    102,
     &  108,    128,    142,    156,    126,     32,   4096,
     & 7264,    360,    424,    501,    614,    745,    895,
     & 1051,   1209,   1207,   1001,    844,    501,    401,
     &  319,    249,    192,    161,    168,    154,    131,
     &  119,    114,    111,    115,    121,    123,    123,
     &  127,    137,     33,   4300,   7470,    385,    456,
     &  541,    671,    865,   1096,   1403,   1552,   1337,
     & 1104,    845,    592,    512,    426,    353,    265,
     &  267,    249,    214,    202,    177,    154,    143,
     &  140,    143,    145,    148,    153,    170,    235,
     &   33,   4505,   7667,    384,    462,    561,    737,
     & 1013,   1261,   1560,   1758,   1772,   1606,   1375,
     &  865,    606,    447,    353,    224,    201,    200,
     &  206,    197,    172,    150,    141,    142,    148,
     &  165,    173,    173,    181,    227,     34,   4710,
     & 7821,    403,    496,    642,    866,   1177,   1537,
     & 1853,   2075,   2055,   1815,   1527,   1286,    768,
     &  602,    399,    315,    296,    288,    263,    224,
     &  198,    188,    180,    169,    164,    163,    163,
     &  166,    172,    184,    202,     35,   4915,   7939,
     &  413,    536,    749,   1022,   1382,   1764,   2129,
     & 2368,   2504,   2219,   1840,   1393,    866,    629,
     &  372,    276,    328,    330,    321,    312,    306,
     &  287,    253,    229,    218,    216,    222,    227,
     &  225,    224,    228,    237,     35,   5120,   8054,
     &  426,    565,    829,   1139,   1575,   2078,   2415,
     & 2649,   2823,   2492,   2137,   1429,   1135,    942,
     &  788,    674,    585,    570,    539,    434,    326,
     &  230,    173,    166,    169,    174,    177,    178,
     &  180,    188,    185,    182,     35,   5324,   8154,
     &  456,    616,    898,   1228,   1697,   2291,   2787,
     & 3227,   3284,   2779,   2305,   1830,   1369,   1110,
     &  887,    723,    552,    479,    382,    335,    320,
     &  315,    318,    320,    322,    324,    327,    326,
     &  325,    328,    336,    352,     36,   5529,   8242,
     &  454,    624,    957,   1391,   1982,   2631,   3196,
     & 3752,   3511,   2947,   2429,   2147,   1892,   1469,
     & 1336,   1144,    991,    610,    593,    507,    410,
     &  317,    309,    296,    304,    311,    316,    319,
     &  323,    331,    335,    338,    355,     36,   5734,
     & 8298,    482,    661,   1010,   1501,   2147,   2969,
     & 3887,   4579,   4226,   3928,   3400,   2362,   1583,
     & 1212,    892,    779,    776,    761,    738,    632,
     &  507,    412,    400,    398,    397,    398,    397,
     &  399,    411,    417,    416,    412,    404,     36,
     & 5939,   8357,    469,    651,   1026,   1560,   2225,
     & 3199,   4296,   5081,   5184,   5025,   4185,   3319,
     & 2015,   1667,   1425,   1280,   1038,    695,    563,
     &  475,    414,    426,    441,    475,    484,    458,
     &  443,    440,    437,    435,    445,    456,    459,
     &   36,   6144,   8378,    486,    675,   1066,   1671,
     & 2601,   3741,   4924,   5834,   6432,   5814,   4874,
     & 3720,   3058,   2452,   1964,   1549,   1119,    924,
     &  732,    592,    464,    434,    372,    318,    319,
     &  313,    305,    305,    321,    324,    319,    319,
     &  316,     36,   6348,   8401,    500,    689,   1086,
     & 1824,   2802,   3976,   5384,   6617,   7000,   5968,
     & 5001,   3943,   2933,   2324,   1933,   1666,   1414,
     & 1272,   1252,   1034,    800,    646,    687,    704,
     &  625,    522,    479,    358,    363,    383,    387,
     &  383,    380,     36,   6553,   8384,    519,    729,
     & 1176,   1960,   3104,   4650,   6180,   7358,   7286,
     & 6088,   5028,   4191,   3055,   2713,   2316,   2055,
     & 1958,   1828,   1767,   1662,   1518,   1129,    991,
     &  915,    875,    755,    617,    373,    268,    253,
     &  254,    254,    257,     36,   6758,   8335,    557,
     &  776,   1280,   2304,   3632,   5290,   7149,   8340,
     & 8130,   7246,   6216,   5555,   3368,   2397,   2256/

      DATA (MAPPMAX (I), I= 4411, 4900) /
     & 2224,   1947,   1704,   1643,   1429,   1374,   1163,
     & 1007,    889,    779,    605,    579,    649,    589,
     &  404,    309,    241,    242,     36,   6963,   8298,
     &  592,    848,   1415,   2376,   3737,   5849,   8348,
     &10073,   9390,   8835,   7683,   5075,   3555,   2464,
     & 1738,   1695,   1454,   1158,    927,    801,    634,
     &  643,    629,    713,    823,    855,    879,    852,
     &  850,    885,    893,    858,    844,     36,   7168,
     & 8214,    636,    960,   1678,   2889,   4357,   6511,
     & 9287,  11044,  11967,   9811,   7710,   5266,   4051,
     & 3176,   2761,   2129,   1953,   1597,   1417,   1221,
     & 1082,   1084,   1022,    769,    733,    692,    651,
     &  630,    661,    685,    702,    671,    662,     35,
     & 7372,   8135,    661,   1033,   1779,   2960,   4680,
     & 7568,  10970,  13804,  11906,   9969,   7413,   5084,
     & 4136,   3373,   2808,   2657,   2137,   1832,   1685,
     & 1447,   1293,   1246,   1158,   1031,    880,    837,
     &  850,    847,    842,    841,    846,    869,     35,
     & 7577,   8044,    720,   1172,   1942,   3371,   5529,
     & 8685,  12837,  16675,  13880,  12144,   6599,   4959,
     & 3534,   2891,   2150,   2075,   1877,   1749,   1691,
     & 1595,   1462,   1371,   1345,   1289,   1222,   1224,
     & 1255,   1260,   1247,   1253,   1256,   1232,     34,
     & 7782,   7929,    684,   1168,   2082,   3496,   5891,
     &10634,  15383,  17134,  13375,  11212,   7605,   5775,
     & 4627,   3525,   3122,   3075,   2893,   2653,   2745,
     & 2495,   1991,   1616,   1384,   1176,   1117,   1076,
     & 1085,   1099,   1083,   1074,   1083,     34,   7987,
     & 7853,    682,   1212,   2319,   4079,   6969,  11080,
     &15678,  19308,  14880,  12531,   8306,   6912,   5826,
     & 4839,   3865,   3128,   2313,   2143,   1995,   1703,
     & 1367,   1216,   1124,   1114,   1270,   1464,   1486,
     & 1452,   1476,   1511,   1499,     34,   8192,   7762,
     &  691,   1166,   2101,   4028,   7073,  11837,  17108,
     &20897,  16271,  13495,  10663,   8079,   6209,   4851,
     & 3601,   3639,   3674,   3622,   3160,   2787,   2417,
     & 2010,   1487,   1297,   1246,   1255,   1313,   1304,
     & 1274,   1281,   1310,     33,   8396,   7665,    726,
     & 1209,   2103,   3957,   7744,  13018,  17517,  20539,
     &19122,  16144,  11954,   7881,   6310,   4929,   4193,
     & 3925,   4044,   3946,   3476,   3007,   2568,   2166,
     & 1888,   1831,   1850,   1944,   1968,   1936,   1897,
     & 1875,     33,   8601,   7555,    715,   1243,   2257,
     & 4335,   8834,  14414,  18950,  22150,  22242,  18072,
     &14230,   8102,   6043,   3441,   3585,   3603,   3535,
     & 3427,   2995,   2666,   2565,   2588,   2552,   2502,
     & 2522,   2561,   2568,   2603,   2582,   2448,     33,
     & 8806,   7446,    724,   1237,   2270,   4505,   9269,
     &15689,  20990,  23403,  22992,  17855,  13080,  10640,
     & 4181,   4117,   3697,   3325,   3474,   3535,   3907,
     & 4343,   3606,   3120,   2920,   2864,   2943,   3045,
     & 3037,   3007,   3010,   3017,     32,   9011,   7336,
     &  669,   1205,   2524,   5361,  10336,  15446,  19712,
     &24382,  27314,  21979,  18711,  11126,  10066,   8103,
     & 5789,   4026,   4084,   3792,   3355,   3025,   2621,
     & 2256,   2225,   2190,   2132,   2045,   2166,   2294,
     & 2243,     32,   9216,   7281,    652,   1330,   2495,
     & 4785,   9922,  15101,  20209,  25869,  29798,  21400,
     &18467,   9554,   6663,   5213,   4329,   4704,   5238,
     & 5545,   4910,   4009,   3695,   3588,   3707,   3745,
     & 3774,   3795,   3836,   3779,   3675,     32,   9420,
     & 7172,    648,   1100,   2163,   4914,   9714,  17103,
     &26120,  32965,  26529,  21928,  16975,  12304,   9400,
     & 7517,   6355,   6261,   6281,   6145,   5807,   4281,
     & 3658,   3420,   3412,   3371,   3350,   3431,   3513,
     & 3412,   3299,     31,   9625,   7065,    672,   1132,
     & 2128,   4687,  10957,  17186,  23417,  29623,  33011,
     &25639,  20833,  12871,   9722,   7032,   5483,   4778,
     & 5129,   5358,   5385,   5545,   5415,   5123,   5054,
     & 5128,   5158,   5143,   5085,   4955,     31,   9830/

      DATA (MAPPMAX (I), I= 4901, 5390) /
     & 6962,    631,   1101,   2226,   5505,  11379,  20400,
     &27863,  32131,  33496,  25980,  19618,  12296,   9749,
     & 7726,   6978,   7083,   7086,   6938,   6282,   4735,
     & 4723,   4655,   4622,   4641,   4701,   4924,   5164,
     & 5256,     30,  10035,   6869,    637,   1158,   2411,
     & 5192,  10356,  19347,  31365,  39256,  36683,  28179,
     &21710,  12515,  10052,   7476,   7094,   6968,   7105,
     & 7070,   6688,   5477,   4954,   5026,   4962,   4812,
     & 4900,   4918,   5176,     30,  10240,   6742,    583,
     & 1152,   2285,   6259,  11563,  20795,  33920,  40679,
     &37693,  30520,  25570,  18554,  14454,  11244,   8857,
     & 7159,   5151,   3848,   2752,   2742,   2720,   2734,
     & 2817,   2945,   3005,   3054,   3466,     29,  10444,
     & 6647,    595,   1233,   2830,   6598,  13936,  19993,
     &27694,  38241,  46214,  32954,  28378,  16005,  11367,
     & 8567,   6233,   5044,   4694,   4737,   4935,   5263,
     & 5576,   5186,   5200,   5235,   5320,   5736,     29,
     &10649,   6562,    577,   1020,   2338,   7960,  16337,
     &25748,  38181,  45510,  35194,  28459,  19134,  15482,
     &11696,   9194,   7185,   6346,   7131,   7027,   6688,
     & 6444,   5784,   4762,   5271,   5804,   7532,   8539,
     &   29,  10854,   6507,    575,   1011,   2010,   6273,
     &13434,  24141,  35664,  45768,  52681,  36664,  29050,
     &17037,  12611,   9089,   5564,   4694,   4789,   5955,
     & 7825,   7211,   7144,   6863,   6469,   6128,   6300,
     & 7173,     29,  11059,   6415,    593,   1058,   2261,
     & 6570,  14624,  26767,  40713,  50212,  51243,  37557,
     &28401,  15931,  12570,   9502,   5386,   4229,   3511,
     & 4441,   7044,   7645,   8458,   8182,   7809,   7581,
     & 7989,   9594,     28,  11264,   6312,    551,   1169,
     & 2819,   8168,  18523,  32619,  43526,  48967,  44626,
     &35790,  28398,  25372,  15571,  11893,   9582,   9086,
     & 8327,   5997,   5955,   6345,   6922,   6836,   7097,
     & 7082,   8164,     28,  11468,   6254,    522,    989,
     & 2063,   7902,  17305,  30520,  47513,  57339,  56910,
     &39981,  31295,  18539,  15808,  13063,  10420,   8466,
     & 6841,   5954,   5985,   5462,   6160,   7607,   9195,
     & 9343,  10548,     28,  11673,   6174,    532,   1148,
     & 2816,   8536,  19564,  37383,  52419,  60140,  48437,
     &37415,  32106,  17427,  14619,  14241,  12927,  11105,
     & 8441,   8585,   6784,   5863,   6320,   6790,   7827,
     & 9776,  12791,     27,  11878,   6118,    495,   1046,
     & 2604,   7110,  16073,  34702,  53822,  61698,  57052,
     &44427,  36891,  22776,  15991,  11714,   9742,   8613,
     & 8573,   8523,   9325,  10671,  10232,  10172,  10805,
     &11651,     27,  12083,   6024,    516,    981,   2537,
     & 6364,  17924,  32643,  47934,  57968,  60744,  42441,
     &37721,  19917,  19844,  17905,  15453,  13517,  12069,
     &10731,  11580,  12610,  12280,  11155,  11214,  11731,
     &   27,  12288,   5958,    518,   1126,   2798,   9189,
     &19144,  33227,  51783,  61997,  59549,  45571,  37066,
     &29545,  20421,  16737,  13552,  10859,   9268,  11698,
     &13140,  11890,   9964,  10136,  11164,  13288,     26,
     &12492,   5856,    546,    994,   2736,   8011,  23367,
     &41794,  53149,  57883,  53521,  44890,  37578,  31088,
     &22362,  17355,  13364,  10932,  13957,  16525,  15436,
     &13308,  14259,  15875,  17840,     26,  12697,   5780,
     &  607,   1329,   3264,   7946,  18952,  38041,  55682,
     &65096,  55790,  45876,  37638,  33674,  21738,  17694,
     &14669,  13995,  13823,  12517,  12277,  14018,  15211,
     &16249,  18523,     26,  12902,   5723,    558,   1326,
     & 3407,   8387,  16907,  29155,  46721,  60510,  66896,
     &50971,  41862,  31743,  23214,  21002,  18420,  17813,
     &18426,  18440,  18144,  20366,  20695,  22654,  24027,
     &   25,  13107,   5632,    567,   1293,   3332,   9680,
     &18302,  31061,  47744,  59548,  61630,  49375,  41346,
     &31700,  28642,  26886,  25110,  25405,  23765,  20695,
     &18811,  18947,  18684,  17407,     25,  13312,   5590,
     &  552,   1350,   3732,   9868,  18713,  32323,  49373,
     &59896,  57954,  50981,  43085,  36275,  24203,  22061/

      DATA (MAPPMAX (I), I= 5391, 5880) /
     &24868,  24655,  25465,  24847,  26000,  26891,  25776,
     &28183,     25,  13516,   5513,    575,   1314,   3366,
     & 8288,  20569,  31190,  43393,  56530,  65647,  56364,
     &46750,  36950,  30241,  30388,  31832,  40346,  34160,
     &29860,  25644,  24299,  24934,  26238,      3,  20480,
     &    0,      3,  32767,      0,   1639,     60,      3,
     &    0,      0,      3,   2334,      0,     10,   2355,
     & 1717,     26,     24,     19,     16,     14,     11,
     &   12,     14,   2396,   2804,     47,     35,     30,
     &   19,     11,     10,      6,      7,      6,      4,
     &    4,     17,   2457,   3514,     86,     73,     58,
     &   50,     43,     30,     23,     17,     12,     10,
     &    5,      6,      5,      5,     22,   2662,   4645,
     &  192,    196,    182,    143,    106,     74,     62,
     &   51,     42,     35,     25,     21,     18,     14,
     &   15,     12,     11,     15,     15,     24,   2867,
     & 5294,    198,    227,    254,    254,    251,    218,
     &  186,    160,     98,     72,     60,     32,     26,
     &   16,     16,     13,     15,     15,     16,     18,
     &   18,     26,   3072,   5850,    229,    257,    289,
     &  310,    313,    321,    312,    273,    228,    195,
     &  141,    108,     84,     62,     50,     37,     27,
     &   20,     15,     10,      8,     10,      9,     28,
     & 3276,   6166,    286,    326,    360,    378,    401,
     &  397,    388,    358,    342,    297,    213,    177,
     &  161,    129,    113,     82,     73,     59,     49,
     &   39,     32,     21,     20,     23,     22,     29,
     & 3481,   6546,    294,    339,    390,    456,    478,
     &  467,    486,    479,    433,    370,    293,    253,
     &  234,    215,    193,    172,    150,    137,    108,
     &   81,     58,     46,     27,     21,     26,     44,
     &   30,   3686,   6802,    315,    365,    418,    473,
     &  526,    564,    606,    606,    553,    508,    476,
     &  404,    333,    282,    250,    215,    185,    176,
     &  160,    131,    112,     94,     76,     58,     38,
     &   39,     38,     31,   3891,   6979,    349,    409,
     &  477,    582,    664,    750,    829,    870,    823,
     &  715,    555,    428,    337,    287,    247,    225,
     &  154,    147,    130,    115,    107,    105,    101,
     &  111,    130,    145,    151,    123,     32,   4096,
     & 7202,    360,    430,    511,    620,    760,    917,
     & 1078,   1252,   1213,   1001,    790,    476,    388,
     &  307,    240,    184,    161,    168,    153,    130,
     &  117,    112,    112,    118,    121,    121,    123,
     &  132,    141,     32,   4300,   7380,    392,    470,
     &  543,    672,    931,   1167,   1468,   1559,   1278,
     & 1051,    800,    578,    496,    414,    343,    271,
     &  269,    247,    215,    201,    174,    153,    143,
     &  141,    141,    145,    149,    157,    173,     33,
     & 4505,   7559,    387,    470,    569,    747,   1031,
     & 1304,   1603,   1773,   1772,   1573,   1374,    815,
     &  591,    434,    333,    226,    206,    203,    211,
     &  199,    172,    148,    142,    143,    150,    166,
     &  173,    173,    184,    241,     34,   4710,   7700,
     &  397,    492,    654,    873,   1163,   1534,   1885,
     & 2091,   2066,   1802,   1512,   1274,    763,    597,
     &  401,    315,    302,    297,    270,    224,    196,
     &  188,    180,    169,    165,    162,    164,    166,
     &  173,    186,    205,     34,   4915,   7817,    399,
     &  520,    723,    983,   1310,   1751,   2151,   2394,
     & 2539,   2282,   1871,   1418,    862,    624,    371,
     &  272,    327,    328,    324,    317,    311,    289,
     &  251,    222,    217,    218,    225,    230,    225,
     &  221,    227,     34,   5120,   7922,    408,    544,
     &  779,   1093,   1501,   2002,   2396,   2626,   2902,
     & 2546,   2192,   1386,   1130,    938,    793,    685,
     &  596,    555,    534,    483,    345,    282,    175,
     &  171,    166,    174,    180,    179,    183,    188,
     &  186,     35,   5324,   8005,    430,    584,    833,
     & 1135,   1610,   2258,   2738,   3148,   3248,   2852/

      DATA (MAPPMAX (I), I= 5881, 6370) /
     & 2394,   1896,   1389,   1160,    948,    744,    598,
     &  496,    390,    340,    321,    318,    321,    322,
     &  319,    318,    327,    334,    330,    329,    337,
     &  345,     35,   5529,   8079,    437,    583,    872,
     & 1239,   1815,   2497,   3008,   3518,   3736,   3032,
     & 2549,   2206,   1981,   1479,   1326,   1179,   1007,
     &  749,    582,    541,    447,    370,    318,    306,
     &  308,    316,    317,    314,    319,    330,    334,
     &  334,     35,   5734,   8120,    451,    616,    928,
     & 1358,   2006,   2723,   3356,   4046,   4373,   4102,
     & 3451,   2925,   1741,   1354,   1020,    818,    786,
     &  790,    778,    704,    554,    454,    400,    395,
     &  391,    391,    394,    403,    413,    418,    419,
     &  419,     35,   5939,   8166,    434,    590,    896,
     & 1358,   2039,   2892,   3808,   4653,   5095,   5213,
     & 4369,   3632,   2476,   1783,   1496,   1321,   1167,
     &  836,    635,    511,    431,    437,    450,    460,
     &  462,    456,    453,    439,    425,    437,    448,
     &  456,     35,   6144,   8172,    443,    613,    956,
     & 1406,   2186,   3270,   4200,   4906,   5635,   6031,
     & 5108,   4300,   3477,   2951,   2416,   1902,   1463,
     & 1075,    856,    701,    581,    485,    422,    310,
     &  319,    315,    315,    315,    314,    319,    321,
     &  312,     35,   6348,   8158,    467,    630,    965,
     & 1535,   2402,   3560,   4758,   5862,   6883,   5815,
     & 5033,   4275,   3462,   2587,   2210,   1875,   1590,
     & 1363,   1233,   1163,    967,    780,    662,    745,
     &  740,    637,    516,    454,    369,    374,    376,
     &  384,     35,   6553,   8134,    472,    634,    982,
     & 1572,   2537,   3766,   5143,   6421,   7133,   6417,
     & 5364,   4185,   3979,   3124,   2798,   2384,   2101,
     & 1941,   1828,   1749,   1599,   1406,   1049,    934,
     &  892,    859,    720,    545,    353,    251,    250,
     &  243,     35,   6758,   8076,    494,    673,   1062,
     & 1683,   2769,   4132,   5565,   6921,   8058,   7400,
     & 6683,   5884,   5410,   3099,   2491,   2264,   2181,
     & 1937,   1719,   1683,   1619,   1418,   1129,    980,
     &  861,    738,    572,    623,    666,    524,    369,
     &  265,     35,   6963,   8014,    519,    720,   1175,
     & 2004,   3122,   4531,   6786,   8972,   8833,   8897,
     & 8207,   7248,   4298,   3142,   2184,   1780,   1732,
     & 1392,   1077,    853,    734,    640,    632,    602,
     &  755,    850,    847,    860,    873,    894,    911,
     &  888,     34,   7168,   7934,    548,    731,   1216,
     & 2189,   3377,   4756,   7082,   9853,  10912,  11606,
     & 9235,   6759,   4877,   3753,   3059,   2662,   2138,
     & 1964,   1583,   1377,   1161,   1053,   1040,    941,
     &  732,    716,    695,    639,    651,    694,    708,
     &   34,   7372,   7838,    549,    744,   1302,   2380,
     & 3749,   5573,   8220,  11210,  12920,  11532,   9567,
     & 6741,   4662,   3856,   3239,   2852,   2608,   2048,
     & 1869,   1647,   1396,   1251,   1163,   1057,    956,
     &  843,    856,    862,    857,    864,    848,     34,
     & 7577,   7736,    604,    905,   1395,   2467,   4166,
     & 6275,   9731,  13695,  16717,  12416,  10530,   6377,
     & 4624,   3628,   2571,   2257,   2100,   1809,   1776,
     & 1658,   1442,   1312,   1331,   1321,   1244,   1196,
     & 1273,   1299,   1237,   1213,   1301,     33,   7782,
     & 7608,    567,    860,   1576,   2793,   4340,   6755,
     &10578,  14463,  17058,  13071,  10830,   8050,   5562,
     & 4457,   3202,   3114,   3079,   2931,   2826,   2642,
     & 2192,   1772,   1448,   1224,   1128,   1070,   1074,
     & 1087,   1073,   1067,     33,   7987,   7523,    557,
     &  878,   1680,   2907,   4828,   7814,  12077,  16238,
     &18660,  13733,  11188,   8037,   7139,   5736,   4711,
     & 3797,   2975,   2282,   2163,   1869,   1555,   1278,
     & 1148,   1023,   1173,   1425,   1518,   1476,   1471,
     & 1478,     32,   8192,   7391,    590,    980,   1693,
     & 2991,   5157,   8283,  12873,  17109,  18991,  15146,
     &12872,  10792,   8759,   6149,   4591,   3927,   3993/

      DATA (MAPPMAX (I), I= 6371, 6860) /
     & 3887,   3506,   2870,   2430,   1958,   1449,   1287,
     & 1257,   1274,   1287,   1292,   1305,     32,   8396,
     & 7281,    543,    891,   1520,   2903,   5143,   8258,
     &13336,  17826,  19772,  18435,  15242,  12815,   8118,
     & 6238,   5110,   4685,   4583,   4314,   3645,   2912,
     & 2482,   2109,   1909,   1851,   1884,   1928,   1927,
     & 1931,   1914,     31,   8601,   7131,    532,    913,
     & 1616,   3018,   5419,   9207,  14456,  19329,  22536,
     &23381,  16778,  13874,   7750,   4716,   3579,   3854,
     & 4079,   3935,   3234,   2625,   2585,   2586,   2572,
     & 2550,   2561,   2513,   2596,   2641,     31,   8806,
     & 6996,    572,    921,   1501,   2980,   5600,   9375,
     &14677,  20369,  24389,  24520,  17408,  13444,   9017,
     & 4857,   4659,   4261,   4147,   4039,   4318,   4344,
     & 3254,   3086,   2869,   2900,   2924,   2944,   3095,
     & 3087,     30,   9011,   6848,    530,    818,   1760,
     & 3185,   6233,  10485,  14794,  20139,  26319,  29906,
     &21873,  17618,  12742,   9031,   7680,   4497,   4400,
     & 3762,   3281,   2890,   2469,   2220,   2200,   2147,
     & 2057,   2016,   2237,     30,   9216,   6747,    523,
     & 1023,   1851,   2894,   5589,   9671,  15792,  23617,
     &29287,  30029,  21959,  13311,   8274,   6171,   5522,
     & 5686,   6305,   6033,   4744,   4050,   3610,   3657,
     & 3831,   3769,   3891,   3915,   3735,     29,   9420,
     & 6630,    506,    864,   1562,   3128,   5983,  10205,
     &16453,  24342,  29366,  27618,  22287,  18444,  15934,
     &10004,   8329,   7515,   7355,   6759,   6134,   3818,
     & 3562,   3340,   3275,   3238,   3322,   3645,     29,
     & 9625,   6476,    477,    797,   1440,   2922,   5784,
     &10417,  15984,  21989,  27546,  31260,  27986,  21611,
     &16924,   9638,   7717,   6683,   6556,   6506,   6349,
     & 6001,   5293,   5209,   4942,   5024,   5189,   5175,
     &   28,   9830,   6306,    464,    782,   1427,   2895,
     & 6024,  10674,  18654,  27231,  33011,  35322,  24660,
     &19722,  14250,  10701,   9764,   9595,   8599,   7452,
     & 6571,   5096,   4920,   4605,   4809,   5160,   5374,
     &   28,  10035,   6198,    462,    764,   1332,   2501,
     & 4925,   9551,  20687,  33597,  38128,  35021,  25249,
     &19731,  14932,  11406,   9445,   8999,   8798,   7941,
     & 6849,   5989,   5120,   5136,   4775,   4954,   5494,
     &   27,  10240,   6012,    438,    780,   1255,   2525,
     & 5944,  11157,  19076,  32094,  37873,  38300,  32050,
     &24974,  19668,  15500,  10208,   8437,   7279,   5732,
     & 4322,   3172,   2437,   2429,   2529,   2820,     26,
     &10444,   5868,    477,    827,   1813,   3849,   7666,
     &12670,  18867,  26402,  34617,  39481,  37032,  28169,
     &17309,  11409,   8718,   6897,   4115,   4406,   4557,
     & 4302,   4362,   4968,   6242,     26,  10649,   5785,
     &  436,    722,   1336,   3601,   7989,  13927,  21969,
     &32624,  39225,  41524,  26749,  21618,  15291,  12291,
     & 9715,   7792,   6324,   5440,   6243,   6564,   6352,
     & 6289,   6542,     26,  10854,   5698,    433,    685,
     & 1289,   2751,   6469,  11923,  20864,  30169,  40848,
     &46406,  39422,  29559,  22108,  13079,  10261,   7799,
     & 5762,   4310,   4430,   5449,   6948,   7797,   8185,
     &   25,  11059,   5595,    446,    768,   1434,   3283,
     & 7281,  12657,  22225,  32734,  42324,  47519,  37322,
     &29369,  23337,  14421,  11182,   9078,   5884,   4565,
     & 3815,   4131,   5703,   7863,     25,  11264,   5475,
     &  439,    724,   1428,   3344,   7002,  13165,  26384,
     &37240,  42053,  41304,  33930,  29309,  25763,  23855,
     &15288,  12002,   8386,   8387,   8272,   7777,   6884,
     & 8015,     25,  11468,   5403,    407,    670,   1361,
     & 3225,   8170,  16210,  22823,  33431,  46122,  51619,
     &44923,  32574,  25451,  16367,  12975,  13330,  12522,
     &10730,   8786,   7241,   6299,   6894,     24,  11673,
     & 5343,    434,    736,   1491,   3523,   8030,  14600,
     &23479,  34264,  41617,  44817,  46284,  38004,  30996,
     &20977,  14110,  12001,  12013,  11583,  10560,  10184,
     &10320,     24,  11878,   5279,    405,    641,   1343/

      DATA (MAPPMAX (I), I= 6861, 7350) /
     & 3166,   8276,  15605,  24300,  35302,  43727,  48280,
     &50072,  42427,  34254,  26641,  17543,  13267,   9969,
     & 7599,   7399,   8688,  10483,     24,  12083,   5226,
     &  376,    589,   1132,   2526,   6004,  13634,  24277,
     &33576,  43140,  49362,  50744,  41298,  32325,  23597,
     &18439,  18196,  15509,  14539,  13375,  11569,  12371,
     &   24,  12288,   5162,    409,    654,   1223,   2547,
     & 6141,  11659,  19324,  32440,  48198,  55163,  54196,
     &41046,  35533,  30532,  26128,  17942,  14105,  11156,
     & 9471,  10682,  15977,     23,  12492,   5074,    399,
     &  535,    939,   2597,   5542,  11726,  22236,  37803,
     &47535,  51838,  51829,  39629,  36029,  32355,  27686,
     &20208,  15186,  12529,  15148,  20934,     23,  12697,
     & 5004,    418,    677,   1236,   2564,   6403,  12421,
     &19437,  30460,  43470,  50770,  51717,  44239,  39498,
     &34625,  30313,  21063,  17651,  16360,  16296,  17820,
     &   23,  12902,   4957,    425,    689,   1206,   2380,
     & 5354,   9575,  16320,  25961,  38164,  48999,  54553,
     &52135,  42613,  35792,  29853,  22750,  19924,  21932,
     &22860,  24688,     23,  13107,   4870,    437,    706,
     & 1300,   2729,   6324,  11258,  18987,  30622,  42236,
     &48906,  49713,  43792,  39070,  35058,  31792,  29581,
     &28997,  29017,  29804,  38809,     22,  13312,   4815,
     &  440,    682,   1260,   2653,   6620,  11436,  18930,
     &30990,  43426,  50102,  48890,  46114,  42962,  36651,
     &29106,  20937,  20907,  26226,  29572,     22,  13516,
     & 4758,    438,    733,   1327,   2456,   5208,  10804,
     &19805,  26527,  34336,  43649,  50222,  50500,  43690,
     &40191,  36566,  32282,  28428,  29400,  39894,      3,
     &20480,      0,      3,  32767,      0,   1556,     80,
     &    3,      0,      0,      3,   2334,      0,     10,
     & 2355,   1712,     26,     24,     19,     16,     14,
     &   11,     12,     14,   2396,   2800,     47,     35,
     &   30,     19,     11,     10,      6,      7,      6,
     &    4,      4,     17,   2457,   3512,     85,     73,
     &   58,     50,     43,     30,     23,     17,     12,
     &   10,      5,      6,      5,      5,     22,   2662,
     & 4643,    192,    192,    186,    143,    106,     72,
     &   64,     51,     41,     35,     26,     20,     18,
     &   14,     15,     12,     12,     14,     14,     24,
     & 2867,   5288,    200,    228,    256,    251,    253,
     &  218,    185,    157,     99,     72,     59,     31,
     &   26,     17,     15,     13,     15,     15,     16,
     &   18,     18,     26,   3072,   5843,    232,    257,
     &  286,    310,    319,    320,    310,    274,    228,
     &  193,    141,    107,     84,     62,     49,     36,
     &   28,     20,     14,     10,      8,     10,      9,
     &   28,   3276,   6157,    287,    329,    359,    378,
     &  400,    401,    385,    358,    343,    296,    212,
     &  176,    164,    127,    111,     83,     73,     59,
     &   48,     39,     32,     21,     20,     22,     22,
     &   29,   3481,   6533,    294,    340,    391,    455,
     &  483,    470,    487,    476,    432,    370,    291,
     &  252,    234,    217,    194,    171,    149,    137,
     &  107,     80,     57,     45,     28,     20,     28,
     &   27,     30,   3686,   6781,    318,    363,    418,
     &  473,    528,    564,    620,    610,    557,    503,
     &  477,    397,    332,    283,    247,    213,    186,
     &  175,    160,    130,    111,     91,     76,     56,
     &   38,     39,     37,     31,   3891,   6945,    345,
     &  410,    499,    573,    662,    756,    853,    850,
     &  808,    705,    573,    418,    341,    290,    257,
     &  221,    151,    149,    131,    114,    106,    105,
     &  103,    111,    132,    146,    149,    103,     31,
     & 4096,   7143,    367,    436,    518,    637,    783,
     &  938,   1121,   1274,   1201,    999,    709,    471,
     &  378,    300,    234,    178,    161,    167,    150,
     &  129,    117,    113,    114,    119,    119,    121,
     &  126,    133,     32,   4300,   7311,    391,    466,
     &  550,    707,    953,   1186,   1493,   1559,   1265/

      DATA (MAPPMAX (I), I= 7351, 7840) /
     & 1029,    771,    564,    490,    412,    339,    272,
     &  273,    247,    218,    201,    172,    152,    142,
     &  141,    142,    145,    151,    157,    177,     33,
     & 4505,   7465,    393,    472,    578,    772,   1052,
     & 1336,   1667,   1852,   1764,   1522,   1340,    751,
     &  570,    416,    315,    224,    209,    211,    214,
     &  196,    167,    144,    143,    145,    153,    166,
     &  171,    174,    186,    252,     33,   4710,   7589,
     &  402,    501,    650,    873,   1176,   1537,   1875,
     & 2124,   2127,   1787,   1466,   1234,    765,    591,
     &  400,    319,    312,    309,    273,    223,    193,
     &  189,    178,    169,    165,    163,    164,    165,
     &  175,    190,     34,   4915,   7700,    389,    507,
     &  716,    959,   1281,   1745,   2195,   2465,   2614,
     & 2263,   1843,   1409,    846,    614,    369,    272,
     &  326,    333,    326,    321,    314,    290,    246,
     &  219,    216,    216,    227,    230,    225,    223,
     &  227,     34,   5120,   7786,    410,    538,    761,
     & 1050,   1454,   1999,   2451,   2688,   2912,   2511,
     & 2167,   1367,   1124,    936,    795,    688,    594,
     &  549,    539,    496,    355,    298,    177,    175,
     &  168,    171,    175,    181,    185,    186,    191,
     &   34,   5324,   7863,    421,    556,    800,   1102,
     & 1572,   2160,   2651,   3003,   3281,   2927,   2441,
     & 1924,   1445,   1219,    988,    786,    634,    516,
     &  400,    348,    324,    320,    324,    324,    320,
     &  320,    324,    328,    328,    330,    342,     34,
     & 5529,   7928,    421,    557,    810,   1141,   1679,
     & 2357,   2950,   3434,   3743,   3102,   2634,   2232,
     & 2050,   1561,   1323,   1191,   1016,    825,    578,
     &  563,    481,    392,    321,    312,    309,    314,
     &  317,    321,    323,    335,    344,     35,   5734,
     & 7957,    424,    563,    828,   1211,   1840,   2576,
     & 3133,   3745,   4305,   4232,   3695,   3194,   1947,
     & 1396,   1075,    843,    809,    811,    802,    752,
     &  591,    471,    402,    406,    400,    391,    389,
     &  397,    408,    417,    422,    423,     35,   5939,
     & 7979,    417,    557,    810,   1211,   1862,   2657,
     & 3461,   4298,   4996,   5104,   4706,   3865,   2950,
     & 1846,   1548,   1368,   1206,    966,    649,    540,
     &  461,    438,    458,    461,    450,    451,    461,
     &  449,    426,    422,    445,    465,     35,   6144,
     & 7970,    423,    573,    859,   1231,   1897,   2829,
     & 3767,   4765,   5570,   6035,   5111,   4351,   3637,
     & 3253,   2746,   2078,   1689,   1122,    972,    787,
     &  645,    527,    459,    320,    317,    316,    315,
     &  317,    315,    320,    319,    316,     35,   6348,
     & 7937,    447,    588,    870,   1373,   2103,   3111,
     & 4319,   5417,   6113,   6040,   5350,   4572,   3585,
     & 3053,   2291,   2006,   1754,   1452,   1248,   1225,
     & 1077,    909,    756,    706,    788,    710,    590,
     &  502,    405,    364,    380,    384,     34,   6553,
     & 7890,    440,    595,    907,   1417,   2156,   3168,
     & 4444,   5772,   6710,   6449,   5538,   4813,   4149,
     & 3799,   2980,   2541,   2244,   2061,   1941,   1806,
     & 1641,   1510,   1144,    990,    899,    866,    756,
     &  652,    369,    287,    247,     34,   6758,   7825,
     &  466,    613,    916,   1512,   2409,   3318,   4798,
     & 6761,   7574,   7119,   6386,   6103,   5421,   4496,
     & 2974,   2335,   2473,   2230,   1818,   1723,   1647,
     & 1540,   1239,   1039,    916,    791,    640,    583,
     &  639,    619,    388,     34,   6963,   7743,    485,
     &  656,   1020,   1649,   2683,   3915,   5517,   7386,
     & 8722,   8956,   9338,   7514,   6186,   3393,   2573,
     & 1904,   1736,   1496,   1111,    910,    797,    699,
     &  646,    594,    687,    814,    841,    865,    896,
     &  903,    889,     33,   7168,   7646,    479,    658,
     & 1082,   1842,   2882,   4119,   5998,   8443,  10340,
     &11163,   9963,   8177,   6183,   4077,   3443,   2885,
     & 2389,   2099,   1739,   1473,   1206,   1076,   1051/

      DATA (MAPPMAX (I), I= 7841, 8330) /
     &  938,    749,    729,    702,    662,    631,    669,
     &   33,   7372,   7555,    476,    663,   1138,   1970,
     & 3109,   4439,   6659,   9376,  11599,  13360,   9950,
     & 8143,   5817,   4311,   3568,   3139,   2883,   2528,
     & 2058,   1792,   1402,   1283,   1213,   1109,    979,
     &  845,    835,    863,    876,    878,     33,   7577,
     & 7437,    539,    785,   1192,   1997,   3526,   5329,
     & 7740,  11292,  14598,  15355,  11332,   9299,   5024,
     & 4037,   3252,   2394,   2346,   1940,   1885,   1723,
     & 1467,   1322,   1337,   1315,   1227,   1193,   1271,
     & 1307,   1276,   1293,     32,   7782,   7295,    511,
     &  705,   1252,   2301,   3650,   5503,   8476,  12639,
     &15812,  15392,  11866,   9085,   7209,   4972,   3970,
     & 3453,   3312,   2971,   2794,   2816,   2224,   1752,
     & 1442,   1260,   1140,   1085,   1091,   1092,   1071,
     &   32,   7987,   7190,    502,    714,   1383,   2454,
     & 3796,   5790,   9051,  14395,  18060,  16458,  12601,
     & 9535,   7736,   7168,   5362,   4191,   3424,   2410,
     & 2314,   1927,   1570,   1281,   1158,   1074,   1224,
     & 1454,   1507,   1488,   1511,     31,   8192,   7036,
     &  486,    794,   1360,   2359,   4038,   6199,   9480,
     &14664,  18205,  17500,  14413,  12761,  11176,   7445,
     & 5532,   4243,   4379,   4129,   3632,   2944,   2366,
     & 1868,   1319,   1283,   1264,   1285,   1304,   1288,
     &   30,   8396,   6902,    457,    738,   1303,   2209,
     & 3853,   6314,   9909,  15420,  18991,  18986,  18542,
     &14443,  11652,   7005,   5938,   5521,   5242,   4610,
     & 3391,   2771,   2252,   1997,   1889,   1873,   1893,
     & 1908,   1924,     30,   8601,   6728,    457,    752,
     & 1336,   2267,   3876,   6301,  10229,  16494,  21539,
     &23787,  22390,  17482,   8945,   6337,   4194,   4217,
     & 4331,   4083,   3412,   2595,   2630,   2564,   2577,
     & 2548,   2524,   2512,   2661,     29,   8806,   6569,
     &  457,    723,   1256,   2122,   3785,   6544,  10828,
     &17044,  22454,  24984,  23083,  16892,  11534,   7127,
     & 5452,   5069,   4583,   4442,   4609,   4300,   3085,
     & 2980,   2889,   2979,   2978,   3004,     29,   9011,
     & 6400,    462,    656,   1206,   2122,   3698,   6933,
     &11289,  16383,  23187,  29122,  31556,  22460,  14831,
     &11112,   8255,   5087,   4734,   3815,   3242,   2723,
     & 2341,   2195,   2181,   2126,   2095,   2285,     28,
     & 9216,   6261,    434,    735,   1289,   1956,   3167,
     & 6271,  10783,  16509,  23931,  29374,  30994,  21299,
     &13546,   8017,   6987,   7011,   7126,   6220,   4460,
     & 4022,   3692,   3663,   3637,   3812,   4120,     27,
     & 9420,   6077,    432,    642,   1076,   1788,   3610,
     & 6066,  10214,  16901,  26698,  31401,  29783,  24611,
     &19786,  14959,   9362,   8425,   7650,   6562,   5384,
     & 3742,   3312,   3215,   3338,   3545,     27,   9625,
     & 5923,    370,    520,    854,   1307,   2557,   4692,
     & 9459,  16594,  24053,  30653,  33324,  31433,  21103,
     &17170,   9274,   7823,   7569,   7117,   6552,   6019,
     & 5421,   4893,   4849,   5165,     26,   9830,   5668,
     &  386,    572,    999,   1637,   3334,   5454,   9418,
     &16494,  26765,  36562,  40773,  25834,  20762,  16167,
     &12054,  12080,   9508,   7658,   5531,   4916,   4622,
     & 5146,   5461,     25,  10035,   5548,    396,    550,
     &  930,   1487,   2891,   5201,   9651,  18417,  28453,
     &35430,  37743,  28829,  22694,  17370,  13866,  11053,
     & 9976,   8521,   7209,   6280,   5355,   5664,     24,
     &10240,   5323,    351,    589,    932,   1391,   2837,
     & 5073,   8208,  13156,  20805,  39264,  40297,  35427,
     &27940,  21849,  16305,  13564,   9662,   7958,   6405,
     & 5222,   3426,     24,  10444,   5166,    362,    571,
     &  874,   1805,   3053,   6146,  10652,  15436,  20221,
     &31703,  40517,  43777,  24694,  20496,  13837,   9977,
     & 7861,   6315,   4602,   4624,   5266,     23,  10649,
     & 5036,    337,    541,    826,   1506,   3367,   5776,
     & 8832,  14592,  25979,  39698,  39881,  36885,  22941,
     &16375,  13300,  11872,   9337,   7346,   5998,   6876/

      DATA (MAPPMAX (I), I= 8331, 8820) /
     &   23,  10854,   4915,    392,    547,    910,   1420,
     & 2951,   5320,   9352,  15929,  23280,  33252,  41521,
     &45345,  32381,  26206,  20327,  12108,   9460,   6661,
     & 4981,   5529,     22,  11059,   4793,    409,    607,
     &  974,   1534,   2847,   5852,   9326,  14883,  24815,
     &36493,  43893,  46480,  31974,  25121,  20469,  12482,
     &10310,   8460,   6912,     22,  11264,   4704,    339,
     &  462,    814,   1359,   2778,   5422,   8815,  15483,
     &29916,  38102,  41880,  42773,  26980,  22290,  21888,
     &20761,  18764,  12334,  11901,     21,  11468,   4605,
     &  335,    466,    859,   1403,   3118,   6303,   9956,
     &14514,  20950,  32695,  46668,  49666,  44052,  30442,
     &24100,  17566,  15950,  17589,     21,  11673,   4547,
     &  357,    473,    836,   1340,   2828,   5857,  10491,
     &16973,  29543,  36570,  40338,  42209,  41250,  34841,
     &28310,  19286,  13188,  15881,     21,  11878,   4468,
     &  346,    467,    893,   1446,   3265,   6190,  10032,
     &16192,  27718,  38213,  45149,  47415,  42719,  36877,
     &32467,  27483,  20373,  14632,     21,  12083,   4406,
     &  357,    464,    690,   1192,   2398,   4842,  10834,
     &18102,  25244,  34231,  47001,  52151,  47180,  36959,
     &31577,  19705,  21031,  21890,     20,  12288,   4335,
     &  360,    503,    828,   1328,   3035,   5923,   9990,
     &17005,  26725,  37269,  48795,  53113,  43614,  35881,
     &32747,  30088,  27497,     20,  12492,   4229,    369,
     &  456,    769,   1575,   2686,   5531,  10557,  18416,
     &30570,  39556,  44518,  45585,  41102,  36204,  35797,
     &34148,  31468,     20,  12697,   4173,    377,    526,
     &  806,   1213,   2356,   4781,   9034,  16500,  25135,
     &37390,  46485,  48999,  41941,  37852,  37329,  36391,
     &34571,     20,  12902,   4109,    336,    465,    815,
     & 1343,   2798,   5313,   8651,  13557,  19965,  28475,
     &40031,  47937,  50038,  44985,  39885,  37628,  35964,
     &   19,  13107,   4062,    382,    505,    809,   1272,
     & 2991,   6120,   9345,  13543,  21895,  31276,  40652,
     &47247,  49474,  45219,  43636,  44917,     19,  13312,
     & 3990,    379,    541,    881,   1383,   3145,   6081,
     &10348,  16846,  25406,  37331,  44763,  46493,  44533,
     &42608,  43082,  45086,     19,  13516,   3968,    353,
     &  423,    753,   1294,   2388,   4262,   6889,  13826,
     &20761,  25962,  32549,  44731,  52072,  48217,  44228,
     &63455,      3,  20480,      0,      3,  32767,      0,
     & 1467,    100,      3,      0,      0,      3,   2334,
     &    0,     10,   2355,   1707,     26,     24,     19,
     &   16,     14,     11,     12,     14,   2396,   2798,
     &   47,     35,     30,     19,     11,     10,      6,
     &    7,      6,      4,      3,     17,   2457,   3510,
     &   85,     72,     58,     50,     44,     29,     23,
     &   17,     12,      9,      6,      6,      5,      5,
     &   22,   2662,   4640,    192,    190,    189,    142,
     &  106,     72,     63,     51,     42,     34,     26,
     &   20,     18,     14,     15,     12,     12,     14,
     &   14,     24,   2867,   5285,    200,    229,    255,
     &  249,    257,    217,    184,    157,     99,     71,
     &   59,     32,     25,     17,     15,     13,     15,
     &   15,     16,     18,     19,     26,   3072,   5838,
     &  229,    260,    287,    311,    314,    325,    313,
     &  272,    227,    192,    141,    107,     82,     63,
     &   48,     37,     27,     20,     14,     10,      8,
     &   10,     10,     28,   3276,   6153,    288,    327,
     &  355,    381,    402,    398,    384,    360,    347,
     &  297,    209,    175,    166,    127,    110,     82,
     &   74,     58,     48,     40,     31,     21,     20,
     &   23,     22,     29,   3481,   6520,    293,    341,
     &  391,    459,    485,    471,    488,    477,    428,
     &  369,    288,    254,    233,    219,    195,    173,
     &  146,    138,    105,     79,     56,     45,     26,
     &   21,     29,     28,     30,   3686,   6761,    315,
     &  368,    416,    471,    529,    576,    625,    615,
     &  559,    505,    473,    390,    328,    284,    246/

      DATA (MAPPMAX (I), I= 8821, 9310) /
     &  212,    184,    175,    162,    129,    109,     90,
     &   74,     55,     38,     39,     36,     30,   3891,
     & 6909,    349,    420,    499,    590,    667,    767,
     &  826,    854,    808,    713,    556,    421,    338,
     &  292,    252,    222,    151,    150,    131,    112,
     &  107,    105,    102,    114,    135,    147,    124,
     &   31,   4096,   7094,    367,    438,    521,    645,
     &  797,    967,   1144,   1301,   1190,   1008,    652,
     &  459,    370,    295,    228,    175,    161,    168,
     &  150,    128,    116,    114,    114,    119,    119,
     &  121,    127,    131,     32,   4300,   7242,    393,
     &  470,    566,    725,    980,   1226,   1507,   1532,
     & 1253,   1015,    755,    563,    479,    402,    330,
     &  274,    272,    245,    220,    201,    170,    150,
     &  142,    141,    142,    147,    151,    162,    182,
     &   32,   4505,   7377,    389,    477,    603,    803,
     & 1084,   1380,   1698,   1885,   1755,   1484,   1308,
     &  716,    552,    405,    291,    221,    211,    213,
     &  213,    194,    165,    144,    145,    147,    156,
     &  167,    169,    175,    193,     33,   4710,   7487,
     &  402,    497,    650,    870,   1182,   1534,   1887,
     & 2151,   2164,   1787,   1427,   1218,    762,    592,
     &  405,    328,    320,    307,    269,    222,    195,
     &  192,    179,    165,    164,    163,    162,    166,
     &  177,    192,     33,   4915,   7586,    385,    507,
     &  716,    958,   1247,   1730,   2186,   2509,   2676,
     & 2305,   1857,   1382,    829,    597,    361,    270,
     &  331,    329,    329,    325,    317,    286,    242,
     &  218,    215,    218,    229,    229,    228,    234,
     &   33,   5120,   7658,    394,    521,    738,   1016,
     & 1403,   1945,   2432,   2777,   2979,   2536,   2184,
     & 1350,   1116,    921,    786,    685,    595,    539,
     &  531,    503,    384,    306,    197,    175,    168,
     &  173,    177,    182,    190,    195,     34,   5324,
     & 7723,    404,    535,    772,   1078,   1518,   2111,
     & 2620,   2998,   3280,   2956,   2457,   1950,   1517,
     & 1226,    998,    801,    659,    541,    406,    348,
     &  327,    322,    325,    327,    321,    320,    323,
     &  325,    330,    331,    344,     34,   5529,   7777,
     &  394,    519,    759,   1101,   1584,   2211,   2829,
     & 3391,   3763,   3252,   2732,   2287,   2053,   1620,
     & 1292,   1192,   1033,    882,    588,    585,    499,
     &  396,    331,    318,    311,    314,    320,    320,
     &  321,    336,    345,     34,   5734,   7795,    380,
     &  508,    763,   1147,   1680,   2342,   3020,   3733,
     & 4289,   4343,   3894,   3250,   2097,   1406,   1115,
     &  885,    816,    822,    823,    779,    634,    475,
     &  408,    402,    393,    385,    387,    404,    417,
     &  429,    438,     34,   5939,   7799,    398,    524,
     &  761,   1154,   1698,   2376,   3267,   4273,   4918,
     & 5020,   4858,   3932,   3110,   2011,   1607,   1426,
     & 1255,   1094,    646,    564,    474,    436,    464,
     &  465,    450,    444,    456,    447,    425,    431,
     &  453,     34,   6144,   7762,    402,    537,    792,
     & 1203,   1853,   2671,   3553,   4507,   5362,   5934,
     & 5192,   4407,   3776,   3402,   2944,   2216,   1761,
     & 1299,   1034,    828,    659,    534,    463,    321,
     &  316,    323,    311,    314,    316,    328,    332,
     &   34,   6348,   7745,    414,    537,    766,   1195,
     & 1871,   2753,   3869,   4960,   5732,   6057,   5350,
     & 4732,   4079,   3394,   2544,   2142,   1839,   1580,
     & 1372,   1231,   1153,    929,    828,    735,    782,
     &  745,    644,    536,    456,    359,    364,     33,
     & 6553,   7670,    420,    543,    794,   1300,   2030,
     & 2882,   4039,   5429,   6392,   6496,   5607,   4886,
     & 4343,   4023,   3278,   2742,   2296,   2101,   2000,
     & 1860,   1710,   1540,   1297,   1020,    896,    875,
     &  789,    650,    447,    305,     33,   6758,   7586,
     &  427,    554,    812,   1363,   2165,   2977,   4285,
     & 6000,   7091,   7183,   6481,   5912,   5614,   5124/

      DATA (MAPPMAX (I), I= 9311, 9800) /
     & 3734,   2728,   2484,   2284,   2016,   1817,   1702,
     & 1566,   1379,   1057,    944,    813,    672,    576,
     &  644,    621,     33,   6963,   7500,    452,    598,
     &  912,   1468,   2366,   3486,   4895,   6738,   8179,
     & 8549,   9508,   8129,   7166,   3783,   2852,   2096,
     & 1831,   1702,   1265,   1007,    828,    718,    651,
     &  603,    637,    814,    849,    858,    895,    914,
     &   32,   7168,   7374,    449,    613,    986,   1670,
     & 2668,   3650,   5361,   7858,   9676,  10469,  10867,
     & 8471,   6632,   4989,   3728,   3097,   2583,   2162,
     & 1838,   1570,   1248,   1110,   1045,    915,    745,
     &  721,    712,    688,    668,     32,   7372,   7256,
     &  470,    640,   1043,   1805,   2897,   4071,   6117,
     & 8924,  11113,  12357,  10978,   8877,   6465,   4643,
     & 3761,   3245,   2917,   2609,   2175,   1902,   1438,
     & 1319,   1215,   1081,    944,    828,    854,    884,
     &  908,     31,   7577,   7120,    536,    788,   1193,
     & 1876,   3111,   4745,   7225,  10543,  13629,  15604,
     &11878,  10514,   5627,   4313,   3479,   2583,   2479,
     & 2060,   1976,   1739,   1413,   1230,   1305,   1313,
     & 1255,   1243,   1270,   1295,     31,   7782,   6939,
     &  522,    755,   1260,   2152,   3389,   5055,   7932,
     &11779,  15030,  16017,  12407,   9994,   7887,   5242,
     & 4287,   3624,   3363,   2964,   2914,   2797,   2082,
     & 1712,   1417,   1220,   1144,   1117,   1122,   1120,
     &   30,   7987,   6841,    479,    692,   1283,   2308,
     & 3594,   5383,   8105,  12537,  16385,  17929,  13139,
     &10882,   8541,   8003,   5678,   4420,   3489,   2454,
     & 2360,   1936,   1591,   1321,   1184,   1161,   1331,
     & 1473,   1524,     29,   8192,   6649,    492,    762,
     & 1253,   2114,   3680,   5727,   8917,  13502,  17405,
     &19287,  15760,  13435,  11276,   7553,   5655,   4840,
     & 4785,   4304,   3445,   2741,   2058,   1621,   1278,
     & 1282,   1288,   1304,     29,   8396,   6495,    436,
     &  686,   1224,   2012,   3347,   5663,   9593,  14141,
     &17315,  18691,  19278,  15721,  13071,   8728,   6666,
     & 6537,   5397,   4507,   2953,   2618,   2067,   1903,
     & 1865,   1852,   1915,   2003,     28,   8601,   6284,
     &  407,    672,   1225,   2011,   3440,   5889,   9939,
     &16037,  21522,  24071,  23418,  15698,  11009,   6282,
     & 4591,   4462,   4805,   4425,   3371,   2614,   2541,
     & 2463,   2461,   2734,   2612,     27,   8806,   6087,
     &  419,    607,   1000,   1579,   3140,   5872,  10110,
     &16095,  21876,  25200,  25680,  16845,  14581,   7110,
     & 6085,   5352,   4952,   5102,   4981,   3033,   3078,
     & 2975,   3000,   2964,     27,   9011,   5904,    403,
     &  555,    957,   1733,   2928,   5790,   9914,  15125,
     &21868,  30213,  35377,  22265,  17469,  12659,   8055,
     & 5634,   4915,   3368,   2891,   2323,   2129,   2267,
     & 2207,   2127,     26,   9216,   5714,    383,    607,
     & 1064,   1653,   2582,   5014,   9185,  14581,  23278,
     &30262,  31744,  22408,  19126,   9720,   8071,   7934,
     & 6526,   5246,   4148,   3586,   3601,   3667,   4024,
     &   25,   9420,   5547,    350,    460,    755,   1190,
     & 2188,   4496,   7724,  13039,  20767,  32944,  35094,
     &30524,  21946,  19605,  10265,   8768,   7317,   6065,
     & 4203,   3658,   3354,   3529,     24,   9625,   5303,
     &  323,    423,    696,   1114,   1838,   3795,   7542,
     &14144,  20797,  27797,  35541,  36383,  26040,  21885,
     &10816,   8801,   7598,   6980,   6170,   5359,   5040,
     &   23,   9830,   5057,    310,    399,    736,   1228,
     & 1993,   3376,   5424,   9831,  19093,  32765,  43399,
     &42822,  27017,  21054,  13976,  12518,   8489,   6746,
     & 5325,   5311,     23,  10035,   4868,    343,    472,
     &  714,   1029,   1680,   3512,   5970,  11278,  20543,
     &33247,  39759,  40138,  28525,  21768,  15985,  11982,
     &10733,   9201,   7576,   6403,     22,  10240,   4636,
     &  277,    416,    666,    965,   1422,   3044,   4816,
     & 7207,  11505,  20471,  42537,  41723,  37898,  27898,
     &20383,  15476,  12570,   9422,   9065,     21,  10444/

      DATA (MAPPMAX (I), I= 9801,10290) /
     & 4415,    267,    409,    573,    810,   1663,   2753,
     & 5748,  10396,  15961,  23613,  33519,  40618,  38254,
     &27810,  19560,  13798,  10772,   9568,     20,  10649,
     & 4302,    275,    399,    561,    810,   1474,   3124,
     & 4895,   7103,  10724,  24362,  36520,  41680,  39755,
     &23757,  18426,  14683,  13249,     20,  10854,   4181,
     &  292,    360,    564,    888,   1292,   2744,   4737,
     & 7439,  10904,  16874,  28297,  41267,  45740,  46439,
     &26337,  21361,  17492,     19,  11059,   3998,    306,
     &  372,    604,   1020,   1665,   2942,   4315,   6365,
     &13006,  21122,  31294,  42792,  45259,  43294,  28877,
     &24915,     19,  11264,   3924,    309,    355,    544,
     &  862,   1231,   2579,   4232,   5960,   9071,  21152,
     &34273,  41233,  41140,  36631,  25294,  28206,     19,
     &11468,   3847,    289,    346,    530,    876,   1223,
     & 2554,   4712,   7483,  12025,  18933,  30753,  43983,
     &48463,  47240,  35703,  32757,     18,  11673,   3706,
     &  320,    406,    645,    935,   1367,   3135,   5383,
     & 8594,  14189,  26142,  38982,  45151,  45366,  39943,
     &44899,     18,  11878,   3665,    326,    408,    612,
     &  927,   1284,   3582,   5888,   8623,  13182,  25953,
     &41810,  45763,  45422,  45276,  50111,     18,  12083,
     & 3614,    303,    373,    485,    767,   1316,   2515,
     & 4084,   7766,  12615,  19202,  30020,  49102,  57736,
     &56950,  48404,     17,  12288,   3497,    311,    394,
     &  644,   1006,   1694,   3708,   5976,   9598,  15923,
     &26355,  37962,  45523,  48895,  53970,     17,  12492,
     & 3431,    335,    419,    607,   1019,   1609,   2424,
     & 4895,   9064,  14710,  22963,  35174,  40769,  44738,
     &57112,     17,  12697,   3350,    343,    404,    542,
     &  762,   1183,   2946,   6019,   9638,  13366,  18200,
     &28539,  41477,  52659,  68765,     16,  12902,   3303,
     &  313,    417,    608,    902,   1405,   3370,   5350,
     & 7578,  11356,  19774,  30635,  34844,  44550,     16,
     &13107,   3240,    339,    450,    688,    948,   1275,
     & 3280,   5657,   8849,  13401,  20936,  28616,  46305,
     &67899,     16,  13312,   3212,    342,    439,    629,
     &  905,   1515,   3714,   5988,   9839,  15478,  22725,
     &33518,  46773,  66925,     16,  13516,   3098,    349,
     &  447,    680,   1029,   1603,   3188,   5529,   9136,
     &15107,  22390,  27256,  35118,  63012,      3,  20480,
     &    0,      3,  32767,      0,   1018,    200,      3,
     &    0,      0,      3,   2334,      0,     10,   2355,
     & 1698,     26,     24,     19,     16,     14,     11,
     &    7,     14,   2396,   2794,     47,     35,     30,
     &   19,     11,     10,      6,      7,      6,      4,
     &    3,     17,   2457,   3509,     84,     73,     58,
     &   49,     44,     29,     23,     17,     12,      9,
     &    6,      6,      5,      5,     22,   2662,   4628,
     &  198,    188,    189,    142,    104,     71,     64,
     &   50,     42,     34,     25,     20,     18,     14,
     &   15,     12,     12,     14,     14,     24,   2867,
     & 5263,    201,    230,    259,    261,    251,    218,
     &  182,    152,     98,     70,     56,     32,     25,
     &   16,     15,     14,     14,     15,     16,     19,
     &   18,     26,   3072,   5806,    231,    261,    287,
     &  311,    322,    332,    312,    275,    228,    186,
     &  138,    104,     80,     61,     47,     35,     26,
     &   20,     13,     10,      8,     10,      9,     27,
     & 3276,   6120,    289,    329,    354,    378,    401,
     &  402,    391,    374,    335,    295,    216,    179,
     &  159,    127,    107,     81,     72,     57,     46,
     &   39,     30,     20,     21,     22,     29,   3481,
     & 6457,    298,    337,    394,    468,    503,    476,
     &  495,    475,    425,    354,    282,    248,    238,
     &  227,    203,    173,    147,    127,    100,     75,
     &   54,     39,     25,     22,     29,     29,     30,
     & 3686,   6660,    310,    368,    427,    505,    559,
     &  606,    636,    614,    553,    510,    466,    349,
     &  320,    289,    238,    201,    183,    186,    162/

      DATA (MAPPMAX (I), I=10291,10780) /
     &  121,    102,     84,     67,     48,     38,     39,
     &   35,     30,   3891,   6779,    351,    420,    516,
     &  625,    692,    789,    824,    831,    789,    696,
     &  555,    413,    340,    293,    253,    220,    159,
     &  153,    128,    114,    111,    105,    105,    124,
     &  140,    143,    143,     30,   4096,   6877,    369,
     &  450,    559,    703,    862,   1046,   1233,   1342,
     & 1104,    953,    566,    432,    344,    269,    208,
     &  165,    172,    171,    144,    126,    116,    114,
     &  117,    120,    121,    124,    131,     31,   4300,
     & 6961,    390,    475,    612,    797,   1035,   1261,
     & 1486,   1509,   1242,   1010,    748,    561,    467,
     &  387,    313,    286,    274,    237,    221,    189,
     &  160,    145,    142,    142,    143,    149,    160,
     &  175,     31,   4505,   7006,    391,    491,    641,
     &  841,   1093,   1454,   1813,   2011,   1845,   1479,
     & 1150,    662,    489,    377,    236,    216,    215,
     &  215,    204,    180,    160,    153,    150,    155,
     &  164,    169,    174,    189,     31,   4710,   7034,
     &  392,    498,    662,    862,   1152,   1611,   2070,
     & 2276,   2146,   1776,   1412,   1171,    740,    548,
     &  380,    344,    345,    309,    247,    217,    202,
     &  183,    167,    165,    163,    162,    163,    178,
     &   31,   4915,   7052,    367,    474,    667,    932,
     & 1261,   1779,   2413,   2912,   3112,   2191,   1965,
     &  884,    688,    466,    295,    312,    352,    352,
     &  343,    331,    298,    246,    218,    219,    222,
     &  228,    232,    240,     31,   5120,   7053,    356,
     &  462,    649,    939,   1356,   1934,   2551,   3001,
     & 3214,   2542,   2157,   1302,   1085,    884,    753,
     &  655,    585,    566,    565,    515,    347,    287,
     &  174,    177,    173,    173,    179,    196,     31,
     & 5324,   7044,    336,    430,    651,    979,   1381,
     & 2068,   2707,   3168,   3518,   3358,   2516,   2002,
     & 1479,   1181,    942,    774,    645,    526,    364,
     &  338,    322,    314,    315,    323,    334,    334,
     &  335,    349,     31,   5529,   7036,    340,    431,
     &  640,    990,   1463,   2210,   2861,   3298,   3675,
     & 3671,   2905,   2421,   2001,   1596,   1226,   1167,
     & 1039,    876,    648,    590,    519,    397,    337,
     &  313,    310,    322,    329,    343,     31,   5734,
     & 6965,    333,    432,    657,   1028,   1521,   2222,
     & 2979,   3609,   4149,   4578,   4425,   3271,   2419,
     & 1473,   1146,    932,    879,    878,    834,    750,
     &  512,    428,    405,    401,    386,    382,    413,
     &  462,     31,   5939,   6914,    344,    446,    636,
     & 1003,   1510,   2147,   2964,   3952,   4855,   5356,
     & 5344,   4066,   3267,   2352,   1672,   1506,   1261,
     & 1078,    650,    573,    478,    442,    445,    448,
     &  449,    453,    482,    482,     30,   6144,   6834,
     &  326,    405,    599,    999,   1560,   2221,   3024,
     & 3994,   5118,   5921,   6091,   4842,   4163,   3676,
     & 3274,   2679,   1830,   1403,   1078,    868,    637,
     &  501,    388,    318,    319,    315,    328,     30,
     & 6348,   6712,    352,    458,    670,   1082,   1696,
     & 2517,   3421,   4337,   5416,   6211,   6078,   5112,
     & 4235,   3299,   2836,   2128,   1980,   1735,   1462,
     & 1202,   1140,   1023,    937,    849,    881,    754,
     &  623,     29,   6553,   6553,    365,    477,    729,
     & 1246,   1893,   2690,   3859,   4947,   6028,   6928,
     & 5537,   5295,   4912,   4315,   3514,   2971,   2434,
     & 2188,   2004,   1830,   1669,   1507,   1232,   1063,
     &  953,    862,     28,   6758,   6385,    364,    498,
     &  809,   1355,   1994,   2780,   4004,   5446,   6763,
     & 7448,   7084,   6519,   6308,   5346,   3969,   2733,
     & 2691,   2351,   2104,   1966,   1768,   1530,   1022,
     &  962,    816,     28,   6963,   6250,    360,    496,
     &  830,   1358,   2091,   3043,   4222,   5678,   7368,
     & 8967,   9902,   9671,   7033,   5149,   3144,   2568,
     & 2059,   1802,   1009,    930,    806,    674,    579/

      DATA (MAPPMAX (I), I=10781,11270) /
     &  674,    882,     27,   7168,   6038,    431,    583,
     &  897,   1535,   2485,   3577,   4995,   7039,   9017,
     &10663,  12466,   9147,   7378,   5604,   3975,   3252,
     & 2640,   2204,   1680,   1242,   1119,   1051,    900,
     &  739,     26,   7372,   5820,    496,    657,    993,
     & 1702,   2830,   4135,   5982,   8752,  11428,  13292,
     &11498,   9436,   6977,   4781,   3950,   3430,   3011,
     & 2371,   2030,   1561,   1359,   1223,   1105,     25,
     & 7577,   5601,    450,    705,   1172,   1745,   2685,
     & 4462,   7118,  10069,  13241,  15493,  15089,  12278,
     & 6074,   4272,   3462,   2957,   2549,   2018,   1419,
     & 1118,   1269,   1367,     24,   7782,   5282,    574,
     &  819,   1347,   2404,   3698,   5239,   8784,  13898,
     &17731,  15705,  12740,   9658,   7335,   5800,   4887,
     & 4128,   3380,   2707,   2031,   1665,   1410,     23,
     & 7987,   5098,    473,    840,   1519,   2341,   3861,
     & 6489,  10094,  14782,  18093,  18675,  13699,  10911,
     & 9966,   6326,   5107,   3071,   2881,   1956,   1619,
     & 1409,     22,   8192,   4786,    487,    841,   1467,
     & 2268,   3640,   6412,  11249,  17056,  20607,  20544,
     &16589,  13719,  11187,   7321,   5851,   4727,   3131,
     & 2135,   1685,     21,   8396,   4563,    411,    673,
     & 1198,   2047,   3378,   6093,  11413,  16780,  21247,
     &23422,  22116,  16118,  11230,   8507,   5966,   4591,
     & 3241,   2364,     20,   8601,   4223,    420,    681,
     & 1145,   1909,   3450,   5769,  10196,  18157,  25687,
     &27769,  20234,  17936,   7858,   5976,   5243,   4753,
     & 4075,     19,   8806,   3912,    396,    643,   1077,
     & 1768,   3407,   6328,  11581,  19201,  30814,  32576,
     &21351,  17332,   9874,   7679,   6359,   5211,     18,
     & 9011,   3612,    370,    534,   1012,   1946,   3315,
     & 6923,  12305,  20310,  37597,  38444,  32061,  22738,
     &11588,   7949,   4386,     16,   9216,   3326,    312,
     &  451,   1104,   1951,   3188,   6008,  12242,  24044,
     &42040,  37953,  30593,  19788,  11928,     15,   9420,
     & 2943,    317,    415,    741,   1268,   2740,   7218,
     &12285,  27310,  54424,  46048,  33240,  23316,     13,
     & 9625,   2548,    279,    362,    625,   1159,   2647,
     & 7886,  16875,  34500,  56505,  47465,     12,   9830,
     & 2198,    210,    272,    389,   1024,   2126,   5412,
     &11339,  22907,  51517,     10,  10035,   1733,    235,
     &  278,    432,   1021,   2277,   7538,  18665,      8,
     &10240,   1241,    167,    232,    357,    594,   1646,
     &    7,  10444,    803,    267,    466,   1543,   7724,
     &    6,  10649,    674,    223,    298,    621,      5,
     &10854,    461,    397,   5011,      4,  11059,    123,
     & 1927,      3,  11264,      0,      3,  32767,      0,
     &  767,    400,      3,      0,      0,      3,   2334,
     &    0,     10,   2355,   1673,     26,     24,     19,
     &   16,     14,     11,     12,     14,   2396,   2783,
     &   47,     35,     30,     19,     11,     10,      6,
     &    7,      6,      4,      3,     17,   2457,   3505,
     &   84,     72,     58,     50,     43,     29,     23,
     &   17,     12,      9,      6,      6,      5,      5,
     &   21,   2662,   4605,    205,    197,    177,    144,
     &  104,     70,     62,     49,     42,     33,     24,
     &   20,     18,     14,     15,     12,     12,     12,
     &   24,   2867,   5219,    206,    236,    266,    271,
     &  251,    217,    179,    143,     94,     68,     51,
     &   31,     23,     16,     15,     14,     15,     15,
     &   17,     18,     19,     26,   3072,   5746,    234,
     &  258,    290,    317,    334,    340,    319,    275,
     &  229,    177,    133,     99,     75,     57,     45,
     &   33,     25,     18,     13,      9,      8,     10,
     &    9,     27,   3276,   6055,    287,    331,    362,
     &  372,    395,    411,    402,    384,    335,    293,
     &  215,    185,    154,    128,    100,     80,     68,
     &   55,     44,     36,     28,     20,     21,     22,
     &   28,   3481,   6287,    315,    370,    423,    465,
     &  505,    517,    517,    468,    395,    332,    274/

      DATA (MAPPMAX (I), I=11271,11760) /
     &  257,    253,    226,    195,    165,    135,    115,
     &   87,     65,     46,     29,     23,     24,     30,
     &   29,   3686,   6434,    322,    393,    472,    566,
     &  610,    642,    663,    605,    544,    494,    425,
     &  340,    316,    272,    225,    196,    192,    173,
     &  130,    111,     86,     70,     50,     40,     38,
     &   35,     29,   3891,   6507,    352,    430,    549,
     &  681,    757,    858,    802,    788,    752,    678,
     &  562,    411,    335,    293,    255,    206,    174,
     &  144,    122,    115,    107,    107,    123,    138,
     &  156,    155,     29,   4096,   6507,    386,    493,
     &  633,    786,    929,   1109,   1254,   1261,   1047,
     &  874,    517,    429,    347,    270,    202,    185,
     &  196,    159,    133,    119,    116,    116,    118,
     &  121,    125,    130,     29,   4300,   6493,    386,
     &  496,    674,    867,   1061,   1300,   1494,   1514,
     & 1243,   1008,    748,    579,    475,    387,    302,
     &  287,    261,    233,    198,    166,    148,    142,
     &  143,    145,    150,    159,     29,   4505,   6432,
     &  396,    496,    656,    910,   1268,   1617,   1933,
     & 2064,   1816,   1523,    940,    612,    459,    280,
     &  224,    230,    232,    216,    184,    164,    159,
     &  159,    163,    166,    170,    177,     28,   4710,
     & 6377,    389,    488,    646,    921,   1345,   1845,
     & 2274,   2351,   2126,   1764,   1438,    905,    680,
     &  464,    397,    403,    344,    256,    225,    204,
     &  183,    170,    166,    167,    173,     28,   4915,
     & 6300,    396,    484,    618,    924,   1428,   2014,
     & 2712,   3193,   2879,   2117,   1652,    893,    625,
     &  399,    332,    408,    393,    365,    344,    296,
     &  247,    220,    218,    223,    235,     28,   5120,
     & 6202,    352,    445,    619,    974,   1504,   2206,
     & 2947,   3345,   3303,   2351,   1888,   1278,    978,
     &  813,    685,    602,    578,    599,    541,    384,
     &  306,    179,    177,    177,    180,     27,   5324,
     & 6071,    343,    444,    649,   1018,   1627,   2557,
     & 3156,   3434,   3771,   2966,   2421,   1743,   1370,
     & 1081,    876,    689,    504,    355,    324,    322,
     &  324,    336,    345,    412,     27,   5529,   5966,
     &  334,    433,    688,   1074,   1590,   2392,   3281,
     & 3995,   4320,   3223,   2648,   2140,   1791,   1324,
     & 1202,   1114,    943,    765,    614,    569,    428,
     &  353,    342,    364,     26,   5734,   5765,    382,
     &  498,    737,   1183,   1869,   2641,   3426,   4449,
     & 5284,   5216,   3734,   2524,   1482,   1173,    971,
     &  924,    878,    798,    614,    460,    395,    424,
     &  437,     26,   5939,   5634,    399,    498,    642,
     & 1041,   1755,   2674,   3787,   5012,   5964,   6256,
     & 4495,   3562,   2399,   1746,   1497,   1268,    952,
     &  630,    539,    483,    478,    484,    484,     25,
     & 6144,   5422,    394,    501,    718,   1163,   1874,
     & 2731,   3849,   5462,   6608,   6661,   5475,   4552,
     & 3681,   3178,   2372,   1863,   1406,   1267,    770,
     &  597,    459,    333,     24,   6348,   5243,    407,
     &  527,    751,   1144,   1805,   2669,   4050,   5511,
     & 6394,   6834,   5806,   4808,   3411,   3014,   2217,
     & 2021,   1858,   1628,   1386,   1267,   1068,     23,
     & 6553,   5015,    404,    496,    677,   1238,   2126,
     & 3132,   4352,   5657,   6700,   7158,   6445,   5336,
     & 4410,   3859,   3349,   2659,   2368,   2248,   2169,
     & 1970,     22,   6758,   4742,    430,    548,    786,
     & 1367,   2315,   3409,   4624,   6617,   8407,   8709,
     & 7243,   6914,   6048,   4564,   3465,   2905,   2531,
     & 2295,   2074,     21,   6963,   4509,    395,    551,
     &  879,   1546,   2388,   3569,   5239,   7957,   9878,
     &10289,  10378,   7522,   6808,   3223,   2606,   2114,
     & 1630,   1119,     20,   7168,   4192,    453,    636,
     & 1054,   1872,   3154,   4476,   6215,   8345,  10790,
     &13091,  13072,   9999,   5958,   3793,   2877,   2271,
     & 1900,     19,   7372,   3873,    525,    727,   1232/

      DATA (MAPPMAX (I), I=11761,12250) /
     & 2269,   3448,   5389,   7831,  11116,  14630,  13677,
     &10766,   8642,   6062,   4747,   3755,   2753,     17,
     & 7577,   3465,    590,    879,   1341,   2304,   3907,
     & 6483,  10574,  15371,  19319,  14372,  13746,   5829,
     & 4683,   3627,     16,   7782,   3122,    480,    747,
     & 1442,   2478,   4046,   6806,  10846,  17943,  21684,
     &17541,  13512,   9250,   6639,     14,   7987,   2691,
     &  694,   1171,   2143,   3772,   6737,  11762,  19356,
     &26584,  20384,  17826,   9399,     12,   8192,   2249,
     &  753,   1268,   2177,   4351,   8664,  14641,  23050,
     &32012,  22946,     11,   8396,   1830,    763,   1352,
     & 2498,   4905,  10886,  21859,  31406,  35860,      9,
     & 8601,   1428,    689,   1241,   2214,   3971,   8171,
     &20046,      7,   8806,    994,    826,   1607,   3237,
     & 8634,      6,   9011,    561,    679,   2514,   5717,
     &    5,   9216,    370,    448,   2591,      3,   9420,
     &    0,      3,  32767,      0,    652,    600,      3,
     &    0,      0,      3,   2334,      0,     10,   2355,
     & 1662,     26,     24,     19,     16,     14,     11,
     &   12,     14,   2396,   2777,     47,     35,     30,
     &   19,     11,     10,      6,      7,      6,      4,
     &    4,     17,   2457,   3499,     85,     72,     57,
     &   50,     43,     29,     22,     18,     11,      9,
     &    6,      6,      5,      5,     21,   2662,   4582,
     &  205,    203,    175,    145,    104,     70,     60,
     &   49,     41,     32,     24,     20,     17,     14,
     &   15,     12,     12,     12,     24,   2867,   5170,
     &  211,    248,    274,    275,    253,    217,    174,
     &  133,     90,     64,     48,     29,     22,     16,
     &   15,     14,     14,     16,     17,     19,     19,
     &   26,   3072,   5656,    238,    269,    290,    326,
     &  359,    367,    317,    271,    220,    161,    122,
     &   90,     69,     53,     40,     31,     22,     16,
     &   12,      9,      9,      9,      9,     27,   3276,
     & 5947,    294,    337,    365,    369,    400,    431,
     &  432,    392,    328,    285,    213,    179,    149,
     &  121,     91,     75,     63,     50,     41,     32,
     &   25,     21,     22,     21,     27,   3481,   6109,
     &  317,    364,    419,    503,    532,    523,    529,
     &  481,    405,    328,    280,    264,    268,    221,
     &  181,    139,    126,    100,     75,     55,     34,
     &   24,     25,     28,     28,   3686,   6190,    343,
     &  405,    470,    562,    651,    730,    708,    626,
     &  537,    504,    379,    363,    306,    253,    208,
     &  200,    178,    140,    115,     92,     70,     53,
     &   39,     39,     36,     28,   3891,   6206,    365,
     &  445,    541,    662,    824,    897,    802,    782,
     &  779,    707,    593,    402,    344,    290,    247,
     &  186,    158,    126,    119,    107,    108,    124,
     &  139,    164,    163,     27,   4096,   6128,    404,
     &  510,    657,    838,   1021,   1178,   1271,   1243,
     & 1049,    792,    528,    424,    341,    247,    204,
     &  216,    167,    138,    119,    114,    116,    120,
     &  123,    129,     27,   4300,   6047,    403,    525,
     &  722,    956,   1208,   1471,   1653,   1459,   1209,
     &  931,    675,    545,    446,    324,    301,    274,
     &  246,    199,    167,    148,    141,    145,    149,
     &  154,     27,   4505,   5954,    413,    539,    755,
     & 1019,   1412,   1893,   2242,   2018,   1607,   1396,
     &  705,    538,    376,    257,    260,    254,    224,
     &  188,    165,    162,    163,    166,    166,    171,
     &   26,   4710,   5831,    393,    526,    783,   1132,
     & 1637,   2207,   2543,   2512,   1967,   1553,   1124,
     &  752,    514,    422,    408,    350,    268,    241,
     &  210,    183,    172,    165,    177,     26,   4915,
     & 5720,    388,    489,    700,   1070,   1761,   2641,
     & 3235,   3380,   2361,   2063,    988,    764,    470,
     &  350,    422,    406,    373,    346,    292,    245,
     &  221,    223,    234,     25,   5120,   5570,    368,
     &  483,    697,   1103,   1823,   2754,   3419,   3662/

      DATA (MAPPMAX (I), I=12251,12740) /
     & 2812,   2345,   1375,   1074,    888,    744,    650,
     &  621,    627,    559,    390,    311,    187,    193,
     &   25,   5324,   5376,    375,    511,    799,   1243,
     & 2053,   2909,   3520,   3867,   3603,   2684,   2042,
     & 1486,   1216,    970,    745,    548,    371,    329,
     &  322,    330,    341,    357,     24,   5529,   5212,
     &  386,    533,    809,   1236,   2047,   3064,   3902,
     & 4358,   3620,   2889,   2296,   2027,   1554,   1290,
     & 1191,   1038,    870,    656,    599,    461,    376,
     &   23,   5734,   4989,    384,    529,    822,   1281,
     & 2165,   3129,   3970,   5050,   5868,   4244,   3773,
     & 1791,   1338,   1062,    974,    917,    818,    652,
     &  476,    461,     22,   5939,   4782,    423,    539,
     &  788,   1347,   2250,   3161,   4395,   6014,   6958,
     & 5178,   4181,   2859,   1915,   1602,   1323,   1001,
     &  667,    578,    526,     21,   6144,   4544,    414,
     &  564,    850,   1299,   2166,   3030,   4228,   6234,
     & 7413,   6343,   5077,   4205,   3680,   2731,   2089,
     & 1697,   1462,    985,     20,   6348,   4316,    427,
     &  559,    850,   1366,   2069,   2996,   4411,   6126,
     & 7189,   6373,   5202,   4296,   3409,   2845,   2359,
     & 2145,   1929,     19,   6553,   4012,    398,    545,
     &  854,   1389,   2037,   3011,   4626,   7005,   8255,
     & 7020,   6005,   5191,   4457,   3471,   3343,   2774,
     &   18,   6758,   3718,    438,    543,    787,   1503,
     & 2470,   3567,   5313,   7725,   9364,   7556,   9145,
     & 7335,   6152,   3911,   3415,     17,   6963,   3376,
     &  411,    624,   1083,   1815,   2937,   4377,   7047,
     & 9559,  11176,  12183,   8484,   8000,   3648,   3056,
     &   15,   7168,   2991,    463,    717,   1292,   2343,
     & 3672,   5133,   8935,  12658,  14292,  13855,  10203,
     & 6825,     14,   7372,   2588,    566,    823,   1413,
     & 2640,   3944,   5917,  10062,  15192,  19220,  13540,
     &11172,     12,   7577,   2082,    703,   1023,   1660,
     & 3035,   5330,   9723,  16136,  22136,  24511,     10,
     & 7782,   1620,    578,    947,   1671,   3213,   5905,
     &10064,  23750,      8,   7987,   1198,    651,   1387,
     & 3101,   6385,  14706,      6,   8192,    625,   1043,
     & 2251,   3934,      4,   8396,    206,   1784,      3,
     & 8601,      0,      3,  32767,      0,    586,    800,
     &    3,      0,      0,      3,   2334,      0,     10,
     & 2355,   1651,     26,     24,     19,     16,     14,
     &   11,      7,     14,   2396,   2773,     47,     35,
     &   30,     19,     11,     10,      6,      7,      6,
     &    4,      4,     17,   2457,   3492,     85,     72,
     &   58,     49,     43,     28,     23,     17,     11,
     &   10,      5,      6,      5,      5,     21,   2662,
     & 4559,    206,    207,    175,    145,    103,     70,
     &   60,     48,     39,     32,     24,     19,     17,
     &   15,     14,     12,     12,     13,     23,   2867,
     & 5083,    235,    268,    293,    276,    248,    210,
     &  166,    118,     81,     59,     42,     27,     20,
     &   15,     15,     14,     15,     16,     17,     18,
     &   25,   3072,   5545,    233,    269,    316,    367,
     &  392,    373,    313,    257,    204,    146,    108,
     &   81,     62,     47,     36,     27,     20,     14,
     &   11,      9,      9,      9,     26,   3276,   5795,
     &  302,    343,    374,    384,    422,    461,    453,
     &  401,    339,    253,    205,    168,    136,    107,
     &   82,     67,     56,     44,     37,     27,     22,
     &   22,     22,     27,   3481,   5892,    316,    381,
     &  456,    516,    552,    557,    556,    489,    413,
     &  321,    298,    288,    237,    191,    152,    132,
     &  108,     79,     58,     38,     26,     24,     28,
     &   28,     27,   3686,   5927,    352,    420,    489,
     &  592,    706,    787,    719,    646,    558,    496,
     &  386,    351,    285,    236,    205,    186,    143,
     &  117,     92,     72,     54,     38,     38,     39,
     &   26,   3891,   5886,    391,    470,    577,    711,
     &  884,    914,    848,    864,    818,    716,    502/

      DATA (MAPPMAX (I), I=12741,13230) /
     &  387,    317,    272,    203,    159,    130,    119,
     &  108,    108,    127,    140,    139,     26,   4096,
     & 5774,    442,    539,    685,    879,   1093,   1225,
     & 1335,   1242,   1087,    722,    518,    407,    296,
     &  226,    235,    172,    137,    120,    113,    116,
     &  119,    122,    127,     26,   4300,   5683,    446,
     &  576,    765,   1046,   1357,   1633,   1709,   1396,
     & 1128,    840,    655,    519,    395,    322,    298,
     &  256,    206,    171,    149,    142,    145,    150,
     &  151,     25,   4505,   5557,    450,    593,    834,
     & 1190,   1637,   2082,   2206,   1949,   1604,   1055,
     &  694,    477,    297,    288,    289,    248,    202,
     &  169,    167,    165,    171,    176,     25,   4710,
     & 5419,    410,    560,    847,   1276,   1911,   2489,
     & 2756,   2372,   1873,   1544,    801,    642,    460,
     &  426,    390,    292,    261,    222,    191,    175,
     &  170,    180,     24,   4915,   5263,    410,    541,
     &  807,   1310,   2261,   3070,   3403,   3048,   2220,
     & 1607,    900,    618,    383,    429,    413,    381,
     &  357,    304,    253,    228,    236,     23,   5120,
     & 5070,    408,    563,    882,   1464,   2411,   3301,
     & 3720,   3436,   2804,   1451,   1183,    964,    805,
     &  694,    641,    645,    587,    403,    327,    195,
     &   23,   5324,   4867,    394,    542,    852,   1405,
     & 2736,   3415,   3626,   4424,   2964,   2313,   1735,
     & 1388,   1061,    831,    627,    440,    356,    336,
     &  342,    350,     22,   5529,   4686,    423,    587,
     &  918,   1448,   2371,   3279,   3998,   4660,   3279,
     & 2659,   2282,   1998,   1422,   1272,   1149,    990,
     &  762,    658,    553,     21,   5734,   4411,    447,
     &  590,    928,   1537,   2579,   3575,   4651,   5781,
     & 5550,   4466,   2260,   1536,   1184,   1011,    957,
     &  862,    744,    539,     20,   5939,   4192,    451,
     &  616,    887,   1415,   2263,   3461,   5136,   6671,
     & 7347,   4944,   3989,   2171,   1788,   1427,   1213,
     &  703,    622,     19,   6144,   3925,    433,    565,
     &  863,   1294,   2158,   3174,   4691,   7093,   8659,
     & 5670,   4798,   4110,   3485,   2431,   1971,   1545,
     &   18,   6348,   3640,    427,    610,    929,   1370,
     & 2001,   3114,   4952,   6909,   8110,   5557,   4896,
     & 4235,   3521,   2541,   2501,     16,   6553,   3273,
     &  444,    569,    798,   1222,   2091,   3677,   6016,
     & 7836,   8649,   6757,   5702,   4846,   4416,     15,
     & 6758,   2918,    465,    597,    921,   1557,   2677,
     & 4612,   7004,   8724,   9333,   9782,   8788,   7654,
     &   14,   6963,   2567,    466,    682,   1138,   1906,
     & 3275,   5345,   8103,  10587,  12655,  13731,  10429,
     &   12,   7168,   2094,    565,    933,   1536,   2760,
     & 4289,   6520,  11930,  17393,  20379,     10,   7372,
     & 1636,    613,    859,   1469,   3002,   5039,   9244,
     &14818,      8,   7577,   1041,    694,   1139,   2026,
     & 4878,   3600,      5,   7782,    453,    954,   2133,
     &    4,   7987,    127,   1247,      3,   8192,      0,
     &    3,  32767,      0,    537,   1000,      3,      0,
     &    0,      3,   2334,      0,     10,   2355,   1641,
     &   26,     24,     19,     16,     14,     11,      7,
     &   14,   2396,   2768,     47,     35,     30,     19,
     &   11,     10,      6,      7,      6,      4,      3,
     &   17,   2457,   3485,     85,     73,     57,     49,
     &   42,     29,     22,     17,     11,      9,      6,
     &    6,      5,      5,     21,   2662,   4533,    208,
     &  210,    177,    145,    101,     70,     58,     48,
     &   38,     31,     24,     19,     17,     14,     14,
     &   12,     13,     12,     23,   2867,   5017,    243,
     &  287,    317,    279,    241,    204,    153,    110,
     &   76,     54,     38,     25,     18,     16,     14,
     &   15,     15,     16,     18,     18,     25,   3072,
     & 5422,    240,    282,    353,    410,    427,    364,
     &  302,    238,    176,    129,     95,     72,     55,
     &   42,     31,     24,     17,     12,     10,      9/

      DATA (MAPPMAX (I), I=13231,13720) /
     &    9,      9,     25,   3276,   5632,    319,    362,
     &  398,    420,    472,    483,    449,    387,    324,
     &  232,    189,    152,    121,     91,     73,     59,
     &   49,     39,     31,     24,     22,     22,     26,
     & 3481,   5668,    334,    384,    476,    548,    594,
     &  608,    579,    491,    403,    331,    311,    260,
     &  213,    163,    139,    114,     83,     60,     41,
     &   28,     24,     26,     27,     26,   3686,   5663,
     &  363,    440,    539,    671,    771,    809,    731,
     &  652,    579,    454,    408,    320,    263,    216,
     &  192,    146,    116,     93,     71,     56,     38,
     &   39,     38,     25,   3891,   5590,    390,    494,
     &  628,    787,    938,    998,    959,    927,    816,
     &  631,    434,    355,    294,    216,    166,    133,
     &  119,    108,    108,    127,    139,    139,     25,
     & 4096,   5488,    439,    548,    712,    956,   1221,
     & 1313,   1423,   1223,   1066,    629,    475,    378,
     &  264,    248,    183,    147,    121,    113,    116,
     &  119,    122,    124,     25,   4300,   5391,    462,
     &  590,    815,   1147,   1486,   1696,   1689,   1395,
     & 1135,    804,    611,    496,    353,    323,    277,
     &  225,    183,    153,    145,    149,    156,    164,
     &   24,   4505,   5237,    471,    620,    909,   1371,
     & 1888,   2275,   2089,   1785,   1583,    827,    651,
     &  413,    313,    328,    285,    226,    188,    174,
     &  169,    173,    177,     23,   4710,   5072,    438,
     &  626,    990,   1489,   2217,   2779,   2530,   2211,
     & 1795,   1297,    841,    558,    447,    420,    331,
     &  279,    238,    205,    184,    183,     23,   4915,
     & 4894,    460,    620,    960,   1647,   2609,   3351,
     & 3469,   2594,   2498,    985,    782,    463,    411,
     &  434,    402,    367,    326,    269,    225,    231,
     &   22,   5120,   4718,    427,    626,   1012,   1647,
     & 2746,   3632,   3955,   2947,   2564,   1415,   1099,
     &  908,    760,    676,    636,    617,    524,    352,
     &  263,     21,   5324,   4536,    399,    567,    895,
     & 1582,   2811,   3566,   3831,   4215,   2942,   2098,
     & 1615,   1350,   1062,    836,    617,    424,    365,
     &  358,     20,   5529,   4299,    427,    611,    984,
     & 1608,   2615,   3672,   4264,   4274,   3217,   2568,
     & 2205,   1828,   1398,   1279,   1130,    960,    753,
     &   19,   5734,   4018,    465,    635,    992,   1699,
     & 2789,   3913,   5166,   6412,   4611,   4242,   1908,
     & 1458,   1114,   1032,    974,    872,     18,   5939,
     & 3720,    480,    637,    992,   1715,   2718,   3856,
     & 6070,   7945,   6072,   4946,   2954,   2044,   1621,
     & 1436,    820,     17,   6144,   3442,    449,    593,
     &  893,   1427,   2226,   3228,   5654,   7596,   7913,
     & 5856,   4834,   3891,   3162,   2365,     16,   6348,
     & 3118,    461,    617,    956,   1419,   2134,   3545,
     & 5429,   6890,   7630,   6088,   5039,   3774,   3354,
     &   14,   6553,   2743,    446,    573,    812,   1315,
     & 2340,   3832,   6452,   8236,   8643,   6810,   6034,
     &   13,   6758,   2336,    458,    601,    971,   1617,
     & 2966,   4546,   6853,   9297,  11021,  12439,     11,
     & 6963,   1940,    512,    724,   1195,   2095,   3359,
     & 5769,   9543,  13441,      9,   7168,   1433,    589,
     & 1003,   1877,   3125,   4547,  12666,      7,   7372,
     &  961,    560,    753,   1531,   4865,      4,   7577,
     &  249,   1099,      3,   7782,      0,      3,  32767,
     &    0,    460,   1500,      3,      0,      0,      3,
     & 2335,      0,     10,   2355,   1618,     26,     24,
     &   19,     16,     14,     11,     12,     14,   2396,
     & 2757,     47,     35,     30,     19,     11,     10,
     &    6,      7,      6,      4,      4,     17,   2457,
     & 3467,     90,     70,     57,     49,     42,     27,
     &   22,     17,     11,      9,      6,      5,      6,
     &    4,     21,   2662,   4434,    231,    225,    181,
     &  139,     86,     71,     54,     43,     36,     28,
     &   22,     18,     16,     14,     13,     13,     13/

      DATA (MAPPMAX (I), I=13721,14210) /
     &   13,     22,   2867,   4860,    285,    357,    329,
     &  280,    230,    181,    128,     89,     63,     44,
     &   30,     21,     17,     15,     14,     15,     16,
     &   17,     17,     24,   3072,   5133,    290,    369,
     &  461,    507,    412,    337,    250,    172,    133,
     &   93,     70,     53,     41,     30,     23,     16,
     &   13,      9,      9,      9,      9,     24,   3276,
     & 5178,    398,    468,    535,    602,    573,    495,
     &  415,    268,    218,    174,    133,     97,     78,
     &   63,     51,     41,     33,     26,     22,     22,
     &   22,     24,   3481,   5150,    400,    500,    628,
     &  714,    716,    646,    557,    451,    370,    293,
     &  237,    178,    140,    112,     86,     63,     40,
     &   28,     24,     27,     26,     23,   3686,   5100,
     &  423,    543,    708,    872,    976,    866,    757,
     &  654,    483,    383,    311,    243,    203,    136,
     &  114,     93,     72,     52,     39,     40,     23,
     & 3891,   4998,    425,    575,    830,   1082,   1251,
     & 1151,   1022,    882,    595,    432,    338,    263,
     &  178,    138,    121,    109,    107,    125,    136,
     &  163,     22,   4096,   4859,    494,    676,   1024,
     & 1421,   1544,   1576,   1303,   1105,    597,    480,
     &  353,    281,    205,    162,    123,    113,    117,
     &  121,    123,     22,   4300,   4749,    521,    751,
     & 1144,   1580,   1891,   1891,   1529,   1178,    818,
     &  651,    527,    385,    331,    250,    199,    158,
     &  148,    148,    154,     21,   4505,   4593,    523,
     &  771,   1245,   1833,   2310,   2340,   1959,   1674,
     &  979,    722,    473,    353,    373,    314,    242,
     &  193,    189,    189,     21,   4710,   4398,    525,
     &  788,   1368,   2161,   2835,   2588,   2454,   2009,
     & 1511,    970,    622,    477,    431,    328,    279,
     &  236,    197,    190,     20,   4915,   4195,    549,
     &  824,   1398,   2339,   3224,   3564,   3047,   3034,
     & 1148,    863,    468,    443,    438,    403,    372,
     &  325,    275,     19,   5120,   4009,    519,    800,
     & 1403,   2333,   3271,   3792,   3722,   3218,   1573,
     & 1236,   1011,    829,    726,    696,    682,    513,
     &   18,   5324,   3794,    474,    771,   1434,   2323,
     & 3228,   3978,   4521,   3173,   2310,   1720,   1444,
     & 1191,    967,    699,    536,     17,   5529,   3507,
     &  521,    787,   1367,   2368,   3422,   4161,   4566,
     & 3452,   2660,   2278,   2046,   1562,   1432,   1330,
     &   16,   5734,   3195,    505,    751,   1293,   2313,
     & 3598,   4998,   6052,   5776,   4857,   2289,   1631,
     & 1249,   1142,     15,   5939,   2856,    501,    716,
     & 1335,   2232,   3228,   4986,   7052,   8190,   5483,
     & 4172,   2302,   1910,     13,   6144,   2483,    486,
     &  693,   1093,   1812,   2940,   4346,   6856,   8898,
     & 6779,   5589,     12,   6348,   2065,    559,    748,
     & 1164,   1891,   2745,   4326,   6488,   8145,   7912,
     &   10,   6553,   1639,    491,    695,   1136,   1829,
     & 3290,   6066,   9598,      8,   6758,   1174,    529,
     &  732,   1352,   3118,   5550,      6,   6963,    765,
     &  656,   1003,   2203,      4,   7168,    217,   1351,
     &    3,   7372,      0,      3,  32767,      0,    417,
     & 2000,      3,      0,      0,      3,   2336,      0,
     &   10,   2355,   1594,     26,     24,     19,     16,
     &   14,     11,      7,     14,   2396,   2748,     47,
     &   35,     30,     19,     11,     10,      6,      7,
     &    6,      4,      4,     17,   2457,   3450,     92,
     &   71,     56,     49,     41,     27,     21,     16,
     &   11,      9,      6,      5,      6,      5,     21,
     & 2662,   4384,    243,    227,    184,    130,     85,
     &   72,     54,     41,     34,     27,     21,     18,
     &   15,     15,     13,     12,     13,     14,     22,
     & 2867,   4750,    336,    402,    324,    268,    226,
     &  156,    116,     77,     55,     38,     26,     19,
     &   16,     15,     15,     15,     16,     18,     18,
     &   23,   3072,   4921,    378,    479,    542,    477/

      DATA (MAPPMAX (I), I=14211,14700) /
     &  408,    273,    194,    150,    106,     74,     56,
     &   43,     32,     24,     17,     14,      9,      9,
     &    9,      9,     23,   3276,   4891,    469,    587,
     &  705,    663,    565,    470,    282,    231,    178,
     &  140,     94,     77,     61,     50,     40,     32,
     &   25,     23,     22,     22,     22,   3481,   4814,
     &  472,    623,    805,    800,    740,    650,    528,
     &  401,    301,    243,    161,    135,    105,     81,
     &   58,     36,     27,     23,     24,     22,   3686,
     & 4705,    504,    696,    943,   1062,   1021,    880,
     &  735,    497,    395,    315,    248,    176,    134,
     &  106,     83,     65,     45,     37,     41,     21,
     & 3891,   4579,    528,    769,   1105,   1400,   1280,
     & 1133,   1004,    621,    467,    356,    243,    166,
     &  128,    117,    109,    112,    128,    143,     21,
     & 4096,   4420,    616,    922,   1367,   1750,   1731,
     & 1440,   1227,    674,    524,    377,    285,    211,
     &  134,    122,    114,    120,    122,    131,     20,
     & 4300,   4288,    658,    979,   1467,   1972,   2098,
     & 1719,   1265,    906,    766,    628,    438,    318,
     &  252,    203,    159,    149,    154,     20,   4505,
     & 4134,    633,    983,   1575,   2203,   2578,   2165,
     & 1770,   1321,    854,    572,    365,    417,    362,
     &  274,    215,    207,    210,     19,   4710,   3916,
     &  664,   1059,   1746,   2522,   2969,   2761,   2278,
     & 1830,   1110,    751,    520,    459,    358,    293,
     &  249,    207,     18,   4915,   3717,    632,   1044,
     & 1792,   2737,   3557,   3928,   2885,   1767,   1393,
     &  575,    419,    454,    430,    401,    363,     17,
     & 5120,   3502,    586,    998,   1767,   2807,   3783,
     & 4275,   3130,   2725,   1366,   1123,    924,    802,
     &  741,    732,     16,   5324,   3264,    555,    892,
     & 1699,   2673,   3535,   4322,   4005,   3328,   2002,
     & 1513,   1362,   1142,    911,     15,   5529,   2944,
     &  581,    920,   1693,   2736,   3789,   4751,   4003,
     & 3348,   2484,   2244,   1827,   1621,     14,   5734,
     & 2591,    513,    805,   1550,   2684,   4050,   5924,
     & 7518,   4639,   4143,   1846,   1569,     12,   5939,
     & 2198,    559,    857,   1592,   2606,   3719,   6539,
     & 8965,   6543,   5582,     11,   6144,   1804,    509,
     &  724,   1211,   2046,   3043,   5020,   8322,   9479,
     &    9,   6348,   1369,    556,    760,   1168,   1792,
     & 2626,   4852,      7,   6553,    904,    542,    769,
     & 1307,   3388,      5,   6758,    384,    589,   1209,
     &    4,   6963,     22,   1965,      3,   7168,      0,
     &    3,  32767,      0,    356,   3000,      3,      0,
     &    0,      3,   2336,      0,     10,   2355,   1541,
     &   26,     24,     19,     16,     14,     11,      7,
     &   14,   2396,   2731,     47,     35,     30,     19,
     &   11,     10,      6,      7,      6,      4,      4,
     &   17,   2457,   3414,     91,     76,     58,     47,
     &   39,     26,     21,     16,     10,     10,      5,
     &    5,      6,      5,     20,   2662,   4305,    259,
     &  225,    188,    123,     87,     70,     54,     40,
     &   32,     25,     20,     17,     15,     14,     12,
     &   13,     13,     22,   2867,   4631,    394,    403,
     &  346,    273,    203,    141,    103,     67,     46,
     &   33,     23,     17,     16,     14,     16,     15,
     &   17,     18,     16,     22,   3072,   4686,    550,
     &  622,    547,    487,    278,    216,    172,    117,
     &   81,     61,     44,     33,     25,     18,     13,
     &   11,      9,      8,      9,     22,   3276,   4612,
     &  625,    786,    779,    669,    520,    329,    243,
     &  185,    142,    105,     80,     62,     49,     40,
     &   31,     25,     23,     22,     22,     21,   3481,
     & 4415,    723,    915,    955,    848,    700,    541,
     &  407,    318,    215,    155,    125,    100,     72,
     &   47,     32,     23,     22,     27,     20,   3686,
     & 4249,    760,   1040,   1233,   1156,   1024,    710,
     &  520,    408,    311,    213,    164,    117,     97/

      DATA (MAPPMAX (I), I=14701,15190) /
     &   70,     54,     43,     37,     19,   3891,   4096,
     &  729,   1084,   1511,   1453,   1283,   1163,    645,
     &  490,    375,    213,    173,    123,    115,    102,
     &  119,    134,     19,   4096,   3855,    988,   1449,
     & 1939,   1901,   1746,   1028,    788,    532,    359,
     &  264,    197,    130,    112,    116,    120,    126,
     &   18,   4300,   3672,    936,   1449,   2080,   2226,
     & 1975,   1360,   1012,    814,    665,    433,    321,
     &  252,    189,    170,    167,     17,   4505,   3465,
     &  941,   1423,   2249,   2649,   2384,   2088,   1236,
     & 1009,    551,    394,    444,    397,    321,    230,
     &   16,   4710,   3209,    973,   1611,   2413,   3340,
     & 3003,   2585,   1706,   1294,    759,    530,    465,
     &  381,    304,     15,   4915,   2972,    926,   1621,
     & 2744,   3852,   3959,   3792,   1435,   1429,    531,
     &  464,    466,    460,     14,   5120,   2727,    860,
     & 1680,   2825,   3893,   4473,   3516,   2560,   1585,
     & 1228,   1025,    879,     13,   5324,   2450,    793,
     & 1420,   2561,   3593,   3924,   4135,   3819,   2078,
     & 1696,   1556,     12,   5529,   2114,    715,   1189,
     & 2140,   3391,   4558,   4884,   3954,   3035,   2751,
     &   10,   5734,   1753,    564,    952,   1934,   3405,
     & 5226,   6766,   7189,      9,   5939,   1283,    647,
     & 1065,   1884,   3106,   5486,   5005,      7,   6144,
     &  836,    576,    814,   1407,   2953,      5,   6348,
     &  357,   1042,   1659,      3,   6553,      0,      3,
     &32767,      0,    301,   5000,      3,      0,      0,
     &    3,   2338,      0,      9,   2355,   1394,     26,
     &   24,     19,     16,     14,     11,     14,   2396,
     & 2686,     47,     35,     30,     19,     11,     10,
     &    6,      7,      6,      4,      4,     17,   2457,
     & 3357,     95,     75,     61,     48,     37,     25,
     &   20,     14,     10,      9,      5,      5,      6,
     &    5,     20,   2662,   4213,    257,    226,    190,
     &  129,     88,     71,     56,     39,     30,     23,
     &   19,     16,     15,     13,     13,     13,     11,
     &   21,   2867,   4496,    419,    415,    351,    284,
     &  185,    147,     89,     62,     40,     28,     20,
     &   16,     15,     15,     16,     16,     18,     17,
     &   21,   3072,   4480,    737,    654,    594,    335,
     &  262,    210,    142,     99,     76,     57,     35,
     &   26,     19,     15,     11,      9,      9,      9,
     &   20,   3276,   4347,    897,    871,    771,    623,
     &  403,    306,    223,    151,    121,     82,     73,
     &   51,     42,     31,     25,     22,     23,     20,
     & 3481,   4100,    910,   1139,   1014,    880,    575,
     &  452,    360,    249,    172,    132,    107,     75,
     &   50,     38,     20,     22,     32,     19,   3686,
     & 3882,    937,   1274,   1334,   1187,    882,    603,
     &  466,    381,    231,    181,    129,    108,     78,
     &   63,     51,     39,     18,   3891,   3611,    987,
     & 1412,   1615,   1496,   1374,    762,    609,    413,
     &  246,    188,    125,    120,    105,    119,    134,
     &   16,   4096,   3313,   1261,   1989,   2112,   1894,
     & 1370,    972,    620,    426,    305,    220,    129,
     &  115,    118,     15,   4300,   3036,   1335,   2063,
     & 2337,   2152,   1625,   1132,    892,    780,    474,
     &  370,    288,    233,     14,   4505,   2745,   1309,
     & 2054,   2620,   2527,   2253,   1536,   1216,    692,
     &  465,    410,    493,     13,   4710,   2428,   1374,
     & 2244,   3316,   3148,   2775,   2037,   1477,    969,
     &  590,    574,     12,   4915,   2118,   1326,   2246,
     & 3819,   4315,   3900,   2004,   1897,    681,    558,
     &   11,   5120,   1798,   1439,   2680,   3952,   4656,
     & 4278,   2684,   1935,   1399,      9,   5324,   1456,
     & 1349,   2202,   3752,   3848,   4096,   4135,      8,
     & 5529,   1058,    971,   1678,   2954,   4556,   6312,
     &    6,   5734,    618,    656,   1147,   3056,      4,
     & 5939,    133,   1385,      3,   6144,      0,      3,
     &32767,      0,    284,   6000,      3,      0,      0/

      DATA (MAPPMAX (I), I=15191,15680) /
     &    3,   2339,      0,      9,   2355,   1321,     26,
     &   24,     19,     16,     14,     13,     14,   2396,
     & 2665,     47,     35,     30,     19,     11,     10,
     &    6,      7,      6,      4,      3,     17,   2457,
     & 3328,     95,     77,     60,     49,     37,     25,
     &   18,     14,     10,      8,      5,      6,      5,
     &    6,     20,   2662,   4166,    260,    228,    190,
     &  130,     87,     71,     56,     38,     29,     22,
     &   19,     16,     14,     13,     13,     13,     14,
     &   21,   2867,   4443,    431,    410,    358,    280,
     &  185,    144,     88,     60,     41,     26,     19,
     &   16,     16,     15,     15,     17,     18,     15,
     &   21,   3072,   4422,    783,    656,    568,    325,
     &  259,    211,    139,    100,     75,     56,     33,
     &   25,     18,     14,     11,      9,      8,      9,
     &   20,   3276,   4258,    954,    881,    776,    595,
     &  409,    294,    213,    147,    120,     77,     68,
     &   49,     40,     30,     24,     22,     22,     19,
     & 3481,   4002,    974,   1162,   1024,    860,    536,
     &  438,    352,    241,    173,    135,    106,     71,
     &   53,     32,     20,     23,     18,   3686,   3762,
     &  973,   1275,   1326,   1196,    857,    602,    463,
     &  372,    223,    178,    126,    108,     81,     65,
     &   54,     17,   3891,   3471,   1050,   1419,   1597,
     & 1477,   1317,    781,    605,    412,    252,    190,
     &  128,    122,    108,    120,     16,   4096,   3149,
     & 1270,   1957,   2134,   1927,   1360,    973,    625,
     &  421,    308,    202,    128,    118,    125,     15,
     & 4300,   2838,   1416,   2073,   2375,   2223,   1497,
     & 1118,    873,    753,    474,    375,    297,    245,
     &   13,   4505,   2528,   1434,   2163,   2597,   2479,
     & 2145,   1446,   1227,    669,    482,    423,     12,
     & 4710,   2175,   1547,   2249,   3279,   3174,   2825,
     & 1838,   1483,    913,    617,     11,   4915,   1848,
     & 1487,   2479,   3663,   4142,   3976,   1933,   1934,
     &  697,      9,   5120,   1507,   1572,   2964,   4074,
     & 4600,   4736,   2293,      8,   5324,   1164,   1324,
     & 2156,   3661,   3914,   5216,      6,   5529,    707,
     & 1050,   1665,   3300,      4,   5734,    236,   1022,
     &    3,   5939,      0,      3,  32767,      0,    261,
     & 8000,      3,      0,      0,      3,   2340,      0,
     &    8,   2355,   1164,     26,     24,     19,     16,
     &   14,     14,   2396,   2599,     47,     35,     30,
     &   19,     11,     10,      6,      7,      6,      4,
     &    4,     16,   2457,   3273,     97,     77,     61,
     &   48,     40,     23,     17,     13,     10,      7,
     &    5,      6,      6,     19,   2662,   4075,    263,
     &  232,    194,    127,     88,     71,     56,     38,
     &   27,     21,     17,     16,     14,     13,     13,
     &   12,     20,   2867,   4345,    433,    416,    356,
     &  281,    183,    149,     88,     57,     43,     23,
     &   18,     16,     15,     15,     16,     17,     18,
     &   20,   3072,   4324,    776,    654,    566,    325,
     &  266,    216,    144,    101,     74,     56,     37,
     &   21,     16,     13,     10,      9,      9,     20,
     & 3276,   4113,    961,    885,    775,    608,    418,
     &  297,    211,    143,    112,     75,     62,     47,
     &   39,     32,     22,     19,     19,     19,   3481,
     & 3843,   1014,   1143,   1015,    821,    530,    422,
     &  343,    258,    189,    150,    118,     71,     59,
     &   27,     20,     24,     17,   3686,   3569,    998,
     & 1222,   1279,   1166,    857,    611,    466,    376,
     &  235,    183,    137,    105,     94,     78,     16,
     & 3891,   3252,   1017,   1337,   1554,   1446,   1282,
     &  821,    643,    488,    255,    215,    132,    129,
     &  117,     15,   4096,   2895,   1258,   1805,   2136,
     & 1922,   1441,    986,    635,    440,    304,    233,
     &  138,    120,     13,   4300,   2550,   1423,   2018,
     & 2319,   2179,   1554,   1136,    882,    765,    497,
     &  389,     12,   4505,   2188,   1558,   2072,   2597/

      DATA (MAPPMAX (I), I=15681,16170) /
     & 2520,   2123,   1387,   1202,    654,    510,     11,
     & 4710,   1817,   1485,   2088,   3140,   3040,   2706,
     & 2027,   1481,   1176,      9,   4915,   1435,   1478,
     & 2445,   3538,   4092,   3752,   2726,      8,   5120,
     & 1050,   1525,   2854,   4308,   4994,   5010,      6,
     & 5324,    624,   1446,   2529,   4453,      4,   5529,
     &  162,   1355,      3,   5734,      0,      3,  32767,
     &    0,    243,  10000,      3,      0,      0,      3,
     & 2342,      0,      7,   2355,   1015,     26,     24,
     &   19,     19,     13,   2396,   2537,     47,     35,
     &   30,     19,     11,     10,      6,      7,      6,
     &    5,     16,   2457,   3215,     98,     78,     61,
     &   50,     39,     23,     16,     13,      9,      6,
     &    5,      6,      6,     19,   2662,   3987,    268,
     &  234,    193,    129,     89,     70,     55,     39,
     &   25,     20,     17,     14,     14,     13,     13,
     &   13,     20,   2867,   4246,    433,    420,    356,
     &  277,    188,    150,     88,     55,     46,     22,
     &   17,     16,     15,     16,     15,     17,     19,
     &   20,   3072,   4223,    768,    648,    563,    335,
     &  271,    216,    149,    101,     73,     55,     36,
     &   26,     15,     12,      9,      9,      9,     19,
     & 3276,   3987,    928,    863,    765,    650,    423,
     &  313,    217,    141,    112,     73,     62,     48,
     &   39,     33,     20,     20,     18,   3481,   3694,
     & 1031,   1103,    987,    788,    520,    416,    338,
     &  278,    218,    173,    135,     80,     62,     29,
     &   21,     17,   3686,   3408,    963,   1170,   1225,
     & 1116,    894,    608,    471,    386,    264,    198,
     &  164,    103,     99,     92,     15,   3891,   3072,
     &  948,   1270,   1464,   1379,   1206,    916,    671,
     &  502,    362,    239,    176,    138,     14,   4096,
     & 2684,   1245,   1668,   2057,   1882,   1565,   1010,
     &  663,    474,    313,    253,    148,     13,   4300,
     & 2326,   1401,   1915,   2285,   2132,   1647,   1153,
     &  903,    777,    509,    412,     11,   4505,   1923,
     & 1493,   1974,   2545,   2484,   2168,   1452,   1173,
     &  786,      9,   4710,   1527,   1291,   1808,   2793,
     & 2910,   2741,   2402,      8,   4915,   1082,   1395,
     & 2062,   3312,   3867,   4519,      6,   5120,    665,
     & 1495,   2923,   4786,      4,   5324,    174,   1699,
     &    3,   5529,      0,      3,  32767,      0,    190,
     &20000,      3,      0,      0,      3,   2351,      0,
     &    5,   2355,    308,     26,     24,     12,   2396,
     & 2165,     47,     35,     30,     19,     11,     10,
     &    6,      7,      6,     15,   2457,   2905,     98,
     &   78,     62,     50,     38,     28,     20,     13,
     &    6,      5,      6,      6,     17,   2662,   3556,
     &  263,    236,    195,    137,     91,     72,     57,
     &   36,     26,     20,     17,     15,     14,     12,
     &   18,   2867,   3770,    416,    422,    355,    275,
     &  187,    144,     90,     61,     46,     30,     20,
     &   15,     14,     16,     17,     18,   3072,   3725,
     &  764,    644,    523,    341,    274,    216,    156,
     &  102,     76,     50,     40,     31,     23,     19,
     &   15,     17,   3276,   3389,    834,    767,    711,
     &  674,    446,    357,    233,    166,    123,     90,
     &   64,     60,     51,     48,     15,   3481,   3049,
     &  819,    979,    879,    744,    545,    415,    349,
     &  324,    304,    258,    204,    149,     14,   3686,
     & 2700,    831,    960,    948,    876,    787,    581,
     &  497,    415,    342,    318,    273,     13,   3891,
     & 2319,    663,    918,   1221,   1167,   1044,    886,
     &  752,    540,    457,    520,     11,   4096,   1858,
     &  934,   1448,   1755,   1802,   1612,   1232,    830,
     &  627,      9,   4300,   1402,    903,   1385,   2215,
     & 2305,   1998,   1368,      7,   4505,    856,   1286,
     & 1601,   2390,   2881,      5,   4710,    283,   1313,
     & 1680,      3,   4915,      0,      3,  32767,      0,
     &  127,  40000,      3,      0,      0,      3,   2369/

      DATA (MAPPMAX (I), I=16171,16291) /
     &    0,      9,   2396,   1332,     47,     35,     30,
     &   19,     11,     10,     12,   2457,   2210,     85,
     &   69,     55,     45,     38,     30,     23,     19,
     &   15,     14,   2662,   2764,    228,    222,    184,
     &  154,     96,     75,     59,     45,     32,     24,
     &   20,     15,   2867,   2923,    368,    410,    329,
     &  237,    198,    138,    113,     70,     53,     45,
     &   36,     26,     15,   3072,   2820,    711,    610,
     &  555,    292,    236,    186,    163,    133,     94,
     &   72,     55,     48,     13,   3276,   2454,    674,
     &  589,    443,    490,    523,    403,    328,    264,
     &  198,    158,     11,   3481,   2040,    605,    685,
     &  723,    677,    605,    417,    353,    332,     10,
     & 3686,   1613,    586,    614,    616,    682,    704,
     &  649,    542,      8,   3891,   1074,    591,    725,
     &  841,    998,   1108,      6,   4096,    521,    880,
     & 1122,   1201,      3,   4300,      0,      3,  32767,
     &    0,      0/

      END
!--------------------------------------------------------------------------------------------
!+
! NAME:
!	Init_AP8min

! PURPOSE:
!	This subroutine sets block data for AP8min
!
! CATEGORY:
!	Radiation belt model
!
! CALLING SEQUENCE:
!	call Init_AP8min
!
! INPUTS:
!	None
!
! OUTPUTS:
!	None
!
! COMMON BLOCKS:
!	COMMON /PROMIN/ IHEADPMIN, MAPPMIN
!
! MODIFICATION HISTORY:
!	Written by: D. Bilitza, 1987
!				S. Bourdarie (introduced in version 3.0) - September 2005
!-
!--------------------------------------------------------------------------------------------
      SUBROUTINE Init_AP8min
!      BLOCK DATA AP8MIN

      COMMON /PROMIN/ IHEADPMIN, MAPPMIN

      INTEGER*4 IHEADPMIN(8)
      INTEGER*4 MAPPMIN(16582)

      DATA IHEADPMIN /2,4,1964,100,2048,2048,1024,16582/

      DATA (MAPPMIN(I), I= 1, 490) /
     & 1895,      10,      3,      0,      0,      3,   2299,
     &    0,      11,   2334,   2001,     17,     12,     13,
     &   12,      12,     10,      9,      9,     14,   2355,
     & 2665,      31,     24,     23,     21,     19,     13,
     &   10,       6,      5,      1,      1,     16,   2396,
     & 3252,      64,     44,     33,     19,     12,     10,
     &    7,       6,      6,      5,      4,      4,      4,
     &   18,    2457,   3659,    124,     96,     72,     43,
     &   34,      29,     23,     20,     16,     12,     10,
     &    8,       7,      5,      9,     22,   2662,   4673,
     &  220,     209,    185,    158,    132,    105,     78,
     &   60,      48,     34,     25,     21,     18,     16,
     &   14,      11,     10,     10,     16,     24,   2867,
     & 5312,     207,    252,    294,    295,    280,    233,
     &  199,     118,     87,     68,     54,     47,     32,
     &   26,      25,     25,     25,     23,     25,     26,
     &   29,      26,   3072,   5865,    229,    257,    288,
     &  311,     315,    332,    327,    300,    258,    211,
     &  137,     109,     78,     58,     43,     37,     34,
     &   31,      30,     30,     29,     29,     32,     28,
     & 3276,    6182,    286,    325,    361,    380,    405,
     &  399,     386,    367,    343,    311,    251,    225,
     &  181,     146,    116,     85,     68,     53,     42,
     &   33,      32,     31,     31,     35,     49,     29,
     & 3481,    6577,    294,    340,    389,    450,    477,
     &  462,     483,    486,    448,    395,    342,    332,
     &  283,     242,    213,    181,    144,    123,     99,
     &   76,      60,     49,     36,     35,     36,     47,
     &   30,    3686,   6855,    312,    364,    416,    478,
     &  522,     543,    590,    612,    572,    548,    519,
     &  455,     384,    339,    300,    268,    230,    185,
     &  165,     106,     83,     63,     60,     57,     57,
     &   65,      88,     31,   3891,   7129,    314,    375,
     &  448,     543,    625,    706,    804,    914,    900,
     &  806,     699,    595,    499,    423,    329,    256,
     &  198,     156,    126,    110,     95,     79,     69,
     &   65,      65,     63,     70,     97,     32,   4096,
     & 7407,     334,    394,    475,    584,    699,    834,
     &  982,    1118,   1269,   1290,   1180,   1010,    882,
     &  527,     420,    357,    211,    139,     96,     60,
     &   31,      22,     17,     18,     19,     15,     17,
     &   20,      18,     33,   4300,   7621,    366,    450,
     &  557,     680,    764,    980,   1288,   1562,   1633,
     & 1462,    1298,   1112,    869,    534,    371,    220,
     &  126,     100,     89,     87,     84,     87,     89,
     &   88,      86,     89,     90,     91,     94,    108,
     &   34,    4505,   7867,    382,    471,    591,    752,
     &  959,    1202,   1526,   1757,   1896,   1944,   1668,
     & 1438,     815,    611,    409,    245,    210,    200,
     &  207,     225,    191,    168,     87,     73,     70,
     &   71,      72,     75,     78,     85,    108,     35,
     & 4710,    8054,    415,    508,    642,    842,   1149,
     & 1487,    1814,   2029,   2078,   1929,   1714,   1478,
     & 1091,     807,    643,    522,    471,    337,    259,
     &  203,     145,    111,    115,    114,    118,    118,
     &  118,     117,    116,    117,    119,    127,     35,
     & 4915,    8191,    429,    552,    758,   1046,   1448,
     & 1883,    2223,   2302,   2425,   2212,   1965,   1705,
     & 1319,     882,    662,    351,    272,    235,    214,
     &  204,     199,    208,    215,    221,    225,    223,
     &  221,     216,    214,    214,    218,    233,     36,
     & 5120,    8348,    436,    579,    831,   1178,   1608,
     & 2082,    2481,   2772,   2699,   2853,   2711,   2386,
     & 2067,    1300,    953,    714,    437,    260,    179,
     &  153,     131,    127,    126,    127,    129,    129,
     &  130,     127,    126,    126,    129,    128,    129,
     &   36,    5324,   8441,    478,    648,    958,   1333,
     & 1838,    2360,   2813,   3385,   3764,   3186,   2699,
     & 2350,    1595,   1221,    876,    463,    387,    248,
     &  248,     244,    250,    251,    257,    263,    265  /

      DATA (MAPPMIN(I), I=  491,  980) /
     &  261,     252,    246,    249,    254,    259,    264,
     &  270,      37,   5529,   8530,    535,    724,   1032,
     & 1504,    2122,   2726,   3348,   4261,   3641,   3439,
     & 3285,    3025,   2062,   1545,   1132,    720,    563,
     &  483,     416,    356,    243,    247,    249,    253,
     &  257,     258,    257,    255,    253,    254,    251,
     &  250,     245,    242,     37,   5734,   8610,    564,
     &  781,    1155,   1669,   2313,   3100,   4106,   5090,
     & 4723,    4171,   3579,   2862,   2142,   1776,   1466,
     & 1275,    1047,    756,    582,    439,    289,    288,
     &  267,     236,    209,    205,    202,    201,    198,
     &  201,     202,    200,    193,    181,     37,   5939,
     & 8675,     582,    833,   1259,   1906,   2829,   3866,
     & 5092,    5899,   5327,   4479,   3301,   2164,   1743,
     & 1398,    1286,   1159,    806,    659,    542,    452,
     &  444,     449,    449,    447,    451,    455,    451,
     &  447,     444,    443,    442,    439,    440,    436,
     &   38,    6144,   8722,    604,    898,   1472,   2354,
     & 3537,    5035,   6435,   7318,   5921,   4842,   3838,
     & 3039,    2426,   1935,   1450,   1152,    921,    737,
     &  566,     486,    412,    325,    316,    310,    311,
     &  309,     319,    328,    328,    325,    317,    302,
     &  301,     295,    280,     38,   6348,   8768,    614,
     &  994,    1649,   2753,   4419,   6504,   8256,   7781,
     & 6309,    4899,   3226,   2528,   1992,   1677,   1409,
     & 1282,    1227,   1049,    858,    720,    721,    714,
     &  631,     538,    442,    368,    364,    370,    375,
     &  375,     375,    370,    361,    349,    325,     38,
     & 6553,    8793,    632,   1027,   1837,   3274,   5576,
     & 7960,    9469,   7581,   6321,   4676,   3714,   3020,
     & 2513,    2112,   1887,   1731,   1408,   1393,   1359,
     & 1236,    1043,    919,    822,    716,    414,    330,
     &  249,     251,    249,    250,    253,    253,    249,
     &  240,     224,     38,   6758,   8790,    663,   1131,
     & 2137,    3947,   6752,   9468,  10966,   8668,   6878,
     & 5892,    3838,   2367,   2154,   2026,   1932,   1789,
     & 1474,    1308,   1252,   1189,   1103,    982,    853,
     &  587,     601,    682,    509,    405,    254,    237,
     &  246,     245,    247,    239,    232,     38,   6963,
     & 8763,     745,   1340,   2634,   4863,   8313,  11557,
     &12095,    9931,   8798,   4460,   3611,   2250,   1768,
     & 1661,    1348,   1094,    886,    860,    794,    728,
     &  706,     870,    860,    847,    854,    868,    878,
     &  878,     875,    870,    869,    850,    832,    804,
     &  778,      38,   7168,   8711,    812,   1573,   2889,
     & 5612,    9755,  13321,  13968,  11093,   8522,   5144,
     & 4171,    3210,   2652,   2098,   1824,   1559,   1306,
     & 1098,     936,   1073,   1128,    738,    712,    677,
     &  668,     659,    668,    673,    670,    674,    679,
     &  679,     665,    639,    583,     37,   7372,   8632,
     &  966,    1835,   3701,   6786,  10735,  15042,  13841,
     &12011,    7985,   5449,   4253,   2902,   2557,   2415,
     & 2032,    1781,   1650,   1528,   1385,   1275,   1156,
     & 1045,     933,    851,    829,    820,    827,    845,
     &  852,     845,    832,    818,    801,    772,     37,
     & 7577,    8559,   1040,   1986,   4117,   7854,  12742,
     &18249,   15231,  13868,   7061,   5599,   3795,   2676,
     & 2102,    1904,   1832,   1706,   1638,   1613,   1519,
     & 1397,    1340,   1304,   1283,   1256,   1245,   1227,
     & 1254,    1281,   1269,   1254,   1223,   1167,   1125,
     & 1010,      37,   7782,   8451,   1037,   2053,   4320,
     & 8234,   13247,  19364,  16095,  14423,   8192,   6289,
     & 5014,    3402,   2768,   2848,   2891,   2928,   2726,
     & 2438,    2119,   1818,   1556,   1405,   1251,   1121,
     & 1098,    1078,   1085,   1087,   1092,   1090,   1060,
     & 1020,     948,    798,     36,   7987,   8398,   1040,
     & 2238,    4737,   9429,  15286,  20257,  17131,  15236,
     & 9150,    7359,   6088,   5045,   3915,   3181,   2606,
     & 2270,    2114,   1920,   1647,   1372,   1191,   1089,
     & 1223,    1438,   1491,   1439,   1440,   1463,   1484  /
      DATA (MAPPMIN(I), I=  981, 1470) /
     & 1484,    1473,   1442,   1371,     36,   8192,   8322,
     &  992,    1981,   5377,  10700,  16271,  21436,  18570,
     &16084,   11366,   8239,   6445,   4987,   3301,   3042,
     & 3084,    3095,   3126,   3010,   2801,   2527,   2169,
     & 1640,    1256,   1294,   1271,   1281,   1287,   1312,
     & 1303,    1303,   1277,   1267,   1200,     36,   8396,
     & 8254,    1041,   2217,   5596,  11187,  17150,  22697,
     &20371,   18072,  11189,   8195,   6516,   4650,   3902,
     & 3445,    3211,   3220,   3271,   3298,   3244,   3038,
     & 2642,    2216,   1927,   1921,   1923,   1934,   1945,
     & 1944,    1931,   1913,   1818,   1611,   1213,     36,
     & 8601,    8201,   1062,   2374,   5797,  11755,  18805,
     &25569,   23593,  20955,  12019,   8725,   6250,   4196,
     & 3353,    3278,   3143,   2868,   2756,   2757,   2671,
     & 2585,    2560,   2549,   2551,   2574,   2602,   2610,
     & 2605,    2580,   2526,   2444,   2308,   2126,   1593,
     &   35,    8806,   8142,   1054,   2438,   6321,  12529,
     &20567,   29223,  23710,  20002,  12878,   9315,   5969,
     & 3639,    3307,   2748,   2263,   2187,   2404,   2992,
     & 3311,    3372,   3364,   3216,   3090,   3025,   2996,
     & 2966,    2948,   2972,   2986,   2965,   2912,   2796,
     &   35,    9011,   8101,   1096,   2839,   6736,  13317,
     &20827,   26672,  27018,  22557,  17850,  11686,   8629,
     & 8386,    7514,   5331,   3986,   3658,   3642,   3478,
     & 3239,    2928,   2551,   2174,   2162,   2157,   2160,
     & 2164,    2211,   2227,   2200,   2127,   2063,   1973,
     &   35,    9216,   8060,   1199,   2964,   6971,  14122,
     &22414,   28972,  29148,  25542,  13803,  10575,   6310,
     & 4825,    4250,   3870,   3880,   3939,   4005,   4006,
     & 3905,    3743,   3714,   3696,   3682,   3697,   3721,
     & 3755,    3751,   3725,   3709,   3663,   3569,   3395,
     &   35,    9420,   8014,   1038,   2625,   7002,  17078,
     &28580,   38218,  28555,  23784,  14905,  11120,   8605,
     & 6453,    4668,   4644,   4592,   4364,   4336,   4353,
     & 4256,    3923,   3552,   3377,   3348,   3337,   3353,
     & 3360,    3388,   3398,   3367,   3347,   3357,   3459,
     &   35,    9625,   7974,   1058,   2778,   8142,  18112,
     &29158,   38589,  31886,  27614,  15379,  11756,   7404,
     & 5758,    4377,   3611,   3336,   3442,   3565,   3785,
     & 4003,    4132,   4345,   4772,   4936,   4992,   5054,
     & 5105,    5101,   5017,   4927,   4705,   4488,   3563,
     &   34,    9830,   7929,   1050,   2852,   7987,  21384,
     &34239,   43063,  31320,  26541,  14358,  11096,   9236,
     & 5255,    5474,   5489,   5449,   5006,   4596,   4461,
     & 4138,    3589,   3142,   3382,   4136,   4720,   4771,
     & 4773,    4900,   4989,   4978,   4896,   4809,     34,
     &10035,    7880,   1018,   3042,   8752,  26094,  39580,
     &47541,   33710,  26599,  13861,  10928,   7420,   4878,
     & 5170,    4977,   4500,   4616,   5115,   6316,   6277,
     & 5592,    5176,   5024,   4955,   4931,   4938,   5003,
     & 5037,    5092,   5157,   5148,   5274,     34,  10240,
     & 7822,    1097,   3528,  11320,  30959,  45133,  53296,
     &39292,   32724,  20426,  14875,  10911,   8529,   5325,
     & 4152,    3082,   3242,   3447,   3667,   3830,   4060,
     & 4536,    4778,   4570,   4412,   4498,   4488,   4240,
     & 4153,    4479,   5171,    839,     34,  10444,   7781,
     &  968,    3330,  11732,  28699,  48053,  61860,  38808,
     &34666,   15336,  12283,   8061,   6482,   6524,   6397,
     & 6063,    5917,   5631,   4974,   4500,   4625,   5421,
     & 5630,    5729,   5650,   5422,   5448,   5637,   5754,
     & 5860,    6570,   1102,     34,  10649,   7726,    949,
     & 3800,   15914,  44698,  55677,  51712,  41313,  22846,
     &15318,   12262,   9180,   9015,   8739,   8014,   5932,
     & 4976,    4967,   5779,   6215,   6448,   6513,   6615,
     & 6554,    6502,   6480,   6498,   6628,   6815,   7346,
     & 3383,     628,     34,  10854,   7702,    967,   3603,
     &14439,   37488,  56986,  69984,  48191,  29525,  18211,
     &11889,    7308,   8061,   8707,   8031,   6669,   6262,
     & 6167,    6127,   6021,   6048,   6238,   6304,   6275,
     & 6251,    6136,   6125,   6194,   6358,   6602,   7457  /
      DATA (MAPPMIN(I), I= 1471, 1960) /
     & 2562,      33,  11059,   7646,   1065,   4282,  17494,
     &43181,   59981,  64441,  49349,  28609,  18246,  13018,
     & 6982,    6107,   7135,   7927,   7979,   7986,   7816,
     & 7772,    7717,   7981,   8334,   8423,   8332,   7992,
     & 7709,    7722,   7903,   8205,   8819,   6261,     33,
     &11264,    7605,    973,   3803,  17998,  46677,  62123,
     &61152,   47671,  35256,  27396,  19708,  13998,  11838,
     & 7860,    7709,   7271,   7088,   7038,   6993,   6975,
     & 7063,    7193,   7321,   7394,   7338,   6942,   6839,
     & 6984,    7192,   7622,   8761,     33,  11468,   7548,
     &  942,    3855,  19292,  44027,  64792,  78143,  55940,
     &35263,   21597,  16626,  13239,   9941,   7579,   7709,
     & 8131,    8605,   8456,   7487,   7011,   7228,   7701,
     & 8007,    8637,   8777,   8512,   8557,   8534,   8790,
     & 9390,   10623,     33,  11673,   7506,    947,   3878,
     &19826,   46235,  65269,  75468,  53427,  46865,  23871,
     &19784,   16734,  10687,   8942,   7088,   6477,   6806,
     & 7200,    7817,   9634,   9854,  10034,  10061,  10011,
     & 9877,    9425,   9649,  10030,  10162,  10325,  11420,
     &   33,   11878,   7460,    907,   3666,  19982,  45897,
     &66887,   80849,  56741,  50996,  23345,  18925,  12802,
     &10291,    9864,   8434,   7716,   8141,   9632,  10005,
     &10160,   10223,  10100,  10126,   9977,   9974,   9772,
     & 9970,   10363,  10692,  10949,  13084,     32,  12083,
     & 7412,     811,   3434,  17320,  42273,  67189,  84315,
     &55937,   48618,  27688,  23055,  19461,  16287,  13763,
     &12860,   12357,  11523,  10926,  10808,  10812,  11237,
     &11275,   11269,  11159,  10811,  10811,  11011,  11278,
     &11900,   13908,     32,  12288,   7377,    828,   3421,
     &18339,   42434,  68271,  86460,  57767,  47630,  30906,
     &22593,   17972,  14424,  12692,  13367,  13302,  12054,
     &11412,   11977,  12466,  12773,  13205,  13097,  12909,
     &12830,   12615,  12871,  13077,  13745,  15513,     32,
     &12492,    7298,    783,   3006,  12800,  41684,  68536,
     &86332,   60430,  51133,  33738,  23827,  18950,  14154,
     &12822,   14521,  14969,  14684,  14673,  14740,  14907,
     &15092,   15430,  15373,  15240,  14907,  14810,  15295,
     &15691,   16197,  19143,     32,  12697,   7264,    884,
     & 3067,   13701,  36703,  65354,  85331,  66156,  54399,
     &38935,   25315,  20801,  17818,  16832,  15661,  14372,
     &14753,   14899,  15089,  15249,  15555,  15778,  15866,
     &15722,   15568,  15074,  15197,  15275,  15974,  17333,
     &   32,   12902,   7218,    872,   2653,  12620,  32563,
     &61213,   80744,  74805,  59579,  41742,  25717,  21047,
     &16106,   17239,  18633,  19574,  19841,  20158,  20443,
     &20652,   20770,  20608,  20472,  20016,  20063,  19981,
     &20821,   22631,  16755,   2324,     31,  13107,   7166,
     &  847,    2232,  12119,  31211,  59073,  77927,  76545,
     &61683,   50501,  41778,  34428,  30660,  25422,  21215,
     &19711,   16970,  16424,  15505,  16446,  17930,  18467,
     &18070,   17821,  17406,  17638,  18178,  18809,  20450,
     &   31,   13312,   7134,    896,   2442,  11240,  29292,
     &56591,   76880,  77929,  63341,  55209,  31200,  24727,
     &22873,   24281,  25283,  25051,  24383,  24423,  24909,
     &25432,   25990,  26029,  25486,  25144,  25155,  25658,
     &26977,   30621,   4292,     31,  13516,   7063,    913,
     & 2335,    9874,  30730,  56329,  75941,  78564,  67183,
     &58077,   49459,  43291,  37201,  31822,  27507,  25464,
     &25615,   24834,  23578,  23705,  23609,  23880,  24027,
     &24368,   24596,  25015,  27160,  16167,   2999,      3,
     &20480,       0,      3,  32767,      0,   1841,     20,
     &    3,       0,      0,      3,   2299,      0,     11,
     & 2334,    1996,     17,     12,     13,     12,     12,
     &   10,       9,      9,     14,   2355,   2653,     31,
     &   24,      23,     21,     19,     13,     10,      6,
     &    4,       2,      2,     16,   2396,   3246,     63,
     &   45,      33,     19,     12,     10,      7,      6,
     &    6,       5,      4,      4,      4,     18,   2457,
     & 3657,     123,     96,     73,     42,     35,     28,
     &   23,      20,     16,     12,     10,      8,      7  /
      DATA (MAPPMIN(I), I= 1961, 2450) /
     &    5,       7,     22,   2662,   4672,    218,    207,
     &  189,     158,    131,    104,     79,     61,     47,
     &   34,      25,     21,     18,     16,     15,     10,
     &   10,      10,     12,     24,   2867,   5309,    208,
     &  252,     294,    292,    283,    233,    198,    117,
     &   88,      68,     53,     47,     32,     26,     25,
     &   25,      26,     23,     24,     26,     26,     26,
     & 3072,    5862,    229,    258,    286,    312,    316,
     &  332,     328,    300,    257,    210,    138,    108,
     &   78,      58,     44,     36,     33,     32,     31,
     &   28,      29,     30,     32,     28,   3276,   6178,
     &  287,     326,    358,    380,    406,    403,    383,
     &  364,     346,    313,    248,    223,    185,    145,
     &  115,      84,     68,     54,     41,     34,     33,
     &   29,      31,     35,     49,     29,   3481,   6571,
     &  296,     337,    391,    450,    481,    460,    486,
     &  486,     447,    393,    342,    335,    281,    242,
     &  213,     180,    143,    122,     98,     76,     60,
     &   50,      36,     33,     37,     47,     30,   3686,
     & 6844,     314,    363,    416,    478,    523,    548,
     &  594,     616,    575,    546,    517,    453,    385,
     &  336,     299,    267,    229,    183,    164,    104,
     &   83,      62,     61,     57,     57,     65,     66,
     &   31,    3891,   7090,    325,    387,    460,    556,
     &  634,     721,    812,    921,    894,    800,    683,
     &  589,     493,    413,    322,    253,    191,    152,
     &  124,     109,     93,     78,     67,     66,     64,
     &   63,      72,     72,     32,   4096,   7354,    343,
     &  411,     488,    588,    721,    860,   1012,   1160,
     & 1293,    1281,   1158,    989,    817,    522,    406,
     &  343,     193,    138,     86,     56,     31,     19,
     &   18,      19,     18,     15,     18,     18,     19,
     &   33,    4300,   7570,    372,    452,    551,    671,
     &  809,    1021,   1339,   1591,   1623,   1456,   1281,
     & 1088,     817,    537,    334,    221,    117,     99,
     &   88,      87,     85,     87,     88,     88,     87,
     &   89,      90,     90,     96,    110,     34,   4505,
     & 7793,     379,    455,    570,    731,    977,   1229,
     & 1552,    1777,   1865,   1884,   1637,   1410,    860,
     &  603,     460,    246,    207,    199,    207,    234,
     &  201,     177,     94,     74,     70,     72,     73,
     &   74,      78,     87,    111,     35,   4710,   7966,
     &  404,     488,    613,    831,   1185,   1531,   1879,
     & 2083,    2078,   1887,   1656,   1425,   1083,    797,
     &  641,     531,    472,    358,    264,    205,    146,
     &  108,     112,    115,    118,    121,    120,    119,
     &  114,     115,    122,    134,     35,   4915,   8093,
     &  426,     545,    746,   1062,   1465,   1851,   2189,
     & 2352,    2467,   2242,   1984,   1710,   1282,    872,
     &  630,     348,    262,    232,    215,    210,    205,
     &  205,     210,    222,    231,    226,    218,    212,
     &  209,     216,    223,    233,     36,   5120,   8218,
     &  448,     595,    877,   1252,   1683,   2106,   2481,
     & 2740,    2867,   2885,   2628,   2264,   1948,   1271,
     &  913,     692,    436,    257,    175,    153,    129,
     &  125,     128,    128,    130,    131,    131,    129,
     &  124,     124,    127,    127,    126,     36,   5324,
     & 8322,     478,    645,    949,   1349,   1886,   2455,
     & 2881,    3392,   3590,   3143,   2697,   2325,   1595,
     & 1206,     893,    478,    393,    253,    248,    239,
     &  246,     250,    260,    269,    262,    255,    252,
     &  248,     252,    255,    260,    267,    277,     36,
     & 5529,    8423,    502,    675,   1010,   1510,   2121,
     & 2851,    3463,   4113,   3560,   3364,   3243,   2996,
     & 2162,    1530,   1177,    722,    566,    487,    424,
     &  367,     278,    252,    245,    253,    267,    262,
     &  253,     252,    253,    255,    252,    250,    252,
     &   37,    5734,   8507,    511,    703,   1047,   1615,
     & 2313,    3200,   4082,   5038,   4357,   4022,   3562,
     & 2763,    2360,   1835,   1542,   1350,   1196,    816  /
      DATA (MAPPMIN(I), I= 2451, 2940) /
     &  627,     538,    303,    303,    291,    259,    222,
     &  212,     209,    203,    194,    190,    195,    197,
     &  200,     187,     37,   5939,   8572,    533,    728,
     & 1124,    1800,   2678,   3732,   4900,   5751,   5401,
     & 4487,    3657,   2377,   1846,   1480,   1313,   1196,
     &  849,     674,    544,    460,    433,    434,    448,
     &  470,     468,    453,    446,    445,    446,    446,
     &  447,     450,    448,    435,     37,   6144,   8612,
     &  558,     774,   1201,   2043,   3199,   4581,   6000,
     & 6729,    6150,   5230,   4177,   3435,   2699,   2119,
     & 1659,    1148,   1024,    808,    647,    476,    444,
     &  353,     311,    308,    306,    306,    309,    321,
     &  330,     322,    317,    317,    315,    307,     37,
     & 6348,    8651,    567,    797,   1288,   2279,   3628,
     & 5301,    7024,   8013,   6774,   5598,   4258,   2803,
     & 2257,    1862,   1452,   1320,   1326,   1278,   1009,
     &  808,     651,    712,    689,    593,    508,    427,
     &  364,     366,    371,    377,    378,    379,    367,
     &  350,      37,   6553,   8664,    568,    851,   1389,
     & 2557,    4452,   6355,   7759,   8490,   6640,   5263,
     & 4350,    3194,   2883,   2465,   2113,   1914,   1751,
     & 1610,    1507,   1383,   1114,    956,    888,    855,
     &  758,     479,    360,    241,    254,    257,    255,
     &  256,     261,    255,     37,   6758,   8629,    627,
     &  970,    1672,   2901,   5100,   7644,   9516,   9425,
     & 8151,    6673,   4935,   3594,   2152,   2092,   2000,
     & 1879,    1764,   1461,   1356,   1258,   1094,    972,
     &  876,     759,    543,    600,    662,    500,    393,
     &  265,     245,    248,    247,    243,     37,   6963,
     & 8587,     692,   1088,   1992,   3453,   5800,   8838,
     &11312,   10008,   9045,   8377,   4344,   3557,   2187,
     & 1751,    1627,   1330,   1087,    885,    832,    748,
     &  674,     665,    784,    829,    847,    853,    863,
     &  871,     877,    881,    872,    866,    849,    833,
     &   37,    7168,   8524,    715,   1232,   2275,   3899,
     & 6702,   10390,  13224,  11731,   9835,   7850,   5081,
     & 4068,    3125,   2531,   2014,   1830,   1581,   1366,
     & 1165,    1009,   1110,   1086,    813,    723,    676,
     &  663,     658,    665,    675,    675,    667,    665,
     &  658,     624,     37,   7372,   8451,    773,   1367,
     & 2618,    4576,   7871,  11806,  13858,  12629,  10811,
     & 7403,    5101,   4004,   2979,   2674,   2530,   2053,
     & 1841,    1661,   1433,   1244,   1213,   1144,   1057,
     &  899,     841,    819,    814,    804,    811,    857,
     &  857,     857,    840,    792,     36,   7577,   8366,
     &  861,    1485,   2861,   5426,   8971,  13447,  18196,
     &14032,   12413,   6198,   4924,   3669,   2440,   2059,
     & 1959,    1818,   1724,   1688,   1612,   1467,   1341,
     & 1318,    1274,   1235,   1225,   1278,   1305,   1267,
     & 1238,    1268,   1264,   1204,   1139,     36,   7782,
     & 8258,     862,   1621,   3341,   5892,   9631,  14765,
     &19120,   14600,  12656,   6845,   5547,   4497,   3150,
     & 2854,    2905,   2883,   2817,   2688,   2447,   2071,
     & 1730,    1483,   1247,   1149,   1112,   1059,   1040,
     & 1063,    1097,   1103,   1101,   1096,   1039,     35,
     & 7987,    8189,    892,   1900,   3740,   6765,  11433,
     &16464,   19962,  15739,  13022,   8269,   6788,   5572,
     & 4483,    3716,   2840,   2311,   2091,   1984,   1747,
     & 1434,    1275,   1107,   1102,   1230,   1434,   1454,
     & 1433,    1429,   1478,   1481,   1482,   1464,     35,
     & 8192,    8121,    893,   1662,   3598,   7085,  11657,
     &16786,   20939,  17674,  15079,  10598,   7683,   6093,
     & 4489,    3115,   3248,   3309,   3329,   3165,   2798,
     & 2588,    2236,   1870,   1297,   1277,   1240,   1259,
     & 1314,    1331,   1373,   1314,   1305,   1263,     35,
     & 8396,    8046,    949,   1822,   4004,   8000,  12815,
     &17519,   21269,  19367,  17952,   9727,   7601,   5913,
     & 4644,    3782,   3391,   3375,   3446,   3450,   3383,
     & 3109,    2671,   2240,   1930,   1876,   1848,   1879,
     & 1912,    1966,   1972,   1891,   1780,   1612,     35  /
      DATA (MAPPMIN(I), I= 2941, 3430) /
     & 8601,    7974,    909,   1760,   4040,   8630,  14664,
     &20515,   24478,  21433,  18823,  11208,   8233,   5687,
     & 3540,    3378,   3348,   3101,   3036,   2968,   2768,
     & 2624,    2540,   2486,   2502,   2548,   2613,   2644,
     & 2580,    2511,   2496,   2451,   2307,   2033,     34,
     & 8806,    7902,    946,   1850,   4436,   9078,  15439,
     &22575,   27647,  22347,  16858,  10728,  10253,   3840,
     & 3736,    3182,   2651,   2470,   2718,   3189,   3482,
     & 3741,    3661,   3257,   3049,   2929,   2880,   2977,
     & 3024,    3006,   3015,   3039,   2989,     34,   9011,
     & 7827,     893,   2088,   4736,   9558,  15551,  21121,
     &24982,   26651,  22471,  16142,  10776,   9585,   8246,
     & 6922,    4064,   3745,   3716,   3543,   3316,   3001,
     & 2625,    2245,   2217,   2160,   2165,   2143,   2147,
     & 2180,    2202,   2213,   2103,     34,   9216,   7787,
     &  969,    2107,   4394,   9918,  16365,  22139,  28447,
     &27878,   26460,  11084,   9183,   5611,   4977,   3908,
     & 3907,    4064,   4355,   4686,   4484,   4063,   3759,
     & 3547,    3563,   3692,   3741,   3758,   3838,   3905,
     & 3749,    3542,   3466,     34,   9420,   7720,    887,
     & 1868,    5173,  10913,  19119,  28304,  35276,  26805,
     &21006,   14469,  10736,   8279,   6230,   4862,   5056,
     & 5146,    5163,   5185,   4901,   4182,   3603,   3372,
     & 3367,    3365,   3396,   3434,   3484,   3467,   3370,
     & 3259,    3202,     33,   9625,   7664,    833,   1782,
     & 5252,   10431,  18537,  28047,  35189,  30570,  25615,
     &16505,   11291,   7874,   6220,   4617,   3941,   3961,
     & 4097,    4264,   4447,   4609,   4886,   5173,   5166,
     & 4911,    4944,   5253,   5267,   5010,   4959,   4783,
     &   33,    9830,   7586,    809,   1782,   5289,  12249,
     &23281,   33366,  39547,  29082,  24363,  14495,  10840,
     & 8907,    5723,   6086,   6165,   6071,   5592,   5431,
     & 4704,    3818,   3428,   3810,   4571,   4732,   4810,
     & 4957,    4999,   4963,   4884,   4844,     33,  10035,
     & 7519,     842,   1923,   5289,  12332,  24004,  35165,
     &42545,   32331,  27205,  16801,  11950,   9101,   6117,
     & 6557,    5528,   5535,   5851,   6465,   6238,   5554,
     & 5119,    4816,   4764,   4870,   5013,   5145,   5175,
     & 5135,    5206,   5030,     33,  10240,   7449,    847,
     & 2149,    5927,  13285,  27861,  38919,  44550,  36595,
     &30908,   25054,  17307,  13158,   9905,   7109,   4811,
     & 3696,    2974,   3018,   3271,   3714,   3828,   3741,
     & 3948,    4071,   3945,   4183,   4416,   4491,   4365,
     & 4285,      32,  10444,   7363,    913,   2481,   6500,
     &15562,   27859,  42378,  54425,  37530,  33806,  15636,
     &12266,    8819,   5946,   5793,   5784,   5861,   6251,
     & 6438,    6028,   5301,   4757,   4736,   4882,   5287,
     & 5526,    5406,   6009,   5916,   5180,     32,  10649,
     & 7301,     869,   2290,   7873,  19873,  40272,  52849,
     &42336,   35009,  22993,  16587,  13010,  10166,   8113,
     & 8567,    8434,   8051,   6759,   5430,   5728,   6792,
     & 6807,    7206,   6976,   6556,   6616,   6484,   6640,
     & 6855,    6693,     32,  10854,   7249,    840,   2024,
     & 7399,   18463,  37618,  53738,  62446,  45182,  28634,
     &16794,   12581,   7275,   6331,   7164,   8492,   7984,
     & 7797,    6957,   6167,   6207,   6023,   5999,   6058,
     & 6361,    6082,   6139,   5709,   5864,   6608,     32,
     &11059,    7177,    983,   2665,   8552,  23797,  43003,
     &57900,   57193,  46483,  26874,  17786,  12822,   6715,
     & 5511,    5561,   6707,   7522,   8023,   7809,   7947,
     & 7609,    7882,   8281,   8195,   8021,   8227,   7350,
     & 7354,    7736,   7697,     31,  11264,   7123,    798,
     & 2023,    9118,  28050,  46908,  59170,  51113,  42329,
     &32758,   26764,  18817,  13107,  11586,   8984,   7765,
     & 7692,    7070,   6926,   6862,   7121,   6862,   7313,
     & 7223,    7169,   7064,   7133,   6529,   7041,     31,
     &11468,    7069,    816,   2218,   9145,  23576,  45698,
     &62901,   70282,  51752,  33441,  21171,  17100,  13787,
     & 9689,    6890,   6219,   6603,   7678,   9234,   8943,
     & 7392,    7545,   8746,   9303,   8955,   8919,   8665  /
      DATA (MAPPMIN(I), I= 3431, 3920) /
     & 8596,    8845,     31,  11673,   6985,    802,   2165,
     &10532,   30534,  50991,  64779,  61210,  47415,  40398,
     &22348,   18785,  15802,  12064,  10115,   7791,   6735,
     & 6658,    7032,   7604,   8852,   9547,   9180,   9627,
     & 9700,    9202,  10118,   9615,  10918,     31,  11878,
     & 6935,     786,   2158,  10637,  30692,  55830,  70648,
     &61262,   49479,  43215,  23275,  18317,  13253,  11368,
     &10653,    8577,   7821,   8762,  10707,  10591,  10145,
     &10013,    9352,   9163,  10316,  10097,  10816,  11137,
     &11150,      30,  12083,   6861,    781,   2085,   8396,
     &32957,   52876,  66387,  67951,  51881,  39716,  25957,
     &20368,   16678,  14465,  12938,  11539,  11738,  11258,
     &11286,   10851,  11017,  10691,  11580,  11117,  11673,
     &11007,   11578,  11159,     30,  12288,   6799,    777,
     & 2110,   10075,  29071,  56377,  72072,  69929,  52809,
     &40026,   28183,  21131,  16579,  12927,  10434,  12876,
     &14824,   12506,  11152,  12073,  12446,  12927,  13002,
     &13291,   12850,  13021,  12279,  13234,     30,  12492,
     & 6697,     770,   2530,   8512,  27223,  57189,  73132,
     &70744,   53750,  42295,  30148,  21479,  16879,  11707,
     &12612,   16502,  16769,  14420,  14744,  14444,  15125,
     &14757,   15252,  14182,  15472,  16062,  15976,  15639,
     &   30,   12697,   6661,    798,   2124,   8389,  28781,
     &61426,   78363,  68808,  53700,  41431,  27940,  22047,
     &18396,   16722,  16322,  16592,  16526,  15177,  13807,
     &15680,   16577,  15893,  14985,  15902,  14450,  15633,
     &15975,   17475,     29,  12902,   6577,    869,   2417,
     & 9169,   25418,  51350,  70191,  75936,  57538,  46416,
     &32032,   24399,  19990,  17923,  19133,  20442,  20406,
     &19557,   19364,  18710,  17371,  19273,  21410,  21267,
     &21752,   20135,  19495,     29,  13107,   6514,    755,
     & 2246,    9174,  25906,  54271,  71536,  76131,  56801,
     &45977,   35401,  33015,  31522,  30385,  24834,  20630,
     &19125,   17975,  17774,  18423,  18585,  18098,  17453,
     &16263,   17724,  18445,  18827,     29,  13312,   6448,
     &  863,    2427,  10327,  27125,  54926,  74155,  75530,
     &59260,   49164,  31744,  25283,  22775,  23179,  24323,
     &24960,   24483,  24390,  25180,  25726,  26282,  26240,
     &25519,   25507,  26115,  26368,  27263,     29,  13516,
     & 6400,     745,   1890,   7095,  25419,  45079,  65303,
     &79899,   68012,  60001,  52424,  45500,  38955,  34167,
     &30242,   25962,  23415,  24372,  24157,  23102,  22849,
     &22146,   24514,  26260,  24343,  22772,  26066,      3,
     &20480,       0,      3,  32767,      0,   1747,     40,
     &    3,       0,      0,      3,   2300,      0,     11,
     & 2334,    1981,     17,     12,     13,     12,     12,
     &   10,       9,      9,     14,   2355,   2642,     31,
     &   24,      23,     21,     19,     13,     10,      6,
     &    5,       1,      2,     16,   2396,   3240,     63,
     &   45,      33,     19,     12,     10,      7,      6,
     &    6,       5,      5,      3,      4,     18,   2457,
     & 3656,     122,     96,     73,     42,     34,     29,
     &   23,      20,     16,     12,     10,      8,      6,
     &    6,       7,     22,   2662,   4669,    217,    207,
     &  190,     158,    131,    104,     78,     61,     48,
     &   34,      24,     21,     18,     16,     16,      9,
     &   10,      10,     11,     24,   2867,   5306,    208,
     &  252,     294,    291,    285,    233,    197,    117,
     &   88,      67,     54,     47,     31,     26,     25,
     &   26,      25,     23,     24,     27,     26,     26,
     & 3072,    5857,    229,    257,    287,    313,    318,
     &  330,     329,    301,    257,    207,    140,    106,
     &   79,      57,     43,     36,     34,     31,     31,
     &   28,      29,     30,     30,     28,   3276,   6170,
     &  288,     326,    362,    378,    405,    403,    389,
     &  359,     348,    311,    251,    219,    186,    144,
     &  115,      83,     68,     53,     41,     34,     33,
     &   29,      32,     35,     47,     29,   3481,   6559,
     &  295,     339,    390,    454,    477,    467,    487,
     &  485,     448,    393,    344,    334,    280,    241  /
      DATA (MAPPMIN(I), I= 3921, 4410) /
     &  213,     178,    141,    123,     97,     75,     59,
     &   50,      36,     33,     37,     47,     30,   3686,
     & 6824,     313,    365,    416,    478,    523,    556,
     &  602,     621,    576,    548,    515,    450,    383,
     &  337,     297,    264,    227,    181,    161,    103,
     &   80,      63,     61,     56,     58,     66,     66,
     &   31,    3891,   7016,    348,    409,    484,    577,
     &  652,     743,    840,    909,    878,    783,    672,
     &  579,     488,    396,    315,    240,    175,    146,
     &  123,     105,     90,     74,     68,     64,     63,
     &   65,      75,     88,     32,   4096,   7264,    359,
     &  425,     503,    612,    749,    900,   1074,   1245,
     & 1332,    1250,   1130,    967,    701,    502,    390,
     &  312,     164,    138,     70,     50,     27,     21,
     &   15,      18,     18,     17,     17,     18,     20,
     &   33,    4300,   7470,    385,    456,    541,    671,
     &  881,    1105,   1440,   1659,   1587,   1434,   1252,
     & 1045,     708,    550,    269,    208,    104,     97,
     &   88,      86,     85,     87,     88,     88,     87,
     &   90,      90,     92,     97,    116,     33,   4505,
     & 7667,     384,    462,    563,    735,   1013,   1264,
     & 1582,    1794,   1858,   1867,   1598,   1349,    855,
     &  605,     461,    247,    214,    209,    216,    235,
     &  195,     172,     84,     73,     70,     70,     72,
     &   75,      79,     89,     34,   4710,   7821,    403,
     &  496,     642,    866,   1177,   1539,   1861,   2081,
     & 2083,    1867,   1626,   1405,   1069,    798,    647,
     &  545,     480,    349,    263,    200,    152,    111,
     &  115,     116,    115,    117,    120,    122,    114,
     &  115,     123,     35,   4915,   7939,    413,    536,
     &  749,    1022,   1382,   1764,   2132,   2395,   2570,
     & 2353,    2051,   1727,   1262,    856,    628,    341,
     &  262,     229,    211,    209,    204,    200,    202,
     &  219,     225,    222,    217,    211,    208,    215,
     &  222,     232,     35,   5120,   8054,    426,    565,
     &  829,    1141,   1582,   2088,   2434,   2703,   3000,
     & 2942,    2590,   2207,   1977,   1293,    996,    790,
     &  392,     315,    183,    163,    132,    124,    123,
     &  125,     128,    129,    130,    130,    129,    126,
     &  127,     126,     35,   5324,   8154,    456,    616,
     &  898,    1228,   1699,   2310,   2836,   3277,   3527,
     & 3241,    2876,   2506,   1751,   1237,    981,    493,
     &  398,     247,    244,    238,    245,    250,    256,
     &  261,     267,    267,    256,    246,    256,    261,
     &  259,     257,     36,   5529,   8242,    454,    624,
     &  957,    1391,   1985,   2643,   3215,   3777,   3758,
     & 3435,    3329,   3038,   2453,   1588,   1264,    877,
     &  587,     496,    441,    384,    319,    250,    248,
     &  244,     261,    273,    267,    261,    253,    251,
     &  253,     248,    245,     36,   5734,   8298,    482,
     &  661,    1010,   1501,   2147,   2975,   3917,   4606,
     & 4269,    4073,   3575,   2803,   2532,   1900,   1612,
     & 1475,    1327,   1064,    738,    561,    421,    298,
     &  295,     277,    240,    221,    206,    197,    193,
     &  196,     204,    208,    200,     36,   5939,   8357,
     &  469,     651,   1026,   1560,   2225,   3199,   4296,
     & 5081,    5184,   5025,   4185,   3319,   2015,   1667,
     & 1425,    1280,   1038,    695,    563,    475,    414,
     &  426,     441,    475,    484,    458,    443,    440,
     &  437,     435,    445,    456,    459,     36,   6144,
     & 8378,     486,    675,   1066,   1671,   2601,   3741,
     & 4924,    5834,   6432,   5814,   4874,   3720,   3058,
     & 2452,    1964,   1549,   1119,    924,    732,    592,
     &  464,     434,    372,    318,    319,    313,    305,
     &  305,     321,    324,    319,    319,    316,     36,
     & 6348,    8401,    500,    689,   1086,   1824,   2802,
     & 3976,    5384,   6617,   7000,   5968,   5001,   3943,
     & 2933,    2324,   1933,   1666,   1414,   1272,   1252,
     & 1034,     800,    646,    687,    704,    625,    522,
     &  479,     358,    363,    383,    387,    383,    380  /
      DATA (MAPPMIN(I), I= 4411, 4900) /
     &   36,    6553,   8384,    519,    729,   1176,   1960,
     & 3104,    4650,   6180,   7358,   7286,   6088,   5028,
     & 4191,    3055,   2713,   2316,   2055,   1958,   1828,
     & 1767,    1662,   1518,   1129,    991,    915,    875,
     &  755,     617,    373,    268,    253,    254,    254,
     &  257,      36,   6758,   8335,    557,    776,   1280,
     & 2304,    3632,   5290,   7149,   8340,   8130,   7246,
     & 6216,    5555,   3368,   2397,   2256,   2224,   1947,
     & 1704,    1643,   1429,   1374,   1163,   1007,    889,
     &  779,     605,    579,    649,    589,    404,    309,
     &  241,     242,     36,   6963,   8298,    592,    848,
     & 1415,    2376,   3737,   5849,   8348,  10073,   9390,
     & 8835,    7683,   5075,   3555,   2464,   1738,   1695,
     & 1454,    1158,    927,    801,    634,    643,    629,
     &  713,     823,    855,    879,    852,    850,    885,
     &  893,     858,    844,     36,   7168,   8214,    636,
     &  960,    1678,   2889,   4357,   6511,   9287,  11044,
     &11967,    9811,   7710,   5266,   4051,   3176,   2761,
     & 2129,    1953,   1597,   1417,   1221,   1082,   1084,
     & 1022,     769,    733,    692,    651,    630,    661,
     &  685,     702,    671,    662,     35,   7372,   8135,
     &  661,    1033,   1779,   2960,   4680,   7568,  10970,
     &13804,   11906,   9969,   7413,   5084,   4136,   3373,
     & 2808,    2657,   2137,   1832,   1685,   1447,   1293,
     & 1246,    1158,   1031,    880,    837,    850,    847,
     &  842,     841,    846,    869,     35,   7577,   8044,
     &  720,    1172,   1942,   3371,   5529,   8685,  12837,
     &16675,   13880,  12144,   6599,   4959,   3534,   2891,
     & 2150,    2075,   1877,   1749,   1691,   1595,   1462,
     & 1371,    1345,   1289,   1222,   1224,   1255,   1260,
     & 1247,    1253,   1256,   1232,     34,   7782,   7929,
     &  684,    1168,   2082,   3496,   5891,  10634,  15383,
     &17134,   13375,  11212,   7605,   5775,   4627,   3525,
     & 3122,    3075,   2893,   2653,   2745,   2495,   1991,
     & 1616,    1384,   1176,   1117,   1076,   1085,   1099,
     & 1083,    1074,   1083,     34,   7987,   7853,    682,
     & 1212,    2319,   4079,   6969,  11080,  15678,  19308,
     &14880,   12531,   8306,   6912,   5826,   4839,   3865,
     & 3128,    2313,   2143,   1995,   1703,   1367,   1216,
     & 1124,    1114,   1270,   1464,   1486,   1452,   1476,
     & 1511,    1499,     34,   8192,   7762,    691,   1166,
     & 2101,    4028,   7073,  11837,  17108,  20897,  16271,
     &13495,   10663,   8079,   6209,   4851,   3601,   3639,
     & 3674,    3622,   3160,   2787,   2417,   2010,   1487,
     & 1297,    1246,   1255,   1313,   1304,   1274,   1281,
     & 1310,      33,   8396,   7665,    726,   1209,   2103,
     & 3957,    7744,  13018,  17517,  20539,  19122,  16144,
     &11954,    7881,   6310,   4929,   4193,   3925,   4044,
     & 3946,    3476,   3007,   2568,   2166,   1888,   1831,
     & 1850,    1944,   1968,   1936,   1897,   1875,     33,
     & 8601,    7555,    715,   1243,   2257,   4335,   8834,
     &14414,   18950,  22150,  22242,  18072,  14230,   8102,
     & 6043,    3441,   3585,   3603,   3535,   3427,   2995,
     & 2666,    2565,   2588,   2552,   2502,   2522,   2561,
     & 2568,    2603,   2582,   2448,     33,   8806,   7446,
     &  724,    1237,   2270,   4505,   9269,  15689,  20990,
     &23403,   22992,  17855,  13080,  10640,   4181,   4117,
     & 3697,    3325,   3474,   3535,   3907,   4343,   3606,
     & 3120,    2920,   2864,   2943,   3045,   3037,   3007,
     & 3010,    3017,     32,   9011,   7336,    669,   1205,
     & 2524,    5361,  10336,  15446,  19712,  24382,  27314,
     &21979,   18711,  11126,  10066,   8103,   5789,   4026,
     & 4084,    3792,   3355,   3025,   2621,   2256,   2225,
     & 2190,    2132,   2045,   2166,   2294,   2243,     32,
     & 9216,    7281,    652,   1330,   2495,   4785,   9922,
     &15101,   20209,  25869,  29798,  21400,  18467,   9554,
     & 6663,    5213,   4329,   4704,   5238,   5545,   4910,
     & 4009,    3695,   3588,   3707,   3745,   3774,   3795,
     & 3836,    3779,   3675,     32,   9420,   7172,    648,
     & 1100,    2163,   4914,   9714,  17103,  26120,  32965  /
      DATA (MAPPMIN(I), I= 4901, 5390) /
     &26529,   21928,  16975,  12304,   9400,   7517,   6355,
     & 6261,    6281,   6145,   5807,   4281,   3658,   3420,
     & 3412,    3371,   3350,   3431,   3513,   3412,   3299,
     &   31,    9625,   7065,    672,   1132,   2128,   4687,
     &10957,   17186,  23417,  29623,  33011,  25639,  20833,
     &12871,    9722,   7032,   5483,   4778,   5129,   5358,
     & 5385,    5545,   5415,   5123,   5054,   5128,   5158,
     & 5143,    5085,   4955,     31,   9830,   6962,    631,
     & 1101,    2226,   5505,  11379,  20400,  27863,  32131,
     &33496,   25980,  19618,  12296,   9749,   7726,   6978,
     & 7083,    7086,   6938,   6282,   4735,   4723,   4655,
     & 4622,    4641,   4701,   4924,   5164,   5256,     30,
     &10035,    6869,    637,   1158,   2411,   5192,  10356,
     &19347,   31365,  39256,  36683,  28179,  21710,  12515,
     &10052,    7476,   7094,   6968,   7105,   7070,   6688,
     & 5477,    4954,   5026,   4962,   4812,   4900,   4918,
     & 5176,      30,  10240,   6742,    583,   1152,   2285,
     & 6259,   11563,  20795,  33920,  40679,  37693,  30520,
     &25570,   18554,  14454,  11244,   8857,   7159,   5151,
     & 3848,    2752,   2742,   2720,   2734,   2817,   2945,
     & 3005,    3054,   3466,     29,  10444,   6647,    595,
     & 1233,    2830,   6598,  13936,  19993,  27694,  38241,
     &46214,   32954,  28378,  16005,  11367,   8567,   6233,
     & 5044,    4694,   4737,   4935,   5263,   5576,   5186,
     & 5200,    5235,   5320,   5736,     29,  10649,   6562,
     &  577,    1020,   2338,   7960,  16337,  25748,  38181,
     &45510,   35194,  28459,  19134,  15482,  11696,   9194,
     & 7185,    6346,   7131,   7027,   6688,   6444,   5784,
     & 4762,    5271,   5804,   7532,   8539,     29,  10854,
     & 6507,     575,   1011,   2010,   6273,  13434,  24141,
     &35664,   45768,  52681,  36664,  29050,  17037,  12611,
     & 9089,    5564,   4694,   4789,   5955,   7825,   7211,
     & 7144,    6863,   6469,   6128,   6300,   7173,     29,
     &11059,    6415,    593,   1058,   2261,   6570,  14624,
     &26767,   40713,  50212,  51243,  37557,  28401,  15931,
     &12570,    9502,   5386,   4229,   3511,   4441,   7044,
     & 7645,    8458,   8182,   7809,   7581,   7989,   9594,
     &   28,   11264,   6312,    551,   1169,   2819,   8168,
     &18523,   32619,  43526,  48967,  44626,  35790,  28398,
     &25372,   15571,  11893,   9582,   9086,   8327,   5997,
     & 5955,    6345,   6922,   6836,   7097,   7082,   8164,
     &   28,   11468,   6254,    522,    989,   2063,   7902,
     &17305,   30520,  47513,  57339,  56910,  39981,  31295,
     &18539,   15808,  13063,  10420,   8466,   6841,   5954,
     & 5985,    5462,   6160,   7607,   9195,   9343,  10548,
     &   28,   11673,   6174,    532,   1148,   2816,   8536,
     &19564,   37383,  52419,  60140,  48437,  37415,  32106,
     &17427,   14619,  14241,  12927,  11105,   8441,   8585,
     & 6784,    5863,   6320,   6790,   7827,   9776,  12791,
     &   27,   11878,   6118,    495,   1046,   2604,   7110,
     &16073,   34702,  53822,  61698,  57052,  44427,  36891,
     &22776,   15991,  11714,   9742,   8613,   8573,   8523,
     & 9325,   10671,  10232,  10172,  10805,  11651,     27,
     &12083,    6024,    516,    981,   2537,   6364,  17924,
     &32643,   47934,  57968,  60744,  42441,  37721,  19917,
     &19844,   17905,  15453,  13517,  12069,  10731,  11580,
     &12610,   12280,  11155,  11214,  11731,     27,  12288,
     & 5958,     518,   1126,   2798,   9189,  19144,  33227,
     &51783,   61997,  59549,  45571,  37066,  29545,  20421,
     &16737,   13552,  10859,   9268,  11698,  13140,  11890,
     & 9964,   10136,  11164,  13288,     26,  12492,   5856,
     &  546,     994,   2736,   8011,  23367,  41794,  53149,
     &57883,   53521,  44890,  37578,  31088,  22362,  17355,
     &13364,   10932,  13957,  16525,  15436,  13308,  14259,
     &15875,   17840,     26,  12697,   5780,    607,   1329,
     & 3264,    7946,  18952,  38041,  55682,  65096,  55790,
     &45876,   37638,  33674,  21738,  17694,  14669,  13995,
     &13823,   12517,  12277,  14018,  15211,  16249,  18523,
     &   26,   12902,   5723,    558,   1326,   3407,   8387,
     &16907,   29155,  46721,  60510,  66896,  50971,  41862  /
      DATA (MAPPMIN(I), I= 5391, 5880) /
     &31743,   23214,  21002,  18420,  17813,  18426,  18440,
     &18144,   20366,  20695,  22654,  24027,     25,  13107,
     & 5632,     567,   1293,   3332,   9680,  18302,  31061,
     &47744,   59548,  61630,  49375,  41346,  31700,  28642,
     &26886,   25110,  25405,  23765,  20695,  18811,  18947,
     &18684,   17407,     25,  13312,   5590,    552,   1350,
     & 3732,    9868,  18713,  32323,  49373,  59896,  57954,
     &50981,   43085,  36275,  24203,  22061,  24868,  24655,
     &25465,   24847,  26000,  26891,  25776,  28183,     25,
     &13516,    5513,    575,   1314,   3366,   8288,  20569,
     &31190,   43393,  56530,  65647,  56364,  46750,  36950,
     &30241,   30388,  31832,  40346,  34160,  29860,  25644,
     &24299,   24934,  26238,      3,  20480,      0,      3,
     &32767,       0,   1657,     60,      3,      0,      0,
     &    3,    2301,      0,     11,   2334,   1970,     17,
     &   12,      13,     12,     12,     10,      9,      9,
     &   14,    2355,   2633,     31,     24,     23,     21,
     &   19,      13,     10,      6,      5,      1,      1,
     &   16,    2396,   3234,     63,     45,     33,     19,
     &   12,      10,      7,      6,      6,      5,      4,
     &    4,       4,     18,   2457,   3655,    122,     93,
     &   75,      42,     35,     28,     23,     20,     16,
     &   12,      10,      8,      6,      6,      6,     22,
     & 2662,    4667,    217,    204,    194,    157,    130,
     &  103,      80,     60,     48,     34,     24,     21,
     &   18,      16,     16,      9,     10,     10,     10,
     &   24,    2867,   5300,    210,    253,    297,    287,
     &  288,     231,    196,    115,     88,     67,     54,
     &   47,      31,     26,     25,     25,     25,     23,
     &   25,      26,     26,     26,   3072,   5850,    230,
     &  257,     289,    312,    316,    332,    333,    300,
     &  255,     205,    141,    105,     78,     57,     43,
     &   36,      33,     32,     31,     28,     29,     30,
     &   30,      28,   3276,   6166,    286,    326,    363,
     &  376,     404,    403,    391,    359,    351,    311,
     &  250,     217,    190,    143,    114,     82,     69,
     &   52,      42,     33,     33,     30,     31,     35,
     &   35,      29,   3481,   6546,    295,    338,    390,
     &  456,     483,    465,    489,    486,    447,    393,
     &  344,     334,    279,    241,    213,    177,    140,
     &  123,      95,     74,     59,     49,     36,     34,
     &   37,      37,     30,   3686,   6802,    315,    364,
     &  419,     475,    524,    562,    616,    623,    578,
     &  548,     517,    445,    380,    334,    297,    263,
     &  225,     178,    158,    100,     80,     62,     61,
     &   56,      59,     66,     66,     31,   3891,   6979,
     &  348,     409,    482,    580,    664,    752,    853,
     &  894,     870,    781,    676,    583,    491,    397,
     &  314,     241,    166,    143,    123,    105,     89,
     &   72,      67,     64,     62,     66,     76,     83,
     &   32,    4096,   7202,    359,    432,    509,    621,
     &  764,     926,   1097,   1301,   1370,   1245,   1089,
     &  966,     639,    490,    383,    282,    160,    136,
     &   61,      46,     25,     21,     15,     19,     16,
     &   20,      18,     18,     19,     32,   4300,   7380,
     &  392,     470,    542,    682,    931,   1190,   1508,
     & 1697,    1549,   1417,   1233,   1028,    605,    563,
     &  219,     179,    102,     93,     88,     86,     86,
     &   87,      87,     88,     88,     90,     90,     92,
     &  101,      33,   4505,   7559,    387,    470,    569,
     &  749,    1043,   1298,   1619,   1814,   1870,   1876,
     & 1591,    1314,    798,    592,    429,    249,    214,
     &  207,     214,    234,    189,    160,     86,     70,
     &   70,      69,     72,     75,     80,     89,     34,
     & 4710,    7700,    397,    492,    654,    873,   1163,
     & 1535,    1891,   2106,   2103,   1835,   1604,   1400,
     & 1057,     797,    651,    553,    487,    346,    260,
     &  200,     150,    111,    115,    113,    115,    117,
     &  121,     120,    113,    113,    124,     34,   4915,
     & 7817,     399,    520,    723,    983,   1310,   1753  /
      DATA (MAPPMIN(I), I=  5881, 6370) /
     & 2161,    2415,   2598,   2410,   2095,   1750,   1257,
     &  843,     639,    332,    268,    231,    210,    203,
     &  198,     198,    202,    212,    217,    218,    219,
     &  214,     212,    214,    220,     34,   5120,   7922,
     &  408,     546,    790,   1081,   1511,   2015,   2419,
     & 2669,    3047,   3001,   2565,   2149,   1918,   1399,
     & 1052,     775,    477,    363,    211,    167,    142,
     &  127,     125,    125,    126,    129,    132,    131,
     &  126,     123,    128,     35,   5324,   8005,    432,
     &  582,     833,   1136,   1616,   2259,   2782,   3208,
     & 3450,    3327,   2980,   2574,   1866,   1258,   1050,
     &  531,     391,    276,    248,    241,    244,    244,
     &  256,     267,    270,    264,    256,    248,    253,
     &  256,     256,    260,     35,   5529,   8079,    437,
     &  583,     872,   1241,   1823,   2508,   3037,   3602,
     & 3975,    3511,   3466,   3100,   2581,   1619,   1283,
     &  943,     666,    517,    458,    393,    333,    248,
     &  248,     251,    261,    274,    264,    255,    253,
     &  252,     251,    247,     35,   5734,   8120,    451,
     &  616,     928,   1358,   2010,   2740,   3373,   4038,
     & 4422,    4209,   3714,   3177,   2592,   2151,   1710,
     & 1529,    1400,   1248,    833,    586,    484,    292,
     &  292,     284,    250,    220,    209,    203,    202,
     &  201,     202,    200,     35,   5939,   8166,    434,
     &  590,     896,   1358,   2039,   2892,   3808,   4653,
     & 5095,    5213,   4369,   3632,   2476,   1783,   1496,
     & 1321,    1167,    836,    635,    511,    431,    437,
     &  450,     460,    462,    456,    453,    439,    425,
     &  437,     448,    456,     35,   6144,   8172,    443,
     &  613,     956,   1406,   2186,   3270,   4200,   4906,
     & 5635,    6031,   5108,   4300,   3477,   2951,   2416,
     & 1902,    1463,   1075,    856,    701,    581,    485,
     &  422,     310,    319,    315,    315,    315,    314,
     &  319,     321,    312,     35,   6348,   8158,    467,
     &  630,     965,   1535,   2402,   3560,   4758,   5862,
     & 6883,    5815,   5033,   4275,   3462,   2587,   2210,
     & 1875,    1590,   1363,   1233,   1163,    967,    780,
     &  662,     745,    740,    637,    516,    454,    369,
     &  374,     376,    384,     35,   6553,   8134,    472,
     &  634,     982,   1572,   2537,   3766,   5143,   6421,
     & 7133,    6417,   5364,   4185,   3979,   3124,   2798,
     & 2384,    2101,   1941,   1828,   1749,   1599,   1406,
     & 1049,     934,    892,    859,    720,    545,    353,
     &  251,     250,    243,     35,   6758,   8076,    494,
     &  673,    1062,   1683,   2769,   4132,   5565,   6921,
     & 8058,    7400,   6683,   5884,   5410,   3099,   2491,
     & 2264,    2181,   1937,   1719,   1683,   1619,   1418,
     & 1129,     980,    861,    738,    572,    623,    666,
     &  524,     369,    265,     35,   6963,   8014,    519,
     &  720,    1175,   2004,   3122,   4531,   6786,   8972,
     & 8833,    8897,   8207,   7248,   4298,   3142,   2184,
     & 1780,    1732,   1392,   1077,    853,    734,    640,
     &  632,     602,    755,    850,    847,    860,    873,
     &  894,     911,    888,     34,   7168,   7934,    548,
     &  731,    1216,   2189,   3377,   4756,   7082,   9853,
     &10912,   11606,   9235,   6759,   4877,   3753,   3059,
     & 2662,    2138,   1964,   1583,   1377,   1161,   1053,
     & 1040,     941,    732,    716,    695,    639,    651,
     &  694,     708,     34,   7372,   7838,    549,    744,
     & 1302,    2380,   3749,   5573,   8220,  11210,  12920,
     &11532,    9567,   6741,   4662,   3856,   3239,   2852,
     & 2608,    2048,   1869,   1647,   1396,   1251,   1163,
     & 1057,     956,    843,    856,    862,    857,    864,
     &  848,      34,   7577,   7736,    604,    905,   1395,
     & 2467,    4166,   6275,   9731,  13695,  16717,  12416,
     &10530,    6377,   4624,   3628,   2571,   2257,   2100,
     & 1809,    1776,   1658,   1442,   1312,   1331,   1321,
     & 1244,    1196,   1273,   1299,   1237,   1213,   1301,
     &   33,    7782,   7608,    567,    860,   1576,   2793,
     & 4340,    6755,  10578,  14463,  17058,  13071,  10830  /
      DATA (MAPPMIN(I), I=  6371, 6860) /
     & 8050,    5562,   4457,   3202,   3114,   3079,   2931,
     & 2826,    2642,   2192,   1772,   1448,   1224,   1128,
     & 1070,    1074,   1087,   1073,   1067,     33,   7987,
     & 7523,     557,    878,   1680,   2907,   4828,   7814,
     &12077,   16238,  18660,  13733,  11188,   8037,   7139,
     & 5736,    4711,   3797,   2975,   2282,   2163,   1869,
     & 1555,    1278,   1148,   1023,   1173,   1425,   1518,
     & 1476,    1471,   1478,     32,   8192,   7391,    590,
     &  980,    1693,   2991,   5157,   8283,  12873,  17109,
     &18991,   15146,  12872,  10792,   8759,   6149,   4591,
     & 3927,    3993,   3887,   3506,   2870,   2430,   1958,
     & 1449,    1287,   1257,   1274,   1287,   1292,   1305,
     &   32,    8396,   7281,    543,    891,   1520,   2903,
     & 5143,    8258,  13336,  17826,  19772,  18435,  15242,
     &12815,    8118,   6238,   5110,   4685,   4583,   4314,
     & 3645,    2912,   2482,   2109,   1909,   1851,   1884,
     & 1928,    1927,   1931,   1914,     31,   8601,   7131,
     &  532,     913,   1616,   3018,   5419,   9207,  14456,
     &19329,   22536,  23381,  16778,  13874,   7750,   4716,
     & 3579,    3854,   4079,   3935,   3234,   2625,   2585,
     & 2586,    2572,   2550,   2561,   2513,   2596,   2641,
     &   31,    8806,   6996,    572,    921,   1501,   2980,
     & 5600,    9375,  14677,  20369,  24389,  24520,  17408,
     &13444,    9017,   4857,   4659,   4261,   4147,   4039,
     & 4318,    4344,   3254,   3086,   2869,   2900,   2924,
     & 2944,    3095,   3087,     30,   9011,   6848,    530,
     &  818,    1760,   3185,   6233,  10485,  14794,  20139,
     &26319,   29906,  21873,  17618,  12742,   9031,   7680,
     & 4497,    4400,   3762,   3281,   2890,   2469,   2220,
     & 2200,    2147,   2057,   2016,   2237,     30,   9216,
     & 6747,     523,   1023,   1851,   2894,   5589,   9671,
     &15792,   23617,  29287,  30029,  21959,  13311,   8274,
     & 6171,    5522,   5686,   6305,   6033,   4744,   4050,
     & 3610,    3657,   3831,   3769,   3891,   3915,   3735,
     &   29,    9420,   6630,    506,    864,   1562,   3128,
     & 5983,   10205,  16453,  24342,  29366,  27618,  22287,
     &18444,   15934,  10004,   8329,   7515,   7355,   6759,
     & 6134,    3818,   3562,   3340,   3275,   3238,   3322,
     & 3645,      29,   9625,   6476,    477,    797,   1440,
     & 2922,    5784,  10417,  15984,  21989,  27546,  31260,
     &27986,   21611,  16924,   9638,   7717,   6683,   6556,
     & 6506,    6349,   6001,   5293,   5209,   4942,   5024,
     & 5189,    5175,     28,   9830,   6306,    464,    782,
     & 1427,    2895,   6024,  10674,  18654,  27231,  33011,
     &35322,   24660,  19722,  14250,  10701,   9764,   9595,
     & 8599,    7452,   6571,   5096,   4920,   4605,   4809,
     & 5160,    5374,     28,  10035,   6198,    462,    764,
     & 1332,    2501,   4925,   9551,  20687,  33597,  38128,
     &35021,   25249,  19731,  14932,  11406,   9445,   8999,
     & 8798,    7941,   6849,   5989,   5120,   5136,   4775,
     & 4954,    5494,     27,  10240,   6012,    438,    780,
     & 1255,    2525,   5944,  11157,  19076,  32094,  37873,
     &38300,   32050,  24974,  19668,  15500,  10208,   8437,
     & 7279,    5732,   4322,   3172,   2437,   2429,   2529,
     & 2820,      26,  10444,   5868,    477,    827,   1813,
     & 3849,    7666,  12670,  18867,  26402,  34617,  39481,
     &37032,   28169,  17309,  11409,   8718,   6897,   4115,
     & 4406,    4557,   4302,   4362,   4968,   6242,     26,
     &10649,    5785,    436,    722,   1336,   3601,   7989,
     &13927,   21969,  32624,  39225,  41524,  26749,  21618,
     &15291,   12291,   9715,   7792,   6324,   5440,   6243,
     & 6564,    6352,   6289,   6542,     26,  10854,   5698,
     &  433,     685,   1289,   2751,   6469,  11923,  20864,
     &30169,   40848,  46406,  39422,  29559,  22108,  13079,
     &10261,    7799,   5762,   4310,   4430,   5449,   6948,
     & 7797,    8185,     25,  11059,   5595,    446,    768,
     & 1434,    3283,   7281,  12657,  22225,  32734,  42324,
     &47519,   37322,  29369,  23337,  14421,  11182,   9078,
     & 5884,    4565,   3815,   4131,   5703,   7863,     25,
     &11264,    5475,    439,    724,   1428,   3344,   7002  /
      DATA (MAPPMIN(I), I=  6861, 7350) /
     &13165,   26384,  37240,  42053,  41304,  33930,  29309,
     &25763,   23855,  15288,  12002,   8386,   8387,   8272,
     & 7777,    6884,   8015,     25,  11468,   5403,    407,
     &  670,    1361,   3225,   8170,  16210,  22823,  33431,
     &46122,   51619,  44923,  32574,  25451,  16367,  12975,
     &13330,   12522,  10730,   8786,   7241,   6299,   6894,
     &   24,   11673,   5343,    434,    736,   1491,   3523,
     & 8030,   14600,  23479,  34264,  41617,  44817,  46284,
     &38004,   30996,  20977,  14110,  12001,  12013,  11583,
     &10560,   10184,  10320,     24,  11878,   5279,    405,
     &  641,    1343,   3166,   8276,  15605,  24300,  35302,
     &43727,   48280,  50072,  42427,  34254,  26641,  17543,
     &13267,    9969,   7599,   7399,   8688,  10483,     24,
     &12083,    5226,    376,    589,   1132,   2526,   6004,
     &13634,   24277,  33576,  43140,  49362,  50744,  41298,
     &32325,   23597,  18439,  18196,  15509,  14539,  13375,
     &11569,   12371,     24,  12288,   5162,    409,    654,
     & 1223,    2547,   6141,  11659,  19324,  32440,  48198,
     &55163,   54196,  41046,  35533,  30532,  26128,  17942,
     &14105,   11156,   9471,  10682,  15977,     23,  12492,
     & 5074,     399,    535,    939,   2597,   5542,  11726,
     &22236,   37803,  47535,  51838,  51829,  39629,  36029,
     &32355,   27686,  20208,  15186,  12529,  15148,  20934,
     &   23,   12697,   5004,    418,    677,   1236,   2564,
     & 6403,   12421,  19437,  30460,  43470,  50770,  51717,
     &44239,   39498,  34625,  30313,  21063,  17651,  16360,
     &16296,   17820,     23,  12902,   4957,    425,    689,
     & 1206,    2380,   5354,   9575,  16320,  25961,  38164,
     &48999,   54553,  52135,  42613,  35792,  29853,  22750,
     &19924,   21932,  22860,  24688,     23,  13107,   4870,
     &  437,     706,   1300,   2729,   6324,  11258,  18987,
     &30622,   42236,  48906,  49713,  43792,  39070,  35058,
     &31792,   29581,  28997,  29017,  29804,  38809,     22,
     &13312,    4815,    440,    682,   1260,   2653,   6620,
     &11436,   18930,  30990,  43426,  50102,  48890,  46114,
     &42962,   36651,  29106,  20937,  20907,  26226,  29572,
     &   22,   13516,   4758,    438,    733,   1327,   2456,
     & 5208,   10804,  19805,  26527,  34336,  43649,  50222,
     &50500,   43690,  40191,  36566,  32282,  28428,  29400,
     &39894,       3,  20480,      0,      3,  32767,      0,
     & 1574,      80,      3,      0,      0,      3,   2301,
     &    0,      11,   2334,   1965,     17,     12,     13,
     &   12,      12,     10,      9,     10,     14,   2355,
     & 2630,      31,     24,     23,     21,     19,     13,
     &   10,       6,      5,      1,      2,     16,   2396,
     & 3231,      63,     45,     33,     19,     12,     10,
     &    7,       6,      6,      5,      4,      4,      4,
     &   18,    2457,   3654,    120,     94,     75,     42,
     &   35,      28,     23,     20,     16,     12,     10,
     &    8,       6,      6,      6,     22,   2662,   4664,
     &  217,     203,    195,    157,    130,    103,     79,
     &   61,      47,     34,     25,     20,     18,     16,
     &   16,       9,     10,     10,     11,     24,   2867,
     & 5297,     209,    254,    296,    287,    290,    232,
     &  194,     115,     88,     67,     53,     47,     31,
     &   26,      25,     25,     26,     22,     25,     26,
     &   27,      26,   3072,   5843,    231,    259,    286,
     &  312,     321,    333,    331,    300,    256,    203,
     &  139,     105,     78,     56,     43,     36,     33,
     &   32,      30,     29,     29,     30,     32,     28,
     & 3276,    6157,    288,    329,    359,    378,    404,
     &  406,     386,    361,    351,    312,    249,    216,
     &  191,     143,    113,     81,     68,     53,     40,
     &   34,      33,     29,     32,     35,     37,     29,
     & 3481,    6533,    294,    340,    391,    455,    489,
     &  467,     490,    484,    447,    395,    345,    333,
     &  276,     244,    211,    175,    138,    123,     94,
     &   73,      59,     48,     36,     33,     38,     38,
     &   30,    3686,   6781,    316,    368,    415,    473,
     &  530,     574,    619,    627,    581,    550,    513  /
      DATA (MAPPMIN(I), I=  7351, 7840) /
     &  443,     379,    332,    293,    261,    224,    176,
     &  154,      99,     78,     62,     60,     57,     59,
     &   67,      85,     31,   3891,   6945,    348,    410,
     &  488,     583,    665,    763,    857,    887,    860,
     &  777,     679,    589,    498,    394,    313,    238,
     &  163,     142,    123,    105,     87,     70,     67,
     &   63,      63,     65,     79,     77,     31,   4096,
     & 7143,     367,    436,    518,    639,    784,    955,
     & 1141,    1332,   1393,   1221,   1056,    959,    588,
     &  472,     381,    255,    154,    126,     60,     41,
     &   25,      18,     19,     17,     16,     20,     16,
     &   19,      32,   4300,   7311,    391,    466,    550,
     &  707,     964,   1208,   1544,   1700,   1547,   1408,
     & 1226,    1001,    580,    551,    200,    168,    101,
     &   91,      87,     86,     87,     87,     86,     87,
     &   88,      89,     91,     95,    100,     33,   4505,
     & 7465,     393,    472,    578,    774,   1067,   1334,
     & 1684,    1903,   1890,   1844,   1556,   1228,    748,
     &  583,     397,    247,    211,    206,    216,    231,
     &  185,     148,     86,     69,     70,     70,     72,
     &   76,      80,     92,     33,   4710,   7589,    402,
     &  501,     650,    873,   1176,   1541,   1879,   2130,
     & 2140,    1836,   1574,   1376,   1044,    799,    657,
     &  559,     493,    335,    256,    198,    145,    112,
     &  113,     114,    116,    116,    120,    119,    112,
     &  117,      34,   4915,   7700,    389,    507,    716,
     &  959,    1281,   1747,   2208,   2482,   2680,   2408,
     & 2079,    1734,   1249,    832,    616,    328,    260,
     &  228,     208,    202,    196,    198,    202,    206,
     &  210,     217,    219,    215,    214,    217,    222,
     &   34,    5120,   7786,    410,    538,    761,   1052,
     & 1467,    2011,   2455,   2752,   3046,   2942,   2495,
     & 2084,    1877,   1472,   1086,    777,    541,    385,
     &  224,     167,    147,    127,    124,    122,    123,
     &  131,     134,    131,    127,    125,    125,     34,
     & 5324,    7863,    421,    556,    800,   1102,   1577,
     & 2181,    2667,   3069,   3456,   3379,   3036,   2591,
     & 2038,    1293,   1094,    576,    399,    297,    247,
     &  240,     244,    253,    260,    267,    265,    259,
     &  251,     247,    251,    256,    259,     34,   5529,
     & 7928,     421,    557,    810,   1141,   1679,   2357,
     & 2969,    3538,   3920,   3591,   3549,   3124,   2644,
     & 1747,    1330,   1005,    757,    545,    466,    404,
     &  343,     268,    256,    247,    254,    260,    254,
     &  253,     259,    262,    253,     35,   5734,   7957,
     &  424,     563,    828,   1211,   1840,   2576,   3137,
     & 3770,    4344,   4324,   3928,   3363,   2553,   2354,
     & 1756,    1555,   1546,   1372,    949,    613,    522,
     &  295,     295,    281,    254,    215,    214,    207,
     &  200,     201,    199,    200,     35,   5939,   7979,
     &  417,     557,    810,   1211,   1862,   2657,   3461,
     & 4298,    4996,   5104,   4706,   3865,   2950,   1846,
     & 1548,    1368,   1206,    966,    649,    540,    461,
     &  438,     458,    461,    450,    451,    461,    449,
     &  426,     422,    445,    465,     35,   6144,   7970,
     &  423,     573,    859,   1231,   1897,   2829,   3767,
     & 4765,    5570,   6035,   5111,   4351,   3637,   3253,
     & 2746,    2078,   1689,   1122,    972,    787,    645,
     &  527,     459,    320,    317,    316,    315,    317,
     &  315,     320,    319,    316,     35,   6348,   7937,
     &  447,     588,    870,   1373,   2103,   3111,   4319,
     & 5417,    6113,   6040,   5350,   4572,   3585,   3053,
     & 2291,    2006,   1754,   1452,   1248,   1225,   1077,
     &  909,     756,    706,    788,    710,    590,    502,
     &  405,     364,    380,    384,     34,   6553,   7890,
     &  440,     595,    907,   1417,   2156,   3168,   4444,
     & 5772,    6710,   6449,   5538,   4813,   4149,   3799,
     & 2980,    2541,   2244,   2061,   1941,   1806,   1641,
     & 1510,    1144,    990,    899,    866,    756,    652,
     &  369,     287,    247,     34,   6758,   7825,    466  /
      DATA (MAPPMIN(I), I= 7841, 8330) /
     &  613,     916,   1512,   2409,   3318,   4798,   6761,
     & 7574,    7119,   6386,   6103,   5421,   4496,   2974,
     & 2335,    2473,   2230,   1818,   1723,   1647,   1540,
     & 1239,    1039,    916,    791,    640,    583,    639,
     &  619,     388,     34,   6963,   7743,    485,    656,
     & 1020,    1649,   2683,   3915,   5517,   7386,   8722,
     & 8956,    9338,   7514,   6186,   3393,   2573,   1904,
     & 1736,    1496,   1111,    910,    797,    699,    646,
     &  594,     687,    814,    841,    865,    896,    903,
     &  889,      33,   7168,   7646,    479,    658,   1082,
     & 1842,    2882,   4119,   5998,   8443,  10340,  11163,
     & 9963,    8177,   6183,   4077,   3443,   2885,   2389,
     & 2099,    1739,   1473,   1206,   1076,   1051,    938,
     &  749,     729,    702,    662,    631,    669,     33,
     & 7372,    7555,    476,    663,   1138,   1970,   3109,
     & 4439,    6659,   9376,  11599,  13360,   9950,   8143,
     & 5817,    4311,   3568,   3139,   2883,   2528,   2058,
     & 1792,    1402,   1283,   1213,   1109,    979,    845,
     &  835,     863,    876,    878,     33,   7577,   7437,
     &  539,     785,   1192,   1997,   3526,   5329,   7740,
     &11292,   14598,  15355,  11332,   9299,   5024,   4037,
     & 3252,    2394,   2346,   1940,   1885,   1723,   1467,
     & 1322,    1337,   1315,   1227,   1193,   1271,   1307,
     & 1276,    1293,     32,   7782,   7295,    511,    705,
     & 1252,    2301,   3650,   5503,   8476,  12639,  15812,
     &15392,   11866,   9085,   7209,   4972,   3970,   3453,
     & 3312,    2971,   2794,   2816,   2224,   1752,   1442,
     & 1260,    1140,   1085,   1091,   1092,   1071,     32,
     & 7987,    7190,    502,    714,   1383,   2454,   3796,
     & 5790,    9051,  14395,  18060,  16458,  12601,   9535,
     & 7736,    7168,   5362,   4191,   3424,   2410,   2314,
     & 1927,    1570,   1281,   1158,   1074,   1224,   1454,
     & 1507,    1488,   1511,     31,   8192,   7036,    486,
     &  794,    1360,   2359,   4038,   6199,   9480,  14664,
     &18205,   17500,  14413,  12761,  11176,   7445,   5532,
     & 4243,    4379,   4129,   3632,   2944,   2366,   1868,
     & 1319,    1283,   1264,   1285,   1304,   1288,     30,
     & 8396,    6902,    457,    738,   1303,   2209,   3853,
     & 6314,    9909,  15420,  18991,  18986,  18542,  14443,
     &11652,    7005,   5938,   5521,   5242,   4610,   3391,
     & 2771,    2252,   1997,   1889,   1873,   1893,   1908,
     & 1924,      30,   8601,   6728,    457,    752,   1336,
     & 2267,    3876,   6301,  10229,  16494,  21539,  23787,
     &22390,   17482,   8945,   6337,   4194,   4217,   4331,
     & 4083,    3412,   2595,   2630,   2564,   2577,   2548,
     & 2524,    2512,   2661,     29,   8806,   6569,    457,
     &  723,    1256,   2122,   3785,   6544,  10828,  17044,
     &22454,   24984,  23083,  16892,  11534,   7127,   5452,
     & 5069,    4583,   4442,   4609,   4300,   3085,   2980,
     & 2889,    2979,   2978,   3004,     29,   9011,   6400,
     &  462,     656,   1206,   2122,   3698,   6933,  11289,
     &16383,   23187,  29122,  31556,  22460,  14831,  11112,
     & 8255,    5087,   4734,   3815,   3242,   2723,   2341,
     & 2195,    2181,   2126,   2095,   2285,     28,   9216,
     & 6261,     434,    735,   1289,   1956,   3167,   6271,
     &10783,   16509,  23931,  29374,  30994,  21299,  13546,
     & 8017,    6987,   7011,   7126,   6220,   4460,   4022,
     & 3692,    3663,   3637,   3812,   4120,     27,   9420,
     & 6077,     432,    642,   1076,   1788,   3610,   6066,
     &10214,   16901,  26698,  31401,  29783,  24611,  19786,
     &14959,    9362,   8425,   7650,   6562,   5384,   3742,
     & 3312,    3215,   3338,   3545,     27,   9625,   5923,
     &  370,     520,    854,   1307,   2557,   4692,   9459,
     &16594,   24053,  30653,  33324,  31433,  21103,  17170,
     & 9274,    7823,   7569,   7117,   6552,   6019,   5421,
     & 4893,    4849,   5165,     26,   9830,   5668,    386,
     &  572,     999,   1637,   3334,   5454,   9418,  16494,
     &26765,   36562,  40773,  25834,  20762,  16167,  12054,
     &12080,    9508,   7658,   5531,   4916,   4622,   5146,
     & 5461,      25,  10035,   5548,    396,    550,    930  /
      DATA (MAPPMIN(I), I=  8331, 8820) /
     & 1487,    2891,   5201,   9651,  18417,  28453,  35430,
     &37743,   28829,  22694,  17370,  13866,  11053,   9976,
     & 8521,    7209,   6280,   5355,   5664,     24,  10240,
     & 5323,     351,    589,    932,   1391,   2837,   5073,
     & 8208,   13156,  20805,  39264,  40297,  35427,  27940,
     &21849,   16305,  13564,   9662,   7958,   6405,   5222,
     & 3426,      24,  10444,   5166,    362,    571,    874,
     & 1805,    3053,   6146,  10652,  15436,  20221,  31703,
     &40517,   43777,  24694,  20496,  13837,   9977,   7861,
     & 6315,    4602,   4624,   5266,     23,  10649,   5036,
     &  337,     541,    826,   1506,   3367,   5776,   8832,
     &14592,   25979,  39698,  39881,  36885,  22941,  16375,
     &13300,   11872,   9337,   7346,   5998,   6876,     23,
     &10854,    4915,    392,    547,    910,   1420,   2951,
     & 5320,    9352,  15929,  23280,  33252,  41521,  45345,
     &32381,   26206,  20327,  12108,   9460,   6661,   4981,
     & 5529,      22,  11059,   4793,    409,    607,    974,
     & 1534,    2847,   5852,   9326,  14883,  24815,  36493,
     &43893,   46480,  31974,  25121,  20469,  12482,  10310,
     & 8460,    6912,     22,  11264,   4704,    339,    462,
     &  814,    1359,   2778,   5422,   8815,  15483,  29916,
     &38102,   41880,  42773,  26980,  22290,  21888,  20761,
     &18764,   12334,  11901,     21,  11468,   4605,    335,
     &  466,     859,   1403,   3118,   6303,   9956,  14514,
     &20950,   32695,  46668,  49666,  44052,  30442,  24100,
     &17566,   15950,  17589,     21,  11673,   4547,    357,
     &  473,     836,   1340,   2828,   5857,  10491,  16973,
     &29543,   36570,  40338,  42209,  41250,  34841,  28310,
     &19286,   13188,  15881,     21,  11878,   4468,    346,
     &  467,     893,   1446,   3265,   6190,  10032,  16192,
     &27718,   38213,  45149,  47415,  42719,  36877,  32467,
     &27483,   20373,  14632,     21,  12083,   4406,    357,
     &  464,     690,   1192,   2398,   4842,  10834,  18102,
     &25244,   34231,  47001,  52151,  47180,  36959,  31577,
     &19705,   21031,  21890,     20,  12288,   4335,    360,
     &  503,     828,   1328,   3035,   5923,   9990,  17005,
     &26725,   37269,  48795,  53113,  43614,  35881,  32747,
     &30088,   27497,     20,  12492,   4229,    369,    456,
     &  769,    1575,   2686,   5531,  10557,  18416,  30570,
     &39556,   44518,  45585,  41102,  36204,  35797,  34148,
     &31468,      20,  12697,   4173,    377,    526,    806,
     & 1213,    2356,   4781,   9034,  16500,  25135,  37390,
     &46485,   48999,  41941,  37852,  37329,  36391,  34571,
     &   20,   12902,   4109,    336,    465,    815,   1343,
     & 2798,    5313,   8651,  13557,  19965,  28475,  40031,
     &47937,   50038,  44985,  39885,  37628,  35964,     19,
     &13107,    4062,    382,    505,    809,   1272,   2991,
     & 6120,    9345,  13543,  21895,  31276,  40652,  47247,
     &49474,   45219,  43636,  44917,     19,  13312,   3990,
     &  379,     541,    881,   1383,   3145,   6081,  10348,
     &16846,   25406,  37331,  44763,  46493,  44533,  42608,
     &43082,   45086,     19,  13516,   3968,    353,    423,
     &  753,    1294,   2388,   4262,   6889,  13826,  20761,
     &25962,   32549,  44731,  52072,  48217,  44228,  63455,
     &    3,   20480,      0,      3,  32767,      0,   1485,
     &  100,       3,      0,      0,      3,   2301,      0,
     &   11,    2334,   1960,     17,     12,     13,     12,
     &   12,      10,      9,      9,     14,   2355,   2624,
     &   31,      24,     23,     21,     19,     13,     10,
     &    6,       5,      1,      1,     16,   2396,   3228,
     &   64,      44,     33,     19,     12,     10,      7,
     &    6,       6,      5,      5,      3,      4,     18,
     & 2457,    3653,    119,     94,     75,     42,     35,
     &   28,      23,     20,     16,     12,     10,      8,
     &    6,       6,      7,     22,   2662,   4662,    216,
     &  201,     199,    156,    130,    101,     81,     60,
     &   48,      33,     25,     20,     18,     16,     16,
     &    9,      10,     10,     10,     24,   2867,   5294,
     &  209,     254,    296,    285,    294,    231,    193,
     &  114,      88,     68,     52,     47,     31,     26  /
      DATA (MAPPMIN(I), I=   8821, 9310) /
     &   25,      25,     26,     23,     24,     26,     29,
     &   26,    3072,   5838,    230,    258,    289,    313,
     &  318,     335,    334,    300,    254,    201,    140,
     &  105,      76,     57,     43,     35,     33,     32,
     &   31,      28,     29,     30,     30,     28,   3276,
     & 6153,     288,    327,    356,    381,    406,    404,
     &  388,     359,    356,    312,    247,    214,    194,
     &  142,     112,     81,     68,     52,     41,     34,
     &   32,      30,     31,     36,     35,     29,   3481,
     & 6520,     293,    341,    392,    459,    485,    472,
     &  492,     488,    443,    394,    345,    337,    273,
     &  245,     209,    174,    136,    122,     94,     72,
     &   58,      48,     35,     34,     38,     39,     30,
     & 3686,    6761,    315,    370,    414,    474,    526,
     &  588,     626,    631,    585,    553,    509,    438,
     &  378,     329,    292,    261,    222,    172,    151,
     &   98,      76,     63,     60,     56,     60,     68,
     &   84,      30,   3891,   6909,    350,    417,    502,
     &  591,     672,    771,    846,    876,    862,    784,
     &  680,     579,    497,    392,    311,    232,    162,
     &  141,     122,    104,     86,     69,     66,     62,
     &   62,      67,     80,     31,   4096,   7094,    367,
     &  438,     521,    647,    799,    978,   1176,   1364,
     & 1406,    1209,   1027,    941,    565,    459,    380,
     &  238,     150,    115,     62,     38,     24,     19,
     &   16,      19,     16,     18,     16,     20,     32,
     & 4300,    7242,    393,    470,    566,    726,    990,
     & 1247,    1559,   1698,   1541,   1403,   1214,    965,
     &  570,     506,    207,    161,    100,     89,     86,
     &   86,      87,     86,     86,     87,     89,     90,
     &   91,      94,    102,     32,   4505,   7377,    389,
     &  477,     603,    803,   1089,   1389,   1722,   1930,
     & 1911,    1825,   1543,   1184,    705,    561,    357,
     &  241,     208,    205,    217,    226,    181,    139,
     &   83,      70,     69,     70,     74,     75,     81,
     &   33,    4710,   7487,    402,    497,    650,    870,
     & 1182,    1539,   1893,   2158,   2195,   1820,   1524,
     & 1362,    1052,    807,    663,    557,    494,    322,
     &  250,     196,    139,    113,    113,    114,    117,
     &  116,     117,    117,    114,    118,     33,   4915,
     & 7586,     385,    507,    716,    960,   1258,   1722,
     & 2198,    2551,   2749,   2459,   2101,   1737,   1203,
     &  819,     559,    319,    250,    227,    208,    202,
     &  197,     197,    200,    204,    207,    213,    217,
     &  217,     216,    218,     33,   5120,   7658,    394,
     &  521,     738,   1016,   1403,   1950,   2460,   2839,
     & 3103,    2971,   2477,   2051,   1855,   1512,   1090,
     &  777,     584,    387,    230,    167,    149,    128,
     &  126,     124,    126,    133,    132,    132,    127,
     &  125,      34,   5324,   7723,    404,    535,    772,
     & 1078,    1518,   2117,   2652,   3058,   3451,   3428,
     & 3050,    2603,   2115,   1336,   1113,    594,    398,
     &  303,     247,    240,    246,    258,    264,    262,
     &  262,     259,    250,    244,    251,    258,    257,
     &   34,    5529,   7777,    394,    519,    759,   1101,
     & 1584,    2218,   2860,   3456,   3898,   3749,   3679,
     & 3177,    2706,   1785,   1349,   1053,    802,    571,
     &  467,     404,    342,    276,    253,    248,    251,
     &  254,     251,    257,    260,    260,    257,     34,
     & 5734,    7795,    380,    508,    763,   1147,   1680,
     & 2344,    3027,   3743,   4316,   4472,   4084,   3401,
     & 2592,    2457,   1836,   1610,   1612,   1355,   1024,
     &  645,     522,    292,    294,    282,    258,    221,
     &  214,     196,    196,    201,    200,     34,   5939,
     & 7799,     398,    524,    761,   1154,   1698,   2376,
     & 3267,    4273,   4918,   5020,   4858,   3932,   3110,
     & 2011,    1607,   1426,   1255,   1094,    646,    564,
     &  474,     436,    464,    465,    450,    444,    456,
     &  447,     425,    431,    453,     34,   6144,   7762,
     &  402,     537,    792,   1203,   1853,   2671,   3553  /
      DATA (MAPPMIN(I), I=  9311, 9800) /
     & 4507,    5362,   5934,   5192,   4407,   3776,   3402,
     & 2944,    2216,   1761,   1299,   1034,    828,    659,
     &  534,     463,    321,    316,    323,    311,    314,
     &  316,     328,    332,     34,   6348,   7745,    414,
     &  537,     766,   1195,   1871,   2753,   3869,   4960,
     & 5732,    6057,   5350,   4732,   4079,   3394,   2544,
     & 2142,    1839,   1580,   1372,   1231,   1153,    929,
     &  828,     735,    782,    745,    644,    536,    456,
     &  359,     364,     33,   6553,   7670,    420,    543,
     &  794,    1300,   2030,   2882,   4039,   5429,   6392,
     & 6496,    5607,   4886,   4343,   4023,   3278,   2742,
     & 2296,    2101,   2000,   1860,   1710,   1540,   1297,
     & 1020,     896,    875,    789,    650,    447,    305,
     &   33,    6758,   7586,    427,    554,    812,   1363,
     & 2165,    2977,   4285,   6000,   7091,   7183,   6481,
     & 5912,    5614,   5124,   3734,   2728,   2484,   2284,
     & 2016,    1817,   1702,   1566,   1379,   1057,    944,
     &  813,     672,    576,    644,    621,     33,   6963,
     & 7500,     452,    598,    912,   1468,   2366,   3486,
     & 4895,    6738,   8179,   8549,   9508,   8129,   7166,
     & 3783,    2852,   2096,   1831,   1702,   1265,   1007,
     &  828,     718,    651,    603,    637,    814,    849,
     &  858,     895,    914,     32,   7168,   7374,    449,
     &  613,     986,   1670,   2668,   3650,   5361,   7858,
     & 9676,   10469,  10867,   8471,   6632,   4989,   3728,
     & 3097,    2583,   2162,   1838,   1570,   1248,   1110,
     & 1045,     915,    745,    721,    712,    688,    668,
     &   32,    7372,   7256,    470,    640,   1043,   1805,
     & 2897,    4071,   6117,   8924,  11113,  12357,  10978,
     & 8877,    6465,   4643,   3761,   3245,   2917,   2609,
     & 2175,    1902,   1438,   1319,   1215,   1081,    944,
     &  828,     854,    884,    908,     31,   7577,   7120,
     &  536,     788,   1193,   1876,   3111,   4745,   7225,
     &10543,   13629,  15604,  11878,  10514,   5627,   4313,
     & 3479,    2583,   2479,   2060,   1976,   1739,   1413,
     & 1230,    1305,   1313,   1255,   1243,   1270,   1295,
     &   31,    7782,   6939,    522,    755,   1260,   2152,
     & 3389,    5055,   7932,  11779,  15030,  16017,  12407,
     & 9994,    7887,   5242,   4287,   3624,   3363,   2964,
     & 2914,    2797,   2082,   1712,   1417,   1220,   1144,
     & 1117,    1122,   1120,     30,   7987,   6841,    479,
     &  692,    1283,   2308,   3594,   5383,   8105,  12537,
     &16385,   17929,  13139,  10882,   8541,   8003,   5678,
     & 4420,    3489,   2454,   2360,   1936,   1591,   1321,
     & 1184,    1161,   1331,   1473,   1524,     29,   8192,
     & 6649,     492,    762,   1253,   2114,   3680,   5727,
     & 8917,   13502,  17405,  19287,  15760,  13435,  11276,
     & 7553,    5655,   4840,   4785,   4304,   3445,   2741,
     & 2058,    1621,   1278,   1282,   1288,   1304,     29,
     & 8396,    6495,    436,    686,   1224,   2012,   3347,
     & 5663,    9593,  14141,  17315,  18691,  19278,  15721,
     &13071,    8728,   6666,   6537,   5397,   4507,   2953,
     & 2618,    2067,   1903,   1865,   1852,   1915,   2003,
     &   28,    8601,   6284,    407,    672,   1225,   2011,
     & 3440,    5889,   9939,  16037,  21522,  24071,  23418,
     &15698,   11009,   6282,   4591,   4462,   4805,   4425,
     & 3371,    2614,   2541,   2463,   2461,   2734,   2612,
     &   27,    8806,   6087,    419,    607,   1000,   1579,
     & 3140,    5872,  10110,  16095,  21876,  25200,  25680,
     &16845,   14581,   7110,   6085,   5352,   4952,   5102,
     & 4981,    3033,   3078,   2975,   3000,   2964,     27,
     & 9011,    5904,    403,    555,    957,   1733,   2928,
     & 5790,    9914,  15125,  21868,  30213,  35377,  22265,
     &17469,   12659,   8055,   5634,   4915,   3368,   2891,
     & 2323,    2129,   2267,   2207,   2127,     26,   9216,
     & 5714,     383,    607,   1064,   1653,   2582,   5014,
     & 9185,   14581,  23278,  30262,  31744,  22408,  19126,
     & 9720,    8071,   7934,   6526,   5246,   4148,   3586,
     & 3601,    3667,   4024,     25,   9420,   5547,    350,
     &  460,     755,   1190,   2188,   4496,   7724,  13039  /
      DATA (MAPPMIN(I), I=  9801,10290) /
     &20767,   32944,  35094,  30524,  21946,  19605,  10265,
     & 8768,    7317,   6065,   4203,   3658,   3354,   3529,
     &   24,    9625,   5303,    323,    423,    696,   1114,
     & 1838,    3795,   7542,  14144,  20797,  27797,  35541,
     &36383,   26040,  21885,  10816,   8801,   7598,   6980,
     & 6170,    5359,   5040,     23,   9830,   5057,    310,
     &  399,     736,   1228,   1993,   3376,   5424,   9831,
     &19093,   32765,  43399,  42822,  27017,  21054,  13976,
     &12518,    8489,   6746,   5325,   5311,     23,  10035,
     & 4868,     343,    472,    714,   1029,   1680,   3512,
     & 5970,   11278,  20543,  33247,  39759,  40138,  28525,
     &21768,   15985,  11982,  10733,   9201,   7576,   6403,
     &   22,   10240,   4636,    277,    416,    666,    965,
     & 1422,    3044,   4816,   7207,  11505,  20471,  42537,
     &41723,   37898,  27898,  20383,  15476,  12570,   9422,
     & 9065,      21,  10444,   4415,    267,    409,    573,
     &  810,    1663,   2753,   5748,  10396,  15961,  23613,
     &33519,   40618,  38254,  27810,  19560,  13798,  10772,
     & 9568,      20,  10649,   4302,    275,    399,    561,
     &  810,    1474,   3124,   4895,   7103,  10724,  24362,
     &36520,   41680,  39755,  23757,  18426,  14683,  13249,
     &   20,   10854,   4181,    292,    360,    564,    888,
     & 1292,    2744,   4737,   7439,  10904,  16874,  28297,
     &41267,   45740,  46439,  26337,  21361,  17492,     19,
     &11059,    3998,    306,    372,    604,   1020,   1665,
     & 2942,    4315,   6365,  13006,  21122,  31294,  42792,
     &45259,   43294,  28877,  24915,     19,  11264,   3924,
     &  309,     355,    544,    862,   1231,   2579,   4232,
     & 5960,    9071,  21152,  34273,  41233,  41140,  36631,
     &25294,   28206,     19,  11468,   3847,    289,    346,
     &  530,     876,   1223,   2554,   4712,   7483,  12025,
     &18933,   30753,  43983,  48463,  47240,  35703,  32757,
     &   18,   11673,   3706,    320,    406,    645,    935,
     & 1367,    3135,   5383,   8594,  14189,  26142,  38982,
     &45151,   45366,  39943,  44899,     18,  11878,   3665,
     &  326,     408,    612,    927,   1284,   3582,   5888,
     & 8623,   13182,  25953,  41810,  45763,  45422,  45276,
     &50111,      18,  12083,   3614,    303,    373,    485,
     &  767,    1316,   2515,   4084,   7766,  12615,  19202,
     &30020,   49102,  57736,  56950,  48404,     17,  12288,
     & 3497,     311,    394,    644,   1006,   1694,   3708,
     & 5976,    9598,  15923,  26355,  37962,  45523,  48895,
     &53970,      17,  12492,   3431,    335,    419,    607,
     & 1019,    1609,   2424,   4895,   9064,  14710,  22963,
     &35174,   40769,  44738,  57112,     17,  12697,   3350,
     &  343,     404,    542,    762,   1183,   2946,   6019,
     & 9638,   13366,  18200,  28539,  41477,  52659,  68765,
     &   16,   12902,   3303,    313,    417,    608,    902,
     & 1405,    3370,   5350,   7578,  11356,  19774,  30635,
     &34844,   44550,     16,  13107,   3240,    339,    450,
     &  688,     948,   1275,   3280,   5657,   8849,  13401,
     &20936,   28616,  46305,  67899,     16,  13312,   3212,
     &  342,     439,    629,    905,   1515,   3714,   5988,
     & 9839,   15478,  22725,  33518,  46773,  66925,     16,
     &13516,    3098,    349,    447,    680,   1029,   1603,
     & 3188,    5529,   9136,  15107,  22390,  27256,  35118,
     &63012,       3,  20480,      0,      3,  32767,      0,
     & 1036,     200,      3,      0,      0,      3,   2303,
     &    0,      11,   2334,   1938,     17,     12,     13,
     &   12,      12,     10,      9,      9,     14,   2355,
     & 2613,      31,     24,     23,     21,     19,     13,
     &   10,       6,      5,      1,      1,     16,   2396,
     & 3225,      63,     45,     33,     19,     12,     10,
     &    7,       6,      6,      5,      4,      4,      4,
     &   18,    2457,   3651,    119,     92,     76,     42,
     &   35,      28,     23,     20,     16,     12,     10,
     &    8,       6,      6,      7,     22,   2662,   4651,
     &  222,     199,    199,    157,    128,    100,     81,
     &   59,      47,     33,     24,     21,     17,     17,
     &   16,       9,      9,     10,     11,     24,   2867  /
      DATA (MAPPMIN(I), I= 10291,10780) /
     & 5270,     213,    259,    303,    293,    288,    229,
     &  187,     111,     87,     66,     52,     45,     31,
     &   25,      25,     26,     25,     23,     25,     26,
     &   28,      26,   3072,   5806,    231,    261,    289,
     &  312,     326,    344,    334,    301,    255,    194,
     &  137,     101,     74,     55,     42,     35,     33,
     &   32,      30,     29,     29,     30,     32,     27,
     & 3276,    6120,    288,    331,    358,    375,    403,
     &  411,     393,    373,    345,    315,    257,    213,
     &  187,     138,    108,     79,     67,     50,     40,
     &   34,      32,     30,     32,     35,     29,   3481,
     & 6457,     298,    338,    393,    468,    503,    480,
     &  499,     486,    447,    388,    351,    338,    272,
     &  239,     207,    167,    132,    116,     89,     69,
     &   55,      44,     36,     35,     37,     38,     30,
     & 3686,    6660,    312,    365,    428,    505,    561,
     &  608,     648,    635,    588,    556,    507,    411,
     &  370,     326,    289,    251,    210,    169,    132,
     &   90,      71,     61,     59,     58,     61,     72,
     &   63,      30,   3891,   6779,    351,    419,    518,
     &  627,     694,    791,    851,    853,    845,    777,
     &  682,     584,    500,    381,    301,    219,    160,
     &  142,     122,    101,     80,     67,     63,     63,
     &   63,      70,     87,     30,   4096,   6877,    369,
     &  450,     560,    705,    865,   1055,   1280,   1443,
     & 1359,    1190,    985,    797,    532,    404,    349,
     &  203,     140,     96,     67,     31,     23,     20,
     &   13,      19,     19,     16,     18,     31,   4300,
     & 6961,     390,    480,    607,    801,   1044,   1291,
     & 1543,    1718,   1614,   1423,   1209,    824,    553,
     &  376,     214,    137,     93,     85,     82,     83,
     &   85,      86,     88,     89,     89,     91,     93,
     &   93,      31,   4505,   7006,    391,    491,    641,
     &  843,    1098,   1466,   1855,   2068,   2068,   1770,
     & 1512,     972,    647,    537,    260,    228,    205,
     &  208,     226,    198,    172,     96,     76,     71,
     &   72,      74,     75,     77,     31,   4710,   7034,
     &  392,     498,    662,    862,   1152,   1611,   2073,
     & 2294,    2179,   1822,   1521,   1346,   1015,    791,
     &  658,     543,    436,    284,    212,    175,    113,
     &  113,     111,    113,    117,    120,    120,    120,
     &   31,    4915,   7052,    367,    474,    667,    933,
     & 1267,    1781,   2419,   2981,   3232,   2512,   2036,
     & 1496,     851,    737,    359,    275,    230,    212,
     &  201,     196,    202,    206,    208,    206,    210,
     &  217,     219,    222,     31,   5120,   7053,    356,
     &  462,     649,    940,   1362,   1945,   2578,   3077,
     & 3349,    3058,   2454,   2003,   1796,   1418,   1023,
     &  741,     574,    359,    232,    177,    152,    125,
     &  120,     122,    127,    135,    134,    139,     31,
     & 5324,    7044,    336,    430,    651,    980,   1384,
     & 2083,    2737,   3277,   3791,   4007,   3289,   2640,
     & 1917,    1113,    908,    496,    349,    253,    253,
     &  251,     247,    245,    253,    259,    270,    268,
     &  260,     260,     31,   5529,   7036,    340,    431,
     &  640,     990,   1467,   2227,   2883,   3367,   3870,
     & 4129,    3958,   3271,   2734,   1794,   1319,   1035,
     &  804,     578,    464,    388,    321,    255,    250,
     &  242,     242,    249,    263,    277,     31,   5734,
     & 6965,     333,    432,    657,   1028,   1521,   2222,
     & 2985,    3637,   4229,   4744,   4830,   3739,   3070,
     & 2635,    1895,   1657,   1374,   1110,    765,    492,
     &  375,     292,    283,    263,    236,    216,    202,
     &  205,      31,   5939,   6914,    344,    446,    636,
     & 1003,    1510,   2147,   2964,   3952,   4855,   5356,
     & 5344,    4066,   3267,   2352,   1672,   1506,   1261,
     & 1078,     650,    573,    478,    442,    445,    448,
     &  449,     453,    482,    482,     30,   6144,   6834,
     &  326,     405,    599,    999,   1560,   2221,   3024,
     & 3994,    5118,   5921,   6091,   4842,   4163,   3676  /
      DATA (MAPPMIN(I), I=  10781,11270) /
     & 3274,    2679,   1830,   1403,   1078,    868,    637,
     &  501,     388,    318,    319,    315,    328,     30,
     & 6348,    6712,    352,    458,    670,   1082,   1696,
     & 2517,    3421,   4337,   5416,   6211,   6078,   5112,
     & 4235,    3299,   2836,   2128,   1980,   1735,   1462,
     & 1202,    1140,   1023,    937,    849,    881,    754,
     &  623,      29,   6553,   6553,    365,    477,    729,
     & 1246,    1893,   2690,   3859,   4947,   6028,   6928,
     & 5537,    5295,   4912,   4315,   3514,   2971,   2434,
     & 2188,    2004,   1830,   1669,   1507,   1232,   1063,
     &  953,     862,     28,   6758,   6385,    364,    498,
     &  809,    1355,   1994,   2780,   4004,   5446,   6763,
     & 7448,    7084,   6519,   6308,   5346,   3969,   2733,
     & 2691,    2351,   2104,   1966,   1768,   1530,   1022,
     &  962,     816,     28,   6963,   6250,    360,    496,
     &  830,    1358,   2091,   3043,   4222,   5678,   7368,
     & 8967,    9902,   9671,   7033,   5149,   3144,   2568,
     & 2059,    1802,   1009,    930,    806,    674,    579,
     &  674,     882,     27,   7168,   6038,    431,    583,
     &  897,    1535,   2485,   3577,   4995,   7039,   9017,
     &10663,   12466,   9147,   7378,   5604,   3975,   3252,
     & 2640,    2204,   1680,   1242,   1119,   1051,    900,
     &  739,      26,   7372,   5820,    496,    657,    993,
     & 1702,    2830,   4135,   5982,   8752,  11428,  13292,
     &11498,    9436,   6977,   4781,   3950,   3430,   3011,
     & 2371,    2030,   1561,   1359,   1223,   1105,     25,
     & 7577,    5601,    450,    705,   1172,   1745,   2685,
     & 4462,    7118,  10069,  13241,  15493,  15089,  12278,
     & 6074,    4272,   3462,   2957,   2549,   2018,   1419,
     & 1118,    1269,   1367,     24,   7782,   5282,    574,
     &  819,    1347,   2404,   3698,   5239,   8784,  13898,
     &17731,   15705,  12740,   9658,   7335,   5800,   4887,
     & 4128,    3380,   2707,   2031,   1665,   1410,     23,
     & 7987,    5098,    473,    840,   1519,   2341,   3861,
     & 6489,   10094,  14782,  18093,  18675,  13699,  10911,
     & 9966,    6326,   5107,   3071,   2881,   1956,   1619,
     & 1409,      22,   8192,   4786,    487,    841,   1467,
     & 2268,    3640,   6412,  11249,  17056,  20607,  20544,
     &16589,   13719,  11187,   7321,   5851,   4727,   3131,
     & 2135,    1685,     21,   8396,   4563,    411,    673,
     & 1198,    2047,   3378,   6093,  11413,  16780,  21247,
     &23422,   22116,  16118,  11230,   8507,   5966,   4591,
     & 3241,    2364,     20,   8601,   4223,    420,    681,
     & 1145,    1909,   3450,   5769,  10196,  18157,  25687,
     &27769,   20234,  17936,   7858,   5976,   5243,   4753,
     & 4075,      19,   8806,   3912,    396,    643,   1077,
     & 1768,    3407,   6328,  11581,  19201,  30814,  32576,
     &21351,   17332,   9874,   7679,   6359,   5211,     18,
     & 9011,    3612,    370,    534,   1012,   1946,   3315,
     & 6923,   12305,  20310,  37597,  38444,  32061,  22738,
     &11588,    7949,   4386,     16,   9216,   3326,    312,
     &  451,    1104,   1951,   3188,   6008,  12242,  24044,
     &42040,   37953,  30593,  19788,  11928,     15,   9420,
     & 2943,     317,    415,    741,   1268,   2740,   7218,
     &12285,   27310,  54424,  46048,  33240,  23316,     13,
     & 9625,    2548,    279,    362,    625,   1159,   2647,
     & 7886,   16875,  34500,  56505,  47465,     12,   9830,
     & 2198,     210,    272,    389,   1024,   2126,   5412,
     &11339,   22907,  51517,     10,  10035,   1733,    235,
     &  278,     432,   1021,   2277,   7538,  18665,      8,
     &10240,    1241,    167,    232,    357,    594,   1646,
     &    7,   10444,    803,    267,    466,   1543,   7724,
     &    6,   10649,    674,    223,    298,    621,      5,
     &10854,     461,    397,   5011,      4,  11059,    123,
     & 1927,       3,  11264,      0,      3,  32767,      0,
     &  786,     400,      3,      0,      0,      3,   2305,
     &    0,      11,   2334,   1902,     17,     12,     13,
     &   12,      12,     10,      9,      9,     14,   2355,
     & 2599,      31,     24,     23,     21,     19,     13,
     &   10,       6,      5,      1,      2,     16,   2396  /
      DATA (MAPPMIN(I), I=  11271,11760) /
     & 3212,      63,     45,     33,     19,     12,     10,
     &    7,       6,      6,      5,      5,      3,      4,
     &   18,    2457,   3649,    119,     92,     75,     42,
     &   35,      28,     23,     20,     16,     12,     10,
     &    8,       6,      6,      5,     22,   2662,   4627,
     &  226,     207,    193,    160,    127,     97,     80,
     &   58,      46,     32,     24,     20,     18,     16,
     &   15,       9,     10,     10,     11,     24,   2867,
     & 5230,     215,    267,    316,    306,    278,    227,
     &  179,     107,     84,     64,     51,     43,     29,
     &   26,      25,     26,     24,     24,     25,     26,
     &   27,      26,   3072,   5746,    234,    259,    291,
     &  318,     338,    353,    342,    303,    254,    184,
     &  130,      95,     70,     51,     41,     35,     33,
     &   31,      30,     28,     30,     30,     32,     27,
     & 3276,    6055,    288,    328,    369,    368,    398,
     &  413,     413,    382,    349,    317,    265,    214,
     &  174,     135,    101,     76,     62,     49,     38,
     &   33,      31,     31,     33,     35,     28,   3481,
     & 6287,     316,    369,    424,    470,    504,    519,
     &  528,     482,    424,    392,    367,    334,    258,
     &  228,     194,    150,    123,     98,     75,     60,
     &   48,      39,     35,     36,     36,     29,   3686,
     & 6434,     322,    393,    473,    565,    609,    650,
     &  681,     627,    590,    550,    485,    407,    358,
     &  309,     272,    232,    186,    142,     98,     73,
     &   62,      60,     56,     60,     72,     64,     29,
     & 3891,    6507,    353,    428,    552,    680,    770,
     &  864,     829,    801,    813,    773,    688,    586,
     &  496,     367,    286,    193,    163,    142,    114,
     &   90,      67,     66,     63,     63,     68,     85,
     &   29,    4096,   6507,    386,    493,    633,    793,
     &  937,    1119,   1295,   1377,   1254,   1113,    957,
     &  729,     527,    398,    333,    193,    136,     96,
     &   72,      30,     25,     20,     17,     19,     18,
     &   19,      29,   4300,   6493,    386,    497,    677,
     &  872,    1072,   1334,   1581,   1773,   1689,   1401,
     & 1180,     647,    504,    289,    207,    110,     85,
     &   83,      80,     80,     80,     86,     93,     92,
     &   92,      97,     29,   4505,   6432,    396,    496,
     &  657,     917,   1263,   1637,   1981,   2166,   2144,
     & 1714,    1447,    755,    592,    425,    252,    213,
     &  211,     226,    198,    170,     96,     76,     69,
     &   69,      74,     77,     28,   4710,   6377,    389,
     &  488,     646,    921,   1347,   1850,   2284,   2383,
     & 2181,    1879,   1573,   1309,    873,    710,    557,
     &  462,     284,    213,    176,    108,    111,    118,
     &  118,     122,    127,     28,   4915,   6300,    396,
     &  484,     618,    924,   1431,   2031,   2749,   3267,
     & 3131,    2486,   1983,   1176,    989,    406,    289,
     &  222,     204,    204,    209,    211,    212,    215,
     &  219,     228,    237,     28,   5120,   6202,    352,
     &  445,     619,    975,   1508,   2214,   2996,   3485,
     & 3577,    2790,   2360,   1945,   1730,   1123,    844,
     &  728,     359,    268,    186,    169,    125,    125,
     &  125,     131,    144,     27,   5324,   6071,    343,
     &  444,     649,   1019,   1634,   2583,   3217,   3632,
     & 4125,    4051,   3002,   2423,   1331,   1038,    524,
     &  365,     256,    252,    247,    250,    254,    261,
     &  270,     283,     27,   5529,   5966,    334,    433,
     &  688,    1074,   1596,   2426,   3364,   4264,   4746,
     & 4490,    3663,   3027,   1988,   1376,   1076,    829,
     &  595,     462,    365,    289,    244,    253,    254,
     &  262,      26,   5734,   5765,    382,    498,    737,
     & 1183,    1869,   2648,   3453,   4539,   5559,   5893,
     & 4296,    3405,   2823,   1827,   1467,   1195,    797,
     &  567,     458,    306,    290,    267,    259,     26,
     & 5939,    5634,    399,    498,    642,   1041,   1755,
     & 2674,    3787,   5012,   5964,   6256,   4495,   3562,
     & 2399,    1746,   1497,   1268,    952,    630,    539  /
      DATA (MAPPMIN(I), I=  11761,12250) /
     &  483,     478,    484,    484,     25,   6144,   5422,
     &  394,     501,    718,   1163,   1874,   2731,   3849,
     & 5462,    6608,   6661,   5475,   4552,   3681,   3178,
     & 2372,    1863,   1406,   1267,    770,    597,    459,
     &  333,      24,   6348,   5243,    407,    527,    751,
     & 1144,    1805,   2669,   4050,   5511,   6394,   6834,
     & 5806,    4808,   3411,   3014,   2217,   2021,   1858,
     & 1628,    1386,   1267,   1068,     23,   6553,   5015,
     &  404,     496,    677,   1238,   2126,   3132,   4352,
     & 5657,    6700,   7158,   6445,   5336,   4410,   3859,
     & 3349,    2659,   2368,   2248,   2169,   1970,     22,
     & 6758,    4742,    430,    548,    786,   1367,   2315,
     & 3409,    4624,   6617,   8407,   8709,   7243,   6914,
     & 6048,    4564,   3465,   2905,   2531,   2295,   2074,
     &   21,    6963,   4509,    395,    551,    879,   1546,
     & 2388,    3569,   5239,   7957,   9878,  10289,  10378,
     & 7522,    6808,   3223,   2606,   2114,   1630,   1119,
     &   20,    7168,   4192,    453,    636,   1054,   1872,
     & 3154,    4476,   6215,   8345,  10790,  13091,  13072,
     & 9999,    5958,   3793,   2877,   2271,   1900,     19,
     & 7372,    3873,    525,    727,   1232,   2269,   3448,
     & 5389,    7831,  11116,  14630,  13677,  10766,   8642,
     & 6062,    4747,   3755,   2753,     17,   7577,   3465,
     &  590,     879,   1341,   2304,   3907,   6483,  10574,
     &15371,   19319,  14372,  13746,   5829,   4683,   3627,
     &   16,    7782,   3122,    480,    747,   1442,   2478,
     & 4046,    6806,  10846,  17943,  21684,  17541,  13512,
     & 9250,    6639,     14,   7987,   2691,    694,   1171,
     & 2143,    3772,   6737,  11762,  19356,  26584,  20384,
     &17826,    9399,     12,   8192,   2249,    753,   1268,
     & 2177,    4351,   8664,  14641,  23050,  32012,  22946,
     &   11,    8396,   1830,    763,   1352,   2498,   4905,
     &10886,   21859,  31406,  35860,      9,   8601,   1428,
     &  689,    1241,   2214,   3971,   8171,  20046,      7,
     & 8806,     994,    826,   1607,   3237,   8634,      6,
     & 9011,     561,    679,   2514,   5717,      5,   9216,
     &  370,     448,   2591,      3,   9420,      0,      3,
     &32767,       0,    670,    600,      3,      0,      0,
     &    3,    2306,      0,     11,   2334,   1883,     17,
     &   12,      13,     12,     12,     10,      9,     10,
     &   14,    2355,   2586,     31,     24,     23,     21,
     &   19,      13,     10,      6,      4,      2,      2,
     &   16,    2396,   3202,     64,     44,     33,     19,
     &   12,      10,      7,      6,      6,      5,      4,
     &    4,       4,     18,   2457,   3644,    119,     90,
     &   77,      42,     35,     28,     22,     20,     16,
     &   12,      10,      8,      6,      6,      5,     21,
     & 2662,    4603,    226,    217,    189,    161,    128,
     &   96,      77,     57,     45,     31,     24,     20,
     &   17,      16,     15,      9,     10,      9,     24,
     & 2867,    5178,    224,    285,    325,    313,    274,
     &  222,     166,    101,     81,     61,     50,     40,
     &   29,      25,     25,     26,     24,     24,     25,
     &   27,      28,     26,   3072,   5658,    235,    267,
     &  294,     329,    363,    380,    347,    296,    242,
     &  166,     119,     86,     64,     47,     39,     34,
     &   32,      31,     29,     30,     29,     32,     31,
     &   27,    3276,   5947,    295,    336,    366,    370,
     &  402,     437,    435,    398,    351,    322,    260,
     &  202,     160,    120,     90,     71,     57,     44,
     &   36,      32,     31,     32,     34,     35,     27,
     & 3481,    6109,    317,    363,    423,    502,    533,
     &  527,     537,    507,    449,    407,    380,    314,
     &  260,     214,    175,    136,    105,     82,     64,
     &   52,      42,     36,     36,     36,     28,   3686,
     & 6190,     344,    404,    472,    560,    657,    739,
     &  732,     653,    595,    558,    461,    411,    349,
     &  302,     251,    197,    153,    106,     73,     62,
     &   58,      57,     59,     69,     69,     28,   3891,
     & 6206,     364,    445,    537,    668,    832,    911  /
      DATA (MAPPMIN(I), I= 12251,12740) /
     &  825,     806,    855,    834,    744,    611,    431,
     &  346,     234,    176,    156,    120,     93,     68,
     &   65,      64,     63,     68,     84,     27,   4096,
     & 6128,     406,    508,    657,    838,   1033,   1193,
     & 1309,    1367,   1214,   1058,    957,    652,    507,
     &  389,     294,    187,    144,     76,     62,     30,
     &   22,      20,     17,     20,     27,   4300,   6047,
     &  403,     525,    723,    963,   1226,   1525,   1781,
     & 1805,    1576,   1318,    912,    574,    462,    195,
     &  171,      87,     87,     79,     79,     79,     82,
     &   87,      93,     94,     27,   4505,   5954,    413,
     &  539,     755,   1022,   1421,   1926,   2296,   2208,
     & 1955,    1665,   1059,    694,    574,    268,    225,
     &  221,     234,    203,    178,     94,     75,     69,
     &   71,      76,     26,   4710,   5831,    393,    526,
     &  783,    1132,   1637,   2215,   2577,   2577,   2073,
     & 1730,    1477,    970,    750,    559,    458,    287,
     &  211,     172,    110,    117,    121,    120,    122,
     &   26,    4915,   5720,    388,    497,    692,   1070,
     & 1761,    2664,   3312,   3498,   2773,   2155,   1628,
     &  928,     616,    335,    231,    203,    201,    213,
     &  215,     224,    229,    233,    239,     25,   5120,
     & 5570,     368,    483,    698,   1111,   1825,   2775,
     & 3533,    3814,   3407,   2644,   2067,   1861,   1274,
     &  956,     772,    393,    314,    198,    171,    129,
     &  127,     129,     25,   5324,   5376,    375,    511,
     &  799,    1252,   2076,   2943,   3672,   4259,   4550,
     & 3384,    2720,   1621,   1169,    593,    400,    257,
     &  255,     258,    251,    250,    272,    286,     24,
     & 5529,    5212,    386,    533,    814,   1234,   2062,
     & 3127,    4126,   4788,   4869,   4178,   3379,   2437,
     & 1502,    1176,    866,    530,    442,    361,    292,
     &  257,     268,     23,   5734,   4989,    384,    529,
     &  822,    1283,   2168,   3139,   4045,   5329,   6279,
     & 5801,    4248,   3229,   1939,   1458,   1152,    757,
     &  566,     458,    308,    313,     22,   5939,   4782,
     &  423,     539,    788,   1347,   2250,   3161,   4395,
     & 6014,    6958,   5178,   4181,   2859,   1915,   1602,
     & 1323,    1001,    667,    578,    526,     21,   6144,
     & 4544,     414,    564,    850,   1299,   2166,   3030,
     & 4228,    6234,   7413,   6343,   5077,   4205,   3680,
     & 2731,    2089,   1697,   1462,    985,     20,   6348,
     & 4316,     427,    559,    850,   1366,   2069,   2996,
     & 4411,    6126,   7189,   6373,   5202,   4296,   3409,
     & 2845,    2359,   2145,   1929,     19,   6553,   4012,
     &  398,     545,    854,   1389,   2037,   3011,   4626,
     & 7005,    8255,   7020,   6005,   5191,   4457,   3471,
     & 3343,    2774,     18,   6758,   3718,    438,    543,
     &  787,    1503,   2470,   3567,   5313,   7725,   9364,
     & 7556,    9145,   7335,   6152,   3911,   3415,     17,
     & 6963,    3376,    411,    624,   1083,   1815,   2937,
     & 4377,    7047,   9559,  11176,  12183,   8484,   8000,
     & 3648,    3056,     15,   7168,   2991,    463,    717,
     & 1292,    2343,   3672,   5133,   8935,  12658,  14292,
     &13855,   10203,   6825,     14,   7372,   2588,    566,
     &  823,    1413,   2640,   3944,   5917,  10062,  15192,
     &19220,   13540,  11172,     12,   7577,   2082,    703,
     & 1023,    1660,   3035,   5330,   9723,  16136,  22136,
     &24511,      10,   7782,   1620,    578,    947,   1671,
     & 3213,    5905,  10064,  23750,      8,   7987,   1198,
     &  651,    1387,   3101,   6385,  14706,      6,   8192,
     &  625,    1043,   2251,   3934,      4,   8396,    206,
     & 1784,       3,   8601,      0,      3,  32767,      0,
     &  604,     800,      3,      0,      0,      3,   2307,
     &    0,      11,   2334,   1850,     17,     12,     13,
     &   12,      12,     10,      9,      9,     14,   2355,
     & 2565,      31,     24,     23,     21,     19,     13,
     &   10,       6,      5,      1,      1,     16,   2396,
     & 3195,      63,     45,     33,     19,     12,     10,
     &    7,       6,      6,      5,      4,      4,      4  /
      DATA (MAPPMIN(I), I= 12741,13230) /
     &   18,    2457,   3639,    119,     91,     76,     41,
     &   35,      28,     23,     19,     16,     12,     10,
     &    8,       6,      6,      7,     21,   2662,   4580,
     &  230,     218,    190,    161,    129,     94,     76,
     &   56,      44,     30,     24,     19,     18,     16,
     &   14,       9,      9,     11,     23,   2867,   5094,
     &  253,     313,    351,    312,    260,    207,    144,
     &   96,      74,     57,     47,     36,     28,     25,
     &   26,      24,     25,     24,     26,     27,     25,
     & 3072,    5547,    233,    268,    318,    369,    398,
     &  397,     343,    283,    214,    146,    105,     77,
     &   56,      44,     37,     33,     32,     30,     29,
     &   29,      31,     31,     26,   3276,   5795,    302,
     &  343,     375,    384,    426,    470,    457,    413,
     &  360,     304,    241,    186,    143,    102,     80,
     &   62,      48,     40,     34,     31,     32,     33,
     &   33,      27,   3481,   5892,    316,    380,    458,
     &  518,     556,    560,    578,    523,    469,    414,
     &  390,     283,    236,    192,    146,    113,     86,
     &   66,      55,     43,     36,     37,     36,     35,
     &   27,    3686,   5927,    353,    419,    490,    594,
     &  712,     800,    751,    690,    629,    572,    470,
     &  389,     327,    268,    207,    160,    112,     72,
     &   62,      58,     57,     59,     68,     71,     26,
     & 3891,    5886,    391,    470,    577,    714,    890,
     &  931,     872,    905,    922,    848,    703,    540,
     &  401,     266,    186,    163,    126,     90,     67,
     &   64,      64,     63,     69,     26,   4096,   5774,
     &  442,     539,    687,    881,   1105,   1241,   1386,
     & 1392,    1243,   1036,    887,    600,    436,    390,
     &  220,     150,    115,     85,     33,     27,     22,
     &   17,      18,     26,   4300,   5683,    446,    576,
     &  768,    1056,   1379,   1713,   1906,   1747,   1466,
     & 1234,     686,    535,    331,    222,    133,     86,
     &   81,      77,     78,     81,     82,     91,     90,
     &   25,    4505,   5557,    450,    593,    834,   1193,
     & 1659,    2113,   2290,   2224,   1867,   1627,    790,
     &  726,     321,    251,    232,    242,    219,    189,
     &  107,      80,     70,     74,     25,   4710,   5419,
     &  410,     560,    847,   1276,   1917,   2510,   2778,
     & 2452,    2046,   1662,   1229,    852,    650,    548,
     &  310,     237,    192,    120,    115,    123,    120,
     &  121,      24,   4915,   5263,    410,    544,    804,
     & 1310,    2277,   3134,   3517,   3380,   2537,   2144,
     &  888,     889,    405,    244,    219,    203,    207,
     &  213,     226,    236,    241,     23,   5120,   5070,
     &  408,     563,    887,   1480,   2418,   3359,   3910,
     & 4020,    2917,   2157,   1948,   1468,   1070,    796,
     &  481,     360,    225,    179,    130,    127,     23,
     & 5324,    4867,    394,    542,    852,   1410,   2764,
     & 3498,    4044,   4972,   3947,   2952,   2192,   1286,
     &  844,     423,    341,    250,    258,    262,    271,
     &  292,      22,   5529,   4686,    423,    587,    920,
     & 1455,    2403,   3394,   4499,   5486,   4804,   3794,
     & 3269,    1809,   1348,   1028,    621,    485,    389,
     &  320,     268,     21,   5734,   4411,    447,    591,
     &  930,    1543,   2606,   3596,   4831,   6343,   7008,
     & 4817,    3800,   2191,   1625,   1243,    817,    611,
     &  511,     312,     20,   5939,   4192,    451,    616,
     &  887,    1415,   2263,   3461,   5136,   6671,   7347,
     & 4944,    3989,   2171,   1788,   1427,   1213,    703,
     &  622,      19,   6144,   3925,    433,    565,    863,
     & 1294,    2158,   3174,   4691,   7093,   8659,   5670,
     & 4798,    4110,   3485,   2431,   1971,   1545,     18,
     & 6348,    3640,    427,    610,    929,   1370,   2001,
     & 3114,    4952,   6909,   8110,   5557,   4896,   4235,
     & 3521,    2541,   2501,     16,   6553,   3273,    444,
     &  569,     798,   1222,   2091,   3677,   6016,   7836,
     & 8649,    6757,   5702,   4846,   4416,     15,   6758,
     & 2918,     465,    597,    921,   1557,   2677,   4612  /
      DATA (MAPPMIN(I), I= 13231,13720) /
     & 7004,    8724,   9333,   9782,   8788,   7654,     14,
     & 6963,    2567,    466,    682,   1138,   1906,   3275,
     & 5345,    8103,  10587,  12655,  13731,  10429,     12,
     & 7168,    2094,    565,    933,   1536,   2760,   4289,
     & 6520,   11930,  17393,  20379,     10,   7372,   1636,
     &  613,     859,   1469,   3002,   5039,   9244,  14818,
     &    8,    7577,   1041,    694,   1139,   2026,   4878,
     & 3600,       5,   7782,    453,    954,   2133,      4,
     & 7987,     127,   1247,      3,   8192,      0,      3,
     &32767,       0,    554,   1000,      3,      0,      0,
     &    3,    2308,      0,     11,   2334,   1821,     17,
     &   12,      13,     12,     11,     11,      9,      9,
     &   13,    2355,   2537,     31,     24,     23,     21,
     &   19,      13,     10,      6,      5,      4,     16,
     & 2396,    3189,     63,     45,     33,     19,     12,
     &   10,       7,      6,      6,      5,      4,      4,
     &    4,      18,   2457,   3635,    120,     89,     76,
     &   42,      34,     28,     23,     19,     17,     12,
     &    9,       8,      6,      6,      7,     21,   2662,
     & 4555,     232,    222,    193,    161,    128,     93,
     &   74,      54,     42,     30,     23,     20,     17,
     &   16,      13,     10,      9,     11,     23,   2867,
     & 5029,     265,    339,    367,    311,    253,    196,
     &  130,      92,     70,     55,     44,     34,     27,
     &   26,      25,     24,     25,     24,     27,     26,
     &   25,    3072,   5422,    239,    284,    357,    413,
     &  438,     393,    324,    262,    181,    129,     90,
     &   67,      51,     40,     35,     32,     31,     30,
     &   29,      30,     31,     31,     25,   3276,   5632,
     &  318,     364,    398,    421,    474,    495,    460,
     &  409,     359,    285,    207,    161,    117,     88,
     &   68,      54,     43,     35,     33,     31,     33,
     &   34,      26,   3481,   5668,    335,    383,    477,
     &  550,     602,    618,    604,    540,    484,    433,
     &  340,     262,    210,    157,    121,     90,     69,
     &   55,      45,     38,     36,     36,     36,     26,
     & 3686,    5663,    363,    440,    541,    673,    774,
     &  834,     771,    721,    659,    560,    462,    350,
     &  287,     223,    168,    117,     70,     62,     58,
     &   57,      59,     67,     68,     25,   3891,   5590,
     &  394,     490,    631,    790,    952,   1021,    990,
     & 1028,     952,    811,    623,    413,    310,    196,
     &  173,     129,     91,     67,     65,     63,     63,
     &   69,      25,   4096,   5488,    445,    542,    716,
     &  960,    1213,   1347,   1488,   1376,   1210,   1024,
     &  755,     559,    414,    319,    194,    153,     76,
     &   62,      31,     22,     18,     18,     25,   4300,
     & 5391,     462,    590,    821,   1156,   1506,   1776,
     & 1937,    1738,   1462,   1053,    661,    516,    239,
     &  216,      93,     88,     80,     78,     79,     81,
     &   87,     100,     24,   4505,   5237,    471,    620,
     &  912,    1389,   1906,   2324,   2226,   2161,   1802,
     & 1323,     786,    606,    290,    247,    238,    237,
     &  196,     143,     93,     72,     75,     23,   4710,
     & 5072,     438,    628,    996,   1485,   2229,   2784,
     & 2591,    2347,   1973,   1660,   1001,    779,    581,
     &  405,     283,    204,    161,    117,    124,    124,
     &   23,    4915,   4894,    460,    622,    972,   1637,
     & 2632,    3410,   3612,   3147,   2448,   1451,   1221,
     &  448,     321,    234,    207,    207,    209,    221,
     &  230,     239,     22,   5120,   4718,    427,    626,
     & 1016,    1653,   2780,   3741,   4100,   3590,   2842,
     & 2118,    1892,   1250,    922,    726,    421,    266,
     &  186,     154,    131,     21,   5324,   4536,    404,
     &  562,     898,   1587,   2846,   3671,   4353,   5169,
     & 3553,    2892,   1829,   1194,    808,    415,    320,
     &  263,     273,    297,     20,   5529,   4299,    428,
     &  611,     988,   1616,   2678,   3862,   4871,   5531,
     & 4787,    3764,   2683,   1593,   1260,    855,    550,
     &  446,     368,     19,   5734,   4018,    465,    635  /
      DATA (MAPPMIN(I), I=  13721,14210) /
     &  992,    1701,   2809,   3985,   5472,   6788,   6300,
     & 4673,    3412,   1959,   1459,   1127,    794,    605,
     &   18,    5939,   3720,    480,    637,    992,   1715,
     & 2718,    3856,   6070,   7945,   6072,   4946,   2954,
     & 2044,    1621,   1436,    820,     17,   6144,   3442,
     &  449,     593,    893,   1427,   2226,   3228,   5654,
     & 7596,    7913,   5856,   4834,   3891,   3162,   2365,
     &   16,    6348,   3118,    461,    617,    956,   1419,
     & 2134,    3545,   5429,   6890,   7630,   6088,   5039,
     & 3774,    3354,     14,   6553,   2743,    446,    573,
     &  812,    1315,   2340,   3832,   6452,   8236,   8643,
     & 6810,    6034,     13,   6758,   2336,    458,    601,
     &  971,    1617,   2966,   4546,   6853,   9297,  11021,
     &12439,      11,   6963,   1940,    512,    724,   1195,
     & 2095,    3359,   5769,   9543,  13441,      9,   7168,
     & 1433,     589,   1003,   1877,   3125,   4547,  12666,
     &    7,    7372,    961,    560,    753,   1531,   4865,
     &    4,    7577,    249,   1099,      3,   7782,      0,
     &    3,   32767,      0,    477,   1500,      3,      0,
     &    0,       3,   2310,      0,     10,   2334,   1731,
     &   17,      12,     13,     12,     12,     10,      9,
     &   13,    2355,   2455,     31,     24,     23,     21,
     &   19,      13,     10,      6,      5,      4,     16,
     & 2396,    3160,     63,     45,     33,     19,     12,
     &   10,       7,      6,      6,      5,      4,      4,
     &    4,      18,   2457,   3624,    124,     86,     76,
     &   41,      34,     28,     23,     19,     16,     12,
     &    9,       8,      6,      6,      7,     21,   2662,
     & 4455,     261,    237,    199,    156,    118,     86,
     &   66,      50,     37,     27,     22,     18,     17,
     &   15,      12,      9,     11,     10,     23,   2867,
     & 4873,     320,    430,    372,    300,    231,    155,
     &  107,      80,     60,     49,     38,     30,     26,
     &   25,      25,     24,     25,     25,     27,     28,
     &   24,    3072,   5133,    291,    370,    470,    518,
     &  438,     357,    275,    176,    135,     88,     64,
     &   50,      39,     35,     32,     31,     29,     30,
     &   30,      31,     31,     24,   3276,   5178,    398,
     &  471,     537,    607,    581,    513,    433,    329,
     &  244,     189,    138,     94,     73,     58,     45,
     &   37,      33,     32,     32,     34,     34,     24,
     & 3481,    5150,    400,    500,    627,    715,    740,
     &  685,     617,    540,    442,    302,    236,    178,
     &  127,      95,     71,     55,     45,     37,     37,
     &   36,      36,     23,   3686,   5100,    422,    543,
     &  716,     879,   1001,    923,    845,    739,    588,
     &  414,     332,    253,    163,    132,     65,     62,
     &   58,      56,     59,     68,     23,   3891,   4998,
     &  426,     580,    836,   1090,   1282,   1218,   1184,
     & 1018,     835,    471,    408,    226,    184,    137,
     &   94,      67,     65,     64,     63,     68,     22,
     & 4096,    4859,    494,    676,   1026,   1434,   1564,
     & 1650,    1440,   1219,   1039,    688,    534,    427,
     &  244,     172,    133,     81,     43,     28,     22,
     &   22,    4300,   4749,    521,    752,   1155,   1611,
     & 1949,    2125,   1793,   1494,   1046,    690,    556,
     &  225,     222,     88,     85,     80,     80,     82,
     &   83,      21,   4505,   4593,    523,    772,   1249,
     & 1851,    2363,   2457,   2281,   1902,   1565,    855,
     &  673,     310,    256,    239,    233,    198,    153,
     &   96,      21,   4710,   4398,    525,    788,   1368,
     & 2161,    2838,   2640,   2555,   2174,   1865,   1060,
     &  840,     633,    411,    305,    214,    165,    119,
     &  123,      20,   4915,   4195,    549,    828,   1394,
     & 2353,    3289,   3694,   3745,   2939,   1448,   1422,
     &  406,     323,    238,    210,    212,    218,    228,
     &   19,    5120,   4009,    519,    800,   1403,   2350,
     & 3347,    3987,   4375,   3345,   2339,   2058,   1383,
     & 1018,     716,    474,    277,    200,     18,   5324,
     & 3794,     474,    771,   1439,   2351,   3316,   4364  /
      DATA (MAPPMIN(I), I= 14211,14700) /
     & 5360,    3926,   3283,   2012,   1289,   1023,    462,
     &  375,     270,     17,   5529,   3507,    521,    791,
     & 1374,    2401,   3585,   4751,   5820,   5261,   4182,
     & 2916,    1740,   1335,    832,    591,     16,   5734,
     & 3195,     505,    751,   1293,   2320,   3647,   5193,
     & 6612,    7340,   5124,   4019,   2089,   1618,   1200,
     &   15,    5939,   2856,    501,    716,   1335,   2232,
     & 3228,    4986,   7052,   8190,   5483,   4172,   2302,
     & 1910,      13,   6144,   2483,    486,    693,   1093,
     & 1812,    2940,   4346,   6856,   8898,   6779,   5589,
     &   12,    6348,   2065,    559,    748,   1164,   1891,
     & 2745,    4326,   6488,   8145,   7912,     10,   6553,
     & 1639,     491,    695,   1136,   1829,   3290,   6066,
     & 9598,       8,   6758,   1174,    529,    732,   1352,
     & 3118,    5550,      6,   6963,    765,    656,   1003,
     & 2203,       4,   7168,    217,   1351,      3,   7372,
     &    0,       3,  32767,      0,    433,   2000,      3,
     &    0,       0,      3,   2312,      0,     10,   2334,
     & 1629,      17,     12,     13,     12,     11,     11,
     &   11,      13,   2355,   2378,     31,     24,     23,
     &   21,      19,     13,     10,      6,      5,      1,
     &   16,    2396,   3134,     63,     45,     33,     19,
     &   12,      10,      7,      6,      6,      5,      5,
     &    3,       4,     18,   2457,   3612,    123,     90,
     &   74,      41,     33,     29,     21,     19,     16,
     &   12,       9,      8,      6,      6,      7,     21,
     & 2662,    4406,    272,    239,    200,    157,    113,
     &   85,      64,     49,     35,     26,     21,     18,
     &   17,      14,     12,      9,     10,     11,     22,
     & 2867,    4759,    367,    484,    372,    292,    201,
     &  135,      98,     72,     55,     43,     35,     28,
     &   26,      25,     25,     24,     25,     26,     26,
     &   23,    3072,   4921,    380,    481,    555,    508,
     &  426,     309,    193,    153,     97,     68,     52,
     &   41,      36,     32,     32,     29,     30,     30,
     &   31,      30,     23,   3276,   4891,    467,    585,
     &  712,     675,    583,    483,    344,    259,    198,
     &  141,      93,     73,     56,     44,     37,     33,
     &   32,      32,     34,     33,     22,   3481,   4814,
     &  472,     632,    806,    814,    782,    712,    616,
     &  447,     319,    239,    168,    121,     95,     69,
     &   51,      43,     37,     34,     35,     22,   3686,
     & 4705,     509,    703,    945,   1086,   1087,    961,
     &  819,     603,    433,    342,    233,    140,    113,
     &   63,      59,     59,     55,     61,     74,     21,
     & 3891,    4579,    528,    770,   1122,   1432,   1348,
     & 1319,    1133,    845,    525,    390,    227,    161,
     &  143,      67,     66,     65,     63,     63,     21,
     & 4096,    4420,    616,    923,   1383,   1763,   1770,
     & 1561,    1329,   1062,    746,    572,    433,    258,
     &  180,     130,     92,     44,     29,     25,     20,
     & 4300,    4288,    660,    983,   1489,   2015,   2310,
     & 1918,    1562,   1191,    796,    624,    277,    267,
     &  104,      88,     86,     83,     83,     20,   4505,
     & 4134,     633,    983,   1581,   2245,   2656,   2372,
     & 2104,    1930,    923,    903,    349,    279,    242,
     &  241,     217,    186,    109,     19,   4710,   3916,
     &  664,    1059,   1747,   2531,   2999,   2812,   2408,
     & 2092,    1194,    964,    713,    514,    369,    238,
     &  188,     131,     18,   4915,   3717,    632,   1045,
     & 1805,    2778,   3685,   4323,   3029,   2374,   1503,
     &  680,     453,    250,    211,    211,    225,     17,
     & 5120,    3504,    581,    997,   1766,   2837,   3926,
     & 4569,    3717,   3120,   2222,   1638,   1233,    898,
     &  493,     356,     16,   5324,   3264,    555,    892,
     & 1707,    2713,   3678,   4706,   5138,   3845,   2834,
     & 1687,    1314,    725,    500,     15,   5529,   2944,
     &  584,     924,   1698,   2818,   4114,   5429,   6089,
     & 4712,    4264,   2016,   1585,   1083,     14,   5734,
     & 2591,     513,    805,   1557,   2705,   4143,   6432  /
      DATA (MAPPMIN(I), I=  14701,15190) /
     & 8251,    5994,   5190,   2656,   1917,     12,   5939,
     & 2198,     559,    857,   1592,   2606,   3719,   6539,
     & 8965,    6543,   5582,     11,   6144,   1804,    509,
     &  724,    1211,   2046,   3043,   5020,   8322,   9479,
     &    9,    6348,   1369,    556,    760,   1168,   1792,
     & 2626,    4852,      7,   6553,    904,    542,    769,
     & 1307,    3388,      5,   6758,    384,    589,   1209,
     &    4,    6963,     22,   1965,      3,   7168,      0,
     &    3,   32767,      0,    370,   3000,      3,      0,
     &    0,       3,   2315,      0,      9,   2334,   1457,
     &   17,      12,     13,     12,     12,     10,     12,
     & 2355,    2243,     31,     24,     23,     21,     19,
     &   13,      10,      6,      7,     16,   2396,   3094,
     &   63,      45,     33,     19,     12,     10,      7,
     &    6,       6,      5,      5,      3,      4,     18,
     & 2457,    3589,    123,     95,     71,     41,     33,
     &   27,      22,     19,     15,     12,      9,      8,
     &    6,       6,      7,     20,   2662,   4329,    285,
     &  244,     203,    155,    106,     84,     60,     46,
     &   35,      27,     21,     17,     16,     14,     10,
     &   10,      11,     22,   2867,   4648,    442,    472,
     &  388,     296,    168,    125,     88,     60,     54,
     &   40,      32,     27,     26,     24,     25,     25,
     &   25,      27,     27,     22,   3072,   4687,    554,
     &  639,     571,    505,    316,    225,    170,    103,
     &   76,      58,     42,     36,     33,     31,     29,
     &   30,      30,     31,     31,     22,   3276,   4612,
     &  625,     792,    793,    690,    558,    351,    274,
     &  210,     141,    106,     77,     55,     43,     37,
     &   33,      32,     32,     34,     33,     21,   3481,
     & 4415,     727,    921,    974,    890,    757,    650,
     &  444,     303,    217,    155,    121,     90,     63,
     &   48,      40,     35,     36,     39,     20,   3686,
     & 4249,     764,   1048,   1256,   1229,   1089,    830,
     &  575,     452,    337,    174,    168,     69,     63,
     &   60,      57,     56,     63,     19,   3891,   4096,
     &  732,    1091,   1545,   1527,   1460,   1319,    799,
     &  618,     364,    222,    171,    109,     74,     65,
     &   65,      63,     19,   4096,   3855,    988,   1449,
     & 1963,    1957,   1790,   1260,    990,    811,    541,
     &  387,     296,    193,    100,    100,     38,     30,
     &   18,    4300,   3672,    945,   1440,   2125,   2362,
     & 2093,    1646,   1242,    927,    672,    325,    279,
     &  129,      95,     89,     90,     17,   4505,   3465,
     &  941,    1434,   2274,   2730,   2600,   2295,   1887,
     & 1175,     831,    456,    295,    237,    242,    235,
     &   16,    4710,   3209,    973,   1612,   2430,   3367,
     & 3054,    2649,   1957,   1348,    990,    754,    500,
     &  408,     255,     15,   4915,   2972,    926,   1625,
     & 2792,    3926,   4488,   3691,   2001,   1828,    583,
     &  468,     266,    207,     14,   5120,   2727,    868,
     & 1676,    2862,   3990,   4853,   3951,   3250,   2282,
     & 1710,    1329,    794,     13,   5324,   2450,    793,
     & 1427,    2595,   3683,   4146,   5234,   4349,   3117,
     & 1956,    1518,     12,   5529,   2114,    718,   1195,
     & 2178,    3600,   5100,   6486,   5691,   5333,   2477,
     &   10,    5734,   1753,    564,    952,   1940,   3447,
     & 5390,    7337,   8574,      9,   5939,   1283,    647,
     & 1065,    1884,   3106,   5486,   5005,      7,   6144,
     &  836,     576,    814,   1407,   2953,      5,   6348,
     &  357,    1042,   1659,      3,   6553,      0,      3,
     &32767,       0,    312,   5000,      3,      0,      0,
     &    3,    2321,      0,      8,   2334,   1233,     17,
     &   12,      13,     12,     12,     11,   2355,   2016,
     &   31,      24,     23,     21,     19,     13,     10,
     &    6,      15,   2396,   3025,     63,     45,     33,
     &   19,      12,     10,      7,      6,      6,      5,
     &    4,       4,     17,   2457,   3541,    123,     94,
     &   75,      42,     33,     26,     21,     19,     14,
     &   12,       8,      8,      6,      6,     20,   2662  /
      DATA (MAPPMIN(I), I=  15191,15680) /
     & 4236,     288,    246,    204,    156,    103,     83,
     &   57,      42,     33,     27,     23,     19,     17,
     &   16,      12,     11,      9,     21,   2867,   4517,
     &  479,     483,    394,    267,    186,    106,     76,
     &   60,      53,     44,     30,     26,     25,     25,
     &   23,      24,     24,     25,     21,   3072,   4480,
     &  757,     682,    600,    385,    277,    210,    121,
     &   92,      68,     52,     39,     34,     32,     30,
     &   29,      29,     29,     31,     20,   3276,   4347,
     &  902,     882,    792,    659,    421,    341,    230,
     &  159,     124,     93,     65,     46,     36,     32,
     &   31,      31,     33,     20,   3481,   4100,    916,
     & 1152,    1056,    914,    684,    478,    361,    228,
     &  177,     140,    117,     66,     53,     42,     35,
     &   35,      40,     19,   3686,   3882,    941,   1293,
     & 1397,    1246,   1011,    650,    504,    369,    226,
     &  168,     127,     74,     61,     60,     57,     58,
     &   18,    3891,   3611,    987,   1439,   1692,   1631,
     & 1450,     984,    749,    438,    263,    196,    113,
     &   82,      67,     66,     64,     16,   4096,   3313,
     & 1262,    2001,   2137,   1958,   1565,   1097,    882,
     &  681,     429,    344,    200,    128,    111,     15,
     & 4300,    3036,   1344,   2080,   2450,   2291,   1930,
     & 1335,    1052,    745,    407,    343,    149,    116,
     &   14,    4505,   2745,   1309,   2076,   2686,   2679,
     & 2466,    2147,   1374,   1112,    502,    401,    272,
     &   13,    4710,   2428,   1374,   2244,   3341,   3171,
     & 2816,    2214,   1560,   1158,    852,    727,     12,
     & 4915,    2118,   1330,   2260,   3864,   4654,   4022,
     & 2652,    2017,    887,    641,     11,   5120,   1799,
     & 1432,    2705,   4025,   5075,   4615,   3688,   2447,
     & 1958,       9,   5324,   1456,   1349,   2216,   3821,
     & 3992,    4773,   4697,      8,   5529,   1058,    971,
     & 1678,    2954,   4556,   6312,      6,   5734,    618,
     &  656,    1147,   3056,      4,   5939,    133,   1385,
     &    3,    6144,      0,      3,  32767,      0,    294,
     & 6000,       3,      0,      0,      3,   2324,      0,
     &    7,    2334,    977,     17,     12,     13,     12,
     &   11,    2355,   1914,     31,     24,     23,     21,
     &   19,      13,     10,     10,     15,   2396,   2984,
     &   63,      45,     33,     19,     12,     10,      7,
     &    6,       6,      5,      5,      4,     17,   2457,
     & 3517,     121,     94,     76,     42,     35,     26,
     &   21,      18,     14,     11,      9,      7,      6,
     &    6,      20,   2662,   4195,    289,    246,    205,
     &  152,     104,     83,     57,     40,     33,     26,
     &   23,      20,     17,     16,     14,     10,      9,
     &   21,    2867,   4464,    485,    481,    395,    265,
     &  189,     104,     74,     61,     52,     44,     30,
     &   24,      25,     25,     23,     24,     25,     27,
     &   21,    3072,   4424,    792,    685,    594,    363,
     &  278,     206,    117,     92,     68,     50,     39,
     &   34,      32,     30,     29,     28,     29,     31,
     &   20,    3276,   4258,    961,    894,    794,    631,
     &  423,     338,    210,    158,    122,     91,     55,
     &   43,      35,     32,     32,     30,     33,     19,
     & 3481,    4003,    979,   1186,   1064,    894,    632,
     &  472,     338,    218,    179,    141,    119,     66,
     &   53,      42,     35,     35,     18,   3686,   3763,
     &  976,    1297,   1383,   1245,    972,    645,    495,
     &  360,     230,    168,    138,     78,     64,     61,
     &   58,      17,   3891,   3472,   1043,   1446,   1675,
     & 1614,    1457,    947,    751,    435,    264,    206,
     &  109,      82,     67,     65,     16,   4096,   3149,
     & 1273,    1972,   2169,   1987,   1547,   1097,    877,
     &  673,     422,    343,    195,    136,    113,     15,
     & 4300,    2838,   1424,   2085,   2485,   2334,   1817,
     & 1314,    1052,    726,    414,    342,    144,    114,
     &   13,    4505,   2528,   1434,   2182,   2670,   2617,
     & 2350,    2091,   1408,   1066,    524,    405,     12  /
      DATA (MAPPMIN(I), I=  15681,16170) /
     & 4710,    2175,   1547,   2249,   3301,   3207,   2851,
     & 2003,    1560,   1142,    897,     11,   4915,   1848,
     & 1487,    2500,   3729,   4451,   4061,   2577,   2172,
     &  864,       9,   5120,   1507,   1573,   2990,   4169,
     & 5173,    5006,   3080,      8,   5324,   1164,   1324,
     & 2156,    3661,   3914,   5216,      6,   5529,    707,
     & 1050,    1665,   3300,      4,   5734,    236,   1022,
     &    3,    5939,      0,      3,  32767,      0,    273,
     & 8000,       3,      0,      0,      3,   2330,      0,
     &    6,    2334,    617,     17,     12,     13,     10,
     & 2355,    1722,     31,     24,     23,     21,     19,
     &   13,      12,     15,   2396,   2913,     64,     44,
     &   33,      19,     12,     10,      7,      6,      6,
     &    5,       4,      5,     17,   2457,   3459,    121,
     &   93,      77,     41,     36,     29,     21,     17,
     &   14,      10,      9,      7,      6,      6,     20,
     & 2662,    4105,    298,    247,    203,    147,    102,
     &   82,      54,     39,     30,     26,     23,     22,
     &   17,      17,     14,     11,     10,     21,   2867,
     & 4365,     490,    478,    392,    266,    193,    102,
     &   75,      59,     51,     43,     29,     24,     25,
     &   24,      24,     24,     25,     27,     20,   3072,
     & 4324,     791,    681,    582,    376,    280,    208,
     &  117,      92,     68,     49,     38,     34,     31,
     &   31,      29,     28,     29,     20,   3276,   4113,
     &  965,     895,    792,    643,    426,    332,    213,
     &  156,     121,     87,     52,     42,     34,     33,
     &   31,      30,     32,     19,   3481,   3844,   1017,
     & 1158,    1047,    879,    582,    461,    341,    242,
     &  192,     153,    135,     68,     55,     44,     35,
     &   36,      17,   3686,   3569,    998,   1238,   1320,
     & 1205,     947,    647,    497,    377,    236,    190,
     &  150,     115,     74,     63,     16,   3891,   3252,
     & 1023,    1356,   1611,   1538,   1362,   1063,    734,
     &  602,     283,    217,    141,     91,     68,     15,
     & 4096,    2895,   1264,   1801,   2168,   2002,   1629,
     & 1101,     880,    695,    438,    336,    219,    154,
     &   13,    4300,   2550,   1436,   2037,   2414,   2298,
     & 1880,    1329,   1055,    743,    430,    334,     12,
     & 4505,    2188,   1561,   2086,   2669,   2632,   2291,
     & 1940,    1426,   1045,    621,     11,   4710,   1817,
     & 1485,    2088,   3158,   3069,   2735,   2144,   1683,
     & 1372,       9,   4915,   1435,   1496,   2440,   3581,
     & 4282,    4012,   3360,      8,   5120,   1050,   1533,
     & 2861,    4372,   5371,   5127,      6,   5324,    624,
     & 1446,    2529,   4453,      4,   5529,    162,   1355,
     &    3,    5734,      0,      3,  32767,      0,    249,
     &10000,       3,      0,      0,      3,   2334,      0,
     &   10,    2355,   1555,     31,     24,     23,     21,
     &   19,      13,     12,     15,   2396,   2845,     63,
     &   45,      33,     19,     12,     10,      7,      6,
     &    6,       5,      5,      3,     17,   2457,   3400,
     &  120,      93,     76,     41,     36,     30,     25,
     &   16,      13,     10,      8,      7,      6,      6,
     &   19,    2662,   4017,    303,    248,    202,    143,
     &  101,      80,     53,     37,     30,     25,     23,
     &   22,      18,     18,     16,     12,     20,   2867,
     & 4267,     487,    475,    388,    270,    195,    104,
     &   74,      59,     49,     42,     31,     23,     25,
     &   25,      24,     24,     25,     20,   3072,   4226,
     &  772,     673,    577,    385,    285,    216,    122,
     &   93,      66,     49,     37,     34,     33,     30,
     &   30,      28,     30,     19,   3276,   3988,    933,
     &  872,     782,    678,    429,    345,    217,    156,
     &  123,      87,     54,     43,     36,     33,     30,
     &   30,      18,   3481,   3695,   1035,   1115,   1011,
     &  843,     570,    457,    355,    263,    217,    168,
     &  153,      71,     60,     49,     38,     17,   3686,
     & 3408,     963,   1181,   1250,   1150,    971,    626,
     &  493,     376,    276,    207,    171,    142,    109  /
      DATA (MAPPMIN(I), I=  16171, 16582) /
     &   70,      15,   3891,   3072,    949,   1281,   1498,
     & 1450,    1316,   1193,    769,    649,    370,    228,
     &  193,      99,     14,   4096,   2684,   1245,   1668,
     & 2081,    1966,   1729,   1107,    900,    741,    472,
     &  341,     246,     13,   4300,   2326,   1406,   1933,
     & 2357,    2249,   1921,   1363,   1074,    819,    443,
     &  353,      11,   4505,   1923,   1498,   1985,   2597,
     & 2582,    2294,   1940,   1468,   1149,      9,   4710,
     & 1527,    1291,   1811,   2800,   2930,   2776,   2460,
     &    8,    4915,   1082,   1395,   2062,   3312,   3893,
     & 4591,       6,   5120,    665,   1495,   2923,   4786,
     &    4,    5324,    174,   1699,      3,   5529,      0,
     &    3,   32767,      0,    194,  20000,      3,      0,
     &    0,       3,   2347,      0,      7,   2355,    797,
     &   31,      24,     23,     21,     13,   2396,   2490,
     &   64,      44,     33,     19,     12,     10,      7,
     &    6,       6,      6,     15,   2457,   3038,    113,
     &   88,      73,     44,     33,     28,     22,     18,
     &   15,      13,     11,     10,     18,   2662,   3591,
     &  307,     248,    201,    134,     96,     72,     48,
     &   36,      29,     24,     21,     21,     23,     21,
     &   20,      18,   2867,   3790,    472,    483,    380,
     &  267,     198,     93,     61,     48,     40,     39,
     &   37,      32,     25,     24,     27,     18,   3072,
     & 3725,     769,    649,    526,    358,    284,    219,
     &  153,     105,     67,     48,     35,     27,     27,
     &   35,      36,     17,   3276,   3389,    836,    770,
     &  723,     683,    477,    367,    244,    155,    124,
     &   99,      76,     68,     47,     41,     15,   3481,
     & 3049,     819,    984,    883,    749,    572,    453,
     &  371,     314,    208,    237,    262,    176,     14,
     & 3686,    2701,    829,    969,    953,    888,    805,
     &  614,     524,    429,    334,    275,    211,     13,
     & 3891,    2319,    663,    925,   1239,   1206,   1098,
     & 1006,     969,    789,    675,    534,     11,   4096,
     & 1858,     934,   1448,   1759,   1805,   1634,   1303,
     &  998,     866,      9,   4300,   1402,    903,   1391,
     & 2227,    2362,   2063,   1579,      7,   4505,    856,
     & 1286,    1601,   2390,   2881,      5,   4710,    283,
     & 1313,    1680,      3,   4915,      0,      3,  32767,
     &    0,     130,  40000,      3,      0,      0,      3,
     & 2363,       0,     11,   2396,   1821,     63,     45,
     &   33,      19,     12,     10,      7,      6,     13,
     & 2457,    2307,     82,     70,     60,     52,     33,
     &   26,      22,     20,     16,     15,     14,   2662,
     & 2792,     262,    235,    192,    152,     88,     68,
     &   50,      40,     35,     30,     25,     15,   2867,
     & 2941,     413,    527,    358,    266,    209,     73,
     &   52,      42,     33,     25,     20,     21,     15,
     & 3072,    2821,    710,    614,    557,    305,    248,
     &  200,     166,    129,     91,     69,     49,     42,
     &   13,    3276,   2454,    675,    589,    445,    494,
     &  524,     415,    346,    281,    196,    132,     11,
     & 3481,    2040,    605,    685,    723,    674,    603,
     &  410,     339,    308,     10,   3686,   1613,    586,
     &  615,     615,    682,    708,    658,    560,      8,
     & 3891,    1074,    591,    725,    841,   1013,   1106,
     &    6,    4096,    521,    880,   1122,   1201,      3,
     & 4300,       0,      3,  32767,      0,      0/

      END
