"""
Copyright 2018 Cognitive Scale, Inc. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""

import logging
import dill
import cuid
import tempfile
import pkg_resources
import jinja2
import docker
import urllib3
import re
from docker.errors import APIError, BuildError
from docker.utils.json_stream import json_stream
import shutil
from pathlib import Path
import jwt
from more_itertools import unique_everseen
from cortex.action import Action
from cortex_client import ActionClient, ConnectionClient
from cortex.utils import md5sum, log_message, is_notebook
from cortex.timer import Timer
from cortex.logger import getLogger
from cortex.exceptions import BuilderException
from typing import Optional

log = getLogger(__name__)

_action_daemon = 'daemon'
_action_function = 'function'
_action_job = 'job'

_base_image = 'continuumio/miniconda3:4.5.4'

_function_main_shim = """# Auto-generated by the Cortex Python SDK
def main(params):
    return {func_name}(params)
"""

_requirements_txt = """# Auto-generated by the Cortex Python SDK
%s

"""

_predict_globals = """
_model_key = "{model_key}"
_model = False
"""

_sklearn_predict_shim = 'predict_function_sklearn.py.j2'
_dockerfile_daemon = 'Dockerfile.daemon.j2'
_docker_daemon_port = 5000
_daemon_main = 'daemon.py.j2'
_function_main = 'func.py.j2'
_dockerfile_func = 'Dockerfile.func.j2'
_job_main = 'job.py.j2'
_dockerfile_job = 'Dockerfile.job.j2'
_cortex_sdk_version = pkg_resources.get_distribution("cortex-client").version
_private_registry_prefix = 'private-registry'

# Configure dill to 'recurse' dependencies when dumping objects
dill.settings['recurse'] = True


class ActionBuilder:

    def __init__(self, name: str, client: ActionClient, camel='1.0.0'):
        self._camel = camel
        self._name = name
        self._image = None
        self._image_prefix = None
        self._tag = cuid.slug()
        self._registry = None
        self._client = client
        self._token = client._serviceconnector.token
        decoded = jwt.decode(self._token, verify=False)
        self._username = decoded['sub']
        self._tenant_id = decoded['tenant']
        self._kind = 'blackbox'
        self._action_type = _action_function
        self._code_s = None
        self._model = None
        self._requirements = []
        self._conda_requirements = []

    def from_image(self, image: str):
        self._image = image
        return self

    def image_prefix(self, prefix):
        """
        Sets a prefix to use with the docker image name.  Ignored when the 'registry' property is set.
        :param prefix:
        :return:
        """
        self._image_prefix = prefix
        return self

    def registry(self, registry_uri: str):
        self._registry = registry_uri
        return self

    def kind(self, kind: str):
        self._kind = kind
        return self

    def daemon(self):
        self._action_type = _action_daemon
        return self

    def job(self):
        self._action_type = _action_job
        return self

    # DISABLED
    # def from_function(self, fn):
    #     func_name = fn.__name__ or 'f'
    #     self._code_s = dill.source.dumpsource(fn, alias=func_name, enclose=False)
    #     self._code_s += "\n%s" % _function_main_shim.format(func_name=func_name)
    #     return self

    def from_source(self, source_str: str):
        """
        Build an Action from source.  The Python source code must contain a function called 'main'.

        :param source_str:
        :return:
        """
        self._code_s = source_str
        return self

    def from_model(self, model, model_type="sklearn", x_pipeline=None, y_pipeline=None, target='y'):
        self._model = {'model': model, 'type': model_type, 'x_pipe': x_pipeline, 'y_pipe': y_pipeline, 'target': target}
        if model_type == 'sklearn':
            self.with_conda_requirements(['numpy', 'scipy', 'scikit-learn'])
            self.with_requirements(['numpy', 'scipy', 'scikit-learn'])
        else:
            raise BuilderException('Unsupported model type: {}'.format(model_type))

        return self

    def with_pip_freeze(self):
        self._requirements = list(self._pip_freeze())
        return self

    def with_requirements(self, requirements: list):
        self._requirements = [str(r) for r in requirements]
        return self

    def with_conda_requirements(self, requirements: list):
        self._conda_requirements = [str(r) for r in requirements]
        return self

    def build(self, **kwargs) -> Optional[Action]:
        _dry_run = kwargs.get('dry_run', False)

        log_message('Building Cortex Action ({}): {}'.format(self._action_type, self._name), log, logging.INFO)

        if self._action_type == _action_daemon:
            self._build_daemon(_dry_run)
        elif self._action_type == _action_job:
            self._build_job(_dry_run, **kwargs)
        else:
            # Assume function
            self._build_function(_dry_run)

        if _dry_run:
            return None

        return Action.get_action(self._name, self._client._serviceconnector)

    def _get_docker_repository(self, use_prefix=False):
        # 1. If the image name is explicitly set, we ignore the registry and prefix settings and also override our
        # default image naming scheme
        if self._image is not None:
            if self._image_prefix and use_prefix:
                return '{}/{}'.format(self._image_prefix, self._image)
            return self._image

        # 2. Use the image prefix in front of our default image name
        if self._image_prefix and use_prefix:
            return '{}/{}:{}'.format(self._image_prefix, self._name, self._tag)

        # 3. Use registry explicitly set by the user
        if self._registry is not None:
            # Prepend registry URI to default image name/tag.  Ignore image prefix.
            return '{}/{}:{}'.format(self._registry, self._name, self._tag)

        # 4. Use built-in private registry in Cortex 5
        api_endpoint = urllib3.util.parse_url(self._client._serviceconnector.url)
        registry_uri = '{}.{}'.format(_private_registry_prefix, '.'.join(api_endpoint.host.split('.')[1:]))

        return '{}/{}/{}:{}'.format(registry_uri, self._tenant_id, self._name.replace('/', '_'), self._tag)

    def _get_docker_auth(self):
        if self._image is None and self._registry is None and self._image_prefix is None:
            # Using the built-in private registry in Cortex 5.  Use the current JWT token for auth.
            return {'username': 'cogscale', 'password': self._client._serviceconnector.token}
        return None

    def _build_function(self, dry_run=False):
        if self._code_s:
            self._kind = 'python:3'
            if not dry_run:
                # Build new docker image with the given action source code
                self._deploy_function_image(self._code_s, 'main', '')

            # DISABLED: push a code archive
            # with tempfile.NamedTemporaryFile(mode='w+b', suffix='.zip') as f:
            #     self._prepare_source_archive(f, self._code_s)
            #     if not dry_run:
            #         self._client.deploy_action(self._name, self._kind, self._get_docker_repository(), f.name)
        elif self._model:
            self._kind = 'python:3'
            # Deploy from ML model object
            self._deploy_model(dry_run)
        else:
            # Deploy existing docker image
            if not dry_run:
                self._client.deploy_action(self._name, self._kind, self._get_docker_repository())

    def _build_daemon(self, dry_run=False):
        if self._code_s:
            if not dry_run:
                # Build a new docker image with the given action source code
                self._deploy_daemon_image(self._code_s, 'main', '')
        elif self._model:
            # Deploy from ML model object
            self._deploy_model(dry_run)
        else:
            # Deploy existing docker image
            if not dry_run:
                self._client.deploy_action(self._name, self._kind, self._get_docker_repository(),
                                           action_type=_action_daemon)

    def _build_job(self, dry_run=False, **kwargs):
        if self._code_s:
            if not dry_run:
                # Build new docker image with the given action source code
                self._deploy_job_image(self._code_s, 'main', '', **kwargs)
        elif self._model:
            # Deploy from ML model object
            self._deploy_model(dry_run, **kwargs)
        else:
            # Deploy existing docker image
            if not dry_run:
                vcpus = 1
                memory = 256
                backend_type = None

                if 'vcpus' in kwargs:
                    vcpus = int(kwargs['vcpus'])

                if 'memory' in kwargs:
                    memory = int(kwargs['memory'])

                if 'backend_type' in kwargs:
                    backend_type = kwargs['backend_type']

                self._client.deploy_action(self._name, self._kind, self._get_docker_repository(),
                                           action_type=_action_job, backend_type=backend_type, vcpus=vcpus,
                                           memory=memory)

    # DISABLED: push a code archive
    # def _prepare_source_archive(self, f, source_code):
    #     with ZipFile(f, 'w') as z:
    #         reqs = self._build_requirements(self._requirements)
    #         conda_reqs = self._build_requirements(self._conda_requirements)
    #         z.writestr('__main__.py', source_code)
    #         z.writestr('requirements.txt', reqs)
    #         z.writestr('conda_requirements.txt', conda_reqs)

    @staticmethod
    def _render_template(template_name, **kwargs):
        contents = ActionBuilder._read_resource(template_name)
        t = jinja2.Template(contents)
        return t.render(**kwargs)

    @staticmethod
    def _read_resource(name):
        file_path = pkg_resources.resource_filename('cortex.builder.resources', name)
        with open(file_path) as f:
            contents = f.read()
        return contents

    @staticmethod
    def _build_requirements(reqs):
        return _requirements_txt % ('\n'.join(list(unique_everseen(reqs))))

    @staticmethod
    def _render_and_save(to_dir, file_name, template_name, **kwargs):
        file_contents = ActionBuilder._render_template(template_name, **kwargs)
        ActionBuilder._dump_file(to_dir, file_name, file_contents)

    @staticmethod
    def _dump_file(to_dir, file_name, file_contents):
        with Path(to_dir, file_name).open('w') as f:
            f.write(file_contents)

    @staticmethod
    def _docker_build_and_push(build_dir, name, repository, auth_config=None):
        d = docker.from_env()
        cli = d.api

        log_message('Building Docker image {}...'.format(name), log, logging.INFO)
        try:
            with Timer() as t:
                stream = cli.build(path=build_dir, tag=name, forcerm=True)
                for obj in json_stream(stream):
                    if 'stream' in obj:
                        line = obj['stream'].strip()
                        if line and not line.startswith('--->'):
                            log_message(line, log, logging.INFO)
                    elif 'error' in obj:
                        msg = 'Docker error: {}'.format(obj['error'])
                        log_message(msg, log, logging.ERROR)
                        raise BuilderException(msg)

            log_message('Image {} built in {:2f} seconds'.format(name, t.interval), log, logging.INFO)

            cli.tag(image=name, repository=repository)
            log_message('Image {} tagged as {}'.format(name, repository), log, logging.INFO)
        except (APIError, ConnectionError) as e:
            log_message('Docker error: {}'.format(str(e)), log, logging.ERROR)
            raise BuilderException(str(e))

        log_message('Pushing image to remote repository...', log, logging.INFO)
        try:
            with Timer() as t:
                stream = cli.push(repository, auth_config=auth_config, stream=True)
                for obj in json_stream(stream):
                    if 'status' in obj:
                        match = re.search(
                            r'(^.*:\s+digest:\s+sha256:)([0-9a-f]+).*',
                            obj['status']
                        )
                        if match:
                            log_message(obj['status'], log, logging.INFO)
                    if 'errorDetail' in obj:
                        msg = 'Docker error: {}'.format(obj['errorDetail'])
                        log_message(msg, log, logging.ERROR)
                        raise BuilderException(msg)

            log_message('Image {} pushed in {:2f} seconds'.format(repository, t.interval), log, logging.INFO)

        except (APIError, ConnectionError) as e:
            log_message('Docker error: {}'.format(str(e)), log, logging.ERROR)
            raise BuilderException(str(e))

        log_message('Cleaning up...', log, logging.INFO)
        try:
            # Remove old versions of the image
            repo_no_tag = repository[0:repository.rindex(':')]
            images = d.images.list(name=repo_no_tag, filters={'before': repository})
            if images and len(images) > 0:
                for img in images:
                    log_message('Removing previous image: ' + img.short_id, log, logging.INFO)
                    d.images.remove(img.id, force=True)
        except (Exception) as e:
            log_message('Docker error during cleanup: {}'.format(str(e)), log, logging.ERROR)

    def _deploy_daemon_image(self, source, func_name, global_code):
        temp_dir = tempfile.mkdtemp()
        try:
            # Generate Dockerfile for daemon container
            self._render_and_save(temp_dir, 'Dockerfile', _dockerfile_daemon, base_image=_base_image,
                                  port=_docker_daemon_port, cortex_sdk_version=_cortex_sdk_version)

            # Generate action.py for the daemon service
            self._render_and_save(temp_dir, 'action.py', _daemon_main, action_globals=global_code,
                                  action_code=self._indent_block(source), func_name=func_name)

            # Generate pip requirements.txt
            reqs = self._build_requirements(self._requirements)
            self._dump_file(temp_dir, 'requirements.txt', reqs)

            # Generate conda requirements
            conda_reqs = self._build_requirements(self._conda_requirements)
            self._dump_file(temp_dir, 'conda_requirements.txt', conda_reqs)

            # Build container and push to registry
            self._docker_build_and_push(temp_dir, self._name, self._get_docker_repository(), self._get_docker_auth())

            # Deploy the Cortex Action
            self._client.deploy_action(self._name, self._kind, self._get_docker_repository(use_prefix=True),
                                       action_type=_action_daemon,
                                       port=str(_docker_daemon_port))
        finally:
            # cleanup temp
            shutil.rmtree(temp_dir, ignore_errors=True)

    def _deploy_function_image(self, source, func_name, global_code):
        temp_dir = tempfile.mkdtemp()
        try:
            # Generate Dockerfile for function container
            self._render_and_save(temp_dir, 'Dockerfile', _dockerfile_func, base_image=_base_image,
                                  cortex_sdk_version=_cortex_sdk_version)

            # Generate action.py for the function logic
            self._render_and_save(temp_dir, 'action.py', _function_main, action_globals=global_code,
                                  action_code=self._indent_block(source), func_name=func_name)

            # Generate pip requirements.txt
            reqs = self._build_requirements(self._requirements)
            self._dump_file(temp_dir, 'requirements.txt', reqs)

            # Generate conda requirements
            conda_reqs = self._build_requirements(self._conda_requirements)
            self._dump_file(temp_dir, 'conda_requirements.txt', conda_reqs)

            # Build container and push to registry
            self._docker_build_and_push(temp_dir, self._name, self._get_docker_repository(), self._get_docker_auth())

            # Deploy the Cortex Action
            self._client.deploy_action(self._name, self._kind, self._get_docker_repository(use_prefix=True))
        finally:
            # cleanup temp
            shutil.rmtree(temp_dir, ignore_errors=True)

    def _deploy_job_image(self, source, func_name, global_code, **kwargs):
        temp_dir = tempfile.mkdtemp()
        try:
            # Generate Dockerfile for job container
            self._render_and_save(temp_dir, 'Dockerfile', _dockerfile_job, base_image=_base_image,
                                  cortex_sdk_version=_cortex_sdk_version)

            # Generate action.py for the job logic
            self._render_and_save(temp_dir, 'action.py', _job_main, action_globals=global_code,
                                  action_code=self._indent_block(source), func_name=func_name)

            # Generate pip requirements.txt
            reqs = self._build_requirements(self._requirements)
            self._dump_file(temp_dir, 'requirements.txt', reqs)

            # Generate conda requirements
            conda_reqs = self._build_requirements(self._conda_requirements)
            self._dump_file(temp_dir, 'conda_requirements.txt', conda_reqs)

            # Build container and push to registry
            self._docker_build_and_push(temp_dir, self._name, self._get_docker_repository(), self._get_docker_auth())

            # Deploy the Cortex Action
            vcpus = 1
            memory = 256
            backend_type = None

            if 'vcpus' in kwargs:
                vcpus = int(kwargs['vcpus'])

            if 'memory' in kwargs:
                memory = int(kwargs['memory'])

            if 'backend_type' in kwargs:
                backend_type = kwargs['backend_type']

            self._client.deploy_action(self._name, self._kind, self._get_docker_repository(use_prefix=True),
                                       action_type=_action_job, backend_type=backend_type, vcpus=vcpus,
                                       memory=memory)
        finally:
            # cleanup temp
            shutil.rmtree(temp_dir, ignore_errors=True)

    def _dump_source(self, action_source):
        print('--- action.py ---')
        print(action_source)
        print('--- requirements.txt ---')
        print(self._build_requirements(self._requirements))
        print('--- conda_requirements.txt ---')
        print(self._build_requirements(self._conda_requirements))
        print('-' * 30)

    @staticmethod
    def _indent_block(s: str, spaces=4):
        lines = s.split('\n')
        lines = map(lambda line: (' ' * spaces) + line, lines)
        return '\n'.join(lines)

    def _deploy_model(self, dry_run=False, **kwargs):
        content_client = ConnectionClient(
            self._client._serviceconnector.url,
            2,
            self._client._serviceconnector.token
        )

        temp_path = None
        try:
            # Upload model version to managed content
            with tempfile.NamedTemporaryFile(mode='w+b', delete=False) as temp:
                temp_path = temp.name
                dill.dump(self._model, temp)

            md5 = md5sum(temp.name)
            model_key = '/cortex/models/%s/%s.pk' % (self._name, md5)

            if not content_client.exists(model_key):
                log_message('model version not found, pushing to remote storage: ' + model_key, log)
                with open(temp.name, 'rb') as f:
                    content_client.uploadStreaming(key=model_key, content_type='application/octet-stream', stream=f)
        finally:
            shutil.rmtree(temp_path, ignore_errors=True)

        # Deploy predict action
        _predict_source = self._render_template(_sklearn_predict_shim)
        _globals = _predict_globals.format(model_key=model_key[1:])

        if self._action_type == _action_daemon:
            if dry_run:
                source = self._render_template(_daemon_main, action_globals=_globals, action_code=_predict_source,
                                               func_name='predict')
                self._dump_source(source)
                return

            self._deploy_daemon_image(_predict_source, 'predict', _globals)
        elif self._action_type == _action_job:
            if dry_run:
                source = self._render_template(_job_main, action_globals=_globals, action_code=_predict_source,
                                               func_name='predict')
                self._dump_source(source)
                return

            self._deploy_job_image(_predict_source, 'predict', _globals, **kwargs)
        else:
            if dry_run:
                source = self._render_template(_function_main, action_globals=_globals, action_code=_predict_source,
                                               func_name='predict')
                self._dump_source(source)
                return

            self._deploy_function_image(_predict_source, 'predict', _globals)

            # DISABLED: push a code archive
            # Assume function with source archive
            # with tempfile.NamedTemporaryFile(mode='w+b', suffix='.zip') as f:
            #     self._prepare_source_archive(f, source)
            #     self._client.deploy_action(self._name, self._kind, self._get_docker_repository(), f.name)

    @staticmethod
    def _pip_freeze():
        try:
            from pip._internal.operations import freeze
        except ImportError:  # pip < 10.0
            from pip.operations import freeze

        return freeze.freeze()
