# Copyright (C) 2015 Stefan C. Mueller


from pydron.backend import worker

from twisted.internet import defer, task

import logging
import twistit
import anycall
import pickle
from pydron.interpreter import traverser
logger = logging.getLogger(__name__)
    
    
class Scheduler(object):
    """
    The scheduler brings the tasks generated by the :class:`Traverser` and the
    :class:`worker.Pool` of workers together. The actual scheduling strategy
    is configurable.
    """
    
    class _Job(object):
        def __init__(self, scheduler, g, tick, task, inputs):
            self.scheduler = scheduler
            self.g = g
            self.tick = tick
            self.task = task
            self.inputs = inputs
            self.result = defer.Deferred(self._cancel)
            
        def _cancel(self, d):
            self.scheduler._cancel_job(self)
            
        def __repr__(self):
            return "Job(%r, %s)" % (self.tick, type(self.task))
    
    def __init__(self, pool, strategy):
        """
        :param strategy: :class:`SchedulingStrategy`.
        """
        self._pool = pool
        self._strategy = strategy
        self._job_queue = set()
        
        #: job -> deferred to cancel evaluation, worker
        self._currently_running = {}

        #: ValueRefs that were used as inputs to
        #: tasks with a syncpoint. The values referenced
        #: by them could now change everytime a syncpoint
        #: task is executed.
        self._leaked_valuerefs = set()
        
        self._master_worker = None
        
        self._statusreport_interval = 2
        self._statusreport_loop = task.LoopingCall(self.log_statusreport)
    
    @twistit.yieldefer
    def schedule_refinement(self, g, tick, task, inputs):
        """
        Refine the graph.
        
        :param g: Graph to be refined.
        :param tick: Tick of the task being refined.
        :param task: Task being refined.
        :param inputs: port -> valueref for the refiner ports.
        """
        me = anycall.RPCSystem.default.local_worker  #@UndefinedVariable
        meremote = anycall.RPCSystem.default.local_remoteworker  #@UndefinedVariable
        
        logger.debug("Refinement request for %r. Transferring inputs to %r" % (tick, me))
        
        in_ports = list(inputs.iterkeys())
        
        # First, get all the required data here.
        data_transfers = []
        for port in in_ports:
            valueref = inputs[port]
            source = self._strategy.choose_source_worker(valueref, meremote)
            
            if hasattr(task, 'refiner_reducer'):
                reducer = task.refiner_reducer.get(port, None)
            else:
                reducer = None
        
            if reducer is None:
                # Get the complete value.
                # We could use a `lambda x:x` reducer, but this way
                # the value ends up in our store, so we might not have
                # to transfer it again.
                d = me.fetch_from(source, valueref.valueid)
                def fetched(_):
                    return me.get_value(valueref.valueid)
                d.addCallback(fetched)
            else:
                d = source.reduce(valueref.valueid, reducer)
            
            data_transfers.append(d)
            
        transfer_results = yield defer.DeferredList(data_transfers, fireOnOneErrback = True)
        
        input_values = {}
        for i, port in enumerate(in_ports):
            _, value = transfer_results[i]
            input_values[port] = value
            
        logger.info("Refining %r..." % tick)
        
        # Lets change the graph
        task.refine(g, tick, input_values)
    
        logger.info("Refining of %r completed." % tick)
    
    def schedule_evaluation(self, g, tick, task, inputs):
        """
        Evaluate a task.
        
        :param g: Graph containing the task
        :param tick: Tick of the task being evaluated.
        :param task: Task being evaluated.
        :param inputs: port -> valueref for the input ports.
        """
        job = self._Job(self, g, tick, task, inputs)
        logger.debug("Job added to queue: %r" % job)
        self._job_queue.add(job)
        
        self._schedule()

        return job.result
       
    def _schedule(self):
        """
        Call this whenever a worker becomes idle or a new job is added to the queue.
        """

        if self._job_queue:
            pairs = list(self._strategy.assign_jobs_to_workers(list(self._job_queue)))

            for workr, job, callback in pairs:
                logger.debug("Job %r scheduled for %r." % (job, workr))
   
                self._job_queue.remove(job)
                self._handle_pair(workr, job, callback)

            
    def _handle_pair(self, workr, job, callback=False):
        """
        :param quick: If false, the worker will be added to the idle list
          once the job has completed. Set this to true for jobs that are started
          when the worker wasn't idle.
        """
        
        props = job.g.get_task_properties(job.tick)
        nosend_ports = props.get("nosend_ports", None)
        syncpoint = props.get("syncpoint", False)
        
        details = []
        if syncpoint:
            details.append("syncpoint")
        logger.info("Running job %r on %r. %s" %(job, workr, " ".join(details)))
        
        # Decide where we'll get the inputs from
        prepared_inputs = {}
        for port, valueref in job.inputs.iteritems():
            source = self._strategy.choose_source_worker(valueref, workr)
            prepared_inputs[port] = (valueref.valueid, source)
            
        if self._master_worker is None:
            self._master_worker = anycall.RPCSystem.default.local_remoteworker #@UndefinedVariable
        runs_on_master = workr is self._master_worker
            
        # Run
        d = workr.evaluate(job.tick, 
                           job.task, 
                           prepared_inputs, 
                           nosend_ports=nosend_ports,
                           fail_on_unexpected_nosend=not runs_on_master)
        
        def catch_pickleerror(reason, job, workr):
            """
            Treat pickling errors as errors thrown by the user's code.
            """
            if reason.check(pickle.PickleError):
                return traverser.EvalResult(reason, 0)
            else:
                return reason
            
        def on_success(evalresult, job, workr):
            logger.info("Job %r completed." % job)
            self._stopped_running(job)
            
            if callback is not None:
                callback(job, workr, False)
            
            
            if evalresult.transfer_results:
                for port, transfer_result in evalresult.transfer_results.iteritems():
                    source = prepared_inputs[port][1]
                    dest = workr
                    self._pool.fire_transmission_time(source, dest, transfer_result.bytecount, transfer_result.duration)
            
            # The inputs are now available on the workr
            # TODO: actually they are available on workr before the job has completed
            for valueref in job.inputs.itervalues():
                valueref.add_worker(workr)
                
            if syncpoint:
                # All the inputs have now potentially leaked into
                # global variables.
                for valueref in job.inputs.itervalues():
                    self._leaked_valuerefs.add(valueref)
            
            if isinstance(evalresult.result, dict):
                # Create ValueRefs for the ValueIds we got back from the worker
                assert all(isinstance(vid, worker.ValueId) for vid in evalresult.result.itervalues())
                
                outs = {}
                for port, valueid in evalresult.result.iteritems():
         
                    if port in evalresult.datasizes:
                        datasize = evalresult.datasizes[port]
                        pickle_support = True
                    else:
                        datasize = None
                        pickle_support = False
                        logger.info("Output %r of %r cannot be pickled." % (port, job))
                        
                    valref = worker.ValueRef(valueid, pickle_support, workr) 
                    valref.datasize = datasize
                    
                    outs[port] = valref
                        
                evalresult.result = outs
                
            if syncpoint:
                leaks_handled_d = self._leaked_valuerefs_affected()
            else:
                leaks_handled_d = defer.succeed(None)
                
            def leaks_handled(_):
                self._schedule() 
                job.result.callback(evalresult)
                
            leaks_handled_d.addCallback(leaks_handled)
            return leaks_handled_d
        
        def on_fail(reason, job, workr):
            self._stopped_running(job)
            
            if reason.check(worker.NoPickleError):
                e = reason.value
                logger.error("The outputs of %r failed to pickle: %r" %
                             (job, e))
                job.g.set_task_property(job.tick, "masteronly", True)
                logger.error("Adding %r to queue for re-try on master." % job)
                
                if callback is not None:
                    callback(job, workr, False)
                
                self._job_queue.add(job)
                self._schedule()
                        
            elif reason.check(defer.CancelledError) and job not in self._currently_running:
                logger.info("Job %r cancelled." % job)
                
                if callback is not None:
                    callback(job, workr, False)
                
                # We cancelled this job
                self._schedule()
                        
            else:
                logger.error("Job %r on %r failed. Assuming Worker is faulty: %s" % (job, workr, reason.getTraceback()))
                
                if callback is not None:
                    callback(job, workr, True)
                
                # Something went wrong with the worker.
                logger.error("Killing %r." % workr)
                d = workr.kill()
                self._pool.remove_worker(workr)
                def unhandled(failure):
                    logger.error("Failed to kill %r: " % (workr, failure.getTraceback()))
                d.addErrback(unhandled)
                
                if not self._pool.get_workers():
                    logger.error("Killed last worker.")
                    job.result.errback(reason)
                else:
                
                    # Put the job back, so that we can try again
                    # on a different worker.
                    logger.error("Adding %r to queue for re-try." % job)
                    self._job_queue.add(job)
                    self._schedule()
        
        self._started_running(job, workr, d)
        d.addErrback(catch_pickleerror, job, workr)
        d.addCallbacks(on_success, on_fail, callbackArgs=(job,workr), errbackArgs=(job,workr))
        
        def unhandled(failure):
            logger.error(failure.getTraceback())
        d.addErrback(unhandled)
        

    def _cancel_job(self, job):
        if job in self._job_queue:
            self._job_queue.remove(job)
        elif job in self._currently_running:
            d, _ = self._currently_running[job]
            d.cancel()
            self._stopped_running(job)
            
    def _leaked_valuerefs_affected(self):
        
        if self._master_worker is None:
            self._master_worker = anycall.RPCSystem.default.local_remoteworker #@UndefinedVariable
        
        copies = []
        for valueref in self._leaked_valuerefs:
            
            old_valueid = valueref.valueid
            new_valueid = worker.ValueId(old_valueid.endpoint, 
                                         old_valueid.nicename)
            
            workers = list(valueref.get_workers())
            
            # We have the modified value on the master, so lets make
            # sure the master knows the new valueid of it.
            if self._master_worker in workers:
                d = self._master_worker.copy(old_valueid, new_valueid)
                copies.append(d)
                have_master = True
            else:
                have_master = False
                raise ValueError("syncpoint values should always be on the master.")
                
            # Free the old valueid from all workers
            # and remember that they don't have it anymore.
            for workr in workers:
                d = workr.free(old_valueid)
                
                def on_err(reason, old_valueid, workr):
                    logger.error("Failed to free %r from %r." %
                                 (old_valueid, workr))
                d.addErrback(on_err, old_valueid, workr)

                valueref.remove_worker(workr)
            
            # The master has the new value
            if have_master:
                valueref.add_worker(self._master_worker)
        
            valueref.valueid = new_valueid
            
        return defer.DeferredList(copies, fireOnOneErrback=True)
         
    def _started_running(self, job, workr, d):
        self._currently_running[job] = (d, workr)
        self._currently_running_changed()
        
    def _stopped_running(self, job):
        del self._currently_running[job]
        self._currently_running_changed()
    
    def _currently_running_changed(self):
        if self._currently_running and not self._statusreport_loop.running:
            self._statusreport_loop.start(self._statusreport_interval, now=False)
            
        if not self._currently_running and self._statusreport_loop.running:
            self._statusreport_loop.stop()
            
    def log_statusreport(self):
        logger.info("Status Report: Currently running jobs:")
        logger.info("--------------------------------------")
        for job, (_, workr) in self._currently_running.iteritems():
            logger.info("   %r on %r" % (job, workr))
            
