#!/usr/bin/env python
#  -*- mode: python; indent-tabs-mode: nil; -*- coding: utf-8 -*-

"""

parts.py

Copyright 2010-2012 by Marcello Perathoner

Distributable under the GNU General Public License Version 3 or newer.

New and tweaked transformers.

"""

import copy
import datetime
import re
import os
import sys
import collections
import functools
import urlparse

from docutils import nodes
import docutils.transforms
import docutils.transforms.parts
from docutils.transforms import components

from epubmaker.mydocutils import broken
from epubmaker.mydocutils import nodes as mynodes
from epubmaker.lib.Logger import error, info, debug, warn
from epubmaker import Unitame
from epubmaker.lib.DublinCore import DublinCore

# pylint: disable=W0142

def copy_and_filter (node, document):
    """ Return a copy of a title, with references, images, etc. removed. """
        
    visitor = ContentsFilter (document)
    node.walkabout (visitor)
    return visitor.get_entry_text ()


class ContentsFilter (nodes.TreeCopyVisitor):

    def get_entry_text (self):
        return self.get_tree_copy ().children

    def ignore_node (self, node):
        raise nodes.SkipNode
    
    def ignore_node_but_process_children (self, node):
        raise nodes.SkipDeparture

    def visit_image (self, node):
        if node.hasattr ('alt'):
            self.parent.append (nodes.Text (node['alt']))
        raise nodes.SkipNode

    visit_newline = nodes.TreeCopyVisitor.default_visit
    depart_newline = nodes.TreeCopyVisitor.default_departure
    
    visit_citation_reference = ignore_node
    visit_footnote_reference = ignore_node
    visit_raw = ignore_node

    visit_interpreted = ignore_node_but_process_children
    visit_problematic = ignore_node_but_process_children
    visit_reference = ignore_node_but_process_children
    visit_target = ignore_node_but_process_children


class TocPageNumberTransform (docutils.transforms.Transform):
    """ Finds the page number all sections, tables and figures are in. """

    default_priority = 715 # before contents transform
    
    def apply (self, **kwargs):
        pageno = '' # pageno is actually a string
        
        for n in self.document.traverse ():
            if isinstance (n, nodes.target) and 'pageno' in n:
                pageno = n['pageno']
                # debug ("pageno: >%s<" % self.pageno)
            elif isinstance (n, (nodes.section, nodes.figure, nodes.table)):
                n['pageno'] = pageno


class TocEntryTransform (docutils.transforms.Transform):
    """ Moves data of pending node onto next header. """

    default_priority = 717 # before Contents transform
    
    def apply (self, **kwargs):
        # debug ("TocEntryTransform %s" % repr (self.startnode.details))

        iter_ = self.startnode.traverse (
            lambda x: isinstance (x, (nodes.title, nodes.caption)), ascend = 1, descend = 1)

        details = self.startnode.details
        if len (iter_):
            title = iter_[0]
            title['toc_entry'] = details['content']
                
            # copy depth
            if 'toc_depth' in details:
                section = title.parent
                if isinstance (section, nodes.section):
                    section['toc_depth'] = details['toc_depth']
                    # debug ("Setting toc_depth: %d" % section['toc_depth'])

        self.startnode.parent.remove (self.startnode)
            

class ContentsTransform (docutils.transforms.Transform):
    """ A modified contents transform that obeys contents-depth directives. """

    default_priority = 720

    def __init__ (self, document, startnode = None):
        docutils.transforms.Transform.__init__ (self, document, startnode)
        self.depth = startnode.details.get ('depth', sys.maxint) if startnode else sys.maxint
        self.toc_depth = sys.maxint
        self.use_pagenos = 'page-numbers' in startnode.details
        self.maxlen = 0 # length of longest page no.

    def apply(self):
        details = self.startnode.details
        if 'local' in details:
            startnode = self.startnode.parent.parent
            while not (isinstance (startnode, nodes.section)
                       or isinstance (startnode, nodes.document)):
                # find the ToC root: a direct ancestor of startnode
                startnode = startnode.parent
        else:
            startnode = self.document

        contents = self.build_contents (startnode)
        if len (contents):
            self.startnode.replace_self (contents)
        else:
            self.startnode.parent.parent.remove (self.startnode.parent)

    def build_contents (self, node, level=0):
        # debug ('build_contents level %d' % level)
    
        details = self.startnode.details
        backlinks = details.get ('backlinks', self.document.settings.toc_backlinks)
        try:
            toc_id = self.startnode.parent['ids'][0]
        except:
            toc_id = None

        entries = []
        for n in node:
            if isinstance (n, nodes.section):
                if 'toc_depth' in n:
                    self.toc_depth = n['toc_depth']
                    # debug ("New toc_depth: %d" % self.toc_depth)
                if level < self.depth and level < self.toc_depth:
                    subsects = self.build_contents(n, level + 1)
                    title = n[0]
                    if not isinstance (title, nodes.title):
                        continue
                    # debug ('title: %s level: %d depth: %d' % (title, level, self.toc_depth))

                    pagenos = []
                    if self.use_pagenos and 'pageno' in n:
                        self.maxlen = max (self.maxlen, len (n['pageno']))
                        inline = nodes.inline ('', ' ' + n['pageno'])
                        inline['classes'].append ('toc-pageref')
                        pagenos.append (inline)
                    
                    if 'toc_entry' in title:
                        container = title['toc_entry']
                        if container is None: # suppress toc entry if emtpy
                            continue
                        # debug ("Setting TOC entry")
                        entrytext = copy_and_filter (container, self.document)
                    else:
                        entrytext = copy_and_filter (title, self.document)
                        
                    reference = nodes.reference (
                        '', '', refid = n['ids'][0], *entrytext)
                    ref_id = self.document.set_id (reference)

                    entry = nodes.paragraph ('', '', reference, *pagenos)
                    item = nodes.list_item ('', entry)
                    item['refid'] = n['ids'][0]
                    item['classes'].append ('toc-entry')
                    if 'level' in title:
                        item['level'] = title['level']
                        item['classes'].append ('level-%d' % (title['level']))
                    if (backlinks in ('entry', 'top')
                         and title.next_node (nodes.Referential) is None):
                        if backlinks == 'entry':
                            title['refid'] = ref_id
                        elif backlinks == 'top' and toc_id is not None:
                            title['refid'] = toc_id
                    item += subsects
                    entries.append (item)
        if entries:
            return nodes.bullet_list ('', *entries, **{'classes': ['compact', 'toc-list'],
                                                       'enumtype': 'none',
                                                       'pageno_maxlen': self.maxlen})
        else:
            return []


class ListOfAnythingTransform (docutils.transforms.Transform):
    """ Create a List Of Figures or List Of Tables etc. """

    default_priority = 718 # before empty section remover

    def __init__ (self, document, startnode = None):
        docutils.transforms.Transform.__init__ (self, document, startnode)
        self.maxlen = 0 # length of longest page no.
    
    def apply (self, **kwargs):
        entries = []
        details = self.startnode.details
        
        self.backlinks = details.get ('backlinks',
                                      self.document.settings.toc_backlinks)
        self.use_pagenos = 'page-numbers' in details
        directive_name = details.get ('directive_name', '')
        
        condition = None
        if 'selector' in details:
            condition = mynodes.node_selector (details['selector'])
        elif directive_name == 'lof':
            condition = nodes.figure
        elif directive_name == 'lot':
            condition = nodes.table
        
        if not condition:
            self.document.reporter.warning (
                "directive %s needs an option 'selector'" % directive_name,
                base_node=self.startnode.parent)
            raise docutils.transform.TransformError

        if 'local' in details:
            startnode = self.startnode.parent.parent
            while not (isinstance (startnode, (nodes.section, nodes.document))):
                # find the ToC root: a direct ancestor of startnode
                startnode = startnode.parent
        else:
            startnode = self.document

        try:
            toc_id = self.startnode.parent.parent['ids'][0]
        except:
            toc_id = None

        for node in startnode.traverse (condition):
                
            title = list (node.traverse (nodes.caption) + node.traverse (nodes.title))
            if len (title) != 1:
                # cannot put anonymous X in list of X
                continue
            
            title = title[0]
            
            pagenos = []
            if self.use_pagenos and 'pageno' in node:
                self.maxlen = max (self.maxlen, len (node['pageno']))
                inline = nodes.inline ('', ' ' + node['pageno'])
                inline['classes'].append ('toc-pageref')
                pagenos.append (inline)
                    
            if 'toc_entry' in title:
                container = title['toc_entry']
                if container is None: # suppress toc entry if emtpy
                    # debug ("Suppressing TOC entry")
                    continue
                # debug ("Setting TOC entry")
                entrytext = copy_and_filter (container, self.document)
            else:
                entrytext = copy_and_filter (title, self.document) # returns list of nodes

            reference = nodes.reference ('', '', *entrytext, refid = node['ids'][0])
            ref_id = self.document.set_id (reference) # target for backlink
                        
            list_item = nodes.list_item ('', nodes.inline ('', '', reference, *pagenos))
            list_item['refid'] = node['ids'][0]
            list_item['level'] = 2
            list_item['classes'].append ('level-2')
            list_item['classes'].append ('toc-entry')

            if (self.backlinks in ('entry', 'top')
                 and title.next_node (nodes.Referential) is None):
                if self.backlinks == 'entry':
                    title['refid'] = ref_id
                elif self.backlinks == 'top' and toc_id is not None:
                    title['refid'] = toc_id
                    
            entries.append (list_item)

        if entries:
            contents = nodes.bullet_list ('', *entries, **{'classes': ['compact', 'toc-list'],
                                                           'enumtype': 'none',
                                                           'pageno_maxlen': self.maxlen})
            self.startnode.replace_self (contents)
        else:
            self.startnode.parent.remove (self.startnode)
            # self.startnode.parent.parent.remove (self.startnode.parent)
            

class FootnotesDirectiveTransform (docutils.transforms.Transform):
    """
    Collects footnotes into this section.

    The 'footnotes::' directive creates a section and puts this
    pending transform into it.  This transform either moves all
    footnotes into the generated section or removes this section from
    non-HTML formats.
    
    """

    # run this before contents transform. contents should not grab
    # this section's title because it can disappear.
    
    default_priority = 718 # before empty section remover
    
    def apply (self, **kwargs):
        pending = self.startnode
        section = pending.parent
        section.remove (pending)

        writer = self.document.transformer.components['writer']
        if writer.supports ('html'):
            group = mynodes.footnote_group ()
            section += group

            for footnote in self.document.traverse (nodes.footnote):
                footnote.parent.remove (footnote)
                group += footnote
        #else:
        #    section.parent.remove (section)


class XetexFootnotesTransform (docutils.transforms.Transform):
    """
    Moves footnote bodies near to footnote references.

    Inserts the footnote body immediately after the footnote reference
    in Xetex format.
    
    """

    default_priority = 717
    
    def apply (self, **kwargs):
        # reversed () makes footnotes inside of footnotes come out in
        # the right order
        for footnote_reference in reversed (
            self.document.traverse (nodes.footnote_reference)):
            if 'refid' not in footnote_reference:
                # further references to footnote
                continue
            # pull in footnote body
            footnote = self.document.ids [footnote_reference['refid']]
            footnote_section = footnote.parent
            footnote_section.remove (footnote)
            
            parent = footnote_reference.parent
            index = parent.index (footnote_reference)
            parent.insert (index + 1, footnote)

            # if the footnote is inside a table environment, move the
            # footnote behind the table
            while (parent):
                table = parent
                parent = parent.parent
                if isinstance (table, (nodes.table, nodes.title, nodes.footnote)):
                    footnote.parent.remove (footnote)
                    index = parent.index (table)
                    parent.insert (index + 1, footnote)
            
        
class EmptySectionRemover (docutils.transforms.Transform):
    """
    Removes this section if it contains only a title or less.
    Use for footnotes or loa sections.

    """

    # run this before contents transform. contents should not grab
    # this section before we make it disappear.
    
    default_priority = 719 # before contents transform
    
    def apply (self, **kwargs):
        pending = self.startnode
        section = pending.parent
        if not isinstance (section, nodes.section):
            return
        # print "\n\n", section
        index = section.index (pending)
        container = section[index + 1]
        # title = section[index - 1] if index > 0 else None
        title = section[0]
        if not isinstance (title, nodes.title):
            title = None
        section.remove (pending)

        if len (container) == 0:
            section.remove (container)
            if title is not None: 
                section.remove (title)
            if len (section) == 0:
                section.parent.remove (section)


class PageNumberMoverTransform (docutils.transforms.Transform):
    """ Moves paragraphs that contain only one page number into the
    next paragraph. """

    default_priority = 721 # after contents transform
    
    def apply (self, **kwargs):
        for target in self.document.traverse (nodes.target):
            if (isinstance (target.parent, nodes.paragraph) and len (target.parent) == 1):
                # move onto next appropriate node
                for next_node in target.traverse (nodes.TextElement, include_self = 0, ascend = 1):
                    if (not isinstance (next_node, (nodes.Structural, nodes.Special))):
                        target.parent.parent.remove (target.parent)
                        next_node.insert (0, target)
                        break
            
                    
class DropCapTransform (docutils.transforms.Transform):
    """ Inserts a dropcap into the following paragraph. """

    default_priority = 719 # run before Contents Transform
    
    def apply (self, **kwargs):
        iter_ = self.startnode.traverse (nodes.paragraph, siblings = 1)

        if len (iter_):
            para = iter_[0]
            iter_ = para.traverse (nodes.Text)
            details = self.startnode.details

            if len (iter_):
                textnode = iter_[0]
                charnode = spannode = restnode = None

                char = details['char']
                if not textnode.startswith (char):
                    error ("Dropcap: next paragraph doesn't start with: '%s'." % char)
                    return

                span = details.get ('span', '')
                if not textnode.startswith (span):
                    error ("Dropcap: next paragraph doesn't start with: '%s'." % span)
                    return
                if span and not span.startswith (char):
                    error ("Dropcap: span doesn't start with: '%s'." % char)
                    return
                if span == char:
                    span = ''

                if span:
                    # split into char/span/rest
                    restnode = nodes.Text (textnode.astext ()[len (span):])
                    spannode = nodes.inline ()
                    spannode.append (nodes.Text (textnode.astext ()[len (char):len (span)]))
                    spannode['classes'].append ('dropspan')
                else:
                    # split into char/rest
                    restnode = nodes.Text (textnode.astext ()[len (char):])
                    spannode = nodes.inline ('', '')
                    spannode['classes'].append ('dropspan')
                
                if (not self.document.settings.no_images) and ('image' in details):
                    charnode = nodes.image ()
                    charnode['uri'] = details['image']
                    charnode['alt'] = char
                    # debug ("Inserting image %s as dropcap." % uri)
                else:
                    charnode = nodes.inline ()
                    charnode.append (nodes.Text (char))
                    # debug ("Inserting char %s as dropcap." % char)

                charnode['classes'].append ('dropcap')
                charnode.attributes.update (details)

                para.replace (textnode, [charnode, spannode, restnode])

        self.startnode.parent.remove (self.startnode)


class CharsetTransform (docutils.transforms.Transform):
    """
    Translates text into smaller charset.

    This does not change the encoding, it just emulates all characters
    with characters from the smaller charset.

    """

    default_priority = 899 # last
    
    def apply (self, **kwargs):
        if self.document.settings.encoding != 'utf-8':
            charset = self.document.settings.encoding
            del Unitame.unhandled_chars[:]

            for n in self.document.traverse (nodes.Text):
                text  = n.astext ()
                text2 = text.encode (charset, 'unitame').decode (charset)
                if text != text2:
                    n.parent.replace (n, nodes.Text (text2)) # cannot change text nodes

            if Unitame.unhandled_chars:
                error ("unitame: unhandled chars: %s" % u", ".join (set (Unitame.unhandled_chars)))
            

class DefaultPresentation (docutils.transforms.Transform):
    """
    Styles nodes with default presentation.

    This way we avoid having to handle each of these elements separately
    in the writers and we can just treat them like every other styled node.

    """
    
    default_priority = 730 # after ..contents:: sections are generated

    default_presentation = {
        nodes.emphasis:        ['italics'],
        nodes.strong:          ['bold'],
        nodes.title_reference: ['italics'],
        nodes.literal:         ['monospaced'],
        # nodes.caption:         ['italics'],
        nodes.subscript:       ['subscript'],
        nodes.superscript:     ['superscript'],
        }

    def apply (self):
        wanted_nodes = tuple (self.default_presentation.keys ())
        for node in self.document.traverse (lambda n: isinstance (n, wanted_nodes)):
            node['classes'] += self.default_presentation[type (node)]


class StyleTransform (docutils.transforms.Transform):
    """
    Add classes to elements.

    Works in a way similar to CSS, though you can select only on
    element and class.

    Works on all following elements in the section it is used, if used
    before the first section works on the rest of the document.
    
    """

    default_priority = 731 # after default presentation
    
    def apply (self, **kwargs):
        pending  = self.startnode
        details  = pending.details
        selector = details.get ('selector', '')

        if selector == 'document':
            # look at document node only
            node_list = [self.document]
        else:
            # traverse all following nodes and their children
            node_list = pending.traverse (
                mynodes.node_selector (selector), siblings = 1, include_self = 0)
            
        classes    = frozenset ([c for c in details.get ('class', []) if not c.startswith ('-')])
        rmclasses  = frozenset ([c[1:] for c in details.get ('class', []) if c.startswith ('-')])

        classes = frozenset (details.get ('class', []))
        rmclasses = frozenset ()
        
        for n in node_list:
            n['classes'] = list ((set (n['classes']) | classes) - rmclasses)
            for a in ('align', 'width'):
                if a in details:
                    n.setdefault (a, details[a])
                
        pending.parent.remove (pending)
        

class AlignTransform (docutils.transforms.Transform):
    """
    Transforms align attribute into align-* class.

    """
    
    default_priority = 801 # late

    def apply (self):
        for body in self.document.traverse (nodes.Body):
            if 'align' in body:
                body['classes'].append ('align-%s' % body['align'])


class NewlineTransform (docutils.transforms.Transform):
    """
    Inserts newline node for newlines.

    """

    default_priority = 897 # next to last

    def apply (self, **kwargs):
        for node in list (self.document.traverse (nodes.Text)):
            text = node.astext ()
            if '\n' in text:
                newnodes = []
                for line in text.splitlines ():
                    newnodes.append (nodes.Text (line))
                    newnodes.append (mynodes.newline ())
                if not text.endswith ('\n'):
                    newnodes.pop () # destroy last newline
                node.parent.replace (node, newnodes)


class TextTransform (docutils.transforms.Transform):
    """
    Implements CSS text-transform.

    """

    default_priority = 897 # next to last

    smartquotes_map = {
        0x0027: u'’',
        # 0x0022: u'”',
        }
    
    re_quotes_thin_space = re.compile (ur'([“„‟’])([‘‚‛”])')
    
    def apply (self, **kwargs):
        self.recurse (self.document, {})

    # FIXME this has been obsoleted by class inheritance

    def recurse (self, node, text_transform):
        if isinstance (node, nodes.Text):
            if len (text_transform) > 0:
                oldtext = text = node.astext ()
                if text_transform.get ('uppercase', False):
                    text = text.upper ()
                if text_transform.get ('smartquotes', False):
                    text = text.translate (self.smartquotes_map)
                    # insert thin space between quotes
                    text = self.re_quotes_thin_space.sub (ur'\1 \2', text)
                if text != oldtext:
                    node.parent.replace (node, nodes.Text (text)) # cannot change text nodes
            return

        ntt = text_transform.copy ()
        classes = node['classes']
        
        if 'text-transform-uppercase' in classes:
            ntt['uppercase'] = True
        elif 'text-transform-smartquotes' in classes:
            ntt['smartquotes'] = True
        elif 'text-transform-none' in classes:
            ntt['uppercase'] = False
            ntt['smartquotes'] = False

        if 'white-space-pre-line' in classes:
            ntt['pre-line'] = True
            
        for child in node:
            self.recurse (child, ntt)


class TitleLevelTransform (docutils.transforms.Transform):
    """
    Adds some useful classes to sections and titles.

    Add `level-N` to `section`, `title` and `subtitle` nodes.

    Add `document-title` or `section-title`.

    Add `title` or `subtitle`.

    Add `with-subtitle`.

    """
    
    default_priority = 360 # after title promotion

    def apply (self, **kwargs):
        self.recurse (self.document, 1)

    def recurse (self, parent, level):
        for node in parent:
            if isinstance (node, nodes.Text):
                continue

            if isinstance (node, (nodes.title, nodes.subtitle)):
                nclass  = node.__class__.__name__
                pclass  = parent.__class__.__name__
                classes = [nclass,
                           '%s-%s' % (pclass, nclass),
                           'level-%d' % level]
                if (isinstance (node, (nodes.title)) and
                    len (parent) >= 2 and
                    isinstance (parent[1], nodes.subtitle)):
                    classes.append ('with-subtitle')
                node['classes'].extend (classes)
                node['level'] = level
            
            if isinstance (node, nodes.section):
                node['classes'].append ('level-%d' % (level + 1))
                node['level'] = level + 1

                self.recurse (node, level + 1)
            else:
                self.recurse (node, level)


class FirstParagraphTransform (docutils.transforms.Transform):
    """
    Mark first paragraphs.

    With indented paragraphs, the first paragraph following a
    vertical space should not be indented. This transform tries
    to figure out which paragraphs should not be indented.
    
    Add the classes `pfirst` and `pnext` to paragraphs.

    """
    
    default_priority = 800 # late

    def apply (self, **kwargs):
        self.recurse (self.document, False)

    def recurse (self, parent, follows_paragraph):
        # follows_paragraph = False # flag: previous element is paragraph
        
        for node in parent:
            
            if isinstance (node, (nodes.paragraph)):
                node['classes'].append ('pnext' if follows_paragraph else 'pfirst')
                follows_paragraph = True
            elif isinstance (node, (nodes.title, nodes.subtitle)):
                # title may also be output as <html:p>
                node['classes'].append ('pfirst')
                follows_paragraph = False
            elif isinstance (node, (mynodes.page)):
                # explicit vertical space or page breaks
                follows_paragraph = False
            elif isinstance (node, (nodes.container, nodes.compound, 
                                    nodes.Invisible, nodes.footnote)):
                # invisible nodes are neutral, footnotes are pulled away
                pass
            else:
                # everything else
                follows_paragraph = False

            if not isinstance (node, nodes.Text):
                self.recurse (node, follows_paragraph)


class MetaCollector (docutils.transforms.Transform):
    """
    Collects meta nodes in document.meta_block (dict of lists).

    In an ill-advised effort to clean up, the `Filter` transform
    throws out all `pending` nodes with meta information if the writer
    format is not HTML. But PDF can have metadata too. We grab the
    metadata and store them in ``document.meta_block``.

    """
    
    default_priority = 200
    """ Before any variables transform. """

    def apply (self):
        self.document.meta_block = collections.defaultdict (list)
        
        for pending in self.document.traverse (nodes.pending):
            if pending.transform == components.Filter:
                try:
                    meta = pending.details['nodes'][0]
                    name = meta.attributes['name']
                    content = meta.attributes['content']
                    self.document.meta_block[name].append (content)
                except:
                    pass


class ImageReplacer (docutils.transforms.Transform):
    """
    Replaces images with 'broken' image to minimize file size.

    """
    
    default_priority = 800

    def apply (self):
        broken_uri = urlparse.urljoin (self.document.settings.base_url, broken)
        for image in self.document.traverse (nodes.image):
            image['uri'] = broken_uri
            image['width'] = '5%'
            image['align'] = 'center'
            if isinstance (image.parent, nodes.figure):
                figure = image.parent
                figure['width'] = figure.attributes.get ('width', '80%')
                if figure['width'] == 'image':
                    figure['width'] = '80%'


class ImageRemover (docutils.transforms.Transform):
    """
    Removes images.

    Caveat: sometimes removing an image makes the doctree invalid.
    eg. removing an image between two transitions.

    """
    
    default_priority = 200 # early before lists are collected

    def apply (self):
        for image in self.document.traverse (nodes.image):
            parent = image.parent
            parent.remove (image)
            if isinstance (parent, nodes.figure):
                # fixup figures that are now empty
                figure = parent
                figure['float'] = ['none'] # do not float bare captions
                figure['width'] = figure.attributes.get ('width', '80%')
                if figure['width'] == 'image':
                    figure['width'] = '80%'
            

class ImageWrapper (docutils.transforms.Transform):
    """
    Wrap a block-level image into a figure.

    """
    
    default_priority = 801 # after ImageReplacer

    def get_width (self, uri):
        # calculate a sensible default width for images
        # assume images are processed for a viewport 980px wide,
        # the same as the iPhone browser assumes

        if (self.document.settings.get_image_size and
            callable (self.document.settings.get_image_size)):

            size = self.document.settings.get_image_size (uri)
            if size is not None:
                w = int (float (size[0]) / (980.0 * 0.8) * 100.0 + 0.5)
                width = "%d%%" % min (100, w)
                debug ('Got dimension of image: %s: %s' % (uri, width))
                return width

        warn ('Could not get dimension of image: %s' % uri)
        return '100%'


    def apply (self):

        for image in self.document.traverse (nodes.image):

            # skip inline images
            # (See also: class `TextElement` in `docutils.nodes`.)
            if isinstance (image.parent, nodes.TextElement):
                continue

            # wrap all images into figures
            if not isinstance (image.parent, nodes.figure):
                figure = nodes.figure ()
                figure['float'] = ['none'] # do not float bare images
                figure['width'] = image.attributes.get ('width', 'image')
                image['width'] = '100%'
                figure['align'] = image.attributes.get ('align', 'center')
                image.replace_self (figure)
                figure.append (image)
            
            # set a default width for block images
            image['width'] = image.attributes.get ('width', '100%')

            figure = image.parent
            if figure['width'] == 'image':
                figure['width'] = self.get_width (image['uri'])
                figure['classes'].append ('auto-scaled')


class KindleTableCaption (docutils.transforms.Transform):
    """ Moves table captions out of table element.

    The Kindle cannot handle anything inside tables beside rows.
    If it finds a caption it just makes a table cell out of it.

    """

    default_priority = 900
    
    def apply (self, **kwargs):
        for table in self.document.traverse (nodes.table):
            for title in table.traverse (nodes.title):
                title.parent.remove (title)
                caption = nodes.caption ()
                caption[:] = title[:]
                table.parent.replace (table, [caption, table])
            
                    
class Lineblock2VSpace (docutils.transforms.Transform):
    """
    Turn empty line_blocks into page nodes.

    """
    
    default_priority = 200 # early, before vspace

    def apply (self):
        for lb in self.document.traverse (nodes.line_block):
            if lb.astext ().strip () == '':
                gap_len = len (lb)
                page = mynodes.page ()
                page['classes'].append ('vspace')
                page['length'] = gap_len
                lb.replace_self (page)

