#!/usr/bin/env python

# Copyright (c) 2016 Adam Karpierz
# SPDX-License-Identifier: BSD-3-Clause

# Copyright (c) 1988, 1989, 1990, 1991, 1992, 1993, 1994, 1995, 1996, 1997, 2000
#  The Regents of the University of California.  All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that: (1) source code distributions
# retain the above copyright notice and this paragraph in its entirety, (2)
# distributions including binary code include the above copyright notice and
# this paragraph in its entirety in the documentation or other materials
# provided with the distribution, and (3) all advertising materials mentioning
# features or use of this software display the following acknowledgement:
# ``This product includes software developed by the University of California,
# Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
# the University nor the names of its contributors may be used to endorse
# or promote products derived from this software without specific prior
# written permission.
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

import sys
import os
import getopt
import ctypes as ct

import libpcap as pcap
from pcaptestutils import *  # noqa

#ifndef lint
copyright = "@(#) Copyright (c) 1988, 1989, 1990, 1991, 1992, 1993, 1994, "\
            "1995, 1996, 1997, 2000\n"\
            "The Regents of the University of California.  "\
            "All rights reserved.\n"
#endif

MAXIMUM_SNAPLEN = 262144


def main(argv=sys.argv[1:]):

    global program_name
    program_name = os.path.basename(sys.argv[0])

    try:
        opts, args = getopt.getopt(argv, "i:Ips:aB:")
    except getopt.GetoptError:
        usage()

    device = None
    dorfmon = False
    dopromisc = False
    snaplen = MAXIMUM_SNAPLEN
    bufsize = 0
    useactivate = False
    useopen = False
    for opt, optarg in opts:
        if opt == '-i':
            device = optarg.encode("utf-8")
        elif opt == '-I':
            dorfmon = True
            useactivate = True  # required for rfmon
        elif opt == '-o':
            useopen = True
        elif opt == '-p':
            dopromisc = True
        elif opt == '-s':
            try:
                long_snaplen = int(optarg)
            except:
                error("invalid snaplen {}", optarg)
            if not (0 <= long_snaplen <= MAXIMUM_SNAPLEN):
                error("invalid snaplen {}", optarg)
            elif long_snaplen == 0:  # <AK> fix, was: snaplen == 0:
                snaplen = MAXIMUM_SNAPLEN
            else:
                snaplen = long_snaplen
        elif opt == '-B':
            try:
                bufsize = int(optarg) * 1024
            except:
                error("invalid packet buffer size {}", optarg)
            if bufsize <= 0:
                error("invalid packet buffer size {}", optarg)
            useactivate = True  # required for bufsize
        elif opt == '-a':
            useactivate = True
        else:
            usage()

    status = 0

    ebuf = ct.create_string_buffer(pcap.PCAP_ERRBUF_SIZE)

    if device is None:
        devlist = ct.POINTER(pcap.pcap_if_t)()
        if pcap.findalldevs(ct.byref(devlist), ebuf) == -1:
            error("{}", ebuf2str(ebuf))
        if not devlist:
            error("no interfaces available for capture")
        device = devlist[0].name
        pcap.freealldevs(devlist)

    if useactivate:
        pd = pcap.create(device, ebuf)
        if not pd:
            error("{}: pcap.create failed: {}",
                  device2str(device), ebuf2str(ebuf))

        status = pcap.set_snaplen(pd, snaplen)
        if status != 0:
            error("{}: pcap.set_snaplen failed: {}",
                  device2str(device), status2str(status))

        if dopromisc:
            status = pcap.set_promisc(pd, 1)
            if status != 0:
                error("{}: pcap.set_promisc failed: {}",
                      device2str(device), status2str(status))

        if dorfmon:
            try:
                status = pcap.set_rfmon(pd, 1)
            except AttributeError:
                error("pcap.set_rfmon is not available on this platform")
            if status != 0:
                error("{}: pcap.set_rfmon failed: {}",
                      device2str(device), status2str(status))

        status = pcap.set_timeout(pd, 1000)
        if status != 0:
            error("{}: pcap.set_timeout failed: {}",
                  device2str(device), status2str(status))

        if bufsize != 0:
            status = pcap.set_buffer_size(pd, bufsize)
            if status != 0:
                error("{}: pcap.set_buffer_size failed: {}",
                      device2str(device), status2str(status))

        status = pcap.activate(pd)
        if status < 0:
            # pcap.activate() failed.
            error("{}: {}\n({})",
                  device2str(device), status2str(status), geterr2str(pd))
        elif status > 0:
            # pcap.activate() succeeded, but it's warning us
            # of a problem it had.
            warning("{}: {}\n({})",
                    device2str(device), status2str(status), geterr2str(pd))
        else:
            print("{} opened successfully".format(device2str(device)))
    elif useopen:
        ebuf[0] = b"\0"
        pd = pcap.open(device, 65535, pcap.PCAP_OPENFLAG_PROMISCUOUS, 1000,
                       None, ebuf)
        if not pd:
            error("{}", ebuf2str(ebuf))
        elif ebuf.value:
            warning("{}", ebuf2str(ebuf))
        else:
            print("{} opened successfully".format(device2str(device)))
    else:
        ebuf[0] = b"\0"
        pd = pcap.open_live(device, 65535, 0, 1000, ebuf)
        if not pd:
            error("{}", ebuf2str(ebuf))
        elif ebuf.value:
            warning("{}", ebuf2str(ebuf))
        else:
            print("{} opened successfully".format(device2str(device)))

    device = None
    pcap.close(pd)

    return 1 if status < 0 else 0


def usage():
    print("Usage: {} [ -Ipa ] [ -i interface ] [ -s snaplen ] "
          "[ -B bufsize ]".format(program_name), file=sys.stderr)
    sys.exit(1)


if __name__.rpartition(".")[-1] == "__main__":
    sys.exit(main())
