"""
Module for serializing Python objects to and from XML
"""

from lxml import etree
from datetime import datetime, date
from decimal import Decimal
import iso8601

def parse_date(value):
    return datetime.strptime(value[:10], "%Y-%m-%d").date()

def parse_datetime(value):
    return iso8601.parse_date(value)

def split_tag(tag):
    if tag.startswith('{'):
        return tag[1:].split('}')
    else:
        return None, tag

def get_nsmap(obj):
    if not hasattr(obj, '_xml_metadata'):
        return None
    
    if 'namespace' not in obj._xml_metadata:
        return None        
    
    return {None: obj._xml_metadata['namespace']}

def get_namespace(obj):
    if hasattr(obj, '_xml_metadata'):
        if 'namespace' in obj._xml_metadata:
            return '{%s}' % obj._xml_metadata['namespace']

    return ''
            
def toxml(obj, root=None):
    """
    Serialize an xsdtopy generated Python object to XML
    
    @param obj: The Python object to serialize
    @type obj: Python object generated by xsdtopy
    @return: lxml etree element containing the serialized XML representation of obj
    @rtype: lxml.etree._Element
    """
    namespace = get_namespace(obj)
    
    if hasattr(obj, '_xml_metadata'):
        if 'sequence' in obj._xml_metadata:            
            elements = obj._xml_metadata['sequence']
        else:
            elements = []
            
        if 'attributes' in obj._xml_metadata:
            attributes = obj._xml_metadata['attributes']
        else:
            attributes = []
    else:
        elements = obj.__dict__
        attributes = []
   
    if root == None:
        root = etree.Element(namespace + obj.__class__.__name__, nsmap=get_nsmap(obj))

    if obj.__class__.__base__ == str:
        root.text = str(obj)

    for attribute in attributes:
        value = getattr(obj, attribute)
        if value is not None:
            if isinstance(value, str): 
                root.set(attribute, value)
            elif isinstance(value, unicode):
                root.set(attribute, value)
            elif isinstance(value, bool):
                if value == True: root.set(attribute, 'true')
                else: root.set(attribute, 'false')
            elif hasattr(value, 'isoformat') and hasattr(value, 'time'):
                root.set(attribute, value.isoformat('T'))
            elif hasattr(value, 'isoformat'):
                root.set(attribute, value.isoformat())
            else:
                root.set(attribute, str(value))
                
    for element in elements:
        values = getattr(obj, element)
        if values is None:
            continue
        
        if not isinstance(values, list):
            values = [values]
        
        for value in values:
            if hasattr(value, '_xml_metadata'):
                toxml(value, etree.SubElement(root, get_namespace(value) + element, nsmap=get_nsmap(value)))
            elif isinstance(value, str): 
                etree.SubElement(root, namespace + element).text = value
            elif isinstance(value, unicode):
                etree.SubElement(root, namespace + element).text = value
            elif isinstance(value, bool):
                if value == True: etree.SubElement(root, namespace + element).text = 'true'
                else: etree.SubElement(root, namespace + element).text = 'false'
            elif hasattr(value, 'isoformat') and hasattr(value, 'time'):
                etree.SubElement(root, namespace + element).text = value.isoformat('T')
            elif hasattr(value, 'isoformat'):
                etree.SubElement(root, namespace + element).text = value.isoformat()
            elif hasattr(value, '_xml_metadata') or hasattr(value, '__dict__'):
                toxml(value, etree.SubElement(root, get_namespace(value) + element, nsmap=get_nsmap(value)))
            else:
                etree.SubElement(root, namespace + element).text = str(value)
                
    return root

def fromxml(root, obj):
    """
    Deserialize provided XML to a Python object
    
    @param root: XML to be deserialized into a Python object
    @type root: lxml.etree._Element
    @param obj: Python object which will contain the results of the deserialization from XML
    @type obj: Python object generated by xsdtopy
    @return: None ('obj' is populated during deserialization from XML)
    @rtype: None
    
    Note: When deserializing a root node into an Python primitive (str, int, etc), or an object which
    inherits from a Python primitive, the 'text' will not be deserialized. For example, when the following
    is deserialized:
    <test>x</test>
    The text node containing the value 'x' will not be deserialized. This is a known limitation of fromxml.
    """
    
    # read in the XML attributes
    for attrib in root.attrib:
        if hasattr(obj, '_xml_metadata') and attrib in obj._xml_metadata['types']:
            child_type = obj._xml_metadata['types'][attrib]()
        else:
            child_type = None        
        
        if child_type is str:
            setattr(obj, attrib, root.attrib[attrib])
        elif child_type is Decimal:
            setattr(obj, attrib, Decimal(root.attrib[attrib]))
        elif child_type is int:
            setattr(obj, attrib, int(root.attrib[attrib]))             
        elif child_type is bool:
            if root.attrib[attrib] == 'true': setattr(obj, attrib, True)
            else: setattr(obj, attrib, False)
        elif child_type is datetime:
            if root.attrib[attrib] is None or len(root.attrib[attrib]) == 0:
                setattr(obj, attrib, None)
            else:
                setattr(obj, attrib, parse_datetime(root.attrib[attrib]))
        elif child_type is date:
            if root.attrib[attrib] is None or len(root.attrib[attrib]) == 0:
                setattr(obj, attrib, None)
            else:
                setattr(obj, attrib, parse_date(root.attrib[attrib]))
        else:
            setattr(obj, attrib, str(root.attrib[attrib]))

    # read in all the XML elements
    values = {}
    for child in root.iterchildren(tag=etree.Element):
        ns, name = split_tag(child.tag)
        
        child_type_is_list = False #child type is a Python list, not just a single object
        
        # determine the Python type that should be created from the data
        if hasattr(obj, '_xml_metadata') and name in obj._xml_metadata['types']:
            child_type = obj._xml_metadata['types'][name]()
            if isinstance(child_type, list):
                child_type_is_list = True
                child_type = child_type[0]
        else:
            # object not defined in the _xml_metadata, we'll still do our best
            # to deserialize the data contained in the element
            child_type = None
        
        if len(child) == 0:
            # the current element contains no child elements
            
            if child_type is str:
                value = child.text
            elif child_type is int:
                if child.text is None or len(child.text) == 0:
                    value = None
                else:
                    value = int(child.text)
            elif child_type is bool:
                if child.text == 'true': value = True
                else: value = False
            elif child_type is datetime:
                if child.text is None or len(child.text) == 0:
                    value = None
                else:
                    value = parse_datetime(child.text)
            elif child_type is date:
                if child.text is None or len(child.text) == 0:
                    value = None
                else:
                    value = parse_date(child.text)
            elif hasattr(child_type, '_xml_metadata'):
                # means that the child is also an xmltopy generated object
                if child.text:
                    child_obj = child_type(child.text)
                else:
                    child_obj = child_type()
                value = fromxml(child, child_obj)            
            else:
                value = child.text
        elif child_type is not None:
            child_obj = child_type()
            value = fromxml(child, child_obj)
        else:
            error = 'Not enough metadata to deserialize the \'%s\' element as a member of a \'%s\' object.' % (name, type(obj))
            root_ns, root_name = split_tag(root.tag)
            if obj.__class__.__name__ != root_name:
                error += ' Did you mean to use a %s object instead of a %s object?' % (root_name, obj.__class__.__name__ )
            raise RuntimeError(error)
        
        # 'name' is the name of the element, ex: 'person' for the element <person>
        if name in values and isinstance(values[name], list):
            values[name].append(value)
        elif child_type is None and name in values:
            values[name] = [values[name], value] # child_type undefined, and existing value found, then turn it into a list
        elif child_type_is_list == True:
            values[name] = [value] # if child_type is defined, and indicates a list, then always create a list
        else:
            values[name] = value

    for value in values:
        setattr(obj, value, values[value])
            
    return obj


