"""Functions intended for library internal use."""

from __future__ import annotations  # Allows forward annotations in Python < 3.10

import datetime
from enum import Enum
import io
import os
from pathlib import Path
import re
import time
import warnings
from zipfile import ZipFile

import numpy as np
import pandas as pd
import requests


def get_latest_timestamped_file_path(file_path: str) -> str:
    """Get the latest version of a file, based on the unix timestamp in its name.

    Args:
        file_path: The file path of which to search for a timestamped version of the file

    Raises:
        FileNotFoundError: Raised if no timestamped version of the file can be found.

    Returns:
        The path of an existing timestamped version of the provided path.
    """
    file_no_extension, extension = os.path.splitext(file_path)

    # Regular expression for matching timestamped files in the format:
    # <filename>_<timestamp><extension> e.g. '/foo/bar_12345.baz'
    file_regex = rf'{re.escape(file_no_extension)}_(?P<unix_timestamp>\d+){re.escape(extension)}'

    # Get all files in directory of given file
    files_in_dir = [str(f) for f in Path(file_path).parent.glob('*') if f.is_file()]

    # Find version of requested file with latest timestamp
    latest_file_path = None
    latest_unix_timestamp = -1
    for file_name in files_in_dir:
        match = re.match(file_regex, file_name)

        # No match -> file is not a timestamped version of file_path
        if match is None:
            continue

        unix_timestamp = int(match.group('unix_timestamp'))

        # Unix timestamp is higher -> file is newer
        if unix_timestamp > latest_unix_timestamp:
            latest_file_path = file_name
            latest_unix_timestamp = unix_timestamp

    # Raise error if no timestamped file exists
    if latest_file_path is None:
        raise FileNotFoundError(f"No timestamped file for {file_path}")

    return latest_file_path


def generate_timestamped_file_path(file_path: str, exists_ok: bool = True) -> str:
    """Add a unix timestamp at the end of the filename but before the file extension.
    This function only generates the path name, it does not actually create or modify any files.

    Args:
        file_path: The path to which to add a unix timestamp
        exists_ok: Whether to raise an error if a file already exists at the generated path. Defaults to True.

    Raises:
        FileExistsError: Raised if `exists_ok` is `False` and a file at the path generated by this
            function already exists.

    Returns:
        A path with an added unix timestamp.
    """
    file_no_extension, extension = os.path.splitext(file_path)

    # Get unix timestamp, rounded down to seconds
    unix_timestamp = int(time.time())

    # Add timestamp to file path
    timestamped_file_path = f'{file_no_extension}_{unix_timestamp}{extension}'

    # Raise an exception if the file already exists
    if not exists_ok and os.path.exists(timestamped_file_path):
        raise FileExistsError(f'Timestamped file {timestamped_file_path} already exists')

    return timestamped_file_path


def try_read_cached_csv(filename: str, **kwargs) -> pd.DataFrame | None:
    """Try to read a CSV from cache.

    Args:
        filename: The name of the file (NOT the name of the cache file!).

    Returns:
        The contents of the CSV as a dataframe or None if the file is not found in cache.
    """
    try:
        timestamped_file_path = get_latest_timestamped_file_path(filename)
        df = pd.read_csv(timestamped_file_path, **kwargs)
    except FileNotFoundError:
        # File is not in cache
        df = None

    return df


def download_csv(
    url: str,
    cache_filename: str | None = None,
    zipped_filename: str | None = None,
    timeout: int = 120, **kwargs
) -> pd.DataFrame:
    """Download a CSV and load it into a dataframe.

    Args:
        url: The full HTTP(S) URL of the CSV file.
        cache_filename: The name of the file the downloaded CSV should be cached to. \
            If None, no caching is performed. Defaults to None.
        zipped_filename: If the given file is a zip, this should be set to the file \
            path of the desired CSV in the zip. Defaults to None.
        timeout: The timeout (in seconds) for the HTTP request. Defaults to 120.

    Returns:
        The CSV data as a pandas DataFrame.
    """
    # Get CSV file from URL
    with requests.get(url, timeout=timeout) as response:
        if zipped_filename is not None:
            # CSV file is in a zip, get zip file and extract CSV file in memory
            zip_file_content = response.content
            speeches_zip = ZipFile(io.BytesIO(zip_file_content))
            csv_io = speeches_zip.open(zipped_filename, 'r')
        else:
            # Read CSV file
            # WORKAROUND: File may contain characters not compatible with utf8 encoding. "Fix" them
            # by ignoring them during decode.
            csv_io = io.StringIO(response.content.decode("utf8", errors="ignore"))

        # Read CSV into dataframe
        with csv_io as csv_file:
            df = pd.read_csv(csv_file, **kwargs)

    # Write file to cache
    if cache_filename is not None:
        timestamped_file_path = generate_timestamped_file_path(cache_filename)
        Path(timestamped_file_path).parent.mkdir(exist_ok=True)  # Ensure parent dir exists
        df.to_csv(timestamped_file_path, index=False)

    return df


def verify_cached_csv(file_path: str) -> None:
    """Verify that a given file exists in the cache and can be read as CSV.

    Args:
        file_path: The name of the file (NOT the name of the cache file!)

    Raises:
        RuntimeError: Raised if verification fails.
    """
    try:
        timestamped_file_path = get_latest_timestamped_file_path(file_path)
        verify_df = pd.read_csv(timestamped_file_path)
        assert len(verify_df) > 0, (
            f'Cached CSV at {timestamped_file_path} is empty. '
            'Manually remove the file or replace with correct dataset.'
        )
    except Exception as ex:
        raise RuntimeError("Verification error. See previous exception.") from ex


class Frequency(Enum):
    """Enum class representing valid frequency codes for time series data.

    These frequency codes align with pandas' frequency aliases and can be used
    for operations like resampling or date range generation.

    Attributes:
        DAILY (str): Daily frequency ('D')
        WEEKLY (str): Weekly frequency ('W')
        MONTHLY (str): Monthly start frequency ('MS')
        QUARTERLY (str): Quarterly start frequency ('QS')
    """

    DAILY = "D"
    WEEKLY = "W"
    MONTHLY = "MS"
    QUARTERLY = "QS"


FrequencyLike = str | Frequency
DateTimeLike = pd.Timestamp | datetime.datetime | np.datetime64 | str | datetime.date


class DayFeatures(Enum):
    """Enum class representing day-based temporal features.

    These features provide various ways to represent a day within different
    time periods (week, month, quarter, year).

    Attributes:
        DAY_OF_WEEK (str): Day of the week (0-6, where 0 is Monday)
        DAY_OF_MONTH (str): Day of the month (1-31)
        DAY_OF_QUARTER (str): Day of the quarter (1-92)
        DAY_OF_YEAR (str): Day of the year (1-366)
    """

    DAY_OF_WEEK = "day_of_week"
    DAY_OF_MONTH = "day_of_month"
    DAY_OF_QUARTER = "day_of_quarter"
    DAY_OF_YEAR = "day_of_year"


class WeekFeatures(Enum):
    """Enum class representing week-based temporal features.

    These features provide various ways to represent a week within different
    time periods (month, quarter, year).

    Attributes:
        WEEK_OF_MONTH (str): Week of the month (1-5)
        WEEK_OF_QUARTER (str): Week of the quarter (1-13)
        WEEK_OF_YEAR (str): Week of the year (1-53)
    """

    WEEK_OF_MONTH = "week_of_month"
    WEEK_OF_QUARTER = "week_of_quarter"
    WEEK_OF_YEAR = "week_of_year"


class MonthFeatures(Enum):
    """Enum class representing month-based temporal features.

    These features provide ways to represent a month within a quarter and a year.

    Attributes:
        MONTH_OF_QUARTER (str): Month of the quarter (1-3)
        MONTH_OF_YEAR (str): Month of the year (1-12)
    """

    MONTH_OF_QUARTER = "month_of_quarter"
    MONTH_OF_YEAR = "month_of_year"


class QuarterFeatures(Enum):
    """Enum class representing quarter-based and year-end temporal features.

    These features provide ways to represent quarters within a year and
    indicate if a date is at the end of a quarter or year.

    Attributes:
        QUARTER_OF_YEAR (str): Quarter of the year (1-4)
        QUARTER_END (str): Boolean indicator for end of quarter (0 or 1)
        YEAR_END (str): Boolean indicator for end of year (0 or 1)
            Note: This is not strictly a quarterly feature but is included
            here for convenience in temporal feature generation.
    """

    QUARTER_OF_YEAR = "quarter_of_year"
    QUARTER_END = "quarter_end"
    YEAR_END = "year_end"


class InvalidTemporalFeature(Exception):
    """Exception raised when an invalid temporal feature is passed to a method."""

    def __init__(self, feature_name: str, message: str = "Invalid temporal feature passed"):
        self.feature_name = feature_name
        self.message = f"{message}: '{feature_name}'"
        super().__init__(self.message)


def validate_and_get_freq(freq: FrequencyLike) -> Frequency:
    """Validate and return a pandas-compatible frequency string.

    This function checks if the provided frequency string is valid according to
    the Frequency enum and pandas' internal frequency validation. It converts
    the input to uppercase before validation.

    Args:
        freq (FrequencyLike): The input frequency to validate.

    Returns:
        str: A valid pandas frequency string.

    Raises:
        ValueError: If the input frequency is not valid or not supported.
    """
    try:
        if isinstance(freq, str):
            freq = Frequency(freq.upper())
        # Additional check with pandas
        pd.tseries.frequencies.to_offset(freq.value)
        return freq
    except ValueError as exc:
        raise ValueError(
            f"Invalid frequency: {freq}. Allowed values are {[f.value for f in Frequency]}"
        ) from exc


def _get_temporal_feature_names(freq: Frequency | None = None) -> list[str]:
    """Returns all supported temporal feature names, optionally only those valid for a given
    frequency.

    Args:
        freq (Frequency | None): Optional input frequency

    Returns:
        list[str]: List of feature names supported by the arguments
    """
    if not freq or freq == Frequency.DAILY:
        all_features = [DayFeatures, WeekFeatures, MonthFeatures, QuarterFeatures]
    elif freq == Frequency.WEEKLY:
        all_features = [WeekFeatures, MonthFeatures, QuarterFeatures]
    elif freq == Frequency.MONTHLY:
        all_features = [MonthFeatures, QuarterFeatures]
    else:
        all_features = [QuarterFeatures]

    return [f.value for features in all_features for f in features]


def _check_valid_features(features: list[str], freq: Frequency) -> list[str]:
    """Warns if a feature is requested that does not match the provided input frequency"""
    valid_features = []
    freq_features = _get_temporal_feature_names(freq)
    all_features = _get_temporal_feature_names()
    for feature in features:
        if feature in freq_features:
            valid_features.append(feature)
        elif feature in all_features:
            warnings.warn(
                f"Requested temporal feature {feature} "
                f"not available for data with frequency {freq.value}!",
                category=UserWarning,
                stacklevel=2,  # Warning traceback points to caller of this function
            )
        else:
            raise InvalidTemporalFeature(feature)
    return valid_features


def _get_day_features(dt_index: pd.DatetimeIndex) -> pd.DataFrame:
    """Calculate day-related temporal features from a DatetimeIndex.

    Args:
        dt_index (pd.DatetimeIndex): The DatetimeIndex to extract features from.

    Returns:
        pd.DataFrame: A DataFrame containing day-related features
    """
    return pd.DataFrame(
        index=dt_index,
        data={
            DayFeatures.DAY_OF_WEEK.value: dt_index.dayofweek,
            DayFeatures.DAY_OF_MONTH.value: dt_index.day,
            DayFeatures.DAY_OF_QUARTER.value: dt_index.dayofyear
            - dt_index.to_period("Q").start_time.dayofyear
            + 1,
            DayFeatures.DAY_OF_YEAR.value: dt_index.dayofyear,
        },
    )


def _get_week_features(dt_index: pd.DatetimeIndex) -> pd.DataFrame:
    """Calculate week-related temporal features from a DatetimeIndex.

    Args:
        dt_index (pd.DatetimeIndex): The DatetimeIndex to extract features from.

    Returns:
        pd.DataFrame: A DataFrame containing week-related features
    """
    return pd.DataFrame(
        index=dt_index,
        data={
            WeekFeatures.WEEK_OF_MONTH.value: dt_index.day.map(lambda x: (x - 1) // 7 + 1),
            WeekFeatures.WEEK_OF_QUARTER.value: (
                dt_index.dayofyear - dt_index.to_period("Q").start_time.dayofyear
            )
            // 7
            + 1,
            WeekFeatures.WEEK_OF_YEAR.value: dt_index.isocalendar().week,
        },
    )


def _get_month_features(dt_index: pd.DatetimeIndex) -> pd.DataFrame:
    """Calculate month-related temporal features from a DatetimeIndex.

    Args:
        dt_index (pd.DatetimeIndex): The DatetimeIndex to extract features from

    Returns:
        pd.DataFrame: A DataFrame containing month-related features
    """
    return pd.DataFrame(
        index=dt_index,
        data={
            MonthFeatures.MONTH_OF_QUARTER.value: dt_index.month.map(lambda x: (x - 1) % 3 + 1),
            MonthFeatures.MONTH_OF_YEAR.value: dt_index.month,
        },
    )


def _get_quarter_features(dt_index: pd.DatetimeIndex) -> pd.DataFrame:
    """Calculate quarter-related temporal features from a DatetimeIndex.

    Args:
        dt_index (pd.DatetimeIndex): The DatetimeIndex to extract features from.

    Returns:
        pd.DataFrame: A DataFrame containing quarter-related features
    """
    return pd.DataFrame(
        index=dt_index,
        data={
            QuarterFeatures.QUARTER_OF_YEAR.value: dt_index.quarter,
            QuarterFeatures.QUARTER_END.value: dt_index.is_quarter_end.astype(int),
            QuarterFeatures.YEAR_END.value: dt_index.is_year_end.astype(int),
        },
    )


def _get_filtered_data(frame: pd.DataFrame, cutoff_date: DateTimeLike) -> pd.DataFrame | None:
    """Filter data up to a given date.

    Args:
        frame (pd.DataFrame): The input DataFrame to filter.
        cutoff_date (DateTimeLike): The cutoff date.

    Returns:
        Optional[pd.DataFrame]: Filtered DataFrame or None if empty.
    """
    filtered_data = frame.loc[frame.index <= cutoff_date]
    return filtered_data if not filtered_data.empty else None
