import sys
from flopy.mbase import Package

class ModflowGmg(Package):
    '''GMG package
    Only programmed to work with the default values; may need work for other options'''
    def __init__(self, model, mxiter=50, iiter=30, iadamp=0, 
                 hclose=1e-5, rclose=1e-5, relax=1.0, 
                 ioutgmg=0, iunitmhc=0, 
                 ism=0, isc=0, damp=1.0, dup=0.75, dlow=0.01,
                 chglimit=1.0,extension='gmg', unitnumber=27):
        Package.__init__(self, model, extension, 'GMG', unitnumber) # Call ancestor's init to set self.parent, extension, name and unit number
        self.heading = '# GMG for MODFLOW, generated by Flopy.'
        self.url = 'gmg.htm'
        self.mxiter = mxiter
        self.iiter = iiter
        self.iadamp = iadamp
        self.hclose = hclose
        self.rclose = rclose
        self.relax = relax
        self.ism = ism
        self.isc = isc
        self.dup = dup
        self.dlow = dlow
        self.chglimit = chglimit
        self.damp = damp
        self.ioutgmg = ioutgmg
        self.iunitmhc = iunitmhc
        self.parent.add_package(self)

    def __repr__( self ):
        return 'Geometric multigrid solver package class'

    def write_file(self):
        # Open file for writing
        f_gmg = open(self.fn_path, 'w')
        f_gmg.write('%s\n' % self.heading)        
        # dataset 0
        f_gmg.write('{} {} {} {}\n'\
             .format(self.rclose, self.iiter, self.hclose, self.mxiter))        
        # dataset 1
        f_gmg.write('{} {} {} {}\n'\
             .format(self.damp, self.iadamp, self.ioutgmg, self.iunitmhc))
        # dataset 2
        f_gmg.write('{} {} '.format(self.ism, self.isc))
        if self.iadamp == 2:
            f_gmg.write('{} {} {}'\
                 .format(self.dup, self.dlow, self.chglimit))
        f_gmg.write('\n')
        # dataset 3
        f_gmg.write('{}\n'.format(self.relax))
        f_gmg.close()


    @staticmethod
    def load(f, model, ext_unit_dict=None):
        """
        Load an existing package.

        Parameters
        ----------
        f : filename or file handle
            File to load.
        model : model object
            The model object (of type :class:`flopy.modflow.mf.Modflow`) to
            which this package will be added.
        ext_unit_dict : dictionary, optional
            If the arrays in the file are specified using EXTERNAL,
            or older style array control records, then `f` should be a file
            handle.  In this case ext_unit_dict is required, which can be
            constructed using the function
            :class:`flopy.utils.mfreadnam.parsenamefile`.

        Returns
        -------
        gmg : ModflowGmg object

        Examples
        --------

        >>> import flopy
        >>> m = flopy.modflow.Modflow()
        >>> gmg = flopy.modflow.ModflowGmg.load('test.gmg', m)

        """

        if model.verbose:
            sys.stdout.write('loading gmg package file...\n')

        if not hasattr(f, 'read'):
            filename = f
            f = open(filename, 'r')
        #dataset 0 -- header
        while True:
            line = f.readline()
            if line[0] != '#':
                break
        #dataset 0
        t = line.strip().split()
        rclose = float(t[0])
        iiter = int(t[1])
        hclose = float(t[2])
        mxiter = int(t[3])
        #dataset 1
        line = f.readline()
        t = line.strip().split()
        damp = float(t[0])
        iadamp = int(t[1])
        ioutgmg = int(t[2])
        try:
            iunitmhc = int(t[3])            
        except:
            iunitmhc = 0
        #dataset 2
        line = f.readline()
        t = line.strip().split()
        ism = int(t[0])
        isc = int(t[1])
        dup, dlow, chglimit = 0.75, 0.01, 1.0
        if iadamp == 2:
            dup = float(t[2])
            dlow = float(t[3])
            chglimit = float(t[4])
        #dataset 3
        line = f.readline()
        t = line.strip().split()
        relax = float(t[0])

        # close the open file
        f.close()

        # create the gmg object
        gmg = ModflowGmg(model, mxiter=mxiter, iiter=iiter, iadamp=iadamp,
                         hclose=hclose, rclose=rclose, relax=relax,
                         ioutgmg=ioutgmg, iunitmhc=iunitmhc,
                         ism=ism, isc=isc, damp=damp,
                         dup=dup, dlow=dlow, chglimit=chglimit)
        return gmg
