#!/usr/bin/env python

import urllib

import click
import gitlab

from tabulate import tabulate

from ..log import echo_warning
from ..log import get_logger
from ..log import verbosity_option
from ..pipelines import process_log
from ..release import get_gitlab_instance
from . import bdt

logger = get_logger(__name__)


@click.command(
    epilog="""
Examples:

  1. Process all the jobs from a particular pipeline

     $ bdt gitlab process-pipelines bob/nightlies pipelines

  2. Process a particular job from a pipeline

     $ bdt gitlab process-pipelines bob/nightlies pipelines --job-id xxx

"""
)
@click.argument("package")
@click.argument("pipeline")
@click.option("--job-id", default=None, help="A job id from a pipeline")
@verbosity_option()
@bdt.raise_on_error
def process_pipelines(package, pipeline, job_id):
    """Returns the last tag information on a given PACKAGE."""

    if "/" not in package:
        raise RuntimeError('PACKAGE should be specified as "group/name"')

    gl = get_gitlab_instance()

    # we lookup the gitlab package once
    try:
        project = gl.projects.get(package)
        pipeline = project.pipelines.get(pipeline)

        jobs = [j for j in pipeline.jobs.list()]
        if job_id is not None:
            jobs = [j for j in jobs if int(j.attributes["id"]) == int(job_id)]

        if len(jobs) == 0:
            print(
                "Job %s not found in the pipeline %s. Use `bdt gitlab get-pipelines` to search "
                % (job_id, pipeline.attributes["id"])
            )

        # Reading log
        try:
            for j in jobs:
                print(
                    "Pipeline %s, Job %s"
                    % (pipeline.attributes["id"], int(j.attributes["id"]))
                )
                web_url = j.attributes["web_url"] + "/raw"
                log = str(urllib.request.urlopen(web_url).read()).split("\\n")
                process_log(log)
        except urllib.error.HTTPError:
            logger.warn(
                "Gitlab access error - Log %s can't be found" % web_url,
                package,
                exc_info=True,
            )
            echo_warning("%s: unknown" % (package,))

        logger.info(
            "Found gitlab project %s (id=%d)",
            project.attributes["path_with_namespace"],
            project.id,
        )

        pass
    except gitlab.GitlabGetError:
        logger.warn(
            "Gitlab access error - package %s does not exist?", package, exc_info=True
        )
        echo_warning("%s: unknown" % (package,))


@click.command(
    epilog="""
Examples:

  1. Get the most recent pipelines from a particular project wit their corresponding job numbers

     $ bdt gitlab get-pipelines bob/nightlies


"""
)
@click.argument("package")
@verbosity_option()
@bdt.raise_on_error
def get_pipelines(package):
    """Returns the CI pipelines given a given PACKAGE."""

    if "/" not in package:
        raise RuntimeError('PACKAGE should be specified as "group/name"')

    gl = get_gitlab_instance()

    # we lookup the gitlab package once
    try:
        project = gl.projects.get(package)
        logger.info(
            "Found gitlab project %s (id=%d)",
            project.attributes["path_with_namespace"],
            project.id,
        )

        pipelines = project.pipelines.list()
        description = [["Pipeline", "Branch", "Status", "Jobs"]]
        for pipeline in pipelines:
            jobs = [j.attributes["id"] for j in pipeline.jobs.list()]
            description.append(
                [
                    "%s" % pipeline.attributes["id"],
                    "%s" % pipeline.attributes["ref"],
                    "%s" % pipeline.attributes["status"],
                    "%s" % jobs,
                ]
            )
        print("Jobs from project %s" % package)
        print(tabulate(description))

    except gitlab.GitlabGetError:
        logger.warn(
            "Gitlab access error - package %s does not exist?", package, exc_info=True
        )
        echo_warning("%s: unknown" % (package,))
