'''
The MIT License (MIT)

Copyright (c) 2007 ~ 2015, Hong-She Liang <starofrainnight@gmail.com>. 

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
'''

'''
Bootstrap rabird setup environment

@date 2015-08-20
@author Hong-She Liang <starofrainnight@gmail.com>
'''

import os
import os.path
import platform
import subprocess
import sys

def _clean_check(cmd, target):
    """
    Run the command to download target. If the command fails, clean up before
    re-raising the error.
    """
    try:
        subprocess.check_call(cmd)
    except subprocess.CalledProcessError:
        if os.access(target, os.F_OK):
            os.unlink(target)
        raise

def download_file_powershell(url, target):
    """
    Download the file at url to target using Powershell (which will validate
    trust). Raise an exception if the command cannot complete.
    """
    target = os.path.abspath(target)
    cmd = [
        'powershell',
        '-Command',
        "(new-object System.Net.WebClient).DownloadFile(%(url)r, %(target)r)" % vars(),
    ]
    _clean_check(cmd, target)

def has_powershell():
    if platform.system() != 'Windows':
        return False
    cmd = ['powershell', '-Command', 'echo test']
    devnull = open(os.path.devnull, 'wb')
    try:
        try:
            subprocess.check_call(cmd, stdout=devnull, stderr=devnull)
        except:
            return False
    finally:
        devnull.close()
    return True

download_file_powershell.viable = has_powershell

def download_file_curl(url, target):
    cmd = ['curl', url, '--silent', '--output', target]
    _clean_check(cmd, target)

def has_curl():
    cmd = ['curl', '--version']
    devnull = open(os.path.devnull, 'wb')
    try:
        try:
            subprocess.check_call(cmd, stdout=devnull, stderr=devnull)
        except:
            return False
    finally:
        devnull.close()
    return True

download_file_curl.viable = has_curl

def download_file_wget(url, target):
    cmd = ['wget', url, '--quiet', '--output-document', target]
    _clean_check(cmd, target)

def has_wget():
    cmd = ['wget', '--version']
    devnull = open(os.path.devnull, 'wb')
    try:
        try:
            subprocess.check_call(cmd, stdout=devnull, stderr=devnull)
        except:
            return False
    finally:
        devnull.close()
    return True

download_file_wget.viable = has_wget

def download_file_insecure(url, target):
    """
    Use Python to download the file, even though it cannot authenticate the
    connection.
    """
    try:
        from urllib.request import urlopen
    except ImportError:
        from urllib2 import urlopen
    src = dst = None
    try:
        src = urlopen(url)
        # Read/write all in one block, so we don't create a corrupt file
        # if the download is interrupted.
        data = src.read()
        dst = open(target, "wb")
        dst.write(data)
    finally:
        if src:
            src.close()
        if dst:
            dst.close()

download_file_insecure.viable = lambda: True

def get_best_downloader():
    downloaders = [
        download_file_powershell,
        download_file_curl,
        download_file_wget,
        download_file_insecure,
    ]

    for dl in downloaders:
        if dl.viable():
            return dl
        
def download(url):
    downloader = get_best_downloader()
    downloader(url, os.path.basename(url))

def use_pip():
    try:
        import pip
    except:
        import os
        import sys
        
        # If we do not have pip, we fetch and install one. It will also install
        # setuptools and wheel. 
        url = "https://bootstrap.pypa.io/get-pip.py"
        filename = os.path.basename(url)
        download(url)
        os.system("%s %s" % (sys.executable, filename))
        
def use_pydgutils():
    try:
        import pydgutils
    except:
        use_pip()
        import pip
        pip.main(["install", "pydgutils"])
        
def use_rabird():
    try:
        import rabird.core
    except:
        use_pydgutils()    
        import pip
        pip.main(["install", "rabird.core"])
        
        module_dirs = "rabird/core/__init__.py"         
        for apath in sys.path:
            module_path = os.path.join(apath, module_dirs)
            if os.path.exists(module_path) and os.path.isfile(module_path):
                # Generate empty __init__.py into rabird/, an ugly fix 
                # for can't find rabird.core module during installation.
                #
                # Because there does not have any __init__.py in the 
                # namespace directory and we can't import it immediately
                # after pip installed in the same process, so we added 
                # an empty __init__.py into rabird/ namespace directory
                # for satisfy it.
                afile = open(os.path.join(os.path.dirname(os.path.dirname(module_path)), "__init__.py"), "wb")
                afile.close()
                break
