
import sys, copy
from itertools import groupby, count, chain
import re

_PY3 = sys.version_info[0] > 2
if _PY3:
    basestring = str

SPANRE = re.compile(r'^\s*(?P<start>-?\d+)\s*(-\s*(?P<stop>-?\d+))?\s*$')
class ParseError(ValueError):
    pass

class intspan(set):
    def __init__(self, initial=None):
        super(intspan, self).__init__()
        if initial:
            self.update(initial)

    def copy(self):
        return copy.copy(self)

    def update(self, items):
        super(intspan, self).update(self._parse_range(items))
        return self

    def intersection_update(self, items):
        super(intspan, self).intersection_update(self._parse_range(items))
        return self

    def difference_update(self, items):
        super(intspan, self).difference_update(self._parse_range(items))
        return self

    def symmetric_difference_update(self, items):
        super(intspan, self).symmetric_difference_update(self._parse_range(items))
        return self

    def discard(self, items):
        for item in self._parse_range(items):
            super(intspan, self).discard(item)

    def remove(self, items):
        for item in self._parse_range(items):
            super(intspan, self).remove(item)

    def add(self, items):
        for item in self._parse_range(items):
            super(intspan, self).add(item)

    def issubset(self, items):
        return super(intspan, self).issubset(self._parse_range(items))

    def issuperset(self, items):
        return super(intspan, self).issuperset(self._parse_range(items))

    def union(self, items):
        return intspan(super(intspan, self).union(self._parse_range(items)))

    def intersection(self, items):
        return intspan(super(intspan, self).intersection(self._parse_range(items)))

    def difference(self, items):
        return intspan(super(intspan, self).difference(self._parse_range(items)))

    def symmetric_difference(self, items):
        return intspan(super(intspan, self).symmetric_difference(self._parse_range(items)))

    __le__   = issubset
    __ge__   = issuperset
    __or__   = union
    __and__  = intersection
    __sub__  = difference
    __xor__  = symmetric_difference
    __ior__  = update
    __iand__ = intersection_update
    __isub__ = difference_update
    __ixor__ = symmetric_difference_update

    def __eq__(self, items):
        return super(intspan, self).__eq__(self._parse_range(items))

    def __lt__(self, items):
        return super(intspan, self).__lt__(self._parse_range(items))

    def __gt__(self, items):
        return super(intspan, self).__gt__(self._parse_range(items))

    def __iter__(self):
        """
        Iterate in ascending order.
        """
        return iter(sorted(super(intspan, self).__iter__()))

    def pop(self):
        if self:
            min_item = min(self)
            self.discard(min_item)
            return min_item
        else:
            raise KeyError('pop from an empty set')

        # This method added only for PyPy, which otherwise would get the wrong
        # answer (unordered).

    def complement(self, low=None, high=None):
        """
        Return the complement of the given intspan--that is, all of the
        'missing' elements between its minimum and missing values.
        Optionally allows the universe set to be manually specified.
        """
        cls = self.__class__
        low = low if low is not None else min(self)
        high = high if high is not None else max(self)
        universe = cls.from_range(low, high)
        if not universe:
            raise ValueError('cannot represent infinite set')
        return universe - self

    @classmethod
    def from_range(cls, low, high):
        """
        Construct an intspan from the low value to the high value,
        inclusive. I.e., closed range, not the more typical Python
        half-open range.
        """
        return cls(range(low, high+1))

    @classmethod
    def from_ranges(cls, ranges):
        """
        Construct an intspan from a sequence of (low, high) value
        sequences (lists or tuples, say). Note that these values are
        inclusive, closed ranges, not the more typical Python
        half-open ranges.
        """
        return cls( chain( *(range(r[0], r[1]+1) for r in ranges) ) )

    @staticmethod
    def _parse_range(datum):

        def parse_chunk(chunk):
            """
            Parse each comma-separated chunk. Hyphens (-) can indicate ranges,
            or negative numbers. Returns a list of specified values. NB Designed
            to parse correct input correctly. Results of incorrect input are
            undefined.
            """
            m = SPANRE.search(chunk)
            if m:
                start = int(m.group('start'))
                if m.group('stop'):
                    stop = int(m.group('stop'))
                    return list(range(start, stop+1))
                return [ start ]
            else:
                raise ParseError("Can't parse chunk '{0}'".format(chunk))

        if isinstance(datum, basestring):
            result = []
            for part in datum.split(','):
                result.extend(parse_chunk(part))
            return result
        else:
            return datum if hasattr(datum, '__iter__') else [ datum ]

    @staticmethod
    def _as_range(iterable):
        l = list(iterable)
        if len(l) > 1:
            return (l[0], l[-1])
        else:
            return (l[0], l[0])

    @staticmethod
    def _as_range_str(iterable):
        l = list(iterable)
        if len(l) > 1:
            return '{0}-{1}'.format(l[0], l[-1])
        else:
            return '{0}'.format(l[0])

    def __repr__(self):
        """
        Return the representation.
        """
        clsname = self.__class__.__name__
        return '{0}({1!r})'.format(clsname, self.__str__())

    def __str__(self):
        """
        Return the stringification.
        """
        items = sorted(self)
        return ','.join(self._as_range_str(g) for _, g in groupby(items, key=lambda n, c=count(): n-next(c)))

    def ranges(self):
        """
        Return a list of the set's contiguous (inclusive) ranges.
        """
        items = sorted(self)
        return [ self._as_range(g) for _, g in groupby(items, key=lambda n, c=count(): n-next(c)) ]

    # see Jeff Mercado's answer to http://codereview.stackexchange.com/questions/5196/grouping-consecutive-numbers-into-ranges-in-python-3-2
    # see also: http://stackoverflow.com/questions/2927213/python-finding-n-consecutive-numbers-in-a-list


# It might be interesting to have a metaclass factory that could create
# spansets of things other than integers. For example, enumerateds defined
# by giving a universe of possible options. Or characters.