import numpy


class SubedgeRoutingInfo(object):
    """ Associates a partitioned_edge to its routing information
        (keys and masks)
    """

    def __init__(self, keys_and_masks, subedge, key_list_function=None):
        """
        :param keys_and_masks: The keys allocated to the partitioned edge
        :type keys_and_masks: iterable of\
                    :py:class:`pacman.model.routing_info.key_and_mask.KeyAndMask`
        :param subedge: The partitioned edge to which the keys and masks have\
                    been assigned
        :type subedge:\
                    :py:class:`pacman.model.partitioned_graph.partitioned_edge.PartitionedEdge`
        :param key_list_function: Optional function which will be called to\
                    translate the keys_and_masks list into individual keys.\
                    If missing, the keys will be generated by iterating\
                    through the keys_and_masks list directly.  The function\
                    parameters are:
                    * An iterable of keys and masks
                    * A partitioned edge
                    * Number of keys to generate (may be None)
        :type key_list_function: (iterable of\
                    :py:class:`pacman.model.routing_info.key_and_mask.KeyAndMask`,\
                    :py:class:`pacman.model.partitioned_graph.partitioned_edge.PartitionedEdge`,
                    int)\
                    -> iterable of int
        """
        self._keys_and_masks = keys_and_masks
        self._subedge = subedge
        self._key_list_function = key_list_function

    def get_keys(self, n_keys=None):
        """ Get the ordered list of individual keys allocated to the subedge

        :param n_keys: Optional limit on the number of keys to return
        :type n_keys: int
        :return: An array of keys
        :rtype: array-like of int
        """
        if self._key_list_function is not None:
            return self._key_list_function(self._keys_and_masks, self._subedge,
                                           n_keys)

        if n_keys is None:
            n_keys = 0
            for key_and_mask in self._keys_and_masks:
                n_keys += key_and_mask.n_keys

        key_array = numpy.zeros(n_keys, dtype="<u4")
        offset = 0
        for key_and_mask in self._keys_and_masks:
            _, offset = key_and_mask.get_keys(
                key_array=key_array, offset=offset, n_keys=(n_keys - offset))
        return key_array

    @property
    def keys_and_masks(self):
        return self._keys_and_masks

    @property
    def subedge(self):
        return self._subedge

    def __repr__(self):
        return "subedge:{}, keys_and_masks:{}"\
            .format(self._subedge, self._keys_and_masks)

    def __str__(self):
        return self.__repr__()
