# -*- coding: utf-8 -*-
#
# Copyright (c) 2007 - 2009 -- Lars Heuer - Semagia <http://www.semagia.com/>.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#
#     * Redistributions in binary form must reproduce the above
#       copyright notice, this list of conditions and the following
#       disclaimer in the documentation and/or other materials provided
#       with the distribution.
#
#     * Neither the name 'Semagia' nor the name 'Mappa' nor the names of the
#       contributors may be used to endorse or promote products derived from 
#       this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
"""\
This module provides a ``TopicMapWriter`` which writes 
`XML Topic Maps (XTM) 2.0 <http://www.isotopicmaps.org/sam/sam-xtm/>`_

:author:       Lars Heuer (heuer[at]semagia.com)
:organization: Semagia - http://www.semagia.com/
:version:      $Rev: 161 $ - $Date: 2009-06-15 15:42:51 +0200 (Mo, 15 Jun 2009) $
:license:      BSD license
"""
from cStringIO import StringIO
from xml.sax import make_parser
from xml.sax.xmlreader import InputSource
import xml.sax.handler as sax_handler
from mappa import XSD
from mappa.utils import is_default_name
from mappa.writer import AbstractTopicMapWriter
from mappa.writer.xmlutils import XMLWriter, xmlwriter_as_contenthandler

from mappa import voc
_NS_XTM = voc.XTM
del voc


class XTM20TopicMapWriter(AbstractTopicMapWriter):
    """\
    The XTM 2.0 writer.
    """
    def __init__(self, out, base, encoding='utf-8'):
        super(XTM20TopicMapWriter, self).__init__(base)
        if not out:
            raise TypeError('"out" is not specified')
        self._parser = None    # SAX parser used to serialize occurrence / variants values of datatype xs:anyType
        self._writer = None    # XMLWriter instance to serialize a topic map
        self._out = out
        self._encoding = encoding
        self.export_iids = True
        self.prettify = False

    def write(self, topicmap):
        """\
        Serializes the specified ``topicmap``.
        """
        self._writer = XMLWriter(self._out, self._encoding)
        writer = self._writer
        writer.prettify = self.prettify
        writer.startDocument()
        attrs = {'xmlns': _NS_XTM, 'version': '2.0'}
        if topicmap.reifier:
            attrs['reifier'] = '#%s' % self._id(topicmap.reifier)
        writer.startElement('topicMap', attrs)
        write_topic = self._write_topic
        for topic in topicmap.topics:
            write_topic(topic)
        write_assoc = self._write_association
        for assoc in topicmap.associations:
            write_assoc(assoc)
        writer.endElement('topicMap')
        writer.endDocument()

    def _write_topic(self, topic):
        """\
        Serializes a topic and its characteristics.
        
        `topic`
            The topic to serialize
        """
        self._writer.startElement('topic', {'id': self._id(topic)})
        self._write_locators('subjectIdentifier', topic.sids)
        self._write_locators('subjectLocator', topic.slos)
        self._write_iids(topic)
        write_name = self._write_name
        for name in topic.names:
            write_name(name)
        write_occurrence = self._write_occurrence
        for occ in topic.occurrences:
            write_occurrence(occ)
        self._writer.endElement('topic')

    def _write_occurrence(self, occ):
        """\
        Serializes the occurrence.
        
        `occurrence`
            An occurrence.
        """
        self._writer.startElement('occurrence', self._reifier(occ))
        self._write_type(occ)
        self._write_scope(occ)
        self._write_iids(occ)
        self._write_data(occ)
        self._writer.endElement('occurrence')

    def _write_name(self, name):
        """\
        Serializes the topic name and its variants.
        
        `name`
            A name.
        """
        self._writer.startElement('name', self._reifier(name))
        if not is_default_name(name):
            self._write_type(name)
        self._write_scope(name)
        self._write_iids(name)
        self._writer.dataElement('value', name.value)
        write_variant = self._write_variant
        for variant in name.variants:
            write_variant(variant)
        self._writer.endElement('name')

    def _write_variant(self, variant):
        """\
        Serializes a variant.
        
        `variant`
            The variant to serialize.
        """
        self._writer.startElement('variant', self._reifier(variant))
        self._write_scope(variant)
        self._write_iids(variant)
        self._write_data(variant)
        self._writer.endElement('variant')

    def _write_association(self, assoc):
        """\
        Serializes an association.
        
        `association`
            An association.
        """
        roles = tuple(assoc.roles)
        if not roles:
            return
        self._writer.startElement('association', self._reifier(assoc))
        self._write_type(assoc)
        self._write_scope(assoc)
        self._write_iids(assoc)
        write_role = self._write_role
        for role in roles:
            write_role(role)
        self._writer.endElement('association')

    def _write_role(self, role):
        """\
        Serializes the ``role``.
        
        `role´
            The role to serialize.
        """
        self._writer.startElement('role', self._reifier(role))
        self._write_iids(role)
        self._write_type(role)
        self._write_topic_ref(role.player)
        self._writer.endElement('role')

    def _write_data(self, datatyped):
        """\
        Serializes the ``value`` and ``datatype`` property of an occurrence or
        variant.

        `datatyped`
            Either an occurrence instance or a variant instance.
        """
        startElement = self._writer.startElement
        endElement = self._writer.endElement
        characters = self._writer.characters
        dt, val = datatyped.datatype, datatyped.value
        if dt == XSD.anyURI:
            self._writer.emptyElement('resourceRef', self._href(val))
        else:
            attrs = None
            if dt != XSD.string:
                attrs = {'datatype': dt}
            startElement('resourceData', attrs)
            if dt == XSD.anyType:
                if self._parser is None:
                    self._parser = make_parser()
                    self._parser.setFeature(sax_handler.feature_namespaces, True)
                self._parser.setContentHandler(AnyTypeContentHandler(xmlwriter_as_contenthandler(self._writer)))
                src = InputSource()
                src.setByteStream(StringIO(val))
                self._parser.parse(src)
            else:
                characters(val)
            endElement('resourceData', indent=False)

    def _write_iids(self, tmc):
        """\
        Serializes item identifiers identifiers.
        
        `iids`
            An iterable of item identifiers.
        """
        if not self.export_iids:
            return
        self._write_locators('itemIdentity', tmc.iids)

    def _write_locators(self, name, locs):
        """\
        Serializes the specified locators under the specified ``name`` if
        ``locs`` is not empty.
        
        `name`
            The element name of the IRIs
        `locs`
            An iterable of IRIs.
        """
        emptyElement, href = self._writer.emptyElement, self._href
        for loc in locs:
            emptyElement(name, href(loc))

    def _write_type(self, typed):
        """\
        Serizalizes the ``type`` of a typed Topic Maps construct.
        
        `typed`
            A typed Topic Maps construct (association, role, occurrence, name).
        """
        self._writer.startElement('type')
        self._write_topic_ref(typed.type)
        self._writer.endElement('type')

    def _write_scope(self, scoped):
        """\
        Serizalizes the ``scope`` of the `scoped` Topic Maps construct.
        
        `scoped`
            The scoped Topic Maps construct (association, occurrence, name, variant)
        """
        write_topic_ref = self._write_topic_ref
        written = False
        for i, theme in enumerate(scoped.scope):
            if not i:
                self._writer.startElement('scope')
                written = True
            write_topic_ref(theme)
        if written:
            self._writer.endElement('scope')

    def _write_topic_ref(self, topic):
        """\
        Writes a ``<topicRef href="#topic-ref"/>`` element.
        """
        self._writer.emptyElement('topicRef', self._href('#%s' % self._id(topic)))

    def _reifier(self, reifiable):
        """\
        Returns a ``dict```iff the ``reifiable`` Topic Maps construct has a reifier.
        """
        reifier = reifiable.reifier
        if not reifier:
            return None
        return {'reifier': '#%s' % self._id(reifier)}

    def _href(self, iri):
        """\
        Returns a ``{"href": iri}`` dict.
        """
        return {'href': iri}

class AnyTypeContentHandler(sax_handler.ContentHandler):
    """\
    ``ContentHandler`` implementation that redirects events to another
    ``ContentHandler``. The ``startDocument`` and ``endDocument`` events are
    ignored.
    """
    def __init__(self, sub_handler):
        sax_handler.ContentHandler.__init__(self)
        self._handler = sub_handler

    def startPrefixMapping(self, prefix, uri):
        self._handler.startPrefixMapping(prefix, uri)

    def endPrefixMapping(self, prefix):
        self._handler.endPrefixMapping(prefix)

    def startElement(self, name, attrs):
        self._handler.startElement(name, attrs)

    def endElement(self, name):
        self._handler.endElement(name)

    def startElementNS(self, name, qname, attrs):
        self._handler.startElementNS(name, qname, attrs)

    def endElementNS(self, name, qname):
        self._handler.endElementNS(name, qname)

    def characters(self, content):
        self._handler.characters(content)

    def ignorableWhitespace(self, content):
        self._handler.ignorableWhitespace(content)

    def processingInstruction(self, target, data):
        self._handler.processingInstruction(target, data)
