# -*- coding: utf-8 -*-
#
# Copyright (c) 2007 - 2009 -- Lars Heuer - Semagia <http://www.semagia.com/>.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#
#     * Redistributions in binary form must reproduce the above
#       copyright notice, this list of conditions and the following
#       disclaimer in the documentation and/or other materials provided
#       with the distribution.
#
#     * Neither the name 'Semagia' nor the name 'Mappa' nor the names of the
#       contributors may be used to endorse or promote products derived from 
#       this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
from xml.sax.saxutils import XMLGenerator, escape, quoteattr

__all__ = ['XMLWriter', 'xmlwriter_as_contenthandler']

class XMLWriter(object):
    """\
    Simple SAX alike XML writer
    """
    __slots__ = ['_out', '_encoding', '_depth', 'prettify']
    
    def __init__(self, out, encoding='utf-8'):
        """\
        
        """
        self._out = out
        self._encoding = encoding
        self._depth = 0
        self.prettify = False

    def startDocument(self):
        """\
        Writes the <?xml version="1.0" ... ?> declaration.
        """
        out, nl = self._out, self._newline
        out.write('<?xml version="1.0" encoding="%s" standalone="yes"?>' % self._encoding)
        nl()
        out.write('<!-- Generated by Mappa - http://mappa.semagia.com/ -->')
        if not self.prettify:
            nl()
        self._depth = 0
    
    def endDocument(self):
        """\
        Flushes to the output.
        """
        self._newline()
        self._out.flush()
    
    def startElement(self, name, attrs=None):
        """\
        Writes a start tag with the optional attributes (a dict).
        """
        self._indent()
        self._out.write('<%s' % name)
        self._write_attributes(attrs)
        self._out.write('>')
        self._depth+=1
    
    def endElement(self, name, indent=True):
        """\
        Writes an end tag. 
        
        `name`
            The name of the tag.
        `indent`
            Indicating if whitespaces in front of the element are allowed.
        """
        self._depth-=1
        if indent:
            self._indent()
        self._out.write('</%s>' % name)
    
    def dataElement(self, name, data, attrs=None):
        """\
        Writes a start tag, the data and an end tag.
        """
        self.startElement(name, attrs)
        self.characters(data)
        self.endElement(name, False)

    def emptyElement(self, name, attrs=None):
        """\
        Writes ``<name att1="attr-val1" attr2="attr-val2"/>``
        """
        self._indent()
        out = self._out
        out.write('<%s' % name)
        self._write_attributes(attrs)
        out.write('/>')
    
    def characters(self, content):
        """\
        Writes an escaped value.
        """
        self._out.write(escape(content))

    def _write_attributes(self, attrs):
        """\
        Serializes the attributes (a ``dict`` or ``None``), if any.
        """
        if attrs:
            write = self._out.write
            for k, v in attrs.items():
                write(' %s=%s' % (k, quoteattr(v)))

    def _indent(self):
        """\
        Indents a line.
        """
        if self.prettify:
            self._newline()
            self._out.write(' ' * self._depth * 2)

    def _newline(self):
        """\
        Writes a newline character.
        """
        self._out.write('\n')


def xmlwriter_as_contenthandler(writer):
    """\
    Returns a ContentHandler which serializes the events.
    
    All events are serialized to the ``_out`` property of the `writer` using 
    the ``writer._encoding``.
    """
    # pylint: disable-msg=W0212
    return XMLGenerator(writer._out, writer._encoding)
