from numpy import atleast_2d
from flopy.mbase import Package


class ModflowGhb(Package):
    '''General head boundary package class
    layer_row_column_head_cond is a list of 2D arrays. Each array has 5 columns containing
    the layer #, row #, column #, head value, and conductance value of a ghb cell.
    Enter a separate array for each stress period. When fewer arrays are supplied than stress periods,
    the ghb cells of the last array are repeated.
    No parameters implemented'''

    def __init__(self, model, ighbcb=0, layer_row_column_data=None, no_print=False,
                 options=None, naux=0,
                 extension='ghb', unitnumber=23):
        Package.__init__(self, model, extension, 'GHB',
                         unitnumber)  # Call ancestor's init to set self.parent, extension, name and unit number
        self.heading = '# GHB for MODFLOW, generated by Flopy.'
        self.url = 'ghb.htm'
        self.ighbcb = ighbcb  # no cell by cell terms are written
        self.mxactb, self.layer_row_column_data = self.assign_layer_row_column_data(layer_row_column_data,
                                                                                    5 + naux)
        self.no_print = no_print
        self.np = 0
        if options is None:
            options = []
        if self.no_print:
            options.append('NOPRINT')
        self.options = options
        self.parent.add_package(self)

    def __repr__(self):
        return 'GHB package class'

    def ncells(self):
        # Returns the  maximum number of cells that have a well (developped for MT3DMS SSM package)
        return self.mxactb

    def write_file(self):
        f_ghb = open(self.fn_path, 'w')
        f_ghb.write('%s\n' % self.heading)
        f_ghb.write('%10i%10i' % (self.mxactb, self.ighbcb))
        for option in self.options:
            f_ghb.write('  {}'.format(option))
        f_ghb.write('\n')
        self.write_layer_row_column_data(f_ghb, self.layer_row_column_head_cond)
        f_ghb.close()

    @staticmethod
    def load(f, model, nper=None, ext_unit_dict=None):
        """
        Load an existing package.

        Parameters
        ----------
        f : filename or file handle
            File to load.
        model : model object
            The model object (of type :class:`flopy.modflow.mf.Modflow`) to
            which this package will be added.
        nper : int
            The number of stress periods.  If nper is None, then nper will be
            obtained from the model object. (default is None).
        ext_unit_dict : dictionary, optional
            If the arrays in the file are specified using EXTERNAL,
            or older style array control records, then `f` should be a file
            handle.  In this case ext_unit_dict is required, which can be
            constructed using the function
            :class:`flopy.utils.mfreadnam.parsenamefile`.

        Returns
        -------
        rch : ModflowGhb object
            ModflowGhb object.

        Examples
        --------

        >>> import flopy
        >>> m = flopy.modflow.Modflow()
        >>> ghb = flopy.modflow.ModflowGhb.load('test.ghb', m)

        """
        if type(f) is not file:
            filename = f
            f = open(filename, 'r')
        # dataset 0 -- header
        while True:
            line = f.readline()
            if line[0] != '#':
                break
        # --check for parameters
        if "parameter" in line.lower():
            raw = line.strip().split()
            assert int(raw[1]) == 0, "Parameters are not supported"
            line = f.readline()
        #dataset 2a
        t = line.strip().split()
        mxactb = int(t[0])
        ighbcb = None
        ighncb = 0
        try:
            if int(t[1]) != 0:
                ighbcb = 53
        except:
            pass

        options = []
        naux = 0
        if len(t) > 2:
            for toption in t[3:-1]:
                if toption.lower() is 'noprint':
                    options.append(toption)
                elif 'aux' in toption.lower():
                    naux += 1
                    options.append(toption)
        if nper is None:
            nrow, ncol, nlay, nper = model.get_nrow_ncol_nlay_nper()
        #read data for every stress period
        layer_row_column_data = []
        current = []
        nitems = 5 + naux
        for iper in xrange(nper):
            print "   loading ghbs for kper {0:5d}".format(iper + 1)
            line = f.readline()
            t = line.strip().split()
            itmp = int(t[0])
            if itmp == 0:
                current = []
            elif itmp > 0:
                for ibnd in xrange(itmp):
                    line = f.readline()
                    t = line.strip().split()
                    bnd = []
                    for jdx in xrange(nitems):
                        if jdx < 3:
                            bnd.append(int(t[jdx]))
                        else:
                            bnd.append(float(t[jdx]))
                    current.append(bnd)
            layer_row_column_data.append(current)
        ghb = ModflowGhb(model, ighbcb=ighbcb,
                         layer_row_column_data=layer_row_column_data,
                         options=options, naux=naux)
        return ghb
