#!/usr/bin/env python
# -*- coding: utf-8 -*-
import functools
import subprocess
from .core import core, Timer
from .targets import Target
from .variable import Variable


class BaseRule(object):

    def __init__(self, func, *args, **kwargs):
        # Name, function and description from decorator.
        self._name = func.__name__
        self._func = func
        self._desc = func.__doc__

        # Optional rule dependencies (`depends` keyword argument).
        self._rule_depends = self._normalise(kwargs.get("depends", []), Target)
        # TODO: Keyword argument `options`.
        # Optional rule arguments (`args` keyword argument.)
        self._rule_args = kwargs.get("args", None)

        # Targets and dependencies.
        self._targets = []
        self._depends = []

        # Process decorator positional arguments.
        for i, arg in enumerate(args):
            if not self._process(i, arg):
                break

        # TODO: Check internal data.
        self._targets = tuple(self._targets)
        self._depends = tuple(self._depends)

    def __call__(self):
        results = {
            # Total number of targets, updated targets.
            "total": len(self._targets),
            "updated": 0,
            # Run results.
            "results": {},
        }

        with Timer() as t:

            # Check rule dependencies are up to date.
            # TODO: Collect timing results from dependencies.
            upd = False
            for dep in self._rule_depends:
                # TODO: Change less than to method call.
                if dep.out_of_date():
                    upd = True

            # If no targets, RulePattern1.
            if results["total"] == 0:
                results["total"] += 1
                results["results"] = self.call(upd)
            else:
                results["results"] = self.call(
                    upd, self._targets, self._depends
                )

        return self._results(t, results)

    @classmethod
    def _normalise(cls, arg, check):
        # Wrap argument as tuple for consistency.
        if isinstance(arg, check):
            return tuple([arg])
        elif isinstance(arg, list) or isinstance(arg, tuple):
            return tuple(arg)
        # TODO: Raise error for unexpected input.
        return tuple([])

    def _process(self, i, arg):
        # If argument is Target instance, RulePattern2.
        if isinstance(arg, Target):
            self._targets.append(arg)
            self._depends.append(None)
            return True

        # Else if argument is list or tuple.
        elif isinstance(arg, list) or isinstance(arg, tuple):

            # TODO: Raise error if length greater than 2.
            if len(arg) > 2:
                return False

            # Extract targets, dependencies from argument list.
            targets = arg[0] if len(arg) > 0 else None
            depends = arg[1] if len(arg) > 1 else None

            # If targets is Target instance.
            if isinstance(targets, Target):
                self._targets.append(targets)

                # If dependencies is Target instance, RulePattern3.
                if isinstance(depends, Target):
                    self._depends.append(tuple([depends]))
                    return True

                # Else if dependencies is list or tuple, RulePattern4.
                elif isinstance(depends, list) or isinstance(depends, tuple):
                    self._depends.append(tuple(depends))
                    return True

            # Else if targets is list or tuple.
            elif isinstance(targets, list) or isinstance(targets, tuple):

                # If dependencies is a Target instance, RulePattern5.
                if isinstance(depends, Target):
                    for targ in targets:
                        self._targets.append(targ)
                        self._depends.append(tuple([depends]))
                    return True

                # Else if dependencies is list or tuple.
                elif isinstance(depends, list) or isinstance(depends, tuple):

                    # If not equal in length, RulePattern7.
                    if len(targets) != len(depends):
                        for targ in targets:
                            self._targets.append(targ)
                            self._depends.append(tuple(depends))
                        return True

                    # If equal in length.
                    for targ, dep in zip(targets, depends):
                        self._targets.append(targ)

                        # If dependency is Target, RulePattern6.
                        if isinstance(dep, Target):
                            self._depends.append(tuple([dep]))

                        # Else if dependency is list or tuple, RulePattern8.
                        elif isinstance(dep, list) or isinstance(dep, tuple):
                            self._depends.append(tuple(dep))

                        # Unknown dependency argument.
                        else:
                            return False
                    return True

        # No arguments, RulePattern1.
        # TODO: Raise error for unknown argument.
        return False

    def _results(self, timer, results):
        # TODO: Timing information for collected results.
        rc = True

        for i, result in results["results"].items():
            # Write error messages to stderr.
            if result["error"] is not None:
                core.write_stderr(result["error"])
                rc = False
            else:
                results["updated"] += 1

        results["time"] = timer.seconds
        return (rc, results)

    def call_func(self, target=None, depends=None):
        result = {"error": None}
        # TODO: Variable context management.
        ctx = Variable.save()

        if target is not None:
            Variable("_T", str(target))
            if depends is not None:
                Variable("_D", " ".join([str(x) for x in depends]))

        with Timer() as t:
            try:
                self._func(target, depends, self._rule_args)
            except KeyboardInterrupt:
                result["error"] = "keyboard interrupt"
            except subprocess.CalledProcessError as e:
                # Subprocess error during function call.
                result["error"] = e.output
            # TODO: Catch unknown exceptions.

        Variable.restore(ctx)

        # Record time in seconds.
        result["time"] = t.seconds
        return result

    def call(self, update=False, targets=None, depends=None):
        if targets is None:
            return {0: self.call_func()}

        results = {}
        # Run function with out of date targets and dependencies.
        for i, pair in enumerate(zip(self._targets, self._depends)):
            targ, depends = pair

            # Use the rule dependencies update override.
            upd = update

            # Check if target is out of date.
            # TODO: Debugging information.
            if targ < None:
                upd = True

            # Check if target is out of date compared to each dependency.
            if depends is not None:
                for dep in depends:
                    try:
                        if targ.out_of_date(dep):
                            upd = True
                    except Exception:
                        # TODO: Better error handling.
                        results[i] = {
                            "error": "dependency error",
                            "time": 0,
                        }
                        return results

            # If update required, run function.
            if upd:
                results[i] = self.call_func(targ, depends)

        return results

    def add_options(self, parser):
        pass

    def call_options(self, args):
        pass

    def get_description(self):
        return self._desc

    def get_targets(self):
        return self._targets


def rule(*args, **kwargs):
    def _decorator(func):
        # TODO: Use multiprocessing subclass here, keyword argument?
        _rule = BaseRule(func, *args, **kwargs)

        @functools.wraps(func)
        def _method():
            return _rule()

        _method._rule = _rule
        return _method
    return _decorator


def is_rule(obj):
    return hasattr(obj, "_rule")


# TODO: Complete refactoring/documentation and remove deprecated code.
# from .option import Option
# from .exceptions import (InvalidTargetError, InvalidOptionError)
#
#
# class _Rule(object):
#     # Error messages.
#     EKWARG_DEPENDS = "rule `{}` argument `depends` is not an instance of `Target` class or a list"  # noqa
#     EKWARG_OPTIONS = "rule `{}` argument `options` is not an instance of `Option` class or a list"  # noqa
#     EARG_LEN = "rule `{}` argument {} is a list of length greater than two"  # noqa
#     EARG_DEPEND = "rule `{}` argument {} dependencies is not an instance of `Target` class or a list"  # noqa
#     EARG_TARGET = "rule `{}` argument {} targets is not an instance of `Target` class or a list"  # noqa
#     EDEPEND = "rule `{}` dependency {} is not an instance of `Target` class"  # noqa
#     EOPTION = "rule `{}` option {} is not an instance of `Option` class"
#     ETARGET = "rule `{}` target {} is not an instance of `Target` class"
#     ETARGET_DEPENDS = "rule `{}` target {} dependencies is not a list"
#     ETARGET_DEPEND = "rule `{}` target {} dependency {} is not an instance of `Target` class"  # noqa
#     EEXECUTE = "execute `{}`"
#     ETARGET_OUT_OF_DATE = "target `{}` out of date"
#     EDEPEND_OUT_OF_DATE = "target `{}` dependency `{}` out of date"
#
#     def __init__(self, *args, **kwargs):
#         # Name, callback and description provided by default.
#         self._name = kwargs.get("name")
#         self._callback = kwargs.get("callback")
#         self._description = kwargs.get("description")
#
#         # Optional rule dependencies (`depends` keyword argument).
#         rule_depends = kwargs.get("depends", [])
#
#         # TODO: Tuple arguments support.
#
#         # Wrap argument in list for easier handling.
#         if isinstance(rule_depends, Target):
#             self._rule_depends = [rule_depends]
#         # Set attribute now (checked later).
#         elif isinstance(rule_depends, list):
#             self._rule_depends = rule_depends
#         # Else raise an error for invalid argument.
#         else:
#             core.raise_exception(
#                 self.EKWARG_DEPENDS, self._name,
#                 cls=InvalidTargetError,
#             )
#
#         # Optional rule command line options (`options` keyword argument).
#         rule_options = kwargs.get("options", [])
#
#         # Wrap argument in list for easier handling.
#         if Option.is_option(rule_options):
#             self._rule_options = [rule_options]
#         # Set attribute now (checked later).
#         elif isinstance(rule_options, list):
#             self._rule_options = rule_options
#         # Else raise error for invalid argument.
#         else:
#             core.raise_exception(
#                 self.EKWARG_OPTIONS, self._name,
#                 cls=InvalidOptionError,
#             )
#
#         # Optional rule arguments (`args` keyword argument).
#         self._args = kwargs.get("args", None)
#
#         # Targets and target dependencies attributes.
#         self._targets = []
#         self._depends = []
#
#         # Decorator positional arguments.
#         for i, arg in enumerate(args):
#             # If argument is a Target instance, RulePattern2.
#             if isinstance(arg, Target):
#                 self._targets.append(arg)
#                 self._depends.append(None)
#
#             # Else if argument is list.
#             elif isinstance(arg, list):
#
#                 # Raise error if list has more than two elements.
#                 if len(arg) > 2:
#                     core.raise_exception(
#                         self.EARG_LEN, self._name, i,
#                         cls=InvalidTargetError,
#                     )
#
#                 # Extract target, dependencies from argument list.
#                 targets = arg[0] if len(arg) > 0 else None
#                 depends = arg[1] if len(arg) > 1 else None
#
#                 # If targets is a Target instance.
#                 if isinstance(targets, Target):
#                     self._targets.append(targets)
#
#                     # If depends is a Target instance, RulePattern3.
#                     if isinstance(depends, Target):
#                         self._depends.append([depends])
#                     # Else if depends is a list, RulePattern4.
#                     elif isinstance(depends, list):
#                         self._depends.append(depends)
#                     # Else raise error for invalid argument.
#                     else:
#                         core.raise_exception(
#                             self.EARG_DEPEND, self._name, i,
#                             cls=InvalidTargetError,
#                         )
#
#                 # Else if targets is a list of Target instances.
#                 elif isinstance(targets, list):
#
#                     # If depends is a Target instance, RulePattern5.
#                     if isinstance(depends, Target):
#                         for target in targets:
#                             self._targets.append(target)
#                             self._depends.append([depends])
#
#                     # Else if depends is a list.
#                     elif isinstance(depends, list):
#
#                         # If lists are equal in length.
#                         if len(targets) == len(depends):
#                             for target, depend in zip(targets, depends):
#                                 self._targets.append(target)
#
#                                 # If depend is a Target, RulePattern6.
#                                 if isinstance(depend, Target):
#                                     self._depends.append([depend])
#                                 # Else if depend is a list, RulePattern8.
#                                 elif isinstance(depend, list):
#                                     self._depends.append(depend)
#                                 # Else unknown depends argument.
#                                 else:
#                                     core.raise_exception(
#                                         self.EARG_DEPEND, self._name, i,
#                                         cls=InvalidTargetError,
#                                     )
#
#                         # Else, RulePattern7.
#                         else:
#                             for target in targets:
#                                 self._targets.append(target)
#                                 self._depends.append(depends)
#                     # Else unknown depends argument.
#                     else:
#                         core.raise_exception(
#                             self.EARG_DEPEND, self._name, i,
#                             cls=InvalidTargetError,
#                         )
#                 # Else unknown targets argument.
#                 else:
#                     core.raise_exception(
#                         self.EARG_TARGET, self._name, i,
#                         cls=InvalidTargetError,
#                     )
#             # Else unknown primary argument.
#             else:
#                 core.raise_exception(
#                     self.EARG_TARGET, self._name, i,
#                     cls=InvalidTargetError,
#                 )
#
#         # No arguments, RulePattern1.
#
#         # Test internal data for correctness.
#         # Rule dependencies must be a list of Target instances.
#         for i, depend in enumerate(self._rule_depends):
#             assert isinstance(depend, Target), (
#                 self.EDEPEND.format(self._name, i)
#             )
#         # Rule options must be a list of Option instances.
#         for i, option in enumerate(self._rule_options):
#             assert Option.is_option(option), (
#                 self.EOPTION.format(self._name, i)
#             )
#         # Targets must be a list of Target instances.
#         # Target dependencies must be a list of lists of Target instances.
#         pairs = zip(self._targets, self._depends)
#         for i, pair in enumerate(pairs):
#             target, depends = pair
#             assert isinstance(target, Target), (
#                 self.ETARGET.format(self._name, i)
#             )
#             if depends is not None:
#                 assert isinstance(depends, list), (
#                     self.ETARGET_DEPENDS.format(self._name, i)
#                 )
#                 for j, depend in enumerate(depends):
#                     assert isinstance(depend, Target), (
#                         self.ETARGET_DEPEND.format(self._name, i, j)
#                     )
#
#     def __call__(self):
#         """Return tuple of boolean indicating success and data providing
#         further execution information (updated, total, elapsed time).
#         """
#         # TODO: Rule callback concurrency using deco?
#         def _rule_callback(callback, target=None, depends=None, args=None):
#             # Execute rule callback.
#             core.debug(__name__, self.EEXECUTE, self._name)
#             result = {"error": None}
#
#             with Timer() as timer:
#                 try:
#                     callback(target, depends, args)
#                 except KeyboardInterrupt:
#                     result["error"] = "keyboard interrupt"
#                 except subprocess.CalledProcessError as e:
#                     # Subprocess error during callback.
#                     result["error"] = e.output
#                 # TODO: Do not catch unknown exceptions yet.
#                 # except Exception as e:
#                 #     # Catch generic/unknown exceptions.
#                 #     result["error"] = str(e)
#
#             # Record time taken in seconds.
#             result["time"] = timer.seconds
#             return result
#
#         def _rule_variables(target, depends):
#             # Set rule variables based on target and dependencies.
#             # Current target string.
#             Variable("_T", str(target))
#             # Current dependencies list.
#             if depends is not None:
#                 Variable("_D", " ".join([str(x) for x in depends]))
#
#         def _rule_single(callback, args=None):
#             # Execute single rule callback without target/dependencies.
#             # TODO: Set variables here?
#             return {
#                 0: _rule_callback(callback, args=args),
#             }
#
#         def _rule_multiple(callback, targets, depends, args, update=False):
#             # Execute multiple rule callbacks with targets/dependencies.
#             # For each Target instance and dependencies list.
#             pairs = zip(targets, depends)
#             results = {}
#
#             for i, pair in enumerate(pairs):
#                 target, depends = pair
#
#                 # Use update override.
#                 upd = update
#
#                 # Save current automatic variables context.
#                 ctx = Variable.save()
#
#                 # Populate automatic variables.
#                 _rule_variables(target, depends)
#
#                 # Update if target out of date.
#                 if target < None:
#                     core.debug(
#                         __name__, self.ETARGET_OUT_OF_DATE, target
#                     )
#                     upd = True
#
#                 # Update if target out of date compared to dependencies.
#                 if depends is not None:
#                     for depend in depends:
#                         if target < depend:
#                             if not upd:
#                                 core.debug(
#                                     __name__, self.EDEPEND_OUT_OF_DATE,
#                                     target, depend
#                                 )
#                             upd = True
#
#                 # If update required, run rule callback.
#                 if upd:
#                     results[i] = _rule_callback(
#                         callback, target, depends, args
#                     )
#
#                 # Restore variable context.
#                 Variable.restore(ctx)
#
#             return results
#
#         # Rule keyword argument dependencies override update.
#         update = False
#
#         # Check rule dependencies are up to date.
#         # TODO: Include elapsed time from rule dependencies.
#         for depend in self._rule_depends:
#             if depend < None:
#                 update = True
#
#         # Total number of targets. updated.
#         total = len(self._targets)
#
#         # If targets list empty, single thread rule callback.
#         if total == 0:
#             # Adjust total to one for output readability.
#             total = 1
#             results = _rule_single(
#                 self._callback,
#                 self._args,
#             )
#         # Else pass target list to multiple threaded rule callback.
#         else:
#             results = _rule_multiple(
#                 self._callback,
#                 self._targets,
#                 self._depends,
#                 self._args,
#                 update,
#             )
#
#         # Track success, number of updated targets and time taken.
#         success = True
#         updated = 0
#         elapsed = 0
#
#         for i, result in results.items():
#             # Write error messages to stderr.
#             if result["error"] is not None:
#                 core.write_stderr(result["error"])
#                 success = False
#             else:
#                 updated += 1
#
#             # Increment elapsed counter.
#             elapsed += result["time"]
#
#         return (success, updated, total, elapsed)
#
#     def add_options(self, parser):
#         """Add rule options to argument parser.
#
#         `parser`
#             ArgumentParser instance.
#         """
#         for opt in self._rule_options:
#             opt.add_argument(parser)
#
#     def call_options(self, args):
#         """Call rule options with arguments return by argument parser.
#
#         `args`
#             Argument parser Namespace instance.
#         """
#         for opt in self._rule_options:
#             opt(getattr(args, opt.get_name()))
#
#     def get_description(self):
#         """Return rule description string."""
#         return self._description
#
#     def get_targets(self):
#         """Return internal list of rule targets."""
#         return self._targets
#
#
# class Rule(object):
#     """Rule function decorator, decorated callback function is run based on the
#     state of targets, dependencies and other keyword arguments.
#
#     `*args`
#         Target arguments in a defined rule pattern.
#     `**kwargs`
#         `depends`
#             Optional additional dependencies for rule.
#         `options`
#             Option instances for command line configuration.
#         `args`
#             Optional arguments passed to decorated callback function.
#     """
#     def __init__(self, *args, **kwargs):
#         self._args = args
#         self._kwargs = kwargs
#
#     def __call__(self, callback):
#         """Construct an internal rule instance from decorator arguments.
#
#         `callback`
#             Callback function accepting (target, depends, args) arguments.
#         """
#         self._kwargs.setdefault("name", callback.__name__)
#         self._kwargs.setdefault("callback", callback)
#         self._kwargs.setdefault("description", callback.__doc__)
#         return _Rule(*self._args, **self._kwargs)
#
#     @staticmethod
#     def is_rule(obj):
#         """Return true if object is a rule instance.
#
#         `obj`:
#             Object instance.
#         """
#         if isinstance(obj, _Rule):
#             return True
#         return False
