from zope.interface import verify

from twisted.internet import defer
from twisted.trial import unittest
from twisted.words.protocols.jabber.jid import JID

from wokkel import disco, iwokkel
from wokkel.generic import parseXml
from wokkel.test.helpers import XmlStreamStub

from jarn.xmpp.collaboration.tests import mock


class DifferentialSyncronisationHandlerTest(unittest.TestCase):
    """
    Tests for the DifferentialSynchronisationProtocol.
    """

    def setUp(self):
        self.stub = XmlStreamStub()
        self.protocol = mock.MockDifferentialSyncronisationHandler()
        self.protocol.xmlstream = self.stub.xmlstream
        self.protocol.connectionInitialized()

    def test_onPresence(self):
        """
        Upon receiving a presence, the protocol MUST set itself up,
        as well as send the initial text to the user. When the user leaves
        there's more bookkeeping.
        """
        self.protocol.mock_text['test-node'] = 'foo'
        xml = """<presence from='test@example.com' to='example.com'>
                    <query xmlns='http://jarn.com/ns/collaborative-editing'
                           node='test-node'/>
                 </presence>"""
        self.stub.send(parseXml(xml))
        self.assertEqual({u'test-node': set([u'test@example.com'])},
                         self.protocol.node_participants)
        self.assertEqual({u'test@example.com': set([u'test-node'])},
                          self.protocol.participant_nodes)
        self.assertEqual({u'test-node': 'foo'},
                         self.protocol.shadow_copies)

        message = self.stub.output[-1]
        self.assertEqual(
            "<message to='test@example.com'>" +
            "<x xmlns='http://jarn.com/ns/collaborative-editing'>" +
            "<item action='set' node='test-node'>foo</item>" +
            "</x></message>", message.toXml())

        xml = """<presence from='test@example.com' to='example.com'
                    type='unavailable'/>"""
        self.stub.send(parseXml(xml))
        self.assertEqual({}, self.protocol.node_participants)
        self.assertEqual({}, self.protocol.participant_nodes)
        self.assertEqual({}, self.protocol.shadow_copies)

    def test_onPatch(self):
        # 'foo' is the initial text. foo and bar present.
        self.protocol.mock_text['test-node'] = 'foo'
        xml = """<presence from='foo@example.com' to='example.com'>
                    <query xmlns='http://jarn.com/ns/collaborative-editing'
                           node='test-node'/>
                 </presence>"""
        self.stub.send(parseXml(xml))
        xml = """<presence from='bar@example.com' to='example.com'>
                    <query xmlns='http://jarn.com/ns/collaborative-editing'
                           node='test-node'/>
                 </presence>"""
        self.stub.send(parseXml(xml))

        # bar sends a patch changing the text to 'foobar'.
        xml = """<message from='bar@example.com' to='example.com'>
                    <x xmlns='http://jarn.com/ns/collaborative-editing'>
                        <item node='test-node' action='patch'>@@ -1,3 +1,6 @@\n foo\n+bar\n</item>
                    </x>
                </message>"""
        self.stub.send(parseXml(xml))

        # foo receives the same patch.
        message = self.stub.output[-1]
        self.assertEqual(
            "<message to='foo@example.com'>" +
            "<x xmlns='http://jarn.com/ns/collaborative-editing'>" +
            "<item action='patch' node='test-node' user='bar@example.com'>@@ -1,3 +1,6 @@\n foo\n+bar\n</item>" +
            "</x></message>",
            message.toXml())

        # The shadow copy is 'foobar'
        self.assertEqual(u'foobar', self.protocol.shadow_copies['test-node'])

    def test_interfaceIDisco(self):
        """
        The handler should provice Service Discovery information.
        """
        verify.verifyObject(iwokkel.IDisco, self.protocol)

    def test_getDiscoInfo(self):
        """
        The namespace should be returned as a supported feature.
        """

        def cb(info):
            discoInfo = disco.DiscoInfo()
            for item in info:
                discoInfo.append(item)
            self.assertIn('http://jarn.com/ns/collaborative-editing',
                          discoInfo.features)

        d = defer.maybeDeferred(self.protocol.getDiscoInfo,
                                JID('user@example.org/home'),
                                JID('pubsub.example.org'),
                                '')
        d.addCallback(cb)
        return d

    def test_getDiscoInfoNode(self):
        """
        The namespace should not be returned for a node.
        """

        def cb(info):
            discoInfo = disco.DiscoInfo()
            for item in info:
                discoInfo.append(item)
            self.assertNotIn('http://jarn.com/ns/collaborative-editing',
                             discoInfo.features)

        d = defer.maybeDeferred(self.protocol.getDiscoInfo,
                                JID('user@example.org/home'),
                                JID('pubsub.example.org'),
                                'test')
        d.addCallback(cb)
        return d

    def test_getDiscoItems(self):
        """
        Items are not supported by this handler.
        """

        def cb(items):
            self.assertEquals(0, len(items))

        d = defer.maybeDeferred(self.protocol.getDiscoItems,
                                JID('user@example.org/home'),
                                JID('pubsub.example.org'),
                                '')
        d.addCallback(cb)
        return d
