#:RICH: pylint this and see what makes sense to do
from __future__ import absolute_import
import os
import logging
import csv

from osgeo import gdal
from osgeo import ogr

from natcap.invest.coastal_vulnerability import coastal_vulnerability_core as cvc
from .. import validation
from .. import utils

LOGGER = logging.getLogger(__name__)

def execute(args):
    """Coastal Vulnerability.

    Parameters:
        workspace_dir (string): The path to the workspace directory on disk
            (required)
        aoi_uri (string): Path to an OGR vector on disk representing the area
            of interest. (required)
        landmass_uri (string): Path to an OGR vector on disk representing the
            global landmass. (required)
        bathymetry_uri (string): Path to a GDAL raster on disk representing
            the bathymetry.  Must overlap with the Area of Interest if
            if provided.
            (optional)
        bathymetry_constant (int): An int between 1 and 5 (inclusive).
            (optional)
        relief_uri (string): Path to a GDAL raster on disk representing the
            elevation within the land polygon provided. (optional)
        relief_constant (int): An int between 1 and 5 (inclusive).
            (optional)
        elevation_averaging_radius (int): a positive int.  The radius around
            which to compute the average elevation for relief.  Must be in
            meters. (required)
        mean_sea_level_datum (int): a positive int. This input is the
            elevation of Mean Sea Level (MSL) datum relative to the datum of
            the bathymetry layer that they provide. The model transforms all
            depths to MSL datum by subtracting the value provided by the user
            to the bathymetry. This input can be used to run the model for a
            future sea-level rise scenario. Must be in meters. (required)
        cell_size (int): Cell size in meters. The higher the value, the faster
            the computation, but the coarser the output rasters produced by
            the model. (required)
        depth_threshold (int): Depth in meters (integer) cutoff to determine
            if fetch rays project over deep areas. (optional)
        exposure_proportion (float): Minimum proportion of rays that project
            over exposed and/or deep areas need to classify a shore segment
            as exposed. (required)
        geomorphology_uri (string): A OGR-supported polygon vector file that
            has a field called "RANK" with values between 1 and 5 in the
            attribute table. (optional)
        geomorphology_constant (int): Integer value between 1 and 5. If layer
            associated to this field is omitted, replace all shore points for
            this layer with a constant rank value in the computation of the
            coastal vulnerability index. If both the file and value for the
            layer are omitted, the layer is skipped altogether.
        habitats_directory_uri (string): Directory containing OGR-supported
            polygon vectors associated with natural habitats. The name of
            these shapefiles should be suffixed with the ID that is specified
            in the natural habitats CSV file provided along with the habitats
            (optional)
        habitats_csv_uri (string): A CSV file listing the attributes for each
            habitat. For more information, see 'Habitat Data Layer' section in
            the model's documentation. (required if
            ``args['habitat_directory_uri']`` is provided)
        habitat_constant (int): Integer value between 1 and 5. If layer
            associated to this field is omitted, replace all shore points
            for this layer with a constant rank value in the computation of
            the coastal vulnerability index. If both the file and value for
            the layer are omitted, the layer is skipped altogether.
            (optional)
        area_computed (string):  Determine if the output data is about all the
            coast about sheltered segments only. Either ``'sheltered'`` or
            ``'both'`` (required)
        suffix (string):  A string that will be added to the end of the output
            file. (optional)
        climatic_forcing_uri (string):  An OGR-supported vector containing
            both wind wave information across the region of interest.
            (optional)
        climatic_forcing_constant (int):  Integer value between 1 and 5. If
            layer to this field is omitted, replace all shore points for this
            layer with a constant rank value in the computation of the coastal
            vulnerability index. If both the file and value for the layer are
            omitted, the layer is skipped altogether. (optional)
        continental_shelf_uri (string):  An OGR-supported polygon vector
            delineating edges of the continental shelf. Default is global
            continental shelf shapefile. If omitted, the user can specify
            depth contour. See entry below. (optional)
        depth_contour (int):  Used to delineate shallow and deep areas.
            Continental limit is at about 150 meters. (optional)
        sea_level_rise_uri (string):  An OGR-supported point or polygon
            vector file features with "Trend" fields in the attributes
            table. (optional)
        sea_level_rise_constant (int):  Integer value between 1 and 5. If layer
            to this field is omitted, replace all shore points for this layer
            with a constant rank value in the computation of the coastal
            vulnerability index. If both the file and value for the layer are
            omitted, the layer is skipped altogether. (optional)
        structures_uri (string):  An OGR-supported vector file containing
            rigid structures to identify the portions of the coast that is
            armored. (optional)
        structures_constant (int):  Integer value between 1 and 5. If layer
            associated this field is omitted, replace all shore points for
            this layer with a constant rank value in the computation of the
            coastal vulnerability index. If both the file and value for the
            layer are omitted, the layer is skipped altogether. (optional)
        population_uri (string):  A GDAL-supported raster file representing
            the population. (required)
        urban_center_threshold (int):  Minimum population required to consider
            shore segment a population center. (required)
        additional_layer_uri (string):  An OGR-supported vector file
            representing level rise, and will be used in the computation of
            coastal vulnerability and coastal vulnerability without habitat.
            (optional)
        additional_layer_constant (int):  Integer value between 1 and 5. If
            layer to this field is omitted, replace all shore points for this
            layer with a constant rank value in the computation of the coastal
            vulnerability index. If both the file and value for the layer are
            omitted, the layer is skipped altogether. (optional)
        rays_per_sector (int):  Number of rays used to subsample the fetch
            distance each of the 16 sectors. (required)
        max_fetch (int):  Maximum fetch distance computed by the model
            (>=60,000m). (optional)
        spread_radius (int):  Integer multiple of 'cell size'. The coast from
            geomorphology layer could be of a better resolution than the global
            landmass, so the shores do not necessarily overlap. To make them
            coincide, the shore from the geomorphology layer is widened by 1 or
            more pixels. The value should be a multiple of 'cell size' that
            indicates how many pixels the coast from the geomorphology layer is
            widened. The widening happens on each side of the coast (n pixels
            landward, and n pixels seaward). (required)
        population_radius (int):  Radius length in meters used to count the
            number people leaving close to the coast. (optional)

    Note:
        If neither ``args['bathymetry_uri']`` nor
        ``args['bathymetry_constant']`` is provided, bathymetry is ignored
        altogether.

        If neither ``args['relief_uri']`` nor ``args['relief_constant']`` is
        provided, relief is ignored altogether.

        If neither ``args['geomorphology_uri']`` nor
        ``args['geomorphology_constant']`` is provided, geomorphology is
        ignored altogether.

        If neither ``args['climatic_forcing_uri']`` nor
        ``args['climatic_forcing_constant']`` is provided, climatic_forcing is
        ignored altogether.

        If neither ``args['sea_level_rise_uri']`` nor
        ``args['sea_level_rise_constant']`` is provided, sea level rise is
        ignored altogether.

        If neither ``args['structures_uri']`` nor
        ``args['structures_constant']`` is provided, structures is ignored
        altogether.

        If neither ``args['additional_layer_uri']`` nor
        ``args['additional_layer_constant']`` is provided, the additional layer
        option is ignored altogether.

    Example args::

        args = {
            u'additional_layer_uri': u'CoastalProtection/Input/SeaLevRise_WCVI.shp',
            u'aoi_uri': u'CoastalProtection/Input/AOI_BarkClay.shp',
            u'area_computed': u'both',
            u'bathymetry_uri': u'Base_Data/Marine/DEMs/claybark_dem/hdr.adf',
            u'cell_size': 1000,
            u'climatic_forcing_uri': u'CoastalProtection/Input/WaveWatchIII.shp',
            u'continental_shelf_uri': u'CoastalProtection/Input/continentalShelf.shp',
            u'depth_contour': 150,
            u'depth_threshold': 0,
            u'elevation_averaging_radius': 5000,
            u'exposure_proportion': 0.8,
            u'geomorphology_uri': u'CoastalProtection/Input/Geomorphology_BarkClay.shp',
            u'habitats_csv_uri': u'CoastalProtection/Input/NaturalHabitat_WCVI.csv',
            u'habitats_directory_uri': u'CoastalProtection/Input/NaturalHabitat',
            u'landmass_uri': u'Base_Data/Marine/Land/global_polygon.shp',
            u'max_fetch': 12000,
            u'mean_sea_level_datum': 0,
            u'population_radius': 1000,
            u'population_uri': u'Base_Data/Marine/Population/global_pop/w001001.adf',
            u'rays_per_sector': 1,
            u'relief_uri': u'Base_Data/Marine/DEMs/claybark_dem/hdr.adf',
            u'sea_level_rise_uri': u'CoastalProtection/Input/SeaLevRise_WCVI.shp',
            u'spread_radius': 250,
            u'structures_uri': u'CoastalProtection/Input/Structures_BarkClay.shp',
            u'urban_center_threshold': 5000,
            u'workspace_dir': u'coastal_vulnerability_workspace'
        }

    Returns:
        None"""

    LOGGER.info('preprocessing inputs...')

    # cast all this stuff to an int, since the UI doesn't do it for us any
    # more.
    args = args.copy()
    for args_key in ('bathymetry_constant', 'relief_constant',
                     'elevation_averaging_radius', 'mean_sea_level_datum',
                     'cell_size', 'depth_threshold', 'geomorphology_constant',
                     'habitat_constant', 'climatic_forcing_constant',
                     'depth_contour', 'sea_level_rise_constant',
                     'structures_constant', 'urban_center_threshold',
                     'additional_layer_constant', 'rays_per_sector',
                     'max_fetch', 'spread_radius', 'population_radius'):
        try:
            args[args_key] = int(args[args_key])
        except (KeyError, ValueError):
            # KeyError when args_key not present
            # valueerror when args_key can't be cast to an int
            LOGGER.debug('Key %s not in args or not an int', args_key)

    try:
        args['exposure_proportion'] = float(args['exposure_proportion'])
    except KeyError:
        pass

    #:RICH: PULL ALL THE PREPROCESING INTO THIS FILE SINCE IT IS UI SPECIFIC AND coastal_vulnerability_core IS MODEL SPECIFIC.
    core_args = cvc.preprocess_inputs(args)
    LOGGER.info('passing arguments to execute_core')
    cvc.execute(core_args)


@validation.invest_validator
def validate(args, limit_to=None):
    """Validate an input dictionary for Coastal Vulnerability.

    Parameters:
        args (dict): The args dictionary.
        limit_to=None (str or None): If a string key, only this args parameter
            will be validated.  If ``None``, all args parameters will be
            validated.

    Returns:
        A list of tuples where tuple[0] is an iterable of keys that the error
        message applies to and tuple[1] is the string validation warning.
    """
    warnings = []
    missing_keys = set([])
    keys_missing_values = set([])
    for required_key in ('workspace_dir',
                         'area_computed',
                         'aoi_uri',
                         'landmass_uri',
                         'bathymetry_uri',
                         'relief_uri',
                         'cell_size',
                         'exposure_proportion',
                         'population_uri',
                         'urban_center_threshold',
                         'rays_per_sector',
                         'spread_radius',
                         'population_radius'):
        try:
            if args[required_key] in ('', None):
                keys_missing_values.add(required_key)
        except KeyError:
            missing_keys.add(required_key)

    if len(missing_keys) > 0:
        raise KeyError('Args is missing required keys: %s'
                       % ', '.join(sorted(missing_keys)))

    if len(keys_missing_values) > 0:
        warnings.append((keys_missing_values, 'Parameter must have a value'))

    for vector_key in ('aoi_uri',
                       'landmass_uri',
                       'geomorphology_uri',
                       'climatic_forcing_uri',
                       'continental_shelf_uri',
                       'sea_level_rise_uri',
                       'structures_uri',
                       'additional_layer_uri'):
        if limit_to in (vector_key, None):
            try:
                with utils.capture_gdal_logging():
                    vector = gdal.OpenEx(args[vector_key])
                if vector is None:
                    warnings.append(([vector_key],
                                     ('Parameter must be a filepath to an '
                                      'OGR-compatible vector')))
            except KeyError:
                # Not all these keys are required
                pass

    for raster_key in ('bathymetry_uri',
                       'relief_uri',
                       'population_uri'):
        if limit_to in (raster_key, None):
            # All rasters are required inputs
            with utils.capture_gdal_logging():
                raster = gdal.OpenEx(args[raster_key])
            if raster is None:
                warnings.append(([raster_key],
                                 ('Parameter must be a filepath to a '
                                  'GDAL-compatible raster')))

    for constant_key in ('bathymetry_constant',
                         'relief_constant',
                         'geomorphology_constant',
                         'habitat_constant',
                         'climatic_forcing_constant',
                         'sea_level_rise_constant',
                         'structures_constant',
                         'additional_layer_constant'):
        if limit_to in (constant_key, None):
            # None of these constants are required.
            if constant_key not in args or args[constant_key] in ('', None):
                continue

            try:
                if not (1 <= int(args[constant_key]) <= 5):
                    raise ValueError()
            except ValueError:
                warnings.append(([constant_key],
                                 ('Parameter must be an int between 1 and 5 '
                                  '(inclusive)')))

    # mean_sea_level_datum can be negative
    for positive_int_key in ('cell_size',
                             'depth_threshold',
                             'depth_contour',
                             'urban_center_threshold',
                             'elevation_averaging_radius',
                             'rays_per_sector',
                             'max_fetch',
                             'spread_radius',
                             'population_radius'):
        if (limit_to not in (positive_int_key, None) or
                positive_int_key not in args):
            continue

        try:
            if int(args[positive_int_key]) < 0:
                warnings.append((
                    [positive_int_key],
                    'Parameter must be a positive integer'))
        except ValueError:
            warnings.append((
                [positive_int_key],
                'Parameter must be a number'))
        except KeyError:
            pass

    if limit_to in ('exposure_proportion', None):
        try:
            assert (0 <= float(args['exposure_proportion']) <= 1)
        except (ValueError, AssertionError):
            warnings.append((
                ['exposure_proportion'],
                'Parameter must be a number between 0 and 1.'))

    if limit_to in ('area_computed', None):
        valid_options = ('both', 'sheltered')
        if args['area_computed'] not in valid_options:
            warnings.append((
                ['area_computed'],
                'Parameter must be one of "%s", "%s"' % valid_options))

    if limit_to in ('habitats_directory_uri', None):
        try:
            if args['habitats_directory_uri'] not in ('', None):
                if not os.path.isdir(args['habitats_directory_uri']):
                    warnings.append((
                        ['habitats_directory_uri'],
                        'Parameter must be a path to a folder on disk.'))
        except KeyError:
            # This is not a required input
            pass

    if limit_to in ('habitats_csv_uri', None):
        try:
            if args['habitats_csv_uri'] not in ('', None):
                csv.reader(open(args['habitats_csv_uri'], 'r'))
        except KeyError:
            pass
        except (csv.Error, IOError):
            warnings.append((
                ['habitats_csv_uri'],
                'Parameter must be a path to a valid CSV'))

    if limit_to in ('mean_sea_level_datum', None):
        try:
            if args['mean_sea_level_datum'] not in ('', None):
                float(args['mean_sea_level_datum'])
        except KeyError:
            pass
        except ValueError:
            warnings.append(
                (['mean_sea_level_datum'], ('Parameter must be a number')))

    if limit_to is None:
        # If the habitats_directory validates, be sure that the habitats_csv
        # also validates.
        try:
            if args['habitats_directory_uri'] not in ('', None):
                if args['habitats_csv_uri'] in ('', None):
                    warnings.append((
                        ['habitats_directory_uri', 'habitats_csv_uri'],
                        ('Habitats directory and CSV must both be provided')))
        except KeyError:
            pass

    return warnings
